/*@@@**************************************************************************
* \file  rectifierop
* \author Hernan Badino
* \date  Wed Apr  8 14:27:15 GMT 2009
* \notes 
*******************************************************************************
******************************************************************************/

/* INCLUDES */
#include <QImage>

#include "rectifierOp.h"
#include "logger.h"
#include "drawingList.h"
#include "stereoCamera.h"
#include "imageConversion.h"

#include "ippDefs.h"

#include <QTextDocument>

using namespace VIC;

/// Constructors.
CRectifierOp::CRectifierOp ( COperator * const f_parent_p )
        : COperator ( f_parent_p, "Rectifier" ),
          m_downscale_i (                   2 ),
          m_rectify_b (                  true ),
          m_saveFiles_b (               false ),
          m_lutFilePath_str ( "./rectLUT.dat" ),
          m_cutTopRows_i (                  0 ),
          m_cutBotRows_i (                  0 ),
          m_gaussPyramidLeft (      16, 16, 1 ),
          m_gaussPyramidRight (     16, 16, 1 ),
          m_offsetU_f (                   0.f )
{
    m_lutLU.setDataType ( CImage::IDT_FLOAT );
    m_lutLV.setDataType ( CImage::IDT_FLOAT );
    m_lutRU.setDataType ( CImage::IDT_FLOAT );
    m_lutRV.setDataType ( CImage::IDT_FLOAT );

    m_lutLU.setBytesPerPixel ( sizeof(float) );
    m_lutLV.setBytesPerPixel ( sizeof(float) );
    m_lutRU.setBytesPerPixel ( sizeof(float) );
    m_lutRV.setBytesPerPixel ( sizeof(float) );

    registerDrawingList ("Rectified Left Image", 
                         S2D<int> (0, 0),
                         true );

    registerDrawingList ("Rectified Right Image", 
                         S2D<int> (1, 0),
                         true );

    ADD_BOOL_PARAMETER( "Rectify", 
                        "Should the rectification be applied?", 
                        m_rectify_b, 
                        this,
                        Rectify, 
                        CRectifierOp );

    ADD_INT_PARAMETER( "Downscale", 
                       "A scale with a factor of 1/downscale will be applied to the resulting rectified images", 
                       m_downscale_i, 
                       this,
                       Downscale, 
                       CRectifierOp );

    ADD_BOOL_PARAMETER( "Save files", 
                        "Save rectified files into a new created \"rectified\" directory.", 
                        m_saveFiles_b, 
                        this,
                        SaveFiles, 
                        CRectifierOp );

    ADD_FPATH_PARAMETER( "Path to LUT file", 
                         "Path to the file were the LUTs are contained.", 
                         m_lutFilePath_str, 
                         this,
                         LutFilePath, 
                         CRectifierOp );

    ADD_INT_PARAMETER( "Cut Top Rows", 
                       "Clear top rows.", 
                       m_cutTopRows_i, 
                       this,
                       CutTopRows, 
                       CRectifierOp );

    ADD_INT_PARAMETER( "Cut Bot Rows", 
                       "Clear bottom rows.", 
                       m_cutBotRows_i, 
                       this,
                       CutBotRows, 
                       CRectifierOp );

    ADD_FLOAT_PARAMETER( "Left Image Offset U", 
                         "The left image will be horizontally [px].", 
                         m_offsetU_f,
                         this,
                         OffsetU, 
                         CRectifierOp );

    BEGIN_PARAMETER_GROUP("Displays", false, CColor::red );

      CDrawingList *list_p;
      list_p = getDrawingList("Rectified Left Image");
      ADD_DISPSTATE_PARAMETER( list_p -> getName(),
                               "Rectified left image.",
                               list_p -> getDisplayState(),
                               list_p,
                               DisplayState,
                               CDrawingList );
      
      list_p = getDrawingList("Rectified Right Image");
      ADD_DISPSTATE_PARAMETER( list_p -> getName(),
                               "Rectified right image.",
                               list_p -> getDisplayState(),
                               list_p,
                               DisplayState,
                               CDrawingList );
    END_PARAMETER_GROUP;
}

/// Virtual destructor.
CRectifierOp::~CRectifierOp ()
{
}

/// Cycle event.
bool CRectifierOp::cycle()
{
    applyEqualization();

    remapImages();

    /// Obtain original camera.
    CStereoCamera * cam_p = dynamic_cast<CStereoCamera *>(getInput ( "Camera" ) );
    m_rectifiedCamera = *cam_p;
    m_rectifiedCamera.scale ( 1./m_downscale_i );
    registerOutput ( "Rectified Camera",  &m_rectifiedCamera );

    static CIO_float s_scaleFactor = 1./m_downscale_i;
    registerOutput ( "Rectification Scale Factor",  &s_scaleFactor );

    /// Register standard output.
    printf("REGISTERING \"Rectified Left Image\"\n");
    printf("REGISTERING \"Rectified Right Image\"\n");
    
    registerOutput ( "Rectified Left Image",  &m_rectLImg );
    registerOutput ( "Rectified Right Image", &m_rectRImg );

    /// Register float image outputs.
    registerOutput ( "Rectified Float Left Image",  &m_fltLImgDst );
    registerOutput ( "Rectified Float Right Image", &m_fltRImgDst );

    /// Register original float image outputs.
    registerOutput ( "Float Left Image",  &m_fltLImgSrc );
    registerOutput ( "Float Right Image", &m_fltRImgSrc );

    m_saveFiles_b = false;
    
    if ( m_saveFiles_b )
    {
        CUShortImage *imgs_p[2] = {&m_rectLImg, &m_rectRImg };
        
        /// 
        char inputId_p [2][13] = {"Image 0 Path", "Image 1 Path"};

        for (int imgNr_i = 0;  imgNr_i < 2; ++ imgNr_i)
        {    
            std::string  filePath0_str = getCastedInputObject<CIOString, std::string> (inputId_p[imgNr_i], "");

            int pos_i = filePath0_str.find_last_of ("/\\");
        
            std::string filename_str = filePath0_str.substr(pos_i+1, filePath0_str.length()-pos_i-5);

            if ( pos_i != -1 )
                filePath0_str.erase(pos_i);
            else
                filePath0_str = std::string("."); 
        
            filePath0_str += "/rectified/";
        
            char cmd_str[256];
            sprintf(cmd_str, "mkdir %s", filePath0_str.c_str());
            system(cmd_str);

            filePath0_str += filename_str;

            QImage qimage ( imgs_p[imgNr_i]->getWidth(), 
                            imgs_p[imgNr_i]->getHeight(), 
                            QImage::Format_RGB32 );
        
            unsigned short int * src_p = (unsigned short int *)imgs_p[imgNr_i]->getData();
            SRgba *  ptr;
            for (unsigned int i = 0; i < imgs_p[imgNr_i]->getHeight(); ++i)
            {   
                ptr = (SRgba *)qimage.scanLine(i);
                for (unsigned int j = 0; j < imgs_p[imgNr_i]->getWidth(); ++j, ++ptr, ++src_p)
                {
                    ptr->a = 255;
                    ptr->r = ptr->g = ptr->b = std::min((unsigned char)(*src_p/256), (unsigned char)255);
                }
            }

            //qimage = qimage.convertToFormat ( QImage::Format_Indexed8 );

            filePath0_str += ".png";
            printf("Saving file to %s\n", filePath0_str.c_str());
            qimage.save(QString(filePath0_str.c_str()));
        }    
    }

    return COperator::cycle();
}

bool
CRectifierOp::remapImages()
{
    const int interpolation_i = IPPI_INTER_LINEAR; //IPPI_INTER_CUBIC; 
    IppiSize  srcSize_o;

    srcSize_o.width  = m_eqLImg.getWidth();
    srcSize_o.height = m_eqLImg.getHeight();

    IppiRect srcROI_o;
    srcROI_o.x      = 0;
    srcROI_o.y      = 0;
    srcROI_o.width  = srcSize_o.width;
    srcROI_o.height = srcSize_o.height;

    IppiSize dstSize_o;
    dstSize_o.width  = m_fltLImgSrc.getWidth();
    dstSize_o.height = m_fltLImgSrc.getHeight();

    IppStatus ippStatus_i;

    /// Convert unsigned short image to float image. LEFT
    ippStatus_i =
        ippiConvert_16u32f_C1R ( (Ipp16u*) m_eqLImg.getData(),
                                 dstSize_o.width * sizeof(unsigned short int),
                                 (Ipp32f*) m_fltLImgSrc.getData(),
                                 dstSize_o.width * sizeof(float),
                                 srcSize_o );

    if (ippStatus_i != ippStsNoErr)
    {
        logger::warn("IPP failed:");
        printIPPError( ippStatus_i );
        return false;
    }

    /// Convert unsigned short image to float image. RIGHT
    ippStatus_i =
        ippiConvert_16u32f_C1R ( (Ipp16u*) m_eqRImg.getData(),
                                 dstSize_o.width * sizeof(unsigned short int),
                                 (Ipp32f*) m_fltRImgSrc.getData(),
                                 dstSize_o.width * sizeof(float),
                                 srcSize_o );

    if (m_rectify_b)
    {
        if (ippStatus_i != ippStsNoErr)
        {
            logger::warn("IPP failed:");
            printIPPError( ippStatus_i );
            return false;
        }

        dstSize_o.width  = m_fltLImgDst.getWidth();
        dstSize_o.height = m_fltLImgDst.getHeight();

        /// Remap left image.
        ippStatus_i =
            ippiRemap_32f_C1R( (Ipp32f*) m_fltLImgSrc.getData( ),        // src
                               srcSize_o,                                // size
                               srcROI_o.width * sizeof(float),           // step src
                               srcROI_o,                                 // roi src
                               (Ipp32f*) m_lutLU.getData ( ),            // pxMap
                               dstSize_o.width * sizeof(float),          // step pxMap
                               (Ipp32f*) m_lutLV.getData ( ),            // pyMap
                               dstSize_o.width * sizeof(float),          // step pyMap
                               (Ipp32f*) m_fltLImgDst.getData ( ),       // dest
                               dstSize_o.width * sizeof(float),          // step dest
                               dstSize_o,                                // roi dest
                               interpolation_i );                      // interpolation

        if (ippStatus_i != ippStsNoErr)
        {
            logger::warn("IPP failed:");
            printIPPError( ippStatus_i );
            return false;
        }

        /// Remap right image.
        ippStatus_i =
            ippiRemap_32f_C1R( (Ipp32f*) m_fltRImgSrc.getData( ),        // src
                               srcSize_o,                                // size
                               srcROI_o.width * sizeof(float),           // step src
                               srcROI_o,                                 // roi src
                               (Ipp32f*) m_lutRU.getData ( ),            // pxMap
                               dstSize_o.width * sizeof(float),          // step pxMap
                               (Ipp32f*) m_lutRV.getData ( ),            // pyMap
                               dstSize_o.width * sizeof(float),          // step pyMap
                               (Ipp32f*) m_fltRImgDst.getData ( ),       // dest
                               dstSize_o.width * sizeof(float),          // step dest
                               dstSize_o,                                // roi dest
                               interpolation_i );                      // interpolation

        if (ippStatus_i != ippStsNoErr)
        {
            logger::warn("IPP failed:");
            printIPPError( ippStatus_i );
            return false;
        }

        if (m_cutTopRows_i > 0)
        {
            int size_i = sizeof(float) * dstSize_o.width * m_cutTopRows_i;
            memset( m_fltLImgDst.getData(), 0, size_i );
            memset( m_fltRImgDst.getData(), 0, size_i );       
        }

        if (m_cutBotRows_i > 0)
        {
            int size_i = sizeof(float) * dstSize_o.width * m_cutBotRows_i;
            memset( m_fltLImgDst.getScanline(dstSize_o.height-m_cutBotRows_i), 0, size_i );
            memset( m_fltRImgDst.getScanline(dstSize_o.height-m_cutBotRows_i), 0, size_i );       
        }    

        srcSize_o = dstSize_o;
        
        /// Convert float image to unsigned short image. LEFT
        ippStatus_i =
            ippiConvert_32f16u_C1R ( (Ipp32f*) m_fltLImgDst.getData(),
                                     dstSize_o.width * sizeof(float),
                                     (Ipp16u*) m_rectLImg.getData(),
                                     dstSize_o.width * sizeof(unsigned short int),
                                     srcSize_o, ippRndNear );
        
        if (ippStatus_i != ippStsNoErr)
        {
            logger::warn("IPP failed:");
            printIPPError( ippStatus_i );
            return false;
        }
        
        /// Convert float image to unsigned short image. RIGHT
        ippStatus_i =
            ippiConvert_32f16u_C1R ( (Ipp32f*) m_fltRImgDst.getData(),
                                     dstSize_o.width * sizeof(float),
                                     (Ipp16u*) m_rectRImg.getData(),
                                     dstSize_o.width * sizeof(unsigned short int),
                                     srcSize_o, ippRndNear );
        
        if (ippStatus_i != ippStsNoErr)
        {
            logger::warn("IPP failed:");
            printIPPError( ippStatus_i );
            return false;
        }
    }
    else
    {
        if (m_downscale_i > 1)
        {
            if ( (int) m_gaussPyramidLeft.getLevels() < m_downscale_i ||
                 m_gaussPyramidLeft.getWidth() != m_eqLImg.getWidth() || 
                 m_gaussPyramidLeft.getHeight() != m_eqLImg.getHeight() )
            {
                m_gaussPyramidLeft.setImageSize ( m_eqLImg.getWidth(), 
                                                  m_eqLImg.getHeight() );
                m_gaussPyramidLeft.setLevels ( m_downscale_i );

                m_gaussPyramidRight.setImageSize ( m_eqLImg.getWidth(), 
                                                   m_eqLImg.getHeight() );
                m_gaussPyramidRight.setLevels ( m_downscale_i );
            }

            
            CImageConversion::convert<uint16_t, float> ( m_eqLImg,
                                                         m_fltLImgSrc );

            CImageConversion::convert<uint16_t, float> ( m_eqRImg,
                                                         m_fltRImgSrc );

            m_gaussPyramidLeft.compute ( m_fltLImgSrc );
            m_gaussPyramidRight.compute( m_fltRImgSrc );

            m_fltLImgDst.copyImageContentFrom ( *m_gaussPyramidLeft.getLevelImage  ( m_downscale_i - 1 ) );
            m_fltRImgDst.copyImageContentFrom ( *m_gaussPyramidRight.getLevelImage ( m_downscale_i - 1 ) );
            

            CImageConversion::convert<float, uint16_t> ( m_fltLImgDst,
                                                         m_rectLImg );

            CImageConversion::convert<float, uint16_t> ( m_fltRImgDst,
                                                         m_rectRImg );

        }
        // Just copy the images.
        else
        {   
            m_rectLImg   .copyImageContentFrom ( m_eqLImg );
            m_rectRImg   .copyImageContentFrom ( m_eqRImg );
            m_fltLImgDst .copyImageContentFrom ( m_fltLImgSrc );
            m_fltRImgDst .copyImageContentFrom ( m_fltRImgSrc );
        }   
    }
    
    return true;
}


void CRectifierOp::applyEqualization()
{
    /// Get input from parent.
    CUShortImage * imgL_p = dynamic_cast<CUShortImage *>(getInput ( "Image Left" ) );
    CUShortImage * imgR_p = dynamic_cast<CUShortImage *>(getInput ( "Image Right" ) );

    unsigned int size_ui = imgL_p -> getWidth() * imgR_p -> getHeight();

    m_eqLImg.copyFrom ( *imgL_p );
    m_eqRImg.copyFrom ( *imgR_p );

    /// Here we will modify the original images.
    if ( imgL_p && imgR_p && 
         imgL_p -> hasSameFormat ( *imgR_p ) )
    {
        if ( imgL_p->getBytesPerPixel() == 1 )
            equalizeImages<unsigned char> ( (unsigned char *) m_eqLImg.getData(), 
                                            (unsigned char *) m_eqRImg.getData(), 
                                            size_ui );
        else
            equalizeImages<unsigned short int> ( (unsigned short int *) m_eqLImg.getData(), 
                                                 (unsigned short int *) m_eqRImg.getData(), 
                                                 size_ui );
    }
}

/// Show event.
bool CRectifierOp::show()
{
    CDrawingList *  list_p;

    list_p = getDrawingList ("Rectified Left Image");
    list_p -> clear();
    list_p -> addImage ( m_rectLImg, 0, 0,
                         800, 600 );
    
    list_p = getDrawingList ("Rectified Right Image");
    list_p -> clear();
    list_p -> addImage ( m_rectRImg, 0, 0,
                         800, 600 );

    
    return COperator::show();
}

/// Init event.
bool CRectifierOp::initialize()
{
    /// Get input from parent.
    CUShortImage * imgL_p = dynamic_cast<CUShortImage *>(getInput ( "Image Left" ) );
    CUShortImage * imgR_p = dynamic_cast<CUShortImage *>(getInput ( "Image Right" ) );

    if (!imgL_p || !imgR_p)
    {
        logger::error("Missing input images.");
        return false;
    }

    if ( not imgL_p -> hasSameFormat (*imgR_p))
    {
        logger::error("Left and right images do not have the same format.");
        return false;        
    }

    int scaledWidth_i  = imgL_p -> getWidth()  / m_downscale_i;
    int scaledHeight_i = imgL_p -> getHeight() / m_downscale_i;
    
    ////////////////////////////////////

    m_fltLImgSrc.freeMemory();

    m_fltLImgSrc.setWidth  ( imgL_p->getWidth() );
    m_fltLImgSrc.setHeight ( imgL_p->getHeight() );
    m_fltLImgSrc.ensureAllocation();

    m_fltRImgSrc = m_fltLImgSrc;
    m_fltRImgSrc.ensureAllocation();

    ////////////////

    m_fltLImgDst.freeMemory();

    m_fltLImgDst.setWidth  ( scaledWidth_i  );
    m_fltLImgDst.setHeight ( scaledHeight_i );
    m_fltLImgDst.ensureAllocation();
    
    m_fltRImgDst = m_fltLImgDst;
    m_fltRImgDst.ensureAllocation();

    /////////////////////
    m_rectLImg.freeMemory();

    m_rectLImg = *imgL_p;
    m_rectLImg.setWidth  ( scaledWidth_i  );
    m_rectLImg.setHeight ( scaledHeight_i  );
    m_rectLImg.ensureAllocation();

    m_rectRImg.freeMemory();
    m_rectRImg = m_rectLImg;    
    m_rectRImg.ensureAllocation();

    /// Register standard output.
    printf("REGISTERING \"Rectified Left Image\"\n");
    printf("REGISTERING \"Rectified Right Image\"\n");

    registerOutput ( "Rectified Left Image",  &m_rectLImg );
    registerOutput ( "Rectified Right Image", &m_rectRImg );

    static CIO_float s_scaleFactor = 1./m_downscale_i;
    registerOutput ( "Rectification Scale Factor",  &s_scaleFactor );

    ////////////////////////////////////

    std::string  filePath0_str = getCastedInputObject<CIOString, std::string> ("Image 0 Path", "");
    std::string  filePath1_str = getCastedInputObject<CIOString, std::string> ("Image 1 Path", "");

    int pos_i = filePath0_str.find_last_of ("/\\");

    if ( pos_i != -1 )
        filePath0_str.erase(pos_i);
    else
        filePath0_str = std::string("."); 

    filePath0_str += "/rectLUT.dat";

    //m_lutFilePath_str = "/home/badino/cv/testApplication/bin/rectLUT.dat";
    if ( m_lutFilePath_str != "" )
        filePath0_str = m_lutFilePath_str;
    
    printf("Opening LUT file %s\n", filePath0_str.c_str() );
    FILE *file_p = fopen ( filePath0_str.c_str(), "r");
    
    if (!file_p)
    {
        logger::warn("LUTs file \"rectLUT.dat\" was not found");
        return false;
    }
    else
    {
        unsigned int width_ui, height_ui, count_ui;
        
        fscanf(file_p, "%i %i %i\n", 
               &width_ui,
               &height_ui,
               &count_ui );
        
        if (count_ui != 2 || 
            width_ui != imgL_p -> getWidth() ||
            height_ui != imgL_p -> getHeight() )
        {
            logger::warn("LUTs file \"rectLUT.dat\" has incorrect format or does not correspond to the current sequence."); 
            fclose(file_p);
            return false;
        }
        
        m_lutLU.freeMemory ( );
        m_lutLU.setWidth  ( scaledWidth_i  );
        m_lutLU.setHeight ( scaledHeight_i );
        m_lutLU.ensureAllocation();
        
        m_lutLV = m_lutLU;
        m_lutLV.ensureAllocation();
        
        m_lutRU = m_lutLU;
        m_lutRU.ensureAllocation();
        
        m_lutRV = m_lutLU;
        m_lutRV.ensureAllocation();        
        
        unsigned int size_ui = width_ui * height_ui * sizeof(float);
        
        // if no scale must be applied, then we just read the info in the 
        // corresponding buffers.
        if ( m_downscale_i == 1)
        {    
            if ( fread ( m_lutLU.getData(), size_ui, 1, file_p) != 1 )
            {
                logger::warn("LUTs file \"rectLUT.dat\": left U LUT could not be read.");
                return false;
            }
        
            if ( fread ( m_lutLV.getData(), size_ui, 1, file_p) != 1 )
            {
                logger::warn("LUTs file \"rectLUT.dat\": left V LUT could not be read.");
                return false;
            }
        
            if ( fread ( m_lutRU.getData(), size_ui, 1, file_p) != 1 )
            {
                logger::warn("LUTs file \"rectLUT.dat\": right U LUT could not be read.");
                return false;
            }
        
            if ( fread ( m_lutRV.getData(), size_ui, 1, file_p) != 1 )
            {
                logger::warn("LUTs file \"rectLUT.dat\": right V LUT could not be read.");
                return false;
            }
        }
        else
        {
            /// Allocate an image with appropriate format.
            CImage temp = m_fltLImgSrc;
            temp.ensureAllocation();
            
            IppiSize srcSize;
            srcSize.width  = temp.getWidth();
            srcSize.height = temp.getHeight();
            IppiRect srcRoi;
            srcRoi.x = 0;
            srcRoi.y = 0;
            srcRoi.width  = srcSize.width;
            srcRoi.height = srcSize.height;
            
            IppiSize dstRoiSize;
            
            //dstRoiSize.x = 0;
            //dstRoiSize.y = 0;
            dstRoiSize.width  = scaledWidth_i;
            dstRoiSize.height = scaledHeight_i;

            if ( fread ( temp.getData(), size_ui, 1, file_p) != 1 )
            {
                logger::warn("LUTs file \"rectLUT.dat\": left U LUT could not be read.");
                return false;
            }

            ippiResize_32f_C1R ( (float * ) temp.getData(), srcSize,
                                 srcSize.width * sizeof(float), srcRoi,
                                 (float * ) m_lutLU.getData(), dstRoiSize.width * sizeof(float),
                                 dstRoiSize, 1./m_downscale_i, 1./m_downscale_i, IPPI_INTER_LINEAR);

            if ( fread ( temp.getData(), size_ui, 1, file_p) != 1 )
            {
                logger::warn("LUTs file \"rectLUT.dat\": left U LUT could not be read.");
                return false;
            }
            
            ippiResize_32f_C1R ( (float * ) temp.getData(), srcSize,
                                 srcSize.width * sizeof(float), srcRoi,
                                 (float * ) m_lutLV.getData(), dstRoiSize.width * sizeof(float),
                                 dstRoiSize, 1./m_downscale_i, 1./m_downscale_i, IPPI_INTER_LINEAR);

            if ( fread ( temp.getData(), size_ui, 1, file_p) != 1 )
            {
                logger::warn("LUTs file \"rectLUT.dat\": left U LUT could not be read.");
                return false;
            }
            
            ippiResize_32f_C1R ( (float * ) temp.getData(), srcSize,
                                 srcSize.width * sizeof(float), srcRoi,
                                 (float * ) m_lutRU.getData(), dstRoiSize.width * sizeof(float),
                                 dstRoiSize, 1./m_downscale_i, 1./m_downscale_i, IPPI_INTER_LINEAR);

            if ( fread ( temp.getData(), size_ui, 1, file_p) != 1 )
            {
                logger::warn("LUTs file \"rectLUT.dat\": left U LUT could not be read.");
                return false;
            }
            
            ippiResize_32f_C1R ( (float * ) temp.getData(), srcSize,
                                 srcSize.width * sizeof(float), srcRoi,
                                 (float * ) m_lutRV.getData(), dstRoiSize.width * sizeof(float),
                                 dstRoiSize, 1./m_downscale_i, 1./m_downscale_i, IPPI_INTER_LINEAR);
            
        }        
        
        fclose (file_p);

        if ( m_offsetU_f ) 
        {
            int size_i = m_lutLU.getWidth() * m_lutLU.getHeight();
            float *p = (float *)m_lutLU.getData();
            float *l = p + size_i;

            for (; p < l; ++p)
                *p += m_offsetU_f;            

            /*
            p = (float *)m_lutRU.getData();
            l = p + size_i;
            for (; p < l; ++p)
                *p += m_offsetU_f;
                */
            /*
            p = m_lutLV.getData();
            l = p + size_i;
            for (; p < l; ++p)
                *p += m_offsetU_f;

            p = m_lutRV.getData();
            l = p + size_i;
            for (; p < l; ++p)
                *p += m_offsetU_f;
                */
        }
        
    }

    return COperator::initialize();
}

/// Reset event.
bool CRectifierOp::reset()
{
    logger::msg("Reset called");
    return COperator::reset();
}

bool CRectifierOp::exit()
{
    logger::msg("Exit called");
    return COperator::exit();
}

void 
CRectifierOp::keyPressed ( CKeyEvent * f_event_p )
{
   return COperator::keyPressed ( f_event_p );    
}
