#ifndef __TRIANGLELIST_H
#define __TRIANGLELIST_H

/**
 *******************************************************************************
 *
 * @file triangleList.h
 *
 * \class CTriangleList
 * \date  Tue Oct 13, 2009
 * \author Hernan Badino (hernan.badino@gmail.com)
 *
 * \brief Handles a list of triangles for displaying in screen or writing in file.
 *
 * The class is derived from CDrawingElementList implementing a list of triangles.
 * Triangles are composed of: 
 *  - rows and columns of the 3 corresponding vertices,
 *  - an outline color, 
 *  - a fill color, and 
 *  - a line width.
 *
 *******************************************************************************/

/* INCLUDES */
#include "drawingElementList.h"
#include "standardTypes.h"
#include "colors.h"

#include <vector>

/* CONSTANTS */


namespace VIC
{
    class CTriangleList: public CDrawingElementList
    {
        /// Constructor, Destructor
    public:
        /// Constructor
        CTriangleList( int f_bufferSize_i = -1);

        /// Destructor.
        virtual ~CTriangleList();

        /// Operations.
    public:

        // Add drawing from other list.
        virtual bool add (  const CTriangleList & f_otherList );

        // Add empty triangle.
        virtual bool add (  float   f_u1_f, 
                            float   f_v1_f,
                            float   f_u2_f,
                            float   f_v2_f,
                            float   f_u3_f, 
                            float   f_v3_f,
                            const SRgba   &f_color,
                            float   f_lineWidth_i );

        // Add empty triangle.
        virtual bool add (  const S2D<float> &f_vertex1, 
                            const S2D<float> &f_vertex2, 
                            const S2D<float> &f_vertex3, 
                            const SRgba   &f_color,
                            float   f_lineWidth_i );

        // Add filled triangle.
        virtual bool add (  float   f_u1_f, 
                            float   f_v1_f,
                            float   f_u2_f,
                            float   f_v2_f,
                            float   f_u3_f,
                            float   f_v3_f,
                            const SRgba   &f_outLineColor,
                            const SRgba   &f_fillColor,
                            float   f_lineWidth_i );

        // Add filled triangle.
        virtual bool add (  const S2D<float> &f_vertex1, 
                            const S2D<float> &f_vertex2, 
                            const S2D<float> &f_vertex3, 
                            const SRgba   &f_outLineColor,
                            const SRgba   &f_fillColor,
                            float   f_lineWidth_i );

        // Is blendable.
        virtual bool isBlendable () { return true; }

        // Clear all triangles.
        virtual bool clear ();

        // Draw all triangles.
        virtual bool show () const;

        // Write triangle in a SVG file.
        virtual bool write ( FILE*                f_file_p,
                             const float          f_offsetU_f = 0.0,
                             const float          f_offsetV_f = 0.0,
                             const std::string    f_parameters_str = "") const;

        // Return number of elements.
        virtual int  getSize () const;

    protected:
        typedef struct
        {
            /// Color information
            SRgba          outlineColor;

            /// Color information
            SRgba          fillColor;

            S2D<float>     vertices[3];

            /// Line width
            float          lineWidth_f;
        } STriangle;

        /// Private Members
    private:
        std::vector<STriangle>    m_triangle_v;
    };
} // Namespace VIC


#endif // __TRIANGLELIST_H

/* ////////////  Version History ///////////////
 *  $Log: triangleList.h,v $
 *  Revision 1.3  2009/11/18 15:51:01  badino
 *  badino: documentation added. Some other global changes.
 *
 *  Revision 1.2  2009/10/13 19:52:26  badino
 *  badino: added documentation.
 *
 *//////////////////////////////////////////////
