#ifndef __SCREEN_H
#define __SCREEN_H

/**
 *******************************************************************************
 *
 * @file standardTypes.h
 *
 * \class S2D
 * \date  Tue Oct 13, 2009
 * \author Hernan Badino (hernan.badino@gmail.com)
 *
 * \brief Implements a template class for handling 2 dimensional values.
 *
 * The class defines a template for handling 2 dimensional values. The 2 
 * public values can be accessed in multiple ways (by defining an Union):
 *   - x, y
 *   - width, height
 *   - min, max
 *
 *
 *******************************************************************************/

/* INCLUDES */
#include <limits>
#include "ioObj.h"

/* CONSTANTS */

namespace VIC
{
    template <class Type>
    class S2D: public CIOObj
    {
    public:
        /// Constructor.
        S2D ( const Type f_x = std::numeric_limits<Type>::max(), 
              const Type f_y = std::numeric_limits<Type>::max() )
                : x ( f_x ),
                  y ( f_y ) {}

        int operator < ( const S2D<Type> &f_other ) const
        { 
            return ( y < f_other.y || 
                     ( y == f_other.y && 
                       x < f_other.y ) );
        }

        bool operator == ( const S2D<Type> &f_other ) const
        { 
            return ( y == f_other.y && x == f_other.x );
        }

        bool operator != ( const S2D<Type> &f_other ) const
        { 
            return ! (operator == ( f_other ));
        }

        bool isValid () const
        { 
            return ( x != std::numeric_limits<Type>::max() || 
                     y != std::numeric_limits<Type>::max() );
        }

        void invalidate () 
        { 
            x = std::numeric_limits<Type>::max();
            y = std::numeric_limits<Type>::max();
        }

        union
        {
            struct {             
                /// x
                Type x;
                
                /// y
                Type y;
            };
            
            struct {
                /// width
                Type width;
                
                /// height
                Type height;
            };
            
            struct {             
                /// min
                Type min;
                
                /// max
                Type max;
            };
        };
    };

    /// We have to define the types inside a struct
    /// because typedef does not work with templates
    /// in C++ so far.
    template <class Type>
    struct SType
    {
        typedef S2D<Type> S2DCoord;
        typedef S2D<Type> SPosition;
    };

/**
 * @file standardTypes.h
 *
 * \class SDisplayState
 * \date  Tue Oct 13, 2009
 * \author Hernan Badino (hernan.badino@gmail.com)
 *
 * \brief Implements a container of the state of a display screen.
 *
 * Implements a container with a S2D<int> for storing position and 
 * a boolean variable for storing visibility of a screen.
 *
 */
    struct SDisplayState
    {
        /// Constructor
        SDisplayState (S2D<int> pos, bool f_visible_b)
                : position (          pos ),
                  visible_b(  f_visible_b ) {}
        
        /// Operator ==
        bool operator == (const SDisplayState & other ) const
        {
            return other.position == position && other.visible_b == visible_b;
        }    
        
        /// Line width.
        S2D<int>                   position;
        
        /// Line width.
        bool                       visible_b;
    };
}

#endif // __SCREEN_H

/* ////////////  Version History ///////////////
 *  $Log: standardTypes.h,v $
 *  Revision 1.3  2009/11/18 15:51:01  badino
 *  badino: documentation added. Some other global changes.
 *
 *  Revision 1.2  2009/10/13 19:52:26  badino
 *  badino: added documentation.
 *
 *//////////////////////////////////////////////
