#ifndef __SEQDEVHDIMGSEQ_H
#define __SEQDEVHDIMGSEQ_H

/**
 *******************************************************************************
 *
 * @file seqDevHDImg.h
 *
 * \class seqDevHDImgSeq
 * \date  Tue Oct 13, 2009
 * \author Hernan Badino (hernan.badino@gmail.com)
 *
 * \brief Device control class for reading sequence of image from the hard disk.
 *
 * This class is derived from CSeqDeviceControl and implements the virtual
 * functions in the parent class. It registers as output the images read from a 
 * directory specified throught parameters obtained from a xml file (with a call to
 * loadNewSequence ( const std::string )).
 *
 *******************************************************************************/

/* INCLUDES */

/* CONSTANTS */

/* INCLUDES */
#include "seqDeviceControl.h"
#include "ioObj.h"
#include "image.h"

#include <vector>
#include <map>
#include <QObject>

/* CONSTANTS */
#define VIC_SEQDEVHD_MAX_IMG_PER_FRAME 256

/* PROTOTYPES */
class QTimer;

namespace VIC
{
    /// Max number of images per frame.
    const unsigned char   m_maxImgsPerFrame_uc = 255;

    /* PROTOTYPES */
    class CSeqDevHDImgDlg;
    
    class CSeqDevHDImg: public CSeqDeviceControl
    {
        Q_OBJECT

        typedef std::vector<std::string> StringList_t;
        
        /// Constructors, Destructors
    public:
        /// Constructor
        CSeqDevHDImg();

        /// Destructor
        virtual ~CSeqDevHDImg();

    /// Sequence Handling.
    public:
        /// Initialize Device.
        virtual bool initialize();

        /// Load next frame
        virtual bool nextFrame();

        /// Load previous frame
        virtual bool prevFrame();

        /// Load next frame
        virtual bool goToFrame( int f_frameNumber_i );

        /// Stop/Stand
        virtual bool stop();

         /// Play
        virtual bool startPlaying();

         /// Play Backwards
        virtual bool startPlayingBackward();

         /// Pause
        virtual bool pause();

    /// Get/Set 
    public:
        
        /// Get number of frames in this sequence.
        virtual int getNumberOfFrames() const;
 
        /// Get current frame in the sequence.
        virtual int getCurrentFrame() const;

        /// Is a forward/backward device?
        virtual bool isBidirectional() const;

        /// Get the dialogs of this device.
        virtual std::vector<QDialog *> getDialogs ( ) const;

    /// Register outputs
    public:
        virtual bool registerOutputs ( 
                std::map< std::string, CIOObj* > &fr_map );

    /// Register outputs
    public slots:
        virtual bool loadNewSequence ( const std::string &f_confFilePath_str );

        /// Get current frame in the sequence.
        virtual void timeOut();

    /// Virtual signals
    signals:
        virtual void start();
        virtual void cycle();
        virtual void reset();

    /// Private methods.
    private:
        void   findFiles ( std::string    f_fullPath_str, 
                           std::string    filter_str, 
                           StringList_t  &fr_fileNames ) const;
        
        bool   loadCurrentFrame();

        bool   loadImageFile( std::string f_filePath_str, CImage * f_image_p );

        void   freeImages();
        
        void   allocateImages();
        
        double getTimeStampFromFilename( std::string f_fileName_p );
        
    /// Private constants.
    private:
        
    /// Protected members
    private:
        /// Dialog.
        CSeqDevHDImgDlg *             m_dialog_p;

        /// Timer for handling play actions.
        QTimer *                      m_qtPlay_p;

        /// Current frame
        int                           m_currentFrame_i;

        /// Number of frames of the sequence.
        int                           m_framesCount_i;

        /// Backward play.
        bool                          m_backward_b;

        /// Max number of images per frame.
        unsigned char                 m_imagesPerFrame_uc;

        /// Buffer of output images.
        CImage *                      m_output_p[m_maxImgsPerFrame_uc];

        /// Buffer of directories where files are located.
        std::string                   m_directoryPath_p[m_maxImgsPerFrame_uc];

        /// Buffer of output images.
        StringList_t                  m_fileName_p[m_maxImgsPerFrame_uc];

        /// Name of the sequence.
        std::string                   m_seqName_str;

        /// Name of the sequence.
        CIOString                     m_filePaths_p[m_maxImgsPerFrame_uc];

        /// Output object for frame number.
        CIO_ui32                      m_imgNr;

        /// Time stamps of the images.
        CIO_double                    m_timeStamps_p[m_maxImgsPerFrame_uc];
        
    };
}

#endif // __SEQDEVHDIMGSEQ_H

/* ////////////  Version History ///////////////
 *  $Log: seqDevHDImg.h,v $
 *  Revision 1.2  2009/11/18 15:51:01  badino
 *  badino: documentation added. Some other global changes.
 *
 *//////////////////////////////////////////////
