// ========== PARTICLE METHODS =============

// Update the position based on force and velocity
function particleUpdate() {
    if (this.bFixed == false) {
        this.vx *= this.damping;
        this.vy *= this.damping;
  
        this.limitVelocities();
        this.handleBoundaries();
        this.px += this.vx;
        this.py += this.vy;
    }
}


// Prevent particle velocity from exceeding maxSpeed
function particleLimitVelocities() {
    if (this.bLimitVelocities) {
        var speed = sqrt(this.vx * this.vx + this.vy * this.vy);
        var maxSpeed = 10;
        if (speed > maxSpeed) {
            this.vx *= maxSpeed / speed;
            this.vy *= maxSpeed / speed;
        }
    }
}


// do boundary processing if enabled
function particleHandleBoundaries() {
    if (this.bPeriodicBoundaries) {
        if (this.px > width) this.px -= width;
        if (this.px < 0) this.px += width;
        if (this.py > height) this.py -= height;
        if (this.py < 0) this.py += height;
    } else if (this.bHardBoundaries) {
        if (this.px >= width) {
            this.vx = -abs(this.vx);
        }
        if (this.px <= 0) {
            this.vx = abs(this.vx);
        }
        if (this.py >= height) {
            this.vy = -abs(this.vy);
        }
        if (this.py <= 0) {
            this.vy = abs(this.vy);
        }
    }
}


// draw the particle as a white circle
function particleDraw() {
    fill(255);
    ellipse(this.px, this.py, 9, 9);
}


// add a force to the particle using F = mA
function particleAddForce(fx, fy) {
    var ax = fx / this.mass;
    var ay = fy / this.mass;
    this.vx += ax;
    this.vy += ay;
}


// make a new particle
function makeParticle(x, y, dx, dy) {
    var p = {px: x, py: y, vx: dx, vy: dy,
             mass: 1.0, damping: 0.9,
             bFixed: false,
             bLimitVelocities: false,
             bPeriodicBoundaries: false,
             bHardBoundaries: false,
             addForce: particleAddForce,
             update: particleUpdate,
             limitVelocities: particleLimitVelocities,
             handleBoundaries: particleHandleBoundaries,
             draw: particleDraw
            }
    return p;
}
  
  
// ========== THE MAIN PROGRAM ============

var myParticles = [];
var nParticles = 30; 
var group = true;
var follow = true;
 
function setup() {
    createCanvas(400, 400);
 
    for (var i = 0; i < nParticles; i++) {
        var rx = random(width);
        var ry = random(height);
        var p = makeParticle(rx, ry, 0, 0);
        p.bHardBoundaries = true;
        p.damping = 0.99;
        myParticles[i] = p;
    }
    frameRate(10);
}
 
 
function keyPressed() {
    if (key === "S") {
        for (var i = 0; i < myParticles.length; i++) {
            myParticles[i].px = random(width);
            myParticles[i].py = random(height);
        }
    } else if (key === "G") {
        group = !group;
    } else if (key === "F") {
        follow = !follow;
    }
}


// apply separation rule to particle p
var desiredSeparation = 50;
var separationFactor = 0.01;
function separate(p) { 
    // rule applies if separation is < desired separation, 
    // apply an opposing force to achieve greater separation
    // opposing force grows as the separation becomes less
    for (var i = 0; i < myParticles.length; i++) {
        var boid = myParticles[i]; // get each other particle
        var d = dist(p.px, p.py, boid.px, boid.py);
        if (d > 1 && d < desiredSeparation) {
            // divide by distance so that the total force is 1
            var fx = (p.px - boid.px) / d;
            var fy = (p.py - boid.py) / d;
            // scale force by (desiredSeparation - d), which
            // is 0 at desiredSeparation and grows as distance
            // becomes less
            var factor = (desiredSeparation - d) * separationFactor;
            p.addForce(fx * factor, fy * factor);
        }
    }
}


// desired velocity is the average velocity of others
var alignmentFactor = 0.01;
function alignment(p) {
    // similar in separate, get the average velocity of neighbors
    // (note: we're getting velocity here, not position)
    var sumx = 0;  // sum of x and y velocities
    var sumy = 0;
    for (var i = 0; i < myParticles.length; i++) {
        var boid = myParticles[i]; // get each other particle
        sumx += boid.vx;
        sumy += boid.vy;
    }
    p.addForce((sumx / myParticles.length - p.vx) * alignmentFactor,
               (sumy / myParticles.length - p.vy) * alignmentFactor);
}    


// steer toward center of flock
var cohesionFactor = 0.0001;
function cohesion(p) {
    // get average position of all particles
    var sumx = 0;  // sum of x and y locations
    var sumy = 0;
    var count = 0; // how many neighbors
    for (var i = 0; i < myParticles.length; i++) {
        var boid = myParticles[i]; // get each other particle
        sumx += boid.px;
        sumy += boid.py;
    }
    var factor = cohesionFactor;
    if (!group) {
        factor = -cohesionFactor;
    }
    p.addForce((sumx / myParticles.length - p.px) * factor,
               (sumy / myParticles.length - p.py) * factor);
}    


var seekFactor = 0.0003;
function seek(p) {
    var desiredX = mouseX - p.px;
    var desiredY = mouseY - p.py;
    p.addForce(desiredX * seekFactor, desiredY * seekFactor);
}


// avoid is similar to separate but it acts to move away from
// the given point at x, y, with desired separation size s
var avoidFactor = 0.05;
function avoid(p, x, y, s) {
    var d = dist(p.px, p.py, x, y);
    if (d > 1 && d < s) {
        // divide by distance so that the total force is 1
        var fx = (p.px - x) / d;
        var fy = (p.py - y) / d;
        // scale force by (s - d), which
        // is 0 at s and grows as distance becomes less
        var factor = (s - d) * avoidFactor;
        p.addForce(fx * factor, fy * factor);
    }
}



//==========================================================
function draw() {
    background(200);
    text("Type F or G", 10, 15);
    if (follow) text("(F)ollow", 10, 30);
    if (group) text("(G)roup", 10, 45);
 
    ellipseMode(CENTER);
    color(200, 0, 0);
    ellipse(height / 2, width / 2, 30, 30);

    color(0);
    for (var i = 0; i < myParticles.length; i++) {
        var ithParticle = myParticles[i];
        ithParticle.fx = 0;
        ithParticle.fy = 0;
        separate(ithParticle);
        alignment(ithParticle);
        cohesion(ithParticle);
        // this avoids the point in the middle of canvas
        avoid(ithParticle, height / 2, width / 2, 40);
        if (follow) seek(ithParticle);
    }
 
    for (var i = 0; i < myParticles.length; i++) {
        var p = myParticles[i]
        p.update(); // update all locations
        p.draw(); // draw all particles
    }
}