// ==++==
//
//
//		Copyright (c) 2002 Microsoft Corporation.	 All rights reserved.
//
//		The use and distribution terms for this software are contained in the file
//		named license.txt, which can be found in the root of this distribution.
//		By using this software in any fashion, you are agreeing to be bound by the
//		terms of this license.
//
//		You must not remove this notice, or any other, from this software.
//
//
// ==--==
/*
	Wraps handle table to implement various handle types (Strong,
	Weak, etc.)
*/

#include "common.h"
#include "vars.hpp"
#include "object.h"
#include "log.h"
#include "eeconfig.h"
#include "gc.h"
#include "nstruct.h"
#include "gcscan.h"
#include "gcsmppriv.h"	// fake generation numbers

#define ARRAYSIZE(a) (sizeof(a) / sizeof(a[0]))

#include "objecthandle.h"

/*
	The following are simple routines to lift promote_func
	callbacks to HANDLESCANPROC callbacks.
*/
static void CALLBACK
ShortWeakHandle(Object** root, LPARAM* udata, LPARAM lp1, LPARAM lp2)
{
	promote_func* fn = (promote_func*)lp1;
	ScanContext* sc = (ScanContext*)lp2;
	ASSERT(fn && sc);

	// FIXME SPOONS other flags?
	logRoot(root, "short-lived weak handle");
	(fn)(root, sc, 0);
}

static void CALLBACK
LongWeakHandle(Object** root, LPARAM* udata, LPARAM lp1, LPARAM lp2)
{
	promote_func* fn = (promote_func*)lp1;
	ScanContext* sc = (ScanContext*)lp2;

	ASSERT(fn && sc);

	// FIXME SPOONS other flags?
	logRoot(root, "long-lived weak handle");
	(fn)(root, sc, 0);
}

static void CALLBACK
StrongHandle(Object** root, LPARAM* udata, LPARAM lp1, LPARAM lp2)
{
	promote_func* fn = (promote_func*)lp1;
	ScanContext* sc = (ScanContext*)lp2;
	ASSERT(fn && sc);

	// FIXME SPOONS other flags?
	logRoot(root, "strong handle");
	(fn)(root, sc, 0);
}

static void CALLBACK
PinnedHandle(Object** root, LPARAM* udata, LPARAM lp1, LPARAM lp2)
{
	promote_func* fn = (promote_func*)lp1;
	ScanContext* sc = (ScanContext*)lp2;

	ASSERT(fn && sc);

	// FIXME SPOONS other flags?
	logRoot(root, "pinned handle");
	(fn)(root, sc, GCHeap::GC_CALL_PINNED);
}

#ifdef GC_PROFILING
	void CALLBACK
	ToProfiler(
		Object** pObjRef,
		LPARAM* pExtraInfo,
		LPARAM lp1,
		LPARAM lp2)
	{
		LOG((LF_GC | LF_CORPROF, LL_INFO100000,
			LOG_HANDLE_OBJECT_CLASS("Notifying profiler of ",
				pObjRef, "to ", *pObjRef)));

		g_profControlBlock.pProfInterface->RootReference(
			(ObjectID)*pObjRef,
			NULL);
	}
#endif // GC_PROFILING

HHANDLETABLE g_hGlobalHandleTable = NULL;

/*
	The definition of this structure *must* be kept up to date with the
	definition in dump-tables.cpp.
*/
struct HandleTableMap{
	HHANDLETABLE* pTable;
	struct HandleTableMap *pNext;
	DWORD dwMaxIndex;
};

HandleTableMap g_HandleTableMap = {NULL,0,0};

#define INITIAL_HANDLE_TABLE_ARRAY_SIZE 10

// flags describing the handle types
static UINT s_rgTypeFlags[] ={
	HNDF_NORMAL, // HNDTYPE_WEAK_SHORT
	HNDF_NORMAL, // HNDTYPE_WEAK_LONG
	HNDF_NORMAL, // HNDTYPE_STRONG
	HNDF_NORMAL, // HNDTYPE_PINNED
	HNDF_EXTRAINFO, // HNDTYPE_VARIABLE
};

static UINT AllHandleTypes[] = {
	HNDTYPE_WEAK_SHORT,
	HNDTYPE_WEAK_LONG,
	HNDTYPE_STRONG,
	HNDTYPE_PINNED,
	HNDTYPE_VARIABLE,
};

BOOL
Ref_Initialize()
{
	// sanity
	_ASSERTE(g_hGlobalHandleTable == NULL);

	// Create an array to hold the handle tables
	HHANDLETABLE* pTable = new HHANDLETABLE [
		INITIAL_HANDLE_TABLE_ARRAY_SIZE ];
	if (pTable == NULL) {
		return FALSE;
	}
	ZeroMemory(pTable,
		INITIAL_HANDLE_TABLE_ARRAY_SIZE * sizeof (HHANDLETABLE));
	g_HandleTableMap.pTable = pTable;
	g_HandleTableMap.dwMaxIndex = INITIAL_HANDLE_TABLE_ARRAY_SIZE;
	g_HandleTableMap.pNext = NULL;

	// create the handle table
	g_hGlobalHandleTable = HndCreateHandleTable(
		s_rgTypeFlags, ARRAYSIZE(s_rgTypeFlags), 0);
	if (!g_hGlobalHandleTable)
		FailFast(GetThread(), FatalOutOfMemory);
	HndSetHandleTableIndex(g_hGlobalHandleTable, 0);
	g_HandleTableMap.pTable[0] = g_hGlobalHandleTable;

	// return true if we successfully created a table
	return (g_hGlobalHandleTable != NULL);
}

void
Ref_Shutdown()
{
	// are there any handle tables?
	if (g_hGlobalHandleTable)
	{
		/*
			don't destroy any of the
			indexed handle tables; they should
			be destroyed externally.
		*/
		// destroy the global handle table
		HndDestroyHandleTable(g_hGlobalHandleTable);

		// destroy the handle table array
		HandleTableMap* walk = &g_HandleTableMap;
		while (walk) {
			delete [] walk->pTable;
			walk = walk->pNext;
		}

		// null out the handle table array
		g_HandleTableMap.pNext = NULL;
		g_HandleTableMap.dwMaxIndex = 0;

		// null out the global table handle
		g_hGlobalHandleTable = NULL;
	}
}

HHANDLETABLE
Ref_CreateHandleTable(UINT uADIndex)
{
	HHANDLETABLE result = NULL;
	HandleTableMap* walk;

	walk = &g_HandleTableMap;
	HandleTableMap* last = NULL;
	UINT offset = 0;

	result = HndCreateHandleTable(
		s_rgTypeFlags, ARRAYSIZE(s_rgTypeFlags),
		uADIndex);
	if (!result)
		FailFast(GetThread(), FatalOutOfMemory);

retry:
	// Do we have free slot
	while (walk) {
		for (UINT i = 0; i < INITIAL_HANDLE_TABLE_ARRAY_SIZE; i ++) {
			if (walk->pTable[i] == 0) {
				HndSetHandleTableIndex(result, i+offset);
				void* r = FastInterlockCompareExchangePointer(
					(void**)&walk->pTable[i], (void*)result, 0);
				if (r == 0) {
					// Get a free slot.
					return result;
				}
			}
		}
		last = walk;
		offset = walk->dwMaxIndex;
		walk = walk->pNext;
	}

	/*
		No free slot.
		Let's create a new node
	*/
	HandleTableMap *newMap = new (nothrow) HandleTableMap;
	if (newMap == NULL) {
		return NULL;
	}
	newMap->pTable = new (nothrow) HHANDLETABLE [
		INITIAL_HANDLE_TABLE_ARRAY_SIZE];
	if (newMap->pTable == NULL) {
		delete newMap;
		return NULL;
	}
	newMap->dwMaxIndex = last->dwMaxIndex
		+ INITIAL_HANDLE_TABLE_ARRAY_SIZE;
	newMap->pNext = NULL;
	ZeroMemory(newMap->pTable,
		INITIAL_HANDLE_TABLE_ARRAY_SIZE*sizeof(HHANDLETABLE));

	void* r = FastInterlockCompareExchangePointer(
		(void**)&last->pNext,newMap,NULL);
	if(r != NULL) {
		// This thread loses.
		delete [] newMap->pTable;
		delete newMap;
	}
	walk = last->pNext;
	offset = last->dwMaxIndex;
	goto retry;
}

void
Ref_RemoveHandleTable(HHANDLETABLE hTable)
{
	UINT index = HndGetHandleTableIndex(hTable);
	HandleTableMap *walk = &g_HandleTableMap;
	UINT offset = 0;

	while (walk) {
		if (index < walk->dwMaxIndex) {
			/*

				During AppDomain unloading, we first
				remove a handle table and then destroy
				the table.  As soon as the table is
				removed, the slot can be reused.
			*/
			if (walk->pTable[index-offset] == hTable)
				walk->pTable[index-offset] = NULL;
			return;
		}
		offset = walk->dwMaxIndex;
		walk = walk->pNext;
	}

	_ASSERTE (!"Should not reach here");
}

void
Ref_DestroyHandleTable(HHANDLETABLE table)
{
	HndDestroyHandleTable(table);
}

/*
	Creates a variable-strength handle.

	N.B. This routine is not a macro since we do
	validation in RETAIL.  We always validate the
	type here because it can come from external callers.
*/
OBJECTHANDLE
CreateVariableHandle(HHANDLETABLE hTable, OBJECTREF object, UINT type)
{
	// verify that we are being asked to create a valid type
	if (!IS_VALID_VHT_VALUE(type)){
		// bogus value passed in
		_ASSERTE(FALSE);
		return NULL;
	}

	// create the handle
	return HndCreateHandle(hTable, HNDTYPE_VARIABLE, object, (LPARAM)type);
}

/*
	Changes the dynamic type of a variable-strength handle.

	N.B. This routine is not a macro since we do
	validation in RETAIL.  We always validate the
	type here because it can come from external callers.
*/
void
UpdateVariableHandleType(OBJECTHANDLE handle, UINT type)
{
	// verify that we are being asked to set a valid type
	if (!IS_VALID_VHT_VALUE(type)){
		// bogus value passed in
		_ASSERTE(FALSE);
		return;
	}

	/*
		If/when concurrent GC is implemented, we need to
		make sure variable handles DON'T change type
		during an asynchronous scan, OR that we properly recover
		from the change.  Some changes are benign,
		but for example changing to or from a pinning handle
		in the middle of a scan would not be fun.
	*/

	// store the type in the handle's extra info
	HndSetHandleExtraInfo(handle, HNDTYPE_VARIABLE, (LPARAM)type);
}

/*
	BasicHandleApp applies f(root,lp1,lp2) to every handle root
	with one of the given types.  Variable-typed handles are not
	treated specially; use HandleApp for that.
*/
static void
BasicHandleApp(
	HANDLESCANPROC f,
	LPARAM lp1,
	LPARAM lp2,
	UINT* types,
	UINT ntypes,
	UINT flags)
{
	HandleTableMap* walk = &g_HandleTableMap;
	while (walk) {
		for (UINT i = 0; i < INITIAL_HANDLE_TABLE_ARRAY_SIZE; i ++) {
			HHANDLETABLE hTable = walk->pTable[i];
			if (hTable)
				HndScanHandlesForGC(
					hTable, f, lp1, lp2, types, ntypes,
					CONDEMN_GEN_NUM, MAX_GENERATION,
					flags);
		}
		walk = walk->pNext;
	}
}

/*
	HandleApp applies f(root,lp1,lp2) to every handle root with
	one of the given types.  It checks the current type of
	variable-typed handles.  Note that the types array may not
	mention HNDTYPE_VARIABLE.
*/
struct VAHCTX{
	HANDLESCANPROC f;
	LPARAM lp1;
	LPARAM lp2;
};

static void CALLBACK
VarAppHelper(Object** root, LPARAM* udata, LPARAM lp1, LPARAM lp2)
{
	UINT vht = *udata;
	UINT mask = (UINT)lp1;
	VAHCTX* c = (VAHCTX*)lp2;

	if(vht & mask){
		c->f(root,udata,c->lp1,c->lp2);
	}
}

static void
HandleApp(
	HANDLESCANPROC f,
	LPARAM lp1,
	LPARAM lp2,
	UINT* types,
	UINT ntypes,
	UINT flags)
{
	/* Apply to non-variable handles. */
	BasicHandleApp(f,lp1,lp2,types,ntypes,flags);

	/*
		Apply to variable handles.  A variable handle's type
		is always HNDTYPE_VARIABLE; its "variable type" is
		stored in userdata.
	*/
	UINT vmask = 0;
	for(UINT i=0; i<ntypes; i++){
		switch(types[i]){
		case HNDTYPE_WEAK_SHORT: vmask |= VHT_WEAK_SHORT; break;
		case HNDTYPE_WEAK_LONG: vmask |= VHT_WEAK_LONG; break;
		case HNDTYPE_STRONG: vmask |= VHT_STRONG; break;
		case HNDTYPE_PINNED: vmask |= VHT_PINNED; break;
		default:
			ASSERT(!"HandleApp on bad handle type");
		}
	}
	VAHCTX c = { f, lp1, lp2 };
	UINT vtype = HNDTYPE_VARIABLE;
	UINT vflags = flags | HNDGCF_EXTRAINFO;
	BasicHandleApp(VarAppHelper, (LPARAM)vmask, (LPARAM)&c,
		&vtype, 1, vflags);
}

void
Ref_ScanPinnedHandles(promote_func* fn, ScanContext* sc)
{
	LOG((LF_GC, LL_INFO10000, "Scanning pinned handles\n"));
	UINT type = HNDTYPE_PINNED;
	UINT flags = sc->concurrent ? HNDGCF_ASYNC : HNDGCF_NORMAL;
	HandleApp(PinnedHandle, (LPARAM)fn, (LPARAM)sc, &type, 1, flags);
}

void
Ref_ScanStrongHandles(promote_func* fn, ScanContext* sc)
{
	LOG((LF_GC, LL_INFO10000, "Scanning strong handles\n"));
	UINT type = HNDTYPE_STRONG;
	UINT flags = sc->concurrent ? HNDGCF_ASYNC : HNDGCF_NORMAL;
	HandleApp(StrongHandle, (LPARAM)fn, (LPARAM)sc, &type, 1, flags);
}

void
Ref_ScanLongWeakHandles(promote_func* fn, ScanContext* sc)
{
	LOG((LF_GC, LL_INFO10000, "Scanning long-lived weak handles\n"));
	UINT type = HNDTYPE_WEAK_LONG;
	UINT flags = sc->concurrent ? HNDGCF_ASYNC : HNDGCF_NORMAL;
	HandleApp(LongWeakHandle, (LPARAM)fn, (LPARAM)sc, &type, 1, flags);
	/*
		SPOONS: Is there confusion over whether SyncBlockCache
		contains short- or long-lived weak handles?
	*/
	SyncBlockCache::GetSyncBlockCache()->
		GCWeakPtrScan(LongWeakHandle, (LPARAM)fn, (LPARAM)sc);
}

// FIXME make a profiling version
// SPOONS: Dan, please clarify.  Ref_ScanPointersForProfiler scans short weak handles.
void
Ref_ScanShortWeakHandles(promote_func* fn, ScanContext* sc)
{
	LOG((LF_GC, LL_INFO10000, "Scanning short-lived weak handles\n"));
	UINT type = HNDTYPE_WEAK_SHORT;
	UINT flags = sc->concurrent ? HNDGCF_ASYNC : HNDGCF_NORMAL;
	HandleApp(ShortWeakHandle, (LPARAM)fn, (LPARAM)sc, &type, 1, flags);
}

#ifdef GC_PROFILING
	void
	Ref_ScanHandlesForProfiler()
	{
		LOG((LF_GC | LF_CORPROF, LL_INFO10000,
			"Scanning all handles for profiler.\n"));
		/*
			SPOONS: There was a comment "Don't scan the
			sync block because they should not be
			reported.  They are weak handles only." This
			makes no sense to me since we are reporting
			other weak handles.  -dave
		*/
		UINT types[] = {
			HNDTYPE_WEAK_SHORT,
			HNDTYPE_WEAK_LONG,
			HNDTYPE_STRONG,
			HNDTYPE_PINNED,
		};
		UINT flags = HNDGCF_NORMAL;
		HandleApp(ToProfiler, (LPARAM)fn, (LPARAM)sc,
			types, ARRAYSIZE(types), flags);
	}
#endif // GC_PROFILING

void
Ref_AgeHandles()
{
	LOG((LF_GC, LL_INFO10000, "Aging handles\n"));
	/*
		NB HndScanHandlesForGC specifically supports a NULL
		scanning function with HNDGCF_AGE while BasicHandleApp
		only accidentally supports it and should not be used
		here.
	*/
	HandleTableMap* walk = &g_HandleTableMap;
	while (walk) {
		for (UINT i = 0; i < INITIAL_HANDLE_TABLE_ARRAY_SIZE; i ++) {
			HHANDLETABLE hTable = walk->pTable[i];
			if (hTable){
				HndScanHandlesForGC(
					hTable, NULL, 0, 0,
					AllHandleTypes, ARRAYSIZE(AllHandleTypes),
					CONDEMN_GEN_NUM, MAX_GENERATION,
					HNDGCF_AGE);
			}
		}
		walk = walk->pNext;
	}
}

void
Ref_RejuvenateHandles()
{
	LOG((LF_GC, LL_INFO10000, "Rejuvenating handles.\n"));
	HandleTableMap* walk = &g_HandleTableMap;
	while (walk) {
		for (UINT i = 0; i < INITIAL_HANDLE_TABLE_ARRAY_SIZE; i ++) {
			HHANDLETABLE hTable = walk->pTable[i];
			if (hTable)
				HndResetAgeMap(hTable,
					AllHandleTypes, ARRAYSIZE(AllHandleTypes),
					HNDGCF_NORMAL);
		}
		walk = walk->pNext;
	}
}
