// ==++==
//
//
//		Copyright (c) 2002 Microsoft Corporation.	 All rights reserved.
//
//		The use and distribution terms for this software are contained in the file
//		named license.txt, which can be found in the root of this distribution.
//		By using this software in any fashion, you are agreeing to be bound by the
//		terms of this license.
//
//		You must not remove this notice, or any other, from this software.
//
//
// ==--==
/*
	Generational GC handle manager.  Entrypoint Header.

	Implements generic support for external handles into a GC
	heap.

	A handle is essentially an Object**; that is, a fixed location
	of a pointer to an object.  It looks like the JIT compiler and
	runtime use handles to store all globally scoped values.

	Every handle table has a number of parameters set when it is
	created.  Note that all handle tables are allocated by
	objecthandle.cpp and that there is a single global handle
	table (allocated in objecthandle.cpp:/^Ref_Initialize) as well
	as a handle table for each app domain (allocated with
	objecthandle.cpp:/^Ref_CreateHandleTable in
	appdomain.cpp:/^AppDomain::Init).

	A handle table's App Domain Index is set at creation.  The
	global handle table has ADIndex zero.  Each app domain d has
	its own handle table whose ADIndex is d.GetIndex().

	A handle table's Index is -1 at creation and immediately set
	to the index of the handle table in a segmented array
	maintained by objecthandle.cpp:/Ref_CreateHandleTable that
	keeps a reference to every handle table allocated.  It is used
	by objecthandle.cpp:/Ref_RemoveHandleTable to zero out the
	reference when a handle table is discarded.  (Because the
	array is segmented into a linked list of fixed-size arrays,
	finding the reference to clear --- even given this index ---
	takes linear time.  Stupid optimization?)

	A handle table has a "type" count (the types are
	0..TypeCount-1) and flags for each type.  (This generality is
	unused: Every handle table is assigned the same type count and
	type flags by objecthandle.cpp.  See objecthandle.h for a
	description of the types used.) Every handle allocated in a
	table with type count N is assigned a type in 0..N-1.  The
	only interesting handle type flag is HNDF_EXTRAINFO.  Handles
	whose type have that flag include space for a word of "user
	data".  This is used by objecthandle.cpp to implement apparant
	type changes.  The table scanning routines can efficiently
	filter handles by type.

	SOME INTERNAL DETAILS

	[These notes are from a cursory examination of the code in
	July 2005.  Use with a grain of salt.  -dave]

	Handle tables are divided into (large) segments.  Segments are
	divided into 256 blocks (eight-bit block indices are
	pervasive).  Blocks are divided into four clumps (32-bit
	arithmetic on words comprising eight-bit-per-clump data is
	pervasive).  The basic unit of allocation is the block;
	currently, a block contains 64 handles.  Each clump has an
	eight-bit generation number.

	The blocks in a segment are classified as either free or
	allocated to a particular handle "type" where the handle types
	are those defined by the client plus a type reserved for user
	data blocks.  This classification is represented by linked
	lists (called chains).  Each segment has an alloc bit for
	every handle, an allocation chain for each handle "type", and
	a free chain.  Each block has an eight-bit handle type, an
	eight-bit buddy block index, an eight-bit lock count, and an
	eight-bit next block index.  HNDF_EXTRAINFO is supported by
	allocating two blocks rather than one: One block holds the
	handles while its buddy block holds the user data.  To
	traverse a chain, you start from its tail and then follow the
	per-block next indices until you find a sentinal.

	Each segment has eight-bit indices tracking the last committed
	block and the first block in the highest committed page.

	Handle tables have a two-level cache for each handle "type" to
	speed up handle allocation and deallocation.  The quick cache
	may contain a single handle that was either allocated from the
	main cache but not yet returned to the user or freed by the
	user but not yet returned to the main cache.  Each main cache
	comprises two small banks of handles (currently 63 handles per
	bank).  One bank holds handles allocated from a segment but
	not yet from the main cache and the other bank holds handles
	freed to the main cache but not yet to their segments.  Each
	segment has an eight-bit sequence number (mod 256) maintained
	by the segment allocation and scanning code.  These are used
	by the main cache to approximate the relative age of two
	handles allocated from different segments of the handle table.

	SOME THOUGHTS

	It looks like handle tables support concurrent but not
	parallel scanning.  The per-block lock counts apparantly
	support a single collector thread performing a scan while
	mutator threads are running.  After such a scan, the collector
	thread has to lock the whole table and rescan to find any new
	tables.

	Going forward, one possible unit of work for incremental
	scanning is the segment.  It looks like it will be easy to
	modify handle table code to support mirrored handles for a
	constant-time flip when collection is over.  (I do not know
	how easy it would be to modify the JIT compiler.)
*/

#ifndef _HANDLETABLE_H
#define _HANDLETABLE_H


/*
	FLAGS, CONSTANTS AND DATA TYPES
*/

/*
	handle flags used by HndCreateHandleTable
*/
#define HNDF_NORMAL (0x00)
#define HNDF_EXTRAINFO (0x01)

/*
	handle to handle table.
	See ../../../palrt/inc/rotor_palrt.h:/DECLARE_HANDLE
*/
DECLARE_HANDLE(HHANDLETABLE);

/*
	PUBLIC ROUTINES AND MACROS
*/

/*
	handle manager init and shutdown routines
*/

HHANDLETABLE HndCreateHandleTable(
	UINT* pTypeFlags,
	UINT uTypeCount,
	UINT uADIndex);

void HndDestroyHandleTable(
	HHANDLETABLE hTable);

UINT HndGetHandleTableADIndex(
	HHANDLETABLE hTable);

void HndSetHandleTableIndex(
	HHANDLETABLE hTable,
	UINT uTableIndex);

UINT HndGetHandleTableIndex(
	HHANDLETABLE hTable);

/*
	individual handle allocation and deallocation
*/
OBJECTHANDLE HndCreateHandle(
	HHANDLETABLE hTable,
	UINT uType,
	OBJECTREF object,
	LPARAM lExtraInfo = 0);

void HndDestroyHandle(
	HHANDLETABLE hTable,
	UINT uType,
	OBJECTHANDLE handle);

void HndDestroyHandleOfUnknownType(
	HHANDLETABLE hTable,
	OBJECTHANDLE handle);

/*
	bulk handle allocation and deallocation
*/
UINT HndCreateHandles(
	HHANDLETABLE hTable,
	UINT uType,
	OBJECTHANDLE* pHandles,
	UINT uCount);

void HndDestroyHandles(
	HHANDLETABLE hTable,
	UINT uType,
	const OBJECTHANDLE* pHandles,
	UINT uCount);

/*
	User data associated with handles.  These only work for those
	handles whose type has the flag HNDF_EXTRAINFO.
*/
void HndSetHandleExtraInfo(
	OBJECTHANDLE handle,
	UINT uType,
	LPARAM lExtraInfo);

LPARAM HndGetHandleExtraInfo(
	OBJECTHANDLE handle);

/*
	get parent table of handle
*/
HHANDLETABLE HndGetHandleTable(
	OBJECTHANDLE handle);

/*
	write barrier
*/
void HndWriteBarrier(
	OBJECTHANDLE handle);

void HndWriteBarrierForGC(
	OBJECTHANDLE handle,
	OBJECTREF objref);

/*
	NON-GC handle enumeration
*/
typedef void (CALLBACK *HNDENUMPROC)(
	OBJECTHANDLE handle,
	LPARAM lExtraInfo,
	LPARAM lCallerParam);

void HndEnumHandles(
	HHANDLETABLE hTable,
	const UINT* puType,
	UINT uTypeCount,
	HNDENUMPROC pfnEnum,
	LPARAM lParam,
	BOOL fAsync);

/*
	GC-time handle scanning
*/
#define HNDGCF_NORMAL (0x00000000) // normal scan
#define HNDGCF_AGE (0x00000001) // age handles while scanning
#define HNDGCF_ASYNC (0x00000002) // drop the table lock while scanning
#define HNDGCF_EXTRAINFO (0x00000004) // iterate per-handle data while scanning

typedef void (CALLBACK *HANDLESCANPROC)(
	Object** pref,
	LPARAM* pExtraInfo,
	LPARAM param1,
	LPARAM param2);

void HndScanHandlesForGC(
	HHANDLETABLE hTable,
	HANDLESCANPROC scanProc,
	LPARAM param1,
	LPARAM param2,
	const UINT* types,
	UINT typeCount,
	UINT condemned,
	UINT maxgen,
	UINT flags);

void HndResetAgeMap(
	HHANDLETABLE hTable,
	const UINT* types,
	UINT typeCount,
	UINT flags);

void HndNotifyGcCycleComplete(
	HHANDLETABLE hTable,
	UINT condemned,
	UINT maxgen);

#if defined(USE_CHECKED_OBJECTREFS) && !defined(_NOVM)
	#define OBJECTREF_TO_OBJECT(objref) (*((Object**)&(objref)))
	#define OBJECT_TO_OBJECTREF(obj) (OBJECTREF(obj))
#else
	#define OBJECTREF_TO_OBJECT(objref) (objref)
	#define OBJECT_TO_OBJECTREF(obj) (obj)
#endif

#ifdef _DEBUG
	void ValidateAssignObjrefForHandle(
		OBJECTREF,
		UINT appDomainIndex);

	void ValidateFetchObjrefForHandle(
		OBJECTREF,
		UINT appDomainIndex);
#endif

/*
	handle assignment
*/
__inline void
HndAssignHandle(OBJECTHANDLE handle, OBJECTREF objref)
{
	// sanity
	_ASSERTE(handle);

	#ifdef _DEBUG
		// Make sure the objref is valid before it is assigned to a handle
		ValidateAssignObjrefForHandle(objref,
			HndGetHandleTableADIndex(HndGetHandleTable(handle)));
	#endif
	// unwrap the objectref we were given
	Object* value = OBJECTREF_TO_OBJECT(objref);

	// if we are doing a non-NULL pointer store then invoke the write-barrier
	if (value) {
		HndWriteBarrier(handle);

		/*
			FIXME spoons: This seems like an appropriate place to
			implement a write barrier for handle assignments, though
			I'm not sure what "HndWriteBarrier" is actually doing.
			The point here is that "value" may have been allocated
			while the collector is "in cycle", in which case, we
			should not be surprised if there is a root pointing to it
			at the end of the cycle.  We shade "value" to ensure that
			the collector is not caught unawares.
		*/
		HndWriteBarrierForGC(handle, objref);
	}

	// store the pointer
	*(Object**)handle = value;
}

/*
	interlocked-exchange assignment
*/
__inline void
HndInterlockedCompareExchangeHandle(
	OBJECTHANDLE handle,
	OBJECTREF objref,
	OBJECTREF oldObjref)
{
	// sanity
	_ASSERTE(handle);

	#ifdef _DEBUG
		// Make sure the objref is valid before it is assigned to a handle
		ValidateAssignObjrefForHandle(objref,
			HndGetHandleTableADIndex(HndGetHandleTable(handle)));
	#endif
	// unwrap the objectref we were given
	Object* value = OBJECTREF_TO_OBJECT(objref);
	Object* oldValue = OBJECTREF_TO_OBJECT(oldObjref);

	// if we are doing a non-NULL pointer store then invoke the write-barrier

	if (value) {
		HndWriteBarrier(handle);

		// Write-barrier -- see note above
		HndWriteBarrierForGC(handle, objref);
	}

	// store the pointer

	InterlockedCompareExchangePointer((PVOID *)handle, value, oldValue);
}

/*
	Note that HndFirstAssignHandle is similar to HndAssignHandle,
	except that it only succeeds if transitioning from NULL to
	non-NULL.  In other words, if this handle is being initialized for
	the first time.
*/
__inline BOOL
HndFirstAssignHandle(OBJECTHANDLE handle, OBJECTREF objref)
{
	// sanity
	_ASSERTE(handle);

	#ifdef _DEBUG
		// Make sure the objref is valid before it is assigned to a handle
		ValidateAssignObjrefForHandle(objref,
			HndGetHandleTableADIndex(HndGetHandleTable(handle)));
	#endif
	// unwrap the objectref we were given
	Object* value = OBJECTREF_TO_OBJECT(objref);

	// store the pointer if we are the first ones here
	LPVOID result = FastInterlockCompareExchangePointer(
		(void **)handle, (void*)value, NULL);
	BOOL success = result == NULL;

	/*
		if we successfully did a non-NULL pointer store then invoke
		the write-barrier
	*/
	if (value && success) {
		HndWriteBarrier(handle);

		// Write-barrier -- see note above
		HndWriteBarrierForGC(handle, objref);
	}

	// return our result
	return success;
}

FORCEINLINE Object**
HndAddr(OBJECTHANDLE handle)
{
	// sanity
	_ASSERTE(handle);

	return (Object**)handle;
}

/*
	inline handle dereferencing
*/
FORCEINLINE OBJECTREF
HndFetchHandle(OBJECTHANDLE handle)
{
	// sanity
	_ASSERTE(handle);

	#ifdef _DEBUG
		// Make sure the objref for handle is valid
		ValidateFetchObjrefForHandle(ObjectToOBJECTREF(*(Object **)handle),
			HndGetHandleTableADIndex(HndGetHandleTable(handle)));
	#endif
	// wrap the raw objectref and return it
	return OBJECT_TO_OBJECTREF(*(Object**)handle);
}

/*
	inline null testing (needed in certain cases where we're in the
	wrong GC mod)
*/
FORCEINLINE BOOL
HndIsNull(OBJECTHANDLE handle)
{
	// sanity
	_ASSERTE(handle);

	return NULL == *(Object **)handle;
}

/*
	inline handle checking
*/
FORCEINLINE BOOL
HndCheckForNullUnchecked(OBJECTHANDLE *pHandle)
{
	// sanity
	_ASSERTE(pHandle);

	return (*(Object***)pHandle == NULL
		|| (**(Object***)pHandle) == NULL);
}

#endif //_HANDLETABLE_H
