// ==++==
//
//
//		Copyright (c) 2002 Microsoft Corporation.	 All rights reserved.
//
//		The use and distribution terms for this software are contained in the file
//		named license.txt, which can be found in the root of this distribution.
//		By using this software in any fashion, you are agreeing to be bound by the
//		terms of this license.
//
//		You must not remove this notice, or any other, from this software.
//
//
// ==--==

#ifndef _GCSMPPRIV_H_
#define _GCSMPPRIV_H_

// optimize for speed
#ifndef _DEBUG
	#ifdef _MSC_VER
		#pragma optimize( "t", on )
	#endif
#endif

#define inline FORCEINLINE

#include "wsperf.h"
#include "perfcounters.h"
#include "log.h"
#include "eeconfig.h"
#include "gc.h"

#ifdef GC_PROFILING
	#include "profilepriv.h"
#endif

#if defined(_X86_)

	#ifdef _DEBUG
		#ifdef _MSC_VER
			inline void
			RetailDebugBreak()
			{
				_asm int 3
			}
		#elif defined(__GNUC__)
			inline void
			RetailDebugBreak()
			{
				__asm("int $3");
			}
		#else
			#error Unsupported compiler
		#endif
	#else
		inline void
		RetailDebugBreak()
		FATAL_EE_ERROR()
	#endif

#else // _X86_
	#define RetailDebugBreak()
#endif

#ifdef _MSC_VER
	#pragma inline_depth(20)
#endif

/* DIAGNOSTIC OPTIONS */
#define TRACE_GC //debug trace gc operation
//#define CATCH_GC //catches exception during GC
//#define MEASURE_GC_SPACE //measure space used by garbage collection
//#define GC_COUNT_ALLOC_FITS

/* End of optional features */

#ifdef _DEBUG
	#define TRACE_GC
	#define CATCH_GC
#endif

#if defined(_DEBUG) && defined(CATCH_GC)
	/*
		note that we can't use COMPLUS_TRY because the
		gc_thread isn't known
	*/
	#define GC_TRY PAL_TRY
	#define GC_EXCEPT(e) PAL_EXCEPT(e)
	#define GC_EXCEPT_FILTER(e,a) PAL_EXCEPT_FILTER(e,a)
	#define GC_ENDTRY PAL_ENDTRY
#else
	#define GC_TRY if(1){
	#define GC_EXCEPT(e) } else {
	#define GC_EXCEPT_FILTER(e,a) } else {
	#define GC_ENDTRY }
#endif

// Fake information about the "generations"
#define NUMBERGENERATIONS 1 //Max number of generations
#define CURRENT_GENERATION 0
#define CONDEMN_GEN_NUM 0
#define MAX_GENERATION 0

extern int gc_count;

extern int successful_alloc;
extern int unsuccessful_alloc;

#ifdef TRACE_GC

	extern int print_level;
	extern BOOL trace_gc;

	//#include "log.h"
	//#define dprintf(l,x) {if (trace_gc &&(l<=print_level)) {LogSpewAlways x;LogSpewAlways ("\n");}}
	#define dprintf(l,x)\
		{\
			if (trace_gc && (l<=print_level)) {\
				printf x ;\
				printf ("\n");\
				fflush(stdout);\
			}\
		}

#else //TRACE_GC

	#define dprintf(l,x)

#endif //TRACE_GC

#undef assert
#define assert _ASSERTE
#undef ASSERT
#define ASSERT _ASSERTE

#ifdef _DEBUG

	struct GCDebugSpinLock {
		volatile LONG lock; // -1 if free, 0 if held
		Thread* holding_thread; // -1 if no thread holds the lock.
	};
	typedef GCDebugSpinLock GCSpinLock;

	#define SPIN_LOCK_INITIALIZER {-1, (Thread*) -1}

#else

	typedef volatile LONG GCSpinLock;
	#define SPIN_LOCK_INITIALIZER -1

#endif

/*
	In the concurrent version, the Enable/DisablePreemptiveGC is
	optional because the gc thread call WaitLonger.
*/
void WaitLonger(int i);

inline static void
enter_spin_lock (LONG volatile* lock)
{
retry:
	if (FastInterlockExchange (lock, 0) >= 0){
		unsigned int i = 0;
		while (*lock >= 0){
			if (++i & 7)
				__SwitchToThread (0);
			else{
				WaitLonger(i);
			}
		}
		goto retry;
	}
}

inline static void
leave_spin_lock(LONG volatile * lock)
{
	*lock = -1;
}


#ifdef _DEBUG

	inline static void
	enter_spin_lock(GCDebugSpinLock* pSpinLock)
	{
		enter_spin_lock(&pSpinLock->lock);
		pSpinLock->holding_thread = GetThread();
	}

	inline static void
	leave_spin_lock(GCDebugSpinLock* pSpinLock)
	{
		_ASSERTE(pSpinLock->holding_thread == GetThread());
		pSpinLock->holding_thread = (Thread*) -1;
		if (pSpinLock->lock != -1)
			leave_spin_lock(&pSpinLock->lock);
	}

	#define ASSERT_HOLDING_SPIN_LOCK(pSpinLock) \
		_ASSERTE((pSpinLock)->holding_thread == GetThread());

#else

	#define ASSERT_HOLDING_SPIN_LOCK(pSpinLock)

#endif

extern "C" GCHeap* g_pGCHeap;

#if !defined(_WIN64) && (DATA_ALIGNMENT > 4)
	#define ALIGNFIXUP 4
#else
	#define ALIGNFIXUP 0
#endif

class CFinalize{
	friend struct MEMBER_OFFSET_INFO(CFinalize);

	private:

	Object** m_Array;
	Object** m_FillPointers[NUMBERGENERATIONS+2];
	Object** m_EndArray;
	int m_PromotedCount;
	volatile LONG lock;

	BOOL GrowArray();

	void MoveItem(
		Object** fromIndex,
		unsigned int fromSeg,
		unsigned int toSeg);

	BOOL
	IsSegEmpty(unsigned int i)
	{
		ASSERT (i <= NUMBERGENERATIONS+1);
		return (i==0)
			? (m_FillPointers[0] == m_Array)
			: (m_FillPointers[i] == m_FillPointers[i-1]);
	}

	public:

	CFinalize ();
	~CFinalize();
	void EnterFinalizeLock();
	void LeaveFinalizeLock();

	void RegisterForFinalization(
		Object* obj);

	Object* GetNextFinalizableObject();

	BOOL ScanForFinalization(
		promote_func* fn,
		ScanContext *sc);

	void RelocateFinalizationData(
		promote_func* fn,
		ScanContext *sc);

	void GcScanRoots(
		promote_func* fn,
		ScanContext* sc);

	void UpdatePromotedGenerations(
		BOOL gen_0_empty_p);

	int GetPromotedCount();

	//Methods used by the shutdown code to call every finalizer
	void SetSegForShutDown(
		BOOL fHasLock);

	size_t GetNumberFinalizableObjects();

	/*
		Methods used by the app domain unloading call to
		finalize objects in an app domain
	*/
	BOOL FinalizeAppDomain(
		AppDomain* pDomain,
		BOOL fRunFinalizers);

	void CheckFinalizerObjects();
};

#ifdef GC_REQUIRES_HEADER
	#define plug_skew (sizeof(ObjHeader)+sizeof(GCHeader))
	#define runtime_skew (sizeof(GCHeader))
#else // !GC_REQUIRES_HEADER
	#define plug_skew sizeof(ObjHeader)
	#define runtime_skew 0
#endif //GC_REQUIRES_HEADER

// FIXME doc
#define min_obj_size\
	(sizeof(BYTE*)+plug_skew+sizeof(size_t))//syncblock + vtable+ first field

#define min_gap_size (min_obj_size + sizeof(size_t))

struct plug{
	BYTE * skew[sizeof(plug_skew) / sizeof(BYTE *)];
};

class Page;
class PageQueue;
struct Work;
class WorkList;

/*
	need to be careful to keep enough pad items to fit a
	relocation node padded to QuadWord before the plug_skew
*/
class heap_segment{
	public:

	BYTE* allocated;
	BYTE* committed;
	BYTE* reserved;
	BYTE* used;
	BYTE* mem;
	heap_segment* next;
	BYTE* plan_allocated;
	// Page* first_page;
	BYTE* padx;

	BYTE* pad0;
	#if ((SIZEOF_OBJHEADER+ALIGNFIXUP) % 8) != 0
		// Must pad to quad word
		BYTE pad1[8 - ((SIZEOF_OBJHEADER+ALIGNFIXUP) % 8)];
	#endif
	plug mPlug;
};

inline BYTE*&
heap_segment_reserved(heap_segment* inst)
{
	return inst->reserved;
}

inline BYTE*&
heap_segment_committed(heap_segment* inst)
{
	return inst->committed;
}

inline BYTE*&
heap_segment_used (heap_segment* inst)
{
	return inst->used;
}

inline BYTE*&
heap_segment_allocated (heap_segment* inst)
{
	return inst->allocated;
}

inline heap_segment*&
heap_segment_next (heap_segment* inst)
{
	return inst->next;
}

inline BYTE*&
heap_segment_mem (heap_segment* inst)
{
	return inst->mem;
}

inline BYTE*&
heap_segment_plan_allocated (heap_segment* inst)
{
	return inst->plan_allocated;
}

// This must be defined here so that it is accessible in jitinterfacex86.cpp.
typedef enum {
	NOT_IN_CYCLE = 0,
	WAITING_TO_START = 1,
	INITIALIZING = 2,
	INITIAL_ROOT_SCAN = 3,
	COPYING = 4,
	WAITING_TO_FINISH = 5,
	FINAL_ROOT_SCAN = 6,
	FINISHING = 7,
} GC_STAGE;

extern "C" GC_STAGE g_stage;

class gc_heap{

	friend class GCHeap;

	public:

	// Configuration option defaults:
	enum {
		// The constant "k" -- bytes collected / bytes allocated
		INCREMENTAL_CONSTANT = 3,

		// These are currently set to match Windows OS page size.
		MOSTLY_PAGE_SIZE = 4096,
		MOSTLY_LOG_PAGE_SIZE = 12,

		PAGE_FREE_SIZE = (MOSTLY_PAGE_SIZE - sizeof(plug) - min_obj_size),

		// Forces a full collection
		GC_FULL_COLLECTION = UINT_MAX,

		// Default values for residency thresholds
		PIN_RESIDENCY_THRESHOLD = (MOSTLY_PAGE_SIZE / 4) * 3,
		THREAD_RESIDENCY_THRESHOLD = (MOSTLY_PAGE_SIZE / 2),
	};

	/*
		Initializes the collector, allocating the underlying
		memory and initializing the member variables.
	*/
	static HRESULT Initialize(
		GCHeap* pGCHeap,
		size_t segment_size);

	static void Destroy();

	inline static BOOL
	IsInCycle()
	{
		return g_stage != NOT_IN_CYCLE;
	}

	inline static BOOL
	ReplicateWrites()
	{
		return g_stage > WAITING_TO_START;
	}

	static void StartCycle();

	// Ensures that the underlying memory of a page has been committed.
	static BOOL CommitPageMemory(
		Page* page);

	inline static BOOL
	GcSupportsVerifyHeap()
	{
		#if defined (VERIFY_HEAP)
			return TRUE;
		#else
			return FALSE;
		#endif
	}

	inline static BOOL
	GcSupportsFinalizers()
	{
		#if defined (GC_SUPPORTS_FINALIZERS)
			return TRUE;
		#else
			return FALSE;
		#endif
	}

	inline static BOOL
	GcSupportsLargeObjects()
	{
		#if defined (GC_SUPPORTS_LARGE_OBJECTS)
			return TRUE;
		#else
			return FALSE;
		#endif
	}

	inline static BOOL
	GcSupportsInteriorPointers()
	{
		#if defined (GC_SUPPORTS_INTERIOR_POINTERS)
			return TRUE;
		#else
			return FALSE;
		#endif
	}

	inline static BOOL
	GcSupportsWeakReferences()
	{
		#if defined (GC_SUPPORTS_WEAK_REFERENCES)
			return TRUE;
		#else
			return FALSE;
	#endif
	}

	inline static BOOL
	GcIsReplicatingCollector()
	{
		#if defined (GC_REPLICATING_COLLECTOR)
			return TRUE;
		#else
			return FALSE;
		#endif
	}

	inline static BOOL
	GcIsConcurrentCollector()
	{
		#if defined (GC_CONCURRENT_COLLECTOR)
			return TRUE;
		#else
			return FALSE;
		#endif
	}

	/*
		Reset the pointers in the given alloc_context,
		removing pages from the free page list as necessary.
		Returns TRUE if the allocation context contains enough
		room to allocate an object of size "min_size"; FALSE,
		otherwise.
	*/
	static BOOL ResetRContext(
		alloc_context& rcontext);

	static BOOL ResetContext(
		alloc_context& acontext,
		alloc_context& rcontext,
		size_t min_size);

	static Page* FindAdjacentPages(
		size_t min_size,
		size_t& actual_size);

	/*
		Must hold the g_shared_lock before calling this
		method!  Returns a pointer to the first free gap (if
		any).
	*/
	static void GarbageCollect(
		size_t bytesToCollect,
		Object*& next_gap,
		Object*& last_gap);

	// Returns whether or not a given object is a live heap object.
	static BOOL IsHeapObject(
		BYTE* obj);

	static Object* FindObject(
		BYTE* interior);

	static Page* GetPage(
		BYTE* obj);

	static CFinalize*
	get_finalize_queue()
	{
		return g_finalize_queue;
	}

	/*
		Threads must hold this lock when modifying shared data
		structures (i.e.  the free page list) or calling
		GarbageCollect.
	*/
	static GCSpinLock*
	GetSharedLock()
	{
		return &g_shared_lock;
	}

	static heap_segment*
	get_segment()
	{
		return segment;
	}

	static void ShadeRoot(
		Object** root);

	static void ShadeObject(
		Object* object);

	#if (defined(GC_REQUIRES_FULL_WRITEBARRIER) && defined(GC_CONCURRENT_COLLECTOR))
		typedef enum {
			GC_WRITE_8BIT_INT = 2,
			GC_WRITE_16BIT_INT = 4,
			GC_WRITE_32BIT_INT = 6,
			GC_WRITE_64BIT_INT = 8,
			GC_WRITE_PTR = 10,
			GC_WRITE_STRUCT = 1,
			GC_WRITE_MULTI = 12,
		} GC_WRITE_KIND;
	#endif

	static void RecordWrite8(
		Object* obj,
		__int8* dst);

	static void RecordWrite16(
		Object* obj,
		__int16* dst);

	static void RecordWrite32(
		Object* obj,
		__int32* dst);

	static void RecordWrite64(
		Object* obj,
		__int64* dst);

	static void RecordWritePtr(
		Object* obj,
		Object** dst);

	static void RecordWriteStruct(
		void* dst,
		MethodTable* pMT);

	static void RecordMultipleWrites(
		Object* obj);

	/*
		The current implementation requires that all pages be
		allocated in a contiguous block.  These two fields
		delimit this block.  Note that upper_page does NOT
		point to a valid page -- instead it points to
		(last_page + 1).
	*/
	static Page* lower_page;
	static Page* upper_page;

	/*
		Free pages used to allocate new mutator data, replica
		objects and copied data.
	*/
	static PageQueue g_free_queue;
	static PageQueue g_evacuated_queue;
	static PageQueue g_promoted_queue;
	static PageQueue g_replica_queue;

	private:

	static void SetStage(
		GC_STAGE new_stage);

	static void InitiateCollection();

	static void ScanStrongRoots(
		ScanContext& sc,
		promote_func* fn);

	static void ScanWeakRoots(
		ScanContext& sc);

	static size_t ThreadPage(
		Page* page,
		Object*& first_gap,
		Object*& prev_gap);

	// Returns the first free gap.
	static void FinishCollection(
		ScanContext& sc,
		Object*& first_gap,
		Object*& last_gap);

	static Object* FindObjectWithLock(
		BYTE* interior,
		size_t gap_end = 0);

	static size_t CopyObjects(
		size_t bytesToCollect);

	static BOOL SetupAllocContext(
		alloc_context* context,
		Object* gap,
		size_t size,
		Object* next_gap,
		Object* last_gap,
		Page* page,
		BOOL clear_memory,
		BOOL check_reserve);

	// FIXME: document page alignment?
	/*
		Extends the usable memory in a segment by committing
		more of the reserved memory, if possible.  TRUE on
		success.
	*/
	static BOOL GrowHeapSegment(
		heap_segment* segment,
		BYTE* to);

	/*
		Given an arbitrary object, promote it (if it is a
		heap-managed object) or ignore it (otherwise).  Return
		the new scan_limit.
	*/
	inline static BYTE*
	gc_heap::CheneyScanChild(
		size_t bytesToCollect,
		size_t& bytesCollected,
		alloc_context* scontexts,
		BYTE** gray_ptrs,
		BOOL scanning,
		Object*& scan_ptr,
		BYTE* scan_limit,
		WorkList& work_list,
		Object* obj,
		BOOL redirect_ptrs,
		Object** child,
		Object** next_ptr = NULL,
		Page** next_page = NULL);

	static void CheneyScan(
		size_t bytesToCollect,
		size_t& bytesCollected,
		alloc_context* scontexts,
		BYTE** gray_ptrs,
		Object* scan_ptr,
		BYTE*& scan_limit,
		WorkList& work_list);

	/*
		Ensure that the given page (and any continued pages)
		are live and in the proper queue.
	*/
	static void ProcessRange(
		Work& work,
		size_t bytesToCollect,
		size_t& bytesCollected,
		alloc_context* scontexts,
		BYTE** gray_ptrs,
		WorkList& work_list);

	static void ProcessWrites(
		Work& work,
		size_t bytesToCollect,
		size_t& bytesCollected,
		WorkList& work_list,
		PageQueue& dead_queue);

	static void ProcessObject(
		Object* obj,
		size_t bytesToCollect,
		size_t& bytesCollected,
		BYTE** gray_ptrs);

	static void RegisterRoot(
		Object** root,
		ScanContext* sc,
		DWORD flags);

	/*
		Update the object reference if the object is live and
		has been moved.  If the object is dead, then set the
		reference to NULL.
	*/
	static void Relocate(
		Object** root,
		ScanContext* sc,
		DWORD flags);

	static Page* ClearAdjacentPages(
		Page* current_page,
		BYTE* end_byte,
		int& page_count);

	// SPOONS: FIXME
	inline static BOOL HasSpace(
		Object* obj,
		alloc_context* scontext,
		size_t& size);

	inline static void ExtendContext(
		int copy_age,
		alloc_context* scontext,
		BYTE*& gray_ptr,
		BYTE*& scan_limit,
		BOOL reset_limit,
		WorkList& work_list,
		size_t& size);

	/*
		obj is the object to by copied.

		copy_age is the age that should be used if a new page
		is allocated for the copy.

		scontext is the allocation context into which the
		object should be copied, or if a new page is
		allocated, the context that should hold that page.

		scan_limit is the end of the current scan.  this may
		or may not be in one of the current allocation
		contexts.

		reset_limit says whether or not we have reached the
		end of the current scan.
	*/
	// SPOONS: inline?
	static Object* gc_heap::CopyObject(
		Object* obj,
		int copy_age,
		alloc_context* scontext,
		BYTE*& gray_ptr,
		BYTE*& scan_limit,
		BOOL reset_limit,
		WorkList& work_list,
		size_t& size);

	static Object* gc_heap::CopyObjectForMark(
		Object* obj,
		int copy_age,
		alloc_context* scontext,
		BYTE*& gray_ptr,
		WorkList& work_list,
		size_t& size);

	#if defined(GC_CONCURRENT_COLLECTOR)
		static void RecordWrite(
			Object* obj,
			BYTE* data,
			GC_WRITE_KIND kind);
	#endif

	#ifdef VERIFY_HEAP

		static void VerifyRoot(
			Object** root,
			ScanContext* sc,
			DWORD flags);

		static void VerifyPage(
			Page* current_page,
			BOOL verify_page_is_live);

		static void VerifyHeap();

		static void DumpRoot(
			Object** root,
			ScanContext* sc,
			DWORD flags);

		static void ClearRootFlags(
			Object** root,
			ScanContext* sc,
			DWORD flags);

		static void DumpHeap();

	#endif

	/*
		Currently there is only one segment; evenually this
		will become a set/list of segments.
	*/
	static heap_segment* segment;
	static Page* first_page;
	static BYTE* first_page_data;
	static int total_pages; // <-- Currently only used during verify

	/*
		"Write" pages hold sets of updates captured by the
		write barrier -- this is used only in the concurrent
		collector.
	*/
	static PageQueue g_write_queue;

	/*
		The next gap in the free list.  The gap may appear on:
		- a mutator page (while the mutator is running)
		- XXX FIXME (during collection)
		- XXX FIXME (in incremental collection)
	*/
	static Object* g_first_gap;

	/*
		FIXME spoons concurrent/parallel: this worklist and
		context should be per-thread
	*/
	static WorkList g_work_list;

	/*
		Should we use the residency of pages to determine when
		to promote in-place?
	*/
	static bool pin_using_residency;

	/*
		For mostly copying collection: the amount of live data
		in a page required for the page to be promoted in
		place (i.e.  pinned).
	*/
	static size_t pin_residency_threshold;

	/*
		For mostly copying collection with free object lists:
		how sparse should a page be before we consider
		threading the free space on that page.
	*/
	// FIXME how should this apply to "continued" pages.
	static size_t thread_residency_threshold;

	/*
		Should we use the age of objects to determine when to
		promote in-place?  If true, then be more conservative
		in pinning pages -- use an age-based heuristic to
		determine when to promote pages in-place.
	*/
	static int min_pinning_age;

	// the average residency of young pages.
	static size_t average_young_page_residency;

	/*
		This is sort of analogous to the number of
		generations: it's the number of discrete object ages
		that we will distinguish during copying.  If
		pin_using_age is not set, then 1 will always be
		enough.  Otherwise, if set to 1, the the collector
		will distinguish between objects allocated since the
		last collection cycle and all other objects; if set to
		2, then three classes of objects are defined: alloc'd
		since the last cycle, alloc'd just before the last
		cycle, and all other objects, etc.
	*/
	//#define GC_NUMBER_OF_CONTEXTS 2
	#define GC_NUMBER_OF_CONTEXTS 1

	static alloc_context g_scontexts[GC_NUMBER_OF_CONTEXTS];
	static Object* g_next_gap;

	static alloc_context g_wcontext;

	static CFinalize* g_finalize_queue;

	/*
		The shared_lock is used to synchronized data
		structures that are shared among more than one thread.
		It should be held before modifiying any shared state
		(i.e.  the free page list) or calling GarbageCollect.
	*/
	static GCSpinLock g_shared_lock;

	// How much space is free to be used for new allocations or new replicas?
	static size_t unreserved_size;

	// Approximately how much space is occupied by live objects?
	static size_t live_size;
};


#ifdef GC_REPLICATING_COLLECTOR
	// N.B. This macro works only in CObjHeader!!!
	#define GC_FORWARD_PTR (*(Object**)(((DWORD*)this) - 2))
#else
	#define GC_FORWARD_PTR (*(Object**)0)
#endif // STOP_THE_WORLD

class CObjectHeader : public Object{
	private:

	enum {
		BLACK = 0x2,
		GRAY = 0x1,
	};

	public:

	// Header Status Information

	MethodTable*
	GetMethodTable() const
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			return( (MethodTable *) ((size_t) m_pMethTab));
		}
		else {
			return( (MethodTable *) (((size_t) m_pMethTab) & (~(BLACK | GRAY))));
		}
	}

	/*
		The Stop-the-world collector only distinguishes
		between White and Gray.
	*/
	inline BOOL
	IsBlack()
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			return !!(((size_t) GC_FORWARD_PTR) & BLACK);
		}
		else {
			return !!(((size_t) m_pMethTab) & BLACK);
		}
	}

	inline void
	SetBlack()
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			GC_FORWARD_PTR = (Object*)(((size_t)GC_FORWARD_PTR & ~GRAY)
				| BLACK);
		}
		else {
			m_pMethTab = (MethodTable *) (((size_t) m_pMethTab & ~GRAY) | BLACK);
		}
	}

	inline BOOL
	IsGray()
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			return !!(((size_t) GC_FORWARD_PTR) & GRAY);
		}
		else {
			return !!(((size_t) m_pMethTab) & GRAY);
		}
	}

	inline void
	SetGray()
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			GC_FORWARD_PTR = (Object*)((size_t)GC_FORWARD_PTR | GRAY);
		}
		else {
			m_pMethTab = (MethodTable *) (((size_t) m_pMethTab) | GRAY);
		}
	}

	inline BOOL
	IsWhite() //FIXME const?
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			return !GC_FORWARD_PTR;
		}
		else {
			return !(((size_t)m_pMethTab) & (BLACK | GRAY));
		}
	}

	inline void
	SetWhite()
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			ASSERT(GC_FORWARD_PTR);
			GC_FORWARD_PTR = NULL;
		}
		else {
			SetMethodTable( GetMethodTable() );
		}
	}

	void SetFree(size_t size)
	{
		I1Array* pNewFreeObject;

		_ASSERTE( size >= sizeof(ArrayBase));
		ASSERT(size >= min_obj_size);
		assert (size == (size_t)(DWORD)size);
		if (gc_heap::GcIsReplicatingCollector()) {
			GC_FORWARD_PTR = NULL;
		}
		pNewFreeObject = (I1Array *) this;
		pNewFreeObject->SetMethodTable(g_pFreeObjectMethodTable);
		int base_size = g_pFreeObjectMethodTable->GetBaseSize();
		assert(g_pFreeObjectMethodTable->GetComponentSize() == 1);
		((ArrayBase *)pNewFreeObject)->m_NumComponents =
			(DWORD)(size - base_size);
		#ifdef _DEBUG
			((DWORD*) this)[-1] = 0;		// clear the sync block,
		#endif //_DEBUG
		if (gc_heap::GcSupportsVerifyHeap()) {
			assert ((DWORD)(((ArrayBase *)pNewFreeObject)->m_NumComponents)
				>= 0);
			if (g_pConfig->IsHeapVerifyEnabled())
				memset (((DWORD*)this)+2, 0xcc,
					((ArrayBase *)pNewFreeObject)->m_NumComponents);
		}
	}

	BOOL
	IsFree() const
	{
		return (GetMethodTable() == g_pFreeObjectMethodTable);
	}

	BOOL
	ContainsPointers() const
	{
		return GetMethodTable()->ContainsPointers();
	}

	Object*
	GetObjectBase() const
	{
		return (Object*) this;
	}

	// Must be called before SET_GRAY
	void
	SetRelocation(Object* newlocation)
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			ASSERT(!GC_FORWARD_PTR);
			GC_FORWARD_PTR = newlocation;
		}
		else {
			*(Object**)(((DWORD**)this)-1) = newlocation;
		}
	}

	Object*
	GetRelocated() const
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			ASSERT((size_t)GC_FORWARD_PTR & ~(BLACK | GRAY));
			ASSERT((size_t)GC_FORWARD_PTR != 0xCCCCCCCC);
			return (Object*)((size_t)GC_FORWARD_PTR & ~(BLACK | GRAY));
		}
		else {
			ASSERT(*(((DWORD**)this)-1));
			return (Object*)*(((DWORD**)this)-1);
		}
	}

	BOOL
	IsRelocated()
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			ASSERT((size_t)GC_FORWARD_PTR != 0xCCCCCCCC);
			return (BOOL)((size_t)GC_FORWARD_PTR & ~(BLACK | GRAY));
		}
		else {
			return GetRelocated() != NULL;
		}
	}

	inline DWORD
	GetSize()
	{
		MethodTable* mT = GetMethodTable();
		return (mT->GetBaseSize()
			+ (mT->GetComponentSize()
				? (((Object*)this)->GetNumComponents() * mT->GetComponentSize())
				:0));
	}

	inline void
	ClearGCHeader()
	{
		#ifdef GC_REQUIRES_HEADER
			/*
				FIXME spoons should be parameterized
				over the size of the header.  For now
				we assume that if the header exists,
				then it is 4 bytes long.
			*/
			ASSERT(sizeof(GCHeader) == 4);
			GC_FORWARD_PTR = NULL;
		#endif //GC_REQUIRES_HEADER
	}

	// FIXME can we store the next gap in the sync block index?
	inline void
	SetNextGap(Object* next)
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			*(Object**)(((DWORD**)this)-1) = next;
		}
		else {
			*(((Object**)this)+2) = next;
		}
	}

	inline Object*
	GetNextGap()
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			return *(Object**)(((DWORD**)this)-1);
		}
		else {
			return *(((Object**)this)+2);
		}
	}

	inline void
	ClearNextGap()
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			*(Object**)(((DWORD**)this)-1) = NULL;
		}
		else {
			*(((Object**)this)+2) = NULL;
		}
	}
};

#undef GC_FORWARD_PTR

#define header(i) ((CObjectHeader*)(i))

#endif // _GCSMPPRIV_H_
