// ==++==
//
//
//		Copyright (c) 2002 Microsoft Corporation.	 All rights reserved.
//
//		The use and distribution terms for this software are contained in the file
//		named license.txt, which can be found in the root of this distribution.
//		By using this software in any fashion, you are agreeing to be bound by the
//		terms of this license.
//
//		You must not remove this notice, or any other, from this software.
//
//
// ==--==
// gc.cpp
/*
	Automatic memory manager.
*/

#include "common.h"
#include "object.inl"
#include "gcsmppriv.h"
#include "gcdata.h"

#ifdef GC_PROFILING
	#include "profilepriv.h"	// ../inc/profilepriv.h
#endif

/*
	Alignment primitives
*/

// Alignment constant for allocation
#define ALIGNCONST (DATA_ALIGNMENT-1)

inline size_t
Align(size_t nbytes)
{
	return (nbytes + ALIGNCONST) & ~ALIGNCONST;
}

inline BOOL
Aligned(size_t n)
{
	return (n & ALIGNCONST) == 0;
}

inline size_t
align_on_page(size_t add)
{
	return ((add + OS_PAGE_SIZE - 1) & - (OS_PAGE_SIZE));
}

inline BYTE*
align_on_page (BYTE* add)
{
	return (BYTE*)align_on_page ((size_t) add);
}

inline BOOL
power_of_two_p(size_t integer)
{
	return !(integer & (integer-1));
}


/*
	TIME is an integral type.  The difference of two times fits in a
	TIME but not necessarily a long.  We have arrows:

		GetTime : 1 -> TIME
		TimeToSeconds : TIME -> double
		+, - : TIME * TIME -> TIME

	FIXME: We probably want ReportGcTime to to be folded into the
	other printing flags.
*/

#ifdef MEASURE_GC_TIME
	enum { ReportGcTime = 1 };

	typedef __int64 TIME;

	static inline TIME
	GetTime()
	{
		LARGE_INTEGER time;
		QueryPerformanceCounter(&time);
		return time.QuadPart;
	}

	static double
	TimeToSeconds(TIME t)
	{
		static TIME freq;
		static double recip;
		static BOOL init = FALSE;
		if(!init){
			init = TRUE;
			LARGE_INTEGER f;
			QueryPerformanceFrequency(&f);
			freq = f.QuadPart;
			recip = 1/((double)freq);
		}
		// Doing it this way because ((double)t) might overflow.
		TIME q = t/freq;
		TIME r = t%freq;
		return q+r*recip;
	}

#else /* ! MEASURE_GC_TIME */
	enum { ReportGcTime = 0 };

	typedef unsigned TIME;

	static inline TIME
	GetTime()
	{
		return 0;
	}

	static double
	TimeToSeconds(TIME t)
	{
		return 0.0;
	}

#endif	/* ReportGcTime, TIME, GetTime, and TimeToSeconds */

class TIMER{
	public:

	TIMER();

	void reset();

	/*
		Elapsed time since constructor or reset.
	*/
	TIME checkTime();

	private:

	TIME start;
};

TIMER::TIMER()
{
	start = GetTime();
}

TIME
TIMER::checkTime()
{
	return GetTime() - start;
}

void
TIMER::reset()
{
	start = GetTime();
}

static double
checkTimeSeconds(TIMER& timer)
{
	return TimeToSeconds(timer.checkTime());
}

static TIMER mutator_timer;
static double mutator_time; /* Time between collections. */
static double gc_time; /* Time in collections. */

/* By analogy to "TIMER"... */
class SIZER{
	public:

	SIZER();

	void addSize(
		size_t s);

	void addSize(
		Object* obj);

	size_t checkSize();

	void reset();

	private:

	size_t size;
};

SIZER::SIZER()
{
	size = 0;
}

#ifdef MEASURE_GC_SPACE
	enum { ReportGcSpace = 1 };

	void
	SIZER::addSize(size_t s)
	{
		size += s;
	}
	
	void
	SIZER::addSize(Object* obj)
	{
		size += Align(header(obj)->GetSize());
	}
	
	size_t
	SIZER::checkSize()
	{
		return size;
	}
	
	void
	SIZER::reset()
	{
		size = 0;
	}

#else /* ! MEASURE_GC_SPACE */
	enum { ReportGcSpace = 0 };

	void
	SIZER::addSize(size_t s)
	{
	}
	
	void
	SIZER::addSize(Object* obj)
	{
	}
	
	size_t
	SIZER::checkSize()
	{
		return 0;
	}
	
	void
	SIZER::reset()
	{
	}

#endif	/* ReportGcSpace */

/*
	These sizes tell how much space was used for allocation and
	for replicated objects.
*/
static SIZER allocated_used_size;
static SIZER allocated_total_size;
static SIZER replicated_used_size;
static SIZER replicated_total_size;

/*
	Scanned sizes give some idea about where time is spent during
	scanning.
*/
static SIZER black_scanned_size;
static SIZER cheney_scanned_size;
static SIZER queue_scanned_size;
static SIZER white_scanned_size;

/*
	These sizes indicate how much data was pinned or copied in
	this cycle (*live*) and how much space was used (*total*) for
	pinned or copied data (i.e.  including fragmentation).
*/
static SIZER pinned_live_size;
static SIZER pinned_total_size;
static SIZER copied_live_size;
static SIZER copied_total_size;
static SIZER write_total_size;

/*
	These sizes give upper bounds on the current amount of live
	data that will be pinned or copied in the next cycle -- they
	are always used (even if we are not collecting statistics
	about the collector).
*/
static size_t promoted_live_size;
static size_t evacuated_live_size;
static size_t promoted_threaded_size;
static size_t evacuated_threaded_size;

/*
	This is the amount of space that was reclaimed, i.e.  pages
	from which data was copied.  This is generally pages that were
	freshly alloc'd or older pages with no live data.
*/
static SIZER reclaimed_total_space;

/*
	This space was (conservatively) reserved for either replicated
	objects or copied objects, but in the end, unused.
*/
static SIZER unused_total_space;

int gc_count;

extern "C" int successful_alloc = 0;
extern "C" int unsuccessful_alloc = 0;

#ifdef TRACE_GC

	int print_level = DEFAULT_GC_PRN_LVL;	 //level of detail of the debug trace
	BOOL trace_gc;

#endif //TRACE_GC


// Data associated with the Collector
heap_segment* gc_heap::segment;
Page* gc_heap::first_page;
BYTE* gc_heap::first_page_data;
Page* gc_heap::upper_page;
Page* gc_heap::lower_page;
int gc_heap::total_pages;
bool gc_heap::pin_using_residency;
size_t gc_heap::pin_residency_threshold;
size_t gc_heap::thread_residency_threshold;
int gc_heap::min_pinning_age;
size_t gc_heap::average_young_page_residency;
PageQueue gc_heap::g_free_queue;
PageQueue gc_heap::g_evacuated_queue;
PageQueue gc_heap::g_promoted_queue;
PageQueue gc_heap::g_replica_queue;

Object* gc_heap::g_first_gap;

CFinalize* gc_heap::g_finalize_queue;

WorkList gc_heap::g_work_list;
alloc_context gc_heap::g_scontexts[GC_NUMBER_OF_CONTEXTS];
Object* gc_heap::g_next_gap = NULL;
alloc_context gc_heap::g_wcontext;

GCSpinLock gc_heap::g_shared_lock = SPIN_LOCK_INITIALIZER;

PageQueue gc_heap::g_write_queue;

 
size_t gc_heap::unreserved_size;
size_t gc_heap::live_size;

static inline void
addSize(size_t* size, size_t delta)
{
	ASSERT((int)delta >= 0);
	size_t before = *size;
	size_t after = before + delta;
	ASSERT(after>=before);
	*size = after;
}

static inline void
subSize(size_t* size, size_t delta)
{
	ASSERT((int)delta >= 0);
	size_t before = *size;
	size_t after = before - delta;
	ASSERT(after<=before);
	*size = after;
}

size_t Page::g_gap_end;

#undef COMPlusThrowOM
#define COMPlusThrowOM() exit(1)

static HRESULT AllocateCFinalize(
	CFinalize** pCFinalize);

/* per heap static initialization */

// GLOBAL

volatile LONG m_AllocLock = -1;
volatile LONG m_WriteLock = -1;
extern "C" {
	alloc_context g_acontext;
	GC_STAGE g_stage = NOT_IN_CYCLE;
}

// New objects are allocated in both from- and to-space.
alloc_context g_rcontext;

/* end of per heap static initialization */

/* static initialization */

// XXX globals
BOOL g_bGCHeapInit = FALSE;
GCHeap* g_pGCHeap = (GCHeap *)NULL;

/* end of static initialization */

#define method_table(obj) ((CObjectHeader*)(obj))->GetMethodTable()

#define TracePointers(mt, obj, size, param, exp) \
	{	 \
		CGCDesc* map = CGCDesc::GetCGCDescFromMT((MethodTable*)(mt)); \
		CGCDescSeries* cur = map->GetHighestSeries(); \
		CGCDescSeries* last = map->GetLowestSeries(); \
		BYTE* o = (BYTE*)obj; \
		\
		if (cur >= last) { \
			do { \
				Object** param = (Object**)((o) + cur->GetSeriesOffset()); \
				Object** ppstop = \
				(Object **)((BYTE*)param + cur->GetSeriesSize() + (size)); \
				while (param < ppstop) { \
					{exp} \
					param++; \
				} \
				cur--; \
			} while (cur >= last); \
		} \
		else {	 \
			SSIZE_T cnt = (SSIZE_T)map->GetNumSeries(); \
			Object** param = (Object**)((o) + cur->startoffset); \
			while ((BYTE*)param < ((o)+(size)-plug_skew)) {	 \
				for (SSIZE_T __i = 0; __i > cnt; __i--) {	 \
					HALF_SIZE_T skip = cur->val_serie[__i].skip; \
					HALF_SIZE_T nptrs = cur->val_serie[__i].nptrs; \
					Object** ppstop = param + nptrs; \
					do { \
						{exp} \
						param++; \
					} while (param < ppstop); \
					param = (Object**)((BYTE*)param + skip); \
				} \
			} \
		} \
	}

#define TracePointers2(mt, obj, size, param, exp1, exp2) \
	{ \
		CGCDesc* map = CGCDesc::GetCGCDescFromMT((MethodTable*)(mt)); \
		CGCDescSeries* cur = map->GetHighestSeries(); \
		CGCDescSeries* last = map->GetLowestSeries(); \
		BYTE* o = (BYTE*)obj; \
	 \
		if (cur >= last){ \
			Object** param = (Object**)((o) + cur->GetSeriesOffset()); \
			Object** ppstop = \
				(Object **)((BYTE*)param + cur->GetSeriesSize() + (size)); \
			if (param < ppstop){ \
				{exp1} \
				param++; \
			} \
			while (param < ppstop){ \
				{exp2} \
				param++; \
			} \
			cur--; \
	 \
			while (cur >= last){ \
				Object** param = (Object**)((o) + cur->GetSeriesOffset()); \
				Object** ppstop = \
					(Object **)((BYTE*)param + cur->GetSeriesSize() + (size)); \
				while (param < ppstop){ \
					{exp2} \
					param++; \
				} \
				cur--; \
			} \
		} \
		else{ \
			SSIZE_T cnt = (SSIZE_T)map->GetNumSeries(); \
			Object** param = (Object**)((o) + cur->startoffset); \
			if ((BYTE*)param < ((o)+(size)-plug_skew)){ \
				for (SSIZE_T __i = 0; __i > cnt; __i--){ \
					HALF_SIZE_T skip =	cur->val_serie[__i].skip; \
					HALF_SIZE_T nptrs = cur->val_serie[__i].nptrs; \
					Object** ppstop = param + nptrs; \
					{exp1} \
					param++; \
					while (param < ppstop){ \
						{exp2} \
						param++; \
					} \
					param = (Object**)((BYTE*)param + skip); \
				} \
				while ((BYTE*)param < ((o)+(size)-plug_skew)){ \
					for (SSIZE_T __i = 0; __i > cnt; __i--){ \
						HALF_SIZE_T skip =	cur->val_serie[__i].skip; \
						HALF_SIZE_T nptrs = cur->val_serie[__i].nptrs; \
						Object** ppstop = param + nptrs; \
						do{ \
							{exp2} \
							param++; \
						} while (param < ppstop); \
						param = (Object**)((BYTE*)param + skip); \
					} \
				} \
			} \
		} \
	}

#define LoopOverContexts(var,exp) \
{ \
	for (int var = 0; var < GC_NUMBER_OF_CONTEXTS; var++) { \
		exp; \
	} \
}

BOOL
gc_heap::SetupAllocContext(
	alloc_context* context, 
	Object* gap,
	size_t size,
	Object* next_gap,
	Object* last_gap,
	Page* page,
	BOOL clear_memory,
	BOOL check_reserve)
{
	BYTE* current_byte;
	if (gap == NULL) {
		if (!gc_heap::CommitPageMemory(page)) {
			return FALSE;
		}
		current_byte = page->GetData();
		//next_gap = NULL;
		ASSERT(size == MOSTLY_PAGE_SIZE);
	}
	else {
		current_byte = (BYTE*)gap;
	}

	size_t new_gap_size = 0;
	Object* new_gap = NULL;

	if (check_reserve) {
		size_t reserve_factor;

		if (page->HasStatus(Page::STATUS_FREE)) {
			if (Page::AGE_YOUNG >= min_pinning_age 
			&& average_young_page_residency > pin_residency_threshold) {
				// Objects will be promoted in-place.
				reserve_factor = 1;
			}
			else {
				// Objects will be evacuated.
				reserve_factor = 2;
			}
		}
		else {
			if (page->HasStatus(Page::STATUS_PROMOTED)) {
				reserve_factor = 1;
			}
			else {
				ASSERT(page->HasStatus(Page::STATUS_EVACUATED));
				reserve_factor = 2;
			}
		}

		if (size > (unreserved_size / reserve_factor)) {
			// Gap must be divided.
			new_gap_size = (size - Align(unreserved_size / reserve_factor));
			if (new_gap_size < min_gap_size) {
				/*
					If we divide the gap then the
					remaining part must be large
					enough to be a gap:
				*/
				new_gap_size = min_gap_size;
			}

			size = size - new_gap_size;
			unreserved_size = 0;
			live_size += size;

			if (size < min_gap_size) {
				// The remaining gap is too small to be divided.
				return FALSE;
			}

			new_gap = (Object*)(current_byte + size);
			header(new_gap)->SetFree(new_gap_size);
		}
		else {
			subSize(&unreserved_size, size * reserve_factor);
			live_size += size;
		}
	}

	ASSERT(Align(size));

	context->alloc_ptr = context->alloc_start = current_byte;
	context->alloc_limit = current_byte + size - min_obj_size;
	context->alloc_end = current_byte + size;
	
	/*
		We MUST set the end of the gap before we set the
		active flag and BEFORE we clear out the gap!
	*/
	/* from SetupAllocContextFromPage
		XXX if this case were analogous (to the FromGap case),
		we would set the GapEnd here.  But!  This case is used
		for replica contexts and for scan (copying) contexts:
		these should not have the active bit (nor the GapEnd)
		set.
	*/
	// XXX This is an unfortunate overloading of this flag!!!
	if (check_reserve) {

		// FIXME REPLICATING
		page->SetGapEnd(context->alloc_end - context->alloc_ptr);
		/*
			size_t gap = page->AcquireGapLock();
			memclr(g_acontext.alloc_start - plug_skew, Align(size));
			page->ReleaseGapLock(gap);
		*/

		if (page->HasStatus(Page::STATUS_FREE)) {
			g_free_queue.RemovePage(page);
			page->SetAge(Page::AGE_YOUNG);

			if (!ReplicateWrites()
			&& unreserved_size == 0
			/* && new_gap_size > pin_residency_threshold */) {
				page->SetStatus(Page::STATUS_PROMOTED);
				g_promoted_queue.PushPage(page);
				dprintf(4, ("page" FMT_ADDR "promoted from free list",
					DBG_ADDR(page)));

				if (new_gap != NULL) {
					header(new_gap)->ClearNextGap();
					if (last_gap != NULL) {
						header(last_gap)->SetNextGap(new_gap);
					}
					last_gap = new_gap;
				}
			}
			else if (!ReplicateWrites()
			&& Page::AGE_YOUNG >= min_pinning_age 
			&& average_young_page_residency > pin_residency_threshold) {
				// Objects will be promoted in-place.
				page->SetStatus(Page::STATUS_PROMOTED);
				g_promoted_queue.PushPage(page);
				dprintf(4, ("page" FMT_ADDR "promoted by young residency",
					DBG_ADDR(page)));

				if (new_gap != NULL) {
					header(new_gap)->ClearNextGap();
					if (last_gap != NULL) {
						header(last_gap)->SetNextGap(new_gap);
					}
					last_gap = new_gap;
				}
			}
			else {
				// Objects will be evacuated.
				page->SetStatus(Page::STATUS_EVACUATED);
				g_evacuated_queue.PushPage(page);
				dprintf(4, ("page" FMT_ADDR "evacuated by alloc"
					" with %u live and %u unreserved",
					DBG_ADDR(page), live_size, unreserved_size));

				if (new_gap != NULL) {
					header(new_gap)->SetNextGap(next_gap);
					next_gap = new_gap;
				}
			}
		}
	}

	context->alloc_next = (BYTE*)next_gap;
	context->alloc_last = (BYTE*)last_gap;
	
	if (clear_memory) {
		memclr(current_byte - plug_skew, Align(size));
	}
	return TRUE;
}

inline static void
ClearAllocContext(alloc_context* context, BOOL reset_bytes)
{
	// Fill in the rest of the context with a "free" object.
	if (context->alloc_end > context->alloc_ptr){
		header(context->alloc_ptr)->SetFree(
			context->alloc_end - context->alloc_ptr);
	}
	else {
		ASSERT(context->alloc_ptr == context->alloc_end);
	}
	
	context->alloc_start = context->alloc_ptr = 
		context->alloc_limit = context->alloc_end = NULL;

	ASSERT(context->alloc_next == NULL);
}

HRESULT
gc_heap::Initialize(GCHeap* pGCHeap, size_t segment_size)
{
	HRESULT hres = S_OK;

	gc_count = 0;

	successful_alloc = 0;
	unsuccessful_alloc = 0;

	#ifdef TRACE_GC
		if (gc_count >= g_pConfig->GetGCtraceStart())
			trace_gc = 1;
		if (gc_count >=	 g_pConfig->GetGCtraceEnd())
			trace_gc = 0;
	#endif // TRACE_GC

	size_t reserved = 0;

	size_t limit = CNameSpace::AskForMoreReservedMemory(0, segment_size);
	if ((reserved + segment_size) > limit) {
		return E_OUTOFMEMORY;
	}
	reserved += segment_size;
	
	BYTE* start = (BYTE*)VirtualAlloc (0, segment_size, 
		MEM_RESERVE, PAGE_READWRITE);
	if (!start) {
		return E_OUTOFMEMORY;
	}

	/*
		XXX This could be more precise (it doesn't account for
		the overhead associated with the headers themselves).
	*/
	size_t initial_commit = align_on_page(
		((segment_size / MOSTLY_PAGE_SIZE) * sizeof(Page))
		+ Align(sizeof(heap_segment)));

	if (!VirtualAlloc(start, initial_commit, MEM_COMMIT, PAGE_READWRITE)) {
		return E_OUTOFMEMORY;
	}

	segment = (heap_segment*)start;
	heap_segment_mem (segment) = start
		+ Align (sizeof (heap_segment) + ALIGNFIXUP) - ALIGNFIXUP;
	heap_segment_reserved (segment) = start + reserved;
	heap_segment_committed (segment) = start + initial_commit;
	heap_segment_next (segment) = 0;
	heap_segment_plan_allocated (segment) = heap_segment_mem (segment);
	heap_segment_allocated (segment) = heap_segment_mem (segment);
	heap_segment_used (segment) = heap_segment_allocated (segment);

	/*
		Divide the segment into pages, skipping the first page
		(which is used for segment/page book keeping).
	*/
	// XXX Again, this is imprecise.  And ugly.
	BYTE* ptr = start
		+ ((initial_commit / MOSTLY_PAGE_SIZE + 1) * MOSTLY_PAGE_SIZE);
	Page* current_page = (Page*)(start
		+ Align(sizeof(heap_segment))
		- sizeof(Page));
	first_page = (Page*)((BYTE*)current_page + sizeof(Page));

	total_pages = 0;

	/*
		Add half the pages to the free list.  This is the
		unused part of the "from-space".
	*/
	g_free_queue.Clear();
	lower_page = current_page + 1;
	for ( ; ptr < start + segment_size; ptr += MOSTLY_PAGE_SIZE){
		current_page = (Page*)((BYTE*)current_page + sizeof(Page));
		BYTE* next_page = (BYTE*)current_page + sizeof(Page);
		if (next_page > heap_segment_committed(segment)){
			if (!GrowHeapSegment(segment,next_page)) {
				return E_OUTOFMEMORY;
			}
		}

		current_page->SetData(ptr);

		current_page->SetStatus(Page::STATUS_FREE);
		g_free_queue.PushPage(current_page);
		total_pages++;
	}

	upper_page = current_page + 1;

	first_page_data = first_page->GetData();

	// Nothing to put in the mutator list, yet.
	g_evacuated_queue.Clear();
	g_promoted_queue.Clear();
	g_replica_queue.Clear();

	pin_using_residency = g_pConfig->IsPinningUsingDensityEnabled();
	// Set the residency thresholds
	pin_residency_threshold = g_pConfig->GetPinDensityThreshold();
	thread_residency_threshold = g_pConfig->GetThreadDensityThreshold();
	min_pinning_age = g_pConfig->GetMinimumPinningAge();
	// This will ensure that MS collectors behave properly starting from the
	// first collection.  (Any positive value is sufficient.)
	average_young_page_residency = 1;

	#ifdef TRACE_GC
		print_level = g_pConfig->GetGCprnLvl();
	#endif

	unreserved_size = (size_t)(g_free_queue.GetLength() * MOSTLY_PAGE_SIZE);

	/*
		if (GcIsReplicatingCollector()) {
			cycle_threshold = (size_t)((INCREMENTAL_CONSTANT * cycle_threshold) 
				/ (INCREMENTAL_CONSTANT + 1.0));
		}
	*/

	dprintf(2, ("%d pages in free list", g_free_queue.GetLength()));
	dprintf(2, ("unreserved size is %d\n",  unreserved_size));

	HRESULT hr = AllocateCFinalize(&g_finalize_queue);

	if (FAILED(hr))
		return hr;

	g_work_list.Init();

	LoopOverContexts(i, ClearAllocContext(&g_scontexts[i], TRUE));

	allocated_used_size.reset();
	allocated_total_size.reset();
	replicated_used_size.reset();
	replicated_total_size.reset();

	mutator_timer.reset();
	return hres;
}

void
gc_heap::Destroy()
{
	dprintf(2, ("Destroying segment [%p, %p[",
		segment, heap_segment_reserved(segment)));

	VirtualFree(segment,
		heap_segment_committed(segment) - (BYTE*)segment, 
		MEM_DECOMMIT);
	VirtualFree(segment,
		heap_segment_reserved(segment) - (BYTE*)segment, 
		MEM_RELEASE);

	if (g_finalize_queue)
		delete g_finalize_queue;
}

static char*
stageName(GC_STAGE stage)
{

	switch(stage){
	#define CASE(n) case n: return #n;
	CASE(NOT_IN_CYCLE);
	CASE(WAITING_TO_START);
	CASE(INITIALIZING);
	CASE(INITIAL_ROOT_SCAN);
	CASE(COPYING);
	CASE(WAITING_TO_FINISH);
	CASE(FINAL_ROOT_SCAN);
	CASE(FINISHING);
	#undef CASE
	default:	return "bogus stage";
	}
}

void
gc_heap::SetStage(GC_STAGE new_stage)
{
	dprintf(1,("Moving to stage %s...", stageName(new_stage)));
	g_stage = new_stage;
}

/*
	In the concurrent version, the Enable/DisablePreemptiveGC is
	optional because the gc thread call WaitLonger.
*/
void
WaitLonger(int i)
{
	// every 8th attempt:
	Thread* pCurThread = GetThread();
	BOOL bToggleGC;

	{
		bToggleGC = pCurThread->PreemptiveGCDisabled();
		if (bToggleGC)
			pCurThread->EnablePreemptiveGC();
	}

	if (g_SystemInfo.dwNumberOfProcessors > 1){
		pause(); // indicate to the processor that we are spining
		if (i & 0x01f)
			__SwitchToThread (0);
		else
			__SwitchToThread (5);
	}
	else
		__SwitchToThread (5);

	{
		if (bToggleGC)
			pCurThread->DisablePreemptiveGC();
	}
}

// Grow by committing more pages (request bytes)
BOOL
gc_heap::GrowHeapSegment(heap_segment* segment, BYTE* pointer)
{
	BYTE* reserved = heap_segment_reserved(segment);
	BYTE* committed = heap_segment_committed(segment);
	BYTE* aligned = (BYTE*)align_on_page((size_t)pointer);
	size_t request = aligned - committed;

	assert(committed <= reserved);
	assert((size_t)reserved == align_on_page((size_t)reserved));
	assert((size_t)committed == align_on_page((size_t)committed));
	assert (request == align_on_page(request));

	size_t c_size = max(request, 16*OS_PAGE_SIZE);
	
	c_size = min(c_size, (size_t)(reserved - committed));
	if (c_size == 0)
		return FALSE;
	assert (c_size >= request);
	WS_PERF_SET_HEAP(GC_HEAP);
	if (!VirtualAlloc(committed, c_size, MEM_COMMIT, PAGE_READWRITE)){
		return FALSE;
	}
	WS_PERF_UPDATE(
		"GC:gc_heap:grow_heap_segment",
		c_size, heap_segment_committed(segment));

	heap_segment_committed(segment) += c_size;
	return TRUE;
}

BOOL
gc_heap::CommitPageMemory(Page* page)
{
	ASSERT(page);

	/*
		Make sure that we've committed the underlying virtual
		memory for this page.
	*/
	BYTE* pageend = page->GetData() + MOSTLY_PAGE_SIZE;
	if(pageend > heap_segment_committed(segment)){
		if (!GrowHeapSegment(segment, pageend)) {
			return FALSE;
		}
	}
	return TRUE;
}

BOOL
gc_heap::ResetRContext(alloc_context& rcontext)
{
	enter_spin_lock(&g_shared_lock);

	Page* replica_page = NULL;
	if (g_free_queue.IsEmpty()) {
		leave_spin_lock(&g_shared_lock);
		return FALSE;
	}
	replica_page = g_free_queue.PopPage();
	
	// XXX Will we ever uses free gaps for replica contexts?
	if (!SetupAllocContext(
		&rcontext, NULL, MOSTLY_PAGE_SIZE, NULL, NULL,
		replica_page, /* XXX ? */ FALSE, FALSE)
	) {
		leave_spin_lock(&g_shared_lock);
		return FALSE;
	}

	replicated_total_size.addSize(MOSTLY_PAGE_SIZE);
	replica_page->SetAge(Page::AGE_YOUNG);
	replica_page->SetStatus(Page::STATUS_REPLICA);
	g_replica_queue.PushPage(replica_page);
	
	leave_spin_lock(&g_shared_lock);
	return TRUE;
}

BOOL 
gc_heap::ResetContext(
	alloc_context& acontext,
	alloc_context& rcontext,
	size_t min_size)
{
	ASSERT(min_size <= PAGE_FREE_SIZE);

	enter_spin_lock(&g_shared_lock);

	Object* next_gap = NULL;
	Object* last_gap = NULL;

	// Reset the context, filling in its unused space.
	if (acontext.alloc_start != NULL) {

		if (GcIsReplicatingCollector()) {
			Page* page = GetPage((BYTE*)acontext.alloc_start);
			page->ClearActive();
			page->ClearGapEnd();
		}

		next_gap = (Object*)acontext.alloc_next;
		last_gap = (Object*)acontext.alloc_last;
		acontext.alloc_next = NULL;

		allocated_used_size.addSize(acontext.alloc_ptr - acontext.alloc_start);
		// No need to reserve space for replicas of the unused part of the context:
		if (GetPage(acontext.alloc_start)->HasStatus(Page::STATUS_EVACUATED)) {
			addSize(&unreserved_size, acontext.alloc_end - acontext.alloc_ptr);
		}
		// Also the end of the page is not live.
		live_size -= acontext.alloc_end - acontext.alloc_ptr;
		ClearAllocContext(&acontext, FALSE);
	}

	// This will only be true for incremental collectors.
	if (IsInCycle()) {
		if (rcontext.alloc_start != NULL) {
			Page* page = GetPage((BYTE*)rcontext.alloc_start);
			page->SetResidency(rcontext.alloc_ptr - rcontext.alloc_start);
			ClearAllocContext(&rcontext, FALSE);
		}

		// Do some work!
		// FIXME new_gap?!?
		// FIXME promote the pages of any remaining gaps
		Object* new_gap = next_gap;

		/*
			Throw away any gaps that will be evacuated --
			these are of no use to the collector.
		*/
		
		/*
			XXX Ideally we would keep the to-be-evacuated
			gaps separate from the to-be-promoted gaps
			(using the former only for new allocation.)
		*/
		while (new_gap != NULL
		&& gc_heap::GetPage((BYTE*)new_gap)->HasStatus(Page::STATUS_EVACUATED)) {
			new_gap = header(new_gap)->GetNextGap();
		}
		
		gc_heap::GarbageCollect(
			gc_heap::MOSTLY_PAGE_SIZE * gc_heap::INCREMENTAL_CONSTANT,
			new_gap, last_gap);

		if (new_gap != NULL) {
			next_gap = new_gap;
		}
	}

	BOOL ran_gc = FALSE;

retry:

	if (ran_gc) {
		leave_spin_lock(&g_shared_lock);
		return FALSE;
	}

	// Check to see if we need to start a new collection cycle.
	if (GcIsReplicatingCollector()
	&& !IsInCycle()
	&& ((unreserved_size / 2) * INCREMENTAL_CONSTANT) <= live_size) {
		StartCycle();
	}

	// Check to see if we have enough space to continue allocation.
	// FIXME is this the correct test for a replicating collector?
	// XXX This "2 *" is conservative if we are acting more mark-sweep-like.
	if (unreserved_size < (2 * min_size)){
		/*
			if (GcIsReplicatingCollector()) {
				if (!IsInCycle()) {
					StartCycle();
				}
			}
			else
		*/
		{
			// XXX is this ClearAllocContext necessary?
			//ClearAllocContext(&acontext, FALSE);
			if (!IsInCycle()) {
				StartCycle();
			}
		
			/*
				Throw away any gaps that will be
				evacuated -- these are of no use to
				the collector.
			*/
			while (next_gap != NULL
			&& GetPage((BYTE*)next_gap)->HasStatus(Page::STATUS_EVACUATED)) {
				next_gap = header(next_gap)->GetNextGap();
			}

			gc_heap::GarbageCollect(
				gc_heap::GC_FULL_COLLECTION,
				next_gap, last_gap);

			if (unreserved_size < (2 * min_size)) {
				leave_spin_lock(&g_shared_lock);
				return FALSE;
			}
			ran_gc = TRUE;
		}
	}

	// Go find more space for allocation...
	Page* current_page;

	Object* first_gap = next_gap;
	Object* prev_gap = NULL;

	while(next_gap != NULL && header(next_gap)->GetSize() < min_size) {
		prev_gap = next_gap;
		next_gap = header(next_gap)->GetNextGap();
		#ifdef GC_COUNT_ALLOC_FITS
				unsuccessful_alloc++;
		#endif 
	}

	if (next_gap != NULL) {
		if (prev_gap != NULL) {
			header(prev_gap)->SetNextGap(header(next_gap)->GetNextGap());
			header(next_gap)->SetNextGap(first_gap);
		}
		if (next_gap == last_gap) {
			last_gap = prev_gap;
		}

		current_page = GetPage((BYTE*)next_gap);

		if (!SetupAllocContext(&acontext, next_gap, 
			header(next_gap)->GetSize(),
			header(next_gap)->GetNextGap(),
			last_gap,
			current_page, TRUE, TRUE)
		) {
			leave_spin_lock(&g_shared_lock);
			return FALSE;
		}
	}
	else {
		// Look for a page in the free list.	If one exists the use it.
		if (g_free_queue.IsEmpty()) {
			unreserved_size = 0;
			goto retry;
		}

		current_page = g_free_queue.PeekFirstPage();

		// Setup the context.  Also, restore the gap list to alloc_next.
		if (!SetupAllocContext(&acontext, NULL, MOSTLY_PAGE_SIZE, first_gap,
			last_gap, current_page, TRUE, TRUE)
		) {
			leave_spin_lock(&g_shared_lock);
			return FALSE;
		}
	}

	if (GcIsReplicatingCollector()) {
		current_page->SetActive();
	}

	allocated_total_size.addSize(acontext.alloc_end - acontext.alloc_start);

	leave_spin_lock(&g_shared_lock);
	return TRUE;
}

void
gc_heap::StartCycle()
{
	SetStage(WAITING_TO_START);
}

Page*
gc_heap::FindAdjacentPages(size_t min_size, size_t& actual_size)
{
	Page* first_page = g_free_queue.PeekFirstPage();
	int page_count = 1;

	if (!first_page){
		return NULL;
	}

	Page* last_page = first_page;
	actual_size = MOSTLY_PAGE_SIZE;
	Page* previous_page = NULL;

	while (actual_size < min_size + min_obj_size
	|| actual_size == min_size){
		if (last_page->GetNextPage() != NULL
		&& (last_page->GetData() + MOSTLY_PAGE_SIZE
		== last_page->GetNextPage()->GetData())){
			last_page = last_page->GetNextPage();
			actual_size += MOSTLY_PAGE_SIZE;
			page_count++;
		}
		else if (last_page->GetNextPage() != NULL){
			previous_page = last_page;
			first_page = last_page = last_page->GetNextPage();
			actual_size = MOSTLY_PAGE_SIZE;
			page_count = 1;
		}
		else{
			// End of the free list.
			return NULL;
		}
	}

	if(actual_size > unreserved_size)
		return NULL;

	// Fix up the free_page_list.
	g_free_queue.RemovePages(first_page, last_page, page_count);

	for (Page* current_page = first_page;
		current_page != NULL;
		current_page = current_page->GetNextPage())
	{
		current_page->SetAge(Page::AGE_YOUNG);

		if (current_page == first_page){
			current_page->SetLarge();
		}
		else{
			current_page->SetContinued();
		}
		
		current_page->SetStatus(Page::STATUS_PROMOTED);
		current_page->SetPromotedReason(Page::PROMOTED_REASON_LARGE);
		dprintf(4, ("page" FMT_ADDR "promoted by large alloc",
			DBG_ADDR(current_page)));
	}

	// Assumes that large objects will be promoted in place:
	subSize(&unreserved_size, actual_size);
	live_size += min_size;

	allocated_total_size.addSize(actual_size);
	// XXX Should large objects always be promoted in-place?
	g_promoted_queue.PushPages(first_page, last_page, page_count);

	/*
		Make sure that we've committed the underlying virtual
		memory for all pages.
	*/
	BYTE* lastpageend = last_page->GetData() + MOSTLY_PAGE_SIZE;
	if (lastpageend > heap_segment_committed (segment)) {
		if (!GrowHeapSegment (segment,lastpageend)){
			return NULL;
		}
	}

	return first_page;
}

// XXX factor out parts common to non-gc allocation and copying-allocation
inline Object*
GCHeap::AllocateInContext(size_t size, BOOL pinned)
{
	size = Align (size);
	assert (size >= Align (min_obj_size));
	{
retry:
		/*
			N.B.  The inlined allocation code must agree
			with the following code!!  It currently
			implements a slightly more conservative test
			(it only checks against alloc_limit), but it
			must do the appropriate initialization (e.g.
			relocation data).  It must also duplicate the
			code in the allocation functions found in
			gcscan.cpp.
		*/
		BYTE* result = g_acontext.alloc_ptr;
		g_acontext.alloc_ptr += size;

		if (g_acontext.alloc_ptr <= g_acontext.alloc_limit
		|| g_acontext.alloc_ptr == g_acontext.alloc_end){
			if (pinned) {
				// FIXME assumes that the Page coincides with the alloc_context.
				Page* page = gc_heap::GetPage(g_acontext.alloc_start);
				if (!page->HasStatus(Page::STATUS_PROMOTED)) {
					if (gc_heap::ReplicateWrites()) {
						if (result != g_acontext.alloc_start) {
							// FIXME abort allocation in this context
							g_acontext.alloc_ptr = g_acontext.alloc_end;
							goto retry;
						}
					}

					ASSERT(page->HasStatus(Page::STATUS_EVACUATED));
					gc_heap::g_evacuated_queue.RemovePage(page);
					page->SetStatus(Page::STATUS_PROMOTED);
					page->SetPromotedReason(Page::PROMOTED_REASON_PERM);
					gc_heap::g_promoted_queue.PushPage(page);
					dprintf(4, ("page" FMT_ADDR "promoted by pinned alloc",
						DBG_ADDR(page)));
				}
				else if (!page->HasPromotedReason(Page::PROMOTED_REASON_PERM)) {
					page->SetPromotedReason(Page::PROMOTED_REASON_PERM);
				}
			}

			if (gc_heap::GcIsReplicatingCollector()
			&& gc_heap::ReplicateWrites()) {
				if (pinned 
				|| gc_heap::GetPage(g_acontext.alloc_start)
					->HasStatus(Page::STATUS_PROMOTED))
				{
					header(result)->SetBlack();
				}
				else {
					Object* replica = (Object*)g_rcontext.alloc_ptr;
					g_rcontext.alloc_ptr += size;
					if (g_rcontext.alloc_ptr > g_rcontext.alloc_limit
					&& g_rcontext.alloc_ptr != g_rcontext.alloc_end) {
						g_rcontext.alloc_ptr -= size;
						if (!gc_heap::ResetRContext(g_rcontext)) {
							return NULL;
						}
						replica = (Object*)g_rcontext.alloc_ptr;
						g_rcontext.alloc_ptr += size;
					}

					header(result)->SetRelocation(replica);
					header(result)->SetGray();
					dprintf(3, ("Replicating alloc %X @ %X", result, replica));
					
					replicated_used_size.addSize(size);
				}
			}
			else {
				// FIXME is this necessary?
				header(result)->ClearGCHeader();
			}

			#ifdef GC_COUNT_ALLOC_FITS
				successful_alloc++;
			#endif // GC_COUNT_ALLOC_FITS

			/*
				Note that we don't add to
				allocated_used_size here -- that's
				done at the end of the page instead.
				(This makes easier to account for
				allocation that's done inline.)
			*/
			return (Object*)result;
		}
		else
		{
			g_acontext.alloc_ptr -= size;

			#ifdef GC_COUNT_ALLOC_FITS
				unsuccessful_alloc++;
			#endif // GC_COUNT_ALLOC_FITS

			/*
				Note that this may cause some
				fragmentation in the replica context:
				if the new object is to be pinned,
				then we won't actually double
				allocate.  Thus the empty space in
				g_rcontext might be used for something
				else.  However, this is too much
				trouble currently.
			*/
			// XXX Consider using the replica context more efficiently.
			if (!gc_heap::ResetContext(g_acontext, g_rcontext, size)) {
				return 0;
			}
			goto retry;
		}
	}
}

/*
	Allocates a large object from a set of contiguous pages.
	However, it does NOT clear memory!
*/
inline Object*
GCHeap::AllocateFromPages(size_t size, size_t& actual_size)
{
	/*
		N.B.  Assumes that all alloc'd from pages data is
		pinned and therefore NOT replicated.
	*/
	size = Align (size);
	assert (size >= Align (min_obj_size));

	/*
		XXX We currently do not perform any incremental
		scanning when allocating large objects.
	*/
	BOOL ran_gc = FALSE;

	{
retry:
		Page* first_page = gc_heap::FindAdjacentPages(size, actual_size);

		if (first_page){
			allocated_used_size.addSize(size);

			// FIXME add the remaining space as a gap
			return (Object*)first_page->GetData();
		}
		else if (!ran_gc){
			ran_gc = TRUE;

			if (GarbageCollect(CONDEMN_GEN_NUM, FALSE) == S_OK){
				goto retry;
			}
		}

		return NULL;
	}
}


LONG
HandleGCException(PEXCEPTION_POINTERS pExceptionInfo, PVOID)
{
	/*
		need this to avoid infinite loop if no debugger
		attached as will keep calling our handler as it's
		still on the stack.
	*/
	if (pExceptionInfo->ExceptionRecord->ExceptionCode == STATUS_BREAKPOINT)
		return EXCEPTION_CONTINUE_SEARCH;

	_ASSERTE_ALL_BUILDS(!"Exception during GC");
	/*
		Set the debugger to break on AV and return a value of
		EXCEPTION_CONTINUE_EXECUTION (-1) here and you will
		bounce back to the point of the AV.
	*/
	return EXCEPTION_EXECUTE_HANDLER;
}

int
Page::PageListLength(Page* page_list)
{
	int pages = 0;
	for(Page* tmp_page = page_list;
		tmp_page != NULL;
		++pages, tmp_page = tmp_page->GetNextPage())
	{
	}
	return pages;
}

Page*
gc_heap::ClearAdjacentPages(Page* current_page, BYTE* end_byte, int& page_count)
{
	current_page->ClearFlags();
	current_page->ClearAge();
	current_page->ClearResidency();
	#ifdef _DEBUG
		memset (current_page->GetData(), 0xCC, PAGE_FREE_SIZE);
	#endif //_DEBUG

	++page_count;
	BYTE* current_byte = current_page->GetData() + MOSTLY_PAGE_SIZE;

	while (current_byte < end_byte){
		current_page = current_page->GetNextPage();
		current_page->ClearFlags();
		current_page->ClearAge();
		current_page->ClearResidency();

		#ifdef _DEBUG
			memset (current_page->GetData(), 0xCC, PAGE_FREE_SIZE);
		#endif //_DEBUG
		++page_count;
		current_byte += MOSTLY_PAGE_SIZE;
	}

	return current_page;
}

void
gc_heap::InitiateCollection()
{
	SetStage(INITIALIZING);

	gc_count++;

	#ifdef TRACE_GC
		if (gc_count >= g_pConfig->GetGCtraceStart()) {
			trace_gc = 1;
		}
		if (gc_count >=	 g_pConfig->GetGCtraceEnd()) {
			trace_gc = 0;
		}
	#endif // TRACE_GC

	dprintf(1,(" ****Garbage Collection**** %d", gc_count));

	dprintf( 2, ("%d pages to be evacuated",
		g_evacuated_queue.GetLength()));
	dprintf( 2, ("%d pages to be promoted in place",
		g_promoted_queue.GetLength()));
	dprintf( 2, ("%d pages left in free list",
		g_free_queue.GetLength()));

	if (GcSupportsVerifyHeap() && g_pConfig->IsHeapVerifyEnabled()){
		VerifyHeap();
	}
	//DumpHeap();

	SetStage(INITIAL_ROOT_SCAN);
}

/*
	This must be large enough to hold the gap size and a pointer
	to the next gap.  We currently make it even larger to avoid
	putting really small gaps in the gap list.
*/
#define SMALLEST_USEFUL_GAP (4 * min_obj_size)

size_t
gc_heap::ThreadPage(Page* page, Object*& first_gap, Object*& prev_gap)
{
	BYTE* current_byte = page->GetData();
	BYTE* next_byte;
	BYTE* end_byte = page->GetData() + MOSTLY_PAGE_SIZE;
	size_t size = 0;
	size_t used_size = 0;
	while (current_byte < end_byte){
		// XXX is Align necessary?
		size = Align(header(current_byte)->GetSize());
		if (header(current_byte)->IsFree()){
			next_byte = current_byte + size;
			// Test against end_byte before IsFree:
			while (next_byte < end_byte && header(next_byte)->IsFree()) {
				size = Align(header(next_byte)->GetSize());
				next_byte += size;
			}

			if (next_byte - current_byte >= SMALLEST_USEFUL_GAP) {
				used_size += next_byte - current_byte;
					
				if (prev_gap != NULL) {
					header(prev_gap)->SetNextGap((Object*)current_byte);
				}
				else {
					first_gap = (Object*)current_byte;
				}
				header(current_byte)->SetFree(next_byte - current_byte);
				prev_gap = (Object*)current_byte;
			}

			current_byte = next_byte;
		}
		else {
			current_byte += size;
		}
	}
	if (prev_gap != NULL) {
		header(prev_gap)->ClearNextGap();
	}

	return used_size;
}


/*
	Currently, this function must always be performed without any
	mutator interleaving (even in the replicating case).
*/
void
gc_heap::FinishCollection(
	ScanContext& sc, 
	Object*& the_first_gap,
	Object*& the_last_gap)
{
	SetStage(FINAL_ROOT_SCAN);

	if (GcIsReplicatingCollector()) {
		// STACKLETS
		// Update the roots to point into to-space.
		ScanStrongRoots(sc, gc_heap::Relocate);
	}

	SetStage(FINISHING);

	// STACKLETS
	CNameSpace::GcScanLongWeakHandles(gc_heap::Relocate, &sc);

	// STACKLETS
	dprintf(3,("Relocating finalization data"));
	g_finalize_queue->RelocateFinalizationData (gc_heap::Relocate, &sc);

	// STACKLETS
	CNameSpace::GcPromotionsGranted();

	int reclaimed_pages = 0;

	dprintf(2, ("%d pages evacuated (dead) ", g_evacuated_queue.GetLength()));

	promoted_live_size = 0;
	evacuated_live_size = 0;
	promoted_threaded_size = 0;
	evacuated_threaded_size = 0;
	size_t young_page_count = 0;
	size_t total_young_residency = 0;

	Object* first_gap = NULL;
	Object* last_gap = NULL;

	TIMER free_timer;

	int free_pages = g_free_queue.GetLength();
	unused_total_space.addSize(free_pages * MOSTLY_PAGE_SIZE);

	for(Page* current_page = g_evacuated_queue.PeekFirstPage(); 
		current_page != NULL; )
	{
		ASSERT(current_page->HasStatus(Page::STATUS_EVACUATED));

		if (current_page->GetAge() == Page::AGE_YOUNG) {
			dprintf(2, ("Residency of (evacuated) young page (%x): %x", 
				current_page, current_page->GetResidency()));
			total_young_residency += current_page->GetResidency();
			young_page_count++;
		}

		BYTE* end_byte = current_page->GetData(); 
		if (ReportGcSpace) {
			if (!current_page->IsContinued()) {
				while (end_byte < current_page->GetData() + MOSTLY_PAGE_SIZE) {
					size_t size = Align(header(end_byte)->GetSize());
					if (header(end_byte)->IsWhite() && !header(end_byte)->IsFree()) {
						reclaimed_total_space.addSize(size);
					}
					end_byte += size;
				}
			}
		}

		end_byte = current_page->GetData() + MOSTLY_PAGE_SIZE;
		ClearAdjacentPages(current_page, end_byte, reclaimed_pages);

		Page* next_page = current_page->GetNextPage();

		g_evacuated_queue.RemovePage(current_page);
		current_page->SetStatus(Page::STATUS_FREE);
		g_free_queue.CoalescePages(current_page, current_page, 1);
		
		// Continue with the rest of the evacuated list.
		current_page = next_page;
	}

	unreserved_size = (size_t)(g_free_queue.GetLength() * MOSTLY_PAGE_SIZE);
	live_size = 0;

	if(ReportGcTime){
		double free_time = checkTimeSeconds(free_timer);
		printf ("FR: %d _ _ Pause Time: %f\n", gc_count, free_time);
		fflush(stdout);
	}

	TIMER pro_timer;

	for(Page* current_page = g_promoted_queue.PeekFirstPage(); 
		current_page != NULL; )
	{
		ASSERT(current_page->HasStatus(Page::STATUS_PROMOTED));

		/*
			The marked objects on this page are still
			live, but we can't move them.  If there are no
			live objects, however, then the page really is
			dead and can be reclaimed.
		*/
		size_t residency = 0;

		/*
			Go through each object on the page, if it's
			not marked, then clear it.  If it is marked,
			then just unmark it.
		*/
		BYTE* current_byte = current_page->GetData();
		BYTE* end_byte = current_byte + MOSTLY_PAGE_SIZE;

		// Used to thread the gaps:
		size_t gaps_size = 0;
		Object* my_first_gap = NULL;
		Object* my_last_gap = NULL;

		while (current_byte < end_byte){
			if (!header(current_byte)->IsWhite()){
				// XXX replicas on promoted pages are currently gray
				//ASSERT(header(current_byte)->IsBlack());
				header(current_byte)->SetWhite();
				
				// XXX is Align necessary?
				size_t size = Align(header(current_byte)->GetSize());
				residency += size;
				
				if (GcIsReplicatingCollector()) {
					/*
						Remember that pinned pages act like roots in a replicating
						collector: we must now forward all the ~children~ of pinned
						objects.
					*/
					if (header(current_byte)->ContainsPointers()){
						TracePointers(method_table(current_byte), current_byte, size, child,
							{
								// XXX is this ASSERT necessary?
								ASSERT((*child == NULL) || IsHeapObject((BYTE*)*child));
								if (*child != NULL) {
									if (!header(*child)->IsWhite()) {
										if (GetPage((BYTE*)*child)->HasStatus(Page::STATUS_PROMOTED)) {
											ASSERT(header(*child)->IsBlack());
											// Nothing to do.
										}
										else {
											*child = header(*child)->GetRelocated();
										}
									}
									else {
										/*
											Okay: here's the situation. The current object
											(current_byte) is live. Therefore, its children must be
											live_pages. Then if we see a white child, it just means
											that we've already cleared the color bit for the child.
											When would we clear the color bit? Answer: only if the
											child is pinned.  However, the pinned flags for the
											page may also have been cleared. So just check if is
											live.
										*/
										ASSERT(GetPage((BYTE*)*child)->HasStatus(Page::STATUS_PROMOTED)
											|| GetPage((BYTE*)*child)->HasStatus(Page::STATUS_REPLICA));
									}
								}
							});
					}
				}

				current_byte += size;
			}
			else {

				BYTE* gap_byte = current_byte;
				do {
					// XXX is Align necessary?
					size_t size = Align(header(current_byte)->GetSize());
					current_byte += size;
				}
				while (current_byte < end_byte && header(current_byte)->IsWhite());

				size_t size = current_byte - gap_byte;
				if (size >= SMALLEST_USEFUL_GAP) {
					gaps_size += size;
					if (my_last_gap != NULL) {
						header(my_last_gap)->SetNextGap((Object*)gap_byte);
					}
					else {
						my_first_gap = (Object*)gap_byte;
					}
					header(gap_byte)->SetFree(size);
					my_last_gap = (Object*)gap_byte;
				}
				else {
					header(gap_byte)->SetFree(size);
				}
			}
		}

		if (my_last_gap != NULL) {
			header(my_last_gap)->ClearNextGap();
		}
		
		ASSERT(!current_page->IsContinued());

		if (!current_page->IsLarge()) {
			dprintf (2,("Predicted residency of page %x: %x; Actual residency: %x\n",
				current_page, current_page->GetResidency(), residency));
				current_page->SetResidency(residency);
		}
		else {
			/*
				For pages with large objects, if a
				live object is found, the set the
				residency to the size of the large
				object.
			*/
			if (residency > 0) {
				residency = header(current_page->GetData())->GetSize();
			}
		}

		if (current_page->GetAge() == Page::AGE_YOUNG
		&& !current_page->IsLarge()) {
			dprintf(2, ("Residency of (promoted) young page (%x): %x", 
				current_page, current_page->GetResidency()));
			total_young_residency += residency;
			young_page_count++;
		}

		/*
			This page may be a part of a large object --
			if so, find the last page that underlies that
			object.
		*/
		int count = 0;
		Page* next_page = current_page->GetNextNonContinuedPage(count);

		if (residency == 0) {
			Page* last_page;
			if (next_page == NULL) {
				last_page = current_page;
			}
			else {
				last_page = next_page->GetPrevPage();
			}

			end_byte = last_page->GetData() + MOSTLY_PAGE_SIZE;
			ClearAdjacentPages(current_page, end_byte, reclaimed_pages);

			dprintf(2, ("No marked objects on promoted page %x", current_page));
			g_promoted_queue.RemovePages(current_page, last_page, count);
			current_page->SetStatus(Page::STATUS_FREE);
			g_free_queue.CoalescePages(current_page, last_page, count);
			addSize(&unreserved_size, count * MOSTLY_PAGE_SIZE);
		}
		else {
			if (residency <= thread_residency_threshold) {
				// Thread the gaps.
				//gaps_size = ThreadPage(current_page, my_first_gap, my_last_gap);
				dprintf (2,("Threading (promoted) page %x: using %x of %x bytes\n",
					current_page, gaps_size, MOSTLY_PAGE_SIZE - residency));
			}
		
			BOOL promote = FALSE;
			// Check and see if this page will also be pinned in the next round.
			// Put it in the proper page queue.
			if (current_page->HasPromotedReason(Page::PROMOTED_REASON_PERM) 
			|| current_page->HasPromotedReason(Page::PROMOTED_REASON_LARGE)) {
				promote = TRUE;
			}
			else if (pin_using_residency
			&& (current_page->IsFull() ? MOSTLY_PAGE_SIZE : residency)
				> pin_residency_threshold
			&& current_page->GetAge() >= min_pinning_age) {
				current_page->SetPromotedReason(
					Page::PROMOTED_REASON_DENSE);
				promote = TRUE;
			}
			else if (gaps_size + residency
				+ evacuated_threaded_size + evacuated_live_size
				> promoted_threaded_size + unreserved_size) {
				/*
					Can't evacuate!  If we did we
					might not have enough room for
					replica in the next round.
				*/
				current_page->SetPromotedReason(
					Page::PROMOTED_REASON_DENSE);
				promote = TRUE;
			}
			
			if (promote) {
				addSize(&promoted_live_size, residency);
				addSize(&promoted_threaded_size, gaps_size);
				if (my_first_gap != NULL) {
					if (last_gap != NULL) {
						header(last_gap)->SetNextGap(my_first_gap);
					}
					else {
						first_gap = my_first_gap;
					}
					last_gap = my_last_gap;
				}
			}
			else {
				addSize(&evacuated_live_size, residency);
				addSize(&evacuated_threaded_size, gaps_size);
				g_promoted_queue.RemovePage(current_page);
				current_page->ClearPromotedReason();
				current_page->SetStatus(Page::STATUS_EVACUATED);
				g_evacuated_queue.PushPage(current_page);
				dprintf(4, ("page" FMT_ADDR "evacuated <- promoted",
					DBG_ADDR(current_page)));
				if (my_last_gap != NULL) {
					if (first_gap != NULL) {
						header(my_last_gap)->SetNextGap(first_gap);
					}
					else {
						last_gap = my_last_gap;
					}
					first_gap = my_first_gap;
				}
			}

			// Increase the age of the page
			current_page->SetAge(
				min(Page::AGE_ANCIENT, current_page->GetAge() + 1));

		}
		pinned_total_size.addSize(MOSTLY_PAGE_SIZE * count);
		
		current_page = next_page;
	}

	if(ReportGcTime){
		double pro_time = checkTimeSeconds(pro_timer);
		printf ("PR: %d _ _ Pause Time: %f\n", gc_count, pro_time);
		fflush(stdout);
	}

	TIMER replica_timer;

	for (Page* current_page = g_replica_queue.PeekFirstPage(); 
		current_page != NULL; )
	{
		/*
			This is just an ordinary live page in to-space
			-- it contains copied or replicated data.
			Confirm that the page flags are set correctly
			and move on to the next page.
		*/
		ASSERT(current_page->HasStatus(Page::STATUS_REPLICA));
		size_t residency = current_page->GetResidency();

		size_t gaps_size = 0;
		Object* my_first_gap = NULL;
		Object* my_last_gap = NULL;
		if (residency <= thread_residency_threshold) {
			gaps_size = ThreadPage(current_page, my_first_gap, my_last_gap);
			dprintf (2,("Threading (replica) page %x: using %x of %x bytes\n",
				current_page, gaps_size, MOSTLY_PAGE_SIZE - residency));
		}

		// Currently, we never expect to see a large page in the replica list.
		ASSERT(!current_page->IsLarge());
		Page* next_page = current_page->GetNextPage();

		BOOL promote = FALSE;
		if (pin_using_residency
		&& (current_page->IsFull() ? MOSTLY_PAGE_SIZE : residency)
			> pin_residency_threshold
		&& current_page->GetAge() >= min_pinning_age) {
			promote = TRUE;
		}
		else if (gaps_size + residency 
			+ evacuated_threaded_size + evacuated_live_size
			> promoted_threaded_size + unreserved_size) {
			/*
				Can't evacuate!  If we did we might
				not have enough room for replica in
				the next round.
			*/
			promote = TRUE;
		}

		if (promote) {
			addSize(&promoted_live_size, residency);
			addSize(&promoted_threaded_size, gaps_size);
			g_replica_queue.RemovePage(current_page);
			current_page->SetStatus(Page::STATUS_PROMOTED);
			current_page->SetPromotedReason(Page::PROMOTED_REASON_DENSE);
			g_promoted_queue.PushPage(current_page);
			dprintf(4, ("page" FMT_ADDR "promoted by residency or necessity",
				DBG_ADDR(current_page)));
			if (my_first_gap != NULL) {
				if (last_gap != NULL) {
					header(last_gap)->SetNextGap(my_first_gap);
				}
				else {
					first_gap = my_first_gap;
				}
				last_gap = my_last_gap;
			}
		}
		else {
			addSize(&evacuated_live_size, residency);
			addSize(&evacuated_threaded_size, gaps_size);
			g_replica_queue.RemovePage(current_page);
			current_page->SetStatus(Page::STATUS_EVACUATED);
			g_evacuated_queue.PushPage(current_page);
			dprintf(4, ("page" FMT_ADDR "evacuated <- replica",
				DBG_ADDR(current_page)));

			if (my_last_gap != NULL) {
				if (first_gap != NULL) {
					header(my_last_gap)->SetNextGap(first_gap);
				}
				else {
					last_gap = my_last_gap;
				}
				first_gap = my_first_gap;
			}
		}
		
		current_page->ClearFull();
		current_page = next_page;
	}

	if(ReportGcTime){
		double replica_time = checkTimeSeconds(replica_timer);
		printf ("RE: %d _ _ Pause Time: %f\n", gc_count, replica_time);
		fflush(stdout);
	}

	if (young_page_count == 0) {
		// XXX Just leave it alone?  Or max it out?
		//average_young_page_residency = MOSTLY_PAGE_SIZE;
		dprintf(1, ("Using old average residency of young pages: %x", 
			average_young_page_residency));
	}
	else {
		/*
			Add some hysteresis by splitting the
			difference betweent the last measured
			residency and the currnet measured residency.
		*/
		average_young_page_residency += total_young_residency / young_page_count;
		average_young_page_residency /= 2;
		dprintf(1, ("New average residency of young pages: %x", 
			average_young_page_residency));
	}

	/* broken up because addSize does not like negative deltas */
	addSize(&unreserved_size, promoted_threaded_size + evacuated_threaded_size);
	subSize(&unreserved_size, evacuated_live_size);

	if (GcIsReplicatingCollector()) {
		live_size = promoted_live_size + evacuated_live_size;
	}

	dprintf( 2, ("%d live pages after collection",
		g_evacuated_queue.GetLength() + g_promoted_queue.GetLength()));
	dprintf( 2, ("%d pages in free list",
		g_free_queue.GetLength()));

	the_first_gap = first_gap;
	the_last_gap = last_gap;
}


/*
	spoons: This class is a placeholder until the code in
	GarbageCollect can be restructed.  The problem is that in
	incremental collection, we would like to do just one stage at
	a time, while in stop-the-world, we (obviously) want to do all
	of them at once.
*/
class WorkStatus{
	public:

	WorkStatus();

	// Should we continue to do work?
	BOOL ContinueWork(
		size_t bytesToCollect);

	// Call when ~some~ work has been completed.
	void WorkAccomplished();

	private:

	BOOL accomplished;
};

WorkStatus::WorkStatus()
{
	accomplished = FALSE;
}

#ifdef GC_REPLICATING_COLLECTOR

	BOOL
	WorkStatus::ContinueWork(size_t bytesToCollect)
	{
		return !accomplished
			|| (bytesToCollect == gc_heap::GC_FULL_COLLECTION);
	}
	
	void
	WorkStatus::WorkAccomplished()
	{
		accomplished = TRUE;
	}

#else // !GC_REPLICATING_COLLECTOR

	BOOL
	WorkStatus::ContinueWork(size_t bytesToCollect)
	{
		return TRUE;
	}
	
	void WorkStatus::WorkAccomplished()
	{
	}

#endif // !GC_REPLICATING_COLLECTOR

void 
gc_heap::GarbageCollect(
	size_t bytesToCollect,
	Object*& next_gap,
	Object*& last_gap)
{
	ScanContext sc;
	WorkStatus status;
	Object* first_gap = NULL;

	while (next_gap != NULL
	&& GetPage((BYTE*)next_gap)->HasStatus(Page::STATUS_EVACUATED)) {
		printf ("Warning: skipping evacuated gap %x\n", next_gap);
		next_gap = header(next_gap)->GetNextGap();
	}
	g_next_gap = next_gap;

	GC_STAGE start_stage = g_stage;

	if (!(GcIsReplicatingCollector() && GcIsConcurrentCollector())) {
		g_pGCHeap->SuspendEE(GCHeap::SUSPEND_FOR_GC);
	}

	double pause_time = checkTimeSeconds(mutator_timer);

	if(ReportGcTime){
		printf ("MU: %d _ %d Pause Time: %f\n", gc_count, start_stage, pause_time);
		fflush(stdout);
	}
	mutator_time += pause_time;
	TIMER gc_timer;

	GC_TRY
	{

		ASSERT(g_acontext.alloc_ptr == NULL
			&& g_acontext.alloc_limit == NULL);

		TIMER initiate_timer;

		if (g_stage == WAITING_TO_START) {

			// Reset the size counters.
			black_scanned_size.reset();
			cheney_scanned_size.reset();
			queue_scanned_size.reset();
			white_scanned_size.reset();
			
			pinned_live_size.reset();
			pinned_total_size.reset();
			copied_live_size.reset();
			copied_total_size.reset();
			write_total_size.reset();

			reclaimed_total_space.reset();
			unused_total_space.reset();

			InitiateCollection();
			status.WorkAccomplished();
		}

		if (status.ContinueWork(bytesToCollect)
		&& g_stage == INITIAL_ROOT_SCAN) {
			#if defined (VERIFY_HEAP)
				sc.verify_page_is_live = TRUE;
			#endif

			if (GcIsReplicatingCollector() && GcIsConcurrentCollector()) {
				/*
					In concurrent mode, we must
					suspend other threads while we
					scan the stack -- this ensures
					that all threads are at safe
					points.
				*/
				leave_spin_lock(&g_shared_lock);
				g_pGCHeap->SuspendEE(GCHeap::SUSPEND_FOR_GC);
				enter_spin_lock(&g_shared_lock);
			}

			// STACKLETS
			ScanStrongRoots(sc, gc_heap::RegisterRoot);

			// STACKLETS
			dprintf(3,("Scanning finalization data"));
			g_finalize_queue->GcScanRoots(gc_heap::RegisterRoot, &sc);

			if (GcIsReplicatingCollector() && GcIsConcurrentCollector()) {
				g_pGCHeap->RestartEE(FALSE, TRUE);
			}

			SetStage(COPYING);
			status.WorkAccomplished();
		}

		if(ReportGcTime){
			double initiate_time = checkTimeSeconds(initiate_timer);
			printf ("IN: %d _ _ Pause Time: %f\n", gc_count, initiate_timer);
			fflush(stdout);
		}

		TIMER scan_timer;

		if (status.ContinueWork(bytesToCollect) && g_stage == COPYING) {
			if (g_work_list.HasWork()) {
				// Copy from the roots
				size_t bytesCollected = CopyObjects (bytesToCollect);
				if (bytesCollected >= bytesToCollect) {
					status.WorkAccomplished();
				}
				else {
					ASSERT(!g_work_list.HasWork());
					SetStage(WAITING_TO_FINISH);
				}
			}
			else {
				SetStage(WAITING_TO_FINISH);
			}
		}

		if(ReportGcTime){
			double scan_time = checkTimeSeconds(scan_timer);
			printf ("SC: %d _ _ Pause Time: %f\n", gc_count, scan_time);
			fflush(stdout);
		}
		TIMER finish_timer;

		if (status.ContinueWork(bytesToCollect) && g_stage == WAITING_TO_FINISH){
			if (GcIsReplicatingCollector() && GcIsConcurrentCollector()) {
				leave_spin_lock(&g_shared_lock);
				g_pGCHeap->SuspendEE(GCHeap::SUSPEND_FOR_GC);
				enter_spin_lock(&g_shared_lock);

				/*
					AddWorkAsWrites any remaining
					entries in the write buffer to
					the work list.
				*/
				if (g_wcontext.alloc_start != g_wcontext.alloc_ptr) {
					g_work_list.AddWorkAsWrites(
						(Object*)g_wcontext.alloc_start, 
						g_wcontext.alloc_ptr);
				}
				g_wcontext.alloc_start = g_wcontext.alloc_ptr = 
					g_wcontext.alloc_limit = g_wcontext.alloc_end = NULL;
			}

			// STACKLETS
			ScanWeakRoots(sc);

			// STACKLETS
			//Handle finalization.
			g_finalize_queue->ScanForFinalization (gc_heap::RegisterRoot, &sc);

			// Trace any objects reachable from a object-to-be-finalized.
			CopyObjects(GC_FULL_COLLECTION);

			/*
				Clean up the alloc context and add the
				last live page (if any) to the mutator
				list:
			*/
			for (int i = 0; i < GC_NUMBER_OF_CONTEXTS; i++) {
				if(g_scontexts[i].alloc_start != g_scontexts[i].alloc_ptr){
					Page* page = GetPage(g_scontexts[i].alloc_start);
					page->SetResidency(g_scontexts[i].alloc_ptr - page->GetData());
					ClearAllocContext(&g_scontexts[i], TRUE);
				}
			}

			FinishCollection(sc, next_gap, last_gap);
			SetStage(NOT_IN_CYCLE);
			if (GcIsReplicatingCollector() && GcIsConcurrentCollector()) {
				g_pGCHeap->RestartEE(FALSE, TRUE);
			}

			if (GcSupportsVerifyHeap() && g_pConfig->IsHeapVerifyEnabled())
			{
				VerifyHeap();
			}
			//DumpHeap();
		}

		if(ReportGcTime){
			double finish_time = checkTimeSeconds(finish_timer);
			printf ("FI: %d _ _ Pause Time: %f\n", gc_count, finish_time);
			fflush(stdout);
		}

	}
	GC_EXCEPT_FILTER(HandleGCException, NULL)
	{
		_ASSERTE(!"Exception during GarbageCollectGeneration()r");
	}
	GC_ENDTRY

	pause_time = checkTimeSeconds(gc_timer);
	gc_time += pause_time;

	if(ReportGcTime) {
		int full = (bytesToCollect == GC_FULL_COLLECTION);
		printf ("GC: %d %d %d Pause Time: %f\n", gc_count, bytesToCollect, 
						start_stage, pause_time);
		fflush(stdout);
	}
	if (g_stage == NOT_IN_CYCLE) {
		if (ReportGcSpace) {
			printf("SP: %d %d _ Allocated Used:   % 15d"
				"                                               Allocated Total:  % 15d"
				"                                               Replicated Used:  % 15d"
				"                                               Replicated Total: % 15d"
				"                                               Black Scanned:    % 15d"
				"                                               Cheney Scanned:     % 15d"
				"                                             Promoted Scanned:     % 15d"
				"                                           White Scanned:    % 15d"
				"                                               Pinned Live:      % 15d"
				"                                               Pinned Total:     % 15d"
				"                                               Copied Live:      % 15d"
				"                                               Copied Total:     % 15d"
				"                                               Immobile Live:    % 15d"
				"                                               Copyable Live:    % 15d"
				"                                               Reclaimed Total:  % 15d"
				"                                               Unused Total:     % 15d"
				"                                               Thread Gap Total: % 15d"
				"                                               Write Total:      % 15d\n",
				 gc_count, bytesToCollect,
				 allocated_used_size.checkSize(),
				 allocated_total_size.checkSize(),
				 replicated_used_size.checkSize(),
				 replicated_total_size.checkSize(),
				 black_scanned_size.checkSize(),
				 cheney_scanned_size.checkSize(),
				 queue_scanned_size.checkSize(),
				 white_scanned_size.checkSize(),
				 pinned_live_size.checkSize(),
				 pinned_total_size.checkSize(),
				 copied_live_size.checkSize(),
				 copied_total_size.checkSize(),
				 promoted_live_size,
				 evacuated_live_size,
				 reclaimed_total_space.checkSize(),
				 unused_total_space.checkSize(),
				 promoted_threaded_size + evacuated_threaded_size,
				 write_total_size.checkSize());
			fflush(stdout);
		}
		allocated_used_size.reset();
		allocated_total_size.reset();
		replicated_used_size.reset();
		replicated_total_size.reset();

		#ifdef GC_COUNT_ALLOC_FITS
			printf("AL: %d %d _ Successful Allocations:   % 15d"
				"                                       Unsuccessful Allocations:  % 15d\n", 
				 successful_alloc, 
				 unsuccessful_alloc);
		#endif // GC_COUNT_ALLOC_FITS
	}
	mutator_timer.reset();

	if (!(GcIsReplicatingCollector() && GcIsConcurrentCollector())) {
		g_pGCHeap->RestartEE(TRUE, TRUE);
	}

	//return first_gap;
}

void
gc_heap::ShadeRoot(Object** root)
{
	#ifdef GC_REQUIRES_PTR_WRITEBARRIER
		// XXX Can we assert what stage we are in?  e.g. stage != WAITING_TO_FINISH
	
		/*
			Note that we CANNOT set the gray bit for
			*root, since we cannot guarantee that *root
			will be scanned -- for example, in the case
			that this field is overwritten before the work
			is removed from the list.
		*/
		g_work_list.AddWorkAsRoot(root, FALSE);
	#endif // GC_REQUIRES_PTR_WRITEBARRIER
}

void
gc_heap::ShadeObject(Object* obj)
{
	#ifdef GC_REQUIRES_PTR_WRITEBARRIER
		// XXX Can assert what stage we are in?	 e.g. stage != WAITING_TO_FINISH
	
		/*
			Here we are safe to set the gray bit since we
			are adding a pointer directly to the live
			object (rather than with another level of
			indirection, as in ShadeRoot above).
		*/
		if (GetPage((BYTE*)obj)->HasStatus(Page::STATUS_PROMOTED)) {
			header(obj)->SetGray();
		}
		g_work_list.AddWorkAsPtr(obj, FALSE);
	#endif // GC_REQUIRES_PTR_WRITEBARRIER
}

#if (defined(GC_REQUIRES_FULL_WRITEBARRIER) && defined(GC_CONCURRENT_COLLECTOR))
	enum {
		GC_WRITE_MASK = 0x0000000F,
		GC_WRITE_SHIFT = 4,
	
		GC_WRITE_SIZE = sizeof(Object*) + sizeof(int),
	};

	/*
		The "WriteLock" is used to protect the buffer used by
		the write barrier.  Again this is only necessary
		because all threads currently share the same buffer.
	*/
	inline static void
	write_spin_lock()
	{
		enter_spin_lock(&m_WriteLock);
	}
	
	inline void
	EnterWriteLock()
	{
		#if defined(_X86_) && defined(_MSC_VER)
			__asm {
				inc dword ptr m_WriteLock
				jz gotit
				call write_spin_lock
				gotit:		
			}
		#else //_X86_
			write_spin_lock();
		#endif //_X86_
	}

	inline void
	LeaveWriteLock()
	{
		// Trick this out
		leave_spin_lock (&m_WriteLock);
	}

	inline void
	gc_heap::RecordWrite(Object* obj, BYTE* data, GC_WRITE_KIND kind)
	{
		EnterWriteLock();
	
		// XXX check that GC is still in progress...
	
		/*
			Note that the difference between the end of
			the page the limit pointer must leave room for
			the plug!  (since the write buffers entries
			don't have a plug)
		*/
		if (g_wcontext.alloc_ptr + GC_WRITE_SIZE > g_wcontext.alloc_limit) {
			/*
				XXX Possible race: what if we are
				waiting to get the shared lock while
				another thread (the GC thread) is
				holding the shared lock and waiting
				for our write buffer?
			*/
			enter_spin_lock(&g_shared_lock);
	
			Page* current_page = g_free_queue.PopPage();
			g_write_queue.PushPage(current_page);
			write_total_size.addSize(MOSTLY_PAGE_SIZE);
	
			if (g_wcontext.alloc_start != g_wcontext.alloc_ptr) {
				g_work_list.AddWorkAsWrites(
					(Object*)g_wcontext.alloc_start,
					g_wcontext.alloc_ptr);
			}
	
			leave_spin_lock(&g_shared_lock);
	
			//current_page->ClearFree();
			current_page->SetWrites();
	
			SetupAllocContext(
				&g_wcontext, NULL, MOSTLY_PAGE_SIZE, NULL, NULL,
				current_page, FALSE, FALSE);
		}
	
		*((Object**)(g_wcontext.alloc_ptr)) = obj;
		if (kind == GC_WRITE_STRUCT) {
			*((int*)(g_wcontext.alloc_ptr + sizeof(Object*))) =
				(int)(kind | (size_t)data);
		}
		else if (kind == GC_WRITE_MULTI) {
			*((int*)(g_wcontext.alloc_ptr + sizeof(Object*))) = kind;
		}
		else {
			*((int*)(g_wcontext.alloc_ptr + sizeof(Object*))) 
				= (kind | ((data - (BYTE*)obj) << GC_WRITE_SHIFT));
		}
		g_wcontext.alloc_ptr += GC_WRITE_SIZE;
	
		LeaveWriteLock();
	}
#endif

void
gc_heap::RecordWrite8(Object* obj, __int8* dst)
{
	#if defined (GC_CONCURRENT_COLLECTOR)
		RecordWrite(obj, (BYTE*)dst, GC_WRITE_8BIT_INT);
	#endif
}

void
gc_heap::RecordWrite16(Object* obj, __int16* dst)
{
	#if defined(GC_CONCURRENT_COLLECTOR)
		RecordWrite(obj, (BYTE*)dst, GC_WRITE_16BIT_INT);	
	#endif
}

void
gc_heap::RecordWrite32(Object* obj, __int32* dst)
{
	#if defined(GC_CONCURRENT_COLLECTOR)
		RecordWrite(obj, (BYTE*)dst, GC_WRITE_32BIT_INT);
	#endif
}

void
gc_heap::RecordWrite64(Object* obj, __int64* dst)
{
	#if defined(GC_CONCURRENT_COLLECTOR)
		RecordWrite(obj, (BYTE*)dst, GC_WRITE_64BIT_INT);
	#endif
}

void
gc_heap::RecordWritePtr(Object* obj, Object** dst)
{
	#if defined(GC_CONCURRENT_COLLECTOR)
		RecordWrite(obj, (BYTE*)dst, GC_WRITE_PTR);
	#endif
}

void
gc_heap::RecordWriteStruct(void* obj, MethodTable* pMT)
{
	#if defined(GC_CONCURRENT_COLLECTOR)
		RecordWrite((Object*)obj, (BYTE*)pMT, GC_WRITE_STRUCT);
	#endif
}

void
gc_heap::RecordMultipleWrites(Object* obj)
{
	#if defined(GC_CONCURRENT_COLLECTOR)
		RecordWrite(obj, NULL, GC_WRITE_MULTI);
	#endif
}

void
gc_heap::ProcessWrites(
	Work& work,
	size_t bytesToCollect,
	size_t& bytesCollected,
	WorkList& work_list,
	PageQueue& dead_queue)
{
	#if defined(GC_REQUIRES_FULL_WRITEBARRIER) && defined(GC_CONCURRENT_COLLECTOR)
		BYTE** pObj;
	
		Page* page = GetPage((BYTE*)work.ptr);
		if(!page->IsWrites()) {
			// This write buffer was already processed.
			// XXX It seems that this situation is impossible... is it?
			ASSERT(!"Write buffered processed more than once");
		}
	
		g_write_queue.RemovePage(page);
	
		for (pObj = (BYTE**)work.ptr; (BYTE*)pObj < work.limit;
			pObj = (BYTE**)((BYTE*)pObj + GC_WRITE_SIZE))
		{
			GC_WRITE_KIND kind;
			if (*((int*)(pObj + 1)) & GC_WRITE_STRUCT) {
				kind = GC_WRITE_STRUCT;
			}
			else {
				kind = (GC_WRITE_KIND)(*((int*)(pObj + 1)) & GC_WRITE_MASK);
			}
			ptrdiff_t offset = (*((int*)(pObj + 1))) >> GC_WRITE_SHIFT;
	
			Object* obj;
			if (kind == GC_WRITE_STRUCT) {
				obj = FindObjectWithLock(*pObj);
			}
			else {
				obj = (Object*)*pObj;
			}
			ASSERT(header(obj)->IsRelocated());
			Object* replica = header(obj)->GetRelocated();
	
			switch (kind) {
			case GC_WRITE_8BIT_INT:
				{
					__int8 src = *((__int8*)((BYTE*)obj + offset));
					*((__int8*)((BYTE*)replica + offset)) = src;
					break;
				}
			case GC_WRITE_16BIT_INT:
				{
					__int16 src = *((__int16*)((BYTE*)obj + offset));
					*((__int16*)((BYTE*)replica + offset)) = src;
					break;
				}
			case GC_WRITE_32BIT_INT:
				{
					__int32 src = *((__int32*)((BYTE*)obj + offset));
					*((__int32*)((BYTE*)replica + offset)) = src;
					break;
				}
			case GC_WRITE_64BIT_INT:
				{
					__int64 src = *((__int64*)((BYTE*)obj + offset));
					*((__int64*)((BYTE*)replica + offset)) = src;
					break;
				}
			case GC_WRITE_PTR:
				{
					Object* src = *((Object**)((BYTE*)obj + offset));
					Object** dst = (Object**)((BYTE*)replica + offset);
			
					if (src == NULL) {
						// Write a reference to the (null) ref.
						*dst = NULL;
						dprintf(3, ("Replicating write (null) %X := %X", 
							dst, NULL));
						// No need to shade!!
					}
					else if (header(src)->IsRelocated()) {
						/*
							The ref already exists in to-space,
							so just write a reference to that.
						*/
						Object* srcReplica = header(src)->GetRelocated();
						*dst = srcReplica;
						dprintf(3, ("Replicating write (to-space) %X := %X", 
							dst, srcReplica));
					}
					else {
						// Write a reference to the ref (in from-space).
						*dst = src;
						// Shade the replica:
						dprintf(3, ("Replicating write (shade) %X := %X",
							dst, src));
						work_list.AddWorkAsRoot(dst, FALSE);
					}
					break;
				}
			case GC_WRITE_STRUCT:
				{
					MethodTable* pMT =
						(MethodTable*)((*((size_t*)(pObj + 1))) & ~GC_WRITE_STRUCT);
					size_t size = (pMT->GetBaseSize()
						+ (pMT->GetComponentSize()
							? (obj->GetNumComponents()  pMT->GetComponentSize())
							: 0));
					offset = *pObj - (BYTE*)obj;
					memcopy((BYTE*)replica + offset, (BYTE*)obj + offset, size);
					work_list.AddWorkAsRange(replica,
						(BYTE*)replica + header(replica)->GetSize());
					break;
				}
			case GC_WRITE_MULTI:
				{
					size_t size = Align(header(obj)->GetSize());
					memcopy((BYTE*)replica - plug_skew + runtime_skew, 
						(BYTE*)obj - plug_skew + runtime_skew,
						size - runtime_skew);
					work_list.AddWorkAsRange(replica, (BYTE*)replica + size);
					break;
				}
			}
	
			// XXX How should processing write contribute towards "work completed"?
			//bytesCollected += GC_WRITE_SIZE;
		}
	
		/*
			Add the page to the list of unused pages -- it
			will be added to the free list at the end of
			collection.
		*/
		page->ClearWrites();
		page->SetStatus(Page::STATUS_FREE);
	
		dead_queue.PushPage(page);
	#endif // GC_REQUIRES_FULL_WRITEBARRIER && GC_CONCURRENT_COLLECTOR
}

void
gc_heap::ScanStrongRoots(ScanContext& sc, promote_func* fn)
{
	dprintf(3,("Scanning stack"));
	CNameSpace::GcScanStacks(fn, &sc);

	dprintf(3,("Scanning handle table"));
	CNameSpace::GcScanStrongHandles(fn, &sc);
}

void
gc_heap::ScanWeakRoots(ScanContext& sc)
{
	if (GcSupportsWeakReferences()) {
		// scan for deleted short weak pointers
		CNameSpace::GcScanShortWeakHandles(gc_heap::Relocate, &sc);
	}
}

size_t
gc_heap::CopyObjects (size_t bytesToCollect)
{
	/*
		Normally, the gray region of the heap is described by
		the items in g_work_list.  However, during the
		execution of this function, we relax this constraint
		slightly: recently allocated replica objects (which
		are gray) may not be in the work list.

		The rational for this is that during copying the
		region between alloc_start and alloc_ptr is growing,
		and rather than continually adding new items to the
		work list, we just remember to scan the context at the
		end.

		This function must maintain the invariant that, when
		it completes, all gray objects and gray regions are
		indicated by items in the work list.  During
		execution, it maintains the invariant that all objects
		in the alloc_context and BEFORE scan_limit have
		already been scanned.

		Furthermore, the caller expects that if the result
		(bytesCollected) is less that bytesToCollect, then
		there are no items remaining in the work list.
	*/
	size_t bytesCollected = 0;

	/*
		In the context of this function, gray_ptrs show what
		part of the alloc_contexts has already been scanned.
		If a gray_ptr is non-NULL, then the area between
		scontext[i]->alloc_start and gray_ptr[i] has already
		be scanned (i.e.  is black) or was explicitly added to
		the work queue.
	*/
	BYTE* gray_ptrs[GC_NUMBER_OF_CONTEXTS];
	LoopOverContexts(i, gray_ptrs[i] = NULL);

	while (g_work_list.HasWork()) {

		Work work;

		g_work_list.RemoveWork(work);

		if (work.IsRange()) {

			ProcessRange(work, bytesToCollect, bytesCollected, g_scontexts, 
				gray_ptrs, g_work_list);

		}
		else if (work.IsWrites()) {

			// This case occurs only in the concurrent case.
			ProcessWrites(work, bytesToCollect, bytesCollected, g_work_list,
				g_free_queue);

		}
		else if (work.IsRoot()) { // Not a range or writes

			Object* obj = *(work.root);
			if (work.IsInterior()) {
				// Find the true beginning of the object
				obj = FindObjectWithLock((BYTE*)obj);
			}

			Page* page = GetPage((BYTE*)obj);
			if (page->HasStatus(Page::STATUS_PROMOTED)) {
				ProcessObject(obj, bytesToCollect, bytesCollected, gray_ptrs);
			}
			else {
				Object* obj_copy;
				if (!header(obj)->IsGray()) {
					if (!GcIsReplicatingCollector() 
					|| (!page->HasStatus(Page::STATUS_PROMOTED)
					|| page->HasStatus(Page::STATUS_REPLICA))) {
						/*
							Not on a live page --> we need to copy
			
							Here we invoke CopyObject knowning
							that everything up to scan_limit has
							ALREADY been scanned.  Therefore, if
							the context is reset, the scan_limit
							should also be reset (to obj + size),
							i.e.  reset_limit = TRUE.
						*/
						size_t size = 0;
						int age = page->GetAge(),
							context_age = min(age, GC_NUMBER_OF_CONTEXTS);
						obj_copy = CopyObject(obj, context_age + 1,
							&g_scontexts[context_age - 1], 
							gray_ptrs[context_age - 1],
							gray_ptrs[context_age - 1], TRUE, 
							g_work_list, size);
					
						if (age == Page::AGE_YOUNG) {
							page->SetResidency(page->GetResidency() + size);
						}
						
						Object* scan_ptr;
						if (gray_ptrs[context_age - 1] == (BYTE*)obj_copy + size) {
							scan_ptr = obj_copy;
						}
						else if (gray_ptrs[context_age - 1] == NULL) {
							scan_ptr =
								(Object*)g_scontexts[context_age - 1].alloc_start;
							gray_ptrs[context_age - 1] = (BYTE*)obj_copy + size;
						}
						else {
							ASSERT(gray_ptrs[context_age - 1] < (BYTE*)obj_copy);
							scan_ptr = (Object*)gray_ptrs[context_age - 1];
							gray_ptrs[context_age - 1] = (BYTE*)obj_copy + size;
						}
						
						CheneyScan(bytesToCollect, bytesCollected,
							g_scontexts, gray_ptrs, 
							scan_ptr, gray_ptrs[context_age - 1],
							g_work_list);
					}
					else {
						obj_copy = obj;
					}
				}
				else {
					obj_copy = header(obj)->GetRelocated();
				}
				
				ptrdiff_t offset = (BYTE*)*(work.root) - (BYTE*)obj;
				*(work.root) = (Object*)((BYTE*)obj_copy + offset);
			}
		}
		else {

			ASSERT(!work.IsInterior());
			ProcessObject(work.ptr, bytesToCollect, bytesCollected, gray_ptrs);
		}
		
		if (bytesCollected >= bytesToCollect) {
			/*
				Make sure all the gray alloc_context
				areas are added to the work queue.
			*/
			for (int which_context = 0;
				which_context < GC_NUMBER_OF_CONTEXTS; 
				which_context++)
			{
				if (gray_ptrs[which_context] != NULL){
					ASSERT(gray_ptrs[which_context]
						> g_scontexts[which_context].alloc_start);
					ASSERT(gray_ptrs[which_context]
						<= g_scontexts[which_context].alloc_ptr);
					if (gray_ptrs[which_context]
						< g_scontexts[which_context].alloc_ptr)
					{
						g_work_list.AddWorkAsRange(
							(Object*)gray_ptrs[which_context],
							g_scontexts[which_context].alloc_ptr);
					}
				}
				else if (g_scontexts[which_context].alloc_start != NULL) {
					g_work_list.AddWorkAsRange(
						(Object*)g_scontexts[which_context].alloc_start,
						g_scontexts[which_context].alloc_ptr);
				}
			}
			break;
		}
		else { // bytesCollected < bytesToCollect
			/*
				We cannot return until we've scanned
				the alloc context.  However, we may
				choose either to continue taking items
				from the work list or to scan the
				context right now.
				
			*/
			// XXX Consider these alternatives.
			/*
				if (g_work_list.HasWork()) {
					continue;
				}
				else // !g_work_list.HasWork() {
			*/
			
			for (int which_context = 0;
				which_context < GC_NUMBER_OF_CONTEXTS; )
			{
				Object* scan_ptr;
				
				if (gray_ptrs[which_context] == NULL) {
					if (g_scontexts[which_context].alloc_start != NULL) {
						scan_ptr = (Object*)g_scontexts[which_context].alloc_start;
						gray_ptrs[which_context] =
							g_scontexts[which_context].alloc_ptr;
					}
					else {
						which_context++;
						continue;
					}
				}
				else if (gray_ptrs[which_context]
					== g_scontexts[which_context].alloc_ptr)
				{
					which_context++;
					continue;
				}
				else if (g_scontexts[which_context].alloc_start 
					<= gray_ptrs[which_context] 
				&& gray_ptrs[which_context]
					< g_scontexts[which_context].alloc_ptr)
				{
					scan_ptr = (Object*)gray_ptrs[which_context];
					gray_ptrs[which_context] =
						g_scontexts[which_context].alloc_ptr;
				}
				else {
					// Unexpected gray_ptr!
					RetailDebugBreak();
					// Make the compiler happy:
					scan_ptr = (Object*)gray_ptrs[which_context];
				}

				if ((BYTE*)scan_ptr < gray_ptrs[which_context]) {
					/*
						If (upon completion) scan_limit is
						non-NULL, this function must obey the
						same contract as ProcessRoot/Work (as
						noted above).
					*/
					CheneyScan(bytesToCollect, bytesCollected,
						g_scontexts, gray_ptrs,
						scan_ptr, gray_ptrs[which_context],
						g_work_list);
				}

				if (gray_ptrs[which_context]
				== g_scontexts[which_context].alloc_ptr) {
					which_context++;
				}

				if (!g_work_list.HasWork()) {
					/*
						If there is no more work, we must
						ensure that all of the contexts are
						scanned -- it's not enough to just go
						through the list once.
					*/
					for (int i = 0; i < GC_NUMBER_OF_CONTEXTS; i++) {
						if (gray_ptrs[i] != g_scontexts[i].alloc_ptr) {
							which_context = 0;
						}
					}
				}
			}
		}
	}

	return bytesCollected;
}


Object*
gc_heap::FindObjectWithLock(BYTE* interior, size_t gap_end)
{
	if (GcSupportsInteriorPointers()) {
		Page* page = GetPage(interior);
		if (page->IsContinued()) {
			page = page->GetFirstPage();
		}

		BYTE* current_byte = page->GetData();
		if (page->IsActive()) {
			if (gap_end == 0) {
				gap_end = page->GetGapEnd();
			}
			if (interior > (page->GetData() + gap_end)) {
				current_byte = (page->GetData() + gap_end);
			}
		}
		
		// XXX How will this work in a multi-processor case?

		Object* obj;

		ASSERT(current_byte <= interior);

		do {
			obj = (Object*)current_byte;
			size_t size = Align(header(obj)->GetSize());

			current_byte += size;
		} while (current_byte <= interior);

		return obj;
	}
	else {
		ASSERT(!"GC does not support interior pointers");
		return NULL;
	}
}

Object*
gc_heap::FindObject(BYTE* interior)
{
	if (GcSupportsInteriorPointers()) {
		Page* page = GetPage(interior);
		size_t gap_end = 0;
		/*
			FIXME this locking is not correct -- if the
			page is removed from active allocation after
			we check, and added to a queue in such a way
			that the prev_page field is set, then
			acquiring a lock breaks PageQueue invariants
		*/
		if (page->IsActive()) {
			gap_end = page->AcquireGapLock();
		}

		Object* result = FindObjectWithLock(interior, gap_end);

		if (gap_end > 0) {
			page->ReleaseGapLock(gap_end);
		}
		return result;
	}
	else {
		ASSERT(!"GC does not support interior pointers");
		return NULL;
	}
}


inline BOOL
gc_heap::HasSpace(Object* obj, alloc_context* scontext, size_t& size)
{
	// SPOONS: + sizeof(ArrayBase) if we intend to scan over empty space
	size = Align(header(obj)->GetSize());
	//return (scontext.alloc_ptr + size + sizeof(ArrayBase)
	return (scontext->alloc_ptr + size + min_obj_size <= scontext->alloc_limit);
}


void
gc_heap::ExtendContext(
	int copy_age,
	alloc_context* scontext,
	BYTE*& gray_ptr,
	BYTE*& scan_limit,
	BOOL reset_limit,
	WorkList& work_list,
	size_t& size)
{
	/*
		No room?  Allocate a new page.  But first clean up the
		old scontext (if any).
	*/
	if(scontext->alloc_start != NULL){
		if (scan_limit == scontext->alloc_ptr) {
			if (reset_limit) {
				// We have already scanned the entire context.
				scan_limit = NULL;
			}
			else {
				/*
					We are currently scanning the
					context, but we've haven't yet
					scanned all there is to be
					scanned -- set the limit to
					the end of the context (so
					that we can continue scanning
					if an adjacent page is
					allocated below).
				*/

				/*
					FIXME this causes problems
					where the adjancent page is
					not allocated to extend the
					current context, but is later
					allocated for a scan context
					of a DIFFERENT age.
				*/
				//scan_limit = scontext->alloc_end;
			}
		}
		else {
			/*
				We are not actively scanning the
				context, so add a task to do so.

				Note that we do not shade any objects
				here: since the context is not pinned,
				the color bits will be ignored during
				scanning.
			*/
			if (scontext->alloc_start <= gray_ptr
			&& gray_ptr < scontext->alloc_ptr) {
				work_list.AddWorkAsRange((Object*)gray_ptr,
					scontext->alloc_ptr);
				gray_ptr = NULL;
			}
			else if (gray_ptr == scontext->alloc_ptr) {
				// Nothing to do.
				gray_ptr = NULL;
			}
			else {
				ASSERT(gray_ptr == scan_limit || gray_ptr == NULL);
				work_list.AddWorkAsRange((Object*)scontext->alloc_start,
					scontext->alloc_ptr);
			}
		}

		Page* gray_page = GetPage(scontext->alloc_start);
		gray_page->SetResidency(scontext->alloc_ptr - scontext->alloc_start);
		// This page contains as much live data "as possible."
		gray_page->SetFull();
	}

	ClearAllocContext(scontext, FALSE);

	Page* current_page;

	Object* first_gap = g_next_gap;
	Object* prev_gap = NULL;

	while (g_next_gap != NULL && 
		header(g_next_gap)->GetSize() < (size + min_obj_size)) {
		prev_gap = g_next_gap;
		g_next_gap = header(g_next_gap)->GetNextGap();
	}

	if (g_next_gap != NULL) {
		if (prev_gap != NULL) {
			header(prev_gap)->SetNextGap(header(g_next_gap)->GetNextGap());
			header(g_next_gap)->SetNextGap(first_gap);
		}

		current_page = GetPage((BYTE*)g_next_gap);

		dprintf(3, ("Using gap %x for replicas.", g_next_gap));

		if (!SetupAllocContext(scontext, g_next_gap, 
			header(g_next_gap)->GetSize(),
			header(g_next_gap)->GetNextGap(),
			NULL,
			current_page, FALSE, FALSE)) {
			ASSERT(!"SetupAllocContext failed unexpectedly.");
		}
		g_next_gap = (Object*)scontext->alloc_next;
		scontext->alloc_next = NULL;
	}
	else {
		// No gaps were large enough -- restore the list but allocate from a page.
		g_next_gap = first_gap;

		// Look for a page in the free list.  	If one exists, then use it.
		if (g_free_queue.IsEmpty()) {
			printf ("Exhausted memory during collection.\n");
			COMPlusThrowOM();
		}

		current_page = g_free_queue.PopPage();
		dprintf(3, ("Removing page %p from the free list.", current_page));

		// Setup the context.
		if (!SetupAllocContext(scontext, NULL, MOSTLY_PAGE_SIZE, NULL, NULL,
			current_page, FALSE, FALSE)) {
			ASSERT(!"SetupAllocContext failed unexpectedly.");
		}

		copied_total_size.addSize(MOSTLY_PAGE_SIZE);
		current_page->SetAge(copy_age);
		current_page->SetStatus(Page::STATUS_REPLICA);
		g_replica_queue.PushPage(current_page);
	}

	/*
		If the scan_limit was NULL, then there wasn't anything
		being scanned.  Therefore, no information is lost by
		setting the limit now.
	*/
	if (scan_limit == NULL) {
		scan_limit = scontext->alloc_ptr;
	}

	// Assumes homogeneous pages:
	ASSERT(scontext->alloc_ptr + size <= scontext->alloc_limit);
}

/*
	Copy the object obj into the proper allocation context.
	Update scan_limit if:

		1) we are currently scanning the context (and
		scan_limit can safely be set to alloc_ptr) OR

		2) we allocate a new and ADJACENT page for the
		alloc_context (in which case, again, scan_limit can
		safely be set to alloc_ptr)

	If scan_limit is set, but a new NON-ADJACENT page is allocated
	for the context, then we either 1) set scan_limit to NULL (if
	reset_limit is TRUE) or 2) make sure that scan_limit
	encompasses the entire OLD context (otherwise).
*/
Object*
gc_heap::CopyObject(
	Object* obj,
	int copy_age,
	alloc_context* scontext,
	BYTE*& gray_ptr,
	BYTE*& scan_limit,
	BOOL reset_limit,
	WorkList& work_list,
	size_t& size)
{
	/*
		FIXME This code doesn't attempt to QWord align when it
		might be beneficial to do so.  However, the original
		Rotor code didn't bother either.
	*/
	if (!HasSpace(obj, scontext, size)) {
		ExtendContext(copy_age, scontext, gray_ptr, scan_limit,
			reset_limit, work_list, size);
	}

	BYTE* result = scontext->alloc_ptr;

	if (GcIsReplicatingCollector()) {
		/*
			In the concurrent case, we must set the
			fowarding pointer BEFORE we read the primary:
			if another thread is modifying this object we
			need to ensure that the write barrier records
			any changes made to the object.
		*/
		header(obj)->SetRelocation((Object*)result);
		header(obj)->SetGray();
		// FIXME REPLICATING SetBlack?
	}

	dprintf (3, ("Copying %p to %p", (BYTE*)obj, result));
	memcopy(result - plug_skew + runtime_skew, 
		(BYTE*)obj - plug_skew + runtime_skew,
		size - runtime_skew);
	copied_live_size.addSize(size - runtime_skew);
	header(result)->ClearGCHeader();

	if (!GcIsReplicatingCollector()) {
		/*
			Otherwise we must set the gray bit ~after~ the
			copy -- otherwise the replica would also be
			marked.
		*/
		header(obj)->SetRelocation((Object*)result);
		header(obj)->SetGray();
		// FIXME replicas on promoted pages should (philosophically) be marked
		//header(obj)->SetBlack();
	}

	if (GetPage(result)->HasStatus(Page::STATUS_PROMOTED)) {
		header((Object*)result)->SetBlack();
	}

	scontext->alloc_ptr += size;

	if (scan_limit == result) {
		/*
			N.B.  This may cause the scan_ptr and
			scan_limit to be on different pages, but that
			those pages will always be adjacent.
		*/
		scan_limit = scontext->alloc_ptr;
	}

	return (Object*)result;
}

Object*
gc_heap::CopyObjectForMark(
	Object* obj,
	int copy_age,
	alloc_context* scontext,
	BYTE*& gray_ptr,
	WorkList& work_list,
	size_t& size)
{
	/*
		FIXME This code doesn't attempt to QWord align when it
		might be beneficial to do so.  However, the original
		Rotor code didn't bother either.
	*/

	if (!HasSpace(obj, scontext, size)) {
		BYTE* scan_limit = (BYTE*)obj + size;
		ExtendContext(copy_age, scontext, gray_ptr, scan_limit, FALSE, work_list, size);
	}

	BYTE* result = scontext->alloc_ptr;

	if (GcIsReplicatingCollector()) {
		/*
			In the concurrent case, we must set the
			fowarding pointer BEFORE we read the primary:
			if another thread is modifying this object we
			need to ensure that the write barrier records
			any changes made to the object.
		*/
		header(obj)->SetRelocation((Object*)result);
		header(obj)->SetGray();
		// FIXME REPLICATING SetBlack?
	}

	dprintf (3, ("Copying %p to %p", (BYTE*)obj, result));
	memcopy(result - plug_skew + runtime_skew, 
		(BYTE*)obj - plug_skew + runtime_skew,
		size - runtime_skew);
	copied_live_size.addSize(size - runtime_skew);
	header(result)->ClearGCHeader();

	if (!GcIsReplicatingCollector()) {
		/*
			Otherwise we must set the gray bit ~after~ the
			copy -- otherwise the replica would also be
			marked.
		*/
		header(obj)->SetRelocation((Object*)result);
		header(obj)->SetGray();
		// FIXME replicas on promoted pages should (philosophically) be marked
		//header(obj)->SetBlack();
	}

	if (GetPage(result)->HasStatus(Page::STATUS_PROMOTED)) {
		header((Object*)result)->SetBlack();
	}

	scontext->alloc_ptr += size;

	return (Object*)result;
}


Page*
gc_heap::GetPage(BYTE* object)
{
	// FIXME spoons use >> rather than /
	return (Page*)(
		(((object - first_page_data) >> MOSTLY_LOG_PAGE_SIZE) * sizeof(Page))
		+ (BYTE*)first_page);
	/*
	return (Page*)((((object - segment->first_page->GetData())
									 / MOSTLY_PAGE_SIZE)
									* sizeof(Page)) + (BYTE*)segment->first_page);
									*/
	/*
	for (Page* current_page = page_list;
			 current_page != NULL;
			 current_page = current_page->GetNextPage())
	{
		if (object >= current_page->GetData()
				&& object < (current_page->GetData() + MOSTLY_PAGE_SIZE))
		{
			return current_page;
		}
	}
	return NULL;
	*/
}

/*
	VM Specific support
*/

#include "excep.h"

/*
	The AllocLock protects g_acontext.  This is only necessary
	because there is currently one context shared by all threads.
*/
inline static void
alloc_spin_lock()
{
	enter_spin_lock (&m_AllocLock);
}

inline void
EnterAllocLock()
{
	#if defined(_X86_) && defined(_MSC_VER)
		__asm {
			inc dword ptr m_AllocLock
			jz gotit
			call alloc_spin_lock
			gotit:		
		}
	#else //_X86_
		alloc_spin_lock();
	#endif //_X86_
}

inline void
LeaveAllocLock()
{
	// Trick this out
	leave_spin_lock (&m_AllocLock);
}

/*
	An explanation of locking for finalization:

	Multiple threads allocate objects.  During the allocation,
	they are serialized by the AllocLock above.  But they release
	that lock before they register the object for finalization.
	That's because there is much contention for the alloc lock,
	but finalization is presumed to be a rare case.

	So registering an object for finalization must be protected by
	the FinalizeLock.

	There is another logical queue that involves finalization.
	When objects registered for finalization become unreachable,
	they are moved from the "registered" queue to the
	"unreachable" queue.  Note that this only happens inside a GC,
	so no other threads can be manipulating either queue at that
	time.  Once the GC is over and threads are resumed, the
	Finalizer thread will dequeue objects from the "unreachable"
	queue and call their finalizers.  This dequeue operation is
	also protected with the finalize lock.

	At first, this seems unnecessary.  Only one thread is ever
	enqueuing or dequeuing on the unreachable queue (either the GC
	thread during a GC or the finalizer thread when a GC is not in
	progress).  The reason we share a lock with threads enqueuing
	on the "registered" queue is that the "registered" and
	"unreachable" queues are interrelated.

	They are actually two regions of a longer list, which can only
	grow at one end.  So to enqueue an object to the "registered"
	list, you actually rotate an unreachable object at the
	boundary between the logical queues, out to the other end of
	the unreachable queue -- where all growing takes place.  Then
	you move the boundary pointer so that the gap we created at
	the boundary is now on the "registered" side rather than the
	"unreachable" side.  Now the object can be placed into the
	"registered" side at that point.  This is much more efficient
	than doing moves of arbitrarily long regions, but it causes
	the two queues to require a shared lock.

	Notice that Enter/LeaveFinalizeLock is not a GC-aware spin
	lock.  Instead, it relies on the fact that the lock will only
	be taken for a brief period and that it will never provoke or
	allow a GC while the lock is held.  This is critical.  If the
	FinalizeLock used enter_spin_lock (and thus sometimes enters
	preemptive mode to allow a GC), then the Alloc client would
	have to GC protect a finalizable object to protect against
	that eventuality.  That is too slow!
*/

BOOL
IsValidObject(BYTE *pObject)
{
	#if defined (VERIFY_HEAP)
		if (!((CObjectHeader*)pObject)->IsFree())
			((Object *) pObject)->Validate();
	#endif
	return(TRUE);
}


#ifdef VERIFY_HEAP
	void
	ValidateObjectMember(Object* obj)
	{
		if (!gc_heap::GcIsReplicatingCollector()
		&& header(obj)->ContainsPointers()){
			size_t size = Align(header(obj)->GetSize());
			TracePointers(method_table (obj), obj, size, child,
				{
					/*
						FIXME spoons should we
						confirm that the child
						is a heap pointer?
					*/
					if (*child != NULL){
						MethodTable *pMT = method_table (*child);
						if (pMT->GetClass()->GetMethodTable() != pMT) {
							RetailDebugBreak();
						}
						/*
							Page* child_page = GetPage((BYTE*)*child);
							if (!child_page->IsLive()) {
								RetailDebugBreak();
							}
						*/
					}
				}
			);
		}
	}
#endif	//VERIFY_HEAP

void
GCHeap::EnableFinalization(void)
{
	SetEvent(hEventFinalizer);
}

BOOL
GCHeap::IsCurrentThreadFinalizer()
{
	return GetThread() == FinalizerThread;
}

Thread*
GCHeap::GetFinalizerThread()
{
	_ASSERTE(FinalizerThread != NULL);
	return FinalizerThread;
}

BOOL
GCHeap::HandlePageFault(void* add)
{
	return FALSE;
}

GCHeap::GCHeap()
{
	//formerly "Static member variables."

	GcInProgress = FALSE;
	m_suspendReason = GCHeap::SUSPEND_OTHER;
	GcThread = 0;
	m_GCThreadAttemptingSuspend = NULL;
	WaitForGCEvent = 0;
	GcCount = 0;

	m_Finalize = 0;
	GcCollectClasses = FALSE;
	m_GCFLock = 0;

	hEventFinalizer = 0;
	hEventFinalizerDone = 0;
	hEventFinalizerToShutDown = 0;
	hEventShutDownToFinalizer = 0;
	fQuitFinalizer = FALSE;
	FinalizerThread = 0;
	UnloadingAppDomain = NULL;
	fRunFinalizersOnUnload = FALSE;
}

GCHeap::~GCHeap()
{
	// SPOONS FIXME implement
}

HRESULT
GCHeap::GlobalInitialize()
{
	HRESULT hr = S_OK;

	g_pGCHeap = new (nothrow) GCHeap;
	if (!g_pGCHeap)
		return E_OUTOFMEMORY;

	return hr;
}

GCHeap*
GCHeap::CreateGCHeap()
{
	// SPOONS FIXME implement for real
	/*
		if (!g_bGCHeapInit){
			// thread safe?
			g_pGCHeap->Initialize();
			g_bGCHeapInit = TRUE;
		}
	*/
	return g_pGCHeap;
}

void
GCHeap::DestroyGCHeap(GCHeap* pGCHeap)
{
	// SPOONS FIXME implement
	// cleanup dynamic_data
	if (ReportGcTime) {
		printf("Total GC = %f\n", gc_time);
		printf("Total MU = %f\n", mutator_time);
	}
}

HRESULT
GCHeap::Shutdown()
{
	gc_heap::Destroy();

	return S_OK;
}


// init the instance heap
HRESULT
GCHeap::Init(size_t)
{
	return S_OK;
}

static CFinalize*
AllocateCFinalize()
{
	return new CFinalize();
}

static HRESULT
AllocateCFinalize(CFinalize **ppCFinalize)
{
	CFinalize *pCFinalize = NULL;

	COMPLUS_TRY
	{
		pCFinalize = AllocateCFinalize();
	}
	COMPLUS_CATCH
	{
	}
	COMPLUS_END_CATCH

	if (pCFinalize == NULL)
		return E_OUTOFMEMORY;

	*ppCFinalize = pCFinalize;

	return S_OK;
}

//System wide initialization
HRESULT
GCHeap::Initialize ()
{
	HRESULT hr = S_OK;

	// thread safe?
	if (g_bGCHeapInit){
		return hr;
	}
	g_bGCHeapInit = TRUE;

	hr = Init (0);
	if (hr != S_OK)
		return hr;

	size_t seg_size = GCHeap::GetValidSegmentSize();

	ClearAllocContext(&g_acontext, TRUE);

	hr = gc_heap::Initialize (this, seg_size);

	if (hr != S_OK)
		return hr;

	if (gc_heap::GcSupportsFinalizers()
	&& (WaitForGCEvent = CreateEvent( 0, TRUE, TRUE, 0 )) != 0){
		// Thread for running finalizers...
		if (FinalizerThreadCreate() != 1){
			hr = E_OUTOFMEMORY;
			return hr;
		}
	}
	else{
		return E_OUTOFMEMORY;
	}

	return hr;
};

/*
	GC callback functions
*/

BOOL
GCHeap::IsPromoted(Object* object, ScanContext* sc)
{
	// FIXME SPOONS SKETCHY
	#if defined (_DEBUG)
		object->Validate(FALSE);
	#endif //_DEBUG
	BYTE* o = (BYTE*)object;
	// FIXME spoons should we check that o is a heap allocated object?
	return !header(o)->IsWhite();
}

void
GCHeap::ShadeObject(Object* obj)
{
	if (gc_heap::GcIsReplicatingCollector()) {
		gc_heap::ShadeObject(obj);
	}
}

unsigned int
GCHeap::WhichGeneration(Object* object)
{
	return CURRENT_GENERATION;
}

BOOL
GCHeap::IsEphemeral(Object* object)
{
	return FALSE;
}

#ifdef VERIFY_HEAP

	// returns TRUE if the pointer is in one of the GC heaps.
	BOOL
	GCHeap::IsHeapPointer(void* p, BOOL small_heap_only)
	{
		BYTE* object = (BYTE*)p;
		// FIXME spoons small_heap_only?
		return gc_heap::IsHeapObject (object);
	}

#endif //VERIFY_HEAP

// FIXME SPOONS this used to be verify only!
BOOL
gc_heap::IsHeapObject (BYTE *object)
{
	return ((BYTE*)object >= (BYTE*)(segment))
		&& ((BYTE*)object < (BYTE*)(segment->committed));
}

BYTE*
gc_heap::CheneyScanChild(
	size_t bytesToCollect,
	size_t& bytesCollected,
	alloc_context* scontexts,
	BYTE** gray_ptrs,
	BOOL scanning,
	Object*& scan_ptr,
	BYTE* scan_limit,
	WorkList& work_list,
	Object* obj,
	BOOL redirect_ptrs,
	Object** child,
	Object** next_ptr,
	Page** next_page)
{
	if (*child != NULL) {
		Page* child_page = GetPage((BYTE*)*child);
		if (child_page->HasStatus(Page::STATUS_PROMOTED)) {		
			if (TRUE /*header(*child)->IsWhite() */) {
				/*
					Object is "white" -- it may be
					a copied or replicated object
					in to-space, or a white a
					pinned object.
				*/
				// FIXME (opt) don't need to set gray if the first branch is taken
				if (next_ptr != NULL && *next_ptr == NULL) {
					//header(*child)->SetBlack();
					*next_ptr = *child;
					*next_page = child_page;
				}
				else {
					//header(*child)->SetGray();
					work_list.AddWorkAsPtr(*child, FALSE);
				}
			}
		}
		else if (child_page->HasStatus(Page::STATUS_EVACUATED)) {
			ASSERT(!header(*child)->IsBlack());
			if (!header(*child)->IsWhite()) {
				/*
					redirect_ptrs is FALSE for
					replicating collection only!
					Recall that objects on pinned
					pages are treated like roots:
					we don't update pointers to
					their children until the end
					of collection.  See similar
					case below.
				*/
				if (redirect_ptrs) {
					*child = header(*child)->GetRelocated();
				}
			}
			else {
				// The object is really white: do the dirty work!
				size_t child_size;
				Object* gray_child;
				
				/*
					Here scan_limit indicates our
					INTENTION to scan -- not what
					already has been scanned.
					Therefore if the context is
					reset (and subsequently
					appears on a non-adjacent
					page) the scan_limit should
					NOT be reset.
				*/
				int age = child_page->GetAge();
				int context_age = min(age, GC_NUMBER_OF_CONTEXTS);

				if (scanning) {
					gray_child = CopyObject(*child, context_age + 1,
						&scontexts[context_age - 1], 
						gray_ptrs[context_age - 1],
						scan_limit, FALSE, 
						work_list, child_size);
				}
				else {
					gray_child = CopyObjectForMark(*child, context_age + 1,
						&scontexts[context_age - 1], 
						gray_ptrs[context_age - 1],
						work_list, child_size);
				}

				if (age == Page::AGE_YOUNG) {
					child_page->SetResidency(
						child_page->GetResidency() + child_size);
				}

				// See note "Recall that objects on pinned..." above.
				if (redirect_ptrs) {
					*child = gray_child;
				}
				
				bytesCollected += child_size;
			}
		}
	}

	return scan_limit;
}

/*
	Scans the region initially defined by the pointers scan_ptr
	and scan_limit.  In this sense the region between these
	pointers can be considered gray.

	If and when bytesCollected >= bytesToCollect, we will abort
	the scan and add any remaining space between scan_ptr and
	scan_limit to the work list.
*/
void
gc_heap::CheneyScan(
	size_t bytesToCollect,
	size_t& bytesCollected,
	alloc_context* scontexts,
	BYTE** gray_ptrs,
	Object* scan_ptr,
	BYTE*& scan_limit,
	WorkList& work_list)
{
	while ((BYTE*)scan_ptr < scan_limit) {
		size_t size = Align(header(scan_ptr)->GetSize());
		BOOL redirect_ptrs = TRUE;

		/*
			if (GetPage((BYTE*)scan_ptr)->HasStatus(Page::STATUS_PROMOTED)) {
				if (!header(scan_ptr)->IsGray()) {
					scan_ptr = (Object*)((BYTE*)scan_ptr + size);
					if (header(scan_ptr)->IsBlack()) {
						black_scanned_size.addSize(size);
					}
					else {
						white_scanned_size.addSize(size);
					}
					continue;
				}
				if (GcIsReplicatingCollector()) {
					redirect_ptrs = FALSE;
				}
			}
		*/
		/*
			This isn't always true -- for instance, if we
			copy into unused gaps on promoted pages, and
			then use the cheney scan to scan those gaps.
		*/
		//ASSERT(GetPage((BYTE*)scan_ptr)->HasStatus(Page::STATUS_REPLICA));

		if (header(scan_ptr)->ContainsPointers()) {
			TracePointers(method_table(scan_ptr), scan_ptr, size, child,
				{
					scan_limit = CheneyScanChild(bytesToCollect, bytesCollected,
						scontexts, gray_ptrs, TRUE,
						scan_ptr, scan_limit,
						work_list, scan_ptr, 
						redirect_ptrs, child);
					if (bytesCollected > bytesToCollect) {
						/*
							Save our scan position.  This
							could cause some fields of
							the current object (i.e.
							scan_ptr) to be rescanned the
							next time around, but since
							they are already redirected,
							they won't be changed in the
							future.
		
							Note that we have not yet set
							the black bit (or cleared the
							gray bit) if this is a pinned
							object -- this ensure that we
							will actually rescan the
							current object.
						*/
						/*
							FIXME if we make scan_ptr
							persistent across increments
							then we don't need to this to
							the work list.
						*/
						// FIXME use WORK_RANGE_INTERIOR_PTR
						work_list.AddWorkAsRange(scan_ptr, scan_limit);
						// Abort the rest of the current scan:
						goto FINISH;
					}
				});
		}

		scan_ptr = (Object*)((BYTE*)scan_ptr + size);
		cheney_scanned_size.addSize(size);
	}

	ASSERT((BYTE*)scan_ptr == scan_limit);	

FINISH:

	BOOL scanning_context = FALSE;
	for (int i = 0; i < GC_NUMBER_OF_CONTEXTS; i++) {
		scanning_context |= (scan_limit == scontexts[i].alloc_ptr);
	}
	if (!scanning_context) {
		scan_limit = NULL;
	}
}

void
gc_heap::ProcessRange(
	Work& work,
	size_t bytesToCollect,
	size_t& bytesCollected,
	alloc_context* scontexts,
	BYTE** gray_ptrs,
	WorkList& work_list)
{
	Object* scan_ptr = work.ptr;
	BYTE* scan_limit = work.limit;

	LoopOverContexts(i, gray_ptrs[i] = NULL);
	
	CheneyScan(bytesToCollect, bytesCollected,
		scontexts, gray_ptrs,
		scan_ptr, scan_limit,
		g_work_list);
}

void
gc_heap::ProcessObject(
	Object* obj,
	size_t bytesToCollect,
	size_t& bytesCollected,
	BYTE** gray_ptrs)
{
	do {
		/*
			It's possible (in an incremental collector)
			that this object has been nullified this the
			time it was shaded.  In that case, just
			continue.
		*/
		if (GcIsReplicatingCollector() && obj == NULL) {
			goto get_more_work;
		}

		Page* page = GetPage((BYTE*)obj);
		ASSERT(page->HasStatus(Page::STATUS_PROMOTED));

		if (!header(obj)->IsBlack()) {
			//process_next:
			header(obj)->SetBlack();


			pinned_live_size.addSize(obj);
			queue_scanned_size.addSize(obj);

			bool redirect_ptrs = !GcIsReplicatingCollector();

			size_t size = Align(header(obj)->GetSize());

			if (header(obj)->ContainsPointers()) {
				//BYTE* end = (BYTE*)obj + size;
		
				/*
					Object* next_ptr = NULL;
					Page* next_page = NULL;
				*/
				

				TracePointers (method_table(obj), obj, size, child,
					/*
					{
						CheneyScanChild(bytesToCollect, bytesCollected,
							scontexts, gray_ptrs, obj, end,
							work_list, obj, redirect_ptrs, child,
							&next_ptr, &next_page);
					} , */
					{
						CheneyScanChild(bytesToCollect, bytesCollected,
							g_scontexts, gray_ptrs, FALSE, obj, NULL /* end */,
							g_work_list, obj, redirect_ptrs, child);
					});

				// FIXME test for bytesCollected <= bytesToCollect!
				/*
					if (next_ptr != NULL) {
						bytesCollected += size;

						obj = next_ptr;
						page = next_page;

						//from_loop++;

						goto process_next;
					}
				*/					
			}
			bytesCollected += size;
		}			

get_more_work:
	
		if (g_work_list.HasPtrWork() && bytesCollected <= bytesToCollect) {
			obj = g_work_list.RemovePtrWork();
		}
		else {
			break;
		}
	} while (TRUE);
}

void
gc_heap::RegisterRoot(Object** root, ScanContext* sc, DWORD flags)
{
	// FIXME longterm: what is CHECK_APP_DOMAIN?
	//ASSERT(!(flags & GC_CALL_CHECK_APP_DOMAIN));

	Object* obj = *root;

	// Don't bother adding roots which are NULL.
	if(obj == NULL)
		return;

	if(!IsHeapObject((BYTE*)obj))
		return;

	dprintf(3,("RegisterRoot" FMT_ADDR "->" FMT_ADDR "flags=%d",
		DBG_ADDR(root), DBG_ADDR(obj), flags));

	Page* page = GetPage((BYTE*)obj);

	if (flags & GCHeap::GC_CALL_PINNED) {
		// Pin the object's page.
		if (!page->HasStatus(Page::STATUS_PROMOTED)) {
			ASSERT(page->HasStatus(Page::STATUS_EVACUATED));
			ASSERT(!page->IsLarge());
			g_evacuated_queue.RemovePage(page);
			page->SetStatus(Page::STATUS_PROMOTED);
			page->SetPromotedReason(Page::PROMOTED_REASON_TEMP);
			g_promoted_queue.PushPage(page);
			dprintf(4, ("page" FMT_ADDR "promoted by pinned root"
				FMT_ADDR "->" FMT_ADDR,
				DBG_ADDR(page), root, obj));
		}
	}

	BOOL interior = FALSE;

	if (GcSupportsInteriorPointers()
	&& (flags & GCHeap::GC_CALL_INTERIOR)) {
		dprintf( 2, ("Found an interior root! %x\n", obj));
		interior = TRUE;
	}
	else {
		ASSERT(!(flags & GCHeap::GC_CALL_INTERIOR));
	}

	/*
		Here we are safe to set the gray bit: we will
		only add roots to the work list if we can
		guarantee that there will be no mutator
		changes between now and the point at which the
		work is removed.
	*/
	if (page->HasStatus(Page::STATUS_PROMOTED)) {
		header(obj)->SetGray();
	}

	if (GcIsReplicatingCollector() && g_stage == INITIAL_ROOT_SCAN) {
		g_work_list.AddWorkAsPtr(obj, interior);
	}
	else {
		g_work_list.AddWorkAsRoot(root, interior);
	}

}

#ifdef VERIFY_HEAP
	void
	gc_heap::VerifyRoot(Object** root, ScanContext* sc, DWORD flags)
	{
		Object* obj = *root;

		if (obj == NULL) {
			return;
		}
	
		if(!IsHeapObject((BYTE*)obj)){
			return;
		}
	
		if (flags & GCHeap::GC_CALL_INTERIOR) {
			// Just validate the entire object.
			obj = FindObjectWithLock((BYTE*)obj);
		}
	
		Page* page = GetPage((BYTE*)obj);
	
		if(sc->verify_page_is_live) {
			if(!(page->HasStatus(Page::STATUS_PROMOTED) 
			|| page->HasStatus(Page::STATUS_EVACUATED))) {
				RetailDebugBreak();
			}
		}
	
		if (!header(obj)->IsFree()) {
			obj->Validate();
		}
	}
	
	void
	gc_heap::VerifyPage(Page* current_page, BOOL verify_page_is_live)
	{
		BYTE* current_byte = NULL;
	
		if (verify_page_is_live) {
			if(!(current_page->HasStatus(Page::STATUS_PROMOTED) 
			|| current_page->HasStatus(Page::STATUS_EVACUATED))) {
				RetailDebugBreak();
			}
		}
		
		if (!current_page->IsContinued()){
			current_byte = current_page->GetData();
		}
		
		if (current_byte == NULL) {
			// This page is unused.
			// FIXME perhaps the page flags should be more indicative.
			return;
		}
		
		// FIXME separate page headers?
		while (current_byte < current_page->GetData() + MOSTLY_PAGE_SIZE) {
			Object* obj = (Object*)current_byte;
			size_t size = Align(header(obj)->GetSize());
			
			/*
				There may be dead objects on live
				pages (i.e.  floating garbage), so we
				can't validate every object (its
				fields may be bogus).  Instead we make
				do with just hopping from one object
				to the next.
			*/
			if (!GcIsReplicatingCollector() 
			&& !header(obj)->IsFree()) {
				obj->Validate();
			}
			
			current_byte += size;
		}
	}
	
	
	/*
		SPOONS: VerifyHeap and DumpHeap might want to know
		about other roots: The ReferenceMarshaller's COM ref
		handle and the handle table's weak handles.
	*/

	void
	gc_heap::VerifyHeap()
	{
		ScanContext sc;
		sc.verify_page_is_live = (g_stage != INITIALIZING);
	
		if (GcIsReplicatingCollector() && GcIsConcurrentCollector()) {
			g_pGCHeap->SuspendEE(GCHeap::SUSPEND_FOR_GC);
		}
	
		dprintf(3,("Verifying Roots"));
		CNameSpace::GcScanStacks(gc_heap::VerifyRoot, &sc);

		dprintf(3,("Verifying handle table"));
		CNameSpace::GcScanStrongHandles(gc_heap::VerifyRoot, &sc);

		if (GcIsReplicatingCollector() && GcIsConcurrentCollector()) {
			g_pGCHeap->RestartEE(FALSE, TRUE);
		}
	
		dprintf(3,("Verifying finalization data"));
		sc.verify_page_is_live = FALSE;
		g_finalize_queue->GcScanRoots(gc_heap::VerifyRoot, &sc);
	
		dprintf(3,("Verifying page lists"));
		for (Page* page = g_free_queue.PeekFirstPage(); page != NULL;
			page = page->GetNextPage())
		{
			ASSERT(page->HasStatus(Page::STATUS_FREE));
		}
	
		ASSERT(total_pages == (g_free_queue.GetLength() +
			g_evacuated_queue.GetLength() +
			g_promoted_queue.GetLength() + 
			g_replica_queue.GetLength() + 
			g_write_queue.GetLength()));
	
		dprintf(3,("Scanning heap object by object"));
	
		BYTE* current_byte = NULL;
	
		for(Page* current_page = g_evacuated_queue.PeekFirstPage();
			current_page != NULL;
			current_page = current_page->GetNextPage())
		{
			VerifyPage(current_page, sc.verify_page_is_live);
		}
	
		for(Page* current_page = g_promoted_queue.PeekFirstPage();
			current_page != NULL;
			current_page = current_page->GetNextPage())
		{
			VerifyPage(current_page, sc.verify_page_is_live);
		}
	}
	
	static void
	DumpObject(int prefix, Object* obj)
	{
		for(int i = 0; i < prefix; i++)
			printf(" ");
	
		if (obj == NULL) {
			printf("NULL \n");
			return;
		}
	
		if (header(obj)->IsGray()) {
			printf("%X *\n", obj);
		}
		else {
			ASSERT(header(obj)->IsWhite());
			header(obj)->SetGray();
			#if _DEBUG
				printf("%X (%s) {\n",
					obj, method_table(obj)->GetClass()->GetDebugClassName());
			#else // !_DEBUG
				printf("%X {\n", obj);
			#endif // _DEBUG
	
			if (header(obj)->ContainsPointers()) {
				size_t size = Align(header(obj)->GetSize());
				TracePointers (method_table(obj), obj, size, child,
					{
						DumpObject(prefix+1, *child);
					}
				);
			}
			for(int i = 0; i < prefix; i++)
				printf(" ");
			printf("}\n");
		}
	}
	
	void
	gc_heap::DumpRoot(Object** root, ScanContext* sc, DWORD flags)
	{
		Object* obj = *root;

		if(!IsHeapObject((BYTE*)obj)){
			return;
		}
	
		if (GcSupportsInteriorPointers() && (flags & GCHeap::GC_CALL_INTERIOR)) {
			// Just validate the entire object.
			obj = FindObjectWithLock((BYTE*)obj);
		}
		else {
			ASSERT(!(flags & GCHeap::GC_CALL_INTERIOR));
		}
	
		DumpObject(0, obj);
	}

	static void
	ClearObjectFlags(int prefix, Object* obj)
	{
		if (obj == NULL) {
			return;
		}
	
		if (!header(obj)->IsWhite()) {
			ASSERT(header(obj)->IsGray());
			header(obj)->SetWhite();
			if (header(obj)->ContainsPointers()) {
				size_t size = Align(header(obj)->GetSize());
				TracePointers (method_table(obj), obj, size, child,
					{
						ClearObjectFlags(prefix, *child);
					}
				);
			}
		}
	}
	
	void
	gc_heap::ClearRootFlags(Object** root, ScanContext* sc, DWORD flags)
	{
		Object* obj = *root;
		
		if(!IsHeapObject((BYTE*)obj)){
			return;
		}
	
		if (GcSupportsInteriorPointers() && (flags & GCHeap::GC_CALL_INTERIOR)) {
			// Just validate the entire object.
			obj = FindObjectWithLock((BYTE*)obj);
		}
		else {
			ASSERT(!(flags & GCHeap::GC_CALL_INTERIOR));
		}
	
		ClearObjectFlags(0, obj);
	}

	void
	gc_heap::DumpHeap()
	{
		ScanContext sc;
		sc.verify_page_is_live = TRUE;
	
		if (GcIsReplicatingCollector() && GcIsConcurrentCollector()) {
			g_pGCHeap->SuspendEE(GCHeap::SUSPEND_FOR_GC);
		}
	
		dprintf(3,("Dumping Roots"));
		CNameSpace::GcScanStacks(gc_heap::DumpRoot, &sc);

		dprintf(3,("Dumping handle table"));
		CNameSpace::GcScanStrongHandles(gc_heap::DumpRoot, &sc);
	
		dprintf(3,("Dumping finalization data"));
		sc.verify_page_is_live = FALSE;
		g_finalize_queue->GcScanRoots(gc_heap::DumpRoot, &sc);
		sc.verify_page_is_live = TRUE;
	
		// Reset the mark bits
		CNameSpace::GcScanStacks(gc_heap::ClearRootFlags, &sc);
	
		CNameSpace::GcScanStrongHandles(gc_heap::ClearRootFlags, &sc);
	
		sc.verify_page_is_live = FALSE;
		g_finalize_queue->GcScanRoots(gc_heap::ClearRootFlags, &sc);
		sc.verify_page_is_live = TRUE;
	
		if (GcIsReplicatingCollector() && GcIsConcurrentCollector()) {
			g_pGCHeap->RestartEE(FALSE, TRUE);
		}
	}
#endif //VERIFY_HEAP

void
gc_heap::Relocate(Object** root, ScanContext* sc, DWORD flags)
{
	Object* obj = *root;
	ptrdiff_t offset = 0;

	if (obj == NULL) {
		return;
	}

	if(!IsHeapObject((BYTE*)obj)){
		return;
	}

	if (GcSupportsInteriorPointers() && (flags & GCHeap::GC_CALL_INTERIOR)) {
		Object* containing_obj = FindObjectWithLock((BYTE*)obj);
		offset = (BYTE*)obj - (BYTE*)containing_obj;
		dprintf(3, ("Relocate " FMT_ADDR "->" FMT_ADDR
			"=" FMT_ADDR "+ %lu flags=%d",
			DBG_ADDR(root), DBG_ADDR(obj),
			DBG_ADDR(containing_obj), (unsigned long)offset,
			flags));
		obj = containing_obj;
	}
	else {
		dprintf(3, ("Relocate %p -> %p, %d", root, obj, flags));
		ASSERT(!(flags & GCHeap::GC_CALL_INTERIOR));
	}

	#if defined (_DEBUG)
		if (!header(obj)->IsFree()) {
			obj->Validate();
		}
	#endif

	if (!header(obj)->IsWhite()){
		// The object is live.	Check if it's pinned.
		if (GetPage((BYTE*)obj)->HasStatus(Page::STATUS_PROMOTED)){
			ASSERT(header(obj)->IsBlack());
			dprintf(3, ("%p Reference to pinned object with size %x",
				obj, header(obj)->GetSize()));
			// Nothing to do.
		}
		else{
			BYTE* relocated = (BYTE*)(header(obj)->GetRelocated());
			dprintf (3, ("%p Reference copied to %p",
				obj, relocated));

			// Update the reference.
			*root = (Object*)(relocated + offset);
		}
	}
	else{
		/*
			Object is dead.  All strong roots should be
			reachable at this point.
		*/
		ASSERT(g_stage != FINAL_ROOT_SCAN);

		// Set the reference to null.
		dprintf (3, ("%p Reference is unreachable", obj));
		*root = NULL;
	}
}

/*static*/ BOOL
GCHeap::IsLargeObject(MethodTable* mt)
{
	return mt->GetBaseSize() > gc_heap::PAGE_FREE_SIZE;
}


/*static*/ size_t
GCHeap::MinimumLargeObjectSize()
{
	return gc_heap::PAGE_FREE_SIZE + 1;
}

/*static*/ BOOL
GCHeap::IsObjectInFixedHeap(Object* pObj)
{
	Page* page = gc_heap::GetPage((BYTE*)pObj);
	return (page->HasStatus(Page::STATUS_PROMOTED)
		&& page->HasPromotedReason(Page::PROMOTED_REASON_PERM));
}

/*static*/ BOOL
GCHeap::IsObjRefValid(BYTE* ptr)
{
	return gc_heap::get_segment() && gc_heap::IsHeapObject(ptr);
	/*
		if (g_lowest_address <= ptr && ptr < g_highest_address)
			return TRUE;
		return FALSE;
	*/
}

Object *
GCHeap::Alloc(DWORD size, DWORD flags)
{
	THROWSCOMPLUSEXCEPTION();

	TRIGGERSGC();
	#ifdef _DEBUG
		Thread* pThread = GetThread();
		if (pThread)
			pThread->SetReadyForSuspension();
	#endif

	Object* newAlloc;

	// SPOONS: no global lock is required!!
	EnterAllocLock();

	if (gc_heap::GcSupportsLargeObjects() && (size <= gc_heap::PAGE_FREE_SIZE)){
		newAlloc = AllocateInContext (size, flags & GC_ALLOC_PINNED);

		LeaveAllocLock();

//		ASSERT (newAlloc);
		if (newAlloc != 0){
			if (gc_heap::GcSupportsFinalizers() && (flags & GC_ALLOC_FINALIZE)) {
				gc_heap::get_finalize_queue()->RegisterForFinalization(newAlloc);
			}
		}
		else {
			printf ("Exhausted memory during allocation (small).\n");
			COMPlusThrowOM();
		}
	}
	else if (gc_heap::GcSupportsLargeObjects()){ /* size > PAGE_FREE_SIZE */
		/*
			We need to find a set of contiguous pages
			large enough for the allocation
		*/
		size_t actual_size = 0;
		newAlloc = AllocateFromPages (size, actual_size);
		LeaveAllocLock();
//		ASSERT (newAlloc);
		if (newAlloc != 0){
			header(newAlloc)->ClearGCHeader();

			//Clear the object
			memclr ((BYTE*)newAlloc - plug_skew, Align(size));
			if (gc_heap::GcSupportsFinalizers() && (flags & GC_ALLOC_FINALIZE)) {
				gc_heap::get_finalize_queue()->RegisterForFinalization(newAlloc);
			}

			// Fill in the unused space.
			// FIXME frag
			if (actual_size > size) {
				header((BYTE*)newAlloc + Align(size))->SetFree(actual_size
					- Align(size));
			}
		}
		else {
			printf ("Exhausted memory during large object allocation.\n");
			COMPlusThrowOM();
		}
	}
	else {
		ASSERT(!"Collector doesn't support large objects");
		newAlloc = NULL;
	}

	assert (Align ((size_t)newAlloc + ALIGNFIXUP) == (size_t)newAlloc + ALIGNFIXUP);
	//dprintf (3, ("Alloc %p", newAlloc));
	return newAlloc;
}

HRESULT
GCHeap::GarbageCollect(int generation, BOOL collect_classes_p)
{
	int gen = (generation < 0)
		? MAX_GENERATION
		: min(generation, MAX_GENERATION);
	GarbageCollectGeneration(gen, collect_classes_p);

	return S_OK;
}

BOOL
GCHeap::GarbageCollectGeneration(unsigned int gen, BOOL collect_classes_p)
{
	// FIXME take the alloc lock before messing with the alloc data strcutures
	enter_spin_lock(gc_heap::GetSharedLock());

	// FIXME FIXME FIXME this is reason the GCHeap is a friend of Collector.
	// This is copy-and-paste from ResetContext!
	if (g_acontext.alloc_start != NULL) {

		if (gc_heap::GcIsReplicatingCollector()) {
			Page* page = gc_heap::GetPage((BYTE*)g_acontext.alloc_start);
			page->ClearActive();
			page->ClearGapEnd();
		}
		// FIXME do the other things that ResetContext does before
		// ClearAllocContext!!
		ClearAllocContext(&g_acontext, FALSE);
	}

	Object* next_gap = (Object*)g_acontext.alloc_next;
	g_acontext.alloc_next = NULL;
		
	/*
		Throw away any gaps that will be evacuated -- these
		are of no use to the collector.
	*/
	while (next_gap != NULL
	&& gc_heap::GetPage((BYTE*)next_gap)->HasStatus(Page::STATUS_EVACUATED)) {
		next_gap = header(next_gap)->GetNextGap();
	}

	Object* last_gap = (Object*)g_acontext.alloc_last;
	g_acontext.alloc_last = NULL;

	/*
		Clear the context and fill in the rest of the context
		with a "free" object.
	*/
	// FIXME this is necessary?
	ClearAllocContext(&g_acontext, FALSE);

	if (!gc_heap::IsInCycle()) {
		gc_heap::StartCycle();
	}
	else {
		ClearAllocContext(&g_rcontext, FALSE);
	}

	gc_heap::GarbageCollect(gc_heap::GC_FULL_COLLECTION, next_gap, last_gap);

	g_acontext.alloc_next = (BYTE*)next_gap;
	g_acontext.alloc_last = (BYTE*)last_gap;
	/*
		if (next_gap != NULL) {
			Page* page = gc_heap::GetPage((BYTE*)next_gap);
			gc_heap::SetupAllocContext(&g_acontext, next_gap, 
				header(next_gap)->GetSize(), 
				header(next_gap)->GetNextGap(),
				page, TRUE, TRUE);
		}
	*/

	leave_spin_lock(gc_heap::GetSharedLock());
	return TRUE;
}

size_t
GCHeap::GetTotalBytesInUse()
{
	return ApproxTotalBytesInUse();
}

size_t
GCHeap::ApproxTotalBytesInUse(BOOL small_heap_only)
{
	size_t totsize = 0;
	enter_spin_lock (gc_heap::GetSharedLock());

	totsize = (heap_segment_allocated (gc_heap::get_segment())
		- heap_segment_mem (gc_heap::get_segment()));

	leave_spin_lock (gc_heap::GetSharedLock());
	return totsize;
}

/*
	The spec for this one isn't clear.  This function returns the
	size that can be allocated without triggering a GC of any
	kind.
*/
size_t
GCHeap::ApproxFreeBytes()
{
	enter_spin_lock(gc_heap::GetSharedLock());

	size_t res = g_acontext.alloc_limit - g_acontext.alloc_ptr;

	leave_spin_lock(gc_heap::GetSharedLock());

	return res;
}

// Verify the segment size is valid.
BOOL
GCHeap::IsValidSegmentSize(size_t cbSize)
{
	/* return (power_of_two_p(cbSize) && (cbSize >> 20)); */
	return TRUE;
}

/*
	NB the INITIAL_ALLOC code may no longer be used because of the
	default for GCSegmentSize in eeconfig.cpp:/sync.
*/

// XXX This is a silly constant
#define INITIAL_ALLOC (1024*1024*32)

// Get the segment size to use, making sure it conforms.
size_t
GCHeap::GetValidSegmentSize()
{
	size_t seg_size = g_pConfig->GetSegmentSize();
	if (!GCHeap::IsValidSegmentSize(seg_size))
		seg_size = INITIAL_ALLOC;
	seg_size = align_on_page(seg_size);
	return (seg_size);
}

void
GCHeap::SetReservedVMLimit(size_t vmlimit)
{
	//gc_heap::reserved_memory_limit = vmlimit;
}

Object*
GCHeap::GetNextFinalizableObject()
{
	return gc_heap::get_finalize_queue()->GetNextFinalizableObject();
}

size_t
GCHeap::GetNumberFinalizableObjects()
{
	return gc_heap::get_finalize_queue()->GetNumberFinalizableObjects();
}

size_t
GCHeap::GetFinalizablePromotedCount()
{
	return gc_heap::get_finalize_queue()->GetPromotedCount();
}

BOOL
GCHeap::FinalizeAppDomain(AppDomain* pDomain, BOOL fRunFinalizers)
{
	return gc_heap::get_finalize_queue()->
		FinalizeAppDomain (pDomain, fRunFinalizers);
}

void
GCHeap::SetFinalizeQueueForShutdown(BOOL fHasLock)
{
	gc_heap::get_finalize_queue()->SetSegForShutDown(fHasLock);
}

/*
	Finalized class tracking
*/

void
GCHeap::RegisterForFinalization(Object* obj)
{
	if (gc_heap::GcSupportsFinalizers()) {
		if (((obj->GetHeader()->GetBits()) & BIT_SBLK_FINALIZER_RUN)){
			//just reset the bit
			obj->GetHeader()->ClrBit(BIT_SBLK_FINALIZER_RUN);
		}
		else{
			gc_heap::get_finalize_queue()->RegisterForFinalization (obj);
		}
	}
	else {
		ASSERT(FALSE && "Finalizers not supported!");
	}
}

void
GCHeap::SetFinalizationRun(Object* obj)
{
	if (gc_heap::GcSupportsFinalizers()) {
		obj->GetHeader()->SetBit(BIT_SBLK_FINALIZER_RUN);
	}
	else {
		ASSERT(FALSE && "Finalizers not supported!");
	}
}

/*
	Support for finalization
*/

inline unsigned int
gen_segment(int gen)
{
	return (NUMBERGENERATIONS - gen - 1);
}

CFinalize::CFinalize()
{
	THROWSCOMPLUSEXCEPTION();

	m_Array = new(Object*[100]);

	if (!m_Array){
		ASSERT(m_Array);
		COMPlusThrowOM();
	}
	m_EndArray = &m_Array[100];

	// min_generation?
	for (unsigned int i =0; i < NUMBERGENERATIONS+2; i++){
		m_FillPointers[i] = m_Array;
	}
	m_PromotedCount = 0;
	lock = -1;
}

CFinalize::~CFinalize()
{
	delete m_Array;
}

int
CFinalize::GetPromotedCount()
{
	return m_PromotedCount;
}

inline void
CFinalize::EnterFinalizeLock()
{
	_ASSERTE(dbgOnly_IsSpecialEEThread()
		|| GetThread() == 0
		|| GetThread()->PreemptiveGCDisabled());

retry:		
	if (FastInterlockExchange (&lock, 0) >= 0){
		unsigned int i = 0;
		while (lock >= 0){
			if (++i & 7)
				__SwitchToThread(0);
			else
				__SwitchToThread(5);
		}
		goto retry;
	}
}

inline void
CFinalize::LeaveFinalizeLock()
{
	_ASSERTE(dbgOnly_IsSpecialEEThread()
		|| GetThread() == 0
		|| GetThread()->PreemptiveGCDisabled());

	lock = -1;
}

void
CFinalize::RegisterForFinalization(Object* obj)
{
	THROWSCOMPLUSEXCEPTION();

	EnterFinalizeLock();
	// Adjust gen
	unsigned int dest = gen_segment (CURRENT_GENERATION);

	Object*** s_i = &m_FillPointers [NUMBERGENERATIONS];
	if ((*s_i) == m_EndArray){
		if (!GrowArray()){
			LeaveFinalizeLock();
			COMPlusThrowOM();
		}
	}
	Object*** end_si = &m_FillPointers[dest];
	do{
		//is the segment empty?
		if (!(*s_i == *(s_i-1))){
			//no, swap the end elements.
			*(*s_i) = *(*(s_i-1));
		}
		//increment the fill pointer
		(*s_i)++;
		//go to the next segment.
		s_i--;
	} while (s_i > end_si);

	/*
		We have reached the destination segment store the
		object
	*/
	**s_i = obj;
	// increment the fill pointer
	(*s_i)++;

	if (g_fFinalizerRunOnShutDown) {
		// Adjust boundary for segments so that GC will keep objects alive.
		SetSegForShutDown(TRUE);
	}

	LeaveFinalizeLock();
}

Object*
CFinalize::GetNextFinalizableObject()
{
	Object* obj = 0;
	//serialize
	EnterFinalizeLock();
	if (!IsSegEmpty(NUMBERGENERATIONS)){
		obj = *(--m_FillPointers [NUMBERGENERATIONS]);

	}
	LeaveFinalizeLock();
	return obj;
}

void
CFinalize::SetSegForShutDown(BOOL fHasLock)
{
	int i;

	if (!fHasLock)
		EnterFinalizeLock();
	// min_generation?
	for (i = 0; i < NUMBERGENERATIONS; i++) {
		m_FillPointers[i] = m_Array;
	}
	if (!fHasLock)
		LeaveFinalizeLock();
}

size_t
CFinalize::GetNumberFinalizableObjects()
{
	// FIXME finalizers this still seems sketchy
	return m_FillPointers[NUMBERGENERATIONS]
		- (g_fFinalizerRunOnShutDown
			? m_Array
			: m_FillPointers[NUMBERGENERATIONS-1]);
}

BOOL
CFinalize::FinalizeAppDomain(AppDomain* pDomain, BOOL fRunFinalizers)
{
	BOOL finalizedFound = FALSE;

	unsigned int startSeg = gen_segment (MAX_GENERATION);

	EnterFinalizeLock();

	//reset the N+2 segment to empty
	m_FillPointers[NUMBERGENERATIONS+1] = m_FillPointers[NUMBERGENERATIONS];

	for (unsigned int Seg = startSeg; Seg < NUMBERGENERATIONS; Seg++){
		Object** endIndex = Seg ? m_FillPointers [Seg-1] :m_Array;
		for (Object** i = m_FillPointers [Seg]-1; i >= endIndex ;i--){
			Object* obj = *i;

			/*
				Objects are put into the finalization
				queue before they are complete (ie
				their methodtable may be null) so we
				must check that the object we found
				has a method table before checking if
				it has the index we are looking for.
				If the methodtable is null, it can't
				be from the unloading domain, so skip
				it.
			*/
			if (obj->GetMethodTable() == NULL)
				continue;

			// eagerly finalize all objects except those that may be agile.
			if (obj->GetAppDomainIndex() != pDomain->GetIndex())
				continue;

			if (obj->GetMethodTable()->IsAgileAndFinalizable()){
				/*
					If an object is both agile &
					finalizable, we leave it in
					the finalization queue during
					unload.  This is OK, since
					it's agile.  Right now only
					threads can be this way, so if
					that ever changes, change the
					assert to just continue if not
					a thread.
				*/
				_ASSERTE(obj->GetMethodTable() == g_pThreadClass);

				/*
					However, an unstarted thread
					should be finalized.  It could
					be holding a delegate in the
					domain we want to unload.
					Once the thread has been
					started, its delegate is
					cleared so only unstarted
					threads are a problem.
				*/
				Thread *pThread =
					((THREADBASEREF)ObjectToOBJECTREF(obj))->GetInternal();
				if (! pThread || ! pThread->IsUnstarted())
					continue;
			}

			if (!fRunFinalizers
			|| (obj->GetHeader()->GetBits()) & BIT_SBLK_FINALIZER_RUN){
				/*
					remove the object because we don't want to
					run the finalizer
				*/
				MoveItem (i, Seg, NUMBERGENERATIONS+2);
				/*
					Reset the bit so it will be put back on the queue
					if resurrected and re-registered.
				*/
				obj->GetHeader()->ClrBit (BIT_SBLK_FINALIZER_RUN);
			}
			else{
				finalizedFound = TRUE;
				MoveItem (i, Seg, NUMBERGENERATIONS);
			}
		}
	}

	LeaveFinalizeLock();

	return finalizedFound;
}

void
CFinalize::MoveItem(
	Object** fromIndex,
	unsigned int fromSeg,
	unsigned int toSeg)
{
	int step;
	ASSERT (fromSeg != toSeg);
	if (fromSeg > toSeg)
		step = -1;
	else
		step = +1;
	// Place the element at the boundary closest to dest
	Object** srcIndex = fromIndex;
	for (unsigned int i = fromSeg; i != toSeg; i+= step){
		Object**& destFill = m_FillPointers[i+(step - 1 )/2];
		Object** destIndex = destFill - (step + 1)/2;
		if (srcIndex != destIndex){
			Object* tmp = *srcIndex;
			*srcIndex = *destIndex;
			*destIndex = tmp;
		}
		destFill -= step;
		srcIndex = destIndex;
	}
}

void
CFinalize::GcScanRoots(promote_func* fn, ScanContext* sc)
{
	//scan the finalization queue
	Object** startIndex = m_FillPointers[NUMBERGENERATIONS-1];
	Object** stopIndex	= m_FillPointers[NUMBERGENERATIONS];
	for (Object** po = startIndex; po < stopIndex; po++){
		logRoot(po, "CFinalize::GcScanRoots");
		(*fn)(po, sc, 0);
	}
}

BOOL
CFinalize::ScanForFinalization(promote_func* fn, ScanContext* sc)
{
	BOOL finalizedFound = FALSE;
	m_PromotedCount = 0;

	//start with gen and explore all the younger generations.
	unsigned int startSeg = gen_segment (CONDEMN_GEN_NUM);
	//if (passNumber == 1)
	{
		//reset the N+2 segment to empty
		m_FillPointers[NUMBERGENERATIONS+1] = m_FillPointers[NUMBERGENERATIONS];
		for (unsigned int Seg = startSeg; Seg < NUMBERGENERATIONS; Seg++){
			Object** endIndex = Seg ? m_FillPointers [Seg-1] : m_Array;
			for (Object** i = m_FillPointers [Seg]-1; i >= endIndex ;i--){
				Object* obj = *i;
				if (!GCHeap::IsPromoted (obj, sc)){
					if ((obj->GetHeader()->GetBits()) & BIT_SBLK_FINALIZER_RUN){
						/*
							remove the object because we don't want to
							run the finalizer
						*/
						MoveItem (i, Seg, NUMBERGENERATIONS+2);
						/*
							Reset the bit so it will be put back on the queue
							if resurrected and re-registered.
						*/
						obj->GetHeader()->ClrBit (BIT_SBLK_FINALIZER_RUN);

					}
					else{
						m_PromotedCount++;
						finalizedFound = TRUE;
						MoveItem (i, Seg, NUMBERGENERATIONS);
					}

				}
			}
		}
	}
	if (finalizedFound){
		//Promote the f-reachable objects
		GcScanRoots(fn, sc);
		SetEvent(g_pGCHeap->hEventFinalizer);
	}

	return finalizedFound;
}

//Relocates all of the objects in the finalization array
void
CFinalize::RelocateFinalizationData(promote_func* fn, ScanContext* sc)
{
	//ScanContext sc;
	//sc.condemn_gen_num = condemn_gen_num;
	unsigned int Seg = gen_segment(CONDEMN_GEN_NUM);

	Object** startIndex = Seg ? m_FillPointers [Seg-1] : m_Array;
	for (Object** po = startIndex; po < m_FillPointers [NUMBERGENERATIONS];po++){
		if (GCHeap::IsPromoted (*po, sc)) {
			logRoot(po, "CFinalize::RelocateFinalizationData");
			(*fn)(po, sc, 0);
		}
	}
}

BOOL
CFinalize::GrowArray()
{
	size_t oldArraySize = (m_EndArray - m_Array);
	size_t newArraySize = (oldArraySize* 12)/10;
	WS_PERF_SET_HEAP(GC_HEAP);
	Object** newArray = new(Object*[newArraySize]);
	if (!newArray){
		/*
			It's not safe to throw here, because of the
			FinalizeLock.  Tell our caller to throw for
			us.
		*/
		ASSERT (newArray);
		return FALSE;
	}
	WS_PERF_UPDATE("GC:CRFinalizeGrowArray",
		sizeof(Object*)*newArraySize, newArray);
	memcpy(newArray, m_Array, oldArraySize*sizeof(Object*));

	//adjust the fill pointers
	// min_generation?
	for (unsigned i = 0; i <= NUMBERGENERATIONS+1; i++){
		m_FillPointers [i] += (newArray - m_Array);
	}
	delete m_Array;
	m_Array = newArray;
	m_EndArray = &m_Array [newArraySize];

	return TRUE;
}


#if 0 && defined (VERIFY_HEAP)
	void
	CFinalize::CheckFinalizerObjects()
	{
		/*
			Variable i must be of type size_t or
			dereferencing of m_FillPointers with negative
			index won't work on all platforms
			min_generation?
		*/
		for (size_t i = 0; i < NUMBERGENERATIONS; i++){
			Object **startIndex = (i > 0) ? m_Array :m_FillPointers[i-1];
			Object **stopIndex	= m_FillPointers[i];
	
			for (Object **po = startIndex; po > stopIndex; po++){
				if (g_pGCHeap->WhichGeneration (*po)
				< (NUMBERGENERATIONS - i -1))
					RetailDebugBreak ();
				(*po)->Validate();
			}
		}
	}
#endif
