// ==++==
//
//
//		Copyright (c) 2002 Microsoft Corporation.	 All rights reserved.
//
//		The use and distribution terms for this software are contained in the file
//		named license.txt, which can be found in the root of this distribution.
//		By using this software in any fashion, you are agreeing to be bound by the
//		terms of this license.
//
//		You must not remove this notice, or any other, from this software.
//
//
// ==--==
// GCSCAN.H
/*
	GC Root Scanning
*/

#ifndef _GCSCAN_H_
#define _GCSCAN_H_

#include "gc.h"

typedef void promote_func(
	Object** root,
	ScanContext* sc,
	DWORD flags);

void PromoteCarefully(
	promote_func fn,
	Object** root,
	ScanContext* sc,
	DWORD flags = GCHeap::GC_CALL_INTERIOR);

typedef struct{
	promote_func* f;
	ScanContext* sc;
} GCCONTEXT;

inline void
logRoot(Object** root, char* what)
{
	#ifdef LOGGING
		if(LoggingOn(LF_GCROOTS, LL_INFO10000)){
			#ifdef _DEBUG
				LogSpewAlways(
					"root" FMT_ADDR "->" FMT_ADDR "(%s)\n",
					DBG_ADDR(root), DBG_ADDR(*root), what);
			#else
				LogSpewAlways(
					"root" FMT_ADDR "->" FMT_ADDR "\n",
					DBG_ADDR(root), DBG_ADDR(*root));
			#endif
		}
	#endif
}

class CNameSpace{
	public:

	static void GcScanStacks(
		promote_func* fn,
		ScanContext* sc);

	static void GcScanStrongHandles(
		promote_func* fn,
		ScanContext* sc);

	static void GcScanShortWeakHandles(
		promote_func* fn,
		ScanContext* sc);

	static void GcScanLongWeakHandles(
		promote_func* fn,
		ScanContext* sc);

	#ifdef GC_PROFILING
		static void GcScanHandlesForProfiler();
	#endif // GC_PROFILING

	// post-promotions callback
	static void GcPromotionsGranted();

	// post-promotions callback some roots were demoted
	static void GcDemote(
		ScanContext* sc);

	// post-gc callback.
	static void
	GcDoneAndThreadsResumed()
	{
		_ASSERTE(0);
	}

	static size_t AskForMoreReservedMemory(
		size_t old_size,
		size_t need_size);
};

/*
	Allocation Helpers
*/

// The main Array allocation routine, can do multi-dimensional
OBJECTREF AllocateArrayEx(
	TypeHandle arrayClass,
	INT32* pArgs,
	DWORD dwNumArgs,
	BOOL bAllocateInLargeHeap = FALSE);

// Optimized verion of above
OBJECTREF FastAllocatePrimitiveArray(
	MethodTable* arrayType,
	DWORD cElements,
	BOOL bAllocateInLargeHeap = FALSE);

#ifdef _DEBUG
	extern void Assert_GCDisabled();
#endif //_DEBUG

#ifdef _X86_

	/*
		for x86, we generate efficient allocators for some special
		cases these are called via inline wrappers that call the
		generated allocators via function pointers.
	*/

	// Create a SD array of primitive types
	typedef Object* (F_CALL_CONV * FastPrimitiveArrayAllocatorFuncPtr)(
		CorElementType type,
		DWORD cElements);

	extern FastPrimitiveArrayAllocatorFuncPtr fastPrimitiveArrayAllocator;

	// The fast version always allocates in the normal heap
	inline OBJECTREF
	AllocatePrimitiveArray(CorElementType type, DWORD cElements)
	{
		#ifdef _DEBUG
			Assert_GCDisabled();
		#endif //_DEBUG

		return OBJECTREF(fastPrimitiveArrayAllocator(type,cElements));
	}

	/*
		The slow version is distinguished via overloading by an
		additional parameter
	*/
	OBJECTREF AllocatePrimitiveArray(
		CorElementType type,
		DWORD cElements,
		BOOL bAllocateInLargeHeap);

	// Allocate SD array of object pointers
	typedef Object* (F_CALL_CONV* FastObjectArrayAllocatorFuncPtr)(
		MethodTable* ElementType,
		DWORD cElements);

	extern FastObjectArrayAllocatorFuncPtr fastObjectArrayAllocator;

	// The fast version always allocates in the normal heap
	inline OBJECTREF
	AllocateObjectArray(DWORD cElements, TypeHandle ElementType)
	{

		#ifdef _DEBUG
			Assert_GCDisabled();
		#endif //_DEBUG

		return OBJECTREF(
			fastObjectArrayAllocator(ElementType.AsMethodTable(),
				cElements));
	}

	/*
		The slow version is distinguished via overloading by an
		additional parameter
	*/
	OBJECTREF AllocateObjectArray(
		DWORD cElements,
		TypeHandle ElementType,
		BOOL bAllocateInLargeHeap);

	// Allocate string
	typedef StringObject* ( F_CALL_CONV* FastStringAllocatorFuncPtr)(
		DWORD cchArrayLength);

	extern FastStringAllocatorFuncPtr fastStringAllocator;

	inline STRINGREF
	AllocateString(DWORD cchArrayLength)
	{
		#ifdef _DEBUG
			Assert_GCDisabled();
		#endif //_DEBUG
		return STRINGREF(fastStringAllocator(cchArrayLength-1));
	}

	// The slow version, implemented in gcscan.cpp
	STRINGREF SlowAllocateString(
		DWORD cchArrayLength);

#else


	/*
		On other platforms, go to the (somewhat less efficient)
		implementations in gcscan.cpp
	*/

	// Create a SD array of primitive types
	OBJECTREF AllocatePrimitiveArray(
		CorElementType type,
		DWORD cElements,
		BOOL bAllocateInLargeHeap = FALSE);

	// Allocate SD array of object pointers
	OBJECTREF AllocateObjectArray(
		DWORD cElements,
		TypeHandle ElementType,
		BOOL bAllocateInLargeHeap = FALSE);

	STRINGREF SlowAllocateString(
		DWORD cchArrayLength);

	inline STRINGREF
	AllocateString(DWORD cchArrayLength)
	{
		return SlowAllocateString( cchArrayLength );
	}

#endif

OBJECTREF DupArrayForCloning(
	BASEARRAYREF pRef,
	BOOL bAllocateInLargeHeap = FALSE);

OBJECTREF AllocateUninitializedStruct(
	MethodTable* pMT);

/*
	The JIT requests the EE to specify an allocation helper to use at
	each new-site.  The EE makes this choice based on whether context
	boundaries may be involved, whether the type is a COM object,
	whether it is a large object, whether the object requires
	finalization.  These functions will throw OutOfMemoryException so
	don't need to check for NULL return value from them.
*/

OBJECTREF AllocateObject(
	MethodTable* pMT);

OBJECTREF AllocateObjectSpecial(
	MethodTable* pMT);

OBJECTREF FastAllocateObject(
	MethodTable* pMT);

#ifdef MAXALLOC
	AllocRequestManager* GetGCAllocManager();
#endif

#endif // _GCSCAN_H_
