// ==++==
//
//
//		Copyright (c) 2002 Microsoft Corporation.	 All rights reserved.
//
//		The use and distribution terms for this software are contained in the file
//		named license.txt, which can be found in the root of this distribution.
//		By using this software in any fashion, you are agreeing to be bound by the
//		terms of this license.
//
//		You must not remove this notice, or any other, from this software.
//
//
// ==--==

#include "common.h"
#include "dbginterface.h"

#include "gcsmppriv.h"

#include "remoting.h"
#include "comsynchronizable.h"
#include "comsystem.h"

#include "syncclean.hpp"


// The contract between GC and the EE, for starting and finishing a GC is as follows:
//
//			LockThreadStore
//			SetGCInProgress
//			SuspendEE
//
//			... perform the GC ...
//
//			SetGCDone
//			RestartEE
//			UnlockThreadStore
//
// Note that this is intentionally *not* symmetrical.	 The EE will assert that the
// GC does most of this stuff in the correct sequence.

void GCHeap::RestartEE(BOOL bFinishedGC, BOOL SuspendSucceded)
{
		if (g_fSuspendOnShutdown) {
				// We are shutting down.	The finalizer thread has suspended EE.
				// There will only be one thread running inside EE: either the shutdown
				// thread or the finalizer thread.

#ifdef PROFILING_SUPPORTED
								g_profControlBlock.inprocState = ProfControlBlock::INPROC_PERMITTED;
#endif

				_ASSERTE (g_fEEShutDown);
				m_suspendReason = SUSPEND_FOR_SHUTDOWN;
				return;
		}

#ifdef TIME_CPAUSE
		printf ("Pause time: %d\n", GetCycleCount32() - cstart);
#endif //TIME_CPAUSE

		// SetGCDone();
		SyncClean::CleanUp();
		GcInProgress= FALSE;
		ThreadStore::TrapReturningThreads(FALSE);
		GcThread		= 0;
		SetEvent( WaitForGCEvent );
		_ASSERTE(ThreadStore::HoldingThreadStore());

		Thread::SysResumeFromGC(bFinishedGC, SuspendSucceded);
}

void GCHeap::SuspendEE(SUSPEND_REASON reason)
{
		BOOL gcOnTransitions;

#ifdef TIME_CPAUSE
		cstart = GetCycleCount32();
#endif //TIME_CPAUSE

		if (g_fSuspendOnShutdown) {
				// We are shutting down.	The finalizer thread has suspended EE.
				// There will only be one thread running inside EE: either the shutdown
				// thread or the finalizer thread.
#ifdef PROFILING_SUPPORTED
								if (reason == GCHeap::SUSPEND_FOR_GC || reason == GCHeap::SUSPEND_FOR_GC_PREP)
												g_profControlBlock.inprocState = ProfControlBlock::INPROC_FORBIDDEN;
#endif

				_ASSERTE (g_fEEShutDown);
				m_suspendReason = reason;
				return;
		}

		LOG((LF_SYNC, INFO3, "Suspending the runtime for reason %d\n", reason));

		// lock the thread store which could take us out of our must
		// complete
		// Need the thread store lock here.	 We take this lock before the thread
		// lock to avoid a deadlock condition where another thread suspends this
		// thread while it holds the heap lock.	 While the thread store lock is
		// held, threads cannot be suspended.
		gcOnTransitions = GC_ON_TRANSITIONS(FALSE);				 // dont do GC for GCStress 3

		Thread* pCurThread = GetThread();

		// Set variable to indicate that this thread is preforming a true GC
		// This is needed to overcome deadlock in taking the ThreadStore lock
		if (reason == GCHeap::SUSPEND_FOR_GC || reason == GCHeap::SUSPEND_FOR_GC_PREP)
		{
				m_GCThreadAttemptingSuspend = pCurThread;

		}

retry_for_debugger:
		ThreadStore::LockThreadStore(reason);

		if (ThreadStore::s_hAbortEvt != NULL &&
				(reason == GCHeap::SUSPEND_FOR_GC || reason == GCHeap::SUSPEND_FOR_GC_PREP))
		{
				LOG((LF_SYNC, INFO3, "GC thread is backing out the suspend abort event.\n"));
				ThreadStore::s_hAbortEvt = NULL;

				LOG((LF_SYNC, INFO3, "GC thread is signalling the suspend abort event.\n"));
				SetEvent(ThreadStore::s_hAbortEvtCache);
		}

		// Set variable to indicate that this thread is attempting the suspend because it
		// needs to perform a GC and, as such, it holds GC locks.
		if (reason == GCHeap::SUSPEND_FOR_GC || reason == GCHeap::SUSPEND_FOR_GC_PREP)
		{
				m_GCThreadAttemptingSuspend = NULL;
		}

		{
				// suspend for GC, set in progress after suspending
				// threads which have no must complete
				ResetEvent( WaitForGCEvent );
				// SetGCInProgress();
				{
						GcThread = pCurThread;
						ThreadStore::TrapReturningThreads(TRUE);
						m_suspendReason = reason;

#ifdef PROFILING_SUPPORTED
												if (reason == GCHeap::SUSPEND_FOR_GC || reason == GCHeap::SUSPEND_FOR_GC_PREP)
																g_profControlBlock.inprocState = ProfControlBlock::INPROC_FORBIDDEN;
#endif // PROFILING_SUPPORTED

						GcInProgress= TRUE;
				}

				HRESULT hr;
				{
						_ASSERTE(ThreadStore::HoldingThreadStore() || g_fProcessDetach);
						hr = Thread::SysSuspendForGC(reason);
						ASSERT( hr == S_OK || hr == ERROR_TIMEOUT);
				}

				// If the debugging services are attached, then its possible
				// that there is a thread which appears to be stopped at a gc
				// safe point, but which really is not. If that is the case,
				// back off and try again.

				// If this is not the GC thread and another thread has triggered
				// a GC, then we may have bailed out of SysSuspendForGC, so we
				// must resume all of the threads and tell the GC that we are
				// at a safepoint - since this is the exact same behaviour
				// that the debugger needs, just use it's code.
				if ((hr == ERROR_TIMEOUT)
#ifdef DEBUGGING_SUPPORTED
						 || (CORDebuggerAttached() &&
								 g_pDebugInterface->ThreadsAtUnsafePlaces())
#endif // DEBUGGING_SUPPORTED
						)
				{
						// In this case, the debugger has stopped at least one
						// thread at an unsafe place.	 The debugger will usually
						// have already requested that we stop.	 If not, it will
						// either do so shortly -- or resume the thread that is
						// at the unsafe place.
						//
						// Either way, we have to wait for the debugger to decide
						// what it wants to do.
						//
						// Note: we've still got the more_space_lock lock held.

						LOG((LF_GCROOTS | LF_GC | LF_CORDB,
								 LL_INFO10,
								 "***** Giving up on current GC suspension due "
								 "to debugger or timeout *****\n"));

						if (ThreadStore::s_hAbortEvtCache == NULL)
						{
								LOG((LF_SYNC, INFO3, "Creating suspend abort event.\n"));
								ThreadStore::s_hAbortEvtCache = CreateEvent(NULL, TRUE, FALSE, NULL);
						}

						LOG((LF_SYNC, INFO3, "Using suspend abort event.\n"));
						ThreadStore::s_hAbortEvt = ThreadStore::s_hAbortEvtCache;
						ResetEvent(ThreadStore::s_hAbortEvt);

						// Mark that we're done with the gc, just like at the
						// end of this method.
						RestartEE(FALSE, FALSE);

						LOG((LF_GCROOTS | LF_GC | LF_CORDB,
								 LL_INFO10, "The EE is free now...\n"));

						// Check if we're ready to go to suspend.
						if (pCurThread && pCurThread->CatchAtSafePoint())
						{
								_ASSERTE(pCurThread->PreemptiveGCDisabled());
								pCurThread->PulseGCMode();	// Go suspend myself.
						}
						else
						{
								__SwitchToThread (0); // Wait a little while, before retrying.
						}

						goto retry_for_debugger;
				}
		}
		GC_ON_TRANSITIONS(gcOnTransitions);
}

void CallFinalizer(Object* obj)
{

		MethodTable			*pMT = obj->GetMethodTable();
		LOG((LF_GC, LL_INFO1000, "Finalizing " LOG_OBJECT_CLASS(obj)));

		_ASSERTE(GetThread()->PreemptiveGCDisabled());
		// if we don't have a class, we can't call the finalizer
		// if the object has been marked run as finalizer run don't call either
		if (pMT)
		{
				if (!((obj->GetHeader()->GetBits()) & BIT_SBLK_FINALIZER_RUN))
				{
						if (pMT->IsContextful())
						{
								Object *proxy = OBJECTREFToObject(CRemotingServices::GetProxyFromObject(ObjectToOBJECTREF(obj)));

								_ASSERTE(proxy && "finalizing an object that was never wrapped?????");
								if (proxy == NULL)
								{
										// Quite possibly the app abruptly shutdown while a proxy
										// was being setup for a contextful object. We will skip
										// finalizing this object.
										_ASSERTE (g_fEEShutDown);
										return;
								}
								else
								{
										// This saves us from the situation where an object gets GC-ed
										// after its Context.
										Object* stub = (Object *)proxy->GetPtrOffset(CTPMethodTable::GetOffsetOfStubData());
										Context *pServerCtx = (Context *) stub->UnBox();
										// Check if the context is valid
										if (!Context::ValidateContext(pServerCtx))
										{
												// Since the server context is gone (GC-ed)
												// we will associate the server with the default
												// context for a good faith attempt to run
												// the finalizer
												// We want to do this only if we are using RemotingProxy
												// and not for other types of proxies (eg. SvcCompPrxy)
												OBJECTREF orRP = ObjectToOBJECTREF(CRemotingServices::GetRealProxy(proxy));
												if(CTPMethodTable::IsInstanceOfRemotingProxy(
														orRP->GetMethodTable()))
												{
												*((Context **)stub->UnBox()) = (Context*) GetThread()->GetContext();
										}
										}
										// call Finalize on the proxy of the server object.
										obj = proxy;
								}
						}
						_ASSERTE(pMT->HasFinalizer());
						MethodTable::CallFinalizer(obj);
				}
				else
				{
						//reset the bit so the object can be put on the list
						//with RegisterForFinalization
						obj->GetHeader()->ClrBit (BIT_SBLK_FINALIZER_RUN);
				}
		}
}

// GLOBAL
static char s_FinalizeObjectName[MAX_CLASSNAME_LENGTH+MAX_NAMESPACE_LENGTH+2];
// GLOBAL
static BOOL s_fSaveFinalizeObjectName = FALSE;

void	CallFinalizer(Thread* FinalizerThread,
										Object* fobj)
{
		if (s_fSaveFinalizeObjectName) {
				DefineFullyQualifiedNameForClass();
				LPCUTF8 name = GetFullyQualifiedNameForClass(fobj->GetClass());
				strcat (s_FinalizeObjectName, name);
		}
		CallFinalizer(fobj);
		if (s_fSaveFinalizeObjectName) {
				s_FinalizeObjectName[0] = '\0';
		}
		// we might want to do some extra work on the finalizer thread
		// check and do it
		if (FinalizerThread->HaveExtraWorkForFinalizer())
		{
				FinalizerThread->DoExtraWorkForFinalizer();
		}

		// if someone is trying to stop us, open the gates
		FinalizerThread->PulseGCMode();
}

struct FinalizeAllObjects_Args {
				OBJECTREF fobj;
		int bitToCheck;
};

static Object *FinalizeAllObjects(Object* fobj, int bitToCheck);

static void FinalizeAllObjects_Wrapper(LPVOID ptr)
{
		FinalizeAllObjects_Args *args = (FinalizeAllObjects_Args *) ptr;
		_ASSERTE(args->fobj);
		args->fobj = ObjectToOBJECTREF(FinalizeAllObjects(OBJECTREFToObject(args->fobj), args->bitToCheck));
}

static Object *FinalizeAllObjects(Object* fobj, int bitToCheck)
{
		Thread *pThread = GetThread();
		bool fTerminate = false;

		if (fobj == NULL)
				fobj = pThread->GetGCHeap()->GetNextFinalizableObject();

		// Finalize everyone
		while (fobj)
		{
				if (fobj->GetHeader()->GetBits() & bitToCheck)
				{
						fobj = pThread->GetGCHeap()->GetNextFinalizableObject();
						continue;
				}

				COMPLUS_TRY
				{
						AppDomain* targetAppDomain = fobj->GetAppDomain();
						AppDomain* currentDomain = pThread->GetDomain();
						if (! targetAppDomain || ! targetAppDomain->CanThreadEnter(pThread))
						{
								// if can't get into domain to finalize it, then it must be agile so finalize in current domain
								targetAppDomain = currentDomain;
#if CHECK_APP_DOMAIN_LEAKS
								 // object must be agile if can't get into it's domain
								if (g_pConfig->AppDomainLeaks() && !fobj->SetAppDomainAgile(FALSE))
										_ASSERTE(!"Found non-agile GC object which should have been finalized during app domain unload.");
#endif
						}
						if (targetAppDomain == currentDomain)
						{
								CallFinalizer(fobj);
								fobj = pThread->GetGCHeap()->GetNextFinalizableObject();
						}
						else
						{
								if (! targetAppDomain->GetDefaultContext())
								{
										// can no longer enter domain becuase the handle containing the context has been
										// nuked so just bail. Should only get this if are at the stage of nuking the
										// handles in the domain if it's still open.
										_ASSERTE(targetAppDomain->IsUnloading() && targetAppDomain->ShouldHaveRoots());
										fobj = pThread->GetGCHeap()->GetNextFinalizableObject();
								}
								else
								if (currentDomain != SystemDomain::System()->DefaultDomain())
								{
										// this means we are in some other domain, so need to return back out through the DoADCallback
										// and handle the object from there in another domain.
										fTerminate = true;
								}
								else
								{
										// otherwise call back to ourselves to process as many as we can in that other domain
										FinalizeAllObjects_Args args;
										args.fobj = ObjectToOBJECTREF(fobj);
										args.bitToCheck = bitToCheck;
										GCPROTECT_BEGIN(args.fobj);
										pThread->DoADCallBack(targetAppDomain->GetDefaultContext(), FinalizeAllObjects_Wrapper, &args);
										// process the object we got back or be done if we got back null
										fobj = OBJECTREFToObject(args.fobj);
										GCPROTECT_END();
								}
						}
				}
				COMPLUS_CATCH
				{
						// Should be an out of memory from Thread::EnterDomain.	 Swallow,
						// no where to report this, and get the next object
						fobj = pThread->GetGCHeap()->GetNextFinalizableObject();
				}
				COMPLUS_END_CATCH

				if (fTerminate)
						break;
		}
		return fobj;
}

BOOL GCHeap::IsInCycle()
{
	return gc_heap::IsInCycle();
}

void GCHeap::WaitUntilGCComplete()
{
		DWORD dwWaitResult = NOERROR;

		if (GcInProgress) {
				ASSERT( WaitForGCEvent );

#ifdef GC_REPLICATING_COLLECTOR
				if ( GetThread() == GcThread ) {
					// PreemptiveGC has been enabled, but we don't need it to be if we
					// are actually (and explicitly) invoking a GC.
					GcThread->DisablePreemptiveGC();
					// If we are in the middle of a cycle, then we might be the
					// collecting thread... so just force collection to complete.
					enter_spin_lock (gc_heap::GetSharedLock());
					// XXX The second two parameters to GarbageCollect should reflect
					// the actual gap lists.
					Object* next_gap = NULL;
					Object* last_gap = NULL;
					gc_heap::GarbageCollect(gc_heap::GC_FULL_COLLECTION,
																	next_gap, last_gap);
					// XXX Do something with next_gap and last_gap.
					leave_spin_lock (gc_heap::GetSharedLock());
				}
				else
#else //!GC_REPLICATING_COLLECTOR
				ASSERT( GetThread() != GcThread );
#endif
				{
#ifdef DETECT_DEADLOCK
					// wait for GC to complete
BlockAgain:
					dwWaitResult = WaitForSingleObject( WaitForGCEvent,
																							DETECT_DEADLOCK_TIMEOUT );

					if (dwWaitResult == WAIT_TIMEOUT) {
						//	Even in retail, stop in the debugger if available.	Ideally, the
						//	following would use DebugBreak, but debspew.h makes this a null
						//	macro in retail.	Note that in debug, we don't use the debspew.h
						//	macros because these take a critical section that may have been
						//	taken by a suspended thread.
						RetailDebugBreak();
						goto BlockAgain;
					}

#else	 //DETECT_DEADLOCK


					if (g_fEEShutDown) {
						Thread *pThread = GetThread();
						if (pThread) {
							dwWaitResult = pThread->DoAppropriateAptStateWait(1, &WaitForGCEvent, FALSE, INFINITE, TRUE);
						} else {
							dwWaitResult = WaitForSingleObject( WaitForGCEvent, INFINITE );
						}

					}
					else
					{
						dwWaitResult = WaitForSingleObject( WaitForGCEvent, INFINITE );
					}

#endif //DETECT_DEADLOCK
				}
		}
}



void WaitForFinalizerEvent (HANDLE event)
{
				WaitForSingleObject(event, INFINITE);
}





ULONG __stdcall GCHeap::FinalizerThreadStart(void *args)
{
		ASSERT(args != 0);
		GCHeap *gcheap = (GCHeap *)args;

		ASSERT(gcheap->hEventFinalizer);

		LOG((LF_GC, LL_INFO10, "Finalizer thread starting..."));

		Thread *pFinalizerThread = gcheap->FinalizerThread;

#ifdef _DEBUG
				AppDomain* defaultDomain = pFinalizerThread->GetDomain();
#endif
		_ASSERTE(defaultDomain == SystemDomain::System()->DefaultDomain());

		BOOL		ok = pFinalizerThread->HasStarted();

		_ASSERTE(ok);
		_ASSERTE(GetThread() == pFinalizerThread);

		// finalizer should always park in default domain

		if (ok)
		{
				EE_TRY_FOR_FINALLY
				{
						pFinalizerThread->SetBackground(TRUE);

						BOOL noUnloadedObjectsRegistered = FALSE;

						while (!gcheap->fQuitFinalizer)
						{
								// Wait for work to do...

								_ASSERTE(pFinalizerThread->PreemptiveGCDisabled());
#ifdef _DEBUG
								if (g_pConfig->FastGCStressLevel()) {
										pFinalizerThread->m_GCOnTransitionsOK = FALSE;
								}
#endif
								pFinalizerThread->EnablePreemptiveGC();
#ifdef _DEBUG
								if (g_pConfig->FastGCStressLevel()) {
										pFinalizerThread->m_GCOnTransitionsOK = TRUE;
								}
#endif
								WaitForFinalizerEvent (gcheap->hEventFinalizer);
								pFinalizerThread->DisablePreemptiveGC();

#ifdef _DEBUG
#ifdef TRACE_GC
								if (g_pConfig->GetGCStressLevel() > 1)
								{
										int last_gc_count;
										do {
												last_gc_count = gc_count;
												pFinalizerThread->m_GCOnTransitionsOK = FALSE;
												pFinalizerThread->EnablePreemptiveGC();
												__SwitchToThread (0);
												pFinalizerThread->DisablePreemptiveGC();
														// If no GCs happended, then we assume we are quiescent
												pFinalizerThread->m_GCOnTransitionsOK = TRUE;
										} while (gc_count - last_gc_count > 0);
								}
#endif //TRACE_GC
#endif //_DEBUG

								// we might want to do some extra work on the finalizer thread
								// check and do it
								if (pFinalizerThread->HaveExtraWorkForFinalizer())
								{
										pFinalizerThread->DoExtraWorkForFinalizer();
								}

								LOG((LF_GC, LL_INFO100, "***** Calling Finalizers\n"));
								FinalizeAllObjects(NULL, 0);
								_ASSERTE(pFinalizerThread->GetDomain() == SystemDomain::System()->DefaultDomain());


								if (gcheap->UnloadingAppDomain != NULL)
								{
										// Now schedule any objects from an unloading app domain for finalization
										// on the next pass (even if they are reachable.)
										// Note that it may take several passes to complete the unload, if new objects are created during
										// finalization.

										if (!gcheap->FinalizeAppDomain(gcheap->UnloadingAppDomain, gcheap->fRunFinalizersOnUnload))
										{
												if (!noUnloadedObjectsRegistered)
												{
														//
														// There is nothing left to schedule.	 However, there are possibly still objects
														// left in the finalization queue.	We might be done after the next pass, assuming
														// we don't see any new finalizable objects in the domain.
														noUnloadedObjectsRegistered = TRUE;
												}
												else
												{
														// We've had 2 passes seeing no objects - we're done.
														gcheap->UnloadingAppDomain = NULL;
														noUnloadedObjectsRegistered = FALSE;
												}
										}
										else
												noUnloadedObjectsRegistered = FALSE;
								}

								// Anyone waiting to drain the Q can now wake up.	 Note that there is a
								// race in that another thread starting a drain, as we leave a drain, may
								// consider itself satisfied by the drain that just completed.	This is
								// acceptable.
								SetEvent(gcheap->hEventFinalizerDone);
						}

						// Tell shutdown thread we are done with finalizing dead objects.
						SetEvent (gcheap->hEventFinalizerToShutDown);

						// Wait for shutdown thread to signal us.
						pFinalizerThread->EnablePreemptiveGC();
						WaitForSingleObject(gcheap->hEventShutDownToFinalizer, INFINITE);
						pFinalizerThread->DisablePreemptiveGC();

						AppDomain::RaiseExitProcessEvent();

						SetEvent(gcheap->hEventFinalizerToShutDown);

						// Phase 1 ends.
						// Now wait for Phase 2 signal.

						// Wait for shutdown thread to signal us.
						pFinalizerThread->EnablePreemptiveGC();
						WaitForSingleObject(gcheap->hEventShutDownToFinalizer, INFINITE);
						pFinalizerThread->DisablePreemptiveGC();

						gcheap->SetFinalizeQueueForShutdown (FALSE);

						// Finalize all registered objects during shutdown, even they are still reachable.
						// we have been asked to quit, so must be shutting down
						_ASSERTE(g_fEEShutDown);
						_ASSERTE(pFinalizerThread->PreemptiveGCDisabled());
						//
						//
						FinalizeAllObjects(NULL, BIT_SBLK_FINALIZER_RUN);
						_ASSERTE(pFinalizerThread->GetDomain() == SystemDomain::System()->DefaultDomain());

						// we might want to do some extra work on the finalizer thread
						// check and do it
						if (pFinalizerThread->HaveExtraWorkForFinalizer())
						{
								pFinalizerThread->DoExtraWorkForFinalizer();
						}

						SetEvent(gcheap->hEventFinalizerToShutDown);

						// Wait for shutdown thread to signal us.
						pFinalizerThread->EnablePreemptiveGC();
						WaitForSingleObject(gcheap->hEventShutDownToFinalizer, INFINITE);
						pFinalizerThread->DisablePreemptiveGC();


						SetEvent(gcheap->hEventFinalizerToShutDown);
				}
				EE_FINALLY
				{
						if (GOT_EXCEPTION())
								_ASSERTE(!"Exception in the finalizer thread!");
				}
				EE_END_FINALLY;
		}
		// finalizer should always park in default domain
		_ASSERTE(GetThread()->GetDomain() == SystemDomain::System()->DefaultDomain());

		LOG((LF_GC, LL_INFO10, "Finalizer thread done."));

		// Enable pre-emptive GC before we leave so that anybody trying to suspend
		// us will not end up waiting forever. Don't do a DestroyThread because this
		// will happen soon when we tear down the thread store.
		pFinalizerThread->EnablePreemptiveGC();

		// We do not want to tear Finalizer thread,
		// since doing so will cause OLE32 to CoUninitalize.
		::Sleep(INFINITE);

		return 0;
}

// FIXME GC_SUPPORTS_FINALIZERS
DWORD GCHeap::FinalizerThreadCreate()
{
		DWORD		dwRet = 0;
		HANDLE	h;
		DWORD		newThreadId;

		hEventFinalizerDone = CreateEvent(0, TRUE, FALSE, 0);
		if (hEventFinalizerDone)
		{
				hEventFinalizer = CreateEvent(0, FALSE, FALSE, 0);
				hEventFinalizerToShutDown = CreateEvent(0, FALSE, FALSE, 0);
				hEventShutDownToFinalizer = CreateEvent(0, FALSE, FALSE, 0);
				if (hEventFinalizer && hEventFinalizerToShutDown && hEventShutDownToFinalizer)
				{
						_ASSERTE(FinalizerThread == 0);
						FinalizerThread = SetupUnstartedThread();
						if (FinalizerThread == 0) {
								return 0;
						}

						// We don't want the thread block disappearing under us -- even if the
						// actual thread terminates.
						FinalizerThread->IncExternalCount();

						h = FinalizerThread->CreateNewThread(4096, &FinalizerThreadStart, (void *)this, &newThreadId);
						if (h)
						{
								::SetThreadPriority(h, THREAD_PRIORITY_HIGHEST);

								// Before we do the resume, we need to take note of the new ThreadId.	 This
								// is necessary because -- before the thread starts executing at KickofThread --
								// it may perform some DllMain DLL_THREAD_ATTACH notifications.	 These could
								// call into managed code.	During the consequent SetupThread, we need to
								// perform the Thread::HasStarted call instead of going through the normal
								// 'new thread' pathway.
								_ASSERTE(FinalizerThread->GetThreadId() == 0);
								_ASSERTE(newThreadId != 0);

								FinalizerThread->SetThreadId(newThreadId);

								dwRet = ::ResumeThread(h);
								_ASSERTE(dwRet == 1);
						}
				}
		}
		return dwRet;
}

// Wait for the finalizer thread to complete one pass.
void GCHeap::FinalizerThreadWait()
{
		ASSERT(hEventFinalizerDone);
		ASSERT(hEventFinalizer);
		ASSERT(FinalizerThread);

		// FIXME this should force the current collection to finish (if one is in
		// progress)

		// Can't call this from within a finalized method.
		if (!IsCurrentThreadFinalizer())
		{

				Thread	*pCurThread = GetThread();
				BOOL		 toggleGC = pCurThread->PreemptiveGCDisabled();

				if (toggleGC)
						pCurThread->EnablePreemptiveGC();

				::ResetEvent(hEventFinalizerDone);
				::SetEvent(hEventFinalizer);

				//----------------------------------------------------
				// Do appropriate wait and pump messages if necessary
				//----------------------------------------------------
				//WaitForSingleObject(hEventFinalizerDone, INFINITE);

				pCurThread->DoAppropriateAptStateWait(1, &hEventFinalizerDone, FALSE, INFINITE, TRUE);

				if (toggleGC)
						pCurThread->DisablePreemptiveGC();
		}
}


#ifdef _DEBUG
#define FINALIZER_WAIT_TIMEOUT 250
#else
#define FINALIZER_WAIT_TIMEOUT 200
#endif
#define FINALIZER_TOTAL_WAIT 2000

static BOOL s_fRaiseExitProcessEvent = FALSE;

// FIXME GC_SUPPORTS_FINALIZERS
BOOL GCHeap::FinalizerThreadWatchDog()
{

		Thread *pThread = GetThread();
		HANDLE			h = FinalizerThread->GetThreadHandle();

				// Do not wait for FinalizerThread if the current one is FinalizerThread.
				if (pThread == FinalizerThread)
						return TRUE;

				// If finalizer thread is gone, just return.
				if (h == INVALID_HANDLE_VALUE || WaitForSingleObject (h, 0) != WAIT_TIMEOUT)
						return TRUE;

				// *** This is the first call ShutDown -> Finalizer to Finilize dead objects ***
				if ((g_fEEShutDown & ShutDown_Finalize1) &&
				!(g_fEEShutDown & ShutDown_Finalize2)) {
				// Wait for the finalizer...
				LOG((LF_GC, LL_INFO10, "Signalling finalizer to quit..."));

				fQuitFinalizer = TRUE;
				ResetEvent(hEventFinalizerDone);
				SetEvent(hEventFinalizer);

				LOG((LF_GC, LL_INFO10, "Waiting for finalizer to quit..."));

				pThread->EnablePreemptiveGC();
				BOOL fTimeOut = FinalizerThreadWatchDogHelper();

				if (!fTimeOut) {
						SetEvent(hEventShutDownToFinalizer);

						// Wait for finalizer thread to finish raising ExitProcess Event.
						s_fRaiseExitProcessEvent = TRUE;
						fTimeOut = FinalizerThreadWatchDogHelper();
						if (fTimeOut) {
								s_fRaiseExitProcessEvent = FALSE;
						}
				}

				pThread->DisablePreemptiveGC();

				// Can not call ExitProcess here if we are in a hosting environment.
				// The host does not expect that we terminate the process.
				//if (fTimeOut)
				//{
						//::ExitProcess (GetLatchedExitCode());
				//}

				return !fTimeOut;
		}

				// *** This is the second call ShutDown -> Finalizer to ***
				// suspend the Runtime and Finilize live objects
		if ( g_fEEShutDown & ShutDown_Finalize2 &&
				!(g_fEEShutDown & ShutDown_COM) ) {

				_ASSERTE (g_fEEShutDown & ShutDown_Finalize1);
				SuspendEE(GCHeap::SUSPEND_FOR_SHUTDOWN);

				g_fSuspendOnShutdown = TRUE;

				GcThread = FinalizerThread;
				// !!! We will not resume EE from now on.	 But we are setting the finslizer thread
				// !!! to be the thread that SuspendEE, so that it will be blocked.
				// !!! Before we wake up Finalizer thread, we need to enable preemptive gc on the
				// !!! finalizer thread.	Otherwise we may see a deadlock during debug test.
				pThread->EnablePreemptiveGC();

				g_fFinalizerRunOnShutDown = TRUE;

				// Wait for finalizer thread to finish finalizing all objects.
				SetEvent(GCHeap::hEventShutDownToFinalizer);
				BOOL fTimeOut = FinalizerThreadWatchDogHelper();

				if (!fTimeOut) {
						// We only switch back GcThread if we do not timeout.
						// We check these to decide if we want to enter EE when processing DLL_PROCESS_DETACH.
						GcThread = pThread;
						g_fFinalizerRunOnShutDown = FALSE;
				}

				// Can not call ExitProcess here if we are in a hosting environment.
				// The host does not expect that we terminate the process.
				//if (fTimeOut) {
				//		::ExitProcess (GetLatchedExitCode());
				//}

				pThread->DisablePreemptiveGC();
				return !fTimeOut;
		}

		// *** This is the third call ShutDown -> Finalizer ***
		// to do additional cleanup
				if (g_fEEShutDown & ShutDown_COM) {
				_ASSERTE (g_fEEShutDown & (ShutDown_Finalize2 | ShutDown_Finalize1));

				GcThread = FinalizerThread;
				pThread->EnablePreemptiveGC();
				g_fFinalizerRunOnShutDown = TRUE;

								SetEvent(GCHeap::hEventShutDownToFinalizer);
				DWORD status = pThread->DoAppropriateAptStateWait(1, &hEventFinalizerToShutDown,
																										FALSE, FINALIZER_WAIT_TIMEOUT,
																										TRUE);

								BOOL fTimeOut = (status == WAIT_TIMEOUT) ? TRUE : FALSE;

								if (!fTimeOut) {
						GcThread = pThread;
						g_fFinalizerRunOnShutDown = FALSE;
				}
								pThread->DisablePreemptiveGC();

				return !fTimeOut;
				}

		_ASSERTE(!"Should never reach this point");
				return FALSE;
}

BOOL GCHeap::FinalizerThreadWatchDogHelper()
{
		Thread *pThread = GetThread();
		_ASSERTE (!pThread->PreemptiveGCDisabled());

		DWORD dwBeginTickCount = GetTickCount();

		size_t prevNumFinalizableObjects = GetNumberFinalizableObjects();
		size_t curNumFinalizableObjects;
		BOOL fTimeOut = FALSE;
		DWORD nTry = 0;
		DWORD maxTry = (DWORD)(FINALIZER_TOTAL_WAIT*1.0/FINALIZER_WAIT_TIMEOUT + 0.5);
		DWORD maxTotalWait = (s_fRaiseExitProcessEvent?3000:40000);
		BOOL bAlertable = TRUE; //(g_fEEShutDown & ShutDown_Finalize2) ? FALSE:TRUE;

		static DWORD dwBreakOnFinalizeTimeOut = (DWORD) -1;
		if (dwBreakOnFinalizeTimeOut == (DWORD) -1) {
		dwBreakOnFinalizeTimeOut = g_pConfig->GetConfigDWORD(L"BreakOnFinalizeTimeOut", 0);
		}

		DWORD dwTimeout = FINALIZER_WAIT_TIMEOUT;

		// This used to set the dwTimeout to infinite, but this can cause a hang when shutting down
		// if a finalizer tries to take a lock that another suspended managed thread already has.
		// This results in the hang because the other managed thread is never going to be resumed
		// because we're in shutdown.	 So we make a compromise here - make the timeout for every
		// iteration 10 times longer and make the total wait infinite - so if things hang we will
		// eventually shutdown but we also give things a chance to finish if they're running slower
		// because of the profiler.
#ifdef PROFILING_SUPPORTED
		if (CORProfilerPresent())
		{
				dwTimeout *= 10;
				maxTotalWait = INFINITE;
		}
#endif // PROFILING_SUPPORTED

		while (1) {
				DWORD status = 0;
				COMPLUS_TRY
				{
						status = pThread->DoAppropriateAptStateWait(1,&hEventFinalizerToShutDown,FALSE, dwTimeout, bAlertable);
				}
				COMPLUS_CATCH
				{
						status = WAIT_TIMEOUT;
				}
				COMPLUS_END_CATCH

				if (status != WAIT_TIMEOUT) {
						break;
				}
				nTry ++;
				curNumFinalizableObjects = GetNumberFinalizableObjects();
				if ((prevNumFinalizableObjects <= curNumFinalizableObjects || s_fRaiseExitProcessEvent)
#ifdef _DEBUG
						&& gc_heap::GetSharedLock()->lock == -1
#else
						&& *(gc_heap::GetSharedLock()) == -1
#endif
						&& !(pThread->m_State & (Thread::TS_UserSuspendPending | Thread::TS_DebugSuspendPending))){
						if (nTry == maxTry) {
								if (!s_fRaiseExitProcessEvent) {
								LOG((LF_GC, LL_INFO10, "Finalizer took too long on one object.\n"));
								}
								else
										LOG((LF_GC, LL_INFO10, "Finalizer took too long to process ExitProcess event.\n"));

								fTimeOut = TRUE;
								if (dwBreakOnFinalizeTimeOut != 2) {
										break;
								}
						}
				}
				else
				{
						nTry = 0;
						prevNumFinalizableObjects = curNumFinalizableObjects;
				}
				DWORD dwCurTickCount = GetTickCount();
				if (pThread->m_State & (Thread::TS_UserSuspendPending | Thread::TS_DebugSuspendPending)) {
						dwBeginTickCount = dwCurTickCount;
				}
				if (dwCurTickCount - dwBeginTickCount >= maxTotalWait
						|| (dwBeginTickCount > dwCurTickCount && dwBeginTickCount - dwCurTickCount <= (~0) - maxTotalWait)) {
						LOG((LF_GC, LL_INFO10, "Finalizer took too long on shutdown.\n"));
						fTimeOut = TRUE;
						if (dwBreakOnFinalizeTimeOut != 2) {
								break;
						}
				}
		}

		if (fTimeOut)
		{
				if (dwBreakOnFinalizeTimeOut){
						DebugBreak();
				}
				if (!s_fRaiseExitProcessEvent && s_FinalizeObjectName[0] != '\0') {
						LOG((LF_GC, LL_INFO10, "Currently running finalizer on object of %s\n",
								 s_FinalizeObjectName));
				}
		}
		return fTimeOut;
}


#ifdef _DEBUG

// Normally, any thread we operate on has a Thread block in its TLS.	But there are
// a few special threads we don't normally execute managed code on.
//
// There is a scenario where we run managed code on such a thread, which is when the
// DLL_THREAD_ATTACH notification of an (IJW?) module calls into managed code.	This
// is incredibly dangerous.	 If a GC is provoked, the system may have trouble performing
// the GC because its threads aren't available yet.
static DWORD SpecialEEThreads[10];
static LONG	 cnt_SpecialEEThreads = 0;

void dbgOnly_IdentifySpecialEEThread()
{
		LONG	ourCount = FastInterlockIncrement(&cnt_SpecialEEThreads);

		_ASSERTE(ourCount < (LONG) NumItems(SpecialEEThreads));
		SpecialEEThreads[ourCount-1] = ::GetCurrentThreadId();
}

BOOL dbgOnly_IsSpecialEEThread()
{
		DWORD		ourId = ::GetCurrentThreadId();

		for (LONG i=0; i<cnt_SpecialEEThreads; i++)
				if (ourId == SpecialEEThreads[i])
						return TRUE;

		return FALSE;
}

#endif // _DEBUG

#ifdef GC_REQUIRES_FULL_WRITEBARRIER
void Erect8WriteBarrier(__int8* dst, __int8 src)
{
	// FIXME spoons this is not really a heap "object" but a heap pointer.
	if(gc_heap::ReplicateWrites() && gc_heap::IsHeapObject((BYTE*)dst))
	{

		Object* obj = gc_heap::FindObject((BYTE*)dst);

#if defined(GC_REPLICATING_COLLECTOR)
		if (header(obj)->IsRelocated()) {
#if !defined(GC_CONCURRENT_COLLECTOR)
			Object* replicaObj = header(obj)->GetRelocated();
			__int8* replicaDst =
				(__int8*)((BYTE*)replicaObj + ((BYTE*)dst - (BYTE*)obj));

			*replicaDst = src;
#else // GC_CONCURRENT_COLLECTOR
			gc_heap::RecordWrite8(obj, dst);
#endif
		}
#else // ?
		ASSERT(!"Unimplemented write barrier!");
#endif // !GC_REPLICATING_COLLECTOR
	}
}

void Erect16WriteBarrier(__int16* dst, __int16 src)
{
	// FIXME spoons this is not really a heap "object" but a heap pointer.
	if(gc_heap::ReplicateWrites() && gc_heap::IsHeapObject((BYTE*)dst))
	{

		Object* obj = gc_heap::FindObject((BYTE*)dst);

#ifdef GC_REPLICATING_COLLECTOR
		if (header(obj)->IsRelocated()) {
#if !defined(GC_CONCURRENT_COLLECTOR)
			Object* replicaObj = header(obj)->GetRelocated();
			__int16* replicaDst =
				(__int16*)((BYTE*)replicaObj + ((BYTE*)dst - (BYTE*)obj));

			*replicaDst = src;
#else // GC_CONCURRENT_COLLECTOR
			gc_heap::RecordWrite16(obj, dst);
#endif
		}
#else // ?
		ASSERT(!"Unimplemented write barrier!");
#endif // !GC_REPLICATING_COLLECTOR
	}
}

void Erect32WriteBarrier(__int32* dst, __int32 src)
{
	// FIXME spoons this is not really a heap "object" but a heap pointer.
	if(gc_heap::ReplicateWrites() && gc_heap::IsHeapObject((BYTE*)dst))
	{

		Object* obj = gc_heap::FindObject((BYTE*)dst);

#ifdef GC_REPLICATING_COLLECTOR
		if (header(obj)->IsRelocated()) {
#if !defined(GC_CONCURRENT_COLLECTOR)
			Object* replicaObj = header(obj)->GetRelocated();
			__int32* replicaDst =
				(__int32*)((BYTE*)replicaObj + ((BYTE*)dst - (BYTE*)obj));

			*replicaDst = src;
#else // GC_CONCURRENT_COLLECTOR
			gc_heap::RecordWrite32(obj, dst);
#endif
		}
#else // ?
		ASSERT(!"Unimplemented write barrier!");
#endif // !GC_REPLICATING_COLLECTOR
	}
}

void Erect64WriteBarrier(__int64* dst, __int64 src)
{
	// FIXME spoons this is not really a heap "object" but a heap pointer.
	if(gc_heap::ReplicateWrites() && gc_heap::IsHeapObject((BYTE*)dst))
	{

		Object* obj = gc_heap::FindObject((BYTE*)dst);

#ifdef GC_REPLICATING_COLLECTOR
		if (header(obj)->IsRelocated()) {
#if !defined(GC_CONCURRENT_COLLECTOR)
			Object* replicaObj = header(obj)->GetRelocated();
			__int64* replicaDst =
				(__int64*)((BYTE*)replicaObj + ((BYTE*)dst - (BYTE*)obj));

			*replicaDst = src;
#else // GC_CONCURRENT_COLLECTOR
			gc_heap::RecordWrite64(obj, dst);
#endif
		}
#else // ?
		ASSERT(!"Unimplemented write barrier!");
#endif // !GC_REPLICATING_COLLECTOR
	}
}
#endif //GC_REQUIRES_PTR_WRITEBARRIER

#ifdef GC_REQUIRES_PTR_WRITEBARRIER
	static void
	PerformWrite(Object* obj, Object** dst, Object* ref, Object* old)
	{
		Object* replicaObj = header(obj)->GetRelocated();
		Object** replicaDst =
			(Object**)((BYTE*)replicaObj + ((BYTE*)dst - (BYTE*)obj));

		if (ref == NULL) {
			// Write a reference to the (null) ref.
			*replicaDst = NULL;
			dprintf(3, ("Replicating write (null) %X := %X w/ %X := %X",
				dst, ref, replicaDst, NULL));
			// No need to shade!!
		}
		else if (header(ref)->IsRelocated()) {
			/*
				The ref already exists in to-space, so
				just write a reference to that.
			*/
			Object* refReplica = header(ref)->GetRelocated();
			*replicaDst = refReplica;
			dprintf(3, ("Replicating write (to-space) %X := %X w/ %X := %X",
				dst, ref, replicaDst, refReplica));
		}
		else {
			// Write a reference to the ref (in from-space).
			*replicaDst = ref;
			// Shade the replica:
			dprintf(3, ("Replicating write (shade) %X := %X w/ %X := %X",
				dst, ref, replicaDst, ref));
			gc_heap::ShadeRoot(replicaDst);
		}

		// FIXME spoons this is not really a heap "object" but a heap pointer.
		if (old != NULL  && gc_heap::IsHeapObject((BYTE*)old)) {
			gc_heap::ShadeObject(old);
		}
	}

	void
	ErectPtrWriteBarrier(Object** dst, Object* ref, Object* old)
	{
		// FIXME spoons this is not really a heap "object" but a heap pointer.
		if(gc_heap::ReplicateWrites()){
			Object* obj = gc_heap::FindObject((BYTE*)dst);

			#ifdef GC_REPLICATING_COLLECTOR
				if (header(obj)->IsRelocated()) {
					#if !defined(GC_CONCURRENT_COLLECTOR)
						PerformWrite(obj,dst,ref,old);
					#else
						gc_heap::RecordWritePtr(obj, dst);
					#endif
				}
			#else
				ASSERT(!"Unimplemented write barrier!");
			#endif
		}
	}
#endif //GC_REQUIRES_PTR_WRITEBARRIER

void
GCHeap::WriteBarrierSyncBlockIndex(ObjHeader* hdr)
{
#ifdef GC_REPLICATING_COLLECTOR
	Object* obj = (Object*)((BYTE*)hdr + sizeof(ObjHeader));

	if(gc_heap::ReplicateWrites() && gc_heap::IsHeapObject((BYTE*)obj))
	{
		if (header(obj)->IsRelocated()) {
			Object* replica = header(obj)->GetRelocated();
			ObjHeader* replicaHdr = (ObjHeader*)((BYTE*)replica - sizeof(ObjHeader));
			// FIXME spoons: This is sort of a hack since we are muddling the
			// internal fields of ObjHeader.  However, we don't want to call any of
			// the ObjHeader methods since (as they are currently written) they will
			// acquire locks and invoke the write barrier.

			// Don't carry over the lock bit! since the mutator doesn't have access
			// to to-space, there's no need to do any locking on the replica.
			*(DWORD*)replicaHdr = *(DWORD*)hdr & ~BIT_SBLK_SPIN_LOCK;
		}
	}
#else // ?
	ASSERT(!"Unimplemented write barrier!");
#endif
}

#if defined(GC_CONCURRENT_COLLECTOR)
void
GCHeap::RecordWrite8(Object* obj, __int8* dst)
{
	gc_heap::RecordWrite8(obj, dst);
}
void
GCHeap::RecordWrite16(Object* obj, __int16* dst)
{
	gc_heap::RecordWrite16(obj, dst);
}
void
GCHeap::RecordWrite32(Object* obj, __int32* dst)
{
	gc_heap::RecordWrite32(obj, dst);
}
void
GCHeap::RecordWrite64(Object* obj, __int64* dst)
{
	gc_heap::RecordWrite64(obj, dst);
}
void
GCHeap::RecordWritePtr(Object* obj, Object** dst, Object* old)
{
	gc_heap::RecordWritePtr(obj, dst, old);
}
void
GCHeap::RecordMultipleWrites(Object* obj)
{
	gc_heap::RecordMultipleWrites(obj);
}
#endif // GC_CONCURRENT_COLLECTOR


/* These helpers basically replicate the ErectWriteBarrier code inline */

void ValidateObjectMember(Object *obj);

extern "C" HCIMPL2(VOID, JIT_WriteBarrier8, __int8* dst, __int8 src)
{
	*dst = src;

#ifdef GC_REQUIRES_FULL_WRITEBARRIER
	Erect8WriteBarrier(dst, src);
#endif
}
HCIMPLEND

extern "C" HCIMPL2(VOID, JIT_WriteBarrier16, __int16* dst, __int16 src)
{
	*dst = src;

#ifdef GC_REQUIRES_FULL_WRITEBARRIER
	Erect16WriteBarrier(dst, src);
#endif
}
HCIMPLEND

extern "C" HCIMPL2(VOID, JIT_WriteBarrier32, __int32* dst, __int32 src)
{
	*dst = src;

#ifdef GC_REQUIRES_FULL_WRITEBARRIER
	Erect32WriteBarrier(dst, src);
#endif
}
HCIMPLEND

extern "C" HCIMPL2(VOID, JIT_WriteBarrier64, __int64* dst, __int64 src)
{
	*dst = src;

#ifdef GC_REQUIRES_FULL_WRITEBARRIER
	Erect64WriteBarrier(dst, src);
#endif
}
HCIMPLEND

extern "C" HCIMPL2(VOID, JIT_CheckedWriteBarrierPtr, Object **dst, Object *ref)
{
	Object* old = *dst;
	*dst = ref;

#ifdef VERIFY_HEAP
	if(ref) {
		ValidateObjectMember(ref);
	}
#endif

#ifdef GC_REQUIRES_PTR_WRITEBARRIER

	if(gc_heap::ReplicateWrites() && gc_heap::IsHeapObject((BYTE*)dst))
	{
		// This implementation requires interior pointer support!
		Object* obj = gc_heap::FindObject((BYTE*)dst);

#ifdef GC_REPLICATING_COLLECTOR
		if (header(obj)->IsRelocated()) {
#if !defined(GC_CONCURRENT_COLLECTOR)
			Object* replicaObj = header(obj)->GetRelocated();
			Object** replicaDst =
				(Object**)((BYTE*)replicaObj + ((BYTE*)dst - (BYTE*)obj));

			if (ref == NULL) {
				// Write a reference to the (null) ref.
				*replicaDst = NULL;
				dprintf(3, ("Replicating write (null) %X := %X w/ %X := %X",
										dst, ref, replicaDst, NULL));
				// No need to shade!!
			}
			else if (header(ref)->IsRelocated()) {
				// The ref already exists in to-space, so just write a reference to
				// that.
				Object* refReplica = header(ref)->GetRelocated();
				*replicaDst = refReplica;
				dprintf(3, ("Replicating write (to-space) %X := %X w/ %X := %X",
										dst, ref, replicaDst, refReplica));
			}
			else {
				// Write a reference to the ref (in from-space).
				*replicaDst = ref;
				dprintf(3, ("Replicating write (shade) %X := %X w/ %X := %X",
										dst, ref, replicaDst, ref));
				gc_heap::ShadeRoot(replicaDst);
			}
		}

		// FIXME spoons this is not really a heap "object" but a heap pointer.
		if (old != NULL && gc_heap::IsHeapObject((BYTE*)old)) {
			gc_heap::ShadeObject(old);
		}

#else // GC_CONCURRENT_COLLECTOR
		gc_heap::RecordWritePtr(obj, dst);
#endif

#else // ?
		ASSERT(!"Unimplemented write barrier!");
#endif // !GC_REPLICATING_COLLECTOR
	}
#endif //GC_REQUIRES_PTR_WRITEBARRIER
}
HCIMPLEND

extern "C" HCIMPL2(VOID, JIT_WriteBarrierPtr, Object **dst, Object *ref)
{
	Object* old = *dst;
	*dst = ref;

#ifdef VERIFY_HEAP
	if(ref) {
		ValidateObjectMember(ref);
	}
#endif

#ifdef GC_REQUIRES_PTR_WRITEBARRIER
	if(gc_heap::ReplicateWrites())
	{
		// This implementation requires interior pointer support!
		Object* obj = gc_heap::FindObject((BYTE*)dst);

#ifdef GC_REPLICATING_COLLECTOR
		if (header(obj)->IsRelocated()) {
#if !defined(GC_CONCURRENT_COLLECTOR)
			Object* replicaObj = header(obj)->GetRelocated();
			Object** replicaDst =
				(Object**)((BYTE*)replicaObj + ((BYTE*)dst - (BYTE*)obj));

			if (ref == NULL) {
				// Write a reference to the (null) ref.
				*replicaDst = NULL;
				dprintf(3, ("Replicating write (null) %X := %X w/ %X := %X",
										dst, ref, replicaDst, NULL));
				// No need to shade!!
			}
			else if (header(ref)->IsRelocated()) {
				// The ref already exists in to-space, so just write a reference to
				// that.
				Object* refReplica = header(ref)->GetRelocated();
				*replicaDst = refReplica;
				dprintf(3, ("Replicating write (to-space) %X := %X w/ %X := %X",
										dst, ref, replicaDst, refReplica));
			}
			else {
				// Write a reference to the ref (in from-space).
				*replicaDst = ref;
				// Shade the replica:
				dprintf(3, ("Replicating write (shade) %X := %X w/ %X := %X",
										dst, ref, replicaDst, ref));
				gc_heap::ShadeRoot(replicaDst);
			}
		}

		// FIXME spoons this is not really a heap "object" but a heap pointer.
		if (old != NULL && gc_heap::IsHeapObject((BYTE*)old)) {
			gc_heap::ShadeObject(old);
		}
#else // GC_CONCURRENT_COLLECTOR
		gc_heap::RecordWritePtr(obj, dst);
#endif

#else // ?
		ASSERT(!"Unimplemented write barrier!");
#endif // !GC_REPLICATING_COLLECTOR
	}
#endif //GC_REQUIRES_PTR_WRITEBARRIER
}
HCIMPLEND

HCIMPL3(VOID, JIT_StructWriteBarrier, void *dest, void* src, CORINFO_CLASS_HANDLE typeHnd_)
{
		TypeHandle typeHnd(typeHnd_);
		MethodTable *pMT = typeHnd.AsMethodTable();
		pMT->CheckRestore();

		CopyValueClassUnchecked(dest, src, pMT);

#ifdef GC_REQUIRES_FULL_WRITEBARRIER
		// FIXME spoons this is not really a heap "object" but a heap pointer.
		if(gc_heap::ReplicateWrites() && gc_heap::IsHeapObject((BYTE*)dest))
		{
			// This implementation requires interior pointer support!
			Object* obj = gc_heap::FindObject((BYTE*)dest);

#ifdef GC_REPLICATING_COLLECTOR
			if (header(obj)->IsRelocated()) {
				gc_heap::RecordWriteStruct(dest, pMT);
			}
#endif // GC_REPLICATING_COLLECTOR
		}
#else // !GC_REQUIRES_FULL_WRITEBARRIER
		ASSERT(!"Unimplemented write barrier!");
#endif // !GC_REQUIRES_FULL_WRITEBARRIER
}
HCIMPLEND

