
#ifndef _GCDATA_H_
#define _GCDATA_H_

#include "gcsmppriv.h"

class Page{
	private:

	enum {
		// PAGE FLAG DEFINITIONS
		STATUS_MASK = 0x0003,
		STATUS_SHIFT = 0,

		PROMOTED_REASON_MASK = 0x000C,
		PROMOTED_REASON_SHIFT = 2,

		/*
			Bits related to large objects.  A LARGE page
			has one or more CONTINUED pages after it.
		*/
		LARGE = 0x0010,
		CONTINUED = 0x0020,

		/*
			Pages that contain as many live objects "as
			possible" are marked full -- even if the
			residency is not equal to MOSTLY_PAGE_SIZE,
			the difference is only caused by
			fragmentation.  This flag is used to mark
			pages that are copied into during Cheney
			scanning and should be considered "as full as
			possible," but may be slightly fragmented.
		*/
		FULL = 0x0040,

		// Used by the write barrier in the concurrent case:
		WRITES = 0x0080,

		AGE_MASK = 0x3000,
		AGE_SHIFT = 12,

		ACTIVE = 0x4000,

		RESIDENCY_MASK = 0x1FFF0000,
		RESIDENCY_SHIFT = 16,

	};

	/*
		The active page may contain a portion of zeroed memory
		(i.e.  the portion of memory that may be used to
		satsify an allocation request in the near future).
		The gap_end points to the first object after the
		zeroed region.  This is used to resolve interior
		pointers -- if the interior pointer points into the
		portion of the page after the gap_end, it's not safe
		to scroll forward from the beginning of the page.
		Instead, we must start from the gap_end.

		XXX Multiple allocation areas will require keeping
		track of multiple gaps.  One alternative is to keep a
		gap for EVERY page.  Another is to store it in the
		allocation context.  A third is to reuse some other
		field in the page table.
	*/
	static size_t g_gap_end;

	public:

	typedef enum {

		STATUS_FREE = 0,
		STATUS_EVACUATED = 1,
		STATUS_PROMOTED = 2,
		STATUS_REPLICA = 3,

	} GcPageStatus;

	typedef enum {

		PROMOTED_REASON_TEMP = 0,
		PROMOTED_REASON_PERM = 1,
		PROMOTED_REASON_LARGE = 2,
		PROMOTED_REASON_DENSE = 3,

	} GcPagePromotedReason;

	typedef enum {

		AGE_INFANT = 0x0,
		AGE_YOUNG = 0x1,
		AGE_MATURE = 0x2,
		AGE_ANCIENT = 0x3,

	} GcPageAge;

	friend class PageQueue;

	inline BOOL
	HasStatus(GcPageStatus status)
	{
		return ((flags & STATUS_MASK)
			== (status << STATUS_SHIFT));
	}

	inline GcPageStatus
	GetStatus()
	{
		return (GcPageStatus)((flags & STATUS_MASK)
			>> STATUS_SHIFT);
	}

	inline void
	SetStatus(GcPageStatus status)
	{
		flags = ((flags & ~STATUS_MASK)
			| (status << STATUS_SHIFT));
	}

	inline BOOL
	HasPromotedReason(GcPagePromotedReason reason)
	{
		ASSERT(HasStatus(STATUS_PROMOTED));
		return ((flags & PROMOTED_REASON_MASK)
			== (reason << PROMOTED_REASON_SHIFT));
	}

	inline GcPagePromotedReason
	GetPromotedReason()
	{
		ASSERT(HasStatus(STATUS_PROMOTED));
		return (GcPagePromotedReason)((flags & PROMOTED_REASON_MASK)
			>> PROMOTED_REASON_SHIFT);
	}

	inline void
	SetPromotedReason(GcPagePromotedReason reason)
	{
		ASSERT(HasStatus(STATUS_PROMOTED));
		flags = ((flags & ~PROMOTED_REASON_MASK)
			| (reason << PROMOTED_REASON_SHIFT));
	}

	inline void
	ClearPromotedReason()
	{
		ASSERT(HasStatus(STATUS_PROMOTED));
		flags &= ~PROMOTED_REASON_MASK;
	}

	inline BOOL
	IsFull()
	{
		return flags & FULL;
	}

	inline void
	SetFull()
	{
		flags |= FULL;
	}

	inline void
	ClearFull()
	{
		flags &= ~FULL;
	}

	inline BOOL
	IsActive()
	{
		return flags & ACTIVE;
	}

	inline void
	SetActive()
	{
		ASSERT(GetGapEnd() != NULL);
		flags |= ACTIVE;
	}
	inline void
	ClearActive()
	{
		flags &= ~ACTIVE;
	}

	inline size_t
	AcquireGapLock()
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			ASSERT(IsActive());
			size_t end_gap = g_gap_end;
			while(end_gap == g_gap_end) {
				end_gap = FastInterlockCompareExchange(
					(LONG*)&g_gap_end, NULL, (LONG)end_gap);
			}
			return end_gap;
		}
		else {
			return 0;
		}
	}

	inline void
	ReleaseGapLock(size_t end_gap)
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			ASSERT(IsActive());
			g_gap_end = end_gap;
		}
	}

	inline BOOL
	IsLarge()
	{
		return flags & LARGE;
	}

	inline void
	SetLarge()
	{
		ASSERT(!IsContinued());
		flags |= LARGE;
	}

	inline void
	ClearLarge()
	{
		flags &= ~LARGE;
	}

	inline BOOL
	IsContinued()
	{
		return flags & CONTINUED;
	}

	inline void
	SetContinued()
	{
		ASSERT(!IsLarge());
		flags |= CONTINUED;
	}

	inline void
	ClearContinued()
	{
		flags &= ~CONTINUED;
	}

	inline BOOL
	IsWrites()
	{
		return flags & WRITES;
	}

	inline void
	SetWrites()
	{
		ASSERT(!HasStatus(STATUS_FREE));
		flags |= WRITES;
	}

	inline void
	ClearWrites()
	{
		flags &= ~WRITES;
	}

	// Clears all flags but not age or residency.
	inline void
	ClearFlags()
	{
		flags &= (AGE_MASK | RESIDENCY_MASK);
	}

	inline int
	GetAge()
	{
		return (flags & AGE_MASK) >> AGE_SHIFT;
	}

	inline void
	SetAge(int age)
	{
		ASSERT(age == AGE_INFANT
			|| age == AGE_YOUNG
			|| age == AGE_MATURE
			|| age == AGE_ANCIENT);
		flags = (flags & ~AGE_MASK) | (age << AGE_SHIFT);
	}

	inline void
	ClearAge()
	{
		flags &= ~AGE_MASK;
	}

	inline Page*
	GetNextPage()
	{
		return next_page;
	}

	inline Page*
	GetNextNonContinuedPage(int& count)
	{
		count++;
		Page* page = next_page;
		while (page != NULL && page->IsContinued()) {
			page = page->GetNextPage();
			count++;
		}
		return page;
	}

	inline Page*
	GetPrevPage()
	{
		return prev_page;
	}

	inline void
	SetResidency(size_t size)
	{
		ASSERT(!IsContinued());
		ASSERT(size <= gc_heap::MOSTLY_PAGE_SIZE);
		ASSERT((size << RESIDENCY_SHIFT) <= RESIDENCY_MASK);
		flags = (flags & ~RESIDENCY_MASK)
			| (int)(size << RESIDENCY_SHIFT);
	}

	inline size_t
	GetResidency()
	{
		ASSERT(!IsContinued());
		return (size_t)(flags >> RESIDENCY_SHIFT);
	}

	inline void
	ClearResidency ()
	{
		flags &= ~RESIDENCY_MASK;
	}

	// FIXME what about gaps for large pages?
	inline void
	SetFirstGap(size_t offset)
	{
		ASSERT(!IsContinued());
		flags = (flags & ~RESIDENCY_MASK)
			| (int)(offset << RESIDENCY_SHIFT);
	}

	inline size_t
	GetFirstGap()
	{
		return (size_t)(flags >> RESIDENCY_SHIFT);
	}

	inline void
	ClearFirstGap()
	{
		flags &= ~RESIDENCY_MASK;
	}

	inline void
	SetGapEnd(size_t offset)
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			g_gap_end = offset;
		}
	}

	inline size_t
	GetGapEnd()
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			return g_gap_end;
		}
		else {
			ASSERT(!"Gap end not defined for non-replicating collectors");
			return 0;
		}
	}

	inline void
	ClearGapEnd()
	{
		if (gc_heap::GcIsReplicatingCollector()) {
			g_gap_end = 0;
		}
		else {
			ASSERT(!"Gap end not defined for non-replicating collectors");
		}
	}

	inline Page*
	GetFirstPage()
	{
		ASSERT(IsContinued());
		Page* page = this;
		do{
			page = page->prev_page;
		}
		while (page->IsContinued());
		ASSERT(page->IsLarge());
		return page;
	}

	inline BYTE*
	GetData()
	{
		return data;
	}

	inline void
	SetData(BYTE* d)
	{
		data = d;
	}

	static int PageListLength(
		Page *page_list);

	private:

	inline void
	SetNextPage(Page* page)
	{
		next_page = page;
	}

	inline void
	SetPrevPage(Page* page)
	{
		prev_page = page;
	}

	int flags;
	Page* prev_page;
	Page* next_page;
	BYTE* data;
};

/*
	The PageQueue defines a set of helper functions for
	manipulating pages -- don't see it as an abstract data type,
	but rather as a collection of related code.  In particular,
	some clients may manipulate the interal list directly.
*/
class PageQueue{
	public:

	PageQueue();
	~PageQueue();

	void CopyInto(
		PageQueue& queue);

	Page* PeekFirstPage();
	Page* PeekLastPage();
	BOOL IsEmpty();
	Page* PopPage();

	void PopPages(
		Page*& first,
		Page*& last,
		int& count);

	void PushPage(
		Page* page);

	// count must be the number of pages between first and last (inclusive)
	void PushPages(
		Page* first,
		Page* last,
		int count);

	/*
		Inserts the pages "first" through "last" into the
		queue after "targer".  "target" must already be in
		this queue.
	*/
	void InsertAfter(
		Page* first,
		Page* last,
		Page* target);

	/*
		Inserts the pages "first" through "last" into the
		queue before "targer".  "target" must already be in
		this queue.
	*/
	void InsertBefore(
		Page* first,
		Page* last,
		Page* target);

	void CoalescePages(
		Page* first,
		Page* last,
		int count);

	// Removes a single page from the queue.
	void RemovePage(
		Page* a);

	// Removes the continuous list of pages from a to b from this queue.
	void RemovePages(
		Page* a,
		Page* b,
		int count);

	void Clear();
	int GetLength();

	private:

	Page* first_page;
	Page* last_page;
	int page_count;
};


class WorkList;

typedef enum {
	/*
		Indicates that the root itself may be transient and
		instead we've stored a pointer to the heap object
		itself.
	*/
	WORK_NORMAL_PTR = 0,

	// Indicates the object ~may~ be an interior pointer
	WORK_INTERIOR_PTR = 1,

	WORK_NORMAL_ROOT = 2,

	// Indicates the root ~may~ be contain interior pointer
	WORK_INTERIOR_ROOT = 3,

	/*
		Indicates that the ptr is the beginning of a range of
		range objects, ending with "limit."
	*/
	WORK_RANGE_PTR = 4,

	#if defined (GC_CONCURRENT_COLLECTOR)
		/*
			Used to delimit the range of a write buffer
			used to hold updates captured by the write
			barrier.  In this case, ptr indicates the
			beginning of the range, and limit indicates
			the end of the range.
		*/
		WORK_WRITES = 5,
	#endif // GC_CONCURRENT_COLLECTOR

} WorkFlags;

struct Work {

	friend class WorkList;
	union {
		/*
			A pointer to a cell that contains a pointer to
			an object in from-space which must be
			copied/scanned.  (Only if WORK_NORMAL_ROOT or
			WORK_INTERIOR_ROOT)
		*/
		Object** root;

		/*
			Only if WORK_NORMAL_PTR or WORK_INTERIOR_PTR:
			a pointer to an object in from-space which
			must be copied/scanned.
		*/
		Object* ptr;
	};
	/*
		A pointer to the address after the last object that
		must be scanned.
	*/
	BYTE* limit;

	BOOL
	IsInterior()
	{
		#ifdef GC_SUPPORTS_INTERIOR_POINTERS
			return flags == WORK_INTERIOR_ROOT
				|| flags == WORK_INTERIOR_PTR;
		#else // !GC_SUPPORTS_INTERIOR_POINTERS
			return FALSE;
		#endif // !GC_SUPPORTS_INTERIOR_POINTERS
	}

	BOOL
	IsRoot()
	{
		return flags == WORK_NORMAL_ROOT
			|| flags == WORK_INTERIOR_ROOT;
	}

	BOOL
	IsRange()
	{
		return flags == WORK_RANGE_PTR;
	}

	BOOL
	IsWrites()
	{
		#if defined(GC_CONCURRENT_COLLECTOR)
			return flags == WORK_WRITES;
		#else
			return FALSE;
		#endif
	}

	private:

	WorkFlags flags;
};

class WorkList{

	public:

	WorkList();
	~WorkList();

	// Forces initialization
	void Init();

	void AddWorkAsRoot(
		Object** root,
		BOOL interior);

	inline void
	AddWorkAsPtr(Object* ptr, BOOL interior)
	{
		ASSERT(!interior);

		obj_last_item = (obj_last_item + 1) % obj_list_size;
		if (!(obj_list_free--)) {
			ExpandObjList();
		}
		/* FIXME interior pointer
		obj_list[obj_last_item] = (Object*)
			((size_t)ptr
			| (interior ? WORK_INTERIOR_PTR : WORK_NORMAL_PTR));
		*/
		obj_list[obj_last_item] = ptr;
	}

	void AddWorkAsRange(
		Object* ptr,
		BYTE* limit);

	void AddWorkAsWrites(
		Object* ptr,
		BYTE* limit);

	BOOL HasWork();

	BOOL HasPtrWork();

	int Size();

	void RemoveWork(
		Work& work);

	inline Object*
	RemovePtrWork()
	{
		ASSERT(HasPtrWork());
		Object* result = obj_list[obj_first_item];
		++obj_list_free;

		ASSERT((size_t)result != 0xCCCCCCCC);

		#ifdef _DEBUG
			((size_t*)obj_list)[obj_first_item] = 0xCCCCCCCC;
		#endif

		obj_first_item = (obj_first_item + 1) % obj_list_size;
		return result;
	}


	void PrintList();
	int Capacity();

	void AddAll(
		WorkList* other);

	private:

	int obj_list_free;
	int obj_list_size;
	Object** obj_list;
	int obj_first_item, obj_last_item;

	int work_list_size;
	Work* work_list;
	int work_first_item, work_last_item;

	void ExpandObjList();
	void EnsureSpace();
};

/*
	This function clears a piece of memory.  size has to be Dword
	aligned
*/
inline void
memclr(BYTE* mem, size_t size)
{
	assert ((size & (sizeof (DWORD)-1)) == 0);
	DWORD* m= (DWORD*)mem;
	for (size_t i = 0; i < size / sizeof(DWORD); i++)
		*(m++) = 0;
}

inline void
memcopy(BYTE* dmem, BYTE* smem, size_t size)
{

	assert ((size & (sizeof (DWORD)-1)) == 0);
	DWORD* dm= (DWORD*)dmem;
	DWORD* sm= (DWORD*)smem;
	DWORD* smlimit = (DWORD*)(smem + size);
	do {
		*(dm++) = *(sm++);
	} while	(sm < smlimit);
}

#endif // _GCDATA_H_
