// ==++==
//
//
//		Copyright (c) 2002 Microsoft Corporation.	 All rights reserved.
//
//		The use and distribution terms for this software are contained in the file
//		named license.txt, which can be found in the root of this distribution.
//		By using this software in any fashion, you are agreeing to be bound by the
//		terms of this license.
//
//		You must not remove this notice, or any other, from this software.
//
//
// ==--==
// gc.h

#ifndef __GC_H
#define __GC_H

#ifdef PROFILING_SUPPORTED
	// We don't support profiling
	#undef PROFILING_SUPPORTED
#endif // PROFILING_SUPPORTED

/* forward declerations */
class gc_heap;
class CFinalize;

struct alloc_context{
	/*
		N.B. These fields are accessed BY OFFSET in
		i386/jitinterfacex86.cpp -- DO NOT REORDER them without
		updating that code!
	*/
	BYTE* alloc_ptr;
	BYTE* alloc_limit; // used for fast comparison
	__int64 alloc_bytes; //Number of bytes allocated by this context
	BYTE* alloc_start;
	BYTE* alloc_end; // the true end of this alloc space, must be >= limit
	BYTE* alloc_next; // the next free gap, if any
	BYTE* alloc_last;

	void
	init()
	{
		alloc_ptr = 0;
		alloc_limit = 0;
		alloc_bytes = 0;
		alloc_start = 0;
		alloc_end = 0;
		alloc_next = 0;
	}
};

struct ScanContext{
	BOOL concurrent; //TRUE: concurrent scanning

	#if defined (VERIFY_HEAP)
		BOOL verify_page_is_live; // FALSE when verifying finalizer data
	#endif

	#if CHECK_APP_DOMAIN_LEAKS
		AppDomain* pCurrentDomain;
	#endif

	ScanContext()
	{
		concurrent = FALSE;
		#if defined (VERIFY_HEAP)
			verify_page_is_live = TRUE;
		#endif
	}
};

#define PER_HEAP
#define PER_HEAP_ISOLATED

/*
	Ephemeral Garbage Collected Heap Interface
*/

// SPOONS: used when we have to return a "generation"
#define FAKE_MAX_GENERATION 1

class GCHeap{

	friend HRESULT InitializeMiniDumpBlock();
	friend struct MEMBER_OFFSET_INFO(GCHeap);

	protected:

	#define pGenGCHeap ((gc_heap*)0)

	friend class CFinalize;
	friend class gc_heap;
	friend void EnterAllocLock();
	friend void LeaveAllocLock();
	friend void STDMETHODCALLTYPE EEShutDown(BOOL fIsDllUnloading);

	GCHeap();
	~GCHeap();

	inline Object* AllocateInContext(
		size_t size,
		BOOL pinned);

	inline Object* AllocateFromPages(
		size_t min_size,
		size_t& actual_size);

	public:

	static HRESULT GlobalInitialize();

	static GCHeap* CreateGCHeap();

	static void DestroyGCHeap(
		GCHeap *pGCHeap);

	/* BaseGCHeap Methods*/
	PER_HEAP_ISOLATED HRESULT Shutdown();

	/* SPOONS */
	PER_HEAP size_t GetTotalBytesInUse();

	PER_HEAP_ISOLATED BOOL
	IsGCInProgress()
	{
		return GcInProgress;
	}

	PER_HEAP Thread*
	GetGCThread()
	{
		return GcThread;
	};

	PER_HEAP Thread*
	GetGCThreadAttemptingSuspend()
	{
		return m_GCThreadAttemptingSuspend;
	}

	PER_HEAP BOOL IsInCycle();

	PER_HEAP_ISOLATED void WaitUntilGCComplete();

	PER_HEAP HRESULT Initialize();

	enum {
		GC_ALLOC_FINALIZE = 0x1,
		GC_ALLOC_CONTAINS_REF = 0x2,
		GC_ALLOC_PINNED = 0x4,
	} GC_ALLOC_FLAGS;

	/*
		flags can be GC_ALLOC_CONTAINS_REF, GC_ALLOC_FINALIZE,
		and GC_ALLOC_PINNED
	*/
	PER_HEAP_ISOLATED Object* Alloc(
		DWORD size,
		DWORD flags);

	static BOOL IsLargeObject(
		MethodTable* mt);

	static size_t MinimumLargeObjectSize();

	static BOOL IsObjectInFixedHeap(
		Object *pObj);

	PER_HEAP_ISOLATED HRESULT GarbageCollect(
		int generation = -1,
		BOOL collectClasses=FALSE);

	// Drain the queue of objects waiting to be finalized.
	PER_HEAP_ISOLATED void FinalizerThreadWait();

	//constants for the flags parameter to the gc call back
	enum {
		GC_CALL_INTERIOR = 0x1,
		GC_CALL_PINNED = 0x2,
		GC_CALL_CHECK_APP_DOMAIN = 0x4,
	} GC_CALL_FLAGS;

	/*
		GC callback functions
	*/

	/*
		Check if an argument is promoted (ONLY CALL DURING THE
		PROMOTIONSGRANTED CALLBACK.)

		SPOONS static
	*/
	static PER_HEAP_ISOLATED BOOL IsPromoted(
		Object* object,
		ScanContext* sc);

	static PER_HEAP_ISOLATED void ShadeObject(
		Object* obj);

	static PER_HEAP_ISOLATED void WriteBarrierSyncBlockIndex(
		ObjHeader* hdr);

	#if defined(GC_CONCURRENT_COLLECTOR)

		static PER_HEAP_ISOLATED void RecordWrite8(
			Object* obj,
			__int8* dst);

		static PER_HEAP_ISOLATED void RecordWrite16(
			Object* obj,
			__int16* dst);

		static PER_HEAP_ISOLATED void RecordWrite32(
			Object* obj,
			__int32* dst);

		static PER_HEAP_ISOLATED void RecordWrite64(
			Object* obj,
			__int64* dst);

		static PER_HEAP_ISOLATED void RecordWritePtr(
			Object* obj,
			Object** dst);

		static PER_HEAP_ISOLATED void RecordMultipleWrites(
			Object* obj);

	#endif // GC_CONCURRENT_COLLECTOR

	PER_HEAP HRESULT Init(
		size_t heapSize);

	//Register an object for finalization
	PER_HEAP_ISOLATED void RegisterForFinalization(
		Object* obj);

	//Unregister an object for finalization
	PER_HEAP_ISOLATED void SetFinalizationRun(
		Object* obj);

	/*
		returns the generation number of an object (not valid during
		relocation)
	*/
	PER_HEAP_ISOLATED unsigned WhichGeneration(
		Object* object);

	// returns TRUE is the object is ephemeral
	PER_HEAP_ISOLATED BOOL IsEphemeral(
		Object* object);

	#ifdef VERIFY_HEAP
		PER_HEAP_ISOLATED BOOL IsHeapPointer(
			void* object,
			BOOL small_heap_only = FALSE);
	#endif //VERIFY_HEAP

	PER_HEAP size_t ApproxTotalBytesInUse(
		BOOL small_heap_only = FALSE);

	PER_HEAP size_t ApproxFreeBytes();

	// TRUE handled, FALSE propagate
	static BOOL HandlePageFault(
		void*);

	//suspend all threads

	typedef enum {
		SUSPEND_OTHER = 0,
		SUSPEND_FOR_GC = 1,
		SUSPEND_FOR_APPDOMAIN_SHUTDOWN = 2,
		SUSPEND_FOR_CODE_PITCHING = 3,
		SUSPEND_FOR_SHUTDOWN = 4,
		SUSPEND_FOR_DEBUGGER = 5,
		SUSPEND_FOR_INPROC_DEBUGGER = 6,
		SUSPEND_FOR_GC_PREP = 7
	} SUSPEND_REASON;

	PER_HEAP_ISOLATED void SuspendEE(
		SUSPEND_REASON reason);

	// resumes threads.
	PER_HEAP_ISOLATED void RestartEE(
		BOOL bFinishedGC,
		BOOL SuspendSucceded);

	PER_HEAP_ISOLATED inline SUSPEND_REASON
	GetSuspendReason()
	{
		return (m_suspendReason);
	}

	PER_HEAP_ISOLATED inline void
	SetSuspendReason(SUSPEND_REASON suspendReason)
	{
		m_suspendReason = suspendReason;
	}

	PER_HEAP_ISOLATED Thread* GetFinalizerThread();

	// Returns TRUE if the current thread is the finalizer thread.
	PER_HEAP_ISOLATED BOOL IsCurrentThreadFinalizer();

	// allow finalizer thread to run
	PER_HEAP_ISOLATED void EnableFinalization(void);

	// Start unloading app domain
	PER_HEAP_ISOLATED void
	UnloadAppDomain(AppDomain* pDomain, BOOL fRunFinalizers)
	{
		UnloadingAppDomain = pDomain;
		fRunFinalizersOnUnload = fRunFinalizers;
	}

	/*
		Return current unloading app domain (NULL when unload is
		finished.)
	*/
	PER_HEAP_ISOLATED AppDomain*
	GetUnloadingAppDomain()
	{
		return UnloadingAppDomain;
	}

	/*
		Lock for allocation Public because of the fast allocation
		helper
	*/
	PER_HEAP_ISOLATED unsigned
	GetGcCount()
	{
		return GcCount;
	}

	static BOOL IsValidSegmentSize(
		size_t cbSize);

	static BOOL IsValidGen0MaxSize(
		size_t cbSize);

	static size_t GetValidSegmentSize();

	static size_t GetValidGen0MaxSize(
		size_t seg_size);

	static BOOL IsObjRefValid(
		BYTE* ptr);

	PER_HEAP_ISOLATED void SetReservedVMLimit(
		size_t vmlimit);

	PER_HEAP_ISOLATED Object* GetNextFinalizableObject();

	PER_HEAP_ISOLATED size_t GetNumberFinalizableObjects();

	PER_HEAP_ISOLATED size_t GetFinalizablePromotedCount();

	PER_HEAP_ISOLATED BOOL FinalizeAppDomain(
		AppDomain* pDomain,
		BOOL fRunFinalizers);

	PER_HEAP_ISOLATED void SetFinalizeQueueForShutdown(
		BOOL fHasLock);

	protected:

	// Lock for finalization
	PER_HEAP_ISOLATED volatile LONG m_GCFLock;

	PER_HEAP_ISOLATED BOOL GcCollectClasses;
	PER_HEAP_ISOLATED volatile BOOL GcInProgress; // used for syncing w/GC
	/*
		This contains the reason that the runtime was suspended
	*/
	PER_HEAP_ISOLATED SUSPEND_REASON m_suspendReason;
	PER_HEAP_ISOLATED Thread* GcThread; // thread running GC
	PER_HEAP_ISOLATED Thread* m_GCThreadAttemptingSuspend;
	PER_HEAP_ISOLATED unsigned GcCount;

	// Interface with gc_heap
	PER_HEAP_ISOLATED BOOL GarbageCollectGeneration(
		unsigned int gen=0,
		BOOL collectClasses = FALSE);

	// Finalizer thread stuff.

	PER_HEAP_ISOLATED BOOL FinalizerThreadWatchDog();
	PER_HEAP_ISOLATED BOOL FinalizerThreadWatchDogHelper();
	PER_HEAP_ISOLATED DWORD FinalizerThreadCreate();

	/* SPOONS
		PER_HEAP_ISOLATED ULONG __stdcall FinalizerThreadStart(
			void *args);
	*/
	static ULONG __stdcall FinalizerThreadStart(
		void *args);

	PER_HEAP_ISOLATED HANDLE WaitForGCEvent; // used for syncing w/GC
	PER_HEAP_ISOLATED HANDLE hEventFinalizer;
	PER_HEAP_ISOLATED HANDLE hEventFinalizerDone;
	PER_HEAP_ISOLATED HANDLE hEventFinalizerToShutDown;
	PER_HEAP_ISOLATED HANDLE hEventShutDownToFinalizer;
	PER_HEAP_ISOLATED BOOL fQuitFinalizer;
	PER_HEAP_ISOLATED Thread* FinalizerThread;
	PER_HEAP_ISOLATED AppDomain* UnloadingAppDomain;
	PER_HEAP_ISOLATED BOOL fRunFinalizersOnUnload;

	PER_HEAP_ISOLATED CFinalize* m_Finalize;
};

extern volatile LONG m_AllocLock;

// Go through and touch (read) each page straddled by a memory block.
void TouchPages(
	LPVOID pStart,
	UINT cb);

#ifdef VERIFY_HEAP
	void ValidateObjectMember(
		Object *obj);
#endif

#endif // __GC_H
