// ==++==
//
//
//	  Copyright (c) 2002 Microsoft Corporation.	 All rights reserved.
//
//	  The use and distribution terms for this software are contained in the file
//	  named license.txt, which can be found in the root of this distribution.
//	  By using this software in any fashion, you are agreeing to be bound by the
//	  terms of this license.
//
//	  You must not remove this notice, or any other, from this software.
//
//
// ==--==
#include "common.h"
#include "exceptionhandling.h"
#include "threads.inl"

#ifdef _DEBUG
static void DoEHLog(int nestinglevel, DWORD lvl, char* fmt, ...)
{
	va_list	 args;
	va_start(args, fmt);

	if (nestinglevel)
	{
		int	  cch	   = 2*nestinglevel;
		char* pPadding = new char[cch + 1];
		memset(pPadding, '.', cch);
		pPadding[cch] = 0;

		LOG((LF_EH, lvl, pPadding));

		delete[] pPadding;
	}

	LogSpewValist(LF_EH, lvl, fmt, args);
}
#define EH_LOG(expr)  { DoEHLog expr ; }
#else
#define EH_LOG(expr)
#endif


#ifdef _DEBUG
BOOL ExceptionTracker::ThrowableIsValid()
{
	INDEBUG(ValidateState();)

	BOOL isValid	 = FALSE;
	BOOL fToggleMode = !m_pThread->PreemptiveGCDisabled();
	if (fToggleMode)
		m_pThread->DisablePreemptiveGC();

	isValid = (m_pThread->GetThrowable() != NULL);

	if (fToggleMode)
		m_pThread->EnablePreemptiveGC();

	return isValid;
}
#endif

CLRUnwindStatus ExceptionTracker::ProcessCallFrame(DWORD_PTR dwControlPc, DWORD_PTR dwStackFrame, BOOL fIsFirstPass)
{
	INDEBUG(ValidateState();)

	EH_LOG((m_dwNestingLevel, LL_INFO100, "	 ProcessCallFrame %s pass\n", (fIsFirstPass ? "first" : "second")));

	_ASSERTE(ThrowableIsValid() && "UpdateThrowable has not been called on this object");

	GetMethodInfo(dwControlPc);

	EH_CLAUSE_ENUMERATOR	EnumState;
	EE_ILEXCEPTION_CLAUSE*	pEHClause		 = NULL;
	unsigned				EHCount;
	BOOL					fFoundHandler	 = FALSE;
	BOOL					fResumeFrame	 = (!fIsFirstPass && (dwStackFrame == m_dwResumeStackFrame));
	DWORD_PTR				dwHandlerStartPC = NULL;

	EHCount = m_pJitMan->InitializeEHEnumeration(m_MethodToken, &EnumState);
	for (unsigned i = 0; i < EHCount; i++)
	{
		pEHClause = m_pJitMan->GetNextEHClause(m_MethodToken, &EnumState, NULL);

		EH_LOG((m_dwNestingLevel, LL_INFO100, "	 considering %s clause [%x,%x], ControlPc at %x\n",
				(IsFault(pEHClause)			? "fault"	:
				(IsFinally(pEHClause)		? "finally" :
				(IsFilterHandler(pEHClause) ? "filter"	:
				(IsTypedHandler(pEHClause)	? "typed"	: "unknown")))),
				pEHClause->TryStartPC,
				pEHClause->TryEndPC,
				dwControlPc - m_dwMethodStartPC
				));

		BOOL fTermHandler = IsFaultOrFinally(pEHClause);
		fFoundHandler	  = FALSE;

		if (( fIsFirstPass &&  fTermHandler) ||
			(!fIsFirstPass && !fTermHandler))
		{
			continue;
		}

		if (ClauseCoversPC(pEHClause, dwControlPc))
		{
			EH_LOG((m_dwNestingLevel, LL_INFO100, "	 clause covers ControlPC\n"));

			dwHandlerStartPC = pEHClause->HandlerStartPC;

			if (fResumeFrame && (dwHandlerStartPC > m_dwCatchToCallPC))
			{
				EH_LOG((m_dwNestingLevel, LL_INFO100, "	 ignoring clause, we are catching before its handler\n"));
				continue;
			}

			if (IsFilterHandler(pEHClause))
			{
				DWORD_PTR dwResult;
				DWORD_PTR dwFilterStartPC;

				dwFilterStartPC = pEHClause->FilterOffset + m_dwMethodStartPC;

				EH_LOG((m_dwNestingLevel, LL_INFO100, "	 calling filter\n"));
				dwResult = CallHandler(dwFilterStartPC, dwStackFrame);
				if (EXCEPTION_EXECUTE_HANDLER == dwResult)
				{
					fFoundHandler = TRUE;
				}
				else if (EXCEPTION_CONTINUE_SEARCH != dwResult)
				{
					_ASSERTE(!"return value from filter was not EXCEPTION_EXECUTE_HANDLER or EXCEPTION_CONTINUE_SEARCH");
				}
				EH_LOG((m_dwNestingLevel, LL_INFO100, "	 filter returned %s\n", (fFoundHandler ? "EXCEPTION_EXECUTE_HANDLER" : "EXCEPTION_CONTINUE_SEARCH")));
			}
			else if (IsTypedHandler(pEHClause))
			{
				BOOL fToggleMode = !m_pThread->PreemptiveGCDisabled();
				if (fToggleMode)
					m_pThread->DisablePreemptiveGC();

				EEClass* pThrownClass = NULL;
				if (m_pThread->GetThrowable() != NULL)
					pThrownClass = m_pThread->GetThrowable()->GetTrueClass();

				if (pThrownClass)
				{
					if (((mdToken)(DWORD_PTR)(pEHClause->pEEClass)) == mdTypeRefNil)
					{
						// this is a catch(...)
						fFoundHandler = TRUE;
					}
					else
					{
						if (!HasCachedEEClass(pEHClause))
							m_pJitMan->ResolveEHClause(m_MethodToken, &EnumState, pEHClause);

						EH_LOG((m_dwNestingLevel, LL_INFO100, "	 clause type = %s\n", (HasCachedEEClass(pEHClause) ? pEHClause->pEEClass->m_szDebugClassName : "<couldn't resolve>")));
						EH_LOG((m_dwNestingLevel, LL_INFO100, "	 thrown type = %s\n", pThrownClass->m_szDebugClassName));
						fFoundHandler = HasCachedEEClass(pEHClause) && ExceptionIsOfRightType(pEHClause->pEEClass, pThrownClass);
					}
				}

				if (fToggleMode)
					m_pThread->EnablePreemptiveGC();
			}
			else
			{
				_ASSERTE(fTermHandler);
				fFoundHandler = TRUE;
			}

			if (fFoundHandler)
			{
				EH_LOG((m_dwNestingLevel, LL_INFO100, "	 found handler at 0x%p\n", dwHandlerStartPC));
				if (fIsFirstPass)
				{
					_ASSERTE(IsFilterHandler(pEHClause) || IsTypedHandler(pEHClause));

					m_dwCatchToCallPC		= dwHandlerStartPC;
					m_dwResumePC			= 0xBAADBAAD;	// we don't know the resume address until we call the catch.
					m_dwResumeStackFrame	= dwStackFrame;

					return FirstPassComplete;
				}
				else
				{
					_ASSERTE(fTermHandler);

					CallHandler(dwHandlerStartPC, dwStackFrame);
					//
					// will continue to find next fault/finally in this call frame
					//
				}
			}
		}
	}


	//
	// When doing the second pass unwind for resume, we need to call the catch that caught the
	// exception when we reach the destination frame and all the finallys have been called.
	//
	if (fResumeFrame)
	{
		EH_LOG((m_dwNestingLevel, LL_INFO100, "	 reached resume frame on second pass\n"));
		m_dwResumePC = NULL;
		if (m_dwCatchToCallPC)
		{
			EH_LOG((m_dwNestingLevel, LL_INFO100, "	 calling catch at 0x%p\n", m_dwCatchToCallPC));
			m_dwResumePC = CallHandler(m_dwCatchToCallPC, dwStackFrame);
			m_dwCatchToCallPC = NULL;
			EH_LOG((m_dwNestingLevel, LL_INFO100, "	 resume address should be 0x%p\n", m_dwResumePC));
		}

		EH_LOG((m_dwNestingLevel, LL_INFO100, "	 clearing throwable, signaling that this tracker should be deleted\n"));
		// we've reached the top stack frame
		m_pThread->SetThrowable(NULL);
		return SecondPassComplete;
	}




	return UnwindPending;
}

void ExceptionTracker::GetMethodInfo(DWORD_PTR dwControlPc)
{
	INDEBUG(ValidateState();)

	m_pJitMan = ExecutionManager::FindJitMan((BYTE*)dwControlPc);

	DWORD dwOffset;
	m_pJitMan->JitCode2MethodTokenAndOffset((BYTE*)dwControlPc, &m_MethodToken, &dwOffset);

	MethodDesc* pMD	  = m_pJitMan->JitTokenToMethodDesc(m_MethodToken);

#ifdef _DEBUG
	{
		EH_LOG((m_dwNestingLevel, LL_INFO100, "	   GetMethodInfo for %s::%s\n", pMD->m_pszDebugClassName, pMD->m_pszDebugMethodName));
	}
#endif

	m_dwMethodStartPC = GetMethodStartAddr(pMD);
}

DWORD_PTR ExceptionTracker::CallHandler(DWORD_PTR dwHandlerStartPC, DWORD_PTR dwStackFrame)
{
	INDEBUG(ValidateState();)

	_ASSERTE(!"NYI -- x86 call handler");
	return NULL;
}


void ExceptionTracker::UpdateThrowable()
{
	INDEBUG(ValidateState();)

	BOOL fToggleMode = !m_pThread->PreemptiveGCDisabled();
	if (fToggleMode)
		m_pThread->DisablePreemptiveGC();

	OBJECTREF  oThrowable = m_pThread->GetThrowable();
	if (NULL == OBJECTREFToObject(oThrowable))
	{
		if (EXCEPTION_COMPLUS == m_ExceptionCode)
		{
			oThrowable = m_pThread->LastThrownObject();
		}
		else
		{
			//
			//
			DWORD COMPlusExceptionCode = MapWin32FaultToCOMPlusException(m_ExceptionCode);

			GCPROTECT_BEGIN(oThrowable);
			CreateExceptionObject((RuntimeExceptionKind)COMPlusExceptionCode, &oThrowable);
			CallDefaultConstructor(oThrowable);
			GCPROTECT_END();
		}
		m_pThread->SetThrowable(oThrowable);
	}

	if (fToggleMode)
		m_pThread->EnablePreemptiveGC();
}

//
//
BOOL ExceptionTracker::ClauseCoversPC(EE_ILEXCEPTION_CLAUSE* pEHClause, DWORD_PTR dwControlPc)
{
	INDEBUG(ValidateState();)

	DWORD_PTR dwTryBegin	= pEHClause->TryStartPC;
	DWORD_PTR dwTryEnd		= pEHClause->TryEndPC;

	return ((dwTryBegin < dwControlPc) && (dwControlPc < dwTryEnd));
}


//
//
DWORD_PTR ExceptionTracker::GetMethodStartAddr(MethodDesc* pMD)
{
	{
		return (DWORD_PTR) pMD->GetAddrofJittedCode();
	}
}


#ifdef _DEBUG
void ExceptionTracker::ValidateState()
{
	if ((PRETAG_VALUE != m_dwPreTag) ||
		(POSTTAG_VALUE != m_dwPostTag))
	{
		_ASSERTE(!"ExceptionTracker::ValidateState FAILED");
	}
}
#endif
