/*
	Splunc - A truncated splay-tree benchmark
	Daniel Spoonhower <spoons@cs.cmu.edu>

	This program builds a splay tree of random integer values, truncating the
	resulting tree at a fixed depth after each insertion.  The lifetimes of
	objects inserted into the tree follow a log-normal distribution.

	The splay tree implementation was ported from a version written by Danny
	Sleator <sleator@cs.cmu.edu>.  The original is available at:
	http://www.link.cs.cmu.edu/link/ftp-site/splaying/ 
*/

using System;

public abstract class Node
{
  public int birthday;
  public int value;
  Node left, right;

	public Node() {}

  public Node(int birthday, int value)
  {
    this.birthday = birthday;
    this.value = value;
  }

  static Node splay(Node tree, Node node) 
  {
    Node l, r, y;
    if (tree == null) return tree;
    node.left = node.right = null;
    l = r = node;

    for (;;) {
      if (node.value <= tree.value) {
        if (tree.left == null) break;
        if (node.value <= tree.left.value) {
          y = tree.left;                           /* rotate right */
          tree.left = y.right;
          y.right = tree;
          tree = y;
          if (tree.left == null) break;
        }
        r.left = tree;                               /* link right */
        r = tree;
        tree = tree.left;
      } else if (node.value > tree.value) {
        if (tree.right == null) break;
        if (node.value > tree.right.value) {
          y = tree.right;                          /* rotate left */
          tree.right = y.left;
          y.left = tree;
          tree = y;
          if (tree.right == null) break;
        }
        l.right = tree;                              /* link left */
        l = tree;
        tree = tree.right;
      } else {
        break;
      }
    }
    l.right = tree.left;                                /* assemble */
    r.left = tree.right;
    tree.left = node.right;
    tree.right = node.left;
    return tree;
  }

  public static Node insert(Node tree, Node node) 
  {
    if (tree == null) {
      return node;
    }
    tree = splay(tree, node);
    
    if (node.value <= tree.value) {
			node.left = tree.left;
			node.right = tree;
			tree.left = null;
    } 
    else { // if (i > t->item)
			node.right = tree.right;
			node.left = tree;
			tree.right = null;
    }
    return node;
  }

  public static void trunc( int date, Node tree, int depth)
  {
    if (tree == null) {
      return;
    }

    if (tree.left != null) {
      if (depth == 0) {
        // Kill left
        //tree.left.printAges(date);
        tree.left = null;        
      }
      else {
        trunc( date, tree.left, depth - 1);
      }
    }

    if (tree.right != null) {
      if (depth == 0) {
        // Kill right
        //tree.right.printAges(date);
        tree.right = null;        
      }
      else {
        trunc( date, tree.right, depth - 1);
      }
    }
  }

  public void printAge( int date)
  {
    Console.WriteLine( date - birthday);
  }

  public void printAges( int date)
  {
    Console.WriteLine( date - birthday);
    if( left != null) {
      left.printAges( date);
    }
    if( right != null) {
      right.printAges( date);
    }
  }

  public void printTree( int date, string prefix)
  {
    Console.WriteLine(  prefix + "age: " + (date - birthday) + " value: " 
                        + value);
    if( left != null) {
      left.printTree( date, prefix + "  ");
    }
    if( right != null) {
      right.printTree( date, prefix + "  ");
    }
  }
}


class NodeA : Node {}
class NodeB : Node { public int x1 = 0; }
class NodeC : Node { public int x1 = 0; public int x2 = 0; }
class NodeD : Node { public double x1 = 0; public double x2 = 0;
	                   public double x3 = 0; public double x4 = 0;
	                   public double x5 = 0; public double x6 = 0;
	                   public double x7 = 0; public double x8 = 0; }

class Node28 : Node {}
class Node32 : Node { public int x1 = 0; }
class Node36 : Node { public int x1 = 0; public int x2 = 0; }
class Node40 : Node { public int x1 = 0; public int x2 = 0; 
	                    public int x3 = 0; }
class Node44 : Node { public int x1 = 0; public int x2 = 0; 
	                    public int x3 = 0; public int x4 = 0; }
class Node48 : Node { public int x1 = 0; public int x2 = 0; 
                      public int x3 = 0; public int x4 = 0;
                      public int x5 = 0; }

public class App
{
  static int date = 0;

  public static void Main(string[] args)
  {
		if (args.Length < 4) {
			Console.WriteLine("usage: splunc <seed> <max_size> <iterations> <trunc_depth> [<iterations> <trunc_depth>...]");
			Environment.Exit(1);
		}

		Random rand = new Random(Int32.Parse(args[0]));
		int max_size = Int32.Parse(args[1]);

    Node tree = new NodeA();
		tree.birthday = date++;
		tree.value = rand.Next(0, max_size);

		for (int j = 2; j + 2 <= args.Length; j += 2) {
			int N = Int32.Parse(args[j]);
			int depth = Int32.Parse(args[j + 1]);

			for (int i = 0; i < N; i++) {
				Node node;

				int value = rand.Next(0, 4);
				//node = new Node42();
				if (value < 1) {
					node = new NodeA();
				}
				else if (value < 2) {
					node = new NodeB();
				}
				else if (value < 3) {
				node = new NodeC();
				}
				else {
					node = new NodeD();
				}
				node.birthday = date++;
				node.value = rand.Next(0, max_size);

				tree = Node.insert(tree, node);
				Node.trunc(date, tree, depth);
			}

			//Console.WriteLine("-- " + depth + " --");
			//tree.printTree(date, "");
    }
  }
}
