#include "FEMEnergyFunction.h"
#include "SimulationMesh.h"

FEMEnergyFunction::FEMEnergyFunction(void){
	setToStaticsMode(0.001);
}

FEMEnergyFunction::~FEMEnergyFunction(void){
}

void FEMEnergyFunction::initialize(SimulationMesh* simMesh){
	assert(simMesh != NULL);
	assert(simMesh->nodes.size() >= 1);
	this->simMesh = simMesh;

	m_s0 = this->simMesh->x;
}

//regularizer looks like: r/2 * (p-p0)'*(p-p0). This function can update p0 if desired, given the current value of s.
void FEMEnergyFunction::updateRegularizingSolutionTo(const dVector &currentP){
	m_s0 = currentP;
}

//estimate accelerations given new estimated positions...
void FEMEnergyFunction::estimateNodalAccelerations(const dVector& xNew, dVector& acc){
	//a = (v_{t+h} - v_t)/ h = (x_{t+h} - x_t)/(h*h) - v_t/h;
	acc = xNew / (timeStep*timeStep) - simMesh->x / (timeStep*timeStep) - simMesh->v / timeStep;
}

//The net energy is: 1/2 a'M a + E + x'F, where E is the potential energy stored in the various elements
double FEMEnergyFunction::computeValue(const dVector& s){
	double totalEnergy = 0;

	for (uint i=0;i<simMesh->elements.size();i++)
		totalEnergy += simMesh->elements[i]->getEnergy(s, simMesh->X);
	
	for (uint i=0;i<simMesh->pinnedNodeElements.size();i++)
		totalEnergy += simMesh->pinnedNodeElements[i]->getEnergy(s, simMesh->X);

	if (useDynamics){
		int nDim = simMesh->x.size();
		//estimate the accelerations...
		estimateNodalAccelerations(s, tmpVec);
		//and now add a term to the energy that, when taken the derivative of, results in Ma (1/2 a'Ma)
		for (int i=0;i<nDim;i++)
			totalEnergy += 0.5 * tmpVec[i] * simMesh->m[i] * tmpVec[i] * timeStep*timeStep;
	}
	totalEnergy -= s.dot(simMesh->f_ext);

	//add the regularizer contribution
	if (regularizer > 0){
		tmpVec = s - m_s0;
		totalEnergy += 0.5*regularizer*tmpVec.dot(tmpVec);
	}

	return totalEnergy;
}

//this method gets called whenever a new best solution to the objective function is found
void FEMEnergyFunction::setCurrentBestSolution(const dVector& s){
	updateRegularizingSolutionTo(s);
}





