#include <GUILib/GLUtils.h>
#include "CSTElement2D.h"
#include <FEMSimLib/SimulationMesh.h>

#define SCALE_FACTOR 10000000.0

CSTElement2D::CSTElement2D(SimulationMesh* simMesh, Node* n1, Node* n2, Node* n3) : SimMeshElement(simMesh) {
//	shearModulus = 0.3 * 10e9 / SCALE_FACTOR;
//	bulkModulus = 1.5 * 10e9 / SCALE_FACTOR;

	this->n[0] = n1;
	this->n[1] = n2;
	this->n[2] = n3;

	setRestShapeFromCurrentConfiguration();

	//distribute the mass of this element to the nodes that define it...
	for (int i = 0;i<3;i++)
		n[i]->addMassContribution(getMass() / 3.0);

	matModel = MM_LINEAR_ISOTROPIC;
//	matModel = MM_STVK;
//	matModel = MM_NEO_HOOKEAN;

}

CSTElement2D::~CSTElement2D(){
}

void CSTElement2D::setRestShapeFromCurrentConfiguration(){
	//edge vectors
	V3D V1(n[0]->getWorldPosition(), n[1]->getWorldPosition()), V2(n[0]->getWorldPosition(), n[2]->getWorldPosition());
	//matrix that holds three edge vectors
	Matrix2x2 dX;
	dX << V1[0], V2[0], 
		  V1[1], V2[1];

	dXInv = dX.inverse();

	//compute the area of the element...
	restShapeArea = computeRestShapeArea(this->simMesh->X);
//	Logger::logPrint("CSTElement2D Element area: %lf\n", restShapeArea);
}


double CSTElement2D::getMass() {
	return restShapeArea * massDensity;
}

double CSTElement2D::computeRestShapeArea(const dVector& X) {
	P3D p1 = n[0]->getCoordinates(X);
	P3D p2 = n[1]->getCoordinates(X);
	P3D p3 = n[2]->getCoordinates(X);
	V3D V1(p1, p2), V2(p1, p3);
	//now compute the area of the element...
	return 1 / 2.0 * fabs(V1.cross(V2).length());
}

void CSTElement2D::addEnergyGradientTo(const dVector& x, const dVector& X, dVector& grad) {
	//compute the gradient, and write it out
	computeGradientComponents(x, X);
	for (int i = 0;i<3;i++)
		for (int j = 0;j<2;j++)
		grad[n[i]->dataStartIndex + j] += dEdx[i][j];
}

void CSTElement2D::addEnergyHessianTo(const dVector& x, const dVector& X, std::vector<MTriplet>& hesEntries) {
    //compute the hessian blocks and 
    computeHessianComponents(x, X);
    for (int i = 0;i<3;i++)
        for (int j = 0;j < 3;j++)
            addSparseMatrixDenseBlockToTriplet(hesEntries, n[i]->dataStartIndex, n[j]->dataStartIndex, ddEdxdx[i][j], true);
}

void CSTElement2D::draw(const dVector& x) {
//	glColor3d(1, 1, 1);
	glBegin(GL_LINES);
	for (int i = 0; i<2;i++)
		for (int j = i + 1;j<3;j++) {
			P3D pi = n[i]->getCoordinates(x);
			P3D pj = n[j]->getCoordinates(x);
			glVertex3d(pi[0], pi[1], pi[2]);
			glVertex3d(pj[0], pj[1], pj[2]);
		}
	glEnd();
}

void CSTElement2D::drawRestConfiguration(const dVector& X) {
	glColor3d(1, 0, 0);
	glBegin(GL_LINES);
	for (int i = 0; i<2;i++)
		for (int j = i + 1;j<3;j++) {
			P3D pi = n[i]->getCoordinates(X);
			P3D pj = n[j]->getCoordinates(X);
			glVertex3d(pi[0], pi[1], pi[2]);
			glVertex3d(pj[0], pj[1], pj[2]);
		}
	glEnd();
}

/**
	F maps deformed vectors dx to undeformed coords dX: dx = F*dX (or, in general F = dx/dX. By writing x as a weighted combination of 
	node displacements, where the weights are computed using basis/shape functions, F can be computed anywhere inside the element).
	For linear basis functions, F is constant throughout the element so an easy way to compute it is by looking at the matrix  that 
	maps deformed traingle/tet edges to their underformed counterparts: v = FV, where v and V are matrices containing edge vectors 
	in deformed and undeformed configurations
*/
void CSTElement2D::computeDeformationGradient(const dVector& x, const dVector& X, Matrix2x2& dxdX){
	// TODO: Implement deformation gradient.
}

double CSTElement2D::getEnergy(const dVector& x, const dVector& X){
	//compute the deformation gradient
	computeDeformationGradient(x, X, F);
	double energyDensity = 0;

	// The field shearModulus corresponds to mu, the Young's modulus
	// The field bulkModulus corresponds to lambda, the Poisson's ratio

	if (matModel == MM_STVK){
		// TODO: Implement St. Venant-Kirchhoff model.
		// You should compute the energy density, and add the result to energyDensity.

	}

	else if (matModel == MM_LINEAR_ISOTROPIC){
		// TODO: Implement linear isotropic model
	}

	else if (matModel == MM_NEO_HOOKEAN){
		// TODO: Implement Neo-Hookean model
	}

	return energyDensity * restShapeArea;

}

void CSTElement2D::computeGradientComponents(const dVector& x, const dVector& X){
	//compute the gradient of the energy using the chain rule: dE/dx = dE/dF * dF/dx. dE/dF is the first Piola-Kirchoff stress sensor, for which nice expressions exist.

	//compute the deformation gradient, storing it in F
	computeDeformationGradient(x, X, F);
	dEdF.setZero();
	Finv = F.inverse();
	FinvT = Finv.transpose();

	// The field shearModulus corresponds to mu, the Young's modulus
	// The field bulkModulus corresponds to lambda, the Poisson's ratio

	if (matModel == MM_STVK){
		// TODO: Implement St. Venant-Kirchhoff model.
		// The result (the variable P in the notes) should be written into dEdf.
	}
	else if (matModel == MM_LINEAR_ISOTROPIC){
		// TODO: Implement linear isotropic model.
	}
	else if (matModel == MM_NEO_HOOKEAN){
		// TODO: Implement Neo-Hookean model.
	}

	//dF/dx is going to be some +/- Xinv terms. The forces on nodes 1,2 can be writen as: dE/dF * XInv', while the force on node 0 is -f1-f2;
	dEdx[1] = V3D(dEdF(0,0) * dXInv(0,0) + dEdF(0,1) * dXInv(0,1), dEdF(1,0) * dXInv(0,0) + dEdF(1,1) * dXInv(0,1), 0) * restShapeArea;
	dEdx[2] = V3D(dEdF(0,0) * dXInv(1,0) + dEdF(0,1) * dXInv(1,1), dEdF(1,0) * dXInv(1,0) + dEdF(1,1) * dXInv(1,1), 0) * restShapeArea;
	dEdx[0] = -dEdx[1]-dEdx[2];
}

void CSTElement2D::computeHessianComponents(const dVector& x, const dVector& X) {
	//this function is not used
}