// Author: Sol Boucher <sboucher@cmu.edu>

#include <netinet/in.h>
#include <sys/socket.h>
#include <ctype.h>
#include <netdb.h>
#include <pthread.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define HOSTNAME_LEN 64
#define PORTNUM_LEN 8
#define QUEUE_LEN 8

static pthread_mutex_t visit_lock;
static int visitors = 0;

static void *serveclient(void *fd) {
	pthread_detach(pthread_self());

	int infd = *(int *) fd;
	free(fd);

	int outfd = dup(infd);
	if(outfd < 0) {
		perror("dup()");
		exit(9);
	}

	FILE *in = fdopen(infd, "r");
	if(!in) {
		perror("fdopen()");
		exit(9);
	}

	const char *terminator = "\r\n\r\n";
	int ctr = 0;
	int chr;
	while(ctr < 4 && (chr = getc(in)) != EOF) {
		putchar(chr);
		if(chr == terminator[ctr])
			++ctr;
		else
			ctr = 0;
	}
	fclose(in);

	FILE *out = fdopen(outfd, "w");
	if(!out) {
		perror("fdopen()");
		exit(9);
	}

	pthread_mutex_lock(&visit_lock);
	int hit_ct = ++visitors;
	pthread_mutex_unlock(&visit_lock);

	fprintf(out,
		"HTTP/1.0 200 OK\r\n"
		"Content-Type: text/plain; charset=UTF-8\r\n"
		"\r\n"
		"You are visitor number: %d\r\n",
		hit_ct);
	fclose(out);

	puts("# connection closed");
	return NULL;
}

int main(int argc, char **argv) {
	if(argc != 2) {
		printf("USAGE: %s <port>\n", argv[0]);
		return 1;
	}
	int port = atoi(argv[1]);

	// 1. socket(family, socktype, transport)
	int sockfd;
	if((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("socket()");
		return 2;
	}

	// 2. bind(fd, addr, size)
	struct sockaddr_in addr = {
		.sin_family = AF_INET,
		.sin_port = htons(port),
		.sin_addr = {INADDR_ANY},
	};
	if(bind(sockfd, (struct sockaddr *) &addr, sizeof addr)) {
		perror("bind()");
		return 3;
	}

	// 3. listen(fd, len)
	if(listen(sockfd, QUEUE_LEN)) {
		perror("listen()");
		return 4;
	}

	pthread_mutex_init(&visit_lock, NULL);
	while(true) {
		// 4. accept(fd, addr, size)
		struct sockaddr_in peer;
		socklen_t plen = sizeof peer;
		int infd;
		if((infd = accept(sockfd, (struct sockaddr *) &peer, &plen)) < 0) {
			perror("accept()");
			return 5;
		}

		// 5. getnameinfo(addr, size, name, size, port, size, flags)
		char hostname[HOSTNAME_LEN];
		char portnum[PORTNUM_LEN];
		int stat;
		if((stat = getnameinfo((struct sockaddr *) &peer, plen, hostname, sizeof hostname, portnum, sizeof portnum, 0))) {
			fprintf(stderr, "%s: %s\n", "getnameinfo()", gai_strerror(stat));
			return 6;
		}

		printf("# incoming connection from %s:%s\n", hostname, portnum);

		int *fd = malloc(sizeof *fd);
		if(!fd) {
			perror("malloc()");
			return 7;
		}
		*fd = infd;

		pthread_t tid;
		if(pthread_create(&tid, NULL, serveclient, fd)) {
			perror("pthread_create()");
			return 8;
		}
	}

	close(sockfd);

	return 0;
}
