/* -*- Mode: C++ -*-
 *Header:
 *File: logplayer.C
 *Author: Noda Itsuki
 *Date: 1996/11/28
 *EndHeader:
 */


#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <signal.h>
#include <sys/time.h>
#include <strstream.h>

#include "param.h"
#include "types.h"
#include "logplayer.h"

#define Debug(x) 

extern Player *Play ;
 
static void update(void) ;

Player::Player()
{

	m_plFilePos = new long [MAX_SHOWINFO];
	if (m_plFilePos == NULL)
		exit(-1);

	controler = new Controler();
	if (controler == NULL)
		exit(-1);
}

Player::~Player()
{
	delete controler;
	delete [] m_plFilePos;
}

void Player::init(int argc, char *argv[])
{
	GetOption(argc, argv) ;

	/* open input file */
	OpenForRead(in_file);	

	/* open output file */
	OpenForWrite(out_file);

	state = STATE_WAIT ;
	limit = 10000 ;
	sent = 0 ;
	rect = 0 ;
	to_time = NONE ;
	rec_state = REC_OFF ;
	current = 1;
	port.init() ;

	/* Pat added this! */
	team_size = 11;
	al_player = 1;
	al_level = 0;
	curr_action_log_fp = NULL;
	curr_action_log_player = 0;
	/* end Pat added this */
	
	/* X logplayer */
	if (!no_window) {
		controler->init(argc, argv) ;
		timer_init() ;
		controler->display_status() ;
		controler->Loop() ;
	}
	/* no window logplayer */
	else {
		/* open command file */
		if (com_file[0] != '\0') {
			if ((comfp = fopen(com_file, "r")) == NULL) {
				cerr << "Can't open log file " << com_file << endl ;
				fclose(infp) ;
				if (out_file[0] != '\0')
					fclose(outfp) ;
				exit(-1) ;
			}
		}
		nw_Init() ;
		nw_TimerInit() ;
		nw_Loop() ;
		nw_Quit() ;
	}
}

void Player::GetOption(int argc, char **argv)
{
	option_t opt[] = {
		{"out",					(void *)&(out_file),		V_STRING},
		{"port",				(void *)&(port.portnum),	V_INT},
		{"nw",					(void *)&(no_window),		V_NONE},
		{"nowindow",			(void *)&(no_window),		V_NONE},
		{"f",					(void *)&(com_file),		V_STRING},
		{"show_action_logs",                    (void *)&show_action_logs,      V_ONOFF},
		{"action_logs_dir",                     (void *)&action_logs_dir,       V_STRING},
		{"\0",					NULL,						0}

	} ;

	char buf[MaxMesg] ;
	char com[256] ;
	char onoff[16] ;
	int i, p, n ;
	int flag = 0;
	FILE *fp ;

	/* set default value */
	in_file[0] = '\0' ;
	out_file[0] = '\0' ;
	com_file[0] = '\0' ;
	port.portnum = DEFAULT_PORT_NUMBER ;
	no_window = FALSE ;
	show_action_logs = FALSE;
	action_logs_dir[0] = 0;
	
	/* skip command name */
	argv++ ; argc-- ;

	/* first, search option '-file' */
	for(i = 0 ; i < argc ; i++) {
		if (!strcmp(*(argv + i), "-file")) {
			if (i == argc - 1) {
				cerr << "can't find file name" << endl ;
				break ;
			}

			if ((fp = fopen(*(argv+i+1), "r")) == NULL) {
				cerr << "can't open config file " << *(argv+i+1) << endl ;
				break ;
			}

			while(fgets(buf,MaxMesg,fp) != NULL) {
				/* ignore remark line */
				if (buf[0] == '#' || buf[0] == '\n')
					continue ;

				/* replace from ':' to ' ' */
				char *r = buf ;
				while(*r != '\0') {
					if (*r == ':') *r = ' ' ;
					r++ ;
				}

				n = sscanf(buf, "%s", com) ;
				if (n < 1) {
					cerr << "Illegal line : " << buf ;
					continue ;
				}

				for (p = 0 ; opt[p].vptr != NULL ; p++) {
					if (strcmp(com, opt[p].optname))
						continue ;

					/* match */
					switch(opt[p].vsize) {
					case V_INT:
						n = sscanf(buf, "%s %d", com, (int *)opt[p].vptr) ;
						break ;

					case V_DOUBLE:
						n = sscanf(buf, "%s %lf", com, (double *)opt[p].vptr) ;
						break ;

					case V_STRING:
						n = sscanf(buf, "%s %s", com, (char *)opt[p].vptr) ;
						break ;

					case V_ONOFF:
						n = sscanf(buf, "%s %s", com, onoff) ;
						*((int *)opt[p].vptr) = (!strcmp(onoff, "on")) ? 
							TRUE : FALSE ;
						break ;

					case V_NONE:
						*((int *)opt[p].vptr) = TRUE ;
						n = 2 ;
						break ;
					}

					if (n > 2)
						cerr << "Illegal line (" << com << ") " << endl ;

					break ;
				}

				if (opt[p].vptr == NULL)
					cerr << "Illegal line (" << com << ") " << endl ;
			}

			fclose(fp) ;
			break ;
		}
	}

	/* next, analyze command line option */
	while (argc) {
		if (!strcmp(*argv, "-file")) {
			argv += 2 ;
			argc -= 2 ;
			continue ;
		}

		if (strncmp(*argv, "-", 1)){
			if (!flag++)
				strcpy(in_file, *argv) ;
			else
				in_file[0] = '\0' ;
			argv++ ;
			argc-- ;
			continue ;
		}

		for (p = 0 ; opt[p].vptr != NULL ; p++) {
			if (strcmp(*argv + 1, opt[p].optname))
				continue ;

			/* match */
			argv++ ;
			argc-- ;

			switch(opt[p].vsize) {
			case V_INT:
				*((int *)opt[p].vptr) = atoi(*argv) ;
				break ;

			case V_DOUBLE:
				*((double *)opt[p].vptr) = atof(*argv) ;
				break ;

			case V_STRING:
				strcpy((char *)opt[p].vptr, *argv) ;
				break ;
				
			case V_ONOFF:
				strcpy(onoff, *argv) ;
				*((int *)opt[p].vptr) = (!strcmp(onoff, "on")) ? 
					TRUE : FALSE ;
				break ;
			case V_NONE:
				*((int *)opt[p].vptr) = TRUE ;
				argv-- ;
				argc++ ;
				break ;
			}

			break ;
		}

		if (opt[p].vptr == NULL)
			cerr << "Unrecognized Opion : " << *argv << endl ;

		argv++ ;
		argc-- ;
	}

	if (action_logs_dir[strlen(action_logs_dir)-1] == '/') {
	  /* remove the slash */
	  action_logs_dir[strlen(action_logs_dir)-1] = 0;
	}

}

void Player::Quit(void)
{
	if (fclose(infp))
		cerr << "Can't close input file" << endl ;

	if (out_file[0] != '\0') {
		fflush(outfp) ;
		if (fclose(outfp))
			cerr << "Can't close output file" << endl ;
	}

	if (curr_action_log_fp != NULL) {
	  fclose(curr_action_log_fp);
	}
	
	
	exit(1) ;
}

void Player::timer_init(void)
{
#ifndef X11R5
	struct sigaction alarm_action ;
	struct itimerval itv ;
	void signal_handler(void) ;

	sig_id = XtAppAddSignal(controler->app_context,
							(XtSignalCallbackProc)update, NULL) ;
	itv.it_interval.tv_sec = 0 ;
	itv.it_interval.tv_usec = TIMEDELTA * 1000 ;
	itv.it_value.tv_sec = 0 ;
	itv.it_value.tv_usec = TIMEDELTA * 1000 ;

	alarm_action.sa_handler = (void (*)(...))signal_handler ;
	alarm_action.sa_flags &= (~SA_RESETHAND) ;
	sigaction(SIGALRM, &alarm_action, NULL) ;
	setitimer(ITIMER_REAL, &itv, NULL) ;

#else
	XtIntervalId intid ;
	intid = XtAppAddTimeOut(controler->app_context,
							TIMEDELTA, (XtTimerCallbackProc)update, NULL) ;
#endif

}


void Player::read_file(void) 
{
	// read until a new showinfo is reached
	m_bReadShowInfo = 0;
        while (m_bReadShowInfo == 0) {
		if (ReadLog() == 0) { 
			StopLog();
			return ;
		}

		SendLog();
		DisplayActionLog();
		WriteLog();
	}

	if (current == to_time)		
		StopLog();
}

void Player::rewind_file(void) 
{
	// is there a cycle to rewind to ?
	if ((current <= 1) || (fseek(infp, m_plFilePos[current-1], SEEK_SET) == -1)) {
		clearerr(infp) ;
		RewindLog() ;
		StopLog();
		return ;
	}

	if (ReadLog() == 0){
		StopLog();
		return ;
	}

	SendLog();
	DisplayActionLog();
	WriteLog();

	if (current == to_time)
		StopLog();
}

void Player::jump(void) 
{
	long offset ;

	offset = ftell(infp) ;

	if (to_time == END) {
		fseek(infp, m_plFilePos[m_iMaxPos], SEEK_SET) ;
		for ( ; ; ) {
			// read to the end of the file
			if (ReadLog() == 0) {
				fseek(infp, m_plFilePos[m_iMaxPos], SEEK_SET) ;
				StopLog();
				state = STATE_NOT_JUMP ;
				return ;
			}
		}
	}
	else if (to_time <= m_iMaxPos) {
		// just jump to the position
		fseek(infp, m_plFilePos[to_time], SEEK_SET);
		if (ReadLog() == 0) {
			clearerr(infp) ;
			RewindLog() ;
			StopLog();
		}
	}
	else {		
		// we did not read so far up to now
		fseek(infp, m_plFilePos[m_iMaxPos], SEEK_SET) ;
		for ( ; ; ) {
			if (ReadLog() == 0) {
				clearerr(infp) ;
				RewindLog() ;
				StopLog();
				return;
			}
			else if (current == to_time) {
				to_time = NONE ;
				break ;
			}
		}
	}

	SendLog();
	DisplayActionLog();
	WriteLog();
	StopLog();
}

void Player::send_blank(void) 
{
	m_tBuf.mode = htons(BLANK_MODE) ;
	controler->display_time(ntohs((unsigned short)-1)) ;

	SendLog();
	DisplayActionLog();
	WriteLog();
}

static void update(void)
{
#ifdef X11R5
	XtIntervalId intid ;
#endif

	Play->sent += TIMEDELTA ;
	Play->rect += TIMEDELTA ;

	if (Play->sent >= Play->limit ) {
		switch (Play->state) {
		case STATE_PLAY:
			Play->limit = PLAY_CYCLE ;
			Play->read_file() ;
			break ;

		case STATE_STOP:
			Play->limit = PLAY_CYCLE ;
			break ;

		case STATE_REVERSE:
			Play->limit = PLAY_CYCLE ;
			Play->rewind_file() ;
			break ;

		case STATE_REW:
			Play->limit = FEED_CYCLE ;
			Play->rewind_file() ;
			break ;

		case STATE_FOR:
			Play->limit = FEED_CYCLE ;
			Play->read_file() ;
			break ;

		case STATE_PLUS:
			Play->limit = 10 ;
			Play->to_time = Play->current + 1 ;
			Play->read_file() ;
			break ;

		case STATE_MINUS:
			Play->limit = 10;
			Play->to_time = Play->current - 1 ;
			Play->rewind_file() ;
			break ;

		case STATE_BLANK:
			Play->limit = PLAY_CYCLE ;
			Play->send_blank() ;
			break ;

		case STATE_JUMP:
			Play->limit = 10 ;
			Play->jump() ;
			break ;

		default:
			break ;
		}

		Play->sent = 0 ;
	}
	if (Play->rect >= 400) {
		Play->port.recv_info() ;
		if(Play->state == STATE_WAIT) {
			if(strcmp(Play->port.rbuf, "(dispinit)")) {
				Play->sent = 0 ;		
			}
			else {
				Play->limit = 100 ;
				Play->state = STATE_STOP ;
				Play->controler->display_status() ;
			}
		}
		Play->rect = 0 ;
	}
#ifdef X11R5
	intid = XtAppAddTimeOut(Play->controler->app_context,
							TIMEDELTA, (XtTimerCallbackProc)update, NULL) ;
#endif
}

#ifndef X11R5
void signal_handler(void)
{
	XtNoticeSignal(Play->sig_id) ;
}
#endif

// added 16.04.98 (Klaus Dorer)
// I think this should reduce logfile sizes by a factor of 3 to 7

int Player::ReadLog(void)
{
	short		len;
	long		pos;

	pos  = ftell(infp);

	// m_iVersion may contain the version information of the logfile to be
	// able to read and write older versions
	switch (m_iVersion)
	{
	case REC_OLD_VERSION:
		// it is a logfile of the previous version
		if (fread((char *) (&m_tBuf), sizeof(dispinfo_t), 1, infp) == 0)
		  return 0;
		if (ntohs(m_tBuf.mode) == SHOW_MODE) 
			NewShowinfo(pos);
		else
			m_bReadShowInfo = 0;
		return TRUE;

	case REC_VERSION_2:
		// new version 
		// read the type of next msg
		if (fread((char *)(&m_tBuf.mode), sizeof(m_tBuf.mode), 1, infp) == 0)
			return FALSE;

		switch (ntohs(m_tBuf.mode))
		{
		case SHOW_MODE:
			// read a show struct
			if (fread((char *)(&m_tBuf.body.show),
									sizeof (m_tBuf.body.show), 1, infp) == 0)
				return FALSE;
			NewShowinfo(pos);
			return TRUE;

		case MSG_MODE:
			// it is a message
			m_bReadShowInfo = 0;
			// read the board info
			if (fread((char *)(&m_tBuf.body.msg.board), sizeof(m_tBuf.body.msg.board), 1, infp) == 0)
				return 0;

			// read the string length
			if (fread((char *)(&len), sizeof(len), 1, infp) == 0)
				return 0;

			// read the message
			if (fread(m_tBuf.body.msg.message, len, 1, infp) == 0)
				return 0;
			else
				return 1;
		}
	}
}

void Player::WriteLog(void)
{
	short		len = 1;

	// recording ?
	if(rec_state != REC_ON)
		return;

       	// write the type of next msg
       	fwrite((char *)(&m_tBuf.mode), sizeof(m_tBuf.mode), 1, outfp);

       	switch (ntohs(m_tBuf.mode))
       	{
       	case SHOW_MODE:
       		// write a show struct
       		fwrite((char *)(&m_tBuf.body.show), sizeof (m_tBuf.body.show), 1, outfp);
       		break;

       	case MSG_MODE:
       		// it is a message
       		// write the board info
       		fwrite((char *)(&m_tBuf.body.msg.board), sizeof(m_tBuf.body.msg.board), 1, outfp);

       		// calculate the string length and write it
       		while ((m_tBuf.body.msg.message[len-1] != '\0') && (len < 2048))
       			len++;
       		fwrite((char *)(&len), sizeof(len), 1, outfp);

       		// write the message
       		fwrite(m_tBuf.body.msg.message, len, 1, outfp);
       		break;
       	}
}

void Player::SendLog(void)
{
	displist_t *p ;

	for (p = port.top.next; p != NULL; p = p->next)
		port.send_info(&m_tBuf, p->shost, p->sport) ;
}

void Player::RewindLog(void)
{
	switch (m_iVersion)
	  {
	  case REC_OLD_VERSION:
	    rewind(infp);
	    break;

	  case REC_VERSION_2:
	    // there are 4 bytes at the beginning specifying the version,
		// so overread
	    fseek(infp, 4L, SEEK_SET);
	    break;
	  }
}

void Player::StopLog(void)
{
        state = STATE_STOP ;
       	to_time = NONE ;
       	if (!no_window) {
       		controler->display_time(current) ;
       		controler->display_status() ;
       		controler->buttons_reset() ;
       	}	
}


/* returns a pointer to a static buffer, so be careful! */
char* repeat_char(char c, int n)
{
  const int MAX_REP = 100;
  static char out[MAX_REP+1];
  if (n > MAX_REP)
    n = MAX_REP;
  for (int i=0; i<n; i++)
    out[i] = c;
  out[n] = 0;
  return out;
}
void Player::DisplayActionLog(void)
{
  //printf("I should show player %d at level %d (%d,%d) \n",
  //al_player, al_level, show_action_logs, ntohs(m_tBuf.mode));
  if (!show_action_logs || ntohs(m_tBuf.mode) != SHOW_MODE)
    return;

  if (curr_action_log_player != al_player) {
    /* we need to open the right log file */
    if (curr_action_log_fp)
      fclose(curr_action_log_fp);
    char fn[MAX_FILE_LEN];
    sprintf(fn, "%s/%s%d-%c-actions.log", action_logs_dir,
	    (al_player <= team_size ? m_tBuf.body.show.team[0].name : m_tBuf.body.show.team[1].name),
	    ((al_player-1) % 11) + 1,
	    (al_player <= team_size ? 'l' : 'r'));
    if (!(curr_action_log_fp = fopen (fn, "r"))) {
      char outstr[200];
      sprintf(outstr, "Failed to open file '%s'", fn);
      perror(outstr);
      curr_action_log_player = 0;
      return;
    }
    curr_action_log_player = al_player;
  }
  
  if (al_level <= 0)
    return;
  printf("\n\n\nAction log for %s %d, level %d, at time %d\n",
	 (al_player <= team_size ? m_tBuf.body.show.team[0].name : m_tBuf.body.show.team[1].name),
	 ((al_player-1) % 11) + 1,
	 al_level, ntohs(m_tBuf.body.show.time));
  if (MoveLogToTime(ntohs(m_tBuf.body.show.time))) {
    while(1) {
      /* print out the log lines! */
      char line[MAX_LOG_LINE_SIZE];
      LogLineInfo llinfo;
      int pos = ftell(curr_action_log_fp);
      
      if (!fgets(line, MAX_LOG_LINE_SIZE, curr_action_log_fp))
	break; /* ran out of file */
      llinfo = ProcessLine(line);
      if (llinfo.msg == NULL ||
	  llinfo.time_t != ntohs(m_tBuf.body.show.time) ||
	  llinfo.time_s > 0) {
	fseek(curr_action_log_fp, pos, SEEK_SET);
	break;
      }
      
      if (llinfo.msg == NULL) {
	printf("<ERROR processign line>\n");
	break;
      }

      //the newline should already be on the msg
      if (llinfo.level > al_level)
	continue;
      printf("%s%s", repeat_char('-', llinfo.level/10), llinfo.msg);
    }
  } else {
    printf("<ERROR- Could not find time in logfile>\n");
  }
  printf("\n\n");
}


/* we assume that the log file is always at the beginning of a line */
int Player::MoveLogToTime(int targ_time) 
{
  char line[MAX_LOG_LINE_SIZE];
  int pos;
  LogLineInfo llinfo;

  pos = ftell(curr_action_log_fp);
  do {
    if (!fgets(line, MAX_LOG_LINE_SIZE, curr_action_log_fp))
      return 0;
  } while (line[0] == '\n');
      
  llinfo = ProcessLine(line);

  if (llinfo.time_t == targ_time && llinfo.time_s == 0) {
    /* if the thing we just read has the right t time and s of 0, we're going to assume
       that it's the begginning of this section
       This will make the normal case of goign through faster */
    fseek(curr_action_log_fp, pos, SEEK_SET);
    return 1; /* we got there */
  }

  int seek_backwards_steps = 1000;
  if (llinfo.time_t > targ_time ||
      (llinfo.time_t == targ_time && llinfo.time_s > 0)) {
    /* we need to scan backwards */
    while(1) {
      fseek(curr_action_log_fp, -seek_backwards_steps, SEEK_CUR);
      if (!fgets(line, MAX_LOG_LINE_SIZE, curr_action_log_fp)) // just get to the next line
	return 0;
      do {
	if (!fgets(line, MAX_LOG_LINE_SIZE, curr_action_log_fp))
	  return 0;
      } while (line[0] == '\n');
      llinfo = ProcessLine(line);

      if (llinfo.msg == NULL) {
	Debug(printf("Scan backwards: could not process line\n"));
	return 0; /* The line could not be processed - something bad happened */
      }

      Debug(printf("Scan backwards: at time %d.%d on way to %d\n",
	    llinfo.time_t, llinfo.time_s, targ_time));
      
      if (targ_time > llinfo.time_t) {
	/* we got to the time step before. Break and scan forards from here */
	Debug(printf("Scan backwards: switching to forwards we should be there now\n"));
	break;
      }
    } /* while(1) */
    
  }

  /* we need to scan forwards - this part is easy */
  while (1) {
    /* this shoudl make scanning forwards much faster */
    int avg_chars_per_line = 80;
    if (targ_time - 10 > llinfo.time_t) {
      Debug(printf("Skipping!\n"));
      if (fseek(curr_action_log_fp, 5 *(targ_time - llinfo.time_t)* avg_chars_per_line, SEEK_CUR) != 0) {
	Debug(printf("Scan forwards: skipping failed\n"));
	return 0;
      }
      if (!fgets(line, MAX_LOG_LINE_SIZE, curr_action_log_fp)) {
	Debug(printf("Scan forwards: after skip read failed\n"));
	return 0;
      }
    }
    
    do {
      pos = ftell(curr_action_log_fp);
      if (!fgets(line, MAX_LOG_LINE_SIZE, curr_action_log_fp)) {
	Debug(printf("Scan forwards: I ran out of file to scan\n"));
	return 0;
      }
    } while (line[0] == '\n');
    
    llinfo = ProcessLine(line);
    
    if (llinfo.msg == NULL) {
      Debug(printf("Scan forwards: could not process line\n"));
      return 0; /* The line could not be processed - something bad happened */
    }
    
    Debug(printf("Scan forwards: at time %d.%d on way to %d\n",
		 llinfo.time_t, llinfo.time_s, targ_time));
    
    if (llinfo.time_t == targ_time && llinfo.time_s == 0) {
      fseek(curr_action_log_fp, pos, SEEK_SET);
      Debug(printf("Scan forwards: I went to the right place\n"));
      return 1; /* we got there */
    }
    if (llinfo.time_t > targ_time ||
	(llinfo.time_t == targ_time && llinfo.time_s > 0)) {
      Debug(printf("Scan forwards: that part of the log is missing\n"));
      return 0;
    }
  } /* while(1) */
}

LogLineInfo Player::ProcessLine(char* line) 
{
  LogLineInfo llinfo;
  llinfo.msg = NULL;
  
  int n = sscanf(line, "%d.%d", &llinfo.time_t, &llinfo.time_s);
  if (n != 2)
    return llinfo;

  char* pc = line;
  while (*pc != ' ' ) {
    if (pc - line > MAX_LOG_LINE_SIZE)
      return llinfo;
    else
      pc++; /* advance to the space between time and level - */
  }
  pc++; /* skip space, should not be on the dashes (if there are any) */
    
  /* count the dashes for the level */
  llinfo.level = 0;
  while (*pc == '-') {
    if (pc - line > MAX_LOG_LINE_SIZE)
      return llinfo;
    else {
      llinfo.level += 10;
      pc++; /* advance past this - */
    }
  }

  llinfo.msg = pc;
  return llinfo;
}


void Player::OpenForRead(char* in_file)
{
	char localbuf[6];

	// check if an input file is specified
	if (in_file[0] == '\0') {
		cerr << "Usage: logplayer file_name\n\t\t[-out file_name]\n\t\t";
		cerr << "[-port num]\n\t\t[-nw]\n\t\t[-file file_name]\n" << endl ;
		exit(-1) ;
	}

	// open the file
	if ((infp = fopen(in_file, "rb")) == NULL) {
	       	cerr << "Can't open log file " << in_file << endl ;
       		exit(-1) ;
	}

	// default is old version
	m_iVersion = REC_OLD_VERSION;
	// have not read anything from file up to now
	m_iMaxPos = 0;
	// read version information
	if (fread(localbuf, 4, 1, infp) == 0)
	{
		// could not read from file
		fclose(infp);
	       	cerr << "Can't open log file " << in_file << endl ;
       		exit(-1) ;
	}

	// check version
	if ((localbuf[0] == 'U') && (localbuf[1] == 'L') && (localbuf[2] == 'G'))
		m_iVersion = localbuf[3];
	else
		// no version information, so it is an old logfile
		// (or no logfile at all)
		rewind(infp);
}

void Player::OpenForWrite(char* out_file)
{
	char	buf[5];
	int	append = FALSE;

	if (out_file[0] == '\0') 
		return;

	// check if we append or create a new file, because version information
	// must not be written when appending
	if ((outfp = fopen(out_file, "rb")) != NULL)
	{
		append = TRUE;
		fclose(outfp);
	}
		
	// open the file
	if ((outfp = fopen(out_file, "ab")) == NULL){
		cerr << "Can't open output file " << out_file << endl ;
		fclose(infp) ;
		exit(-1) ;
	}

	// write version information
	if (append != TRUE)
	{
		buf[0] = 'U';
		buf[1] = 'L';
		buf[2] = 'G';
		buf[3] = REC_VERSION_2;
		fwrite(buf, 4, 1, outfp);
	}
}

void Player::NewShowinfo(long pos)
{
	int	i;

	current = ntohs(m_tBuf.body.show.time);
	if (current < 0) {
		cerr << "strange time\n";
		current = 0;
		return;
	}
	  
	if (!no_window)
		controler->display_time(current) ;
	// test, if this info has never been read before
	if (current > m_iMaxPos) {
	  if (current < MAX_SHOWINFO) {
		for (i = m_iMaxPos + 1; i <= current; i++)
			// just for the case, that timesteps are missing
			m_plFilePos[i] = pos;
		m_iMaxPos = current;
	  }
	} 		
        m_bReadShowInfo = 1;
}
