/* -*- Mode: C++ -*- */

/* test.C
 * CMUnited99 (soccer client for Robocup99)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1999 Peter Stone
 *
 * CMUnited-99 was created by Peter Stone, Patrick Riley, and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */


#include <stdio.h>
#include <fstream.h>
#include <iomanip.h>
#include "test.h"
#include "Memory.h"
#include "client.h"
#ifndef RELEASE_VERSION  
#include "intercept.h"
#include "goalie.h"
#endif // RELEASE_VERSION
#include "kick.h"
#include "dribble.h"
#include "behave.h"


/*****************************************************************************************/

void test_scan_with_body()
{
  if ( Mem->PlayMode == PM_Before_Kick_Off ){
    move(-1,0);
    my_stamp; printf("%.1f\n",Mem->MyBodyAng());
    return;
  }

  if ( Mem->TimeToTurnForScan() && Mem->CurrentTime < 50 ){
    my_stamp; printf("turning.  Last sight at %d\n",Mem->LastSightTime.t);
    turn(Mem->MyViewAngle() * 2);
  }
  my_stamp; printf("        %.1f\n",Mem->MyBodyAng());
}

#ifndef RELEASE_VERSION  
/*****************************************************************************************/

void test_DT()
{
  my_stamp; printf(" passing: ");
  for (int i=1; i<=Mem->SP_team_size; i++){
    if ( i!=Mem->MyNumber && Mem->TeammatePositionValid(i) )
      printf("%d: %.3f  ",i,Mem->PassConfidence(i));
  }
  printf("shot: %.3f\n",Mem->PassConfidence(Mem->MarkerPosition(Mem->RM_Their_Goal)));
}

/*****************************************************************************************/

Unum test_control()
{
  if ( !Mem->BallPositionValid() ) return Unum_Unknown;
  Vector ball = Mem->BallAbsolutePosition();
  
  Unum teammate = Mem->ClosestTeammateToBall();
  Unum opponent = Mem->ClosestOpponentToBall();

  float teammate_distance = 400, opponent_distance = 400;
  if ( teammate != Unum_Unknown )
    teammate_distance = Mem->TeammateDistanceTo(teammate,ball);
  if ( opponent != Unum_Unknown )
    opponent_distance = Mem->OpponentDistanceTo(opponent,ball);

  if ( teammate_distance < opponent_distance && teammate_distance < Mem->SP_kickable_area )
    return teammate;
  else if ( opponent_distance < teammate_distance && opponent_distance < Mem->SP_kickable_area )
    return -opponent;
  else
    return Unum_Unknown;
}

/*****************************************************************************************/

void test_2v3(Rectangle *rect)
{  
  static int First_time = TRUE;

  /* put in posssessor stuff */

  if ( First_time ){
    Vector point = rect->random();
    point = Mem->PositionToKickoffPosition(point);
    move(point.x,point.y);
    
    First_time=FALSE;
    return;
  }

  Unum teammate;
  if ( !Mem->MyConf() ){ turn(30); return; }

  if ( !Mem->BallPositionValid() ){ face_neck_and_body_to_ball(); return; }

  if ( Mem->BallKickable() ){
    float congestion_epsilon = .01;
    if ( Mem->KickInProgress() ){
      Mem->LogAction2(50,"continuing kick");
      smart_kick_hard_abs(Mem->kick_in_progress_abs_angle,Mem->kick_in_progress_mode,
			  Mem->kick_in_progress_target_vel,Mem->kick_in_progress_rotation);
      return;
    }
    if ( rect->DistanceToEdge(Mem->MyPos()) < 0 ){
      Mem->LogAction2(50,"dribbling in");
      SmartDribbleTo(rect->Center(), 40);
      return;
    }
    if ( (teammate = Mem->LeastCongestedTeammate()) == Unum_Unknown ){
      Mem->LogAction2(50,"scanning field for pass");
      scan_field_with_body();
      return;
    }
    /* redoing the congestion calc here */
    if ( Mem->Congestion(Mem->MyPos()) > Mem->Congestion(Mem->TeammateAbsolutePosition(teammate)) + congestion_epsilon && Mem->PassConfidence(teammate) > 0){
      Mem->LogAction5(50,"going to pass (me %f  teammate %f  conf %f)",
		      Mem->Congestion(Mem->MyPos()),Mem->Congestion(Mem->TeammateAbsolutePosition(teammate)),
		      Mem->PassConfidence(teammate));
      if ( Mem->TeammatePositionValid(teammate) < .9 ){
	Mem->LogAction3(50,"facing teammate %d",teammate);
	face_neck_to_teammate(teammate);
      }
      else{
	Mem->Say( PMsg_passing_decision, teammate );
	my_stamp; printf("%d passing to %d\n",Mem->MyNumber,teammate);
	Mem->LogAction3(50,"passing to %d",teammate);
	pass_ball(teammate);
      }
      return;
    }
    /* Mem->LogAction5(50,"holding ball (me %f  teammate %f  conf %f)",
		    Mem->Congestion(Mem->MyPos()),Mem->Congestion(Mem->TeammateAbsolutePosition(teammate)),
		    Mem->PassConfidence(teammate)); */
    hold_ball();
    return;
  }
  
  if ( (Mem->MyInterceptionAble() && Mem->FastestTeammateToBall()==Mem->MyNumber) ||
       (!Mem->BallMoving() && Mem->ClosestTeammateToBall()==Mem->MyNumber ) ){
    Mem->LogAction2(50,"going to the ball");
    get_ball();
    return;
  }

  if ( Mem->team_receiver == Mem->MyNumber && Mem->CurrentTime - Mem->team_pass_time < 10 ){
    Mem->LogAction2(50,"watching for the pass");
    face_neck_and_body_to_ball();
    return;
  }

  Unum possessor = Mem->BallPossessor();
  char possessor_side = '?';
  if ( possessor > 0 ) possessor_side = Mem->MySide;
  else if ( possessor < 0 ) { possessor_side = Mem->TheirSide; possessor *= -1; }

  Mem->LogAction4(50,"posssesor = %c %d",possessor_side,possessor);

  if ( possessor_side == '?' ){
    Mem->LogAction2(50,"Getting open in rectangle");
    get_open_in_rectangle(rect);
    return;
  }

  if ( possessor_side == Mem->MySide ){
    Mem->LogAction3(50,"Getting open for pass from %d",possessor);
    get_open_for_pass_from_in_rectangle(rect,Mem->TeammateAbsolutePosition(possessor));
    return;
  }

  if ( possessor_side == Mem->TheirSide && Mem->OpponentPositionValid(Mem->MyNumber) ){
    Mem->LogAction3(50,"Marking %d",Mem->MyNumber);
    //mark_opponent(Mem->MyNumber,3);
    get_open_in_rectangle(rect);
    return;
  }

  Mem->LogAction2(50,"Getting open in rectangle -- nothing else to do");
  get_open_in_rectangle(rect);
}

/*****************************************************************************************/

void test_get_open_in_rectangle(Rectangle *rect)
{
  int x_granularity = 5;
  int y_granularity = 5;
  
  float x_mesh = rect->Width()/(x_granularity+1);
  float y_mesh = rect->Height()/(y_granularity+1);
  
  float start_x = rect->LeftX() + x_mesh/2;
  float start_y = rect->TopY()  + y_mesh/2;

  float x = start_x, y = start_y;

  float best_congestion = Mem->Congestion(Vector(x,y));
  Vector best_point = Vector(x,y);
  float tmp;

  for ( int i=0; i<x_granularity; i++ ){
    for ( int j=0; j<y_granularity; j++ ){
      if ( (tmp = Mem->Congestion(Vector(x,y))) < best_congestion ){
	best_congestion = tmp;
	best_point = Vector(x,y);
      }
      y+=y_mesh;
    }
    x+=x_mesh;
    y =start_y;
  }

  int buffer = 3;

  if ( Mem->DistanceTo(best_point) < buffer ){
    if (Mem->TimeToTurnForScan())
      turn(Mem->MyViewAngle() * 2);
  }
  else 
    test_go_to_point(best_point,buffer,50);
}
#endif // RELEASE_VERSION

/*****************************************************************************************/

void test_random_movement_in_rectangle(Rectangle *rect)
{
  static int First_time = TRUE;
  if ( First_time ){
    Vector point = rect->random();
    point = Mem->PositionToKickoffPosition(point);
    move(point.x,point.y);
    First_time=FALSE;
    return;
  }

  if ( !Mem->MyConf() ) {
    turn(30);
    return;
  }

  if ( rect->DistanceToEdge(Mem->MyPos()) < 5 )
    test_go_to_point(rect->Center(),0);
  else if ( Mem->MyStamina() >= Mem->EffortDecThreshold )
    test_random_movement();
}

/*****************************************************************************************/

void test_1v1()
{
  static int First_time = TRUE;
  if ( First_time ){
    if (Mem->PlayMode == PM_Before_Kick_Off)
      move(-10,0);
    First_time=FALSE;
    return;
  }

  if ( Mem->BallPositionValid() ){
    if ( Mem->BallKickable() ){
      hold_ball();
    }
    else {
      get_ball();
    }
  }
  else
    face_neck_and_body_to_ball();
}

#ifndef RELEASE_VERSION  
/*****************************************************************************************/

void test_dodge()
{
  /* run 2 players at each other */
  if ( Mem->CurrentTime < 10 )
    test_volley();
  else 
    test_go_to_point(Vector(40,0),3);
}

/*****************************************************************************************/

void test_play()
{
  if ( !Mem->MyConf() || Mem->ClockStopped )
    test_go_to_position();
  else if ( !Mem->BallPositionValid() )
    turn( Mem->MyViewAngle() );
  else if ( Mem->ClosestTeammateTo(Mem->BallAbsolutePosition()) == Mem->MyNumber ){

    Pnum PreferenceList[11];
    int NumOptions = Mem->MyReceiverList(PreferenceList);
    Unum player;
    AngleDeg target = Mem->MarkerAngleFromBody(Mem->RM_Their_Goal);

    if ( Mem->MyLocationsPositionType() != PT_Forward || Mem->MyLocationsPositionSide() != PS_Center ){
      for ( int i=0; i<NumOptions; i++ ){
	if ( (player = Mem->PositionPlayer(PreferenceList[i])) != Pnum_Unknown &&
	     Mem->TeammatePositionValid(player) ){
	  target = Mem->TeammateAngleFromBody(player);
	  break;
	}
      }
    }

    test_go_to_ball(target);
  }
  else
    test_go_to_position();
}
#endif // RELEASE_VERSION

/*****************************************************************************************/

void test_volley()
{
  static int First_time = TRUE;
  if ( First_time ){
    move(-20,0);
    First_time=FALSE;
    return;
  }

  if ( Mem->BallPositionValid() && Mem->BallX() < -1 )
    test_go_to_ball();
  else
    test_go_to_point(Vector(-20,0),3);
}

#ifndef RELEASE_VERSION  
/*****************************************************************************************/

void test_say()
{
  if ( Mem->CurrentTime > 0 && !(Mem->CurrentTime.t%40) )
    Mem->Say(PMsg_ping_ball);
}
#endif // RELEASE_VERSION

/*****************************************************************************************/

void test_go_to_ball(AngleDeg kick_angle)
{
  if ( Mem->BallKickable() ){
    if ( !Mem->TestVersion ){
      if (!smart_kick_hard(kick_angle, KM_QuickestRelease))
	my_error("test_go_to_ball: kick failed\n");
    }
    else
      kick(100,kick_angle);
  }
  else if ( Mem->MyConf() && Mem->BallPositionValid() ){
    if ( Mem->TestVersion )
      test_go_to_point(Mem->BallAbsolutePosition(),Mem->SP_kickable_area);
    else {
      if (Mem->MyInterceptionAble())
	test_go_to_point(Mem->MyInterceptionPoint(),0);	
      else 
	test_face_ball();
    }
  }
  else 
    turn(90);
}

/*****************************************************************************************/

void test_go_to_ball()
{
  if ( Mem->MyConf() ) 
    test_go_to_ball(Mem->MarkerAngleFromBody(Mem->RM_Their_Goal));
  else
    turn(90);
}

#ifndef RELEASE_VERSION  
/*****************************************************************************************/

void test_go_to_position()
{
  static int First_time = TRUE;
  if ( First_time && Mem->PlayMode == PM_Before_Kick_Off ){
    if (0)
      /* start random, move to position */
      move(range_random(-50,0),range_random(-Mem->SP_pitch_width/2,Mem->SP_pitch_width/2));
    else{
      /* start at position */
      Vector move_to = Mem->PositionToKickoffPosition( Mem->MyCurrentHome() );
      move(move_to.x,move_to.y);
    }
    First_time=FALSE;
    return;
  }

  /* if ( Mem->CurrentTime.t > 100 ) Mem->SetFormation(FT_72,Mem->CurrentTime); */

  if ( Mem->BallPositionValid() ) Mem->UpdateHome();

  Vector home = Mem->MyCurrentHome();
  float buffer = Mem->MyPositionBuffer();

  test_go_to_point(home,buffer);
}
#endif // RELEASE_VERSION

/*****************************************************************************************/

void test_go_to_point(Vector p, float buffer, float dash_power )
{
  if ( !Mem->MyConf() ) {
    turn(Mem->MyViewAngle());
    return;
  }

  if ( Mem->DistanceTo(p) < buffer ){
    //turn(30);
    test_face_ball();
    return;
  }

  float target_ang  = Mem->AngleToFromBody(p);
  float target_dist = Mem->DistanceTo(p);

  if ( 1 && !Mem->ClockStopped ){ /* dodge players */
    PlayerObject *player;
    float    dodge_dist = Min(Mem->CP_dodge_distance_buffer,target_dist);
    AngleDeg dodge_ang  = Mem->CP_dodge_angle_buffer;
    if ( (player = Mem->GetPlayerWithin( dodge_dist, dodge_ang, 0, target_ang - dodge_ang))
	 != NULL ){
      if ( Mem->NumPlayersWithin( dodge_dist, 2*dodge_ang) ){
	/* Target close, so no players will be within in the next iteration ==> dash */
	Vector new_target = Mem->BodyPolar2Gpos(Mem->SP_player_size,player->get_ang_from_body() + 90);
	test_go_to_point(new_target,0);
      }
      else{
	dash(Mem->CP_dodge_power);
      }
      return;
    }
    if ( (player = Mem->GetPlayerWithin( dodge_dist, dodge_ang, 0, target_ang + dodge_ang))
	 != NULL ){
      if ( Mem->NumPlayersWithin( dodge_dist, 2*dodge_ang) ){
	/* Target close, so no players will be within in the next iteration ==> dash */
	Vector new_target = Mem->BodyPolar2Gpos(Mem->SP_player_size,player->get_ang_from_body() - 90);
	test_go_to_point(new_target,0);
      }
      else{
	dash(Mem->CP_dodge_power);
      }
      return;
    }
  }

  if ( fabs(target_ang) > Mem->CP_max_go_to_point_angle_err ){
    turn(target_ang);
    return;
  }

  if ( Mem->MyStamina() >= Mem->EffortDecThreshold )
    dash( Min( dash_power, Mem->MyStamina()-Mem->EffortDecThreshold ) );
  else
    {my_stamp; printf("recovering\n");} //turn(180);
}

/*****************************************************************************************/

void test_face_ball()
{
  if ( Mem->BallPositionValid() ) 
    turn( Mem->BallAngleFromNeck() );
  else 
    turn(Mem->MyViewAngle());
}

#ifndef RELEASE_VERSION  
/*****************************************************************************************/

void test_move()
{
  static int First_time = TRUE;
  if ( First_time ){
    move(-50,-30 + 10*(Mem->MyNumber-1));
    First_time=FALSE;
    return;
  }

  float recover_thr_excess = Mem->MyStamina() - Mem->RecoveryDecThreshold;
  float effort_thr_excess  = Mem->MyStamina() - Mem->EffortDecThreshold;

  if ( recover_thr_excess < 0 ) recover_thr_excess = 0;
  if ( effort_thr_excess < 0 )  effort_thr_excess = 0;

  static int recovering = FALSE;
  
  switch( Mem->MyNumber ){
  case 1: dash(20); break;
    /* these are all the same as case 1 */
    /*case 2: if ( !(Mem->CurrentTime.t%2) ) dash(40); break;  
      case 3: if ( !(Mem->CurrentTime.t%3) ) dash(60); break;
      case 4: if ( !(Mem->CurrentTime.t%4) ) dash(80); break;
      case 5: if ( !(Mem->CurrentTime.t%5) ) dash(100); break; */
  case 2: 
    if ( recover_thr_excess < 100 ) 
      dash(recover_thr_excess);
    else
      dash(100);
    break;
  case 3: 
    if ( effort_thr_excess < 100 ) 
      dash(effort_thr_excess);
    else
      dash(100);
    break;
  case 4: dash(100); break;
  case 5: 
    /* Looks like this would win over the top ones in the long run -- might want to recover for 
       not so long */
    if ( recovering==TRUE ){
      if ( Mem->MyStamina() == Mem->SP_stamina_max ) recovering = FALSE;
    }
    else{
      if ( effort_thr_excess < 100 ) recovering = TRUE;
      else dash(100);
    }
    break;
  case 6: 
    /* Looks like this would win over the top ones in the long run -- might want to recover for 
       not so long */
    if ( recovering==TRUE ){
      if ( Mem->MyStamina() == Mem->SP_stamina_max ) recovering = FALSE;
    }
    else{
      if ( recover_thr_excess < 100 ) recovering = TRUE;
      else dash(100);
    }
    break;
  case 7: 
    if ( recovering==TRUE ){
      if ( Mem->MyStamina() > Mem->SP_stamina_max * Mem->SP_effort_dec_thr ) recovering = FALSE;
    }
    else{
      if ( recover_thr_excess < 100 ) recovering = TRUE;
      else dash(100);
    }
    break;
  }
}
#endif // RELEASE_VERSION

/*****************************************************************************************/

void test_random_movement()
{
  static int First_time = FALSE;
  if ( First_time ){
    move(range_random(-50,0),range_random(-Mem->SP_pitch_width/2,Mem->SP_pitch_width/2));
    First_time=FALSE;
    return;
  }

  /* if      ( !int_random(100) ) change_view(VW_Wide);
     else if ( !int_random(100) ) change_view(VW_Normal);
     else if ( !int_random(100) ) change_view(VW_Narrow); */

  if ( Mem->ClockStopped ) return;

  if ( Mem->BallKickable() ){
    kick(range_random(0,100),Mem->BallAngleFromBody());
    return;
  }

  if ( Mem->MyConf() && Mem->BallPositionValid() && Mem->BallDistance() < 9 ){
    if ( fabs(Mem->BallAngleFromBody()) > 1 ) 
      turn( Mem->BallAngleFromBody() );
    else
      dash(100);
    return;
  }

  if ( int_random(4) )
    dash(range_random(20,30));
  else 
    turn(range_random(-90,90));

  return;
}

/*****************************************************************************************/

void test_straight_to_ball()
{
  static int First_time = TRUE;
  if ( First_time ){
    if (Mem->MySide == 'l')
      turn(90);
    else
      turn(-90);
    First_time=FALSE;
    return;
  }

  if ( Mem->CurrentTime.t && Mem->CurrentTime.t < 10 && Mem->GetBall()->pos_valid() ){
    turn(Mem->GetBall()->get_ang_from_body());
  }
  else 
    dash(Mem->CurrentTime.t%100); 
  return;
}

/*****************************************************************************************/

void test_run_straight()
{
  static Bool GO = TRUE;
  
  if ( !(Mem->CurrentTime.t%200) ) GO = TRUE;

  if ( Mem->MyStamina() > 600 && GO ) 
    dash(60);
  else if ( Mem->CurrentTime.t ){
    GO = FALSE;
    printf("stamina = %f, effort = %f\n",Mem->MyStamina(),Mem->MyEffort());
  }
}

/*****************************************************************************************/

void test_turn_and_dash_slow()
{
  static int First_time = TRUE;
  if ( First_time ){
    if (Mem->MySide == 'l')
      turn(90);
    else
      turn(-90);
    First_time=FALSE;
    return;
  }

  if ( Mem->CurrentTime.t%5 || !Mem->CurrentTime.t ) return;

  if ( !(Mem->CurrentTime.t%50) )
    turn(40);
  else
    dash(100);

  return;
}

#ifndef RELEASE_VERSION  
/*****************************************************************************************/

void test_print_mode()
{
  my_stamp; printf("mode %d\n",Mem->MyActionMode);
}
#endif // RELEASE_VERSION

/*****************************************************************************************/

void test_print_ball()
{
  my_stamp;

  if ( Mem->BallPositionValid() ) {
    printf(" Ball at %.1f %.1f (%.2f)",Mem->BallX(), Mem->BallY(), Mem->BallPositionValid()); 
    if ( Mem->BallVelocityValid() )
          printf("     Speed %.2f dir %.1f (%.2f)",
		 Mem->BallSpeed(), Mem->BallAbsoluteHeading(), Mem->BallVelocityValid()); 
    else
      printf("    velocity not valid");
  }
  else 
    printf(" Ball not valid");

  printf("\n");
}

/*****************************************************************************************/

void test_print_positions()
{
  Bool me = TRUE;
  Bool ball = FALSE;
  Bool players = TRUE;

  if (me){
    my_stamp; printf("at %d I'm at (%.1f %.1f) facing %.1f with velocity (%f %.1f)\n",
		     Mem->CurrentTime.t,Mem->MyX(),Mem->MyY(),Mem->MyBodyAng(),Mem->MySpeed(),Mem->MyDir());
  }

  if (ball){
    if (Mem->GetBall()->pos_valid())
      printf("Ball: (%f %f)       ",Mem->GetBall()->get_x(),Mem->GetBall()->get_y());

    if (Mem->GetBall()->vel_valid())
      printf("velocity: (%f %f)\n",Mem->GetBall()->get_speed(),Mem->GetBall()->get_abs_heading());
    else if (Mem->GetBall()->pos_valid())
      printf("\n"); 
  }

  if (players){
    for (int i=0; i<Mem->NumTeammates(); i++){
      my_stamp;
      printf("T %d at %.1f %.1f facing %.1f\n",Mem->Teammates()[i]->unum,Mem->Teammates()[i]->get_x(),Mem->Teammates()[i]->get_y(),Mem->Teammates()[i]->get_abs_body_ang());
    }
    for (int i=0; i<Mem->NumOpponents(); i++){
      my_stamp;
      printf("O %d at %.1f %.1f facing %.1f\n",Mem->Opponents()[i]->unum,Mem->Opponents()[i]->get_x(),Mem->Opponents()[i]->get_y(),Mem->Opponents()[i]->get_abs_body_ang());
    }
    for (int i=0; i<Mem->NumTeamlessPlayers(); i++){
      my_stamp;
      printf("? %d at %.1f %.1f facing %.1f\n",Mem->TeamlessPlayers()[i]->unum,Mem->TeamlessPlayers()[i]->get_x(),Mem->TeamlessPlayers()[i]->get_y(),Mem->TeamlessPlayers()[i]->get_abs_body_ang());
    }
  }
}

/*****************************************************************************************/

void test_time()
{
  Time tmp = Mem->CurrentTime - 2;
  printf("time: %d %d         %d %d      (%d %d)",
	 Mem->CurrentTime.t,Mem->CurrentTime.s,tmp.t,tmp.s,Mem->LastStartClockTime.t,Mem->LastStartClockTime.s);
  if ( Mem->CurrentTime.t > 5 ){
    int tmp2 = Mem->CurrentTime - Time(5,0);
    printf(" --------  %d)",tmp2);
  }
  printf("\n");
}

#ifndef RELEASE_VERSION  
/*****************************************************************************************/

/** BRIAN'S STUFF **/

/******************************************************************************
***********/
void test_good_pass(){
  
  if(Mem->BallPositionValid()){

    Unum fastestTeammate = Mem->FastestTeammateToBall();
    Unum fastestOpponent = Mem->FastestOpponentToBall();
    
    int MAX_DIFF_BTW_CYCLES = 5;
    
    if(fastestTeammate != Unum_Unknown)
      if(fastestOpponent != Unum_Unknown)
    if((Mem->TeammatePositionValid(fastestTeammate))&&
(Mem->OpponentPositionValid(fastestOpponent)))
      if(Mem->OpponentInterceptionAble(fastestOpponent))
        if (Mem->TeammateInterceptionAble(fastestTeammate)){
          int teammateMinCycles =
Mem->TeammateInterceptionNumberCycles(fastestTeammate,
Mem->SP_max_power);
          int opponentMinCycles =
Mem->OpponentInterceptionNumberCycles(fastestOpponent,
Mem->SP_max_power);
          
          if(teammateMinCycles + MAX_DIFF_BTW_CYCLES <= opponentMinCycles){
        cout<<Mem->CurrentTime.t<<" TEAMMATE: "<<fastestTeammate<<" has
control, team_cyc = " <<
          teammateMinCycles<<"  opp_cyc = "<<opponentMinCycles<<"\n";
        return;
          }
          else 
        if(opponentMinCycles + MAX_DIFF_BTW_CYCLES <= teammateMinCycles){
          cout<<Mem->CurrentTime.t<<" OPPONENT: "<<fastestOpponent<<"
has control, team_cyc = "<<
            teammateMinCycles<<"  opp_cyc = "<<opponentMinCycles<<"\n";
          return;
        }
        else {
          cout<<Mem->CurrentTime.t<<" NOONE: it is too close to tell,
team_cyc = "<<
            teammateMinCycles<<"  opp_cyc = "<<opponentMinCycles<<"\n";
          return;
        }
        }
        else{
          cout<<Mem->CurrentTime.t<<" OPPONENT: "<<fastestOpponent<<
        " has control away from nearest teammatess" <<"\n";
          return;
        }
      else{
        if(Mem->TeammateInterceptionAble(fastestTeammate)){
          cout<<Mem->CurrentTime.t<<" TEAMMATE: "<<fastestTeammate<<
        " has control away from nearest opponents" <<"\n";
          return;
        }
        else{
          cout<<"NOONE: Neither fastest player is InterceptionAble"<<"\n";
          return;
        }
      }
    else{
      cout<<"UNSURE: Invalid player position"<<"\n";
      return;
    }
      else{
    cout<<Mem->CurrentTime.t<<" TEAMMATE: "<<fastestTeammate<<" has
control away from nearest opponents" <<"\n";
    return;
      }
    else{ 
      if(fastestOpponent != Unum_Unknown){
    cout<<Mem->CurrentTime.t<<" OPPONENT: "<<fastestOpponent<<" has
control away from nearest teammates" <<"\n";
    return;
      }
      else{
    cout<<Mem->CurrentTime.t<<" NOONE: Pass went away from everyone"<<"\n";
    return;
      }
    }
  }
  else {
    cout<<" UNCLEAR: can't locate ball"<<"\n";
    return;
  }
} 
#endif // RELEASE_VERSION

/***************************************************************************/
/* Pat added for parameter tuning */
/**************************************************************************/
void test_turnball()
{
  if (!Mem->MyConf() || !Mem->BallPositionValid())
    scan_field_with_body();
  else if (Mem->BallKickable()) {
    TurnballTo(Mem->BallAngleFromBody() + 180, TURN_CCW);
  } else
    ; /* if we can't kick it, do nothing */
  
}

void test_turnball2()
{
  static TurnDir turn_dir = TURN_CW;
  if (!Mem->MyConf() || !Mem->BallPositionValid()) {
    /* switch the turn direction when we lose the ball,
       this effectively randomizes it */
    turn_dir = (turn_dir == TURN_CW) ? TURN_CCW : TURN_CW;
    scan_field_with_body();
  }  
  else if (Mem->BallKickable()) {
    TurnballTo(-Mem->MyBodyAng(), turn_dir);
  } else
    ; /* if we can't kick it, do nothing */
  
}

#ifdef DEBUG_OUTPUT
#define DEBUG_TEST_KICK(x) 
#else
#define DEBUG_TEST_KICK(x)
#endif
void test_hard_kick(KickMode km)
{
  static float bally;
  static Bool LastActionKick = FALSE;

  if (!Mem->MyConf()) {
    DEBUG_TEST_KICK(cout << "Time: " << Mem->CurrentTime.t << "\tScanning field" << endl);
    scan_field_with_body();
    LastActionKick = FALSE;
  } else if (Mem->PlayMode == PM_Before_Kick_Off) {
    move(-10,0);
  } else {
    DEBUG_TEST_KICK(cout << "Time: " << Mem->CurrentTime.t << "\tMyAng: " << Mem->MyAng() << endl);
    if (fabs(GetNormalizeAngleDeg(Mem->MyBodyAng() -
				  signf(bally)*
				  Mem->CP_hardest_kick_player_ang)) > 9 &&
	(!Mem->BallKickable() || Mem->BallVelocityValid()) &&
	!LastActionKick) {
      DEBUG_TEST_KICK(cout << " Turning: target: "
	   << signf(bally)*Mem->CP_hardest_kick_player_ang
	   << "\tturning: "
	   << signf(bally)*Mem->CP_hardest_kick_player_ang - Mem->MyAng() << endl);
      turn(signf(bally)*Mem->CP_hardest_kick_player_ang - Mem->MyBodyAng());
      LastActionKick = FALSE;
    } else if (Mem->BallKickable()) {
      DEBUG_TEST_KICK(cout << "calling smart_kick_hard" << endl);
      smart_kick_hard_abs(0.0, km);
      if (Mem->Action->type == CMD_kick)
	LastActionKick = TRUE;
      else
	LastActionKick = FALSE;
      bally = Mem->BallY();
    } else
      LastActionKick = FALSE; /* if we can't kick or turn, do nothing */
  }
  
}

void test_intercept()
{
  if (!Mem->MyConf()) {
    Mem->LogAction2(10, "Lost myself, scanning field");
    scan_field_with_body();
  } else if (!Mem->BallPositionValid()) {
    Mem->LogAction2(10, "Lost the ball, scanning field");
    scan_field_with_body();
  } else if (Mem->BallKickable()) {
    Mem->LogAction2(10, "Have ball, holding onto it");
    hold_ball();
  } else {
    Mem->LogAction2(10, "Chasing the ball");
    get_ball();
  }
  
}

void test_go_to_static_ball()
{
 if (!Mem->MyConf() || !Mem->BallPositionValid())
    scan_field_with_body();
  else if (Mem->BallKickable() && Mem->LastActionType() == CMD_kick) {
    smart_kick_hard_abs((-Mem->BallAbsolutePosition()).dir(), KM_HardestKick);
  } else {
    if (go_to_static_ball((-Mem->BallAbsolutePosition()).dir()))
      smart_kick_hard_abs((-Mem->BallAbsolutePosition()).dir(), KM_HardestKick);
    //cout << Mem->CurrentTime.t << " I'm ready to kick" << endl;
  }
}

void test_pred_cycles_to_point()
{
  static Vector pt(0,0);
  static Rectangle r(Vector(0,0), Vector(20,20));
  
  if (!Mem->MyConf())
    scan_field_with_body();
  else {
    cout << "Time: " << setw(4) << Mem->CurrentTime.t << '\t'
	 << "predict " << setw(3) << Mem->PredictedCyclesToPoint(pt) << " cycles "
	 << "to point " << pt << endl;
    if (go_to_point(pt, Mem->CP_at_point_buffer) == AQ_ActionNotQueued) {
      cout << "I got there!" << endl;
      pt = r.random();
    }
  }
}

#ifndef RELEASE_VERSION  
void test_shot(KickMode km) 
{
  static int time_ball_kickable = 0;

  if (Mem->CP_goalie) {
    goalie_behave();
  } else {
    change_view(VW_Narrow);
    
    if (!Mem->MyConf()) {
      cout << "Time: " << Mem->CurrentTime.t << "\tScanning field" << endl;
      scan_field_with_body();
    } else if (Mem->PlayMode == PM_Before_Kick_Off) {
      move(-10,0);
    } else if (Mem->BallKickable()) {
      if (++time_ball_kickable > 10) {
	float ang = (Mem->ShotTarget() - Mem->MyPos()).dir();
	cout << "calling smart_kick_hard with ang: " << ang << endl;
	smart_kick_hard_abs(ang, km);
      }
    } else {
      time_ball_kickable = 0;
    }
  }
}
#endif // RELEASE_VERSION



#ifndef RELEASE_VERSION  
void test_breakaway()
{
#ifdef OLD_CODE
  if (Mem->MySide == 'l') {
    if (Mem->CP_goalie) {
      goalie_behave();
    } else {
      return; /* we'll fall through to normal behave */
    }
  } else {

    if (Mem->idle_cycles > 0) {
      Mem->LogAction2(30, "Idling");
      Mem->idle_cycles--;
      face_neck_and_body_to_point(Mem->MarkerPosition(Mem->RM_Their_Goal));
      return;
    }
    
    
    change_view(VW_Narrow);
    
    if (!Mem->MyConf() || !Mem->BallPositionValid()) {
      //cout << "Time: " << Mem->CurrentTime.t << "\tScanning field" << endl;
      scan_field_with_body();
    } else if (Mem->PlayMode == PM_Before_Kick_Off) {
      move(-10,0);
    } else {
      
      /* regular breakaway action */
      if ( Mem->BallKickable() && Mem->KickInProgress() ){
	/* Already kicking */
	Mem->LogAction2(30, "test_breakaway: continuing kick");
	smart_kick_hard_abs(Mem->kick_in_progress_abs_angle,Mem->kick_in_progress_mode,
			    Mem->kick_in_progress_target_vel,Mem->kick_in_progress_rotation);
	return;
      }

      Vector shot_target = Mem->ShotTarget();
      if (Mem->BallKickable()) {
	Unum oppGoalie     = Mem->ClosestOpponentTo(Mem->MarkerPosition(Mem->RM_Their_Goal)); 
	float dist_to_goal = Mem->DistanceTo(Mem->MarkerPosition(Mem->RM_Their_Goal)); 
	KickMode shot_mode = Mem->ShotMode(); 
	int cycles_to_steal;

	if (oppGoalie != Unum_Unknown) {
	  cycles_to_steal = Mem->EstimatedCyclesToSteal(oppGoalie);
	  Mem->LogAction3(40, "Cycles for goalie to steal: %d", cycles_to_steal);
	}

	/* look to see if we should shoot now */
	if (oppGoalie == Unum_Unknown) {
	  Mem->LogAction5(30, "test_breakaway: shooting because goalie not there: mode: %d at (%.1f, %.1f)",
			  shot_mode, shot_target.x, shot_target.y);
	  kick_ball(shot_target, shot_mode);
	} else if (!Mem->CouldGoalieBlockShot(shot_target, Mem->CP_shot_speed, oppGoalie)) {
	  Mem->LogAction5(30, "test_breakaway: shooting because goalie can't stop it: mode: %d at (%.1f, %.1f)",
			  shot_mode, shot_target.x, shot_target.y);
	  kick_ball(shot_target, shot_mode); 	  
	} else if (cycles_to_steal <= Mem->CP_breakaway_min_goalie_steal_time) {
	  Mem->LogAction5(30, "test_breakaway: shooting because goalie close: mode: %d at (%.1f, %.1f)",
			  shot_mode, shot_target.x, shot_target.y);
	  kick_ball(shot_target, shot_mode);
	} else {
	  /* still not ready to shoot */
	  /* try dribbling or a kick and run */
	  if (Mem->CurrentTime-Mem->CP_breakaway_targ_valid_time >= Mem->my_breakaway_time ||
	      Mem->DistanceTo(Mem->my_breakaway_targ) < Mem->CP_at_point_buffer) {
	    /* stored value too old to reply on  OR
	       we got to the last target */
	    if (Mem->MyX() < Mem->CP_breakaway_approach_x &&
		Mem->DistanceTo(Vector(Mem->CP_breakaway_approach_x,
				       signf(Mem->MyY())*Mem->CP_breakaway_approach_y)) >
		Mem->CP_at_point_buffer) {
	      /* kick-run to approach point */
	      Vector targ;
	      AngleDeg ang;
	      Vector approach_pt;
	      approach_pt.x = Mem->CP_breakaway_approach_x;
	      approach_pt.y = signf(Mem->MyY()) * Mem->CP_breakaway_approach_y;
	      Mem->LogAction2(40, "test breakaway: recomputing kick-run targ, approach");
	      for (ang = (approach_pt - Mem->MyPos()).dir();
		   fabs(ang) < 90;
		   ang += signf(Mem->MyY()) * -5.0 ) {
		if (breakaway_evaluate_kick_run_angle(ang, oppGoalie, &targ))
		  break;
	      }
	      if (fabs(ang) >= 90) {
		my_error("No kick-run target (should probably go to dribble");
	      } else {
		Mem->my_breakaway_dribble = FALSE; //do kick-run
		Mem->my_breakaway_targ = targ;
	      }
	      
	    } else if (fabs(Mem->MyY()) - Mem->CP_at_point_buffer >
		       Mem->CP_breakaway_approach_y) {
	      /* we're on the side, dribble straight across */
	      Mem->my_breakaway_dribble = TRUE; //do dribble
	      Mem->my_breakaway_targ.x = Mem->MyX();
	      Mem->my_breakaway_targ.y = signf(Mem->MyY()) *
		(Mem->CP_breakaway_approach_y - 1);
	      Mem->LogAction2(40, "test breakaway: recomputing dribble targ, across");
	    } else {
	      /* we're inside the approach area,
		 dribble to opposite corner */
	      Mem->my_breakaway_dribble = TRUE; //do dribble
	      Mem->my_breakaway_targ.x = Mem->SP_pitch_length / 2;
	      Mem->my_breakaway_targ.y =
		-1.0 * signf(Mem->MyY()) * Mem->SP_goal_area_width / 2;
	      Mem->LogAction2(40, "test breakaway: recomputing dribble targ, opp corner");
	    }
	  }

	  /* Now that we have a target picked, go to the dribble or kick-run to it */
	  if (Mem->my_breakaway_dribble) {
	    if (!Mem->BallVelocityValid()) {
	      stop_ball();
	      face_only_neck_to_ball();
	    } else {
	      face_only_neck_to_point((oppGoalie != Unum_Unknown) ?
				      Mem->OpponentAbsolutePosition(oppGoalie) :
				      Mem->MarkerPosition(Mem->RM_Their_Goal));
	      DribbleRes res = SmartDribbleTo(Mem->my_breakaway_targ, Mem->CP_dribble_dash_pow);
	      if (res != DR_Going)
		my_error("test_breakaway: dribble did something wrong: %d", res);
	      Mem->LogAction4(30, "test breakaway: dribbling towards (%.1f, %.1f)",
			      Mem->my_breakaway_targ.x, Mem->my_breakaway_targ.y);
	    }
	  } else {
	    /* Do a kick and run */
	    /* now decide whether to kick or run */
	    int my_cyc_to_point = Mem->PredictedCyclesToPoint(Mem->my_breakaway_targ);
	    if (Mem->WillDashBeCollision(Mem->SP_max_power) ||
		Mem->BallPredictedPosition(my_cyc_to_point).dist(Mem->my_breakaway_targ) >
		Mem->SP_kickable_area - Mem->CP_kickable_buffer) {
	      /* need to kick the ball */
	      float target_vel =
		SolveForFirstTermGeomSeries(Mem->SP_ball_decay,
					    Mem->CP_breakaway_kick_run_cycles,
					    Mem->BallAbsolutePosition().dist(Mem->my_breakaway_targ));
	      Mem->LogAction4(30, "Kicking for kick-run targ (%.2f, %.2f)",
			      Mem->my_breakaway_targ.x, Mem->my_breakaway_targ.y);
	      smart_kick_hard_abs((Mem->my_breakaway_targ-Mem->MyPos()).dir(),
				  KM_QuickestRelease, target_vel, TURN_CLOSEST);
	    } else {
	      /* just go to the right point */
	      Mem->LogAction2(30, "Going to breakaway kick-run targ");
	      go_to_point(Mem->my_breakaway_targ, 0/*buffer*/, Mem->SP_max_power, DT_none);
	    }
	    face_only_neck_to_point((oppGoalie != Unum_Unknown) ?
				    Mem->OpponentAbsolutePosition(oppGoalie) :
				    Mem->MarkerPosition(Mem->RM_Their_Goal));
	  }
	  
#ifdef ONLY_KIKC_RUN
	  /* try kick and run */
	  if (Mem->CurrentTime-2 >= Mem->my_breakaway_time ||
	      Mem->DistanceTo(Mem->my_breakaway_targ) < Mem->CP_at_point_buffer) {
	    /* stored value too old to reply on  OR
	       we got to the last target */
	    Mem->LogAction2(40, "test breakaway: recomputing kick-run targ");
	    Vector approach_pt;
	    approach_pt.x = Mem->CP_breakaway_approach_x;
	    approach_pt.y = signf(Mem->MyY()) * Mem->CP_breakaway_approach_y;

	    Vector targ;
	    AngleDeg ang;
	    if (Mem->MyX() < approach_pt.x) {
	      /* favor heading towards the approach point */
	      Mem->LogAction2(40, "test breakaway: recomputing kick-run targ, approach");
	      for (ang = (approach_pt - Mem->MyPos()).dir();
		   fabs(ang) < 90;
		   ang += signf(Mem->MyY()) * -5.0 ) {
		if (breakaway_evaluate_kick_run_angle(ang, oppGoalie, &targ))
		  break;
	      }
	      if (fabs(ang) >= 90)
		my_error("No kick-run target (should probably go to dribble");
	      else
		Mem->my_breakaway_targ = targ;
	    } else {
	      /* favor heading towards middle of goal inwards */
	      Mem->LogAction2(40, "test breakaway: recomputing kick-run targ, cross");
	      for (ang = (Mem->MarkerPosition(Mem->RM_Their_Goal) - Mem->MyPos()).dir();
		   fabs(ang) < 90;
		   ang += signf(Mem->MyY()) * -5.0 ) {
		if (breakaway_evaluate_kick_run_angle(ang, oppGoalie, &targ))
		  break;
	      }
	      if (fabs(ang) >= 90)
		my_error("No kick-run target (should probably go to dribble");
	      else
		Mem->my_breakaway_targ = targ;
	    }

	    Mem->LogAction5(40, "test breakaway: picked a new targ (%.2f, %.2f) which is at ang %.2f",
			    Mem->my_breakaway_targ.x, Mem->my_breakaway_targ.y, ang);

	  } // pick a new kick-run target

	  /* now decide whether to kick or run */
	  int my_cyc_to_point = Mem->PredictedCyclesToPoint(Mem->my_breakaway_targ);
	  if (Mem->WillDashBeCollision(Mem->SP_max_power) ||
	      Mem->BallPredictedPosition(my_cyc_to_point).dist(Mem->my_breakaway_targ) >
 	      Mem->SP_kickable_area - Mem->CP_kickable_buffer) {
	    /* need to kick the ball */
	    float target_vel =
	      SolveForFirstTermGeomSeries(Mem->SP_ball_decay,
					  Mem->CP_breakaway_kick_run_cycles,
					  Mem->BallAbsolutePosition().dist(Mem->my_breakaway_targ));
	    Mem->LogAction4(30, "Kicking for kick-run targ (%.2f, %.2f)",
			    Mem->my_breakaway_targ.x, Mem->my_breakaway_targ.y);
	    smart_kick_hard_abs((Mem->my_breakaway_targ-Mem->MyPos()).dir(),
				KM_QuickestRelease, target_vel, TURN_CLOSEST);
	  } else {
	    /* just go to the right point */
	    Mem->LogAction2(30, "Going to breakaway kick-run targ");
	    go_to_point(Mem->my_breakaway_targ, 0/*buffer*/, Mem->SP_max_power, DT_none);
	  }
	  face_only_neck_to_point((oppGoalie != Unum_Unknown) ?
				  Mem->OpponentAbsolutePosition(oppGoalie) :
				  Mem->MarkerPosition(Mem->RM_Their_Goal));
	
#endif	
	  Mem->my_breakaway_time = Mem->CurrentTime;
	}

      } else {
	/* ball not kickable */
	/* try dribbling, then do a get_ball */
	face_only_neck_to_point(Mem->MarkerPosition(Mem->RM_Their_Goal));
	DribbleRes res = SmartDribbleTo(shot_target, Mem->CP_dribble_dash_pow);
	if (res == DR_LostBall) {
	  Mem->LogAction2(30, "test_breakaway: getting the ball");
	  if (Mem->MyInterceptionAble()) {
	    /* will write over the previous turn_neck, but that's okay */
	    get_ball();
	  } else
	    Mem->LogAction2(40, "test_breakaway: can't get to ball");
	} else {
	  Mem->LogAction2(30, "test_breakaway: dribbling");
	}
      } /* ball not kickable */
    } /* regular breakaway action */ 
  } /* striker */ 
#endif	  

}

#ifdef MOVED_TO_BEHAVE
Bool test_evaluate_kick_run_angle(AngleDeg ang, Unum oppGoalie, Vector* pTarg) 
{
  if (oppGoalie == Unum_Unknown)
    my_error("evaluate_kick_run_angle: goalie must be known");
  if (fabs(GetNormalizeAngleDeg(ang - Mem->MyBodyAng())) <
      Mem->CP_max_go_to_point_angle_err)
    // facing the right dir
    // the 1 at the end is an initial idle cycle
    *pTarg = Mem->MyPredictedPosition(Mem->CP_breakaway_Mem->CP_breakaway_kick_run_cycles, Mem->SP_max_power, 1);
  else
    // the 1 at the end is an initial idle cycle
    *pTarg = Mem->MyPredictedPositionWithTurn(ang - Mem->MyBodyAng(),
					      Mem->CP_breakaway_Mem->CP_breakaway_kick_run_cycles,
					      Mem->SP_max_power,
					      TRUE, 1);
  /* displace it by a dribble dist 
  *pTarg += (*pTarg - Mem->MarkerPosition(Mem->RM_Their_Goal)).
    SetLength(Mem->CP_dribble_ball_dist);
  */
  int goalie_cyc = Mem->OpponentPredictedCyclesToPoint(oppGoalie, *pTarg);
  Mem->LogAction6(205, "Trying ang: %.2f, targ is (%.2f, %.2f), goalie: %f",
		  ang, pTarg->x, pTarg->y, (float)goalie_cyc);
  return (goalie_cyc > Mem->CP_breakaway_Mem->CP_breakaway_kick_run_cycles + Mem->CP_cycles_to_kick)
    ? TRUE : FALSE;
}
#endif

#if 0
void test_breakaway_kick_ball(Vector point, KickMode mode, TurnDir rotation)
{
  /* target_angle relative to body */
  if (rotation == TURN_NONE)
    rotation = KickRotationDirection(Mem->AngleToFromBody(point));

  /* look to see if a dash will help */
  if (Mem->WillDashHelpKick(point, Mem->SP_max_power)) {
    float target_angle =
      (point - Mem->MyPredictedPosition(1, Mem->SP_max_power)).dir() - Mem->MyBodyAng();
    Mem->LogAction3(40, "tb: dashing will help kick to angle: %1.f", target_angle);
    dash(Mem->SP_max_power);
    Mem->StartShot(target_angle, mode, rotation);
  } else {
    float target_angle = Mem->AngleToFromBody(point);
    Mem->LogAction3(40, "tb: starting kick to angle %.1f",target_angle);
    Mem->StartShot(target_angle, mode, rotation);
    smart_kick_hard(target_angle, mode, rotation);
  }
  
  return;
}
#endif
#endif // RELEASE_VERSION
