/* -*- Mode: C++ -*- */

/* dribble.C
 * CMUnited99 (soccer client for Robocup99)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1999 Peter Stone
 *
 * CMUnited-99 was created by Peter Stone, Patrick Riley, and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */


#include <math.h>
#include "Memory.h"
#include "client.h"
#include "kick.h"
#include "dribble.h"
#include "behave.h"

#ifdef DEBUG_OUTPUT
#define DebugDrib(x) 
#define DebugDrib2(x) 
#else
#define DebugDrib(x)
#define DebugDrib2(x)
#endif


void PatsTest_dribble() 
{
  static Vector pos(10,10);
  static AngleDeg drib_ang = 60;
  

  if (!Mem->MyConf()) {
    Mem->LogAction2(10, "Don't know where I am, scanning field");
    scan_field_with_body();
    return;
  }
  
  if (!Mem->BallPositionValid()) {
    Mem->LogAction2(10, "Lost ball, scanning field");
    return;
  }
  
  if (Mem->CurrentTime.t % 40 == 1)
    drib_ang = -drib_ang;
  
  if (!Mem->BallVelocityValid()) {
    if (Mem->BallKickable())
      stop_ball();
    else
      get_ball();
  } else {
    /*printf("\nTime: %d\n", Mem->CurrentTime.t);*/
    DribbleRes res = DribbleTo(pos, Mem->CP_dribble_dash_pow,
			       1, drib_ang, DM_Lazy);
    if (res == DR_GotThere) {    
      Mem->LogAction2(10, "got to dribble target, reversing");
      stop_ball();
      pos = -pos;
    } else if (res == DR_LostBall) {
      get_ball();
    }
  }

  change_view(VW_Narrow);
}

#ifndef RELEASE_VERSION  

void PatsTest_dribble2() 
{
  static Vector pos(52.5,-20);
  //static Vector pos(45,-0);
  AngleDeg drib_ang;
  
  if (!strcmp(Mem->MyTeamName, "CMUnited")) {
    if (Mem->ViewWidth != VW_Wide)
      change_view(VW_Wide);
    
    if (!Mem->MyConf())
      return;
    
    if (!Mem->BallPositionValid())
      return;
    
    if (Mem->CurrentTime.t < 20)
      return;

    //drib_ang = -60 + 2*fabs((Mem->CurrentTime.t % 30)*4 - 60);
    drib_ang = 90;
    
    printf("\nTime: %d\n", Mem->CurrentTime.t);
    //printf("drib_ang: %f\n", drib_ang);
    //float dash_power = 75 + 5 * (Mem->CurrentTime.t / 100);
    //cout << "dash_pow: " << dash_power << endl;
    DribbleRes res = SmartDribbleTo(pos, 75);    
    //DribbleRes res = DribbleTo(pos, 75, 1, drib_ang, DM_Lazy);
    DebugDrib(printf("DribRes: %d\n", res));
    if (res == DR_GotThere) {    
      stop_ball();
      pos = Vector(-pos.x, -pos.y);
    } else if (res == DR_LostBall) {
      /* go to intercept */
      if (Mem->MyInterceptionAble(75))
	go_to_point(Mem->MyInterceptionPoint(75));
      else
	my_error("Lost ball and can't intercept it");
    }
    
  } else {
    /* opponents to dribble around */
    static int FirstTime = TRUE;
    const int ydist = 2;
    if (!FirstTime)
      return;
    switch (Mem->MyNumber) {
      //case 1: move(-10, -ydist); break;
    case 1: move(-40, -15); break;
    case 2: move(-20,  ydist); break;
    case 3: move(-30, -ydist); break;
    case 4: move(-40,  ydist); break;
    case 5: move(-50, -ydist); break;
    }
    FirstTime = FALSE;
  }
}
#endif // RELEASE_VERSION

/* Here's the basic strategy for dribbling:
   Alternate kicks and dashs so that the ball stays close to us.
   Using predicted ball positions and velocity corrected kicks, then given
   an angle we want to dribble at, we can choose a dash or kick that is most
   appropraite */

/* Checks to see if a dash would keep us in control of ball or if it would
   help us catch up to ball. If not, it calls dribble_kick */
TurnKickCommand dribble_dash(Vector vEndPos, float max_pow,
			     AngleDeg drib_ang, DribbleMode mode)
{
  Mem->LogAction2(70, "dribble: trying to do a dash");
  DebugDrib(cout << "in dribble_dash" << endl);
  TurnKickCommand com;
  com.time = -1;
  com.turn_neck = FALSE;
  
  if (!Mem->BallWillBeKickable(1, max_pow, Mem->CP_kickable_buffer)) {    
    if (Mem->BallKickable()) {      
      DebugDrib(printf("Dribble: Ball is not goign to be kickable, so we'll kick it.\n"));
      Mem->LogAction2(80, "dribble: tried dash, but will lose ball, so kicking");
      return dribble_kick(vEndPos, max_pow, drib_ang, mode);
    }    
    if (!Mem->BallWillBeKickable(2, max_pow, Mem->CP_kickable_buffer) &&
	!Mem->BallWillBeKickable(3, max_pow, Mem->CP_kickable_buffer)) {
      DebugDrib(printf("Dribble: looks bad for dash, probably need to go to intercept"));
      Mem->LogAction2(70, "dribble: lost ball it seems");
      return com;
    } else
      DebugDrib(printf("Dribble: It's okay, ball will be kickable soon\n"));
  }

  if (Mem->WillDashBeCollision(max_pow, Mem->CP_collision_buffer)) {
    DebugDrib(printf("Dribble: dash would be a collision, so we'll kick instead\n"));
    if (Mem->BallKickable()) {
      Mem->LogAction2(70, "dribble: dash would collide, so we kick");
      return dribble_kick(vEndPos, max_pow, drib_ang, mode);
    } else {
      /* we'll fall through to a dash-
	 that'll at least get us close to the ball */
      DebugDrib(printf("Dribble: dash would be collision, but we can't kick the ball?"));
      Mem->LogAction2(70, "dribble: dash would collide, but can't kick!");
    }
  }
  
  //SMURF: need to plan for end point
  /* it might be a good idea to plan for the point where we want to stop
     dribbling so we don't overshoot and/or lose the ball when we stop */

  Mem->LogAction2(80, "dribble: doing a dash");
  DebugDrib(printf("Dribble: doing a basic dash\n"));
  com.time = Mem->CurrentTime;
  com.type = CMD_dash;
  com.power = max_pow;
  return com;
}

/* returns a kick that would put that would (assuming we dash next cycle)
   leave us with the ball at the correct angle after the next cycle */
TurnKickCommand GetDribbleKickCom(Vector vMyPred, AngleDeg drib_ang)
{
  Vector vBallTarg;

  Mem->LogAction2(90, "dribble: getting the kick command");
  
  if (!Mem->BallKickable())
    my_error("GetDribbleKickCom: Ball isn't kickable\n");
  
  vBallTarg = vMyPred +
    Polar2Vector(Mem->CP_dribble_ball_dist, drib_ang + Mem->MyBodyAng());

  vBallTarg = Mem->Global2RelativeToMyBody(vBallTarg);
  Vector vBallTraj = vBallTarg - Mem->BallRelativeToBodyPosition();
  /*vBallTraj = vBallTraj.rotate(-Mem->MyAng());*/

  return dokick(vBallTraj.dir(), vBallTraj.mod() / (1 + Mem->SP_ball_decay));
}

/* we may need to do a turnkick if the ball gets stuck behind us for some
   reason */
TurnKickCommand GetDribbleTurnKickCom(AngleDeg drib_ang)
{
  TurnKickCommand com;

  Mem->LogAction2(90, "dribble: getting the turnkick command");
  
  if (!Mem->BallKickable())
    my_error("GetDribbleTurnKickCom: Ball isn't kickable\n");
  
  DebugDrib(printf("Dribble: Having to do turnball\n"));
  TurnDir rot = RotToAvoidOpponent(drib_ang);
  DebugDrib(printf("Dribble: rot dir is %d\n", rot));
  KickToRes res = turnball_kick(drib_ang, rot, FALSE, &com);
  if (res == KT_LostBall)
    my_error("Dribble: turning lost the ball");
  else if (res == KT_Success || res == KT_DidNothing)
    {
      /* if we don;t know where we want to turn, just turn to in front of us
	 so that our next kick won;t be a collision */
      res = turnball_kick(0, TURN_CLOSEST, FALSE, &com);
      if (res == KT_Success || res == KT_DidNothing)
	my_error("Dribble: TurnKick finished in dribble, so we shouldn't have doen it");
    }  
  return com;
}


/* figures out a kick to do to keep dribbling
   (see the dribble_kick comment).
   Checks for collisions and calls turnkick if applicable */
/* the DribbleMode determines what to do when the kick we want to do is a
   collision. In strict mode, we always stop and turnball.
   In lazy mode, we try to get halfway to the desired angle and if that doesn't
   work, then we turnball */
/* lazy mode has been tested and works, but Strict mode has not. It seems to
   lead to many lost balls */
TurnKickCommand dribble_kick(Vector vEndPos, float max_pow,
			     AngleDeg drib_ang, DribbleMode mode)
{
  DebugDrib(cout << "in dribble_kick" << endl);
  if (!Mem->BallKickable())
    my_error("dribble_kick: Ball isn't kickable\n");
  
  TurnKickCommand com;
  /* figure out the point to kick the ball to */
  /* the 1 at the end is an idle cycle */
  Vector vMyPred = Mem->MyPredictedPosition(2, max_pow, 1);

  if ((vMyPred - Mem->MyPos()).mod() > (vEndPos - Mem->MyPos()).mod()) {
    /* if we are going to go far enough to get to our endpos, we may not want
       to dash as far */
    vMyPred = vEndPos;
  }
  
  com = GetDribbleKickCom(vMyPred, drib_ang);

  if (Mem->WillKickBeCollision(com.power, com.angle, Mem->CP_collision_buffer)) {
    if (mode == DM_Strict && fabs(Mem->BallAngleFromBody() - drib_ang) > 20 ) {      
      //	(mode == DM_Lazy && signf(Mem->BallAngle()) == signf(drib_ang))) {
      /* we'll turnball instead */
      DebugDrib(printf("In strict mode, doing turnball\n"));
      Mem->LogAction2(80, "dribble: strict mode, turnballing");
      com = GetDribbleTurnKickCom(drib_ang);
    } else if (mode == DM_Lazy) {
      //SMURF: should we average, or send to 0???
      //com = GetDribbleKickCom(vMyPred, 0);
      DebugDrib(printf("Dribble: In lazy mode, trying to interpolate changeover\n"));
      com = GetDribbleKickCom(vMyPred,
			      (Mem->BallAngleFromBody() + drib_ang)/2);
      if (Mem->WillKickBeCollision(com.power, com.angle, Mem->CP_collision_buffer)) {
	/* still didn't work, use turnball! */
	DebugDrib(printf("Dribble: In lazy mode, didn't work\n"));
	Mem->LogAction2(80, "dribble: lazy mode, but still have to turnball");
	com = GetDribbleTurnKickCom(drib_ang);
      } else {
	Mem->LogAction2(80, "dribble: lazy mode, averaging kick");
      }
    } else
      my_error("Dribble: mode is bad!");    
    
  } else {
    DebugDrib(printf("Dribble: doing a kick\n"));
    Mem->LogAction2(80, "dribble: doing a normal kick");
  }
  
  return com;
}


DribbleRes DribbleTo(Vector vEndPos, float max_dash_pow, float buffer,
		     AngleDeg drib_ang, DribbleMode mode,
		     Bool IsDodge, Vector DodgePoint)
{
  AngleDeg targAng;
  AngleDeg targAngErr = Mem->CP_max_go_to_point_angle_err;
  
  if (!Mem->BallPositionValid()) {
    my_error("must know where ball is to dribble");
    return DR_Error;
  }  

  /* first check velocity confidence */
  if (!Mem->BallVelocityValid()) {
    my_error("Shouldn't call dribble with ball velocity invalid");
    return DR_Error;
    /*Mem->LogAction2(60, "Dribble: ball velocity not valid");
    face_only_neck_to_ball();
    stop_ball();
    return DR_Going; */
  }

  DebugDrib(cout << "MyPos: " << Mem->MyPos() << "\tMyAng: " << Mem->MyAng() << endl);
  DebugDrib(cout << "BallPos: " << Mem->BallAbsolutePosition() << endl);
  DebugDrib(cout << "BallDist: " << Mem->BallDistance() << endl);
  DebugDrib(cout << "BallVel: " << Mem->BallAbsoluteVelocity() << endl);
  DebugDrib(cout << "vEndPos: " << vEndPos << "\tdrib_ang: " << drib_ang << endl);

  Mem->LogAction6(60, "DribbleTo: pow %.1f to (%.1f, %.1f) at ang %.2f",
		  max_dash_pow, vEndPos.x, vEndPos.y, drib_ang);
  
  if (mode == DM_None)
    my_error("Can't pass DM_None as a mode to DribbleTo");
  
  if ((Mem->MyPos() - vEndPos).mod() < buffer) {
    DebugDrib(printf("Got to target!\n"));
    Mem->LogAction2(80, "dribble: I got to target!");
    return DR_GotThere;
  }
  
  
  if (IsDodge) {
    /* there is something we need to dodge */
    /* we'll dodge whatever side of us the ball was supposed to go */
    Vector disp = vEndPos - Mem->MyPos();
    //DebugDrib(cout << "Disp1: " << di	sp << endl);
    disp *= Mem->CP_dodge_distance_buffer / disp.mod();
    //DebugDrib(cout << "Disp2: " <<	 disp << endl);
    disp = disp.rotate(signf(drib_ang) * 90);
    DebugDrib(cout << "Disp: " << disp << endl);
    if (Mem->DistanceTo(DodgePoint) < Mem->CP_dribble_dodge_close_dist) {
      DebugDrib(printf("Dribble: sharp dodge\n"));
      vEndPos = Mem->MyPos() + disp;
    } else {
      DebugDrib(printf("Dribble: regular dodge\n"));
      vEndPos = (DodgePoint + disp - Mem->MyPos())*2 + Mem->MyPos();      
    }
    DebugDrib(cout << "We're dodging a player: disp: " << disp
	      << "\tvEndPos: " << vEndPos << endl);
    Mem->LogAction2(80, "dribble: dodging a player");
    targAngErr = Mem->CP_dribble_dodge_angle_err;
  }    

  targAng = Mem->AngleToFromBody(vEndPos);
  DebugDrib(printf("targAng: %f\n", targAng));
  TurnKickCommand com;  

  /* if the mode is strict and then ball is way off,
     then go ahead and turnball */
  if (mode == DM_Strict &&
      fabs(Mem->BallAngleFromBody() - (targAng + drib_ang)) >
      Mem->CP_dribble_exp_angle_buffer &&
      Mem->BallKickable()) {
    com = GetDribbleTurnKickCom(drib_ang + targAng);
    if (DoTurnKickCommand(com))
      return DR_Going;
    else
      return DR_Error;
  }
  
  /* first turn the right way */
  if (fabs(targAng) > targAngErr) {
    DebugDrib(printf("Want to turn\n"));
    if (!Mem->BallWillBeKickable(1, 0, Mem->CP_kickable_buffer) &&
	!Mem->BallWillBeKickable(2, 0, 0))
      if (Mem->BallKickable()) {
	DebugDrib(printf("Ball will not be kickable, but it is now, so we're kicking\n"));
	Mem->LogAction2(80, "dribble: stopping the ball");
	com = dokick(0,0);
      } else {
	DebugDrib(printf("Ball will not be kickable, trying to catch up\n"));
	Mem->LogAction2(70, "dribble: trying to catch up to ball");
	com = dribble_dash(vEndPos, max_dash_pow, drib_ang, mode);
	if (com.time != Mem->CurrentTime)
	  return DR_LostBall;  
      }    
    else {      
      DebugDrib(printf("Dribble: turning to the right angle %f %f\n", targAng, Mem->MyAng()));
      Mem->LogAction2(80, "dribble: turning the right way");
      com.time = Mem->CurrentTime;
      com.type = CMD_turn;
      com.angle = targAng;
      com.turn_neck = FALSE;
    }    
    if (DoTurnKickCommand(com))
      return DR_Going;
    else
      return DR_Error;
  }

  if (Mem->BallKickable()) {
    AngleDeg expBallAngle;
    expBallAngle = (Mem->BallPredictedPosition() -
		    Mem->MyPredictedPosition(1, max_dash_pow)).dir();
    expBallAngle -= Mem->MyBodyAng();
    NormalizeAngleDeg(&expBallAngle);
    DebugDrib(printf("Expected ball angle: %f\n", expBallAngle));
    if (fabs(expBallAngle-drib_ang) > Mem->CP_dribble_exp_angle_buffer ||
	fabs(expBallAngle) > 90) {      
      /* kick if the ball will end up behind us */
      DebugDrib(printf("The ball will be too far off it's target, so kicking\n"));
      Mem->LogAction2(70, "dribble: kicking because ball will be behind us");
      com = dribble_kick(vEndPos, max_dash_pow, drib_ang, mode);
    } else {      
      /* otherwise dash */
      DebugDrib(printf("Going to try to do a dash\n"));
      Mem->LogAction2(70, "dribble: trying to dash");
      com = dribble_dash(vEndPos, max_dash_pow, drib_ang, mode);
    }    
  } else {
    DebugDrib(printf("Ball is not kickable, better dash\n"));
    Mem->LogAction2(70, "dribble: trying to dash because ball not kickable");
    com = dribble_dash(vEndPos, max_dash_pow, drib_ang, mode);
  }
  
  if (com.time != Mem->CurrentTime)
    return DR_LostBall;
  
  if (DoTurnKickCommand(com))
    return DR_Going;
  else
    return DR_Error;
}

DribbleRes SimpleDribbleTo(Vector vEndPos, float max_dash_pow, float buffer)
{
  return DribbleTo(vEndPos, max_dash_pow, buffer, Mem->CP_dribble_angle_norm,
		   DM_Lazy, FALSE, Vector(0,0));
  
}



AngleDeg GetNormalizeDribAng(AngleDeg ang)
{
  NormalizeAngleDeg(&ang);
  if (ang <= -90)
    ang = -90;
  if (ang >= 90)
    ang = 90;
  return ang;
}

AngleDeg GetNormalizeDribAngWithBuffer(AngleDeg ang)
{
  NormalizeAngleDeg(&ang);
  if (fabs(ang) > 180-Mem->CP_dribble_angle_ignore_buffer)
    /* we want the ball right in back, so use the side the ball is on */
    ang = signf(Mem->BallAngleFromBody())*90; 
  else 
    ang = GetNormalizeDribAng(ang);
  return ang;
}


#ifndef RELEASE_VERSION  

/* the calculated dribble angle is relative to the trajectory */
DribbleRes SmartDribbleTo(Vector vEndPos, float max_dash_pow, 
			  float buffer)
{
  AngleDeg dribAng;
  DribbleMode dribMode = DM_Lazy;

  Mem->LogAction5(50, "SmartDribbleTo (%f, %f) with dash_pow %f",
		 vEndPos.x, vEndPos.y, max_dash_pow);
  
  DebugDrib(cout << "at SmartDribbleTo; vEndPos: " << vEndPos
	    << "\tbuffer: " << buffer << endl);
  DebugDrib(cout << "MyPos: " << Mem->MyPos() << endl);
  /* we need to decide the mode (strict or lazy) and the dribble angle */
  AngleDeg vote_ang = 0.0;
  float total_weight = 0.0;
  float dist_weight = 3.0;
  float ang_weight = 1.0;

  /* we're going to figure out the angle, but we're also goign to keep track
     of the closest player who's in our way right now */
  Vector traj = vEndPos - Mem->MyPos();
  AngleDeg trajAng = traj.dir() - Mem->MyBodyAng();
  Line trajLine;
  trajLine.LineFromTwoPoints(Mem->MyPos(), vEndPos );
  Unum dodgePlayer = -1; /* can't use Unum_Unknown casue it's 0 */
  float dodgePlayerDist = HUGE;
  Vector dodgePlayerPos;
  for (Unum i=0; i <= Mem->SP_team_size; i++) {
    Vector pos;
    float dist;
    float ang;
    if (i==0) {
      /* look for closest teamless player */
      if (Mem->NumTeamlessPlayers() < 1)
	continue;
      pos = Mem->ClosestTeamlessPlayerPosition();
      dist = Mem->DistanceTo(pos);
      ang = Mem->AngleToFromBody(pos);
      if (dist > Mem->CP_dribble_ignore_opp_dist) {
	DebugDrib2(printf("Player %d: player too far away\n", i));
	continue;
      }      
    } else {
      if (!Mem->OpponentPositionValid(i)) {
	continue;
      }    
      if (Mem->OpponentDistance(i) > Mem->CP_dribble_ignore_opp_dist) {      
	DebugDrib2(printf("Player %d: player too far away\n", i));
	continue;
      }
      pos = Mem->OpponentAbsolutePosition(i);
      dist = Mem->OpponentDistance(i);
      ang = Mem->OpponentAngleFromBody(i);
    }
    
    
    //this next check is to put the dribble mode to strict if a player is real close
    //However, lazy mode is working okay, so we're goign to ignore this for now
    /* SMURF
    if (Mem->OpponentDistance(i) < Mem->CP_dribble_worry_opp_dist)
      dribMode = DM_Lazy;//DM_Strict; */
    float weight = (dist_weight * Cos(90 * dist / Mem->CP_dribble_ignore_opp_dist));
    DebugDrib2(cout << "Player " << i << "\tdist weight: " << weight << endl);
    weight *= ang_weight * (Sin((GetNormalizeAngleDeg(ang-trajAng)+180)/2)*.5 + .5);
    DebugDrib2(cout << "Player " << i << "\ttotal weight: " << weight << endl);
    DebugDrib2(cout << "Player " << i << "\tAngle: "
	      << ang << "\tVoteAngle: "
	      << GetNormalizeDribAngWithBuffer(ang-trajAng+180)
	      << endl);
    //weight /= (dist_weight + ang_weight);
    //printf(" In loop: %d\tweight: %f\tang: %f\n", i, weight,
    //	   GetNormalizeAngleDeg(Mem->OpponentAngle(i)+180));
    vote_ang += weight*GetNormalizeDribAngWithBuffer(ang-trajAng+180);
    total_weight += weight;

    /* now look at whether he's on our path */
    if (trajLine.dist(pos) < Mem->CP_dodge_distance_buffer) {      
      Vector ptOnTraj = trajLine.ProjectPoint(pos);
      /* see if they are in a parabola from us */
      if (Sqr((Mem->MyPos() - ptOnTraj).mod()) > (pos - ptOnTraj).mod() ) {	
	float distOnTraj = (ptOnTraj - Mem->MyPos()).mod();
	Vector oppTraj = ptOnTraj - Mem->MyPos();
	if (fabs(oppTraj.dir() - traj.dir()) <
	    Mem->CP_max_go_to_point_angle_err &&
	    distOnTraj < Mem->CP_dribble_dodge_max_dist &&
	    (dodgePlayer == -1 || dodgePlayerDist > dist)) {	  
	  dodgePlayer = i;
	  dodgePlayerDist = dist;
	  dodgePlayerPos = pos;
	}
      }      
    }
  }
  if (total_weight == 0.0) {
    /* no players in the area */
    DebugDrib(printf("No players in the area. Normal dribble\n"));
    dribAng = signf(Mem->BallAngleFromBody())*Mem->CP_dribble_angle_norm;    
  } else {
    //dribAng = vote_ang / total_weight;
    dribAng = GetNormalizeDribAngWithBuffer(vote_ang / total_weight);
    DebugDrib(printf("Using weighted thing: ang: %f\t weight: %f\n",
		     vote_ang, total_weight));
    DebugDrib(printf("Using weighted thing: %f\n", dribAng));
  }

  /* now we're going to make sure that the drib ang isn't too close to the
     sideline */
  Vector dribble_pos = Mem->MyPos() +
    Polar2Vector(Mem->CP_dribble_ball_dist, traj.dir() + dribAng);
  Rectangle field = Mem->FieldRectangle.shrink(Mem->CP_dribble_sideline_buffer);
  if (!field.IsWithin(dribble_pos)) {
    DebugDrib(cout << "Correcting for sideline; dribAng: " << dribAng << endl);
    Line l = field.nearestEdgeLine(dribble_pos);
    Vector sol1, sol2;
    DebugDrib(cout << "edge: " << l
	      << "\tbuff: " << Mem->CP_dribble_sideline_buffer << endl);
    int numSol = LineCircleIntersect(l,
				     Mem->CP_dribble_ball_dist,
				     Mem->MyPos(),
				     &sol1, &sol2);
    DebugDrib(cout << "numSol: " << numSol << "\tsol1: " << sol1 << "\tsol2: " << sol2 << endl);
    if (numSol == 0)
      //my_error("Dribble: correcting for sideline, but no LineCircle Intersection?");
      dribAng =
	Mem->AngleToFromBody(field.AdjustToWithin(Mem->MyPos())) - trajAng;
    else if (numSol == 1)
      dribAng = Mem->AngleToFromBody(sol1) - trajAng;
    else
      dribAng = (Mem->FieldRectangle.DistanceToEdge(sol1) >
		 Mem->FieldRectangle.DistanceToEdge(sol2)) ?
	Mem->AngleToFromBody(sol1) - trajAng :
      Mem->AngleToFromBody(sol2) - trajAng;
    DebugDrib(cout << "Finished Correction; dribAng: " << dribAng << endl);
    dribAng = GetNormalizeDribAng(dribAng);
  }

  
  DribbleRes res = 
    DribbleTo(vEndPos, max_dash_pow, buffer, dribAng, dribMode,
	      (Bool)(dodgePlayer != -1), dodgePlayerPos);

  /* SmartDribbleTo also takes care or turning the neck */
  /* DribbleTo might have turned the neck, so don't want to mess with that */
  if (!Mem->TurnNeckThisCycle() &&
      (res == DR_Going || res == DR_GotThere) ) {
    if (Mem->CP_dribble_scan_field) {
      if (Mem->TimeToTurnForScan()) {
	Mem->LogAction2(90, "Dribble: turning neck for scanning");
	scan_field_with_neck();
      }
    } else {
      /* if velocity conf is high, look around a little.
	 otherwise look at the ball */
      if (Mem->BallVelocityValid() >
	  int_pow(Mem->CP_ball_conf_decay, Mem->PredictedNextSightInterval())) {
	/* enough conf to look around */
	/* Let's see if there is a teamless player that we shoudl look at */
	
	Vector pos;
	if (Mem->NumTeamlessPlayers() > 0 &&
	    Mem->DistanceTo((pos = Mem->ClosestTeamlessPlayerPosition())) < 2.0 * Mem->SP_feel_distance && 
	    Mem->CanSeePointWithNeck(pos)) {
	  Mem->LogAction2(90, "Dribble: turning neck to look at a teamless player");
	  face_only_neck_to_point(pos);
	} else {
	  Mem->LogAction2(90, "Dribble: turning neck to look around");
	  turn_neck(Mem->LimitTurnNeckAngle((dribAng - signf(dribAng)*2.0*Mem->MyViewAngle())
					    - Mem->MyNeckRelAng()));
	}	
      } else {
	/* just look at the ball */
	Mem->LogAction2(90, "Dribble: turning neck to look at ball");
	turn_neck(Mem->LimitTurnNeckAngle(dribAng - Mem->MyNeckRelAng()));
      }
    }
  }

#ifndef RELEASE_VERSION  
  /* Remember that we're dribbling so the action mode can keep dribbling
     even if the ball's not quite kickable */
  if ( res == DR_Going )
    Mem->SetDribbling(vEndPos,max_dash_pow,buffer);
#endif // RELEASE_VERSION

  return res;
}
#endif // RELEASE_VERSION
