/* -*- Mode: C++ -*- */

/* MemAction.C
 * CMUnited99 (soccer client for Robocup99)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1999 Peter Stone
 *
 * CMUnited-99 was created by Peter Stone, Patrick Riley, and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

#include "MemAction.h"

#ifdef DEBUG_OUTPUT
#define DebugClear(x) 
#else
#define DebugClear(x) 
#endif

void ActionInfo::Initialize()
{
#ifndef RELEASE_VERSION  
  MyActionMode = AM_Unknown;
  PullOffside = FALSE;
#endif

  Stored_Fastest_Teammate_Time = 0;
  Stored_Fastest_Opponent_Time = 0;

  for (int i = 1; i <= SP_team_size; i++) {
    TeamIntInfo[i] = new PlayerInterceptInfo;
    TeamIntInfo[i]->time = -1;
    OppIntInfo[i] = new PlayerInterceptInfo;
    OppIntInfo[i]->time = -1;
  }

  kick_in_progress = FALSE;
#ifndef RELEASE_VERSION  
  goalie_last_catch_time = 0;
  GoalieWarnTime = 0;
#endif // RELEASE_VERSION

#ifndef RELEASE_VERSION  
  goalie_after_catch = FALSE;
  
  DribbleTargetAvailable = FALSE;
  DribbleTargetTime = -1;
#endif

  InterceptLookahead = LA_Default;
  IntMinCyc = -1;
  IntMinCycTime = -1;

  HKTime = -1;
  HKStep = -1;
  HKStepNext = -1;
  HKrot = TURN_CW;

#ifndef RELEASE_VERSION  
  def_block_overrun_time = -1;
  def_block_overrun = FALSE;

  my_breakaway_time = 0;
  my_breakaway_targ = Vector(0,0);
  
  ChangeViewForHandleBallTime = 0;
  LastBallPossessorTime = 0;

  team_pass_time = -1;
  team_receiver = Unum_Unknown;
#endif
}




/*****************************************************************************************/
/*****************************************************************************************/
/*****************************************************************************************/

#ifndef RELEASE_VERSION  
void ActionInfo::UpdateActionMode()
{
  /* make this a finite state machine? */

  if ( CP_goalie && !(SetPlay && InSetPlay) && (PlayMode == PM_Play_On || !goalie_after_catch) ){
    LogAction2(10, "Mode: AM_goaltend");
    MyActionMode = AM_goaltend; return;
  }
  if ( !MyConf() ){
    LogAction2(10, "Mode: AM_Localize");
    MyActionMode = AM_Localize; return;
  }
  if ( PlayMode == PM_Before_Kick_Off ){
    LogAction2(10, "Mode: AM_Before_Kick_Off");
    MyActionMode = AM_Before_Kick_Off; return;
  }
  if ( !BallPositionValid() ){
    LogAction2(10, "Mode: AM_Face_Ball (ball position not valid)");
    MyActionMode = AM_Face_Ball; return;
  }
  if ( PlayMode != PM_Play_On && CurrentTime - PlayModeTime <= CP_Setplay_Delay ){
    LogAction2(10, "Mode: AM_Face_Ball (waiting to initialize set play)");
    MyActionMode = AM_Face_Ball; return;
  }
  if ( NeedToInitializeSetPlay() && !SetPlay ){
    LogAction2(10, "Mode: AM_Face_Ball (initializing set play)");
    MyActionMode = AM_Face_Ball; return;
  }
  if ( SetPlay && InSetPlay ){
    LogAction2(10, "Mode: AM_Setplay");
    MyActionMode = AM_Setplay; return;
  }
  if ( SetPlay && MyMark() == Unum_Unknown ){
    LogAction2(10, "Mode: AM_Offense_Passive (Setplay and no mark)");
    MyActionMode = AM_Offense_Passive; return;
  }
  if ( !SetPlay && BallKickable() && ClosestTeammateToBall()==MyNumber ){
    LogAction2(10, "Mode: AM_With_Ball");
    MyActionMode = AM_With_Ball; return;
  }
  if ( MyStamina() < EffortDecThreshold ){
    LogAction2(10, "Mode: AM_Recover");
    MyActionMode = AM_Recover; return;
  }
  if ( !SetPlay && DribbleInProgress() && ClosestTeammateToBall()==MyNumber && 
       BallVelocityValid() && BallWillBeKickable(1,dribble_in_progress_dash_power) ){
    /* continue dribbling */
    LogAction2(10, "Mode: AM_With_Ball (even though ball not kickable -- for dribble)");
    MyActionMode = AM_With_Ball; return;
  }

  /***************************************/

  //MyActionMode = AM_Offense_Passive; return;

  Unum opp_breakaway;
  if ( !SetPlay && !BallMoving() && ClosestTeammateToBall()==MyNumber ){
    if ( (opp_breakaway = TheirBreakaway()) != Unum_Unknown ){
      /* Defense_Active mode has been improved, so now we want to stay in it */
#ifdef NOT_USED
      Vector opp_pos = OpponentAbsolutePosition(opp_breakaway);
      Line traj_line = LineFromTwoPoints(MarkerPosition(RM_My_Goal),opp_pos);
      Vector my_projection = traj_line.GetClosestPtInBetween(MyPos(),MarkerPosition(RM_My_Goal),opp_pos);
      if ( my_projection.dist(opp_pos) <= CP_their_breakaway_kickable_buffer ) /* I'm behind the ball */
	{}
#endif	
      LogAction2(10, "Mode: AM_Defense_Active (ball not moving, I'm closest, but TheirBreakaway)");
      MyActionMode = AM_Defense_Active; return;
    }
    LogAction2(10, "Mode: AM_Offense_Active (ball not moving, I'm closest)");
    MyActionMode = AM_Offense_Active; return;
  }
  if ( !SetPlay && MyInterceptionAble() && FastestTeammateToBall()==MyNumber ){
    if ( (opp_breakaway = TheirBreakaway()) != Unum_Unknown ){
#ifdef NOT_USED
      Vector opp_pos = OpponentAbsolutePosition(opp_breakaway);
      Line traj_line = LineFromTwoPoints(MarkerPosition(RM_My_Goal),opp_pos);
      Vector my_projection = traj_line.GetClosestPtInBetween(MyPos(),MarkerPosition(RM_My_Goal),opp_pos);
      if ( my_projection.dist(opp_pos) <= CP_their_breakaway_kickable_buffer ){ /* I'm behind the ball */
      }
#endif	
      LogAction2(10, "Mode: AM_Defense_Active (I'm fastest, but TheirBreakaway)");
      MyActionMode = AM_Defense_Active; return;
    }
    LogAction2(10, "Mode: AM_Offense_Active (I'm fastest to ball)");
    MyActionMode = AM_Offense_Active; return;
  }

  if ( !SetPlay )
    LogAction3(50,"I'm not fastest to the ball -- %d is faster",FastestTeammateToBall());

  char possession;
  possession = '?';
  if ( PlayMode != PM_Play_On ) possession = TeamInPossession();

  possession = TeamInPossession(); 

  /* if ( MyNumber == 3 )
    printf("%d : %d   with ball %c   possession %c\n",
	   CurrentTime.t,MyNumber,TeamWithBall(),possession); */

  Unum fastest;
  if ( !SetPlay && MyInterceptionAble() && 
       ( (MyPositionType() <= PT_Defender && possession != MySide) ||
	 (MyPositionType() > PT_Defender && possession == TheirSide) ) &&
       ( ( (fastest = FastestTeammateToBall()) != Unum_Unknown && 
	   TeammateDistanceToBall(fastest) < BallDistance() && 
	   NumTeammatesCloserToBall()<=1 ) || 
	 ClosestTeammateToBall() == MyNumber ) ){
    LogAction3(10, "Mode: AM_Defense_Active (possession == %c)",possession);
    MyActionMode = AM_Defense_Active; return;
  }

  /***************************************/
  
  if ( team_receiver == MyNumber && team_pass_time > LastStartClockTime && 
       CurrentTime - team_pass_time < 10 ){
    LogAction2(10, "Mode: AM_Face (receiving pass)");
    MyActionMode = AM_Face_Ball; return;
  }
  if ( VP_train_DT && TrainingPassInProgress() ){
    LogAction2(10, "Mode: AM_Watch_Pass");
    MyActionMode = AM_Watch_Pass; return;
  }
  if ( InOffsidePosition() ){
    LogAction2(10, "Mode: AM_GetOnSide");
    MyActionMode = AM_GetOnSide; return;
  }

  /***************************************/

  /* defenders */
  if ( MyPositionType() <= PT_Defender ){ 
    if ( CP_spar && possession == MySide && 
	 ( BallX() < MyPositionX() ||        
	   NumCloserAdjustedForOffsidePositions(MyPosition(),BallAbsolutePosition() ) <= 2 ) ){
      LogAction2(10, "Mode: AM_Offense_Auxiliary (defender)");
      MyActionMode = AM_Offense_Auxiliary; return; 
    }
    if ( MyPositionType() == PT_Sweeper && TheirBreakaway() != Unum_Unknown ){
      LogAction2(10, "Mode: AM_Defense_Active (sweeper, their breakaway)");
      MyActionMode = AM_Defense_Active; return;
    }
    if ( CP_mark == TRUE ){
      LogAction2(10, "Mode: AM_Defense_Auxiliary (defender)");
      MyActionMode = AM_Defense_Auxiliary; return;  /* auxiliary or passive? */
    }
    LogAction2(10, "Mode: AM_Offense_Passive (defender)");
    MyActionMode = AM_Offense_Passive; return; 
  }
      
  /* forwards */
  if ( MyPositionType() == PT_Forward ){
    if ( CP_spar && MyPositionX(TRUE) - BallX() < 30 ){
      LogAction2(10, "Mode: AM_Offense_Auxiliary (forward)");
      MyActionMode = AM_Offense_Auxiliary; return;
    }
    LogAction2(10, "Mode: AM_Offense_Passive (forward)");
    MyActionMode = AM_Offense_Passive; return;
  }

  /* midfielders */
  if ( MyPositionType() == PT_Midfielder ){
    if ( BallX() > MyPositionX() ){
      if ( !CP_spar || possession == TheirSide ){
	LogAction2(10, "Mode: AM_Offense_Passive (midfielder -- ball in front)");
	MyActionMode = AM_Offense_Passive; return;  /* make this defense_aux?? */
      }
      LogAction2(10, "Mode: AM_Offense_Auxiliary (midfielder -- ball in front)");
      MyActionMode = AM_Offense_Auxiliary; return;  
    }
    else{ /* ball behind position x */
      if ( CP_spar && possession != TheirSide && 
	   NumCloserAdjustedForOffsidePositions(MyPosition(),BallAbsolutePosition() ) <= 3 ){
	LogAction2(10, "Mode: AM_Offense_Auxiliary (midfielder -- ball behind)");
	MyActionMode = AM_Offense_Auxiliary; return;  
      }
      LogAction2(10, "Mode: AM_Offense_Passive (midfielder -- ball behind)");
      MyActionMode = AM_Offense_Passive; return;   /* make this defense_aux?? */
    }
  }

  my_error("shouldn't get here");
  LogAction2(10, "Mode: AM_Offense_Passive (shouldn't get here)");
  MyActionMode = AM_Offense_Passive;
}
#endif




/*****************************************************************************************/
/*****************************************************************************************/
/*****************************************************************************************/

#ifdef DEBUG_OUTPUT
#define DebugInt(x) 
#else
#define DebugInt(x) 
#endif

/* only used for this player */
PlayerInterceptInfo
ActionInfo::CloseBallInterception(float max_pow, int max_lookahead,
				  Vector vBallPos, Vector vBallVel)
{
  Vector vNewPos;
  PlayerInterceptInfo info;
  float dash_dist = max_pow * SP_dash_power_rate;
  info.dash_pow = max_pow;
  info.lookahead = max_lookahead;
  info.res = BI_None;  
  
  vBallPos += vBallVel;
  vBallVel *= SP_ball_decay;
  /* we need to figure out the right dash power so that the ball ends up right in
     front of us. Like many things, this ends up as a LineCircleIntersect problem */
  Vector vMyPred = MyPredictedPosition();
  Ray    rDash(vMyPred, MyBodyAng());
  Line   lDash   = LineFromRay(rDash);
  Vector sol1, sol2;
  
  int num_sol = LineCircleIntersect(lDash, SP_player_size + SP_ball_size + CP_collision_buffer,
				    vBallPos, &sol1, &sol2);
  if (num_sol >= 1) {
    /* we'll make sure that the right answer is in sol1 */
    if (num_sol == 2) {
      /* we have to pick which point is better */
      if (fabs(GetNormalizeAngleDeg((vBallPos - sol2).dir() - MyBodyAng())) < 90) {
	sol1 = sol2; //sol2 is the right solution, put it in sol1
      } else if (!(fabs(GetNormalizeAngleDeg((vBallPos-sol1).dir() - MyBodyAng())) < 90)) {
	my_error("CloseBallInterception: 1 ahead, neither solution looks good %.2f %.2f",
		 GetNormalizeAngleDeg((vBallPos - sol2).dir() - MyBodyAng()),
		 GetNormalizeAngleDeg((vBallPos - sol1).dir() - MyBodyAng()));
      }
    }

    /* now figure out the dash power based on that point */
    float dash_pow = vMyPred.dist(sol1) / SP_dash_power_rate;
    dash_pow = MinMax(SP_min_power, dash_pow, SP_max_power);
    if (!rDash.InRightDir(sol1))
      dash_pow = -dash_pow;

    if (vBallPos.dist(MyPredictedPosition(1, dash_pow)) < SP_kickable_area) {
      /* this works! */
      info.res = BI_CanChase;
      info.numCyc = 1;
      info.dash_pow_to_use = dash_pow;
      //this'll make go_to_point dash
      info.pos = vMyPred + Polar2Vector(signf(dash_pow)*dash_dist, MyBodyAng());
      LogAction5(70, "CloseBallInterception: One dash and we're there: %.2f to (%.2f, %.2f)",
		 dash_pow, sol1.x, sol1.y);
      return info;
    }
    
    
  }
  
  vBallPos += vBallVel;
  vBallVel *= SP_ball_decay;
  //now look two cycles ahead
  //try turn then dash
  float targ_ang = (vBallPos - MyPredictedPosition(2)).dir();
  if (fabs(targ_ang - MyBodyAng()) > CP_max_go_to_point_angle_err) {
    vNewPos = MyPredictedPositionWithTurn(targ_ang - MyBodyAng(), 2, max_pow);
    if (vNewPos.dist(vBallPos) < SP_kickable_area) {
      info.res = BI_CanChase;
      info.numCyc = 2;
      info.dash_pow_to_use = max_pow;
      //this'll make go_to_point turn
      //info.pos = MyPos() + Polar2Vector(dash_dist, targ_ang); 
      info.pos = MyPredictedPosition() + Polar2Vector(dash_dist, targ_ang); 
      LogAction2(70, "CloseBallInterception: Turn then dash and we're there");
      return info;
    }
  }
  //try two dashes
  vNewPos = MyPredictedPosition(2, max_pow);
  if (vNewPos.dist(vBallPos) < SP_kickable_area) {
    info.res = BI_CanChase;
    info.numCyc = 2;
    info.dash_pow_to_use = max_pow;
    //this'll make go_to_point dash
    //info.pos = MyPos() + Polar2Vector(2*dash_dist, MyBodyAng()); 
    info.pos = MyPredictedPosition() + Polar2Vector(dash_dist, MyBodyAng());
    LogAction2(70, "CloseBallInterception: Two dashes and we're there");
    return info;
  }
  return info;
}

/*****************************************************************************************/

/* does not set time field */
/* cyc inc is initially CP_intercept_step, but when an answer is found, we
   then bring cyc back a step, and go up by ones */
PlayerInterceptInfo
ActionInfo::ActiveCanGetThere(float max_pow, int max_lookahead,
			   Vector vBallPos, Vector vBallVel,
			   char side, Unum num,
			   Vector vPlayerPos, Vector vPlayerVel,
			   float fPlayerAng, int PlayerAngValid,
			   bool IsThisMe)
{
  float at_point_buffer = 1;
  PlayerInterceptInfo info;
  //Vector vPredPlayer = vPlayerPos + vPlayerVel;
  Vector vPredPlayer = vPlayerPos +
    vPlayerVel * (SumInfGeomSeries(vPlayerVel.mod(), SP_player_decay));
  Vector vOldBallPos, vOldBallVel;
  float turn_angle;
  int cyc;
  int cyc_inc = (IsThisMe ? CP_my_intercept_step : CP_intercept_step); 
  int max_cyc = (max_lookahead + cyc_inc - 1);  
  max_cyc -= (max_cyc % cyc_inc); 
  /* max_cyc is so that we don't miss an interception if CP_intercept_step is not
     1. For example, if CP_intercept_step is 5, max_look is 14, and we can
     intercept in 13, we would return no intercept if we just used max_lookahead */
  
  DebugInt(printf(" ACGT: BallPos.vel.mod: %f\n", vBallVel.mod() ));

  info.dash_pow_to_use = max_pow;
  NormalizeAngleDeg(&fPlayerAng);

  /* we want to aim a little ahead of the ball, so advance it a little */
  for (int i=0; i < CP_intercept_aim_ahead; i++) {
    vBallPos += vBallVel;
    vBallVel *= SP_ball_decay;
  }

  if (IsThisMe) LogAction4(140, "ActiveBallIntercept: %d %d", (int)max_pow, max_lookahead);
  
  for (cyc=0; cyc<=max_cyc; cyc += cyc_inc) {

    if (!IsPointInBounds(vBallPos,-3)) {  /* expand the field by 3 meters so we don't give up to soon */
      DebugInt(printf("The ball will go out of bounds before we can get it\n"));
      break;
    }
    
    /* decide if we need to turn to ball */
    float ball_ang = (vBallPos - vPredPlayer).dir();
    Vector vEndSpot;
    /* SMURF - we should probably aim for ball 1 cycle ahead or something
       like that */
    //DebugInt(printf(" angle to exp ball pos: %f\n", AngleTo(vBallPos)));
    turn_angle = ball_ang - fPlayerAng;
    if (fabs(turn_angle) < CP_max_go_to_point_angle_err)
      turn_angle = 0.0;      
    if (IsThisMe) {
      vEndSpot = MyPredictedPositionWithTurn(turn_angle, cyc, max_pow,(turn_angle != 0.0));
    } else {
      int run_cyc = cyc;
      if (PlayerAngValid) {
	if (turn_angle != 0.0)
	  run_cyc--;
	run_cyc = Max(0, run_cyc);
      }
      Vector PlayerDash =
	Polar2Vector(max_pow*SP_dash_power_rate, ball_ang);
      vEndSpot =
	PlayerPredictedPosition(side, num, run_cyc, PlayerDash);
    }
    

    float dist_to_ball_after = (vBallPos - vEndSpot).mod();
    /* if we can make it there */
    /* SMURF- is this too lenient? */
    if (dist_to_ball_after <= at_point_buffer || 
        (vEndSpot - vPredPlayer).mod() > (vBallPos - vPredPlayer).mod() + SP_kickable_area) {
      /* we can get to the ball! */
      /* OR we travelled far enough, but somehow missed the ball,
	 return sucess */
      if (dist_to_ball_after <= at_point_buffer) {
	if (IsThisMe) LogAction4(100, "Found a ball interception by being close (%.2f, %.2f)",
				vBallPos.x, vBallPos.y);
	info.numCyc = cyc;
      } else {	
	if (IsThisMe) LogAction4(100, "Found a ball interception by going far (%.2f, %.2f)",
				vBallPos.x, vBallPos.y);
	info.numCyc = cyc;
	//vBallPos += vBallVel; /* advance one spot for that turn*/
      }      
      
      if (cyc_inc > 1 && cyc != 0) {
	/* we want the best answer- go back and go up by ones */
	if (IsThisMe)
	  LogAction2(100, "Found a ball interception, but goign back for accuracy");	
	DebugInt(printf("Found answer, but going back for accuracy: %d\n", cyc));
	cyc -= cyc_inc;
	vBallPos = vOldBallPos;
	vBallVel = vOldBallVel;
	cyc_inc = 1;
	max_cyc = max_lookahead; // don;t need to go above this anymore
      } else {
	/* we want to try avoiding turning towards the ball for only a small savings
	   in time to intercept */
	if (IsThisMe && CP_no_turn_max_cyc_diff > -1 &&
	    turn_angle != 0.0 &&
	    (vBallVel.x >= FLOAT_EPS || vBallVel.y >= FLOAT_EPS)) {
	  Ray rBall(vBallPos, vBallVel);
	  Ray rPlayer(vPredPlayer, fPlayerAng);
	  Vector int_pt;
	  if (rBall.intersection(rPlayer, &int_pt)) {
	    float dist = vEndSpot.dist(int_pt);
	    float num_cyc; /* the number of cycles extra it takes the ball to get to
				this pos */
	    num_cyc = SolveForLengthGeomSeries(vBallVel.mod(), SP_ball_decay, dist);
	    LogAction3(90, "No turn interception: It takes %.2f extra cycles", num_cyc);
	    // if an answer less than 0 is given, the ball will never get there
	    if (num_cyc >= 0 &&
		num_cyc <= CP_no_turn_max_cyc_diff) {
	      /* use this target instead! */
	      LogAction4(70, "Using the new no turning interception point (%.2f, %.2f)",
			 int_pt.x, int_pt.y);
	      info.res = BI_CanChase;
	      info.pos = int_pt;
	      return info;
	    } /* using no turn interseption */
	  } /* there is an intersection */
	  
	} /* no turn interseption */
	

	if (info.numCyc > max_lookahead) {
	  info.res = BI_Failure;
	} else {
	  info.res = BI_CanChase;
	  info.pos = vBallPos;
	}	
	return info;
      }      
    }
    
    /* update ball position estimate */
    vOldBallPos = vBallPos;
    vOldBallVel = vBallVel;
    for (int i=0; i < cyc_inc; i++) {
      vBallPos += vBallVel;
      vBallVel *= SP_ball_decay;
    }
      
    
  } /* cycle loop */

  info.res = BI_Failure; // can't make it to ball before max_lookahead
  return info;
}

/*****************************************************************************************/

void ActionInfo::BallIntercept_active(float max_pow_to_use, int max_lookahead,
				   char PlayerSide, Unum PlayerNum,
				   PlayerInterceptInfo* pInfo)
{
  Vector PlayerPos;
  Vector PlayerVel;
  float PlayerAng;
  int AngValid = FALSE;
  Vector BallVel;

  pInfo->res = BI_None;

  if (!BallPositionValid()) {
    my_error("BallIntercept_active: Can't get to ball if I don;t know where it is");
    pInfo->res = BI_Invalid;
    return;
  }
  
  if (!PlayerPositionValid(PlayerSide, PlayerNum)) {
    my_error("BallIntercept_active: Can't give an answer if I don't know where player is");
    pInfo->res = BI_Invalid;
    return;
  }
  PlayerPos = PlayerAbsolutePosition(PlayerSide, PlayerNum);
  //DebugInt(cout << "PlayerPos: " << PlayerPos << endl);
  
  if (PlayerVelocityValid(PlayerSide, PlayerNum)) {
    PlayerVel = PlayerAbsoluteVelocity(PlayerSide, PlayerNum);
  } else {    
    PlayerVel = Vector(0,0);
  }
  
  if (PlayerBodyAngleValid(PlayerSide, PlayerNum)) {
    AngValid = TRUE;
    PlayerAng = PlayerAbsoluteBodyAngle(PlayerSide, PlayerNum);
  } else
    PlayerAng = 0;
    
  if ((PlayerPos - BallAbsolutePosition()).mod() <
      SP_kickable_area) {
    pInfo->res = BI_ReadyToKick;
    pInfo->numCyc = 0;
    pInfo->pos = PlayerPos;
    return;
  }

  if (BallVelocityValid())
    BallVel = BallAbsoluteVelocity();
  else
    BallVel = Vector(0,0);
  
  DebugInt(printf("At BallIntercept_active  max_pow: %f, max_look: %d\n",
		  max_pow_to_use, max_lookahead));

  if (PlayerSide == MySide && PlayerNum == MyNumber)
    *pInfo = CloseBallInterception(max_pow_to_use, max_lookahead,
				   BallAbsolutePosition(), BallVel);
  
  if (pInfo->res == BI_None)
    *pInfo = 
      ActiveCanGetThere(max_pow_to_use, max_lookahead,
			BallAbsolutePosition(), BallVel,
			PlayerSide, PlayerNum,
			PlayerPos, PlayerVel, PlayerAng, AngValid,
			(PlayerSide == MySide && PlayerNum == MyNumber));
  else
    ;//{ printf("%d:%d Used Close Ball intercept\n",MyNumber,CurrentTime.t);}
} 

/*****************************************************************************************/

PlayerInterceptInfo* ActionInfo::GetPlayerIntInfo(char side, Unum num)
{
  if (side == MySide)
    return TeamIntInfo[num];
  else if (side == TheirSide)
    return OppIntInfo[num];
  else
    my_error("bad side passed to GetPlayerIntInfo");
  return NULL;
}

/*****************************************************************************************/

PlayerInterceptInfo* ActionInfo::VerifyIntInfo(char side, Unum num, float dash_pow)
{
  PlayerInterceptInfo* pInfo = GetPlayerIntInfo(side, num);
  if (pInfo == NULL) {
    my_error("Bad side or number passed to VerifyIntInfo");
    return NULL;
  }

  int lookahead;
  switch (InterceptLookahead) {
  case LA_Default: lookahead = CP_max_int_lookahead; break;
  case LA_BestSoFar:
    lookahead =
      (IntMinCycTime == CurrentTime) ? (IntMinCyc) : CP_max_int_lookahead;
    break;
  default: lookahead = InterceptLookahead; break;
    break;
  }
  if ( pInfo->time != CurrentTime || fabs((pInfo->dash_pow-dash_pow))>FLOAT_EPS ||
       (pInfo->lookahead < lookahead && !IsSuccessRes(pInfo->res)) ) {
    /* set the info struct */
    DebugInt(printf("%d %d Data not current. Calling interception code\n", MyNumber, num));

    if (pInfo->time == CurrentTime && (pInfo->dash_pow-dash_pow)<=FLOAT_EPS &&
	(side != MySide || num != MyNumber))
      my_error("Recomputing %c %d because lookahead got bigger; old: %d\tnew: %d",
	       side,num,pInfo->lookahead, lookahead);
    
    /* let's do a real quick estimate to see if the player can make it there
       if player dist to ball > max ball dist will travel + max_dist we'll
       travel, then there's no way to get there */
    if (!PlayerPositionValid(side, num)) {      
      my_error("VerifyIntInfo: Can't give an answer if I don't know where player is");
      pInfo->res = BI_Invalid;
      return pInfo;
    }
    DebugInt(printf("Lookahead: %d\n", lookahead));
    float ball_travel = SumGeomSeries((BallVelocityValid() ? BallSpeed() : 0),
				      SP_ball_decay, lookahead);
    float player_travel = SP_player_speed_max * lookahead;
    float play_ball_dist = (PlayerAbsolutePosition(side, num) -
			    BallAbsolutePosition()).mod() ;
    if (play_ball_dist > player_travel + ball_travel) {
      pInfo->time = CurrentTime;
      pInfo->dash_pow = dash_pow;
      pInfo->dash_pow_to_use = dash_pow;
      pInfo->lookahead = lookahead;
      pInfo->res = BI_Failure;
      DebugInt(printf("Interception: %d, %d Took shortcut to decide failure\n", MyNumber, num));
    } else {
      DebugInt(printf("Interception: %d, %d About to do actual calculation\n", MyNumber, num));
      BallIntercept_active( dash_pow, lookahead, side, num, pInfo);
      if (IsSuccessRes(pInfo->res))
	SetIntMinCyc(pInfo->numCyc);
      pInfo->time = CurrentTime;
      pInfo->dash_pow = dash_pow;
      pInfo->lookahead = lookahead;
    }    
  }
  else if ( IsSuccessRes(pInfo->res) )
    SetIntMinCyc(pInfo->numCyc);

  return pInfo;
}

/*****************************************************************************************/


InterceptRes ActionInfo::PlayerInterceptionResult(char side, Unum num,
				      float dash_pow)
{
  return (VerifyIntInfo(side, num, dash_pow))->res;
}

/*****************************************************************************************/

Bool ActionInfo::PlayerInterceptionAble(char side, Unum num, float dash_pow)
{
  return IsSuccessRes((VerifyIntInfo(side, num, dash_pow))->res) ? TRUE : FALSE;
}

/*****************************************************************************************/

int ActionInfo::PlayerInterceptionNumberCycles(char side, Unum num,
				   float dash_pow)
{
  PlayerInterceptInfo* pInfo = VerifyIntInfo(side, num, dash_pow);
  if (!IsSuccessRes(pInfo->res))
    my_error("Trying to get number of cycles on invalid result: %c%d %d",
	     side, num, pInfo->res);
  return pInfo->numCyc;
}

/*****************************************************************************************/

Vector ActionInfo::PlayerInterceptionPoint(char side, Unum num,
			       float dash_pow)
{
  PlayerInterceptInfo* pInfo = VerifyIntInfo(side, num, dash_pow);
  if (!IsSuccessRes(pInfo->res))
    my_error("Trying to get interception point on invalid result: %c%d %d", 
	     side, num, pInfo->res);
  return pInfo->pos;  
}

/*****************************************************************************************/

float ActionInfo::PlayerInterceptionDashPower(char side, Unum num, float dash_pow)
{
  PlayerInterceptInfo* pInfo = VerifyIntInfo(side, num, dash_pow);
  if (!IsSuccessRes(pInfo->res))
    my_error("Trying to get interception dash power on invalid result: %c%d %d", 
	     side, num, pInfo->res);
  return pInfo->dash_pow_to_use;  
}


/*****************************************************************************************/

int ActionInfo::GetInterceptionMinCyc()
{
  if (IntMinCycTime != CurrentTime)
    return -1;
  else
    return IntMinCyc;
}

/*****************************************************************************************/

void ActionInfo::SetIntMinCyc(int newval)
{
  if (IntMinCycTime != CurrentTime) {
    IntMinCycTime = CurrentTime;
    IntMinCyc = newval;
  } else if (IntMinCyc > newval)
    IntMinCyc = newval;
}

/*****************************************************************************************/

void ActionInfo::SetInterceptionLookahead(int newval)
{
  if (newval > 0 || newval == LA_Default || newval == LA_BestSoFar) {
    if (IntMinCycTime == CurrentTime) 
      DebugInt(cout << "Changing lookahead mid way through computations. Could be bad" <<endl);
    InterceptLookahead = newval;
  } else {
    my_error("Trying to set InterceptLookahead to an invlaid value");
  }
  
}




/*****************************************************************************************/
/*****************************************************************************************/
/*****************************************************************************************/
/* Passive interception stuff */

int ActionInfo::GetClosestPointToBallPath(Vector* pPt, float* pNumCycles,
				       Vector PlayerPos, Vector BallPos,
				       Vector BallVel)
{
  if (fabs(BallVel.x) < FLOAT_EPS && fabs(BallVel.y) < FLOAT_EPS) {
    *pPt = BallPos;
    *pNumCycles = 0;
    return 1;
  }

  Ray rBallPath(BallPos, BallVel);;
  
  *pPt = rBallPath.GetClosestPoint(PlayerPos);
  
  /* adjust point for sidelines */
  Rectangle field(Vector(0,0), Vector(SP_pitch_length, SP_pitch_width));
  *pPt = AdjustPtToRectOnLine(*pPt, field, LineFromRay(rBallPath));

  /* Now let's reason about how far off we will be if we favor not turning */
  Vector no_turn_pt;
  if (rBallPath.intersection(Ray(MyPos(), MyBodyAng()), &no_turn_pt)) {
    if (no_turn_pt.dist(*pPt) < CP_no_turn_max_dist_diff) {
      LogAction6(110, "BPI: using no turn interception, old: (%.1f, %.1f) new: (%.1f, %.1f)",
		 pPt->x, pPt->y, no_turn_pt.x, no_turn_pt.y);
      *pPt = no_turn_pt;
    }
  }
  
  /* compute the number of cycles to get here */
  *pNumCycles = 0;

  /* now get the number of cycles */
  Vector traj = *pPt - BallPos;
  DebugInt(cout << "Pt: " << *pPt << "\tBallVel: " << BallVel
	   << "\tBallPos: " << BallPos << "\ttraj: " << traj << endl);
  /* first decide if the ball is actually coming towards us */
  if (signf(traj.x) != signf(BallVel.x) ||
      signf(traj.y) != signf(BallVel.y)) {
    DebugInt(printf("  GCPTBP: Ball is goign wrong way for closest intercept!\n"));
    return 0;
  }

  float trajDist = traj.mod();
  float velMod = BallVel.mod();
  float temp = trajDist / velMod * (SP_ball_decay - 1) + 1;
  if (temp < 0.0) {
    /* ball will never make it to closest point */
    /* SMURF - shoudl adjust for actual closest!!!! */
    DebugInt(printf("GCPTBP: Ball will never make it to closest point, adjusting\n"));
    *pPt = BallPos + traj * SumInfGeomSeries(velMod, SP_ball_decay) / traj.mod();
    *pNumCycles = SP_half_time; //just a big number
    return 1; 
  } else
    *pNumCycles = log(temp) / log(SP_ball_decay);

  return 1;
}

/*****************************************************************************************/

void ActionInfo::VerifyBPIInfo()
{
  if (BPItime == CurrentTime)
    return;

  BPItime = CurrentTime;
  
  Vector BallVel;

  if (!MyConf()) {
    my_error("Can't intercept if I don't know where I am");
    BPIvalid = FALSE;
    return;
  }

  if (!BallPositionValid()) {
    my_error("Can't get to ball path if I don't know where it is");
    BPIvalid = FALSE;
    return;
  }
  
  if (BallKickable()) {
    BPIvalid = TRUE;
    BPIable = TRUE;
    BPIdist = 0;
    BPIpoint = MyPos();
    BPIballcyc = 0;
    return;
  }

  if (BallVelocityValid())
    BallVel = BallAbsoluteVelocity();
  else {
    BPIvalid = TRUE;
    BPIable = TRUE;
    BPIdist = BallDistance();
    BPIpoint = BallAbsolutePosition();
    BPIballcyc = 0;
    return;
  }
      
  DebugInt(printf("\nTime: %d\n", CurrentTime.t));
  DebugInt(printf("At BallIntercept_passive\n"));

  int passRet;
  passRet = GetClosestPointToBallPath(&BPIpoint, &BPIballcyc, MyPos(),
				      BallAbsolutePosition(), BallVel);
  DebugInt(printf("Passive Method: ret: %d\tx: %f\ty:%f\tcyc: %f\n",
		  passRet, BPIpoint.x, BPIpoint.y, BPIballcyc));
  if (passRet) {
    BPIvalid = TRUE;
    BPIable = TRUE;
    BPIdist = (BPIpoint - MyPos()).mod();
  } else {
    BPIvalid = TRUE;
    BPIable = FALSE;
  }

  return;
  
}

/*****************************************************************************************/

Vector ActionInfo::BallPathInterceptPoint()
{
  VerifyBPIInfo();
  if (!BPIvalid)
    my_error("Calling BallPathInterceptionPoint when info not valid?");
  return BPIpoint;
}

/*****************************************************************************************/

Bool ActionInfo::BallPathInterceptAmIThere(float buffer)
{
  VerifyBPIInfo();
  if (!BPIvalid)
    my_error("Calling BallPathInterceptionAmIThere when info not valid");
  return (BPIable && (MyPos() - BPIpoint).mod() <= buffer) ? TRUE : FALSE;
}

/*****************************************************************************************/

float ActionInfo::BallPathInterceptDistance()
{
  VerifyBPIInfo();
  if (!BPIable)
    my_error("Calling BallPathInterceptionDistance when I can't get get there");
  return BPIdist;
}

/*****************************************************************************************/

int ActionInfo::BallPathInterceptCyclesForBall()
{
  VerifyBPIInfo();
  if (!BPIable)
    my_error("Calling BallPathInterceptionCyclesForBall when I can't get get there");
  return (int)ceil(BPIballcyc);
}

/*****************************************************************************************/

Bool ActionInfo::BallPathInterceptCanIGetThere(float max_pow)
{
  VerifyBPIInfo();
  if (!BPIable)
    return FALSE;

  AngleDeg targAng = AngleToFromBody(BPIpoint);
  Vector myEnd;
  if (fabs(GetNormalizeAngleDeg(MyBodyAng() - targAng)) >
      CP_max_go_to_point_angle_err) {
    myEnd = MyPredictedPosition((int)ceil(BPIballcyc), max_pow);
  } else {
    myEnd = MyPredictedPositionWithTurn(targAng - MyBodyAng(),
					(int)ceil(BPIballcyc), max_pow);    
  }

  return ( (myEnd - MyPos()).mod() >= (BPIpoint - MyPos()).mod() ) ? TRUE : FALSE;
}




/*****************************************************************************************/
/*****************************************************************************************/
/*****************************************************************************************/

float ActionInfo::VelAtPt2VelAtFoot(Vector pt, float targ_vel_at_pt)
{
  if (targ_vel_at_pt < FLOAT_EPS) {
    return SolveForFirstTermInfGeomSeries(SP_ball_decay, (pt - MyPos()).mod() );
  } else {
    float ball_steps =
      SolveForLengthGeomSeries(targ_vel_at_pt, 1/SP_ball_decay,
			    (pt - MyPos()).mod() );
    return targ_vel_at_pt * pow(1/SP_ball_decay, ball_steps);
  }  
}

/*****************************************************************************************/

/* looks at closeest opponent or teamless player */
/* SMURF: the teamless part is a hack */
KickMode ActionInfo::BestKickModeAbs(AngleDeg abs_ang)
{
  Unum closest = ClosestOpponent();
  if (NumTeamlessPlayers() > 0) {
    Vector teamless_pos = ClosestTeamlessPlayerPosition();
    if (closest == Unum_Unknown ||
	DistanceTo(teamless_pos) < OpponentDistance(closest))
      closest = Unum_Teamless;
  }

  if (closest == Unum_Unknown)
    return KM_HardestKick;
  int cyc_to_steal = EstimatedCyclesToSteal(closest);
  float targ_ang = abs_ang + signf(GetNormalizeAngleDeg(BallAngleFromBody()-abs_ang)) *
    (90 + CP_hardest_kick_ball_ang);
  float ang_diff = GetNormalizeAngleDeg(BallAngleFromBody() - targ_ang);
  NormalizeAngleDeg(&ang_diff);  
  if (cyc_to_steal > fabs(ang_diff)/CP_time_for_full_rotation + CP_cycles_to_kick)
    return KM_HardestKick;
  //if (OpponentWithBall() != Unum_Unknown)
  if (cyc_to_steal <= 1)
    return KM_QuickestRelease;
  if (cyc_to_steal < CP_cycles_to_kick)
    return KM_Quickly;
  if (cyc_to_steal < CP_cycles_to_kick + 1) // time for a dash in KM_Hard
    return KM_Moderate;
  return KM_Hard;
}

/*****************************************************************************************/


/* returns estimated cycles for opponent to get the ball into his kickable
   area */
/* can handle Unum_Teamless SMURF: it's kind of a hack though */
int ActionInfo::EstimatedCyclesToSteal(Unum opp, Vector ball_pos)
{
  if (!BallKickable())
    my_error("EstimatedCyclesToSteal: shouldn't use this if the ball is not kickable");

  if (BallKickableForOpponent(opp)) {
    LogAction2(110, "EstimatedCyclesToSteal: already kickable for opponent");
    return 0;
  }
  
  Vector targ = ball_pos;
  Vector pos;
  int cyc;
  if (opp == Unum_Teamless) {
    if (NumTeamlessPlayers() < 1)
      my_error("EstimatedCyclesToSteal: can't estimate teamless if there aren't any");
    pos = ClosestTeamlessPlayerPosition();
    targ -= (ball_pos - pos).SetLength(SP_kickable_area);
    cyc = (int)ceil(targ.dist(pos) / SP_player_speed_max);
  } else {
    if (!OpponentPositionValid(opp))
      my_error("EstimateCyclesToSteal: can't estimate if I don;t know where opponent is");
    pos = OpponentAbsolutePosition(opp);
    targ -= (ball_pos - pos).SetLength(SP_kickable_area);
    cyc = OpponentPredictedCyclesToPoint(opp, targ);
  }  

  /* now decide if the player will have to dodge */
  if (!pos.ApproxEqual(targ)) {
    Line oppLine = LineFromTwoPoints(pos, targ);
    Vector dodge_pos = oppLine.ProjectPoint(MyPos());
    dodge_pos += (pos - dodge_pos).SetLength(2.0* SP_player_size);
    float dodge_dist = oppLine.dist(MyPos());
    if (dodge_dist < 2.0 * SP_player_size &&
	oppLine.InBetween(dodge_pos, pos, targ)) {
      /* need to take into account a dodge */
      cyc += 2; //have to turn twice
      if (dodge_dist > 2 * SP_player_size - SP_player_speed_max)
	cyc += 1; // one dash will dodge us
      else
	cyc += 2; // it takes two dashes to dodge us
    }
  }
  
  return cyc;  
}

#ifndef RELEASE_VERSION  
/*****************************************************************************************/

/* the idea is to look at the goalie's closest spot to the ball trajectory
   and figure out if he could get to there (cathcably get to there that is)
   before the ball gets there */
int ActionInfo::NumCyclesGoalieBeatsBall(Vector from_spot, float abs_shot_ang,
					 float shot_vel, Unum goalie)
{
  if (goalie == Unum_Unknown)
    goalie = ClosestOpponentTo(MarkerPosition(RM_Their_Goal));

  if (goalie == Unum_Unknown || !OpponentPositionValid(goalie)) {
    LogAction2(110, "CouldGoalieBLockShot: Goalie not there");
    return FALSE;
  }
  
  Line l;
  l.LineFromRay(from_spot, abs_shot_ang);

  Vector int_pt = l.ProjectPoint(OpponentAbsolutePosition(goalie));
  int_pt = AdjustPtToRectOnLine(int_pt, FieldRectangle, l);

  int goalie_cyc =
    OpponentPredictedCyclesToPoint(goalie, int_pt, SP_max_power, SP_catch_area_l);

  int ball_cyc =
    (int)(ceil(SolveForLengthGeomSeries(shot_vel, SP_ball_decay,
					int_pt.dist(from_spot))));

  if (ball_cyc < 0)
    return TRUE; /* ball will never get to shot target */
  
  ball_cyc += (int)CP_cycles_to_kick;
  ball_cyc--; /* the ball moves closer during the kick */
  
  LogAction4(110, "NumCyclesGoalieBeatsBall: goalie %d, ball %d", goalie_cyc, ball_cyc);
  
  return ball_cyc - goalie_cyc;
}
#endif // RELEASE_VERSION


/*****************************************************************************************/

/* this is not an exact function becuase we don't have a perfect mapping of
   ball speed/position to kick power.
   Basically, this function returns whether the ball will be further back but still
   kickable after a dash */
Bool ActionInfo::WillDashHelpKick(Vector pt, float dash_pow)
{
  if (!BallWillBeKickable(1, dash_pow, CP_kickable_buffer)) {
    LogAction2(130, "WillDashHelpKick: ball will not be kickable");
    return FALSE;
  }
  
  /* we're going to assume that a collision is bad.
     but depending on how the ball is actually moving that could be good */
  if (WillDashBeCollision(dash_pow, CP_collision_buffer)) {
    LogAction2(130, "WillDashHelpKick: collision");
    return FALSE;
  }

  /* if we're not facing genrally in the direction we want to kick it,
     dashing will probably not help */
  if (fabs(AngleToFromBody(pt)) > CP_max_dash_help_kick_angle) {
    LogAction2(130, "WillDashHelpKick: not facing");    
    return FALSE;
  }
  
  AngleDeg curr_ang = BallAngleFromBody() - AngleToFromBody(pt);
  NormalizeAngleDeg(&curr_ang);
  Vector my_pred_pos = MyPredictedPosition(1, dash_pow);
  AngleDeg pred_ang =
    (BallPredictedPosition() - my_pred_pos).dir() -
    (pt - my_pred_pos).dir();
  NormalizeAngleDeg(&pred_ang);

  LogAction4(130, "WillDashHelpKick: curr: %.1f  pred: %.1f", curr_ang, pred_ang);
  
  return (fabs(pred_ang) > fabs(curr_ang)) ? TRUE : FALSE;
}

#ifndef RELEASE_VERSION  

/*****************************************************************************************/

Unum ActionInfo::BreakawayPassTarget()
{
  return PassTarget(FALSE,PC_DT_Max,PF_Breakaway);
}

/*****************************************************************************************/

Vector ActionInfo::BreakawayApproachPoint()
{
  Vector approach_pt;
  approach_pt.x = CP_breakaway_approach_x;
  if (fabs(MyY()) < CP_breakaway_middle_buffer) {
    /* pick the less congested side */
    float left_cong  = Congestion(Vector(approach_pt.x, -CP_breakaway_approach_y));
    float right_cong = Congestion(Vector(approach_pt.x, CP_breakaway_approach_y));
    if (left_cong + CP_congestion_epsilon < right_cong) {
      LogAction4(130, "BreakawayApproachPoint: left less congested %.2f < %.2f",
		 left_cong, right_cong);
      approach_pt.y = -CP_breakaway_approach_y;
    } else if (right_cong + CP_congestion_epsilon < left_cong) {
      LogAction4(130, "BreakawayApproachPoint: right less congested %.2f < %.2f",
		 right_cong, left_cong);      
      approach_pt.y =  CP_breakaway_approach_y;
    } else {
      LogAction5(130, "BreakawayApproachPoint: cong similar %.2f ~= %.2f, going %.0f",
		 left_cong, right_cong, signf(MyY()));            
      approach_pt.y = signf(MyY()) * CP_breakaway_approach_y;
    }
    
  } else {
    LogAction3(130, "BreakawayApproachPoint: on side %d", (int)signf(MyY()));
    approach_pt.y = signf(MyY()) * CP_breakaway_approach_y;
  }
  
#ifdef NOT_USED  
  Unum oppGoalie = ClosestOpponentTo(MarkerPosition(RM_Their_Goal)); 
  LogAction3(205, "BreakawayApproachPoint: goalie is %d", oppGoalie);
  if (oppGoalie == Unum_Unknown ||
      !TheirPenaltyArea.IsWithin(OpponentAbsolutePosition(oppGoalie))) {
    LogAction2(120, "Breakaway approach point not adjusted");
    return approach_pt; //no adjustment
  }

  Vector goalie_desired_pos = TheirGoalieDesiredPosition();
  float dist_diff =
    goalie_desired_pos.dist(approach_pt) - 
    OpponentAbsolutePosition(oppGoalie).dist(approach_pt);

  dist_diff = Max(0, dist_diff);

  //approach_pt += (approach_pt - goalie_desired_pos).SetLength(dist_diff);
  approach_pt.x -= dist_diff;
  
  LogAction5(120, "BreakawayApproachPoint: adjust by %.2f to (%.2f, %2.f)",
	     dist_diff, approach_pt.x, approach_pt.y);
#endif
  
  return approach_pt;
}

/*****************************************************************************************/

int ActionInfo::BreakawayKickRunCycles()
{
  Vector approach_pt = BreakawayApproachPoint();

  /* pick the kick run cycles based on distance */
  float kick_run_cycles = 
    (MarkerDistance(RM_Their_Goal) -
     MarkerPosition(RM_Their_Goal).dist(approach_pt)) /
    ((SP_pitch_length/2) -
     MarkerPosition(RM_Their_Goal).dist(approach_pt)) *
    (CP_breakaway_kick_run_max_cycles - CP_breakaway_kick_run_min_cycles) +
    CP_breakaway_kick_run_min_cycles;

  /* Now reduce it based on opponent positions */
  int cyc_diff = CP_breakaway_kick_run_max_cycles - CP_breakaway_kick_run_min_cycles;
  for (int num=1; num <= SP_team_size; num++) {
    if (!OpponentPositionValid(num))
      continue;
    if (IsOpponentBehind(num, MarkerPosition(RM_Their_Goal)))
      continue; //they are behind us
    float reduce = cyc_diff - (OpponentDistance(num) - SP_kickable_area) *
      cyc_diff / (CP_breakaway_kick_run_worry_dist - SP_kickable_area);
    reduce = MinMax(0, reduce, cyc_diff);
    if (reduce < FLOAT_EPS)
      continue;
    kick_run_cycles -= reduce;
    LogAction4(130, "BreakawayKickRunCycles: opp %d caused me to reduce by %.2f",
	       num, reduce);
  }
  
  return (int)rint(kick_run_cycles);
}

/*****************************************************************************************/
/*****************************************************************************************/
/*****************************************************************************************/

Bool ActionInfo::HaveAShot()
{
  if ( HasAShot(MyNumber) ||
       (VP_use_DT && PassConfidence(MarkerPosition(RM_Their_Goal)) == 1) )
    return TRUE;
  return FALSE;

  /* could use DT or something */
}

/*****************************************************************************************/

Bool ActionInfo::HasAShot(Unum teammate)
{
  /* could use DT or something */
  if ( TeammateDistanceTo(teammate,MarkerPosition(RM_Their_Goal)) < CP_shot_distance && 
       (NumOpponentsInCone((SP_goal_width/2) / TeammateDistanceTo(teammate, MarkerPosition(RM_Their_Goal)),
			   MarkerPosition(RM_Their_Goal),
			   TeammateAbsolutePosition(teammate)) <= 1 ||
       !CouldGoalieBlockShotFromSpot(TeammateAbsolutePosition(teammate),
				     ShotTarget(TeammateAbsolutePosition(teammate)),
				     CP_shot_speed, 
				     CP_shot_goalie_react_buffer)) )   
    return TRUE;
  return FALSE;
}

/*****************************************************************************************/

Bool ActionInfo::HaveAGoodShot()
{
  if ( HasAGoodShot(MyNumber) ||
       (VP_use_DT && PassConfidence(MarkerPosition(RM_Their_Goal)) == 1) )
    return TRUE;
  return FALSE;

  /* could use DT or something */
}

/*****************************************************************************************/

Bool ActionInfo::HasAGoodShot(Unum teammate)
{
  /* could use DT or something */
  if ( (TeammateDistanceTo(teammate,MarkerPosition(RM_Their_Goal)) < CP_good_shot_distance && 
	NumOpponentsInCone((SP_goal_width/2) / TeammateDistanceTo(teammate, MarkerPosition(RM_Their_Goal)),
			   MarkerPosition(RM_Their_Goal),
			   TeammateAbsolutePosition(teammate)) <= 1 ) ||
       (TeammateDistanceTo(teammate,MarkerPosition(RM_Their_Goal)) < CP_shot_distance &&
	!CouldGoalieBlockShotFromSpot(TeammateAbsolutePosition(teammate),
				     ShotTarget(TeammateAbsolutePosition(teammate)),
				     CP_shot_speed, 
				     CP_good_shot_goalie_react_buffer)) )

       return TRUE;
  else return FALSE;
}

/*****************************************************************************************/

Bool ActionInfo::HaveAGreatShot()
{
  return (MarkerDistance(RM_Their_Goal) < CP_shot_distance &&
	  !CouldGoalieBlockShot(ShotTarget(), CP_shot_speed)) ? TRUE : FALSE;
}


/*****************************************************************************************/

Bool ActionInfo::HasAGreatShot(Unum teammate)
{
  if (!TeammatePositionValid(teammate))
    my_error("HasAGreatShot: teamamte %d position not valid", teammate);
  return (TeammateDistanceTo(teammate, MarkerPosition(RM_Their_Goal)) < CP_shot_distance &&
	  !CouldGoalieBlockShotFromSpot(TeammateAbsolutePosition(teammate),
					ShotTarget(teammate), CP_shot_speed))
    ? TRUE : FALSE;
}

/*****************************************************************************************/

Bool ActionInfo::HasABetterShot(Unum teammate)
{
  if (TeammateDistanceTo(teammate, MarkerPosition(RM_Their_Goal)) > CP_shot_distance)
    return FALSE;
    
  int teammate_shot_cycles =
    NumCyclesGoalieBeatsBall(TeammateAbsolutePosition(teammate),
			     ShotTarget(teammate), CP_shot_speed);
  int my_shot_cycles =
    NumCyclesGoalieBeatsBall(ShotTarget(), CP_shot_speed);

  return (teammate_shot_cycles <= 1 && //it's a great shot
	  my_shot_cycles - teammate_shot_cycles > CP_better_shot_cyc_diff)
    ? TRUE : FALSE;
}


/*****************************************************************************************/
/*****************************************************************************************/
/*****************************************************************************************/

KickMode ActionInfo::ShotMode()
{
  /* need to work on this too */
  return BestKickMode(MarkerAngleFromBody(RM_Their_Goal));
}

/*****************************************************************************************/

Vector ActionInfo::ShotTarget(Vector from_pos)
{
  Vector goal = MarkerPosition(RM_Their_Goal);

  Vector correction = Vector(0,SP_goal_width/2-2);
  Vector left_post  = goal-correction;
  Vector right_post = goal+correction;

  if ( !NumOpponentsInCone(SP_goal_width/from_pos.dist(goal),
			   goal, from_pos) ) {
    if (TheirGoalieArea.IsWithin(from_pos)) {
      Vector pt;
      pt.x = SP_pitch_length / 2;
      pt.y = MinMax(left_post.y, from_pos.y, right_post.y);
      LogAction4(110, "No goalie and close, shot targ is (%.2f, %.2f)", pt.x, pt.y);
      return pt;
    } else {
      LogAction2(110, "No goalie and far away, shot targ is middle");
      return goal;
    }
  }

  Line left_post_line  = LineFromTwoPoints(from_pos,left_post);
  Line right_post_line = LineFromTwoPoints(from_pos,right_post);

  Unum goalie = ClosestOpponentTo(goal);
  Vector goalie_position = OpponentAbsolutePosition(goalie);
  Vector left_int_pt = left_post_line.ProjectPoint(goalie_position);
  Vector right_int_pt = right_post_line.ProjectPoint(goalie_position);
  //we're using the regular at point buffer
  int left_cyc = OpponentPredictedCyclesToPoint(goalie, left_int_pt);
  int right_cyc = OpponentPredictedCyclesToPoint(goalie, right_int_pt);
  if (left_cyc > right_cyc) {
    LogAction4(210, "ShotTarget: to left post %d > %d", left_cyc, right_cyc);
    return left_post;
  } else if (right_cyc > left_cyc){
    LogAction4(210, "ShotTarget: to right post %d > %d", right_cyc, left_cyc);
    return right_post;
  } else {
    float left_dist2  = goalie_position.dist(left_int_pt);
    float right_dist2 = goalie_position.dist(right_int_pt);
  
    if ( left_dist2 <= right_dist2 ) {
      LogAction4(210, "ShotTarget: to right post (by dist) %.2f > %.2f",
		 right_dist2, left_dist2);
      return right_post;
    } else {
      LogAction4(210, "ShotTarget: to left post (by dist) %.2f > %.2f",
		 left_dist2, right_dist2);
      return left_post;
    }    
  }
}

/*****************************************************************************************/

Unum ActionInfo::PassTarget(Bool use_receiver_list, PassChoiceType choice_type, PassFilterType filter_type)
{
  /* This is a simple one --- get the top player on the preference list that's around */
  Pnum *PreferenceList;
  PreferenceList = new Pnum[MyCurrentFormationSize()];
  int NumOptions;
  if ( use_receiver_list )
    NumOptions = MyReceiverList(PreferenceList);
  else {
    NumOptions = MyCurrentFormationSize();
    for (int i=0; i<MyCurrentFormationSize(); i++) PreferenceList[i] = i+1;
  }

  Unum target = Unum_Unknown;
  Unum player;
  int j;
  float max_conf=-1,tmp,min_congestion=HUGE;

  //printf("%d : %d  ",MyNumber,CurrentTime.t);

  for ( int i=0; i<NumOptions; i++ ){
    if ( choice_type == PC_Random )
      j = int_random(NumOptions);
    else 
      j = i;

    if ( (player = PositionPlayer(PreferenceList[j])) == Pnum_Unknown ||
	 player == MyNumber || !TeammatePositionValid(player) ||
	 TeammateInOffsidePosition(player) ||
	 OwnPenaltyArea.IsWithin(TeammateAbsolutePosition(player)) || 
	 (TeammateX(player) < MyX() &&
	  TeammateX(player) - their_offside_line <= CP_back_pass_offside_buffer) )
      continue;

    /* only consider players closer to the goal, with less congestion, and with a shot */
    if ( filter_type == PF_GoalDist_Congestion_And_Shot ){
      if (TeammateCongestion(player,TRUE) + CP_congestion_epsilon >= MyCongestion() ||
	  (MarkerDistance2(RM_Their_Goal) <= TeammateDistance2To(player,MarkerPosition(RM_Their_Goal))) ||
	  !HasAShot(player) )
	continue;
    }

    /* only consider players closer to the goal and with less congestion */
    if ( filter_type == PF_GoalDist_And_Congestion ){
      if (TeammateCongestion(player,TRUE) + CP_congestion_epsilon >= MyCongestion() ||
	  (MarkerDistance2(RM_Their_Goal) <= TeammateDistance2To(player,MarkerPosition(RM_Their_Goal))) )
	continue;
    }

    /* only consider players closer to the goal or with less congestion (and far enough away) */
    if ( filter_type == PF_GoalDist_Or_Congestion ){
      if ((TeammateCongestion(player,TRUE) + CP_congestion_epsilon >= MyCongestion() || 
	   TeammateDistance(player) <= CP_min_less_congested_pass_dist) &&
	  (MarkerDistance2(RM_Their_Goal) <= TeammateDistance2To(player,MarkerPosition(RM_Their_Goal))) )
	continue;
    }

    /* only consider players closer to the endline, with less congestion, and with a shot */
    if ( filter_type == PF_XPos_Congestion_And_Shot ){
      if (TeammateCongestion(player,TRUE) + CP_congestion_epsilon >= MyCongestion() ||
	  (TeammateX(player) < MyX()) ||
	  !HasAShot(player) )
	continue;
    }

    /* only consider players closer to the endline and with less congestion */
    if ( filter_type == PF_XPos_And_Congestion ){
      if (TeammateCongestion(player,TRUE) + CP_congestion_epsilon >= MyCongestion() ||
	  (TeammateX(player) < MyX()))
	continue;
    }

    /* only consider players closer to the endline or with less congestion (and far enough away) */
    if ( filter_type == PF_XPos_Or_Congestion ){
      if ((TeammateCongestion(player,TRUE) + CP_congestion_epsilon >= MyCongestion() || 
	   TeammateDistance(player) <= CP_min_less_congested_pass_dist) &&
	  (TeammateX(player) < MyX()))
	continue;
    }

    /* only consider passing to a player that's got a good shot */
    if ( filter_type == PF_Breakaway ){
      if ( !HasAGreatShot(player) &&
	   (!HasAGoodShot(player) ||
	    TeammateDistance2To(player, MarkerPosition(RM_Their_Goal)) >= MarkerDistance2(RM_Their_Goal)) )
	continue;
    }
    
    /* only pass to a player with a better shot */
    if ( filter_type == PF_BetterShot ){
      if ( !HasABetterShot(player) )
	continue;
    }

    /* only pass to a player with a better shot */
    if ( filter_type == PF_No_Opponent_Near ){
      Unum opp = ClosestOpponentTo(TeammateAbsolutePosition(player));
      if ( OpponentAbsolutePosition(opp).dist(TeammateAbsolutePosition(player))
	   <= CP_back_pass_opponent_buffer )
	continue;
    }

    if ( choice_type == PC_DT_Max ){
      if ( !VP_use_DT ) my_error("Can't have choice_type DT_Max if not using DT");
      tmp = PassConfidence(player);
      if ( tmp > max_conf && tmp > 0 ){
	max_conf = tmp;
	  target = player;
      }
      //printf("%d:%.2f ",player,tmp);
    }
    else if ( choice_type == PC_DT_Thresh ){
      if ( !VP_use_DT ) my_error("Can't have choice_type DT_Thresh if not using DT");
      if ( PassConfidence(player) > 0 ){
	  target = player;
	  break;
      }
    }
    else if ( choice_type == PC_Congestion ){
      if ( (tmp=TeammateCongestion(player)) < min_congestion ){
	min_congestion = tmp;
	  target = player;
      }
    }
    else{ /* choice_type == PC_Fixed or random */
      target = player;
      break;
    }
  }

  //printf("\n");

  delete [] PreferenceList;
  return target;
}


/*****************************************************************************************/

Bool ActionInfo::NeedToClear()
{
  if ( !NumOpponentsWithin(10) ) return FALSE;

  if ( InOwnPenaltyArea() ) return TRUE;
  if ( MarkerDistance(RM_My_Goal) < 25 ) return TRUE;

  return FALSE;
}

/*****************************************************************************************/

//returns the expected success
float ActionInfo::EvaluateClearAngle(Vector end)
{
  Line l = LineFromTwoPoints(BallAbsolutePosition(), end);
  float exp_success = 1.0;
  for (Unum opp = 1; opp <= SP_team_size; opp++) {
    if (!OpponentPositionValid(opp))
      continue;
    Vector pt = l.ProjectPoint(OpponentAbsolutePosition(opp));
    if (!l.InBetween(pt, BallAbsolutePosition(), end))
      continue;
    float wid = pt.dist(OpponentAbsolutePosition(opp));
    float dist = BallAbsolutePosition().dist(pt);
    if (wid > dist * CP_clear_ball_cone_ratio)
      continue;
    exp_success *= wid / (dist * CP_clear_ball_cone_ratio);
  }
  return exp_success;
}


/*****************************************************************************************/

/* returns a relative to body ang */
AngleDeg ActionInfo::ClearTarget()
{
  AngleDeg ang;
  AngleDeg ang_max;
  float exp_utility_max = -1.0;

  for (ang = -90; ang <= 90; ang += CP_clear_ball_ang_step) {
    Vector end = BallAbsolutePosition() +
      Vector(CP_clear_ball_max_dist, ang*signf(BallY()));
    //float utility = Sin(ang * 1.5); //max utility at 60 degrees
    //max utility at 30 degrees, and 0 at -90
    float utility = .5*(Sin(1.5 * ang + 45) + 1); 
    float exp_success = EvaluateClearAngle(end);
    float exp_utility = utility * exp_success;
    if (exp_utility > exp_utility_max) {
      ang_max = ang;
      exp_utility_max = exp_utility;
    }
  }
  DebugClear(cout << "Clear Target: " << GetNormalizeAngleDeg(signf(BallY())*ang_max) << endl);
  return GetNormalizeAngleDeg(signf(BallY())*ang_max - MyBodyAng());
}

/*****************************************************************************************/

/* returns a relative to body ang */
AngleDeg ActionInfo::ClearTargetOffensive()
{
  AngleDeg ang;
  AngleDeg ang_max;
  float exp_utility_max = -1.0;

  float goal_ang = AngleToFromBody(MarkerPosition(RM_Their_Goal)) + MyBodyAng();
  float start_angle;
  float stop_angle;
  start_angle =
    Max(GetNormalizeAngleDeg(goal_ang-CP_clear_offensive_min_angle),
	GetNormalizeAngleDeg(AngleToFromBody(Vector(MyX()+CP_clear_offensive_min_horiz_dist,
						    -SP_pitch_width/2)) + MyBodyAng()));
  stop_angle =
    Min(GetNormalizeAngleDeg(goal_ang+CP_clear_offensive_min_angle),
	GetNormalizeAngleDeg(AngleToFromBody(Vector(MyX()+CP_clear_offensive_min_horiz_dist,
						    SP_pitch_width/2)) + MyBodyAng()));
  DebugClear(cout << "goal_ang: " << goal_ang << endl);
  DebugClear(cout << "start: " << start_angle << "\tstop: " << stop_angle << endl);
  for (ang = start_angle; ang <= stop_angle; ang += CP_clear_ball_ang_step) {
    Vector end = BallAbsolutePosition() + Vector(CP_clear_ball_max_dist, ang);
    //the utility is linear in the angle
    //the slope is determined by the Y value of the player
    float utility = -.5 / 120 * MyY() / (SP_pitch_width/2) * ang  + .75;
    DebugClear(cout << "ang: " << ang << "\tutility: " << utility << endl);
    float exp_success = EvaluateClearAngle(end);
    float exp_utility = utility * exp_success;
    if (exp_utility > exp_utility_max) {
      ang_max = ang;
      exp_utility_max = exp_utility;
    }
  }
  DebugClear(cout << "Clear Target Offensive: " << GetNormalizeAngleDeg(signf(BallY())*ang_max) << endl);
  return GetNormalizeAngleDeg(ang_max - MyBodyAng());
    
}  

/*****************************************************************************************/

/* SMURF: this is not right yet!!! */
Bool ActionInfo::CanWeDribbleAroundOpponent()
{
  return FALSE;
}

/*****************************************************************************************/

Bool ActionInfo::CanDribbleTowards(Vector pt)
{
  //if ( MySide=='r' ) return FALSE;

  Vector traj = pt - MyPos();
  traj *= CP_dribble_towards_length / traj.mod();
  return CanDribbleTo(MyPos() + traj);
}

/*****************************************************************************************/

Bool ActionInfo::CanDribbleTo(Vector pt)
{
  if ( !CanKeepBall() ) {
    LogAction4(140, "CanDribbleTo (%.2f, %.2f): can't keep ball",
	       pt.x, pt.y);
    return FALSE;
  }
  
  if ( NumOpponentsWithin(CP_dribble_circle_inner_rad,CP_dribble_circle_ang,
			  0,AngleToFromBody(pt)) > 0) {
    LogAction4(140, "CanDribbleTo (%.2f, %.2f): player in small half circle",
	       pt.x, pt.y);
    return FALSE;
  }

  /* only look at outer circle if facing the wrong way */
  if ( fabs(AngleToFromBody(pt)) > CP_max_go_to_point_angle_err &&
       NumOpponentsWithin(CP_dribble_circle_outer_rad,CP_dribble_circle_ang,
			  0,AngleToFromBody(pt)) > 0) {
    LogAction4(140, "CanDribbleTo (%.2f, %.2f): player in large half circle",
	       pt.x, pt.y);
    return FALSE;
  }

  int numCone = NumOpponentsInCone(CP_can_dribble_cone_ratio, pt);
  if ( numCone > 1 || OpponentWithBall() != Unum_Unknown ) {
    LogAction5(140, "CanDribbleTo (%.2f, %.2f): inCone: %.0f",
	       pt.x, pt.y, (float)numCone);
    return FALSE;
  } else {
    if (numCone == 0) {
      LogAction4(140, "CanDribbleTo (%.2f, %.2f): TRUE No players in cone",
		 pt.x, pt.y);
      return TRUE;
    }
    if (CanWeDribbleAroundOpponent() == TRUE &&
	numCone + NumOpponentsWithin(SP_feel_distance) <= 1) {
      LogAction4(140, "CanDribbleTo (%.2f, %.2f): TRUE",
		 pt.x, pt.y);
      return TRUE;
    } else {
      LogAction6(140, "CanDribbleTo (%.2f, %.2f): FALSE %.0f %.0f",
		 pt.x, pt.y, numCone, NumOpponentsWithin(SP_feel_distance));
      return FALSE;
    }
  }
}

/*****************************************************************************************/

Bool ActionInfo::CanDribble()
{
  /* check targets in this order:
     their goal; straight; corner on my side; corner on other side; 
     straight to baseline;
     sideline up 20 on my side; sideline up 20 on other side. 
     Need to do my side first to avoid oscillation at center */

  if ( DribbleTargetTime == CurrentTime )
    return DribbleTargetAvailable;

  Fside my_side = MyLocationSide();
  int max_candidates = 6;
  Vector CandidateTarget[max_candidates];

  int c=0;

  CandidateTarget[c++] = MarkerPosition(RM_Their_Goal);
  CandidateTarget[c++] = MarkerPosition(my_side == FS_Right ? RM_RF_Flag : RM_LF_Flag);
  CandidateTarget[c++] = MarkerPosition(my_side == FS_Right ? RM_LF_Flag : RM_RF_Flag);
  CandidateTarget[c++] = Vector(MarkerX(RM_Their_Goal),MyY()); /* straight ahead */

  float target_x = MyX()+20;
  float target_y = MarkerY( my_side == FS_Right ? RM_RF_Flag : RM_LF_Flag );
  CandidateTarget[c++] = Vector(target_x,target_y);
  CandidateTarget[c++] = Vector(target_x,-target_y);

  if ( c > max_candidates ) my_error("need bigger candidate array");
  
  for (int i=0; i<c; i++){
    Vector target = CandidateTarget[i];
    if ( MyX() < CP_dont_dribble_to_middle_min_x ){
      /* when in defensive end, don't dribble to middle */
      if ( MyY() < OwnPenaltyArea.TopY() && target.y > OwnPenaltyArea.TopY() )
	continue;
      else if ( MyY() > OwnPenaltyArea.BottomY() && target.y < OwnPenaltyArea.BottomY() )
	continue;
    }
    if ( CanDribbleTowards(target) ){
      DribbleTargetAvailable = TRUE;
      DribbleTargetTime = CurrentTime;
      DribbleTargetVector = target;
      //printf("d%d",i); fflush(stdout);
      return DribbleTargetAvailable;
    }
  }

  DribbleTargetAvailable = FALSE;
  DribbleTargetTime = CurrentTime;
  return DribbleTargetAvailable;
}

/*****************************************************************************************/

Vector ActionInfo::DribbleTarget()
{
  if ( DribbleTargetTime == CurrentTime ){
    if ( DribbleTargetAvailable )
      return DribbleTargetVector;
    else 
      my_error("can't dribble");
  }
  else if ( CanDribble() )
    return DribbleTargetVector;
  
  my_error("can't dribble (2)");
  return 0;
}

/*****************************************************************************************/

Bool ActionInfo::CanKeepBall()
{
  Unum* sortPlayers = new Unum[SP_team_size+1];
  float desired_ang = BallAngleFromBody() + MyBodyAng();
  Bool desired_ang_set = FALSE;
  
  int maxPlay = SortPlayersByDistanceToPoint('t', MyPos(), sortPlayers);
  /* SMURF: this is a hack */
  /* now we're goign to mess with sortPlayers to add Unum_Teamless */
  Vector teamless_pos;
  if (NumTeamlessPlayers() > 0) {
    Unum prev = Unum_Teamless;
    teamless_pos   = ClosestTeamlessPlayerPosition();
    float teamless_dist   = DistanceTo(teamless_pos);
    for (int i=0; i < maxPlay; i++) {
      if (OpponentDistance(sortPlayers[i]) > teamless_dist) {
	Unum temp = sortPlayers[i];
	sortPlayers[i] = prev;
	prev = temp;
      }
    }
    sortPlayers[maxPlay] = prev;
    maxPlay++;
  }
  /* now we're ready to use this thing */
  for (int i=0; i < maxPlay; i++) {
    if (BallKickableForOpponent(sortPlayers[i])) {
      delete [] sortPlayers;
      return FALSE;
    }
    if (EstimatedCyclesToSteal(sortPlayers[i], desired_ang) <=
	CP_cycles_to_kick + CP_can_keep_ball_cycle_buffer) {
      /* player can get to the ball */
      if (desired_ang_set) {
	delete [] sortPlayers;
	return FALSE; //too many players too close
      }
      else { 
        float opp_ang = 
	  (sortPlayers[i] == Unum_Teamless) ? 
	  AngleToFromBody(teamless_pos) : OpponentAngleFromBody(sortPlayers[i]); 
        desired_ang = GetNormalizeAngleDeg(opp_ang + MyBodyAng() + 180); 
        desired_ang_set = TRUE; 
      }       
    }
  }

  delete [] sortPlayers;
  return TRUE;
  
  /*  my_error("CanKeepBall: how did I get here?");
  return FALSE;*/
}

/*****************************************************************************************/

Vector ActionInfo::OverrunBlockPos(Vector block_targ, Unum opponent, Line TrajLine)
{
  Vector oppToTarg = block_targ - OpponentAbsolutePosition(opponent);
  Line l = LineFromTwoPoints(OpponentAbsolutePosition(opponent), block_targ);
  oppToTarg *=
    (CP_def_block_dist + CP_def_block_dist_ratio * l.dist(MyPos()))
    / oppToTarg.mod();
  //perpindicular displacement for overrunning
  Vector disp =  oppToTarg +
    (oppToTarg * (CP_overrun_buffer / oppToTarg.mod()))
      .rotate((TrajLine.HalfPlaneTest(MyPos()) ? -1.0 : 1.0) * 90);
  return OpponentAbsolutePosition(opponent) + oppToTarg + disp;
  /* SMURF: we use oppToTarg twice inteh calculation! Not sure if that is correct,
     but we're leavingit like it is for now */
}

/*****************************************************************************************/
Bool ActionInfo::ShouldICross()
{
  if (MarkerDistance(RM_LF_Flag) < CP_should_cross_corner_dist ||
      MarkerDistance(RM_RF_Flag) < CP_should_cross_corner_dist)
    return TRUE;
  if (FieldRectangle.RightEdge().dist(MyPos()) <= CP_should_cross_baseline_buffer)
    return TRUE;
  return FALSE;
}

/*****************************************************************************************/
Bool ActionInfo::ShouldIMoveToCross()
{
  return (MarkerDistance(RM_LF_Flag) < CP_should_move_to_cross_corner_dist ||
	  MarkerDistance(RM_RF_Flag) < CP_should_move_to_cross_corner_dist)
    ? TRUE : FALSE;
}

#endif // RELEASE_VERSION

/*****************************************************************************************/
/*****************************************************************************************/
/*****************************************************************************************/

Bool ActionInfo::KickInProgress()
{
  /* need to have kicked last cycle.  Updates kick_in_progress_time */
  if ( kick_in_progress && kick_in_progress_time == LastActionOpTime ){
    kick_in_progress_time = CurrentTime;
    return TRUE;
  }
  return FALSE;
}

/*****************************************************************************************/

void ActionInfo::StartKick(AngleDeg target_angle, KickMode mode, float target_vel, TurnDir rot)
{
  kick_in_progress = TRUE;
  start_kick_time = kick_in_progress_time = CurrentTime;
  kick_in_progress_abs_angle = GetNormalizeAngleDeg(target_angle + MyBodyAng());
  kick_in_progress_mode = mode;
  kick_in_progress_target_vel = target_vel;
  kick_in_progress_rotation = rot;
}

/*****************************************************************************************/

void ActionInfo::StartShot(AngleDeg target_angle, KickMode mode, TurnDir rot)
{
  StartKick(target_angle,mode,2*SP_ball_speed_max, rot);
}

/*****************************************************************************************/

void ActionInfo::StartPass(Unum target, float target_vel_at_dest, TurnDir rot)
{
  if ( target == Unum_Unknown || !TeammatePositionValid(target) ) my_error("can't start this pass");

  team_passer = MyNumber;
  team_receiver = target;
  team_pass_time = CurrentTime;

  float target_vel = VelAtPt2VelAtFoot(TeammateAbsolutePosition(target),target_vel_at_dest);
  StartKick(TeammateAngleFromBody(target),KM_Moderate,target_vel,rot);
}




/*****************************************************************************************/
/*****************************************************************************************/
/*****************************************************************************************/

/* No reasoning about players being tired yet:  
   if so, need to add dash_pow to the interception calls */

/* These functions are very computationally intensive */

/* the stored value does not include the goalie */
Unum ActionInfo::FastestTeammateToBall()
{
  if ( !BallPositionValid() ) my_error("Need to know ball position to know fastest to it\n");

  Unum closest = ClosestTeammateToBall();
  if ( !BallMoving() && !TeammateTired(closest) ) return closest;

  if ( CurrentTime == Stored_Fastest_Teammate_Time ) return Stored_Fastest_Teammate;
  
  ResetInterceptionMinCyc();
  SetInterceptionLookahead(LA_BestSoFar);

  Unum FastestPlayer = Unum_Unknown;
  int cycles, min_cycles = CP_max_int_lookahead+1;

  for (int i=1; i<=SP_team_size; i++){
    if ( TeammatePositionValid(i) && TeammateInterceptionAble(i) == TRUE &&
	 (cycles=TeammateInterceptionNumberCycles(i)) < min_cycles && 
	 (i != FP_goalie_number || CP_goalie) ){
      min_cycles = cycles;
      FastestPlayer = i;
    }
  }

  Stored_Fastest_Teammate = FastestPlayer;
  Stored_Fastest_Teammate_Time = CurrentTime;

  return Stored_Fastest_Teammate;
}

/*****************************************************************************************/

Unum ActionInfo::FastestOpponentToBall()
{
  if ( !BallPositionValid() ) my_error("Need to know ball position to know fastest to it\n");

  if ( !BallMoving() ) return ClosestOpponentToBall();

  if ( CurrentTime == Stored_Fastest_Opponent_Time ) return Stored_Fastest_Opponent;

  ResetInterceptionMinCyc();
  SetInterceptionLookahead(LA_BestSoFar);

  Unum FastestPlayer = Unum_Unknown;
  int cycles, min_cycles = CP_max_int_lookahead+1;
  for (int i=1; i<=SP_team_size; i++){
    if ( OpponentPositionValid(i) && OpponentInterceptionAble(i) == TRUE &&
	 (cycles=OpponentInterceptionNumberCycles(i)) < min_cycles ){
      min_cycles = cycles;
      FastestPlayer = i;
    }
  }

  Stored_Fastest_Opponent = FastestPlayer;
  Stored_Fastest_Opponent_Time = CurrentTime;

  return FastestPlayer;
}

/*****************************************************************************************/
Unum ActionInfo::BallPossessor(){

  if (!BallPositionValid()) {
    //my_error("BallPossesor: ball position not valid");
    return Unum_Unknown;
  }

  Unum num_with_ball = PlayerWithBall();
  if (num_with_ball != Unum_Unknown)
    return num_with_ball;

  Unum fastestTeammate, fastestOpponent;
  
  if (BallMoving()) {
    int teamCycles, oppCycles;
    fastestOpponent = FastestOpponentToBall();
    fastestTeammate = FastestTeammateToBall(); 
  
    if (fastestTeammate == Unum_Unknown ||
	fastestOpponent == Unum_Unknown) 
      return (fastestTeammate == Unum_Unknown ? -fastestOpponent : fastestTeammate);

    teamCycles = TeammateInterceptionNumberCycles(fastestTeammate); 
    oppCycles = OpponentInterceptionNumberCycles(fastestOpponent); 

    if (teamCycles + CP_possessor_intercept_space < oppCycles)
      return fastestTeammate;
    else if (oppCycles + CP_possessor_intercept_space < teamCycles)
      return -fastestOpponent;
  } else {
    fastestTeammate = ClosestTeammateToBall();
    fastestOpponent = ClosestOpponentToBall();

    if (fastestTeammate == Unum_Unknown ||
	fastestOpponent == Unum_Unknown) 
      return (fastestTeammate == Unum_Unknown ? -fastestOpponent : fastestTeammate);

    /* we'll just ignore facing angles because they probably aren't right anwyay */;
    if (TeammateAbsolutePosition(fastestTeammate).dist(BallAbsolutePosition()) <
	OpponentAbsolutePosition(fastestOpponent).dist(BallAbsolutePosition()))
      return fastestTeammate;
    else
      return -fastestOpponent;
  }

  return Unum_Unknown;
}

 
/*****************************************************************************************/

char ActionInfo::TeamInPossession()
{
  switch ( PlayMode ){
  case PM_Play_On: break;
  case PM_My_Kick_In:
  case PM_My_Corner_Kick:
  case PM_My_Kick_Off:
  case PM_My_Free_Kick:
  case PM_My_Goalie_Free_Kick:
  case PM_My_Offside_Kick:
  case PM_My_Goal_Kick: return MySide;
  case PM_Their_Kick_In:
  case PM_Their_Corner_Kick:
  case PM_Their_Goal_Kick:
  case PM_Their_Kick_Off:
  case PM_Their_Offside_Kick:
  case PM_Their_Free_Kick: 
  case PM_Their_Goalie_Free_Kick: return TheirSide;
  default: break;
  }

  Unum player = BallPossessor();
  if ( player > 0 ) return MySide;
  else if (player < 0) return TheirSide;
  else return '?';
}

#ifndef RELEASE_VERSION  
/*****************************************************************************************/

Unum ActionInfo::TheirBreakaway()
{
  if (!BallPositionValid())
    my_error("breakaway: lost ball");

  Unum opp_ball = OpponentWithBall(-CP_their_breakaway_front_kickable_buffer);

  if (opp_ball == Unum_Unknown)
    return Unum_Unknown;

  /* now we want to make sure that the ball is generally in front of the player */
  if (OpponentDistanceToBall(opp_ball) > CP_their_breakaway_back_kickable_buffer) {
    /* make sure that the ball is in front of the player */
    Vector opp_pos = OpponentAbsolutePosition(opp_ball);
    if (fabs(GetNormalizeAngleDeg((MarkerPosition(RM_My_Goal) - opp_pos).dir() -
				  (BallAbsolutePosition() - opp_pos).dir()))
	> 90) {
      LogAction2(200,"TheirBreakaway: ball is too far away and not in front");
      return FALSE;
    } 
  } 
  
  
  /*
  if ( FastestTeammateToBall() == MyNumber && 
       MyInterceptionAble() && OpponentInterceptionAble(opp_ball) &&
       MyInterceptionNumberCycles() <= OpponentInterceptionNumberCycles(opp_ball) ){
    LogAction2(200,"I can get to ball as quickly as opponent, so not a breakaway");
    return Unum_Unknown;
  }
  */
  
  Unum team_ball = TeammateWithBall(-CP_their_breakaway_back_kickable_buffer);

  float wid_dist_ratio = TheirBreakawayConeRatio(opp_ball);

  int num_team = NumTeammatesInCone( wid_dist_ratio,
				     MarkerPosition(RM_My_Goal),
				     OpponentAbsolutePosition(opp_ball) );
  return (opp_ball != Unum_Unknown &&
	  team_ball == Unum_Unknown &&
	  num_team <= 1)
    ? opp_ball : Unum_Unknown;

#ifdef OLD_CODE	
  Unum opp_ball = OpponentWithBall(-CP_breakaway_kickable_buffer);

  return (opp_ball != Unum_Unknown &&
	  NumTeammatesWithin(OpponentDistanceTo(opp_ball,MarkerPosition(RM_My_Goal))+CP_breakaway_buffer,
			     MarkerPosition(RM_My_Goal)) <= 1 ) /* don't count goalie */
    ? TRUE : FALSE;
#endif	
}

/*****************************************************************************************/

Unum ActionInfo::OurBreakaway()
{
  if (!BallPositionValid())
    my_error("breakway: lost ball");

  Unum team_ball = TeammateWithBall(-CP_our_breakaway_kickable_buffer);

  if (team_ball == Unum_Unknown)
    return Unum_Unknown;

  Unum opp_ball = OpponentWithBall(-CP_our_breakaway_kickable_buffer);

  float wid_dist_ratio =
    (SP_goal_area_width/2) / Min(TeammateDistanceTo(team_ball, MarkerPosition(RM_Their_Goal)),
			     CP_our_breakaway_min_cone_dist_wid);
  int num_opp = NumOpponentsInCone( wid_dist_ratio,
				    MarkerPosition(RM_Their_Goal),
				    TeammateAbsolutePosition(team_ball));
  return (team_ball != Unum_Unknown &&
	  opp_ball == Unum_Unknown &&
	  num_opp <= 1)
    ? team_ball : Unum_Unknown;

#ifdef OLD_CODE	
  return (team_ball != Unum_Unknown && 
	  NumOpponentsWithin(TeammateDistanceTo(team_ball,MarkerPosition(RM_Their_Goal))+CP_breakaway_buffer,
			     MarkerPosition(RM_Their_Goal)) <= 1 ) /* don't count goalie */
    ? TRUE : FALSE;
#endif
  
}
#endif // RELEASE_VERSION

#ifndef RELEASE_VERSION  

/*****************************************************************************************/
/* this is just copied from our goalie.C file.
   It's used in breakaways */
Vector ActionInfo::TheirGoalieDesiredPosition()
{
  if (!BallPositionValid())
    my_error("GoalieDesiredPosition: need to know where ball is");

  Line l = LineFromTwoPoints(BallAbsolutePosition(),
			     MarkerPosition(RM_Their_Goal));
  float d = MinMax(CP_goalie_min_pos_dist,
		   MarkerPosition(RM_Their_Goal).dist(BallAbsolutePosition()),
		   CP_goalie_max_pos_dist);
  float scale = (d-CP_goalie_min_pos_dist)/
    (CP_goalie_max_pos_dist-CP_goalie_min_pos_dist);

  Unum opp = FurthestForwardOpponent(FALSE);
  
  Rectangle r(Min(SP_pitch_length/2 - CP_goalie_baseline_buffer,
		  Max(SP_pitch_length/2 - CP_goalie_baseline_buffer -
		      scale*(SP_penalty_area_length *
			     CP_goalie_max_forward_percent),
		      Max((opp != Unum_Unknown ? OpponentAbsolutePosition(opp).x : -100),
			  TeammateAbsolutePosition(FurthestForwardTeammate()).x) +
		      CP_goalie_baseline_buffer)),//left
	      SP_pitch_length/2, //right
	      MinMax(-SP_goal_area_width/2,
		     -scale*SP_goal_area_width/2,
		     -SP_goal_width / 2), //top
	      MinMax(SP_goal_width / 2, 
		     scale*SP_goal_area_width/2,
		     SP_goal_area_width/2) ); //bottom  

  scale = Min(CP_goalie_ball_ang_for_corner, fabs(l.angle()))/
    CP_goalie_ball_ang_for_corner;
  d = MinMax(CP_goalie_ball_dist_for_corner,
	     MarkerPosition(RM_Their_Goal).dist(BallAbsolutePosition()),
	     CP_goalie_ball_dist_for_center);
  scale *= (CP_goalie_ball_dist_for_center - d)/
    (CP_goalie_ball_dist_for_center-CP_goalie_ball_dist_for_corner);
  Vector guard_pt = MarkerPosition(RM_Their_Goal);
  guard_pt.x += CP_goalie_baseline_buffer;
  guard_pt.y += signf(BallAbsolutePosition().y -
		      MarkerPosition(RM_Their_Goal).y) *
    scale*SP_goal_width / 2;
  Vector pos =
    AdjustPtToRectOnLine(BallAbsolutePosition(), r,
			 LineFromTwoPoints(guard_pt, BallAbsolutePosition()));

  return pos;
}
#endif // RELEASE_VERSION
