/* -*- Mode: C++ -*- */

#ifndef _OBSERVE_H_
#define _OBSERVE_H_

#include <ostream.h>
#include "types.h"
#include "geometry.h"
#include "data.h"
#include "MemPosition.h"

/*********************************** Constants ****************************/

const int MaxBallControllers = 3;

const float MaxSimilarity = 1.0;
const float MinSimilarity = 0.0;
const float ZeroSimilarity = 0.0;


typedef enum {
  OT_None,
  OT_BallPos,
  OT_OppPos,
  OT_BallCont,
  OT_OppPass,
  OT_OppDrib
} ObserveType;

/*********************************** Types ****************************/
typedef struct 
{
  SideType side;
  Unum     num;
  Vector   pos;
  Time     time;
} PlayerID;

/********************************* Misc Utility Functions *****************/
Observation* GetNewObservationOfType(ObserveType obType, OptionInfo* pMemOpt);


/*********************************** Observation **************************/
class ObserveList;
class AdversaryType; /* defined in adv_class.[Ch] */

class Observation {
public:
  Observation()  { enabled = FALSE; }  
  virtual ~Observation() {}

  void SetEnabled(Bool val) { enabled = val; }

  virtual void Initialize(OptionInfo* MemOpt) = 0; 
  virtual void Reset() = 0;

  void Record(PositionInfo* pPIMem);
  void WriteInfoToFile(ostream& output_file);
  void WriteSummaryInfoToFile(ostream& output_file);

  virtual void NotifyOfPlayMode(Pmode playmode, Time CurrTime) {}; 

  virtual Bool ReadFromFile(istream& infile) = 0;  
  virtual Bool ReadDataFromFile(istream& infile) = 0;  

  float SimilarityTo(Observation* pObs);

  void CombineWith(Observation* pObs, float weight = 1.0);
  

protected:
  
  virtual float PrivSimilarityTo(Observation* pObs) = 0;
  virtual void PrivRecord(PositionInfo* pPIMem) = 0;
  virtual void PrivWriteInfoToFile(ostream& output_file) = 0;
  virtual void PrivWriteSummaryInfoToFile(ostream& output_file) = 0;
  virtual void PrivCombineWith(Observation* pObs, float weight) = 0;

  friend ObserveList;
  friend AdversaryType;  
  virtual ObserveType GetObserveType() { return OT_None; }  

private:
    Bool enabled; /* derived classes should not need to know about enabled */
  
};

inline void Observation::Record(PositionInfo* pPIMem)
{
  if (!enabled) return;
  PrivRecord(pPIMem);
}
inline void Observation::WriteInfoToFile(ostream& output_file)
{
  if (!enabled) return;
  PrivWriteInfoToFile(output_file);
}
inline void Observation::WriteSummaryInfoToFile(ostream& output_file)
{
  if (!enabled) return;
  PrivWriteSummaryInfoToFile(output_file);
}

/*********************************** ObserveList **************************/
class ObserveList 
{
 private:
  typedef struct OBSERVELISTITEM {
    struct OBSERVELISTITEM* pNext;
    Observation* pObs;
    float weight;
  } ObserveListItem;
  ObserveListItem* pHead;
  ObserveListItem* pIter;

 public:
  ObserveList()
    { pHead = pIter = NULL; }
  void ClearList();
  ~ObserveList() { ClearList(); }
  
  void Add(Observation* pObs, float weight = 1.0);
  void ResetIter()
    { pIter = pHead; }
  Observation* GetCurrentIter();
  Observation* PeekCurrentIter() 
    { return (pIter != NULL ? pIter->pObs : NULL); }      
  float PeekCurrentIterWeight() 
    { return (pIter != NULL ? pIter->weight : 0.0); }      
  Bool AdvanceCurrentIter()
    { return (pIter == NULL ? FALSE :
	      ((pIter = pIter->pNext) == NULL ? FALSE : TRUE)); }
  /* moves the list from the argument to this,
     destroys what is in this and leaves the argument empty */
  void Transfer(ObserveList* pol);
  
  /* the weights used are in the this pointer, not the polComp */
  float CompareTo(ObserveList* polComp);
};

/**************************** Derived Observation **************************/
/* NOTE: if you want observations to be usable as pieces of adversary classes,
   you have to add code to adv_class.c, AdversaryType::LoadFromFile in order
   to be able to load stuff from files */

/* SMURF: I'm not sure that these destructors are done correctly */

class BallPositionObs : public Observation {
 public:
  void Initialize(OptionInfo* MemOpt); 
  ~BallPositionObs();
  
  void Reset();
  void PrivRecord(PositionInfo* pPIMem);
  void PrivWriteInfoToFile(ostream& output_file);
  void PrivWriteSummaryInfoToFile(ostream& output_file);
  ObserveType GetObserveType() { return OT_BallPos; }

  static const char* FileHeaderString(){ return file_header; }
  Bool ReadFromFile(istream& infile);
  Bool ReadDataFromFile(istream& infile);
  float PrivSimilarityTo(Observation* pObs);
  void PrivCombineWith(Observation* pObs, float weight);
 private:
  RectGrid* prgBallPositions;
  SingleDataSummary sdsConf;
  float sim_M; /* squashing constant; bigger -> more similarity in general */
  static const char file_header[] = "******Ball Positions******";  
};

class OpponentPositionObs : public Observation {
 public:
  void Initialize(OptionInfo* MemOpt); 
  ~OpponentPositionObs();
  
  void Reset();
  void PrivRecord(PositionInfo* pPIMem);
  void PrivWriteInfoToFile(ostream& output_file);
  void PrivWriteSummaryInfoToFile(ostream& output_file);
  ObserveType GetObserveType() { return OT_OppPos; }

  static const char* FileHeaderString(){ return file_header; }
  Bool ReadFromFile(istream& infile);
  Bool ReadDataFromFile(istream& infile);
  float PrivSimilarityTo(Observation* pObs);
  void PrivCombineWith(Observation* pObs, float weight);
 private:
  int TeamSize;
  typedef struct {
    RectGrid rg;
    SingleDataSummary sdsConf;
  } StatPlayerInfo;
  StatPlayerInfo* aPlayerPositions;
  static const char file_header[] = "******Opponent Positions******";  
  float sim_M; /* squashing constant; bigger -> more similarity in general */

  void SortByAverageX(StatPlayerInfo* aInfo, int* sort_order, 
		      Bool use_zero = FALSE);
  
};

class BallControllerObs : public Observation 
{
public:
  void Initialize(OptionInfo* MemOpt); 
  ~BallControllerObs();
  
  void Reset();
  void PrivRecord(PositionInfo* pPIMem);
  void PrivWriteInfoToFile(ostream& output_file);
  void PrivWriteSummaryInfoToFile(ostream& output_file);
  ObserveType GetObserveType() { return OT_BallCont; }

  static const char* FileHeaderString(){ return file_header; }
  Bool ReadFromFile(istream& infile);
  Bool ReadDataFromFile(istream& infile);
  float PrivSimilarityTo(Observation* pObs);
  /* the weight is for the this object; the new object always has weight 1.0 */
  void PrivCombineWith(Observation* pObs, float weight);
private:
  int TeamSize;
  int* aControlCountOpp;
  int* aControlCountTeam;
  int ControlCountNone;
  static const char file_header[] = "******Ball Controllers******";  
};

class OpponentPassingObs : public Observation 
{
public:
  void Initialize(OptionInfo* MemOpt); 
  ~OpponentPassingObs();
  
  void Reset();
  void PrivRecord(PositionInfo* pPIMem);
  void PrivWriteInfoToFile(ostream& output_file);
  void PrivWriteSummaryInfoToFile(ostream& output_file);
  void NotifyOfPlayMode(Pmode playmode, Time CurrTime);
  ObserveType GetObserveType() { return OT_OppPass; }

  static const char* FileHeaderString(){ return file_header; }
  Bool ReadFromFile(istream& infile);  
  Bool ReadDataFromFile(istream& infile);
  float PrivSimilarityTo(Observation* pObs);
  void PrivCombineWith(Observation* pObs, float weight);
private:
  int OP_pass_num_past_cont;
  int OP_pass_timeout;
  PlayerID* aPastCont;
  RectGridMulti* prgmOppPassing;
  int numPasses;
  static const char file_header[] = "******Opponent Passing******";  

  void ClearPastCont(PlayerID apc[]);
};

class OpponentDribblingObs : public Observation 
{
public:
  void Initialize(OptionInfo* MemOpt); 
  ~OpponentDribblingObs();
  
  void Reset();
  void PrivRecord(PositionInfo* pPIMem);
  void PrivWriteInfoToFile(ostream& output_file);
  void PrivWriteSummaryInfoToFile(ostream& output_file);
  void NotifyOfPlayMode(Pmode playmode, Time CurrTime);
  ObserveType GetObserveType() { return OT_OppDrib; }

  static const char* FileHeaderString(){ return file_header; }
  Bool ReadFromFile(istream& infile);
  Bool ReadDataFromFile(istream& infile);
  float PrivSimilarityTo(Observation* pObs);
  void PrivCombineWith(Observation* pObs, float weight);
private:
  int OP_drib_num_past_cont;
  int OP_max_drib_cntl_intvl;
  float OP_min_drib_dist;

  typedef struct 
  {
    SideType side;
    Unum     num;
    Vector   orig_pos;
    Time     orig_time;
    Vector   last_cntl_pos;
    Time     last_cntl_time;
  } PlayerDribbleInfo;  
  PlayerDribbleInfo* aPastCont;
  
  RectGridMulti* prgmOppDribbling;
  int numDribbles;
  static const char file_header[] = "******Opponent Dribbling******";  

  void ClearPastCont(PlayerDribbleInfo apc[]);
  void PurgeControllerArray(Time CurrTime);
};




#endif



