/* -*- Mode: C++ -*- */
#include "observe.h"
#include "MemPosition.h"

#ifdef DEBUG_OUTPUT
/*#define DebugCont(x) {if (pPIMem->MySide == 'l' && pPIMem->MyNumber == 1) x;}*/
#define DebugCont(x) 
#else
#define DebugCont(x) 
#endif

/********************************* Misc Utility Functions *****************/
Observation* GetNewObservationOfType(ObserveType obType, OptionInfo* pMemOpt)
{
  Observation* pObs = NULL; 

  if (obType == OT_BallPos)
    pObs = new BallPositionObs; 
  else if (obType == OT_OppPos)
    pObs = new OpponentPositionObs; 
  else if (obType == OT_BallCont)
    pObs = new BallControllerObs; 
  else if (obType == OT_OppPass)
    pObs = new OpponentPassingObs; 
  else if (obType == OT_OppDrib)
    pObs = new OpponentDribblingObs; 
  else
    my_error("Unrecognized observation type: %d", obType);

  if (pObs != NULL) {  
    pObs->Initialize(pMemOpt);  
    pObs->SetEnabled(TRUE);    
  }
  
  

  return pObs;    
}


/*********************************** Observation **************************/
float Observation::SimilarityTo(Observation* pObs)
{
  if (GetObserveType() == pObs->GetObserveType())
    return PrivSimilarityTo(pObs);
  else
    return 0.0;
}

void Observation::CombineWith(Observation* pObs, float weight)
{
  if (GetObserveType() == pObs->GetObserveType())
    PrivCombineWith(pObs, weight);
  else
    my_error("Tried to combine observations of different types: %d %d", 
	     GetObserveType(), pObs->GetObserveType());
}


/*********************************** ObserveList **************************/
/* the observations will be stored in order of their observe types. If there 
   are multiple ones with the same ObserveType, the order is unspecified */
void ObserveList::ClearList()
{
  /*
  ObserveListItem *pTemp;
  for ( ; pHead != NULL; pHead = pTemp) {
    pTemp = pHead;
    delete pHead->pObs;
    delete pHead;
  } 
  */ 
}


void ObserveList::Add(Observation* pObs, float weight)
{
  ObserveListItem *pNewItem;
  pNewItem = new ObserveListItem;
  pNewItem->pObs = pObs;
  pNewItem->weight = weight;
  
  /* now do an insertion sort */
  ObserveListItem *pItem, *pPrev = NULL;
  for (pItem = pHead; pItem != NULL; pItem = pItem->pNext) {
    if (pNewItem->pObs->GetObserveType() < pItem->pObs->GetObserveType())
      break;
    pPrev = pItem;
  }

  pNewItem->pNext = pItem;
  if (pItem == pHead)
    pHead = pNewItem;

  if (pPrev != NULL)
    pPrev->pNext = pNewItem;
}

Observation* ObserveList::GetCurrentIter()
{
  if (pIter == NULL)
    return NULL;
  Observation* pTemp;
  
  pTemp = pIter->pObs;
  pIter = pIter->pNext;
  //printf("GetCurrIter: %d\n", pTemp->GetObserveType());
  return pTemp;
}

/* the weights used are in the this pointer, not the polComp */
float ObserveList::CompareTo(ObserveList* polComp)
{  
  float total_sim = ZeroSimilarity;
  /*  it's a float because we have to worry about weights */
  float num_comp = 0.0; 

  ObserveListItem *pThisItem = pHead;
  ObserveListItem *pCompItem = polComp->pHead;
  
  while(1) {
    if (pThisItem == NULL) break;
    if (pCompItem == NULL) break;
    
    if (pThisItem->pObs->GetObserveType() == 
	pCompItem->pObs->GetObserveType()) {
      /* they are the same type, so compare them */
      total_sim += pThisItem->weight * 
	pThisItem->pObs->SimilarityTo(pCompItem->pObs);
      num_comp += pThisItem->weight;
      /* now advance both of them */
      pThisItem = pThisItem->pNext;
      pCompItem = pCompItem->pNext;      
    } else if (pThisItem->pObs->GetObserveType() < 
	       pCompItem->pObs->GetObserveType()) {
      pThisItem = pThisItem->pNext;
    } else if (pThisItem->pObs->GetObserveType() >
	       pCompItem->pObs->GetObserveType()) {
      pCompItem = pCompItem->pNext;
    }
  } /* while loop */

  if (num_comp == 0.0)
    return total_sim;
  else {
    /* this normalizes the return to within MinSimilarity and MaxSimilarity */
    return total_sim / num_comp;
  }
}

void ObserveList::Transfer(ObserveList* pol)
{
  ClearList();
  pHead = pol->pHead;
  pIter = pol->pIter;
  pol->pHead = pol->pIter = NULL;
}


/***************************** Shared Functions *****************************/
void GetBallController(PositionInfo* pPIMem, SideType* aside, Unum* anum )
{
  int i;
  
  /* initialize */
  for (i=0; i<MaxBallControllers; i++) {
    aside[i] = ST_NoValue;
    anum[i] = Unum_Unknown;
  }

  if (!pPIMem->BallPositionValid()) {
    aside[0] = ST_Unknown;
    return;
    /* no point in looking at the players if we don't know where the ball is */
  }
  
  Unum players[MAX_PLAYERS];
  int num_players =
    pPIMem->SortPlayersByDistanceToPoint('b', pPIMem->BallAbsolutePosition(),
					 players);

  /* go through the sorted array and if they are close enough, mark them as
     a ball controller */
  for (i=0; i<MaxBallControllers; i++) {
    if (i >= num_players)
      break; /* no more players! */
    char side;
    Unum num;
    if (players[i] <= pPIMem->SP_team_size) {
      side = pPIMem->MySide;
      num = players[i];
    } else {
      side = pPIMem->TheirSide;
      num = players[i] - pPIMem->SP_team_size;
    }
    if (pPIMem->PlayerDistanceTo(side, num, pPIMem->BallAbsolutePosition()) >
	pPIMem->SP_kickable_area + pPIMem->CP_control_buffer)
      break; /* this player is too far away to be considered controlling */
    aside[i] = (side == pPIMem->MySide) ? ST_Ours : ST_Theirs;
    anum[i] = num;
  }

  if (i==0) {
    if (num_players == 0)
      /* there were no players in the list! */
      aside[0] = ST_Unknown;
    else
      aside[0] = ST_Neither;
  }
}

/* this is basically the same as scan conversion in computer graphics
   it's a little more complicated because the RectGridMulti has a more
   complicated relation from Vector coords to RectGrid coords */
void RecordTrajectory(Vector start, Vector end, RectGridMulti* prgm) 
{
  float dy = end.y - start.y;
  float dx = end.x - start.x;
  float m =  dy / dx;
  /* ang is actually the angle + 45 degrees */
  int ang = (((int)(end - start).dir()) + 45 + 360) % 360;
  /* the +45 is so that the case where num_vals is 4 makes more sense graphically */
  int which = ang / ((int)(360 / prgm->num_vals));
  DebugCont(printf("  ang+45: %d\twhich: %d\n", ang, which));
  
  if (-1.0 <= m && m <= 1.0) {
    /* slope is such that we do one increment of x each iteration */
    double xinc = Sign((end - start).x) * prgm->pitch_length / prgm->num_cols;
    double x = start.x;
    double y = start.y;
    for ( ; Sign(xinc)*x <= Sign(xinc)*end.x; x += xinc, y += m*xinc ) {
      prgm->IncrementRectForPoint(x, y, which);
    }
  } else {
    /* slope is such that we do one increment of y each iteration */
    double yinc = Sign((end - start).y) * prgm->pitch_width / prgm->num_rows;
    double x = start.x;
    double y = start.y;
    for ( ; Sign(yinc)*y <= Sign(yinc)*end.y; x += yinc / m, y += yinc ) {
      prgm->IncrementRectForPoint(x, y, which);
    }
  }
}


/***************************** BallPositionObs *****************************/
void BallPositionObs::Initialize(OptionInfo* MemOpt)
{
  //printf("Initializing Ball rg: %d, %d\n", r, c);
  prgBallPositions   = new RectGrid;
  prgBallPositions->Initialize(MemOpt->SP_pitch_length, MemOpt->SP_pitch_width,
				MemOpt->OP_rg_rows, MemOpt->OP_rg_cols);
  sim_M = 1.0; /* SMURF */  
}

BallPositionObs::~BallPositionObs()
{
  delete prgBallPositions;  
}


void BallPositionObs::Reset()
{
  prgBallPositions->Zero();
  sdsConf.Reset();
}

void BallPositionObs::PrivRecord(PositionInfo* pPIMem)
{
  if (!pPIMem->BallPositionValid())
    return;  
  prgBallPositions->IncrementRectForPoint(pPIMem->BallAbsolutePosition());
  sdsConf.AddPoint(pPIMem->BallPositionValid());
}

void BallPositionObs::PrivWriteInfoToFile(ostream& output_file)
{
  output_file << endl << file_header << endl;
  prgBallPositions->WriteToFile(output_file);
  output_file << "Conf: ";
  sdsConf.WriteInfoToFile(output_file);
}

void BallPositionObs::PrivWriteSummaryInfoToFile(ostream& output_file)
{
  output_file << endl << "******Ball Position Summary******" << endl;
  output_file << "Times I saw the ball: "
	      << (prgBallPositions->Sum());
  output_file << "\tConf Mean: " << sdsConf.GetMean() << endl;
}

Bool BallPositionObs::ReadFromFile(istream& infile)
{
  char line[100];  
  skip_white_space(infile);
  infile.getline(line, 100);
  if (!infile) return FALSE;  
  if (strncmp(line, file_header, strlen(file_header)) != 0)
    my_error("BallPositionObs::ReadFromFile: bad header, trying to continue");
  return ReadDataFromFile(infile);  
}

Bool BallPositionObs::ReadDataFromFile(istream& infile)
{
  char line[100];  
  if (!prgBallPositions->ReadFromFile(infile))
    return FALSE;
  skip_white_space(infile);  
  infile.getline(line, 100); /* skip the conf line: SMURF */
  return TRUE;
}

float BallPositionObs::PrivSimilarityTo(Observation* pObs)
{
  BallPositionObs* pComp = (BallPositionObs*) pObs;
  
  return prgBallPositions->SimilarityTo(pComp->prgBallPositions, sim_M);  
}

void BallPositionObs::PrivCombineWith(Observation* pObs, float weight)
{
  BallPositionObs* pComb = (BallPositionObs*) pObs;
  prgBallPositions->CombineWith(pComb->prgBallPositions, weight);
}


/*************************** OpponentPositionObs *****************************/
void OpponentPositionObs::Initialize(OptionInfo* MemOpt)
{
  TeamSize = MemOpt->SP_team_size;
  //printf("Initializing Opponent rg: %d, %d\n", r, c);
  aPlayerPositions = new StatPlayerInfo[TeamSize+1];
  for (int i=0; i<=TeamSize; i++)
    aPlayerPositions[i].rg.Initialize(MemOpt->SP_pitch_length,
				       MemOpt->SP_pitch_width,
				       MemOpt->OP_rg_rows, MemOpt->OP_rg_cols);
  sim_M = 1.0; /* SMURF */  
}
OpponentPositionObs::~OpponentPositionObs()
{
  delete [] aPlayerPositions;  
}


void OpponentPositionObs::Reset()
{
  for (int i = 0; i <= TeamSize; i++) {
    aPlayerPositions[i].rg.Zero();
    aPlayerPositions[i].sdsConf.Reset();
  }
}

void OpponentPositionObs::PrivRecord(PositionInfo* pPIMem)
{
  /* we're going to loop through all player objects we know about */
  for(int num = 0; num < pPIMem->num_their_players; num++) {
    if (!pPIMem->TheirPlayer[num] ||
	pPIMem->TheirPlayer[num]->pos_valid() == 0)
      continue;
    aPlayerPositions[pPIMem->TheirPlayer[num]->unum].rg.
      IncrementRectForPoint(pPIMem->TheirPlayer[num]->get_abs_pos());
    aPlayerPositions[pPIMem->TheirPlayer[num]->unum].sdsConf.
      AddPoint(pPIMem->TheirPlayer[num]->pos_valid());
  }  

  for(int num = 0; num < pPIMem->num_teamless_players; num++) {
    if (!pPIMem->TeamlessPlayer[num] ||
	pPIMem->TeamlessPlayer[num]->pos_valid() == 0)
      continue;
    aPlayerPositions[pPIMem->TeamlessPlayer[num]->unum].rg.
      IncrementRectForPoint(pPIMem->TeamlessPlayer[num]->get_abs_pos());
    aPlayerPositions[pPIMem->TeamlessPlayer[num]->unum].sdsConf.
      AddPoint(pPIMem->TeamlessPlayer[num]->pos_valid());
  }  
}

void OpponentPositionObs::PrivWriteInfoToFile(ostream& output_file)
{
  output_file << endl << file_header << endl;
  for (int num=0; num <= TeamSize; num++) {
    output_file << "*Opponent " << num << "*" << endl;
    aPlayerPositions[num].rg.WriteToFile(output_file);
    output_file << "Conf: ";
    aPlayerPositions[num].sdsConf.WriteInfoToFile(output_file);
    /*output_file << "averageX: " << aPlayerPositions[num].rg.AverageX() << endl;*/
    
  }
  /*int sort_order[20];
  SortByAverageX(aPlayerPositions, sort_order);
  for (int i=0; i<TeamSize; i++)
    output_file << sort_order[i] << " ";
    output_file << endl;*/
  
}

void OpponentPositionObs::PrivWriteSummaryInfoToFile(ostream& output_file)
{
  output_file << endl << "******Opponent Position Summary******" << endl;
  for (int num = 1; num <= TeamSize; num++) {
    output_file << "Times I saw opponent " << num << ": " 
		<< (aPlayerPositions[num].rg.Sum());
    output_file << "\tConf Mean: "
		<< aPlayerPositions[num].sdsConf.GetMean() << endl;
  }
  
}

Bool OpponentPositionObs::ReadFromFile(istream& infile)
{
  char line[100];  
  skip_white_space(infile);
  infile.getline(line, 100);
  if (!infile) return FALSE;  
  if (strncmp(line, file_header, strlen(file_header)) != 0)
    my_error("OpponentPositionObs::ReadFromFile: bad header, trying to continue");
  return ReadDataFromFile(infile);  
}

Bool OpponentPositionObs::ReadDataFromFile(istream& infile)
{
  char line[100]; 
  for (int i=0; i<=TeamSize; i++) {
    /* skip the Opponent # line; should we check to make sure it's right"? */
    infile.getline(line, 100);
    if (!aPlayerPositions[i].rg.ReadFromFile(infile))
      return FALSE;
    skip_white_space(infile);  
    infile.getline(line, 100); /* skip the conf line: SMURF */
  }  
  return TRUE;
}


void OpponentPositionObs::SortByAverageX(StatPlayerInfo* aInfo, 
					 int* sort_order, Bool use_zero)
{
  /* something was goign wrong with allocation, so I changed this to a 
     static allocation */
  float avgX[20];
  int i;
  
  /* avgX = new float[TeamSize + use_zero]; SMURF */
  
  /* fill in average X vals */
  for (i=0; i<TeamSize + use_zero; i++)
    avgX[i] = aInfo[i + !use_zero].rg.AverageX();
  
  /* fill in sort_order thing with increasing nums */
  for (i=0; i<TeamSize + !use_zero; i++)
    sort_order[i] = i + !use_zero;
  
  /* do a BubbleSort */
  BubbleSort(TeamSize + use_zero, sort_order, avgX);
  
  /* delete [] avgX; SMURF */
  
  return;  
}


/* in order to compare arrays of RectGrids, we're going to sort by average
   X value and compare them that way. This is not the best way, but I think it
   will be passable in some cases */
float OpponentPositionObs::PrivSimilarityTo(Observation* pObs)
{
  OpponentPositionObs* pComp = (OpponentPositionObs*) pObs;
  int* thisSortIdx;
  int* compSortIdx;
  float sim = 0.0;
  
  thisSortIdx = new int[TeamSize];
  compSortIdx = new int[TeamSize];
  
  SortByAverageX(aPlayerPositions, thisSortIdx);
  SortByAverageX(pComp->aPlayerPositions, compSortIdx);
  
  for (int i=0; i<TeamSize; i++) {
    sim += (1.0 / TeamSize) * aPlayerPositions[thisSortIdx[i]].rg.
      SimilarityTo(&(pComp->aPlayerPositions[compSortIdx[i]].rg), sim_M);
  }

  sim = sim * (MaxSimilarity - MinSimilarity) + MinSimilarity;  

  delete [] thisSortIdx;
  delete [] compSortIdx;  

  return sim;
}

void OpponentPositionObs::PrivCombineWith(Observation* pObs, float weight)
{
  OpponentPositionObs* pComb = (OpponentPositionObs*) pObs;
  for (int i=0; i<=TeamSize; i++) 
    aPlayerPositions[i].rg.CombineWith(&pComb->aPlayerPositions[i].rg, weight);
}

/*************************** BallControllerObs *****************************/

void BallControllerObs::Initialize(OptionInfo* MemOpt)
{
  TeamSize = MemOpt->SP_team_size;
  aControlCountOpp = new int[TeamSize+1];
  aControlCountTeam = new int[TeamSize+1];
}
BallControllerObs::~BallControllerObs()
{
  delete [] aControlCountOpp;
  delete [] aControlCountTeam;  
}


void BallControllerObs::Reset()
{
  for (int i=0; i<=TeamSize; i++) {
    aControlCountOpp[i] = 0;
    aControlCountTeam[i] = 0;
  }
  ControlCountNone = 0;
}

void BallControllerObs::PrivRecord(PositionInfo* pPIMem)
{
  Unum auCont[MaxBallControllers];
  SideType asCont[MaxBallControllers];
  
  GetBallController(pPIMem, asCont, auCont);
  for (int i=0; i<MaxBallControllers; i++) {
    switch (asCont[i]) {
    case ST_Ours:
      aControlCountTeam[auCont[i]]++;
      DebugCont(printf("Time: %d\t We   control ball: %d\n",
		       pPIMem->CurrentTime.t, auCont[i]));
      break;
    case ST_Theirs:
      aControlCountOpp[auCont[i]]++;
      DebugCont(printf("Time: %d\t They control ball: %d\n",
		       pPIMem->CurrentTime.t, auCont[i]));
      break;
    case ST_Neither:
      DebugCont(printf("Time: %d\t No   control ball\n",
		       pPIMem->CurrentTime.t));
      ControlCountNone++;
      break;
    case ST_NoValue:
    case ST_Unknown:
      break;
    default:
      my_error("Observe: bad ret from GetBallController: '%c'\n", asCont[i]);
      break;
    }
  } /* for loop */
}

void BallControllerObs::PrivWriteInfoToFile(ostream& output_file)
{
  output_file << endl << file_header << endl;
  for (int num=0; num <= TeamSize; num++) {
    output_file << " Opponent " << num << ": " << aControlCountOpp[num] << endl;
  }
  for (int num=0; num <= TeamSize; num++) {
    output_file << " Teammate " << num << ": " << aControlCountTeam[num] << endl;
  }
  output_file << " No Controller: " << ControlCountNone << endl;

  /*output_file << "Sim to self: " << SimilarityTo(this) << endl;  */
}

void BallControllerObs::PrivWriteSummaryInfoToFile(ostream& output_file)
{
  int sum;
  
  output_file << endl << "******Ball Controllers Summary******" << endl;
  sum = 0;
  for (int num=0; num <= TeamSize; num++) {
    sum += aControlCountOpp[num];
  }
  output_file << " Opponents: " << sum << endl;

  sum = 0;
  for (int num=0; num <= TeamSize; num++) {
    sum += aControlCountTeam[num];
  }
  output_file << " Teammates: " << sum << endl;

  output_file << " No Controller: " << ControlCountNone << endl;
}

Bool BallControllerObs::ReadFromFile(istream& infile)
{
  char line[100];  
  skip_white_space(infile);
  infile.getline(line, 100);
  if (!infile) return FALSE;  
  if (strncmp(line, file_header, strlen(file_header)) != 0)
    my_error("BallControllerObs::ReadFromFile: bad header, trying to continue");
  return ReadDataFromFile(infile);  
}


Bool BallControllerObs::ReadDataFromFile(istream& infile)
{
  char line[100];  
  for (int i=0; i<=(TeamSize+1)*2; i++) {
    if (!skip_white_space(infile))
      return FALSE;  
    infile.getline(line, 100);
    if (!infile)
      return FALSE;  
    else {
      if (i<=TeamSize) { /* opponent line */
	if (!GetIntFromLine(line, &aControlCountOpp[i]))
	  return FALSE;
      } else if (i == (TeamSize+1)*2) { /* last line (no cont) */
	if (!GetIntFromLine(line, &ControlCountNone))
	  return FALSE;
      } else { /* teammate line */
	if (!GetIntFromLine(line, &aControlCountTeam[i - (TeamSize+1)]))
	  return FALSE;
      }
    }
  } /* for loop */
  return TRUE;  
}


float BallControllerObs::PrivSimilarityTo(Observation* pObs)
{
  BallControllerObs* pComp = (BallControllerObs*) pObs;
  
  int thisSum=0, compSum=0;
  float scale;
  float diff;
  float sim;
  
  for (int i=0; i<=TeamSize; i++) {
    thisSum += aControlCountOpp[i];
    thisSum += aControlCountTeam[i];
    compSum += pComp->aControlCountOpp[i];
    compSum += pComp->aControlCountTeam[i];
  }
  thisSum += ControlCountNone;
  compSum += pComp->ControlCountNone;
  
  scale = ((float) thisSum) / compSum;
  
  for (int i=0; i <=TeamSize; i++) {
    diff += fabs(aControlCountOpp[i] - scale*pComp->aControlCountOpp[i]);
    diff += fabs(aControlCountTeam[i] - scale*pComp->aControlCountTeam[i]);
  }
  diff += fabs(ControlCountNone - scale*pComp->ControlCountNone);
  
  sim = diff / (2 * thisSum);
  sim = 1 - sim;
  
  return Round(sim, -2);
}

void BallControllerObs::PrivCombineWith(Observation* pObs, float weight)
{
  BallControllerObs* pComb = (BallControllerObs*) pObs;

  for (int i=0; i <=TeamSize; i++)
    aControlCountOpp[i] = 
      RoundToInt((weight*aControlCountOpp[i] + pComb->aControlCountOpp[i]) /
		 (1.0 + weight));

  for (int i=0; i <=TeamSize; i++)
    aControlCountTeam[i] = 
      RoundToInt((weight*aControlCountTeam[i] + pComb->aControlCountTeam[i]) /
		 (1.0 + weight));

  ControlCountNone = 
    RoundToInt((weight*ControlCountNone + pComb->ControlCountNone) /
	       (1.0 + weight));
}


/*************************** OpponentPassingObs *****************************/

void OpponentPassingObs::Initialize(OptionInfo* MemOpt)
{
  OP_pass_num_past_cont = MemOpt->OP_pass_num_past_cont;
  OP_pass_timeout       = MemOpt->OP_pass_timeout;

  prgmOppPassing = new RectGridMulti;
  prgmOppPassing->Initialize(MemOpt->SP_pitch_length, MemOpt->SP_pitch_width,
			     MemOpt->OP_rg_rows, MemOpt->OP_rg_cols,
			     MemOpt->OP_rg_vals);

  aPastCont = new PlayerID[OP_pass_num_past_cont];
  
  /* this is test code 
  prgmOppPassing[0].Zero();
  prgmOppPassing[0].WriteToFile(cout);

  prgmOppPassing[0].Zero();
  RecordTrajectory(Vector(-25, 0), Vector(25, 0), &prgmOppPassing[0]);
  prgmOppPassing[0].WriteToFile(cout);

  prgmOppPassing[0].Zero();
  RecordTrajectory(Vector(0, 25), Vector(0, -25), &prgmOppPassing[0]);
  prgmOppPassing[0].WriteToFile(cout);
  */
}
OpponentPassingObs::~OpponentPassingObs()
{
  delete prgmOppPassing;
  delete [] aPastCont;  
}


void OpponentPassingObs::Reset()
{
  prgmOppPassing->Zero();
  numPasses = 0;
  ClearPastCont(aPastCont);
}

void OpponentPassingObs::PrivRecord(PositionInfo* pPIMem)
{
  Unum auCont[MaxBallControllers];
  SideType asCont[MaxBallControllers];
  PlayerID* anewCont = new PlayerID[OP_pass_num_past_cont];
  int num_new = 0;

  ClearPastCont(anewCont);

  GetBallController(pPIMem, asCont, auCont);
  for (int i=0; i<MaxBallControllers; i++) {
    switch (asCont[i]) {
    case ST_Theirs:
    case ST_Ours:
      if (num_new >= OP_pass_num_past_cont) {
	my_error("OpponentPassingObs: more new controllors than past array can handle\n");
	break;
      }
      anewCont[num_new].side = asCont[i];
      anewCont[num_new].num  = auCont[i];
      anewCont[num_new].pos  =
	pPIMem->PlayerAbsolutePosition(asCont[i]==ST_Ours ? pPIMem->MySide : pPIMem->TheirSide,
				       auCont[i]);
      anewCont[num_new].time = pPIMem->CurrentTime;
      num_new++;
      if (asCont[i] == ST_Ours)
	break;
      /* go through the past controllers, and see if we have a pass */
      /* Question: 2 controllers to 2 controllers would give 2 passes? */
      for (int j=0; j<OP_pass_num_past_cont; j++) {
	/* SMURF: I'm requiring numbers to be different, but that may miss some
	   passes involving unknown players that we could use */
	/* SMURF: if we learn a players number and he's dribbling, it
	   might look like a pass */
	if (aPastCont[j].side == asCont[i] &&
	    pPIMem->CurrentTime-aPastCont[j].time < OP_pass_timeout &&
	    aPastCont[j].num != auCont[i]) {
	  /* record this pass */
	  DebugCont(printf("Time: %d\tI saw a pass!!!!!!\n", pPIMem->CurrentTime.t));
	  DebugCont(cout << " from " << aPastCont[j].pos << " to " << anewCont[i].pos << endl);
	  RecordTrajectory(aPastCont[j].pos, anewCont[i].pos, prgmOppPassing);
	  numPasses++;
	  /* since we have a completed pass, we can break and clear the old
	     controllers array */
	  DebugCont(prgmOppPassing->WriteToFile(cout));
	  ClearPastCont(aPastCont);
	  break;
	}
      }
      break;

    case ST_Neither:
    case ST_NoValue:
    case ST_Unknown:
      break;
    default:
      my_error("Observe: bad ret from GetBallController: '%c'\n", asCont[i]);
      break;
    }
  } /* for loop */

  /* now put in the new controllers. The oldest controllers are pushed furthest
     towards the high numbers of the array */
  if (num_new > 0) {
    for(int i = OP_pass_num_past_cont-num_new-1; i >= 0; i--)
      aPastCont[i+num_new] = aPastCont[i];
    for(int i = 0; i<num_new; i++)
      aPastCont[i] = anewCont[i];
  }

  delete [] anewCont;
}

void OpponentPassingObs::PrivWriteInfoToFile(ostream& output_file)
{
  output_file << endl << file_header << endl;
  output_file << "Num Opponent passes: " << numPasses << endl;
  prgmOppPassing->WriteToFile(output_file);
}

void OpponentPassingObs::PrivWriteSummaryInfoToFile(ostream& output_file)
{
  output_file << endl << "******Opponent Passing Summary******" << endl;
  output_file << "Num Opponent passes: " << numPasses << endl;
}

void OpponentPassingObs::NotifyOfPlayMode(Pmode playmode, Time)
{
  /* on all other play mode changes, the ball is set somewhere, so we should
     clear the controller array */
  if (playmode != PM_Play_On)
    ClearPastCont(aPastCont);
}

void OpponentPassingObs::ClearPastCont(PlayerID apc[])
{
  for (int i=0; i < OP_pass_num_past_cont; i++)
    apc[i].side = ST_NoValue;
}

Bool OpponentPassingObs::ReadFromFile(istream& infile)
{
  char line[100];  
  skip_white_space(infile);
  infile.getline(line, 100);
  if (!infile) return FALSE;  
  if (strncmp(line, file_header, strlen(file_header)) != 0)
    my_error("BallPositionObs::ReadFromFile: bad header, trying to continue");
  return ReadDataFromFile(infile);  
}

Bool OpponentPassingObs::ReadDataFromFile(istream& infile)
{
  char line[100];  
  infile.getline(line, 100);
  if (!GetIntFromLine(line, &numPasses)) {
    my_error("OpponentPassingObs: Failed reading numPasses line");
    return FALSE;
  }
  skip_white_space(infile);  
  return prgmOppPassing->ReadFromFile(infile);  
}


float OpponentPassingObs::PrivSimilarityTo(Observation* pObs)
{
  OpponentPassingObs* pComp = (OpponentPassingObs*) pObs;
  /* SNORK */
  return 0.0;
}

void OpponentPassingObs::PrivCombineWith(Observation* pObs, float weight)
{
  OpponentPassingObs* pComb = (OpponentPassingObs*) pObs;
  prgmOppPassing->CombineWith(pComb->prgmOppPassing, weight);
}

/************************** OpponentDribblingObs *****************************/

/* Here's the idea of the dribbling observation:
   We watch for who is controlling the ball
   We store the last OP_drib_num_past_cont players who have controlled the ball
   If OP_max_drib_cntl_intvl cycles goes by, the player is removed from list
   Anytime a player has been removed from the list, we check to see if
   he has travelled enough distance with the ball. If he has, we then record
   a dribble */

void OpponentDribblingObs::Initialize(OptionInfo* MemOpt)
{
  OP_drib_num_past_cont = MemOpt->OP_drib_num_past_cont;
  OP_max_drib_cntl_intvl = MemOpt->OP_max_drib_cntl_intvl;
  OP_min_drib_dist = MemOpt->OP_min_drib_dist;

  prgmOppDribbling = new RectGridMulti;
  prgmOppDribbling->Initialize(MemOpt->SP_pitch_length, MemOpt->SP_pitch_width,
			       MemOpt->OP_rg_rows, MemOpt->OP_rg_cols,
			       MemOpt->OP_rg_vals);
  
  aPastCont = new PlayerDribbleInfo[OP_drib_num_past_cont];
  ClearPastCont(aPastCont);
}
OpponentDribblingObs::~OpponentDribblingObs()
{
  delete prgmOppDribbling;
  delete [] aPastCont;  
}

  
void OpponentDribblingObs::Reset()
{
  prgmOppDribbling->Zero();
  ClearPastCont(aPastCont);
}

void OpponentDribblingObs::PrivRecord(PositionInfo* pPIMem)
{
  Unum auCont[MaxBallControllers];
  SideType asCont[MaxBallControllers];
  PlayerDribbleInfo* anewCont = new PlayerDribbleInfo[OP_drib_num_past_cont];
  int num_new = 0;

  ClearPastCont(anewCont);

  GetBallController(pPIMem, asCont, auCont);
  for (int i=0; i<MaxBallControllers; i++) {
    if (asCont[i] == ST_Theirs) {
      /* go through the past controllers and find if this player has been
	 controlling the ball */
      Bool player_in_past = FALSE;
      for (int j=0; j < OP_drib_num_past_cont && !player_in_past; j++) {
	if (aPastCont[j].side == asCont[i] && aPastCont[j].num == auCont[i]) {
	  /* the side check is unneccesary, but it's for sanity */
	  /* update this player's time and position */
	  aPastCont[j].last_cntl_pos = 
	    pPIMem->PlayerAbsolutePosition(asCont[i]==ST_Ours ? pPIMem->MySide : pPIMem->TheirSide,
					   auCont[i]);
	  aPastCont[j].last_cntl_time = pPIMem->CurrentTime;
	  player_in_past = TRUE;
	}
      }

      if (!player_in_past) {
	/* this player was not already in the array */
	if (num_new >= OP_drib_num_past_cont) {
	  my_error("OpponentPassingObs: more new controllors than past array can handle\n");
	  continue;
	}
	
	anewCont[num_new].side = asCont[i];
	anewCont[num_new].num  = auCont[i];
	anewCont[num_new].last_cntl_pos  = 
	  anewCont[num_new].orig_pos  = 
	  pPIMem->PlayerAbsolutePosition(asCont[i]==ST_Ours ? pPIMem->MySide : pPIMem->TheirSide,
					 auCont[i]);
	anewCont[num_new].last_cntl_time =
	  anewCont[num_new].orig_time =
	  pPIMem->CurrentTime;
	num_new++;
      }
    } /* if ST_Theirs */
  } /* for loop */

  /* now put in the new controllers.
     First, we purge all old controllers, recording their dribbles, if necc
     That will also push all the controllers to the front of the array
     We then stick our new ones at the end */
  PurgeControllerArray(pPIMem->CurrentTime);
  for(int i=0; num_new > 0; i++) {
    if (i > OP_drib_num_past_cont)
      my_error("OpponentDribblingObs: more controllers than we can store");
    if (aPastCont[i].side == ST_NoValue)
      aPastCont[i] = anewCont[--num_new];
  }

  delete [] anewCont;
  
}

void OpponentDribblingObs::PrivWriteInfoToFile(ostream& output_file)
{
  output_file << endl << file_header << endl;
  output_file << "Num Opponent dribbles: " << numDribbles << endl;
  prgmOppDribbling->WriteToFile(output_file);
}

void OpponentDribblingObs::PrivWriteSummaryInfoToFile(ostream& output_file)
{
  output_file << endl << "******Opponent Dribbling Summary******" << endl;
  output_file << "Num Opponent dribbles: " << numDribbles << endl;
}

void OpponentDribblingObs::NotifyOfPlayMode(Pmode playmode, Time CurrTime)
{
  /* on all other play mode changes, the ball is set somewhere, so we should
     clear the controller array */
  if (playmode != PM_Play_On)
    ClearPastCont(aPastCont);
}

void OpponentDribblingObs::ClearPastCont(PlayerDribbleInfo apc[])
{
  for (int i=0; i < OP_drib_num_past_cont; i++)
    apc[i].side = ST_NoValue;
}

void OpponentDribblingObs::PurgeControllerArray(Time CurrTime)
{
  int offset = 0; /* how far each entry must be shifted along */
  
  for (int i=0; i < OP_drib_num_past_cont; i++) {
    if (aPastCont[i].side == ST_NoValue)
      offset++;
    else {
      if (aPastCont[i].side != ST_Theirs)
	my_error("OpponentDribblingObs: something other than NoValue and Theirs in controllers");
      if (CurrTime - aPastCont[i].last_cntl_time > OP_max_drib_cntl_intvl) {
	/* we have to purge this entry. See if the player moved enough with
	   the ball to warrant recording */
	if ((aPastCont[i].last_cntl_pos - aPastCont[i].orig_pos).mod() >
	    OP_min_drib_dist) {
	  if (aPastCont[i].last_cntl_time - aPastCont[i].orig_time < OP_max_drib_cntl_intvl)
	    my_error("OpponentDribblingObs: it seems like the player moved to far too fast");
	  RecordTrajectory(aPastCont[i].orig_pos, aPastCont[i].last_cntl_pos,
			   prgmOppDribbling);
	  numDribbles++;
	  DebugCont(printf("Time: %d\tI saw a dribble!!!!!!\n", CurrTime.t));
	  DebugCont(cout << " from " << aPastCont[i].orig_pos
		    << " to " << aPastCont[i].last_cntl_pos << endl);

	} /* record the dribble */
	aPastCont[i].side = ST_NoValue;
	offset++;
      } else if (offset > 0)
	aPastCont[i-offset] = aPastCont[i]; /* slide the entry to the left */
    }
    
  } /* for loop */
  
}

Bool OpponentDribblingObs::ReadFromFile(istream& infile)
{
  char line[100];  
  skip_white_space(infile);
  infile.getline(line, 100);
  if (!infile) return FALSE;  
  if (strncmp(line, file_header, strlen(file_header)) != 0)
    my_error("BallPositionObs::ReadFromFile: bad header, trying to continue");
  return ReadDataFromFile(infile);  
}

Bool OpponentDribblingObs::ReadDataFromFile(istream& infile)
{
  char line[100];  
  infile.getline(line, 100);
  if (!GetIntFromLine(line, &numDribbles))
    return FALSE;
  skip_white_space(infile);  
  return prgmOppDribbling->ReadFromFile(infile);  
}


float OpponentDribblingObs::PrivSimilarityTo(Observation* pObs)
{
  OpponentDribblingObs* pComp = (OpponentDribblingObs*) pObs;
  /* SNORK */
  return 0.0;
}

    
void OpponentDribblingObs::PrivCombineWith(Observation* pObs, float weight)
{
  OpponentDribblingObs* pComb = (OpponentDribblingObs*) pObs;
  prgmOppDribbling->CombineWith(pComb->prgmOppDribbling, weight);
}
