/* -*- Mode: C -*- */

#include <string.h>
#include "adv_class.h"


#ifdef DEBUG_OUTPUT
#define DebugAdC(x) 
#else
#define DebugAdC(x)
#endif

#ifdef DEBUG_OUTPUT
#define DebugComb(x) x
#else
#define DebugComb(x)
#endif

const char* AdversaryTypeSeparatorLine = "***********************************";
const int linesize = 200; 


/************************** AdversaryTypeList ********************************/
AdversaryTypeList::~AdversaryTypeList()
{
  AdversaryTypeListItem *pTemp;
  for ( ; pHead != NULL; pHead = pTemp) {
    pTemp = pHead;
    delete pHead->pAdvT;
    delete pHead;
  }  
} 

void AdversaryTypeList::Add(AdversaryType* pAdvT)
{
  AdversaryTypeListItem *pNewItem;
  pNewItem = new AdversaryTypeListItem;
  pNewItem->pAdvT = pAdvT;
  
  pNewItem->pNext = pHead;
  pHead = pNewItem;
}

AdversaryType* AdversaryTypeList::GetCurrentIter()
{
  if (pIter == NULL)
    return NULL;
  AdversaryType* pTemp;
  
  pTemp = pIter->pAdvT;
  pIter = pIter->pNext;
  //printf("GetCurrIter: %d\n", pTemp->GetObserveType());
  return pTemp;
}


AdversaryType* AdversaryTypeList::MatchToObservations(ObserveList* pObsList)
{
  AdversaryTypeListItem *pItem, *pMax = NULL;
  float maxSim = MinSimilarity - 1.0; 

  for (pItem = pHead;
       pItem != NULL; 
       pItem = pItem->pNext) {
    float match = pItem->pAdvT->CompareToObservations(pObsList); 

    if (match > maxSim)
      pMax = pItem; 
    
  }
  return pMax->pAdvT; 
} 


/************************** AdversaryType ********************************/

AdversaryType::AdversaryType()
{
  name[0] = 0; 
} 

Bool AdversaryType::ReadObservationsFromFile(istream& infile,
					     OptionInfo* pMemOpt)
{
  char line[linesize];

  DebugAdC(cout << " About to read observations from file" << endl); 
  
  while (1) {
    if (!skip_white_space(infile))
      break; /* out of file to process */
    infile.getline(line, linesize);
    if (!infile) break;

    /* try to figure out if there is no weight line, and therefore default to
       1.0 */
    float weight = 1.0;
    if (strncmp(line, "***", 3)!=0) {
      if (!GetFloatFromLine(line, &weight)) {
	my_error("Badly formed weight line: %s", line);
	continue; 
      }
      DebugAdC(cout << "  Read in weight line" << endl); 
      if (!skip_white_space(infile))
	break; /* out of file to process */
      infile.getline(line, linesize);
      if (!infile) break;
    } 

    DebugAdC(cout << "Reading Observation type: " << line << endl); 

    Observation* pObs = NULL; 
    if (strncmp(line, BallPositionObs::FileHeaderString(),
		strlen(BallPositionObs::FileHeaderString())) == 0)
      pObs = new BallPositionObs; 
    else if (strncmp(line, OpponentPositionObs::FileHeaderString(),
		strlen(OpponentPositionObs::FileHeaderString())) == 0)
      pObs = new OpponentPositionObs; 
    else if (strncmp(line, BallControllerObs::FileHeaderString(),
		strlen(BallControllerObs::FileHeaderString())) == 0)
      pObs = new BallControllerObs; 
    else if (strncmp(line, OpponentPassingObs::FileHeaderString(),
		strlen(OpponentPassingObs::FileHeaderString())) == 0)
      pObs = new OpponentPassingObs; 
    else if (strncmp(line, OpponentDribblingObs::FileHeaderString(),
		strlen(OpponentDribblingObs::FileHeaderString())) == 0)
      pObs = new OpponentDribblingObs; 
    else
      my_error("Unrecognized observation type in adversary class file: %s", line);

    //DebugComb(cout << "    Read in an observation" << endl); 
    if (pObs != NULL) {
      pObs->Initialize(pMemOpt);
      pObs->SetEnabled(TRUE);
      if (!pObs->ReadDataFromFile(infile)) {
	my_error("Error reading in adv clas, type: %s", line);
	delete pObs; 
      } else 
	listTargetObs.Add(pObs, weight);
    } else
      my_error("AdversaryType::LoadFromFile: Bad AdvType or new failed"); 
    
  } /* while(1) reading loop */
  
  return TRUE; 

}

Bool AdversaryType::LoadFromFile(istream& infile, OptionInfo* pMemOpt)
{
  char line[linesize];
  char *pc; 

  if (!skip_white_space(infile)) return FALSE;

  /* read the first line, which should a seprator or the name line */
  infile.getline(line, linesize);
  if (!infile) return FALSE;
  if (strncmp(line, AdversaryTypeSeparatorLine,
	      strlen(AdversaryTypeSeparatorLine)) == 0) {
    /* it was a separator line, so get the next line */
    if (!skip_white_space(infile)) return FALSE;
    infile.getline(line, linesize);
  } 

  pc = strchr(line, ':'); 
  if (pc == NULL) {
    my_error("Badly formed adversary type name line: %s", line);
    return FALSE; 
  } 
  pc++; /* skip colon */
  if (*pc == ' ') pc++; /* skip space */
  strcpy(name, pc); /* get the name */
  DebugAdC(cout << "Loading in adversary class: " << name << endl); 
  
  /* now read in all the observations */
  return ReadObservationsFromFile(infile, pMemOpt); 
} 

void AdversaryType::WriteToFile(ostream& outfile)
{
  outfile << endl << AdversaryTypeSeparatorLine << endl
	  << "AdversaryType: " << name << endl; 
  
  Observation* pObs;
  for( listTargetObs.ResetIter();
       (pObs = listTargetObs.PeekCurrentIter()) != NULL;
       ) {
    outfile << endl 
            << "Weight: " << listTargetObs.PeekCurrentIterWeight(); 
    pObs->WriteInfoToFile(outfile);
    listTargetObs.AdvanceCurrentIter(); 
  } 

} 

float AdversaryType::CompareToObservations(ObserveList* pObsList)
{
  return listTargetObs.CompareTo(pObsList); 
}


Bool AdversaryType::CreateFromObservations(OptionInfo* pMemOpt)
{
  ifstream listfile, infile;
  char class_dir[MAX_FILE_LEN];
  int class_dir_len; 

  DebugComb(cout << "Create Adversary Type from observations" << endl); 

  listfile.open(pMemOpt->CP_create_adv_class_list_fn);
  if (!listfile)
    my_error("Could not open observation list file: %s",
	     pMemOpt->CP_create_adv_class_list_fn);
  else {
    char line[100];
    get_leading_directory(pMemOpt->CP_create_adv_class_list_fn, class_dir);
    class_dir_len = strlen(class_dir);
    DebugComb(cout << " AdvType: leading class dir: " << class_dir << endl); 
    while(1) {
      if (!skip_white_space(listfile)) break; 
      listfile.getline(line, 100);
      if (!listfile) break; /* file fully read */
      if (line[0] == '#')
	continue;  /* it's a comment line */
      if (line[0] != '/' && line[0] != '~') {
	/* it's a relative path */
	/* we'll put it onto the end of class dir, then copy it back into
	   line */
	strcat(class_dir, line);
	strcpy(line, class_dir);
	class_dir[class_dir_len] = 0;  /* cut off the file name */
      } 
      DebugComb(cout << "Trying to open file: " << line << endl); 
      infile.open(line);
      if (!infile) {
	my_error("Could not open observation file: %s", line);
	continue; 
      } 
	
      ReadObservationsFromFile(infile, pMemOpt); 
      
      infile.close(); 
    } 
  }

  DebugComb(cout << "Done reading files, combining the info" << endl); 

  listfile.close(); 

  /* combine the observations */
  /* ObserveList stores items in sorted order, which makes this much easier */
  Observation *pObs, *pCombObs;
  ObserveList listCombObs; 
  
  listTargetObs.ResetIter();
  pObs = listTargetObs.GetCurrentIter();
  while (pObs != NULL) {
    ObserveType currType = pObs->GetObserveType();
    int numCombined = 0;
    pCombObs = GetNewObservationOfType(currType, pMemOpt); 
    do {
      pCombObs->CombineWith(pObs, (float)numCombined);
      numCombined++; 
      pObs = listTargetObs.GetCurrentIter();
    } while (pObs != NULL && pObs->GetObserveType() == currType);
    /* we have no real weights, so they are all equal */
    listCombObs.Add(pCombObs, 1.0); 
  } /* while (pObs != NULL) */

  listTargetObs.Transfer(&listCombObs); 

  return TRUE; 
} 
