/* -*- Mode: C++ -*- */
#include "MemFormation.h"


/********************************************************************************/
/********************************************************************************/
/********************************************************************************/
/*                        TeamPosition Class                                    */
/********************************************************************************/
/********************************************************************************/
/********************************************************************************/

TeamPosition::TeamPosition()
{
  HomeBuffer = 0;
  type = PT_None;
  side = PS_None;
}

/********************************************************************************/

void TeamPosition::SetHome(Vector p, float buf, Ptype t, Pside s)
{
  Home = p;
  HomeBuffer = buf;

  type = t; 
  side = s;
}

/********************************************************************************/

void TeamPosition::SetHomeRange(Vector center, Vector size)
{
  HomeRange = Rectangle(center,size);
}

/********************************************************************************/

void TeamPosition::SetMaxRange (Vector center, Vector size)
{
  MaxRange = Rectangle(center,size);
}

/********************************************************************************/

void TeamPosition::PrintHomeRange ()
{
  HomeRange.Print();
}

/********************************************************************************/

void TeamPosition::PrintMaxRange ()
{
  MaxRange.Print();
}




/********************************************************************************/
/********************************************************************************/
/********************************************************************************/
/*                        Unit Class                                            */
/********************************************************************************/
/********************************************************************************/
/********************************************************************************/

Unit::Unit()
{
  type = UT_None;
  size = captain = 0;
  members = NULL;
}

/********************************************************************************/

Unit::~Unit()
{
  if ( members != NULL )
    delete [] members;
}

/********************************************************************************/

void Unit::InitializeUnit(Utype tp, int sz, Pnum *mems, Pnum capt)
{
  type = tp;
  size = sz;
  captain = mems[capt];
  members = new Pnum[size];

  for (int i=0; i<size; i++)
    members[i] = mems[i];
}

/********************************************************************************/

Bool Unit::IsMember(Pnum position)
{
  for (int i=0; i<size; i++){
    if ( members[i] == position )
      return TRUE;
  }
  return FALSE;
}

/********************************************************************************/

Bool Unit::IsCaptain(Pnum position)
{
  if ( members[0] == position )
      return TRUE;
  return FALSE;
}

/********************************************************************************/

Pnum Unit::ImportantPosition(Pnum p1, Pnum p2)
{
  int index1, index2;

  for (int i=0; i<size; i++){
    if ( members[i] == p1 )
      index1 = i;
    if ( members[i] == p2 )
      index2 = i;
  }

  if ( index1 < index2 )
    return p1;
  else if ( index2 < index1 )
    return p2;
  else
    my_error("indices shouldn't be the same");
  return 0;
}




/********************************************************************************/
/********************************************************************************/
/********************************************************************************/
/*                        Formation Class                                       */
/********************************************************************************/
/********************************************************************************/
/********************************************************************************/

Formation::Formation(Ftype formation_number, int sz)
{
  type = formation_number;
  size = sz;
  FormationUnits = NULL;

  for (int i=1; i<=size; i++)            /* Positions start at 1            */
    ReceiverList[i] = new Pnum[size];
                                         /* receiver positions              */
  TempReceiverList = new Pnum[size];     /* Used to pass back possible receivers */
}

/********************************************************************************/

Formation::~Formation()
{
  if ( FormationUnits != NULL )
    delete [] FormationUnits;

  for (int i=1; i<=size; i++)
    delete [] ReceiverList[i];
  delete [] TempReceiverList;
}

/********************************************************************************/

void Formation::AllocateUnits(int num)
{
  NumUnits = num;
  FormationUnits = new Unit[NumUnits];
}

/********************************************************************************/

void Formation::SetPosition(Pnum num, Vector home, float buffer,
			    Vector home_center, Vector home_size,
			    Vector max_center,  Vector max_size,
			    Ptype t, Pside s)
{
  /* Use this to put in the parameters for the positions in a given formation */
  PositionList[num].SetHome(home,buffer,t,s);
  PositionList[num].SetHomeRange(home_center,home_size);
  PositionList[num].SetMaxRange(max_center,max_size);
}

/********************************************************************************/

TeamPosition *Formation::GetPosition(Pnum num)
{
  if (num > size || num < 1) 
    my_error("position list index only goes from 1 to size %d (trying to get %d)",size,num);
  return &(PositionList[num]); 
}

/********************************************************************************/

void Formation::SetCandidateReceivers(Pnum position, int num, Pnum *PrefList)
{
  int i;

  for (i=0; i<num; i++)
    ReceiverList[position][i] = PrefList[i];

  for (; i<size; i++)
    ReceiverList[position][i] = Pnum_Unknown;  /* End the list with sequence       */
}					       /* of unknowns----means no reciever */

/********************************************************************************/

Pnum *Formation::GetCandidateReceivers(Pnum position, Fside LocationSide)
{
  /* Either call a function to build the list on the fly, 
     or return the precanned ReceiverList */
  switch(type){
  case FT_433: 
    Build_433_ReceiverList(TempReceiverList,position,LocationSide); 
    return TempReceiverList;
  case FT_442: 
    Build_442_ReceiverList(TempReceiverList,position,LocationSide); 
    return TempReceiverList;
  case FT_352: 
    Build_352_ReceiverList(TempReceiverList,position,LocationSide); 
    return TempReceiverList;
  case FT_72: 
    Build_72_ReceiverList(TempReceiverList,position,LocationSide); 
    return TempReceiverList;
  case FT_334: 
    Build_334_ReceiverList(TempReceiverList,position,LocationSide); 
    return TempReceiverList;
  case FT_244: 
    Build_244_ReceiverList(TempReceiverList,position,LocationSide); 
    return TempReceiverList;
  case FT_532: 
    Build_532_ReceiverList(TempReceiverList,position,LocationSide); 
    return TempReceiverList;
  default: 
    return  ReceiverList[position]; 
  }
}

/********************************************************************************/

Unit *Formation::GetUnit(int num)
{
  return &FormationUnits[num];
}

/********************************************************************************/

Bool Formation::IsUnitMember(int unit, Pnum position)
{
  return GetUnit(unit)->IsMember(position);
}

/********************************************************************************/

Bool Formation::IsUnitCaptain(int unit, Pnum position)
{
  return GetUnit(unit)->IsCaptain(position);
}

/********************************************************************************/

Bool Formation::IsMoreImportantPosition(Pnum pos1, Pnum pos2)
{
  if ( pos1 == pos2 )
    my_error("same positions more important???");

  for (int unit=0; unit<NumUnits; unit++){
    if ( IsUnitMember(unit,pos1) && IsUnitMember(unit,pos2) ){
      if ( GetUnit(unit)->ImportantPosition(pos1,pos2) == pos1 )
	return TRUE;
    }
  }
  return FALSE;
}

/********************************************************************************/

Pnum Formation::LocationToPosition(Vector p)
{
  /* Finds the position whose home range (x,y) falls.  If more than one,
     takes the one closest to the home position.  If none, takes the one
     whose home range rectangle is closest */

  float tiebreakDist = -1000;  /* > 2 * (pitch_length + pitch_width) */
  Pnum position;
  Rectangle *HomeRange;
  
  for (int i=1; i<=GetSize(); i++){
    HomeRange = GetPosition(i)->GetHomeRange();
    if ( HomeRange->IsWithin(p) ){  /* Check if (x,y) is within the home range */
      float HomeDist = DistanceToPositionHome(i,p);
      if ( tiebreakDist < 0 || HomeDist < tiebreakDist ){
	position = i;
	tiebreakDist = HomeDist;
      }
    }    /* We're not in the range of this position. Now check if closest so far */
    else if ( tiebreakDist < 0 && HomeRange->DistanceToEdge(p) > tiebreakDist ){
      position = i;
      tiebreakDist = HomeRange->DistanceToEdge(p);    /* a negative number     */
    }
  }

  return position;
}

/********************************************************************************/

float Formation::DistanceToPositionHome(Pnum position, Vector p)
{
  Vector Home = GetPosition(position)->GetHome();
  return Home.dist(p);
}

/********************************************************************************/

Pnum Formation::ClosestPosition(Vector p)
{
  float dist, minDist = HUGE;
  Pnum closestPosition;
  for (int position=1; position<=size; position++){
    dist = DistanceToPositionHome(position,p);
    if (dist < minDist){
      minDist = dist;
      closestPosition = position;
    }
  }
  return closestPosition;
}




/********************************************************************************/
/********************************************************************************/
/********************************************************************************/
/*                     FormationInfo Class                                   */
/********************************************************************************/
/********************************************************************************/
/********************************************************************************/

FormationInfo::FormationInfo()
{
  MarkChangeMethod = MC_Obey;
  HomeChangeMethod = HC_Obey;
  CurrentFormation = Formation_Unknown;
  FormationTime = 0;
}

/********************************************************************************/

void FormationInfo::Initialize()
{
  Formation_433 = new Formation(FT_433,11);
  Initialize_433_Formation();

  Formation_442 = new Formation(FT_442,11);
  Initialize_442_Formation();

  Formation_352 = new Formation(FT_352,11);
  Initialize_352_Formation();

  Formation_72 = new Formation(FT_72,11);
  Initialize_72_Formation();

  Formation_334 = new Formation(FT_334,11);
  Initialize_334_Formation();

  Formation_244 = new Formation(FT_244,11);
  Initialize_244_Formation();

  Formation_532 = new Formation(FT_532,11);
  Initialize_532_Formation();

  Formation_right = new Formation(FT_right,11);
  Initialize_right_Formation();

  Formation_left = new Formation(FT_left,11);
  Initialize_left_Formation();

  if      ( !strcmp(FP_initial_mc_method,"Obey") )    MarkChangeMethod = MC_Obey;
  else if ( !strcmp(FP_initial_mc_method,"Closest") ) MarkChangeMethod = MC_Closest;
  else if ( !strcmp(FP_initial_mc_method,"Open") )    MarkChangeMethod = MC_Open;
  else my_error("Unknown mc parameter");

  if      ( !strcmp(FP_initial_hc_method,"Obey") )            HomeChangeMethod = HC_Obey;
  else if ( !strcmp(FP_initial_hc_method,"Shift") )           HomeChangeMethod = HC_Shift;
  else if ( !strcmp(FP_initial_hc_method,"Mark") )            HomeChangeMethod = HC_Mark;
  else my_error("Unknown hc parameter");

  PositionPlayers = new Unum[SP_team_size+1];     /* Start map at 1 */
  PlayerPositions = new Pnum[SP_team_size+1];     /* Start map at 1 */
  PlayerPositionTimes = new Time[SP_team_size+1];

  PositionMarks = new Unum[SP_team_size+1];
  PositionMarkTimes = new Time[SP_team_size+1];

  for(int i=1; i<=SP_team_size; i++){
    PositionPlayers[i] = Unum_Unknown;
    PlayerPositions[i] = Pnum_Unknown;
    PlayerPositionTimes[i] = 0;
    PositionMarks[i] = Unum_Unknown;
    PositionMarkTimes[i] = 0;
  }

  for(int i=1; i<=SP_team_size; i++){
    switch (i){
    case 1: 
      PlayerPositions[1] = FP_initial_player_1_pos; 
      PositionPlayers[FP_initial_player_1_pos] = 1;
      break;
    case 2: 
      PlayerPositions[2] = FP_initial_player_2_pos; 
      PositionPlayers[FP_initial_player_2_pos] = 2;
      break;
    case 3: 
      PlayerPositions[3] = FP_initial_player_3_pos; 
      PositionPlayers[FP_initial_player_3_pos] = 3;
      break;
    case 4: 
      PlayerPositions[4] = FP_initial_player_4_pos; 
      PositionPlayers[FP_initial_player_4_pos] = 4;
      break;
    case 5: 
      PlayerPositions[5] = FP_initial_player_5_pos; 
      PositionPlayers[FP_initial_player_5_pos] = 5;
      break;
    case 6: 
      PlayerPositions[6] = FP_initial_player_6_pos; 
      PositionPlayers[FP_initial_player_6_pos] = 6;
      break;
    case 7: 
      PlayerPositions[7] = FP_initial_player_7_pos; 
      PositionPlayers[FP_initial_player_7_pos] = 7;
      break;
    case 8: 
      PlayerPositions[8] = FP_initial_player_8_pos; 
      PositionPlayers[FP_initial_player_8_pos] = 8;
      break;
    case 9: 
      PlayerPositions[9] = FP_initial_player_9_pos; 
      PositionPlayers[FP_initial_player_9_pos] = 9;
      break;
    case 10: 
      PlayerPositions[10] = FP_initial_player_10_pos; 
      PositionPlayers[FP_initial_player_10_pos] = 10;
      break;
    case 11: 
      PlayerPositions[11] = FP_initial_player_11_pos; 
      PositionPlayers[FP_initial_player_11_pos] = 11;
      break;
    default: my_error("Team not that big");
    }
    PlayerPositionTimes[i] = CurrentTime;
  }

  if      ( !strcmp(FP_initial_formation,"433") ) InitialFormation = FT_433;
  else if ( !strcmp(FP_initial_formation,"442") ) InitialFormation = FT_442;
  else if ( !strcmp(FP_initial_formation,"352") ) InitialFormation = FT_352;
  else if ( !strcmp(FP_initial_formation,"72")  ) InitialFormation = FT_72;
  else if ( !strcmp(FP_initial_formation,"334") ) InitialFormation = FT_334;
  else if ( !strcmp(FP_initial_formation,"244") ) InitialFormation = FT_244;
  else if ( !strcmp(FP_initial_formation,"532") ) InitialFormation = FT_532;
  else if ( !strcmp(FP_initial_formation,"right") ) InitialFormation = FT_right;
  else if ( !strcmp(FP_initial_formation,"left") ) InitialFormation = FT_left;
  else my_error("Unknown formation parameter");

  SetFormation(InitialFormation,CurrentTime);

  if      ( !strcmp(FP_formation_when_losing,"initial")) FormationWhenLosing = InitialFormation;
  else if ( !strcmp(FP_formation_when_losing,"433") ) FormationWhenLosing = FT_433;
  else if ( !strcmp(FP_formation_when_losing,"442") ) FormationWhenLosing = FT_442;
  else if ( !strcmp(FP_formation_when_losing,"352") ) FormationWhenLosing = FT_352;
  else if ( !strcmp(FP_formation_when_losing,"72")  ) FormationWhenLosing = FT_72;
  else if ( !strcmp(FP_formation_when_losing,"334") ) FormationWhenLosing = FT_334;
  else if ( !strcmp(FP_formation_when_losing,"244") ) FormationWhenLosing = FT_244;
  else if ( !strcmp(FP_formation_when_losing,"532") ) FormationWhenLosing = FT_532;
  else if ( !strcmp(FP_formation_when_losing,"right") ) FormationWhenLosing = FT_right;
  else if ( !strcmp(FP_formation_when_losing,"left") ) FormationWhenLosing = FT_left;
  else my_error("Unknown formation_when_losing parameter");


  if      ( !strcmp(FP_formation_when_losing_lots,"initial")) FormationWhenLosingByLots = InitialFormation;
  else if ( !strcmp(FP_formation_when_losing_lots,"433") ) FormationWhenLosingByLots = FT_433;
  else if ( !strcmp(FP_formation_when_losing_lots,"442") ) FormationWhenLosingByLots = FT_442;
  else if ( !strcmp(FP_formation_when_losing_lots,"352") ) FormationWhenLosingByLots = FT_352;
  else if ( !strcmp(FP_formation_when_losing_lots,"72")  ) FormationWhenLosingByLots = FT_72;
  else if ( !strcmp(FP_formation_when_losing_lots,"334") ) FormationWhenLosingByLots = FT_334;
  else if ( !strcmp(FP_formation_when_losing_lots,"244") ) FormationWhenLosingByLots = FT_244;
  else if ( !strcmp(FP_formation_when_losing_lots,"532") ) FormationWhenLosingByLots = FT_532;
  else if ( !strcmp(FP_formation_when_losing_lots,"right") ) FormationWhenLosingByLots = FT_right;
  else if ( !strcmp(FP_formation_when_losing_lots,"left") ) FormationWhenLosingByLots = FT_left;
  else my_error("Unknown formation_when_losing_lots parameter");

  if      ( !strcmp(FP_formation_when_winning,"initial")) FormationWhenWinning = InitialFormation;
  else if ( !strcmp(FP_formation_when_winning,"433") ) FormationWhenWinning = FT_433;
  else if ( !strcmp(FP_formation_when_winning,"442") ) FormationWhenWinning = FT_442;
  else if ( !strcmp(FP_formation_when_winning,"352") ) FormationWhenWinning = FT_352;
  else if ( !strcmp(FP_formation_when_winning,"72")  ) FormationWhenWinning = FT_72;
  else if ( !strcmp(FP_formation_when_winning,"334") ) FormationWhenWinning = FT_334;
  else if ( !strcmp(FP_formation_when_winning,"244") ) FormationWhenWinning = FT_244;
  else if ( !strcmp(FP_formation_when_winning,"532") ) FormationWhenWinning = FT_532;
  else if ( !strcmp(FP_formation_when_winning,"right") ) FormationWhenWinning = FT_right;
  else if ( !strcmp(FP_formation_when_winning,"left") ) FormationWhenWinning = FT_left;
  else my_error("Unknown formation_when_winning parameter");

  if      ( !strcmp(FP_formation_when_tied,"initial")) FormationWhenTied = InitialFormation;
  else if ( !strcmp(FP_formation_when_tied,"433") ) FormationWhenTied = FT_433;
  else if ( !strcmp(FP_formation_when_tied,"442") ) FormationWhenTied = FT_442;
  else if ( !strcmp(FP_formation_when_tied,"352") ) FormationWhenTied = FT_352;
  else if ( !strcmp(FP_formation_when_tied,"72")  ) FormationWhenTied = FT_72;
  else if ( !strcmp(FP_formation_when_tied,"334") ) FormationWhenTied = FT_334;
  else if ( !strcmp(FP_formation_when_tied,"244") ) FormationWhenTied = FT_244;
  else if ( !strcmp(FP_formation_when_tied,"532") ) FormationWhenTied = FT_532;
  else if ( !strcmp(FP_formation_when_tied,"right") ) FormationWhenTied = FT_right;
  else if ( !strcmp(FP_formation_when_tied,"left") ) FormationWhenTied = FT_left;
  else my_error("Unknown formation_when_tied parameter");

  stored_PositionOfMyLocation_time = Time(-1,0);
  stored_PositionOfMyLocation      = Pnum_Unknown;
}

/********************************************************************************/

FormationInfo::~FormationInfo()
{
  delete [] PlayerPositions;
  delete [] PositionPlayers;
  delete [] PlayerPositionTimes;  /* Why does this cause a seg fault?? */
  delete [] PositionMarks;
  delete [] PositionMarkTimes;

  delete Formation_433;
  delete Formation_442;
  delete Formation_352;
  delete Formation_72;
  delete Formation_334;
  delete Formation_244;
  delete Formation_532;
  delete Formation_right;
  delete Formation_left;
}

/********************************************************************************/

void FormationInfo::Reset()
{
  if (MyPosition() == Pnum_Unknown )
    SetMyPosition(GetNewPosition());
  ResetCurrentHome();
}

/********************************************************************************/

void FormationInfo::SetPlayerPosition(Unum player, Pnum position, Time time)
{
  /* printf("%d: changing %d's position from %d to %d\n",
	 Mem->MyNumber,player,Mem->PlayerPosition(player),position); */

  if ( time < PlayerPositionTimes[player] ) 
    my_error("Update player position to an earlier time???");

  if ( player == Unum_Unknown )
    my_error("Should we be setting unknown player's position?");
  else{
    /* First check to see if some other player is in the same position */
    int old_position = PlayerPositions[player];
    int old_position_player = Unum_Unknown;

    if ( old_position == position ){ /* just update the time */
      PlayerPositionTimes[player] = time;
      return;
    }

    if ( old_position != Pnum_Unknown && PositionPlayers[old_position] == player ){
      for (int i=1; i<=SP_team_size; i++) {
	if ( i != player && PlayerPositions[i] == old_position )
	  old_position_player = i;
      }
      PositionPlayers[old_position] = old_position_player;
    }
  }

  if (position != Pnum_Unknown)
    PositionPlayers[position] = player;
  if (player   != Unum_Unknown  ){
    PlayerPositions[player]   = position;
    PlayerPositionTimes[player] = time;
  }
}

/********************************************************************************/

void FormationInfo::ClearPlayerPosition(Unum player)
{
  if ( player == Unum_Unknown )
    my_error("Should we be clearing unknown player's position?");

  Pnum position = PlayerPositions[player];
  PlayerPositions[player] = Pnum_Unknown;

  if ( PositionPlayers[position] == player )
    PositionPlayers[position] = Unum_Unknown;
}

/********************************************************************************/

void FormationInfo::ClearPositionPlayer(Pnum position)
{
  if ( position == Pnum_Unknown )
    my_error("Should we be clearing unknown position's player?");

  for (int i=1; i<=SP_team_size; i++){
    if ( PlayerPositions[i] == position )
      PlayerPositions[i] = Pnum_Unknown;
  }
  PositionPlayers[position] = Unum_Unknown;
}

/********************************************************************************/

void FormationInfo::SetMyPosition(Pnum position)
{
  SetPlayerPosition(MyNumber, position, CurrentTime);
  ResetCurrentHome();
#if TEAM_POSITION_DEBUG
  printf("player %d, time %d:  going to position %d\n",
	 MyNumber,CurrentTime, position);
#endif
}

/********************************************************************************/

void FormationInfo::ClearMyPosition()
{
#if 0
  printf("player %d, time %d:  clearing position %d\n",
	 MyNumber,CurrentTime,MyPosition());
#endif
  ClearPlayerPosition(MyNumber);
}

/********************************************************************************/

Bool  FormationInfo::PositionOccupied (Pnum position)
{
  if (PositionPlayer(position) == Unum_Unknown )
    return FALSE;
  else 
    return TRUE;
}

/********************************************************************************/

Pnum FormationInfo::LocationToPosition(Vector p, Bool AdjustForOffsides)
{
  if ( AdjustForOffsides )
    return LocationToAdjustedForOffsidePosition(p);
  else
    return CurrentFormation->LocationToPosition(p);
}

/********************************************************************************/

float FormationInfo::DistanceToPositionHome(Pnum position, Vector p, Bool AdjustForOffsides)
{
  if ( AdjustForOffsides )
    return DistanceToAdjustedForOffsidePositionHome(position,p);
  else
    return CurrentFormation->DistanceToPositionHome(position,p);
}

/********************************************************************************/

Pnum FormationInfo::ClosestPosition(Vector p, Bool AdjustForOffsides)
{
  if ( AdjustForOffsides )
    return ClosestAdjustedForOffsidePosition(p);
  else
    return CurrentFormation->ClosestPosition(p);
}

/********************************************************************************/

Pnum FormationInfo::LocationToAdjustedForOffsidePosition(Vector p)
{
  float tiebreakDist = -1000;  /* > 2 * (pitch_length + pitch_width) */
  Pnum position;
  Rectangle HomeRange;
  
  for (int i=1; i<=MyCurrentFormationSize(); i++){
    HomeRange = PositionHomeRange(i,TRUE);
    if ( HomeRange.IsWithin(p) ){  /* Check if (x,y) is within the home range */
      float HomeDist = DistanceToAdjustedForOffsidePositionHome(i,p);
      if ( tiebreakDist < 0 || HomeDist < tiebreakDist ){
	position = i;
	tiebreakDist = HomeDist;
      }
    }    /* We're not in the range of this position. Now check if closest so far */
    else if ( tiebreakDist < 0 && HomeRange.DistanceToEdge(p) > tiebreakDist ){
      position = i;
      tiebreakDist = HomeRange.DistanceToEdge(p);    /* a negative number     */
    }
  }
  return position;
}

/********************************************************************************/

float FormationInfo::DistanceToAdjustedForOffsidePositionHome(Pnum position, Vector p)
{
  return PositionToAdjustForOffsidePosition(PositionHome(position)).dist(p);
}

/********************************************************************************/

float FormationInfo::Distance2ToAdjustedForOffsidePositionHome(Pnum position, Vector p)
{
  return PositionToAdjustForOffsidePosition(PositionHome(position)).dist2(p);
}

/********************************************************************************/

Pnum FormationInfo::ClosestAdjustedForOffsidePosition(Vector p)
{
  float dist, minDist = HUGE;
  Pnum closestPosition;
  for (int position=1; position<=MyCurrentFormationSize(); position++){
    dist = Distance2ToAdjustedForOffsidePositionHome(position,p);
    if (dist < minDist){
      minDist = dist;
      closestPosition = position;
    }
  }
  return closestPosition;
}

/********************************************************************************/

int FormationInfo::NumCloserAdjustedForOffsidePositions(Pnum pos, Vector p)
{
  float dist, minDist = Distance2ToAdjustedForOffsidePositionHome(pos,p);
  int num = 0;
  for (int position=1; position<=MyCurrentFormationSize(); position++){
    dist = Distance2ToAdjustedForOffsidePositionHome(position,p);
    if (dist < minDist){
      num++;
    }
  }
  return num;
}

/********************************************************************************/

Vector FormationInfo::PositionHome     (Pnum position, Bool AdjustForOffsides)
{
  Vector home = CurrentFormation->GetPosition(position)->GetHome();
  if ( AdjustForOffsides ) 
    home = PositionToAdjustForOffsidePosition(home);
  return home;
}

/********************************************************************************/

float FormationInfo::PositionHomeBuffer(Pnum position)
{
  return CurrentFormation->GetPosition(position)->GetHomeBuffer();
}

/********************************************************************************/

Ptype   FormationInfo::PositionType      (Pnum position)
{
  return CurrentFormation->GetPosition(position)->GetType();
}

/********************************************************************************/

Pside   FormationInfo::PositionSide      (Pnum position)
{
  return CurrentFormation->GetPosition(position)->GetSide();
}

/********************************************************************************/

Rectangle FormationInfo::PositionHomeRange (Pnum position, Bool AdjustForOffsides)
{
  Rectangle range = *(CurrentFormation->GetPosition(position)->GetHomeRange());
  if ( AdjustForOffsides )
    range = RectangleToAdjustForOffsideRectangle(&range);
  return range;
}

/********************************************************************************/

Rectangle FormationInfo::PositionMaxRange (Pnum position, Bool AdjustForOffsides)
{
  Rectangle range = *(CurrentFormation->GetPosition(position)->GetMaxRange());
  if ( AdjustForOffsides )
    range = RectangleToAdjustForOffsideRectangle(&range);
  return range;
}

/********************************************************************************/

float FormationInfo::PositionHomeRangeWidth (Pnum position, Bool AdjustForOffsides)
{
  return PositionHomeRange(position,AdjustForOffsides).Width();
}

/********************************************************************************/

float FormationInfo::PositionHomeRangeHeight(Pnum position, Bool AdjustForOffsides)
{
  return PositionHomeRange(position,AdjustForOffsides).Height();
}

/********************************************************************************/

float FormationInfo::PositionMaxRangeWidth (Pnum position, Bool AdjustForOffsides)
{
  return PositionMaxRange(position,AdjustForOffsides).Width();
}

/********************************************************************************/

float FormationInfo::PositionMaxRangeHeight(Pnum position, Bool AdjustForOffsides)
{
  return PositionMaxRange(position,AdjustForOffsides).Height();
}

/********************************************************************************/

Vector FormationInfo::PlayerHome       (Unum player, Bool AdjustForOffsides )
{
  Vector home = CurrentFormation->GetPosition(PlayerPosition(player))->GetHome();
  if ( AdjustForOffsides ) 
    home = PositionToAdjustForOffsidePosition(home);
  return home;
}

/********************************************************************************/

float FormationInfo::PlayerHomeBuffer  (Unum player  )
{
  return CurrentFormation->GetPosition(PlayerPosition(player))->GetHomeBuffer();
}

/********************************************************************************/

Ptype   FormationInfo::PlayerType        (Unum player  )
{
  return CurrentFormation->GetPosition(PlayerPosition(player))->GetType();
}

/********************************************************************************/

Pside   FormationInfo::PlayerSide        (Unum player  )
{
  return CurrentFormation->GetPosition(PlayerPosition(player))->GetSide();
}

/********************************************************************************/

Pnum FormationInfo::PlayerPosition(Unum player)
{
  if ( player > SP_team_size || player < 0 ) my_error("That's not a player");
  return PlayerPositions[player];
}

/********************************************************************************/

Unum FormationInfo::PositionPlayer(Pnum position)
{
  if ( position > MyCurrentFormationSize() || position < 0 ) my_error("That's not a position");
  return PositionPlayers[position];
}

/********************************************************************************/

Time FormationInfo::PlayerPositionTime(Unum player)
{
  if ( player == MyNumber ) 
    return CurrentTime; /* Know my position for sure */

  return PlayerPositionTimes[player];
}

/********************************************************************************/

Pnum FormationInfo::MyPosition()
{
  return PlayerPosition(MyNumber);
}

/********************************************************************************/

Vector FormationInfo::MyPositionHome(Bool AdjustForOffsides)
{
  return PlayerHome(MyNumber,AdjustForOffsides);
}

/********************************************************************************/

Ptype   FormationInfo::MyPositionType()
{
  return PlayerType(MyNumber);
}

/********************************************************************************/

Pside FormationInfo::MyPositionSide()
{
  return PlayerSide(MyNumber);
}

/********************************************************************************/

float FormationInfo::MyPositionBuffer()
{
  return PositionHomeBuffer(MyPosition());
}

/********************************************************************************/

Rectangle FormationInfo::MyHomeRange(Bool AdjustForOffsides)
{
  return PositionHomeRange(MyPosition(),AdjustForOffsides);
}

/********************************************************************************/

Rectangle FormationInfo::MyMaxRange(Bool AdjustForOffsides)
{
  return PositionMaxRange(MyPosition(),AdjustForOffsides);
}

/********************************************************************************/

float FormationInfo::MyHomeRangeWidth(Bool AdjustForOffsides)
{
  return PositionHomeRangeWidth(MyPosition(),AdjustForOffsides);
}

/********************************************************************************/

float FormationInfo::MyHomeRangeHeight(Bool AdjustForOffsides)
{
  return PositionHomeRangeHeight(MyPosition(),AdjustForOffsides);
}

/********************************************************************************/

float FormationInfo::MyMaxRangeWidth(Bool AdjustForOffsides)
{
  return PositionMaxRangeWidth(MyPosition(),AdjustForOffsides);
}

/********************************************************************************/

float FormationInfo::MyMaxRangeHeight(Bool AdjustForOffsides)
{
  return PositionMaxRangeHeight(MyPosition(),AdjustForOffsides);
}

/********************************************************************************/

void FormationInfo::PrintMyHomeRange(Bool AdjustForOffsides)
{
  MyHomeRange(AdjustForOffsides).Print();
}

/********************************************************************************/

void FormationInfo::PrintMyMaxRange(Bool AdjustForOffsides)
{
  MyMaxRange(AdjustForOffsides).Print();
}

/********************************************************************************/

Bool   FormationInfo::IsWithinMyHomeRange(Vector p,Bool AdjustForOffsides)
{
  return MyHomeRange(AdjustForOffsides).IsWithin(p);
}

/********************************************************************************/

Bool   FormationInfo::IsWithinMyMaxRange(Vector p, Bool AdjustForOffsides)
{
  return MyMaxRange(AdjustForOffsides).IsWithin(p);
}
 
/********************************************************************************/
 
void  FormationInfo::ResetCurrentHome()
{
  Pnum position = MyPosition();

  if (position != Pnum_Unknown)
    SetCurrentHome(PositionHome(position));
}

/********************************************************************************/

void  FormationInfo::SetFormation(Ftype formation_number, Time time)
{
  FormationTime = time;
  Ftype old_form = FT_None;
  
  if ( MyCurrentFormation() != Formation_Unknown )
    old_form = MyCurrentFormationType();

  switch(formation_number){
  case FT_433: CurrentFormation = Formation_433; break;
  case FT_442: CurrentFormation = Formation_442; break;
  case FT_352: CurrentFormation = Formation_352; break;
  case FT_72:  CurrentFormation = Formation_72;  break;
  case FT_334: CurrentFormation = Formation_334; break;
  case FT_244: CurrentFormation = Formation_244; break;
  case FT_532: CurrentFormation = Formation_532; break;
  case FT_right: CurrentFormation = Formation_right; break;
  case FT_left: CurrentFormation = Formation_left; break;
  default: my_error("Not a valid formation number");
  }
  ResetCurrentHome();

  if ( old_form != FT_None && formation_number != old_form )
    Say(PMsg_none); /* Announce the new formation */
}

/********************************************************************************/

Bool FormationInfo::SetCurrentHome(Vector p)
{

/*  printf("%d trying to set to (%.1f,%.1f) -- %d\n",
	 MyNumber,x,y,IsWithinMyHomeRange(x,y)); */

  if ( !FieldRectangle.IsWithin(p) )
    my_error("Myhome shouldn't be off the field!!! (%.1f,%.1f)",p.x,p.y);

  if ( IsWithinMyHomeRange(p) ){
    CurrentHome = p;
    return TRUE;
  }

  /* Need to adjust x and y to be in homerange */
  /*CurrentHome = MyHomeRange().AdjustToWithin(p);
  return FALSE;*/
  CurrentHome = p;
  return TRUE;
}

/********************************************************************************/

Bool FormationInfo::SetCurrentHomeX(float x )
{
  return SetCurrentHome(Vector(x,MyCurrentHomeY()));
}

/********************************************************************************/

Bool FormationInfo::SetCurrentHomeY(float y )
{
  return SetCurrentHome(Vector(MyCurrentHomeX(),y));
}

/********************************************************************************/

void FormationInfo::SetMyMark(Unum mark,Time t)
{
  /* Check if mark is in the maxrange?  No--you can mark a player far away
     Just don't move to it until it's in range */
  /* Announce the player you're leaving?? */
  Say(PMsg_marking,mark);
  SetPositionMark(MyPosition(),mark,t);
}

/********************************************************************************/

void FormationInfo::SetPositionMark(Pnum pos, Unum mark, Time t)
{
  PositionMarks[pos] = mark;
  PositionMarkTimes[pos] = t;
}

/********************************************************************************/

Unum FormationInfo::PositionMark(Pnum pos)
{
  if ( PositionMarks[pos] == Unum_Unknown )
    return Unum_Unknown;
  else if ( PositionMarkTimes[pos] < SecondLastStartClockTime ) /* reset at kickoffs (also offsides?) */
    return Unum_Unknown;
  else if ( CurrentTime - PositionMarkTimes[pos] > CP_mark_persist_time )
    return Unum_Unknown;
  else
    return PositionMarks[pos];
}

/********************************************************************************/

Pnum FormationInfo::OpponentMarker(Unum opp)
{
  for (int i=1; i<=MyCurrentFormationSize(); i++)
    if ( PositionMark(i) == opp )
      return i;
  return Pnum_Unknown;
}

/********************************************************************************/

Bool FormationInfo::OpponentMarked(Unum opp)
{
  Pnum marking_pos = OpponentMarker(opp);

  if ( marking_pos == Pnum_Unknown ) return FALSE;
  if ( PositionPlayer(marking_pos) == Unum_Unknown ) return FALSE;
  return TRUE;
}

/********************************************************************************/

void FormationInfo::HearTeammateFormationPosition(Unum player, Ftype formation, 
						     Time formtime, Pnum position, Time postime)
{
  if ( formation == FT_None )
    return;

  if ( MyCurrentFormation() != Formation_Unknown && /* Need to be initialized */
       formation != MyCurrentFormationType() && formtime > FormationTime )
    SetFormation(formation,formtime);

  if ( postime > PlayerPositionTime(player) ){
    SetPlayerPosition(player,position,postime);
  }
}

/********************************************************************************/

int FormationInfo::UnitSize(int unit)
{
  return CurrentFormation->GetUnit(unit)->GetSize();
}

/********************************************************************************/

Pnum *FormationInfo::UnitMembers(int unit)
{
  return CurrentFormation->GetUnit(unit)->GetMembers();
}

/********************************************************************************/

Bool FormationInfo::IsUnitMember(int unit, Unum player)
{
  return CurrentFormation->IsUnitMember(unit, PlayerPosition(player));
}

/********************************************************************************/

Bool FormationInfo::IsUnitCaptain(int unit, Unum player)
{
  return CurrentFormation->IsUnitCaptain(unit, PlayerPosition(player));
}

/********************************************************************************/

Bool FormationInfo::IsMyUnit(int unit)
{
  return IsUnitMember(unit,MyNumber);
}

/********************************************************************************/

Bool FormationInfo::AmUnitCaptain(int unit)
{
  return IsUnitCaptain(unit,MyNumber);
}

/********************************************************************************/

Bool FormationInfo::IsMyUnitCaptain(Unum player)
{
  for (int i=0; i<CurrentFormation->GetNumUnits(); i++){
    if ( IsMyUnit(i) && IsUnitCaptain(i,player) )
      return TRUE;
  }
  return FALSE;
}

/********************************************************************************/

int *FormationInfo::CandidateReceivers(Pnum position, Fside LocationSide)
{
  return CurrentFormation->GetCandidateReceivers(position,LocationSide);
}

/********************************************************************************/

Bool FormationInfo::AmCandidateReceiverOfPlayer(Unum player)
{
  Pnum position = PlayerPosition(player);

  Fside side;
  if ( TeammatePositionValid(player) )
    side = TeammateLocationSide(player);
  else {
    switch ( PositionSide(position) ){
    case PS_Right:  side = FS_Right; break;
    case PS_Left:   side = FS_Left;  break;
    /* assuming symmetry of candidates and just call it FS_Right if it's a center position */
    case PS_Center: side = FS_Right; break;
    default: my_error("Don't know the position side");
    }
  }
  
  int *candidates = CandidateReceivers(position,side);
  for (int i=0; i<SP_team_size; i++){
    if ( candidates[i] == MyPosition() )
      return TRUE;
    if ( candidates[i] == position )
      return FALSE;
  }
  return FALSE;
}

/********************************************************************************/
							    
Bool FormationInfo::IsMoreImportantPosition(Pnum pos1, Pnum pos2)
{
  return CurrentFormation->IsMoreImportantPosition(pos1, pos2);
}

/********************************************************************************/

/* Return the closest unoccupied position */
Pnum FormationInfo::GetNewPosition()
{
  if (ChangePositions || MyPosition() == Pnum_Unknown){ 

    Pnum bestPosition=Pnum_Unknown;
    float dist,minDist=1000;
    for (int position=1; position<=MyCurrentFormationSize(); position++){
      if ( !PositionOccupied(position) 
#if 0
	   &&
	   (position==EncodePosition(GOALTENDER,CENTER) ||
	    position==EncodePosition(DEFENDER  ,RIGHT ) ||
	    position==EncodePosition(MIDFIELDER,RIGHT ) ||
	    position==EncodePosition(FORWARD   ,RIGHT ) ||
	    position==EncodePosition(FORWARD   ,CENTER) ) 
#endif
	                                                       ){
	dist = DistanceToPositionHome(position,MyPos());
	if (dist < minDist){ /* find closest unoccupied position */
	  bestPosition = position;
	  minDist = dist;
	}
      }
    }
    if ( bestPosition == Pnum_Unknown ) {
      for (int position=1; position<=MyCurrentFormationSize(); position++){
	printf("    *****    position %2d held by %2d\n",
	       position,PositionPlayer(position));
      }
      printf("    *****    I'm at (%.1f, %.1f)\n",MyX(),MyY());
      my_error("Some position should have been unoccupied");
    }
    return bestPosition;
  } 
  else {
    /* The following line allows more than one player to play the same 
       position when ChangePositions is FALSE                    */
/*    return MyPosition(); */
    my_error("Shouldn't be calling GetNewPosition if ChangePositions is FALSE");
    return 0;
  }
}

/********************************************************************************/

/* This must return the position number for true (?) */
Pnum FormationInfo::ConsiderPosition(Pnum position)
{
  int OnlyGoalieSwitch = FALSE;

  if (ChangePositions){

    if ( PositionOccupied(position) )
      return Pnum_Unknown;

    /*int teammate;
    float dist;
    dist = ClosestTeammateTo(PositionHomeX(position),
				  PositionHomeY(position),
			   &teammate);*/  /* Find the closest teammate to the position */
    if ( /* teammate == MyNumber && dist < 40 && */
	IsMoreImportantPosition(position,MyPosition()) ){
      /*printf("%d:%d Decided %d more important than %d\n",
	MyNumber,CurrentTime,position,MyPosition());*/
      if ( OnlyGoalieSwitch && position != 0 )
	return Pnum_Unknown;
      else 
	return position;
    }
    else 
      return Pnum_Unknown; /* Someone else will go */
  }
  else /* Don't change positions */
    return Pnum_Unknown;
}

/********************************************************************************/

/* Return the new position to go to or FALSE if stay the same */
Pnum FormationInfo::ConsiderNewPosition()
{
  if (ChangePositions){

    for (int position=1; position<=MyCurrentFormationSize(); position++){
      if ( position != MyPosition() && 
	   ConsiderPosition(position) != Pnum_Unknown &&
	   int_random(100)                             )
	return position;
    }
    return Pnum_Unknown;
  }
  else /* Keep my Position */
    return Pnum_Unknown;
}

/********************************************************************************/

Bool FormationInfo::InPosition()
{
  if ( MyPos().dist(MyCurrentHome()) > MyPositionBuffer() ) 
    return FALSE;
  else
    return TRUE;
}

/********************************************************************************/

Bool FormationInfo::InKickoffPosition()
{
  if ( MyPosition() == Pnum_Unknown )
    return FALSE;

  /*
  float goalX,goalY;
  if ( SetPlay && MyPositionInSetPlay() )
    GetMySetPlayXY(&goalX,&goalY);
  else{
    goalX = Mem->GetCurrentHomeX();
    goalY = Mem->GetCurrentHomeY();
  }
  */

  if ( MyPos().dist(PositionToKickoffPosition( MyCurrentHome() )) > MyPositionBuffer() )
    return FALSE;
  else 
    return TRUE;
}

/********************************************************************************/

Pnum FormationInfo::PositionOfMyLocation(Bool AdjustForOffsides)
{
  if ( !MyConf() ) my_error("Can't get position of unknown location");

  if ( stored_PositionOfMyLocation_time == CurrentTime )
    return stored_PositionOfMyLocation;

  stored_PositionOfMyLocation      = LocationToPosition(MyPos(),AdjustForOffsides);
  stored_PositionOfMyLocation_time = CurrentTime;
  return stored_PositionOfMyLocation;
}

/********************************************************************************/

Ptype FormationInfo::MyLocationsPositionType(Bool AdjustForOffsides)
{
  return PositionType(PositionOfMyLocation(AdjustForOffsides));
}

/********************************************************************************/

Pside FormationInfo::MyLocationsPositionSide(Bool AdjustForOffsides)
{
  return PositionSide(PositionOfMyLocation(AdjustForOffsides));
}
 
/********************************************************************************/

Bool FormationInfo::IsWithinMyHomeRange(char side, Unum number, Bool AdjustForOffsides)
{
  if ( !PlayerPositionValid(side,number) ) return FALSE;
  return IsWithinMyHomeRange(PlayerAbsolutePosition(side,number),AdjustForOffsides);
}

/********************************************************************************/

Bool FormationInfo::IsWithinMyMaxRange(char side, int number, Bool AdjustForOffsides)
{
  if ( !PlayerPositionValid(side,number) ) return FALSE;
  return IsWithinMyMaxRange(PlayerAbsolutePosition(side,number),AdjustForOffsides);
}

/********************************************************************************/

Bool FormationInfo::AmWithinMyHomeRange(Bool AdjustForOffsides)
{
  if ( !MyConf() ) return FALSE;
  return IsWithinMyHomeRange(MyPos(),AdjustForOffsides);
}

/********************************************************************************/

Bool FormationInfo::AmWithinMyMaxRange(Bool AdjustForOffsides)
{
  if ( !MyConf() ) return FALSE;
  return IsWithinMyMaxRange(MyPos(),AdjustForOffsides);
}

/********************************************************************************/

Bool FormationInfo::BallIsWithinMyHomeRange(Bool AdjustForOffsides)
{
  return IsWithinMyHomeRange(BallAbsolutePosition(),AdjustForOffsides);
}

/********************************************************************************/

Bool FormationInfo::BallIsWithinMyMaxRange(Bool AdjustForOffsides)
{
  return IsWithinMyMaxRange(BallAbsolutePosition(),AdjustForOffsides);
}

/********************************************************************************/

Bool FormationInfo::OpponentIsWithinPlayerHomeRange(Unum opponent, Unum player, Bool AdjustForOffsides)
{
  if ( !OpponentPositionValid(opponent) ) my_error("Don't know where opponent is");
  
  Pnum position = PlayerPosition(player);
  if ( position == Pnum_Unknown ) my_error("Don't know player's position");

  return PositionHomeRange(position,AdjustForOffsides).IsWithin(OpponentAbsolutePosition(opponent));
}

/********************************************************************************/

Bool FormationInfo::OpponentIsWithinPlayerMaxRange(Unum opponent, Unum player, Bool AdjustForOffsides)
{
  if ( !OpponentPositionValid(opponent) ) my_error("Don't know where opponent is");
  
  Pnum position = PlayerPosition(player);
  if ( position == Pnum_Unknown ) my_error("Don't know player's position");

  return PositionMaxRange(position, AdjustForOffsides).IsWithin(OpponentAbsolutePosition(opponent));
}

/********************************************************************************/

float FormationInfo::PlayerDistanceToPositionHome(Unum player, Pnum position, Bool AdjustForOffsides)
{
  if ( player != Unum_Unknown  && player != MyNumber && !TeammatePositionValid(player) )
    my_error("Don't know where the player is (PlayerDistanceToPositionHome)");
  
  Vector p;
  if ( player == Unum_Unknown || player == MyNumber ){
    p = MyPos();
  }
  else 
    p = TeammateAbsolutePosition(player);

  return DistanceToPositionHome(position,p,AdjustForOffsides);
}

/********************************************************************************/

void  FormationInfo::UpdateHome()
{
  Rectangle rect;
  Vector open_point;
  Unum mark;
  Bool AdjustForOffsides = TRUE;

  switch(HomeChangeMethod){
  case HC_Obey: /*printf("%d:%d home obey??\n",Mem->MyNumber,Mem->CurrentTime);*/ 
    AdjustForOffsides = TRUE;
    SetCurrentHome(MyPositionHome(AdjustForOffsides));
    break;
  case HC_Mark: 
    my_error("shouldn't use update home for marking players");
    UpdateMark();
    if ( !BallPositionValid() )
      my_error("Shouldn't be updating home if ball's not valid");
    mark = MyMark();
    if ( mark != Unum_Unknown && OpponentPositionValid(mark) ){
      SetCurrentHome(PointInBetween(OpponentAbsolutePosition(mark),BallAbsolutePosition(),3));
      break;
    }
    /* else shift */
  case HC_Get_Open:
    rect = MyHomeRange();
    rect = RectangleToAdjustForOffsideRectangle(&rect);
    open_point = LeastCongestedValidPointInRectangle(&rect);
    if ( rect.IsWithin(open_point) ){
      SetCurrentHome(open_point);
      break;
    }
    if ( open_point != rect.Center() + Vector(rect.Width(),0) )
      my_error("Why isn't the point in the rectangle???? (updatehome)");
    /* drop through to shift case */
  case HC_Shift:
    /* Was Shade to ball side */
    if ( !BallPositionValid() )
      my_error("Shouldn't be updating home if ball's not valid");

    AdjustForOffsides = TRUE;
    if ( MyPositionType() == PT_Goaltender ) AdjustForOffsides = FALSE;

    float defaultY = MyPositionY(AdjustForOffsides); 
    float defaultX = MyPositionX(AdjustForOffsides); 
    float ballX = BallX();
    float ballY = BallY();
    float ShadeFactor = MyHomeRangeHeight(AdjustForOffsides)/(SP_pitch_width);  /* Scale by field width     */
    float ShadeMag = (ballY - defaultY)*ShadeFactor;

    float ballDist = fabs(ballX - defaultX);
    ShadeMag *= Max(0,(1 - ballDist/(SP_pitch_length)));           /* Scale by ball's distance */
    SetCurrentHomeY(defaultY + ShadeMag);

    /* drop back if the ball's on the other side of the field */
    ShadeFactor = MyHomeRangeWidth(AdjustForOffsides)/SP_pitch_length;         /* Scale by field length    */
    ShadeMag = fabs((ballY - defaultY)*ShadeFactor);
    ShadeMag *= fabs(ballY)/(SP_pitch_width/2);               /* Scale by ball's position */

    if ( ballY*defaultY < 0 ) /* other side */
      SetCurrentHomeX(defaultX - ShadeMag);
    else
      SetCurrentHomeX(defaultX);
    //printf(" home %.1f  default %.1f  final %.1f \n",MyPositionX(),defaultX,MyCurrentHomeX());
    break;
  }

  if ( PlayMode == PM_Before_Kick_Off )
    SetCurrentHome( PositionToKickoffPosition(MyCurrentHome()) );
}

/********************************************************************************/

void  FormationInfo::UpdateMark()
{
  Unum ClosestOp;

  switch(MarkChangeMethod){
  case MC_Obey: break;
  case MC_Closest: 
    /* Might want to prevent switching too hastily--first look for mark? */
    /* if (Mark != Unum_Unknown)
       return; */
    ClosestOp = ClosestOpponent();
    if ( ClosestOp != Unum_Unknown && OpponentIsWithinMyMaxRange(ClosestOp,TRUE)
	 && NumTeammatesCloserTo(OpponentAbsolutePosition(ClosestOp)) == 0 )
      SetMyMark(ClosestOp,CurrentTime); 
    else
      SetMyMark(Unum_Unknown,CurrentTime);
    break;
  case MC_Open: my_error("Haven't defined MC_Open yet"); break;
  }
}

/********************************************************************************/

int FormationInfo::MyReceiverList(Pnum *ReceiverList)
{
  /* Go by my actual location, not my position */
  Pnum MyActualLocation = PositionOfMyLocation(TRUE);
  int *Rlist = CandidateReceivers(MyActualLocation,MyLocationSide());
  for (int i=0; i<SP_team_size; i++)
    ReceiverList[i] = Rlist[i]; /* Copy the list into here */

  int NumOptions = 0;
  while (ReceiverList[NumOptions] != Pnum_Unknown && NumOptions < SP_team_size)
    NumOptions++;
  
  return NumOptions;
}

