/* -*- Mode: C++ -*- */
#include "MemCommunicate.h"


/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

MessageStruct::MessageStruct()
{
  Next = Prev = NULL;
  data1 = data2 = 0;
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

MessageList::MessageList()
{
  head = tail = NULL;
}
 
/*****************************************************************************/

MessageList::~MessageList()
{
  MessageStruct *next;
  while ( head != NULL ){
    next = head->Next;
    delete head;
    head = next;
  }
}

/*****************************************************************************/

void MessageList::AddMessage(MessageStruct *msg)
{
  msg->Prev  = tail;
  tail       = msg;

  if ( head == NULL )
    head = tail;
  else
    tail->Prev->Next = tail;
}

/*****************************************************************************/

void MessageList::DeleteMessage(MessageStruct *msg)
{
  if ( msg == head ){
    head = msg->Next;
    if ( msg->Next != NULL )
      msg->Next->Prev = NULL;
  }
  else if ( msg == tail ){ 
    if ( head == NULL ) my_error("How can there be a tail with no head?!");
    tail = msg->Prev;
    /* This is safe because msg != head */
    msg->Prev->Next = NULL;
  }
  else{
    msg->Prev->Next = msg->Next;
    msg->Next->Prev = msg->Prev;
  }

  delete msg;
}




/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

void CommunicateInfo::Initialize()
{
  say_buffer = new char[SP_say_msg_size];

  for (int i=0; i<SP_say_msg_size; i++) /* Fill with spaces by default */
    say_buffer[i] = ' ';                

  MessageStateIndex = MessageBodyIndex = 0;

  FillMessageHead();

  MinDelay     = 0;                          /* If this is 0, communicate immediately by default */
  MaxDelay     = 10;                         /* Time after which I'll respond before acting more */
  HearInterval = SP_hear_decay/SP_hear_inc;  /* time steps need to wait before hearing again     */
}

/*****************************************************************************/

CommunicateInfo::~CommunicateInfo()
{
  delete say_buffer;
}

/*****************************************************************************/

int CommunicateInfo::EncodeKey(){
  int key;
  if (0)
    key = MyNumber * ( CurrentTime.t + closest_int( MyX() ) );
  else
    key = CurrentTime.t;
  return key;
}

/*****************************************************************************/

Time CommunicateInfo::DecodeKey(int key, int from, Vector from_pos){
  int sentTime;
  if (0)
    sentTime = key/from - closest_int( from_pos.x ); 
  else
    sentTime = key;
  return Time(sentTime,0);
}

/*****************************************************************************/

void CommunicateInfo::FillMessageHead()
{
  sprintf(say_buffer,"(say ");
  
  int say_length = 5; /*(say */
  message_buffer = say_buffer + say_length;
  MessageEndIndex = SP_say_msg_size - say_length;

  int index = 0;
  message_buffer[index] = MySide;
  index+=3;
  put_int( &message_buffer[index++], MyNumber );  

  MessageStateIndex = index;          /*%c %2d */
}

/*****************************************************************************/

Unum CommunicateInfo::ParseMessageHead()
{
  int index = 0;

  if ( hear_buffer[index++] != MySide )
    return Unum_Unknown;

  return (Unum) get_int(&hear_buffer[index]);
}

/*****************************************************************************/

void CommunicateInfo::FillMessageState()
{
  for (int i=MessageStateIndex; i<MessageBodyIndex; i++) /* Fill with spaces by default */
    message_buffer[i] = ' ';                

  int index = MessageStateIndex;

  KeyLength      =  7;
  FormLength     =  2;
  TimeLength     =  7;
  PastTimeLength =  3;
  PositionLength =  2;

  ConfLength     =  5; /* 1 int, 2 decimal places  */
  ConfPrecision  =  2; /* can move down to 1       */

  PosLength      =  3; /* 2 ints, 0 decimal place  */
  PosPrecision   =  0;

  VelLength      =  8; /* 1 int, 5 decimal places  */
  VelPrecision   =  5;

  /* Put in the encoded key */

  index += KeyLength;
  put_int( &message_buffer[index++], EncodeKey() );

  /* Put in the formation and time */

  index += FormLength;
  put_int( &message_buffer[index++], MyCurrentFormationType() );

  index += TimeLength;
  put_int( &message_buffer[index++], FormationTime.t );

  /* Put in my position for hearer's distance calculation                     */
  /* Can take this out and make the hearer index into the later position info */
  
  if ( !MyConf() ) 
    my_error("Can't do a say with all this state info if I don't know where I am");

  index+=PosLength;
  put_float( &message_buffer[index++], MyX(), PosPrecision );
  index+=PosLength;
  put_float( &message_buffer[index++], MyY(), PosPrecision );

  /* Ball info */

  index += ConfLength;
  put_float( &message_buffer[index++], BallPositionValid(), ConfPrecision );
  
  if ( BallPositionValid() ){
    index += PosLength;
    put_float( &message_buffer[index++], BallX(), PosPrecision );
    index += PosLength;
    put_float( &message_buffer[index++], BallY(), PosPrecision );
  }
  else 
    index += 2*PosLength + 2;

  index += ConfLength;
  put_float( &message_buffer[index++], BallVelocityValid(), ConfPrecision );

  if ( BallVelocityValid() ){
    index += VelLength;
    put_float( &message_buffer[index++], BallAbsoluteVelocity().x, VelPrecision );
    index += VelLength;
    put_float( &message_buffer[index++], BallAbsoluteVelocity().y, VelPrecision );
  }
  else 
    index += 2*VelLength + 2;

  /* Teammate info */

  int time;
  for ( int teammate=1; teammate<=SP_team_size; teammate++ ){
    index += PastTimeLength;
    if ( PlayerPositionTime(teammate) != 0 ){
      if ( PlayerPositionTime(teammate) < SecondLastStartClockTime ){
	//printf("subtraction would've been bad\n");
	put_int( &message_buffer[index++], -1 ); 
      }
      else if ( (time=CurrentTime-PlayerPositionTime(teammate)) >= pow(10,PastTimeLength) ){
	/* teammate probably not around -- convention: past_time < 0 ==> pos_time == 0*/
	printf("Teammate %d gone?\n",teammate);
	put_int( &message_buffer[index++], -1 ); 
      }
      else if ( put_int( &message_buffer[index++], CurrentTime - PlayerPositionTime(teammate)) > 
		PastTimeLength ){
	my_error("PastTimeLength too long %d %d %d (%d)",CurrentTime - PlayerPositionTime(teammate),
		 PlayerPositionTime(teammate).t,PlayerPositionTime(teammate).s,teammate);
	exit(0);
      }
    }
    else
      /* teammate probably not around -- convention: past_time < 0 ==> pos_time == 0*/
      put_int( &message_buffer[index++], -1 ); 
    index += PositionLength;

    if ( PlayerPosition(teammate) > SP_team_size || PlayerPosition(teammate) < 0 ) 
      my_error("what kind of position is that? %d\n",PlayerPosition(teammate));

    put_int( &message_buffer[index++], PlayerPosition(teammate) );
    
    index += ConfLength;
    float pos_conf = TeammatePositionValid(teammate);
    if ( teammate == MyNumber && pos_conf ) pos_conf = 1.0; /* I'm sure of my own position */
    put_float( &message_buffer[index++], pos_conf, ConfPrecision );

    if ( TeammatePositionValid(teammate) ){
      index += PosLength;
      put_float( &message_buffer[index++], TeammateX(teammate), PosPrecision );
      index += PosLength;
      put_float( &message_buffer[index++], TeammateY(teammate), PosPrecision );
    }
    else 
      index += 2*PosLength + 2;
  }

  /* Opponent info */

  for ( int opponent=1; opponent<=SP_team_size; opponent++ ){
    index += ConfLength;
    put_float( &message_buffer[index++], OpponentPositionValid(opponent), ConfPrecision );

    if ( OpponentPositionValid(opponent) ){
      index += PosLength;
      put_float( &message_buffer[index++], OpponentX(opponent), PosPrecision );
      index += PosLength;
      put_float( &message_buffer[index++], OpponentY(opponent), PosPrecision );
    }
    else 
      index += 2*PosLength + 2;
  }
  
  index++;
  MessageBodyIndex = index;
}

/*****************************************************************************/

Bool CommunicateInfo::ParseMessageState(Unum from, Time time)
{
  char *parsed_message = &hear_buffer[MessageStateIndex];

  /* Get the encoded key */

  int key = get_int(&parsed_message);

  /* Get the formation and time */

  Ftype formation = (Ftype) get_int(&parsed_message);
  Time  form_time = Time(get_int(&parsed_message),0);

  if ( formation > FT_left || formation < 0 ){
    my_error("Bad formation number %d from %d",formation,from);
    return FALSE;
  }
  if ( form_time > CurrentTime.t + 100 || form_time < -1 ){
    my_error("Bad formation time %d from %d",form_time.t,from);
    return FALSE;
  }

  if ( formation != FT_None ){
    if ( formation != MyCurrentFormationType() && form_time > FormationTime ){
      SetFormation(formation,form_time);
    }
  }
    
  /* Get from's position for  distance calculation             */
  /* Can take this out and  index into the later position info */

  float x = get_float(&parsed_message);
  float y = get_float(&parsed_message);
  Vector from_pos  = Vector(x,y);
  float  from_dist;
  
  if ( fabs(x) > 200 ) {my_error("bad x %.1f from %d",x,from); return FALSE; }
  if ( fabs(y) > 200 ) {my_error("bad y %.1f from %d",y,from); return FALSE; }

  /* Check to see if the key is valid */

  Time sent_time = DecodeKey(key,from,from_pos);
  if ( time.t - sent_time.t >= 60 ){
    my_error("Invalid message!");
    return FALSE;
  }

  /* ball info */

  float pconf = get_float(&parsed_message);
  if ( pconf > 1 || pconf < 0 ) {my_error("bad pconf %.1f from %d",pconf,from); return FALSE; }
  if ( pconf ){
    x = get_float(&parsed_message);
    y = get_float(&parsed_message);
    if ( fabs(x) > 200 ) {my_error("bad x %.1f from %d",x,from); return FALSE; }
    if ( fabs(y) > 200 ) {my_error("bad y %.1f from %d",y,from); return FALSE; }
    from_dist = from_pos.dist(Vector(x,y));
  }
  float vconf = get_float(&parsed_message);
  if ( vconf > 1 || vconf < 0 ) {my_error("bad vconf %.1f from %d",pconf,from); return FALSE; }
  float dx,dy;
  if ( vconf ){
    dx = get_float(&parsed_message);
    dy = get_float(&parsed_message);
    if ( fabs(dx) > SP_ball_speed_max ) {my_error("bad dx %.1f from %d",dx,from); return FALSE; }
    if ( fabs(dy) > SP_ball_speed_max ) {my_error("bad dy %.1f from %d",dy,from); return FALSE; }
  }

  if ( vconf && !ClockStopped ){ /* If clock's stopped, ignore ball movement info--it's still */
    if ( !pconf ){
      LogAction7(200,"Message from %d: pos (%.1f %.1f), vel (%.1f %.1f)",from,x,y,dx,dy);
      my_error("should know ball position if know velocity");
    }
    HearBall(x,y,pconf,dx,dy,vconf,from_dist,sent_time);
  }
  else if ( pconf )
    HearBall(x,y,pconf,from_dist,sent_time);

  /* teammate info */

  int past_time;
  Time pos_time;
  Pnum position;
  for ( int teammate=1; teammate<=SP_team_size; teammate++ ){
    past_time = get_int(&parsed_message);
    if ( past_time > CurrentTime.t+100 ) {my_error("bad past_time %d from %d",past_time,from); return FALSE; }
    if ( past_time >= 0 && CurrentTime.t - past_time > SecondLastStartClockTime.t )
      pos_time  = sent_time - past_time;
    else
      pos_time = 0;  /* teammate probably not around -- convention: past_time < 0 ==> pos_time == 0*/
    position  = (Pnum) get_int(&parsed_message);
    if (position > MyCurrentFormationSize() || position < 0)
      {my_error("bad position %d from %d",position,from); return FALSE; }
    if ( pos_time > PlayerPositionTime(teammate) && teammate != MyNumber )
      SetPlayerPosition(teammate,position,sent_time);

    pconf = get_float(&parsed_message);
    if ( pconf > 1 || pconf < 0 ) {my_error("bad pconf %.1f from %d",pconf,from); return FALSE; }
    if ( pconf ){
      x = get_float(&parsed_message);
      y = get_float(&parsed_message);     
      if ( fabs(x) > 200 ) {my_error("bad x %.1f from %d",x,from); return FALSE; }
      if ( fabs(y) > 200 ) {my_error("bad y %.1f from %d",y,from); return FALSE; }
      from_dist = from_pos.dist(Vector(x,y));
      if ( teammate != MyNumber ) /* Don't need to update for myself */
	HearPlayer(MySide,teammate,x,y,pconf,from_dist,sent_time);
    }
  }

  for ( int opponent=1; opponent<=SP_team_size; opponent++ ){
    pconf = get_float(&parsed_message);
    if ( pconf > 1 || pconf < 0 ) {my_error("bad pconf %.1f from %d",pconf,from); return FALSE; }
    if ( pconf ){
      x = get_float(&parsed_message);
      y = get_float(&parsed_message);     
      if ( fabs(x) > 200 ) {my_error("bad x %.1f from %d",x,from); return FALSE; }
      if ( fabs(y) > 200 ) {my_error("bad y %.1f from %d",y,from); return FALSE; }
      from_dist = from_pos.dist(Vector(x,y));
      HearPlayer(TheirSide,opponent,x,y,pconf,from_dist,sent_time);
    }
  }

  if ( MessageBodyIndex == 0 ){ /* Temporary for the first time around */
    while (*parsed_message != '*') parsed_message++;
    MessageBodyIndex = parsed_message - hear_buffer;
  }

  return TRUE;
}

/*****************************************************************************/

Bool CommunicateInfo::FillMessageBody(MessageStruct *msg)
{
  for (int i=MessageBodyIndex; i<MessageEndIndex; i++) /* Fill with spaces by default */
    message_buffer[i] = ' ';                

  int index = MessageBodyIndex;

  int TargetTypeLength  = 1;
  int TargetLength      = 2;
  int MessageTypeLength = 2;
  int MessageDataLength = 3;

  message_buffer[index++] = '*';
  message_buffer[index++] = ' ';

  switch( msg->target_type ){
  case TT_All: case TT_Player: case TT_Position: case TT_Unit: break;
  default: my_error("That's not a target type! %d",msg->target_type);
  }
  
  index+=TargetTypeLength;
  put_int( &message_buffer[index++], msg->target_type);

  index+=TargetLength;
  put_int( &message_buffer[index++], msg->target);

  if ( msg->respond == TRUE ){ /* need to put in the appropriate response type, data */

    switch( msg->message_type ){ /* Check if message is still appropriate */
    case PMsg_ping_ball:     if ( !BallPositionValid() )               return FALSE;  break; 
    case PMsg_ping_teammate: if ( !TeammatePositionValid(msg->data1) ) return FALSE;  break;
    case PMsg_ping_opponent: if ( !OpponentPositionValid(msg->data1) ) return FALSE;  break;
    case PMsg_Setplay_Ready: if ( !SetPlay || Player_Setplay_Ready[MyNumber] != SPnum_Unknown ) /* got ack */
                                                                       return FALSE;  break;
    case PMsg_Setplay_Starter:								       
    case PMsg_Setplay_Ping_Starter: if ( !SetPlay )                    return FALSE;  break;
    default: ;
    }

    switch( msg->message_type ){
    case PMsg_ping:
    case PMsg_ping_ball: 
    case PMsg_ping_teammate: 
    case PMsg_ping_opponent: 
    case PMsg_leaving_position:
    case PMsg_already_there:
    case UMsg_assign_position:
      index+=MessageTypeLength;
      put_int( &message_buffer[index++], PMsg_none);

      index+=MessageDataLength;
      put_int( &message_buffer[index++], 0);

      index+=MessageDataLength;
      put_int( &message_buffer[index++], 0);
      break;
    case PMsg_Setplay_Starter:
    case PMsg_Setplay_Ping_Starter:
      index+=MessageTypeLength;
      put_int( &message_buffer[index++], PMsg_Setplay_Starter);

      index+=MessageDataLength;
      if ( Confirmed_Setplay_Starter == MyNumber )
	put_int( &message_buffer[index++], 1);
      else
	put_int( &message_buffer[index++], 0);

      index+=MessageDataLength;
      put_int( &message_buffer[index++], 0);
      break;
    case PMsg_Setplay_Ready:
      index+=MessageTypeLength;
      put_int( &message_buffer[index++], PMsg_Setplay_OK_Ready);

      index+=MessageDataLength;
      put_int( &message_buffer[index++], 0);

      index+=MessageDataLength;
      put_int( &message_buffer[index++], 0);
      break;
    default: my_error("No response for that message type");
    }
  }
  else{ /* respond == FALSE:  it's an original message */
    index+=MessageTypeLength;
    put_int( &message_buffer[index++], msg->message_type);

    index+=MessageDataLength;
    put_int( &message_buffer[index++], msg->data1);

    index+=MessageDataLength;
    put_int( &message_buffer[index++], msg->data2);
  }

  message_buffer[index++] = ')';
  message_buffer[index++] = '\0';

  return TRUE;
}

/*****************************************************************************/

void CommunicateInfo::ParseMessageBody(Unum from, Time time)
{
  char *parsed_message = &hear_buffer[MessageBodyIndex];

  parsed_message+=2; /* '* ' */

  TargetType ttype  = (TargetType) get_int( &parsed_message );
  int        target = get_int( &parsed_message );
  MessageType mtype = (MessageType) get_int( &parsed_message );

  int data1 = get_int( &parsed_message );
  int data2 = get_int( &parsed_message );

  TargetType response_target_type; 
  int        response_target;      

  /* Ignore messages not for me */
  switch ( ttype ){
  case TT_Player:   
    if ( target > SP_team_size ) my_error("Player %d???",target);
    if ( target != MyNumber ){
      LogAction4(200,"Heard message from %d -- not for me (for %d)",from,target);
      return;
    }
    else { response_target_type = TT_Player; response_target = from; }
    break;
  case TT_Position: 
    if ( target != MyPosition() ){
      LogAction4(200,"Heard message from %d -- not for me (for position %d)",from,target);
      return; 
    }
    else { response_target_type = TT_Player; response_target = from; }
    break;
  case TT_Unit:     
    if ( !IsMyUnit(target) ){
      LogAction4(200,"Heard message from %d -- not for me (for unit %d)",from,target);
      return; 
    }
    else { response_target_type = TT_Unit;   response_target = target; }
    break;
  case TT_All:
    response_target_type = TT_All; response_target = target;
    break;
  default: 
    my_error("Not a valid target type %d",ttype);
  }

  Bool delay=FALSE, respond=FALSE, stagger=FALSE, NoDelay=FALSE;
  Pnum position;
  Unum player, mark;

  switch ( mtype ){
  case CMsg_new_coach:                                        break;
  case PMsg_none:                                             break;    
  case PMsg_ping:                        respond = TRUE;      break;
  case PMsg_ping_ball: 
    if ( BallPositionValid() )           respond = TRUE;      
    response_target_type = TT_Player;
    NoDelay = TRUE;
    response_target = from;                                   break;
  case PMsg_ping_teammate:
    player = (Unum) data1;
    if ( TeammatePositionValid(player) ) respond = TRUE;      
    response_target_type = TT_Player;
    NoDelay = TRUE;
    response_target = from;                                   break;
  case PMsg_ping_opponent:
    player = (Unum) data1;
    if ( OpponentPositionValid(player) ) respond = TRUE;      
    response_target_type = TT_Player;
    NoDelay = TRUE;
    response_target = from;                                   break;
  case PMsg_ready_to_pass:     /* indicate passer */          break;
  case PMsg_ready_to_receive:  /* indicate receiver */        break;
  case PMsg_passing_decision:  
    team_passer = from;
    team_receiver = data1;
    team_pass_time = time;                                    break;
  case PMsg_my_ball:           /* indicate ball */            break;
  case PMsg_leaving_position:  
    position = (Pnum) data1;
    if ( PlayerPosition(from) != Pnum_Unknown )
      my_error("should have already cleared this position");
    if ( PositionPlayer(position) == from ){
      my_error("should have already cleared this player");
    }
      
    if ( position != MyPosition() )
      position = ConsiderPosition(position);
    if ( position != Pnum_Unknown && position != MyPosition() ){ 
      /* If I'm going to take the freed position */
      SetMyPosition(position);      
      respond = TRUE;
    }                                                         break;
  case PMsg_already_there:
    if ( PlayerPosition(from) == MyPosition() ){
      SetMyPosition(GetNewPosition());  
      respond = TRUE;
    }
    break;
  case PMsg_tired: SetTeammateTired(from,time);               break;
  case PMsg_Setplay_Starter:
    if ( data1 == TRUE ){
      if ( Confirmed_Setplay_Starter == MyNumber ){
	EndSetPlay();
	respond = TRUE;
	delay = TRUE;
      }
      else 
	Confirmed_Setplay_Starter = from;
    }
    else 
      EndSetPlay();                                           break;
  case PMsg_Setplay_Ready:
    if ( InSetPlay )
      Player_Setplay_Ready[from] = data1;        
    if ( Confirmed_Setplay_Starter == MyNumber ){
      response_target_type = TT_Player; 
      response_target = from; 
      respond = TRUE;
    }                                                         break;
  case PMsg_Setplay_OK_Ready:
    if ( PlayerPosition(from) != SetPlayStarterPosition() )
      my_error("why isn't it the starter responding? (%d responding not %d)", 
	       from, PositionPlayer(SetPlayStarterPosition()));
    Player_Setplay_Ready[MyNumber] = MySetPlayPosition();     break;
  case PMsg_Setplay_Ping_Starter:
    respond = TRUE;                                           break;
  case PMsg_marking:
    if ( data1 != Unum_Unknown && MyMark() == data1 ) 
      SetMyMark(Unum_Unknown,time);
    SetPositionMark(PlayerPosition(from),data1,time);         break;
  case UMsg_assign_mark:
    mark   = (Unum) data1;
    player = (Unum) data2;
    if ( ttype != TT_Unit ) 
      my_error("mark message should go to unit");
    /* If not for me, keep track of who's marking whom? */
    if ( player == MyNumber && IsUnitCaptain(target,from) ){
      SetMyMark(mark,time);  /* Announce player that's being left? */
      MarkChangeMethod = MC_Obey;
      HomeChangeMethod = HC_Mark;
    }                                                         break;
  case UMsg_assign_position:
    position = (Pnum) data1;
    player =   (Unum) data2;
    if ( ttype != TT_Unit ) 
      my_error("mark message should go to unit");
    /* If not for me, keep track of who's assigned to a position? */
    if ( player == MyNumber && IsUnitCaptain(target,from) ){
      SetMyPosition(position);  /* Announce player that's being left? */
      respond = TRUE;
    }                                                         break;
  default: my_error("Unknown message type");
  }

  if ( respond && !NoDelay && ( ttype == TT_All || ttype == TT_Unit ) ){
    delay = TRUE;
    stagger = TRUE;
  }

  int delay_time = MinDelay;  /* The default */
  if ( delay ){
    delay_time = HearInterval;
    if ( stagger ){
      if ( MyNumber > from )
	delay_time += (((MyNumber-from)-1)*2   )*HearInterval;
      else /* MyNumber < from */
	delay_time += (((from-MyNumber)-1)*2 +1)*HearInterval;
    }
  }

  if ( respond )
    AddMessage(time,delay_time,TRUE,response_target_type,response_target,mtype,data1,data2);

  LogAction4(200,"Heard message of type %d from %d",mtype,from);
}

/*****************************************************************************/

Bool CommunicateInfo::fill_message(MessageStruct*msg)
{
  FillMessageState();
  /* if false, no longer need to send the message */
  return FillMessageBody(msg);
}

/*****************************************************************************/

void CommunicateInfo::ParsePlayerSound(char *msg, Time time)
{
  if ( msg[1] == 's' ){    /* "self" */
    LastSayTime = time;
    return;
  }

  float ang = get_float(&msg);
  hear_buffer = msg+1; /* advance past the space */

  Unum from = ParseMessageHead();
  if ( from == Unum_Unknown ){
    /* store opp message for learning to parse it */
    /* can use the angle to learn about opp pos   */
    ang++; /* take out */
    return;
  }
  else{ /* it's from a teammate */
    LastHearTime = time;
    LastSpeaker  = from;
  }

  if ( !ParseMessageState(from, time) ) 
    return;

  ParseMessageBody(from, time);
}

/*****************************************************************************/

void CommunicateInfo::AddMessage(Time time, int delay, Bool respond, TargetType ttype, int target, 
				 MessageType mtype, int d1, int d2)
{
  MessageStruct *msg = new MessageStruct;
  msg->time = time;
  msg->delay = delay;
  msg->target_type = ttype;
  msg->target = target;
  msg->message_type = mtype;
  msg->data1 = d1;
  msg->data2 = d2;
  msg->respond = respond;
  MsgList.AddMessage(msg);
  LogAction3(200,"Planning to say message of type %d",mtype);
}

/*****************************************************************************/

void CommunicateInfo::Say(MessageType mtype, int d1, int d2)
{
  AddMessage(CurrentTime,0,FALSE,TT_All,0,mtype,d1,d2);
}

/*****************************************************************************/

void CommunicateInfo::SayToPlayer(Unum player, MessageType mtype, int d1, int d2)
{
  AddMessage(CurrentTime,0,FALSE,TT_Player,player,mtype,d1,d2);
}

/*****************************************************************************/

void CommunicateInfo::SayToPosition(Pnum position, MessageType mtype, int d1, int d2)
{
  AddMessage(CurrentTime,0,FALSE,TT_Position,position,mtype,d1,d2);
}

/*****************************************************************************/

void CommunicateInfo::SayToUnit(int unit, MessageType mtype, int d1, int d2)
{
  if ( !IsMyUnit(unit) )
    my_error("Can't SayToUnit unless I'm in the unit");
  AddMessage(CurrentTime,0,FALSE,TT_Unit,unit,mtype,d1,d2);
}

/*****************************************************************************/

Bool CommunicateInfo::Communicate()
{
  /* cycle through the messagelist to see if any are ready      */
  /* take the first one that's ready---it's been on the longest */

  if ( !CP_communicate ) return FALSE;

  MessageStruct *msg = MsgList.head;

  if ( CurrentTime == LastComCheckTime ) 
    return FALSE;
  else
    LastComCheckTime = CurrentTime;

  if ( CurrentTime - LastSayTime < HearInterval ) 
    return FALSE;

  /* This is conservative:  could speak anyway and hope it's heard                              */
  /* Could also reason about who LastSpeaker was and if the target is likely to be able to hear */
  /* if ( CurrentTime - LastHearTime < HearInterval ) 
    return FALSE; */

  /* If ball position isn't valid, might want to ping it here so it happens before behave */
  /*if ( !ClockStopped && !BallPositionValid() )
    Say(PMsg_ping_ball); */

  if ( (LastSayTime < SecondLastStartClockTime || CurrentTime - LastSayTime > CP_max_say_interval) &&
       MsgList.head == NULL ) /* nothing waiting */
    Say(PMsg_none);

  if ( LastHearTime.t > 0 && CurrentTime - LastHearTime > HearInterval*3 && MsgList.head == NULL ) 
    /* nothing heard for a time */
    Say(PMsg_none);

  if ( !MyConf() ) /* Just because I have all the state -- could make a state-free say */
    return FALSE;
  
  while ( msg != NULL ){
    if ( CurrentTime - msg->time >= msg->delay ){
      Bool still_send = fill_message(msg);
      MsgList.DeleteMessage(msg);
      if ( still_send )  /* else, keep looping */
	return TRUE; 
    }
    msg = msg->Next;
  }

  return FALSE;
}

/*****************************************************************************/

Bool CommunicateInfo::MessageTypePending(MessageType mtype)
{
  MessageStruct *msg = MsgList.head;
  while (msg != NULL){
    if ( msg->message_type == mtype )
      return TRUE;
    msg = msg->Next;
  }
  return FALSE;
}
