/* -*- Mode: C -*- */

/* test.C
 * CMUnited98 (soccer client for Robocup98)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1998 Peter Stone
 *
 * CMUnited-98 was created by Peter Stone, Manuela Veloso, and Patrick Riley
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* test.C contains functions that demonstrate our client's abilities
 * They should be called from behave.C
 */

#include "test.h"
#include "Memory.h"
#include "client.h"
#include "kick.h"
#include "dribble.h"

#ifndef RELEASE_VERSION
#include "intercept.h"
#endif

/*****************************************************************************************/

void test_dribble_to_goal()
{
  if ( Mem->BallKickable() && Mem->MarkerDistance(Mem->RM_Their_Goal) > 18 )
    SimpleDribbleTo(Mem->MarkerPosition(Mem->RM_Their_Goal));
  else if ( Mem->BallPositionValid() && !Mem->MyInterceptionAble() )
    /* go anyway */
    test_go_to_point( Mem->BallAbsolutePosition(),0 );
  else 
    test_go_to_ball();
}

/*****************************************************************************************/

void test_random_movement_in_rectangle(Rectangle *rect)
{
  static int First_time = TRUE;
  if ( First_time ){
    Vector point = rect->random();
    point = Mem->PositionToKickoffPosition(point);
    move(point.x,point.y);
    First_time=FALSE;
    return;
  }

  if ( !Mem->MyConf() ) {
    turn(30);
    return;
  }

  if ( rect->DistanceToEdge(Mem->MyPos()) < 5 )
    test_go_to_point(rect->Center(),0);
  else if ( Mem->MyStamina() >= Mem->EffortDecThreshold )
    test_random_movement();
}

/*****************************************************************************************/

void test_1v1()
{
  static int First_time = TRUE;
  if ( First_time ){
    move(-10,0);
    First_time=FALSE;
    return;
  }

  if ( Mem->BallPositionValid() ){
    if ( Mem->BallKickable() ){
      AngleDeg ang = 0;
      Unum opponent;
      if ( (opponent = Mem->ClosestOpponent()) != Unum_Unknown ){
	ang = Mem->OpponentAngle(opponent) + 180;
	NormalizeAngleDeg(&ang);
      }

      if ( opponent == Unum_Unknown ){
	if ( Mem->FirstActionOpSinceLastSight )
	  turn(Mem->MyViewAngle() * 2);
      }
      else if ( Mem->OpponentPositionValid(opponent) < .9 )
	turn( Mem->OpponentAngle(opponent) );
      else
	TurnballTo(ang);
    }
    else
      test_go_to_ball();
  }
  else
    test_face_ball();
}

/*****************************************************************************************/

void test_volley()
{
  static int First_time = TRUE;
  if ( First_time ){
    move(-20,0);
    First_time=FALSE;
    return;
  }

  if ( Mem->BallPositionValid() && Mem->BallX() < -1 )
    test_go_to_ball();
  else
    test_go_to_point(Vector(-20,0),3);
}

/*****************************************************************************************/

void test_go_to_ball(AngleDeg kick_angle)
{
  if ( Mem->BallKickable() ){
    if ( !Mem->TestVersion ){
      if (!smart_kick_hard(kick_angle, KM_Moderate))
	my_error("test_go_to_ball: kick failed\n");
    }
    else
      kick(100,kick_angle);
  }
  else if ( Mem->MyConf() && Mem->BallPositionValid() ){
    if (Mem->MyInterceptionAble())
      test_go_to_point(Mem->MyInterceptionPoint(),0);	
    else 
      test_go_to_point(Mem->BallAbsolutePosition(),Mem->SP_kickable_area);
  }
  else 
    turn(90);
}

/*****************************************************************************************/

void test_go_to_ball()
{
  if ( Mem->MyConf() ) 
    test_go_to_ball(Mem->MarkerAngle(Mem->RM_Their_Goal));
  else
    turn(90);
}

/*****************************************************************************************/

void test_go_to_point(Vector p, float buffer, float dash_power )
{
  if ( !Mem->MyConf() ) {
    turn(Mem->MyViewAngle());
    return;
  }

  if ( Mem->DistanceTo(p) < buffer ){
    //turn(30);
    test_face_ball();
    return;
  }

  float target_ang  = Mem->AngleTo(p);
  float target_dist = Mem->DistanceTo(p);

  if ( 1 && !Mem->ClockStopped ){ /* dodge players */
    PlayerObject *player;
    float    dodge_dist = Min(Mem->CP_dodge_distance_buffer,target_dist);
    AngleDeg dodge_ang  = Mem->CP_dodge_angle_buffer;
    if ( (player = Mem->GetPlayerWithin( dodge_dist, dodge_ang, 0, target_ang - dodge_ang))
	 != NULL ){
      if ( Mem->NumPlayersWithin( dodge_dist, 2*dodge_ang) ){
	/* Target close, so no players will be within in the next iteration ==> dash */
	Vector new_target = Mem->Polar2Gpos(.8,player->get_ang() + 90);
	test_go_to_point(new_target,0);
      }
      else{
	dash(Mem->CP_dodge_power);
      }
      return;
    }
    if ( (player = Mem->GetPlayerWithin( dodge_dist, dodge_ang, 0, target_ang + dodge_ang))
	 != NULL ){
      if ( Mem->NumPlayersWithin( dodge_dist, 2*dodge_ang) ){
	/* Target close, so no players will be within in the next iteration ==> dash */
	Vector new_target = Mem->Polar2Gpos(.8,player->get_ang() - 90);
	test_go_to_point(new_target,0);
      }
      else{
	dash(Mem->CP_dodge_power);
      }
      return;
    }
  }

  if ( fabs(target_ang) > Mem->CP_max_go_to_point_angle_err ){
    turn(target_ang);
    return;
  }

  if ( Mem->MyStamina() >= Mem->EffortDecThreshold )
    dash( Min( dash_power, Mem->MyStamina()-Mem->EffortDecThreshold ) );
  else
    {my_stamp; printf("recovering\n");} //turn(180);
}

/*****************************************************************************************/

void test_face_ball()
{
  if ( Mem->BallPositionValid() ) 
    turn( Mem->BallAngle() );
  else 
    turn(Mem->MyViewAngle());
}

/*****************************************************************************************/

void test_random_movement()
{
  static int First_time = FALSE;
  if ( First_time ){
    move(range_random(-50,0),range_random(-Mem->SP_pitch_width/2,Mem->SP_pitch_width/2));
    First_time=FALSE;
    return;
  }

  /* if      ( !int_random(100) ) change_view(VW_Wide);
     else if ( !int_random(100) ) change_view(VW_Normal);
     else if ( !int_random(100) ) change_view(VW_Narrow); */

  if ( Mem->ClockStopped ) return;

  if ( Mem->BallKickable() ){
    kick(range_random(0,100),Mem->BallAngle());
    return;
  }

  if ( Mem->MyConf() && Mem->BallPositionValid() && Mem->BallDistance() < 9 ){
    if ( fabs(Mem->BallAngle()) > 1 ) 
      turn( Mem->BallAngle() );
    else
      dash(100);
    return;
  }

  if ( int_random(4) )
    dash(range_random(20,30));
  else 
    turn(range_random(-90,90));

  return;
}

/*****************************************************************************************/

void test_straight_to_ball()
{
  static int First_time = TRUE;
  if ( First_time ){
    if (Mem->MySide == 'l')
      turn(90);
    else
      turn(-90);
    First_time=FALSE;
    return;
  }

  if ( Mem->CurrentTime.t && Mem->CurrentTime.t < 10 && Mem->GetBall()->pos_valid() ){
    turn(Mem->GetBall()->get_ang());
  }
  else 
    dash(Mem->CurrentTime.t%100); 
  return;
}

/*****************************************************************************************/

void test_run_straight()
{
  static Bool GO = TRUE;
  
  if ( !(Mem->CurrentTime.t%200) ) GO = TRUE;

  if ( Mem->MyStamina() > 600 && GO ) 
    dash(60);
  else if ( Mem->CurrentTime.t ){
    GO = FALSE;
    printf("stamina = %f, effort = %f\n",Mem->MyStamina(),Mem->MyEffort());
  }
}

/*****************************************************************************************/

void test_turn_and_dash_slow()
{
  static int First_time = TRUE;
  if ( First_time ){
    if (Mem->MySide == 'l')
      turn(90);
    else
      turn(-90);
    First_time=FALSE;
    return;
  }

  if ( Mem->CurrentTime.t%5 || !Mem->CurrentTime.t ) return;

  if ( !(Mem->CurrentTime.t%50) )
    turn(40);
  else
    dash(100);

  return;
}

/*****************************************************************************************/

void test_print_ball()
{
  my_stamp;

  if ( Mem->BallPositionValid() ) {
    printf(" Ball at %.1f %.1f (%.2f)",Mem->BallX(), Mem->BallY(), Mem->BallPositionValid()); 
    if ( Mem->BallVelocityValid() )
          printf("     Speed %.2f dir %.1f (%.2f)",
		 Mem->BallSpeed(), Mem->BallAbsoluteHeading(), Mem->BallVelocityValid()); 
    else
      printf("    velocity not valid");
  }
  else 
    printf(" Ball not valid");

  printf("\n");
}

/*****************************************************************************************/

void test_print_positions()
{
  Bool me = TRUE;
  Bool ball = FALSE;
  Bool players = TRUE;

  if (me){
    my_stamp; printf("at %d I'm at (%.1f %.1f) facing %.1f with velocity (%f %.1f)\n",
		     Mem->CurrentTime.t,Mem->MyX(),Mem->MyY(),Mem->MyAng(),Mem->MySpeed(),Mem->MyDir());
  }

  if (ball){
    if (Mem->GetBall()->pos_valid())
      printf("Ball: (%f %f)       ",Mem->GetBall()->get_x(),Mem->GetBall()->get_y());

    if (Mem->GetBall()->vel_valid())
      printf("velocity: (%f %f)\n",Mem->GetBall()->get_speed(),Mem->GetBall()->get_abs_heading());
    else if (Mem->GetBall()->pos_valid())
      printf("\n"); 
  }

  if (players){
    for (int i=0; i<Mem->NumTeammates(); i++){
      my_stamp;
      printf("T %d at %.1f %.1f facing %.1f\n",Mem->Teammates()[i]->unum,Mem->Teammates()[i]->get_x(),Mem->Teammates()[i]->get_y(),Mem->Teammates()[i]->get_abs_face());
    }
    for (int i=0; i<Mem->NumOpponents(); i++){
      my_stamp;
      printf("O %d at %.1f %.1f facing %.1f\n",Mem->Opponents()[i]->unum,Mem->Opponents()[i]->get_x(),Mem->Opponents()[i]->get_y(),Mem->Opponents()[i]->get_abs_face());
    }
    for (int i=0; i<Mem->NumTeamlessPlayers(); i++){
      my_stamp;
      printf("? %d at %.1f %.1f facing %.1f\n",Mem->TeamlessPlayers()[i]->unum,Mem->TeamlessPlayers()[i]->get_x(),Mem->TeamlessPlayers()[i]->get_y(),Mem->TeamlessPlayers()[i]->get_abs_face());
    }
  }
}



