/* -*- Mode: C++ -*- */

/* kick.C
 * CMUnited98 (soccer client for Robocup98)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1998 Peter Stone
 *
 * CMUnited-98 was created by Peter Stone, Manuela Veloso, and Patrick Riley
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* kick.C contains functions for kicking the ball
   Created by Patrick Riley
 */


#include <math.h>
#include "Memory.h"
#include "client.h"
#include "kick.h"
#include "test.h"
#include "behave.h"

#ifdef DEBUG_OUTPUT
#define DebugKick(x) x
#else
#define DebugKick(x) 
#endif

int DoTurnKickCommand(TurnKickCommand com)
{
  if (com.time != Mem->CurrentTime) {
    my_error("DoTurnKickCommand- told to do command not set this cycle");
    return 0;
  }
    
  switch (com.type) {
  case CMD_dash:
    DebugKick(printf("DoTurnKickCommand: dash\n"));
    dash(com.power);
    return 1;
  case CMD_turn:
    DebugKick(printf("DoTurnKickCommand: turn\n"));
    turn(com.angle);
    return 1;
  case CMD_kick:
    DebugKick(printf("DoTurnKickCommand: kick\n"));
    kick(com.power, com.angle);
    return 1;

  default:
    my_error("PatInfo::DoTurnKickCommand- unimplemented type!");
    return 0;
  }
  
}


/* decides if we can kick staright to the decired point around the player
   without a collision.
   (EndDist, dir) is a relative polar vector for the ball's final position
   closeMarg is what the radius of the player is considered to be */
int is_straight_kick(float dir, float EndDist, float closeMarg)
{
  Vector btraj = Polar2Vector(EndDist, dir) -
    Mem->BallRelativePosition();
  float ang;
  float dist;
  int res;

  DebugKick(printf("    isStriaght ball abs pos mod: %f\t dir: %f\n",
	 Mem->BallAbsolutePosition().mod(), Mem->BallAbsolutePosition().dir()));
  DebugKick(printf("    isStriaght ball rel pos mod: %f\t dir: %f\n",
	 Mem->BallRelativePosition().mod(), Mem->BallRelativePosition().dir()));
  DebugKick(printf("    isStriaght btraj mod: %f\t dir: %f\n", btraj.mod(), btraj.dir()));
  /* Apply the law of cosines to the anle formed by the player's center(A),
     the ball's current position(B), and the ball target position(C).
     The angle calculated is ABC */
  ang = ACos( (Sqr(EndDist) - Sqr(btraj.mod()) -
	       Sqr(Mem->BallDistance()))/
	      (-2 * Mem->BallDistance() * btraj.mod()) );
  DebugKick(printf("   isStraight ang: %f\n", ang));
  if (fabs(ang) > 90) {
    DebugKick(printf("    isStraight: Obtuse!\n"));
    return 1; /* obtuse angle implies definately straight */
  }
  /* get the height of the triangle, ie how close to the player the ball will
     go */
  dist = Sin(ang) * Mem->BallDistance(); 
  DebugKick(printf("   isStraight dist: %f\n", dist));
  res = (fabs(dist) > closeMarg);
  return ( res );  
} 

/* picks a rotation (CW or CCW) to turnball based on the nearest opponent or
   teamless player */
TurnDir RotToAvoidOpponent(float abs_dir)
{
  TurnDir rot;
  float dist = HUGE;
  float ang;
  Unum opp = Mem->ClosestOpponent();
  if (opp != Unum_Unknown) {
    dist = Mem->OpponentDistance(opp);
    ang = Mem->OpponentAngle(opp);
  }
  if (Mem->NumTeamlessPlayers() > 0) {
    Vector pos = Mem->ClosestTeamlessPlayerPosition();
    float d = Mem->DistanceTo(pos);
    if (d < dist) {
      dist = d;
      ang = Mem->AngleTo(pos);
    }
  }
  if (dist < Mem->CP_turnball_opp_worry_dist) {
    /* there is an opponent near enough to worry about */
    DebugKick(printf("In RotToAvoidOpponent, avoiding opponent\n"));
    if (GetNormalizeAngleDeg(ang - (abs_dir - Mem->MyAng()))
	< 0)
      rot = TURN_CCW;
    else
      rot = TURN_CW;
  } else
    rot = TURN_CLOSEST;
  return rot;
}

/* Picks the rotation direction (CW or CCW) such that the ball has to travel
   the least distance */
TurnDir RotClosest(float abs_dir)
{
  AngleDeg cw_ang = (abs_dir - Mem->MyAng()) - Mem->BallAngle();
  AngleDeg ccw_ang = Mem->BallAngle() - (abs_dir - Mem->MyAng());
  DebugKick(printf("Test1: cw_ang: %f\tccw_ang: %f\n", cw_ang, ccw_ang));
  if (cw_ang < 0) cw_ang += 360;
  if (ccw_ang < 0) ccw_ang += 360;
  DebugKick(printf("Test2: cw_ang: %f\tccw_ang: %f\n", cw_ang, ccw_ang));
  if (cw_ang < ccw_ang)
    return TURN_CW;
  else
    return TURN_CCW;  
}

  

/* Kick with direction ddir and power such that the ball moves distance ddist
   If ddist is too big, kick it as hard as possible.
   corrects for ball velocity
   Returns a command object partially filled out
   The distance is multiplied by the distFactor */
/* returns KickCommand.time = -1 for error */
TurnKickCommand dokick(AngleDeg ddir, float ddist, float distFactor,
		       bool* pCanKickToTraj) 
{
  float v0;
  float power;
  float kick_dir = ddir;
  TurnKickCommand com;
  com.time = -1;
  com.type = CMD_kick;
  if (pCanKickToTraj)
    *pCanKickToTraj = TRUE;

  v0 = ddist * distFactor;

  NormalizeAngleDeg(&ddir);

  DebugKick(printf(" dokick: ddir: %f\tv0: %f\n", ddir, v0));
  DebugKick(printf(" kickrate: %f\tmyang: %f\n", Mem->BallKickRate(), Mem->MyAng() ));

  if (Mem->BallKickRate() == 0.0)
    my_error("dokick: Huh? BallKickRate is 0!");

  if (!Mem->BallVelocityValid())
    DebugKick(printf("In dokick with velocity not valid. Assuming it's 0."));
  
  if (Mem->BallVelocityValid() &&
      Mem->BallAbsoluteVelocity() != 0) { /* correct for ball velocity */
    Vector tmp = Polar2Vector(v0, ddir) - Mem->BallRelativeVelocity();
    kick_dir = tmp.dir();
    power = tmp.mod() / Mem->BallKickRate();
    DebugKick(printf(" Correcting for ball velocity# vel.x: %f\tvel.y:%f\n",
	  Mem->BallRelativeVelocity().x, Mem->BallRelativeVelocity().y));
    DebugKick(printf(" New stuff# power: %f\t ddir:%f\n", power, kick_dir));
  } else
     power = v0  / Mem->BallKickRate();

  if (power > Mem->SP_max_power) {
    DebugKick(printf("Trying to kick over SP_max_power! Correcting...\n"));
    if (!Mem->BallVelocityValid() || Mem->BallAbsoluteVelocity() == 0) {
      power = Mem->SP_max_power;
    } else {
      /* this is a ray circle intersection problem
         We want to kick in the right direction, but not as hard as desired */
      Vector sol1, sol2;
      int numSol;
      Vector vNewTraj;
      numSol =
	RayCircleIntersect(Mem->BallRelativePosition(),
			   Polar2Vector(1, ddir),
			   Mem->SP_max_power * Mem->BallKickRate(),
			   Mem->Global2RelativeToMe(Mem->BallPredictedPosition()),
			   &sol1, &sol2);
      /* we want the solution that's furthest along the ray - that's sol2
	 if there are two solution */
      if (numSol == 0) {
	/* we can't kick ball to desired trajectory, so let's just slow it
	   down by kicking directly against velocity
	   It might be better to return an error and have the caller decide
	   what to do, but this works pretty well */
	DebugKick(printf("Can't kick ball to right trajectory!\n"));
	power = Mem->SP_max_power;
	kick_dir = Mem->BallRelativeHeading() + 180;
	if (pCanKickToTraj)
	  *pCanKickToTraj = FALSE;
      } else {
	if (numSol == 2)
	  vNewTraj = sol2;
	else
	  vNewTraj = sol1;

	Vector vNewKick = vNewTraj -
	  Mem->Global2RelativeToMe(Mem->BallPredictedPosition());
	power = vNewKick.mod() / Mem->BallKickRate();
	kick_dir = vNewKick.dir();

	DebugKick(printf(" Correcting for ball velocity AND max power# vel.x: %f\tvel.y:%f\n",
	       Mem->BallRelativeVelocity().x, Mem->BallRelativeVelocity().y));
	DebugKick(printf("New stuff# power: %f\t dir:%f\n", power, kick_dir));
      }
    }    
  } 

  power = Min(Round(power, -2), Mem->SP_max_power);
  kick_dir = Round(kick_dir, -2);
  NormalizeAngleDeg(&kick_dir);
  DebugKick(printf("kicking with power: %f at dir: %f\n", power, kick_dir));
  com.time = Mem->CurrentTime;
  com.power = power;
  com.angle = kick_dir;
  return com;
}

/* all the turnball reasoning is done in relative coordinates
   call this before calling dokick to correct for movement of the player */
void PlayerMovementCorrection(AngleDeg* pdir, float* pdist)
{
  DebugKick(printf("Before corr- dir: %f\t dist: %f\n", *pdir, *pdist));
  Vector vRelBallTarg = Mem->BallRelativePosition() +
    Polar2Vector(*pdist, *pdir);
  Vector vNewRelTarg = Mem->MyPredictedPosition() +
    vRelBallTarg - Mem->MyPos();
  Vector vNewRelTraj = vNewRelTarg - Mem->BallRelativePosition();
  *pdir  = vNewRelTraj.dir();
  *pdist = vNewRelTraj.mod();
  DebugKick(printf("After corr- dir: %f\t dist: %f\n", *pdir, *pdist));
}


/* kick ball so that it is at angle ddir and dist EndDist
   If you have to kick around the player, kick rotway(clockwise or counter-)
   */
/* we always reason about the right trajectory for the ball leave velocity
   correction for dokick */
KickToRes turnball_kick(AngleDeg target_dir, TurnDir rotate, 
		  Bool StopBall, TurnKickCommand* pCom,
		  float EndDist, float closeMarg, float kickFac)
{
  float dir;
  float dist;
  Vector btraj;

  pCom->time = -1;
  
  DebugKick(printf("\nat turnball_kick: target_dir: %f\n", target_dir));

  NormalizeAngleDeg(&target_dir);
  
  //DebugKick(printf("HERE Time: %d\n", Mem->CurrentTime.t));
  /* the pos valid is not really right - if we are turning the ball and didn't
     actually see it last time, then there's a problem */
  if ( !Mem->BallPositionValid() || !Mem->BallKickable() ) {
    return KT_LostBall;
  }

  /* if the velocity isn's valid, turn to face ball */
  if ( !Mem->BallVelocityValid() ) {
    DebugKick(printf("turning to face ball\n"));
    /* we should reason about whether we can see the ball before turning,
       or maybe we should try and stop the ball */
    pCom->time = Mem->CurrentTime;
    pCom->type = CMD_turn;
    pCom->angle = Mem->BallAngle();
    return KT_TurnedToBall;
  } 
  
  DebugKick(printf(" ball.dist: %f\t.dir: %f\n",
	 Mem->BallDistance(), Mem->BallAngle()));
  DebugKick(printf(" HERE ball.vel.x: %f\t.y: %f\tmod: %f\n",
	 Mem->BallRelativeVelocity().x, Mem->BallRelativeVelocity().y,
	 Mem->BallSpeed()));
  DebugKick(printf(" ball.rpos.x: %f\t.y: %f\n",
	 Mem->BallRelativePosition().x, Mem->BallRelativePosition().y));
  DebugKick(printf(" target_dir: %f\n", target_dir));

  if ( fabs(GetNormalizeAngleDeg(target_dir - Mem->BallAngle())) < Mem->CP_KickTo_err) {
    /* Do something to indicate we are done */
    if (!StopBall || Mem->BallSpeed() < Mem->CP_max_ignore_vel)
      return KT_DidNothing;
    DebugKick(printf("  Stop ball kick\n"));
    dir = 0;
    dist = 0;    
    PlayerMovementCorrection(&dir, &dist);
    *pCom = dokick(dir, dist, 1.0);
    return KT_Success;
  }

  if (rotate == TURN_AVOID) {
    rotate = RotToAvoidOpponent(target_dir + Mem->MyAng());
  }
  
  if (rotate == TURN_CLOSEST) {
    rotate = RotClosest(target_dir + Mem->MyAng());
  }
  
  
  if (is_straight_kick(target_dir, EndDist, closeMarg)) {
    float pow;
      
    btraj = Polar2Vector(EndDist, target_dir) -
      Mem->BallRelativePosition();
    dir = btraj.dir();
    dist = btraj.mod();
    
    /* now we're goign to do some distance twiddling to get the ball to
       get to the right angle and stop */
    pow = dist / Mem->BallKickRate();
    pow = Min(pow, Mem->CP_max_turn_kick_pow);
    dist = pow * Mem->BallKickRate();
      
    DebugKick(printf("  Straight kick# dir: %f dist: %f\n", dir, dist));
    PlayerMovementCorrection(&dir, &dist);
    *pCom = dokick(dir, dist, 1.0);

  } else if (Mem->BallDistance() < closeMarg) {

    /* ball is too close to do a tangent kick, so do a kick at 90 degrees */
    dir = ((int)rotate)*(-90) + Mem->BallAngle();
    dist = 2.0*sqrt(Sqr(Mem->CP_opt_ctrl_dist) - Sqr(Mem->BallDistance()));
    DebugKick(printf("  Close kick# dir: %f dist: %f\n", dir, dist));
    PlayerMovementCorrection(&dir, &dist);
    *pCom = dokick(dir, dist, kickFac);

  } else {

    /* do a turning kick */
    /* we make a circle around the player of radius closeMarg
       and calculate the trajectory that goes in the right direction and is
       tangent to the circle */
    dir = 180 + Mem->BallAngle() +
      ((int)rotate)*ASin(closeMarg / Mem->BallDistance());
    DebugKick(printf(" ball dist: %f\tclosest_margin: %f\n",
	   Mem->BallDistance(), closeMarg));
    dist = sqrt(Sqr(Mem->BallDistance()) - Sqr(closeMarg));
    dist +=
      sqrt(Sqr(Mem->CP_opt_ctrl_dist) - Sqr(closeMarg));
    DebugKick(printf("  Turning ball# dir: %f dist: %f\n", dir, dist));
    PlayerMovementCorrection(&dir, &dist);
    *pCom = dokick(dir, dist, kickFac);

  }

  return KT_DidKick;
}


KickToRes TurnballTo(AngleDeg rel_dir, TurnDir rotate)
{
  TurnKickCommand com;
  KickToRes res = turnball_kick(rel_dir, rotate, TRUE, &com);
  if (res == KT_Success || res == KT_DidKick || res == KT_TurnedToBall)
    DoTurnKickCommand(com);

  return res;
}

/*********************************************************************************************/

/* if we kick the ball as hard as possible in the right direction, will it
   be a collision with the player? */
/* all we have to do is look at the predicted ball position with that kick and
   see if it is within the player's radius of the player */
int is_hard_kick_coll(float abs_dir, TurnKickCommand* pcom,
		      Vector* pvPredBall, float targ_vel)
{
  /* the Mem->SP_ball_speed_max is just to get as hard a kick as possible */
  *pcom = dokick(abs_dir - Mem->MyAng(), targ_vel);
  *pvPredBall = Mem->BallPredictedPosition(1, pcom->power, pcom->angle);
  DebugKick(cout << "IsColl: PredBall: " << *pvPredBall
	    << "\tMyPos: " << Mem->MyPredictedPosition() << endl);
  DebugKick(cout << "diff: " << (*pvPredBall - Mem->MyPredictedPosition()).mod()
	    << "\tmarg: " << Mem->CP_hard_kick_margin << endl);
  return (*pvPredBall - Mem->MyPredictedPosition()).mod() <=
    Mem->CP_hard_kick_margin;
}

/* Used when we decide a kick in the right direction woudl be a collision,
   so we need to turnball to kick the ball more to the side of us */
TurnKickCommand hard_kick_turnball(float abs_dir, TurnDir rot, Bool StopBall = FALSE)
{
  TurnKickCommand com;
  //TurnDir rot = RotToAvoidOpponent(abs_dir);
  /* SMURF - should this have a larger dokick_factor? */
  KickToRes res = turnball_kick(abs_dir - Mem->MyAng(), rot, StopBall,
				&com, Mem->CP_hard_kick_end_turn_dist,
			  Mem->CP_hard_kick_margin, Mem->CP_hard_kick_factor);
  if (res == KT_DidNothing || res == KT_LostBall)
    my_error("	hard_kick_turnball: Something weird happened: %d", res);
  return com;
}

TurnKickCommand kick_hard_moderate(AngleDeg abs_dir, float targ_vel, TurnDir rot)
{
  /* Here's the idea:
       See if one strong kick will be a collision (if it is, turnball)
       o.w. manipulate our kick so that we get another one next cycle
       (but make sure the final velocity is higher)
       it that makes it a collision, just do the strong kick */
    /* there is no reasning about turning the ball backward to get max power.
       See KM_HardestKick for that */
  TurnKickCommand kickCom;
  TurnKickCommand HKCommand;
  Vector vPredBall;
  if (is_hard_kick_coll(abs_dir, &kickCom, &vPredBall, targ_vel)) {
    DebugKick(printf(" Moderate: collision- doing turnball\n"));
    return hard_kick_turnball(abs_dir, rot);      
  } else {
    DebugKick(cout << " MyPos: " << Mem->MyPos() << endl);
    DebugKick(cout << " vPredBall: " << vPredBall << endl);
    Vector vPredBallRel =
      vPredBall.Global2Relative(Mem->MyPredictedPosition(), Mem->MyAng());
    DebugKick(cout << " vPredBallRel: " << vPredBallRel << endl);
    if (vPredBallRel.mod() < Mem->SP_kickable_area ||
	!Mem->BallWillBeKickable()) {
      /* we're going to get another kick next time or this will be our last
	   kick anyway - do the strong one! */
      DebugKick(printf(" Moderate: strong with another kick or last kick!\n"));
      return kickCom;
    }
    /* we're goign to set vBall to be the relative vector (to new pos)
	 to the position that will give us another kick next time */
    float oneKickVel = Min(Mem->SP_ball_speed_max,
			   (vPredBall - Mem->BallAbsolutePosition()).mod());
    float twoKickVel = 0.0;
    Vector sol1, sol2;
    int numSol;
    Vector vBallTarg;
    numSol =
      RayCircleIntersect(Mem->BallAbsolutePosition(),
			 vPredBall - Mem->BallAbsolutePosition(),
			 Mem->SP_kickable_area - Mem->CP_hard_kick_err,
			 Mem->MyPredictedPosition(),
			 &sol1, &sol2);
    /* we want the solution that's furthest along the ray - that's sol2
	 if there are two solution */
    if (numSol != 0) {
	
      if (numSol == 2)
	vBallTarg = sol2;
      else
	vBallTarg = sol1;

      /* see if the first of the two kicks is a coll */
      if ((vBallTarg - Mem->MyPredictedPosition()).mod() >=
	  Mem->CP_hard_kick_margin) {
	/* we can do it without collision */
	/* now see if this is actually goign to be better */
	vBallTarg =
	  vBallTarg.Global2Relative(Mem->BallAbsolutePosition(), Mem->MyAng());
	float kickrate =
	  Mem->GetBall()->calc_kick_rate(vBallTarg.mod(), vBallTarg.dir());
	/* the first kick */
	//DebugKick( cout << "vBallTarg: " << vBallTarg << "\tBallAbsPos: " <<
	//	 Mem->BallAbsolutePosition() << endl );
	twoKickVel = (vBallTarg - Mem->BallRelativePosition()).mod() *
	  Mem->SP_ball_decay;
	DebugKick(printf("  twoKickVel: first kick: %f\n", twoKickVel));
	/* the second kick */
	twoKickVel = Min(Mem->SP_ball_speed_max,
			 twoKickVel + kickrate * Mem->SP_max_power);
	DebugKick(printf("  oneKickVel: %f\ttwoKickVel: %f\n", oneKickVel, twoKickVel));
      } else
	my_error("kick_hard_moderate- no ray intersection?");	
    }
      
    /* remember if twoKick is a collision, then it's velocity will be 0 */
    if (numSol == 0 || oneKickVel >= twoKickVel) {
      /* do the one hard kick */
      DebugKick(printf(" Moderate- Doing one hard kick\n"));
      return kickCom;
    } else {
      /* do the weaker kick */
      DebugKick(printf(" Moderate- doing first of two kicks\n"));
      //DebugKick(printf(" Predicted distance: %f\n",
      //		 (vBallTarg - Mem->MyPredictedPosition()).mod() ));
      //DebugKick(cout << " BallCurrPos: " << Mem->BallAbsolutePosition() << endl);
      HKCommand = dokick(vBallTarg.dir(), vBallTarg.mod());
      DebugKick(cout << " KickTraj: " << vBallTarg << endl);
      DebugKick(cout << " PredPos: " << Mem->MyPredictedPosition() << endl);
      return HKCommand;
    }
      
  }    
}


TurnDir KickRotationDirectionAbs(AngleDeg abs_ang, TurnDir rot)
{
  if (rot == TURN_AVOID) 
    rot = RotToAvoidOpponent(abs_ang);    
  if (rot == TURN_CLOSEST) 
    rot = RotClosest(abs_ang);
  if (rot != TURN_CW && rot != TURN_CCW)
    my_error("KickRotationDirection: bad TurnDir");

  return rot;
}


/* see above for description of KM_Moderate */
/* KM_HardestKick: Moves the ball to the side of us (relative to the kick
   direction) then uses KM_Moderate */
/* KM_Quickly, KM_QuickestRelease: get rid of the ball as fast as possible,
   will turnball if neccesary. KM_Quickly will turn to face the ball if it can
   so that we get a harder kick */
/* returns 1 if a kick actually done */
int smart_kick_hard_abs(float abs_dir, KickMode mode, float targ_vel,
			TurnDir rot)
{
  TurnKickCommand HKCommand;
  HKCommand.time = -1;
  
  DebugKick(printf("\nsmart_kick_hard: Time: %d\n", Mem->CurrentTime.t));

  if (!Mem->BallPositionValid()) {
    my_error("smart_kick_hard called with ball position not valid");
    return 0;
  }

  if (!Mem->BallKickable()) {
    my_error("smart_kick_hard called with ball not kickable!");
    return 0;
  }
  
  if (!Mem->BallVelocityValid()) {
    DebugKick(printf("smart_kick_hard: turning to face ball\n"));
    HKCommand.time = Mem->CurrentTime;
    HKCommand.type = CMD_turn;
    HKCommand.angle = Mem->BallAngle();
    return DoTurnKickCommand(HKCommand);
  }

  rot = KickRotationDirectionAbs(abs_dir, rot);
  
  if (mode <= KM_Moderate &&
      Mem->IsPointBehind(Mem->BallAbsolutePosition(), abs_dir)) {
    /* see if we need to rotate one way */
    DebugKick(cout << "smart_kick_hard: decign if rotating " << rot << endl);
    /* now decide if ball is on the wrong side	 */
    if (Mem->IsPointBehind(Mem->BallAbsolutePosition(), abs_dir+((int)rot)*90)) {
      /* start rotating right way */
      DebugKick(cout << "smart_kick_hard: special turnball to avoid opp" << endl);
      TurnKickCommand com;
      KickToRes res = turnball_kick( abs_dir - Mem->MyAng(),
				     //abs_dir + ((int)rot)*90 - Mem->MyAng(),
				     rot, FALSE, &com);
      if (res == KT_DidNothing || res == KT_LostBall)
	my_error("smart_kick_hard: special turnball; turnkick failed, res: %d", res);
      else
	return DoTurnKickCommand(com);
    }
  }
  
  
  
  switch (mode) {
  case KM_None:
    my_error("KM_None is not a valid kick mode for smart_kick_hard!");
    break;
    
  case KM_HardestKick: {      
    if (Mem->CurrentTime - 1 != Mem->HKTime) {
      DebugKick(printf("First in a chain of calls\n"));
      /* this is the first in a chain of calls */      
      Mem->HKStep = 0;
      /* decide which way to rotate ball */
      Mem->HKrot = rot;
      /*AngleDeg ang = Mem->BallAngle() + Mem->MyAng() - abs_dir;
      NormalizeAngleDeg(&ang);
      if (ang >= 0)
	Mem->HKrot = TURN_CW;
      else
	Mem->HKrot = TURN_CCW;*/
	  
      /* see if we need to turn */
      AngleDeg target_dir = abs_dir - Mem->MyAng() + ((int)Mem->HKrot)*90;
      if (fabs(target_dir) > Mem->CP_hard_kick_err) {
	DebugKick(printf("turning to get hard kick\n"));
	Mem->HKTime = Mem->CurrentTime;
	Mem->HKStepNext = 0;
	HKCommand.type = CMD_turn;
	HKCommand.angle = target_dir;
	HKCommand.time = Mem->CurrentTime;
	return DoTurnKickCommand(HKCommand);
      }
    } else
      Mem->HKStep = Mem->HKStepNext;

    /* AngleDeg turn_target = abs_dir + 180 +
      ((int)Mem->HKrot)*Mem->CP_hardest_kick_angle_disp; */
    AngleDeg turn_target = abs_dir + 
      ((int)Mem->HKrot)*(90 + Mem->CP_static_kick_ang);
    NormalizeAngleDeg(&turn_target);
    if ( Mem->HKStep == 0 &&
	 fabs(GetNormalizeAngleDeg(turn_target-Mem->MyAng()-Mem->BallAngle()))>
	 Mem->CP_KickTo_err) {
      /* on step 0, we turn ball to back of us */      
      HKCommand = hard_kick_turnball(turn_target, TURN_CLOSEST, TRUE);//TRUE to stop ball
      Mem->HKStepNext = 0;
    }
    if (Mem->HKStep == 1 ||
	(Mem->HKStep == 0 && HKCommand.time != Mem->CurrentTime)) {
      /* on step 1, we call the moderate code */
      /* or if step 0 had turnball problems, we'll go ahead and drop
	 to this code */
      HKCommand = kick_hard_moderate(abs_dir, targ_vel, rot);
      Mem->HKStepNext = 1;
    }
    if (Mem->HKStep != 0 && Mem->HKStep != 1)
      my_error("HKstep is not in a valid state: %d", Mem->HKStep);
    
    Mem->HKTime = Mem->CurrentTime;
    return DoTurnKickCommand(HKCommand);
    break;
  }
  
  case KM_Hard:
    my_error("KM_Hard not implemented!");
    break;
    
  case KM_Moderate:
    return DoTurnKickCommand(kick_hard_moderate(abs_dir, targ_vel, rot));
  break;
    
  case KM_Quickly:
  case KM_QuickestRelease: {
      Mem->HKStep = Mem->HKStepNext = -1;
      /* see if the hardest kick in the direction we want will be collision */
      /* use our dokick function to correct for vel */
      TurnKickCommand kickCom;
      Vector vPredBall;
      if (!is_hard_kick_coll(abs_dir, &kickCom, &vPredBall, targ_vel)) {
	/* there is no collsion! */
	if (mode == KM_Quickly &&
	    Mem->BallWillBeKickable() &&
	    fabs(Mem->BallAngle()) > Mem->CP_max_hard_kick_angle_err) {
	  /* In KM_Quickly mode, we can turn to ball and get it next time */
	  HKCommand.time = Mem->CurrentTime;
	  HKCommand.type = CMD_turn;
	  HKCommand.angle = Mem->BallAngle();
	} else
	  HKCommand = kickCom;
	return DoTurnKickCommand(HKCommand);
      } else {
	/* do turnball */
	return DoTurnKickCommand(hard_kick_turnball(abs_dir, rot));
      }
    
    }
    
    break;
    
  default:
    my_error("Invalid/Unimplemented kick mode passed to smart_kick_hard");
    break;
  }

  my_error("Got to end of smart_kick_hard?");
  return 0;
  
}

/* good passes require that the ball is not moving too quickly when it reaches
   the intended recipient, so this cover function helps acheive that */
int smart_pass(Vector pt, float targ_vel_at_pt, KickMode mode, TurnDir rot)
{
  return smart_kick_hard(Mem->AngleTo(pt), mode,
			 Mem->VelAtPt2VelAtFoot(pt, targ_vel_at_pt),
			 rot);
}


