/* -*- Mode: C++ -*- */

/* behave.C
 * CMUnited98 (soccer client for Robocup98)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1998 Peter Stone
 *
 * CMUnited-98 was created by Peter Stone, Manuela Veloso, and Patrick Riley
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* behave.C is called once every server cycle.
   It determines the action to be sent to the server
 */


#include "behave.h"
#include "Memory.h"
#include "client.h"
#include "test.h"
#include "kick.h"
#include "dribble.h"

#ifndef RELEASE_VERSION
#include "intercept.h"
#include "goalie.h"
#include "setplay.h"
#endif

#define DebugHandle(x) 
#define DebugMark(x) 

#ifdef DEBUG_OUTPUT
#define DebugBlock(x) x
#else
#define DebugBlock(x) 
#endif


void behave(){

  /* Put any one of these test functions in -- but only one at a time */

  /* Start one player -- move the ball around the field */
  //face_ball();

  /* Start one player, press kickoff -- it will kick to the goal */
  test_go_to_ball();

  /* Start one player, press kickoff -- it will dribble to the goal */
  //test_dribble_to_goal();

  /* Start 2 players on different teams, press 'drop ball' near one player */
  //test_1v1();

  /* Start 2 players on different teams, press 'drop ball' near one player */
  //test_volley();

  /* These are to print out the positions of objects */
  //test_print_ball();
  //test_print_positions();

  /* Start 3 players on each team and press 'drop ball' in the middle of the field */
  // Rectangle *rect = new Rectangle(Vector(0,0),Vector(30,30));
  // test_random_movement_in_rectangle(rect);
  // delete rect;
}

/*****************************************************************************************/
/*****************************************************************************************/
/*****************************************************************************************/

ActionQueueRes scan_field()
{
  if ( Mem->FirstActionOpSinceLastSight ){
    turn(Mem->MyViewAngle() * 2);
    return AQ_ActionQueued;
  }
  else 
    return AQ_ActionNotQueued;
}

/*****************************************************************************************/

void face_point(Vector point)
{
  turn( Mem->AngleTo(point) );
}

/*****************************************************************************************/

ActionQueueRes face_teammate(Unum teammate)
{
  if ( Mem->TeammatePositionValid(teammate) ){
    if ( fabs(Mem->TeammateAngle(teammate)) < 1 )
      return AQ_ActionNotQueued;
    turn( Mem->TeammateAngle(teammate) );
    return AQ_ActionQueued;
  }
  else
    return scan_field();
}

/*****************************************************************************************/

ActionQueueRes face_opponent(Unum opponent)
{
  if ( Mem->OpponentPositionValid(opponent) ){
    if ( fabs(Mem->OpponentAngle(opponent)) < 1 )
      return AQ_ActionNotQueued;
    turn( Mem->OpponentAngle(opponent) );
    return AQ_ActionQueued;
  } 
  else
    return scan_field();
}

/*****************************************************************************************/

ActionQueueRes face_ball()
{
  if ( Mem->BallPositionValid() ) {
    if ( fabs(Mem->BallAngle()) < 1 )
      return AQ_ActionNotQueued;
    turn( Mem->BallAngle() );
    return AQ_ActionQueued;
  }
  else 
    return scan_field();
}

/*****************************************************************************************/

void get_ball()
{
  if ( !Mem->MyConf() || !Mem->BallPositionValid() ) my_error("not enough info to get ball");

  if ( !Mem->BallMoving() ){
    if ( go_to_point(Mem->BallAbsolutePosition()) == AQ_ActionNotQueued ){
      my_error("already there???");
      face_ball();
    }
  }
  else {
    if ( !Mem->MyInterceptionAble() ){
      my_error("Can't get to the ball");
      face_ball();
    }
    else if ( go_to_point(Mem->MyInterceptionPoint()) == AQ_ActionNotQueued ) {
      my_error("already there (moving) ???");
      face_ball();
    }
  }
}

/*****************************************************************************************/

void stop_ball()
{
  if (!Mem->BallPositionValid())
    my_error("Need to know where ball is to stop it");

  if (Mem->BallVelocityValid()) {
    DoTurnKickCommand(dokick(0,0));
  } else {
    //DebugDrib(printf("Stop kick; don't know velocity, so goign to kick to us"));
    kick(30, GetNormalizeAngleDeg(Mem->BallAngle() + 180));
  }
}

/*****************************************************************************************/

void hold_ball()
{
  if (!Mem->BallPositionValid())
    my_error("Need to know where ball is to hold it");

  Unum opponent = Mem->ClosestOpponent();

  if ( opponent == Unum_Unknown && Mem->BallMoving() )
    stop_ball();
  else if ( opponent == Unum_Unknown && Mem->FirstActionOpSinceLastSight )
    scan_field();
  else if ( Mem->OpponentPositionValid(opponent) < .9 && !Mem->InViewAngle(Mem->OpponentAngle(opponent)) )
    face_opponent(opponent);
  else if ( Mem->FirstActionOpSinceLastSight ) /* want this??? */
    scan_field();
  else {
    AngleDeg ang = GetNormalizeAngleDeg(Mem->OpponentAngle(opponent) + 180);
    TurnballTo(ang);
  }
}

/*****************************************************************************************/

void pass_ball(Unum teammate, float target_vel)
{
  if ( teammate == Unum_Unknown ) my_error("Need to pass to a teammate");
  if ( !Mem->TeammatePositionValid(teammate) ) my_error("can't pass to invalid teammmate");

  if ( Mem->TeammatePositionValid(teammate) < .9 ){
#ifndef RELEASE_VERSION
    Mem->Say(PMsg_ping_teammate,teammate);
#endif
    if ( face_teammate(teammate) == AQ_ActionNotQueued )
      hold_ball();
    return;
  }

  Vector   target = Mem->TeammateAbsolutePosition(teammate);
  AngleDeg target_angle = Mem->AngleTo(target);
  TurnDir  rotation = KickRotationDirection(target_angle);

#ifndef RELEASE_VERSION
  Mem->Say( PMsg_passing_decision, teammate );
#endif
  Mem->StartPass(teammate,target_vel,rotation);
  smart_pass(target,target_vel,KM_Moderate,rotation);
}

/*****************************************************************************************/

void kick_ball(AngleDeg target_angle, KickMode mode, TurnDir rotation)
{
  if (rotation == TURN_NONE)
    rotation = KickRotationDirection(target_angle);
  Mem->StartShot(target_angle, mode, rotation);
  smart_kick_hard(target_angle, mode, rotation);
  return;
}

void kick_ball(Vector point, KickMode mode, TurnDir rotation)
{
  kick_ball(Mem->AngleTo(point),mode, rotation);
}

/*****************************************************************************************/

ActionQueueRes go_to_point( Vector p, float buffer, float dash_power, Bool dodge )
{
  if ( !Mem->MyConf() ) my_error("Can't go to a point if not localized");

  if ( Mem->DistanceTo(p) < buffer ){
    if ( Mem->SP_use_offside && fabs(Mem->MyX() - Mem->my_offside_line) < 5 ){ /* hack */
      Unum opp = Mem->FurthestForwardOpponent();
      if ( opp != Unum_Unknown && Mem->OpponentPositionValid(opp) < .9 ){ /* hack */
	return face_opponent(opp);
      }
    }
    return AQ_ActionNotQueued;
  }

#ifndef RELEASE_VERSION
  if ( Mem->PlayMode != PM_Play_On && Mem->TeamInPossession() == Mem->TheirSide &&
       !Mem->OffsidePosition(p,Mem->MySide) &&
       //p.dist(Mem->BallAbsolutePosition()) > Mem->SP_free_kick_buffer &&
       Mem->InOffsidePosition() && Mem->BallDistance() < Mem->SP_free_kick_buffer+1 ){
    if ( Mem->BallY() > Mem->MyY() )
      go_to_point(Vector(Mem->MyX(),Mem->BallY()-(Mem->SP_free_kick_buffer+1)));
    else
      go_to_point(Vector(Mem->MyX(),Mem->BallY()+(Mem->SP_free_kick_buffer+1)));
    return AQ_ActionQueued;
  }
#endif

  float target_ang  = Mem->AngleTo(p);
  float target_dist = Mem->DistanceTo(p);

  //if ( Mem->ClockStopped ) dodge = FALSE;

  if ( dodge ){ /* dodge players */
    PlayerObject *player;
    float    dodge_dist = Min(Mem->CP_dodge_distance_buffer,target_dist);
    AngleDeg dodge_ang  = Mem->CP_dodge_angle_buffer;
    if ( (player = Mem->GetPlayerWithin( dodge_dist, dodge_ang, 0, target_ang - dodge_ang))
	 != NULL ){
      if ( Mem->NumPlayersWithin( dodge_dist, 2*dodge_ang) ){
	/* Target close, so no players will be within in the next iteration ==> dash */
	Vector new_target = Mem->Polar2Gpos(.8,player->get_ang() + 90);
	go_to_point(new_target,0);
      }
      else{
	dash(Mem->CorrectDashPowerForStamina(Mem->CP_dodge_power));
      }
      return AQ_ActionQueued;
    }
    if ( (player = Mem->GetPlayerWithin( dodge_dist, dodge_ang, 0, target_ang + dodge_ang))
	 != NULL ){
      if ( Mem->NumPlayersWithin( dodge_dist, 2*dodge_ang) ){
	/* Target close, so no players will be within in the next iteration ==> dash */
	Vector new_target = Mem->Polar2Gpos(.8,player->get_ang() - 90);
	go_to_point(new_target,0);
      }
      else{
	dash(Mem->CorrectDashPowerForStamina(Mem->CP_dodge_power));
      }
      return AQ_ActionQueued;
    }
  }

  if ( fabs(target_ang) > Mem->CP_max_go_to_point_angle_err ){
    turn(target_ang);
    return AQ_ActionQueued;
  }

  dash_power = Mem->CorrectDashPowerForStamina(dash_power);
  if ( dash_power > 0 ){
    dash( dash_power );
    return AQ_ActionQueued;
  }
  else
    {my_stamp; printf("recovering\n");} 

  return AQ_ActionNotQueued;
}
