/* -*- Mode: C -*- */

/* MemPosition.h
 * CMUnited98 (soccer client for Robocup98)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1998 Peter Stone
 *
 * CMUnited-98 was created by Peter Stone, Manuela Veloso, and Patrick Riley
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* MemPosition.C stores all the information relating to other objects on the field:
 * other players, the ball, markers, etc.
 */

#ifndef _POSITION_H_
#define _POSITION_H_

#include "MemPlayer.h"



/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/

class Object {
public:

  /* Methods to get the position  */
  float    get_dist();  /* relative */
  AngleDeg get_ang();

  inline Vector get_abs_pos()
    { if (!pos_valid()) my_error ("can't get_abs_pos %d",type);  return gpos; }
  Vector get_rel_pos();
  inline float get_x() { return get_abs_pos().x; }
  inline float get_y() { return get_abs_pos().y; }
  inline float pos_valid() { return ( (gconf >= min_valid_conf) ? gconf : 0 ); }
  
  /* Methods to change the position                      */
  /* Changes do NOT take effect until update() is called */
  virtual void set_polar(float d, float a, Time time);
  virtual void set_angle(AngleDeg a, Time time);
  virtual void set_chinfo(float dist, float dir, Time time);

  /* Method which processes all changes */
  virtual void update();  
  virtual void reset ();  /* 0 confidences */
  virtual void sanitize_times();
  
  ObjType type;

protected:

  float max_conf;
  float min_valid_conf;

  Bool    seen;
  Bool    seen_moving;
  Time    seen_time;

  /* Based on the object's current position, should it be seen? */
  virtual Bool in_view_range(Time time, float angle_buffer = 0, float distance_buffer = 0);
  
  float    dist;  /* polar */
  AngleDeg ang;
  Time     pdtime;
  Time     patime;

  inline Time ptime() { return Min(pdtime,patime); }

  float  distch;  /* polar */
  float  dirch;
  Time   chtime;

  Vector rpos;
  Time   rtime;

  Vector gpos;    /* global   */
  Time   gtime;
  float  gconf;
};



/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/

class StationaryObject : public Object {
public:

  void Initialize(MarkerType m, Vector pos, float max, float min_valid, Bool rotate);
  void Initialize(SideLine  l, Vector pos, float max, float min_valid, Bool rotate);

  Vector get_my_pos(AngleDeg my_ang);
  AngleDeg get_my_ang();
  Vector get_my_vel(AngleDeg my_ang);

  int object_id;
};



/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/

class MobileObject : public Object {
public:

  virtual void Initialize(ObjType t, float max, float min_valid, float decay, float motion, float speed);

  AngleDeg get_rel_heading();
  Vector get_rel_vel(); /* relative direction, absolute speed */

  /* Methods to get the velocity */
  inline Vector get_abs_vel()  /* global   */
    { if (!vel_valid()) my_error ("can't get_abs_vel %d",type); return gvel; }

  inline float    get_speed()       { return get_abs_vel().mod(); }
  inline AngleDeg get_abs_heading() { return get_abs_vel().dir(); }     
  inline float    vel_valid()       { return ( (gvconf >= min_valid_conf) ? gvconf : 0 ); }
  virtual inline Bool moving()      { return vel_valid() && get_speed() != 0 ? TRUE : FALSE; } 
      
    /* Methods to change the position                      */
  /* Changes do NOT take effect until update() is called */
  virtual void set_heard_info (float x, float y, float conf, float dist, Time time);
  virtual void set_heard_info (float x, float y, float pconf, float dx, float dy, float vconf, 
			       float dist, Time time);

  /* Method which processes all changes */
  virtual void  update_kick(Time time) = 0; /* pure virtual function */
  virtual void  estimate_pos(Time time);
  void          estimate_vel(Time time);
  virtual void  update(Time time);
  virtual void  update_seen(Time time);
  void          update_estimate(Time time);
  void          update_heard(Time time);
  virtual void  reset();
  virtual void  forget();
  virtual void  sanitize_times();

  Vector        estimate_future_pos(int steps, Vector extra_vel=0.0, Vector extra_vel_per=0.0);

protected:

  float conf_decay;
  float motion_decay;

  float max_speed;

  Vector gvel;    /* global   */

  Time   gvtime;
  float  gvconf;

private:

  Bool heard;
  Bool heard_moving;

  Vector rvel;    /* relative direction, absolute speed */
  Time   rvtime;

  Vector hpos;    /* heard */
  Vector hvel;    
  Time   hptime;
  Time   hvtime;
  float  hpdist;
  float  hvdist;
  float  hpconf;
  float  hvconf;
};



/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/

class BallObject : public MobileObject {
public:
  void Initialize(float max, float min_valid, float decay, float motion, float max_speed);

  Bool moving();
  Bool kickable(float buffer = 0.0);
  Bool catchable();
  float get_kick_rate(Time time);

  void update_kick(Time time);
  void estimate_pos(Time time);
  void update(Time time);
  void update_seen(Time time);
        
  float calc_kick_rate(float dist, float ang);
  float calc_kick_rate() {return calc_kick_rate(get_dist(), get_ang());}

  void forget_past_kick(Time t);

 protected:
  Bool  in_view_range(Time time, float angle_buffer = 3, float distance_buffer = .1);
  
private:
  float effective_kick_rate;
  Time  ektime;

  Bool use_pos_based_vel_estimate;
  Time pos_based_vel_time;
  
  Time   last_seen_time;
  Vector last_seen_pos;

  /* we keep a record of the most recent kicks we try to do so that
     we can estimate the velocity of the ball from subsequent positions */
#define num_past_kicks (7)		
  int past_kick_idx;
  struct PastKick {
    Time time;
    Vector kick_effect;
  } past_kicks[num_past_kicks];

  int past_kick_inc(int i)
    { return (i+1) % num_past_kicks; }
  int past_kick_dec(int i)
    { return ((i-1)+num_past_kicks) % num_past_kicks; }
};


/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/

class PlayerObject : public MobileObject {
public:
  void Initialize(float max, float min_valid, float decay, float motion, float max_speed);

  char side;
  Unum unum;

  AngleDeg get_rel_face();

  /* Methods to get the velocity */
  inline AngleDeg get_abs_face() { return gface; }  /* global   */
  inline float  face_valid() { return ( (gfconf >= min_valid_conf) ? gfconf : 0 ); }

  /* Methods to change the position                      */
  /* Changes do NOT take effect until update() is called */
  void set_face      (AngleDeg face , Time time);
  void set_heard_info_w_face (float x, float y, float pconf, float dx, float dy, float vconf, 
				  AngleDeg face, float fconf, float dist, Time time);

  /* Method which processes all changes */
  void  update(Time time);
  void  update_seen_face(Time time);
  void  update_estimate(Time time);
  void  update_heard(Time time);
  void  reset();
  void  forget();
  void  sanitize_times();

  /* This is a pure virtual MobileObject fn.  Shouldn't be called for playerobject */
  inline void update_kick(Time time) { my_error("Can't kick players %d",time.t); }

protected:
  Bool  in_view_range(Time time, float angle_buffer = 3, float distance_buffer = 1);

private:

  Bool     seen_face;
  Bool     heard_face;

  AngleDeg rface;  /* relative */
  Time     rftime;

  AngleDeg gface;  /* global   */
  float    gfconf;
  Time     gftime;

  AngleDeg hface;  /* heard    */
  float    hfconf;
  float    hfdist;
  Time     hftime;
};


/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/

class TempPlayerObject {
public:

  float    dist;  /* relative */
  AngleDeg ang;

  Time     time;
  char     side;

  inline void set(char s, float d, float a, Time t) { time = t; side = s; dist = d; ang = a; }
};




/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/

#define MAX_MARKERS 100
#define MAX_PLAYERS 25

#define Unum_Unknown  0
#define Unum_Teamless 100

class PositionInfo : public PlayerInfo{
public:

  ~PositionInfo();
  void Initialize();

  void SeeLine  (SideLine l, float dist, float ang, Time tm);
  void SeeLine  (SideLine l, float ang, Time tm);

  void SeeMarker(MarkerType marker, float dist, float ang, Time tm);
  void SeeMarker(MarkerType marker, float ang, Time tm);
  void SeeMarker(MarkerType marker, float dist, float ang, float distChng, float dirChng, Time tm);

  void SeeBall(float ang, Time tm);
  void SeeBall(float dist, float ang, Time tm);
  void SeeBall(float dist, float ang, float distChng, float dirChng, Time tm);

  void SeePlayer(float ang, Time time);
  void SeePlayer(float dist, float ang, Time time);
  void SeePlayer(char side, float ang, Time time);
  void SeePlayer(char side, float dist, float ang, Time time);
  void SeePlayer(char side, Unum num, 
		 float dist, float ang, float distChng, float dirChng, float facedir, Time time);
  void SeePlayer(char side, Unum num, float dist, float ang, Time time);
  void SeePlayer(char side, Unum num, float ang, Time time);

  void HearBall(float x, float y, float conf, float dist, Time time);
  void HearBall(float x, float y, float pconf, float dx, float dy, float vconf, float dist, Time time);

  void HearPlayer(char side, Unum num, float x, float y, float conf, float dist, Time time);
  void HearPlayer(char side, Unum num,
		  float x, float y, float pconf, float dx, float dy, float vconf, float dist, Time time);
  void HearPlayer(char side, Unum num, float x, float y, float pconf, float dx, float dy, float vconf, 
		  AngleDeg face, float fconf, float dist, Time time);

  /* Access shortcuts -- Markers */
  inline float    MarkerDistance     (MarkerType m) { return GetMarker(m)->get_dist();  }
  inline AngleDeg MarkerAngle        (MarkerType m) { return GetMarker(m)->get_ang();   }
  inline Vector   MarkerPosition     (MarkerType m) { return GetMarker(m)->get_abs_pos();   }
  inline float    MarkerX            (MarkerType m) { return GetMarker(m)->get_x();     }
  inline float    MarkerY            (MarkerType m) { return GetMarker(m)->get_y();     }
  inline float    MarkerPositionValid(MarkerType m) { return GetMarker(m)->pos_valid(); }

  /* Access shortcuts -- Ball */
  inline float    BallDistance     () { return GetBall()->get_dist();  }
  inline AngleDeg BallAngle        () { return GetBall()->get_ang();   }
  inline Vector   BallAbsolutePosition() { return GetBall()->get_abs_pos();   }
  inline Vector   BallRelativePosition() { return GetBall()->get_rel_pos();   }
  inline float    BallX            () { return GetBall()->get_x();     }
  inline float    BallY            () { return GetBall()->get_y();     }
  inline float    BallPositionValid() { return GetBall()->pos_valid(); }

  inline float    BallSpeed          () { return GetBall()->get_speed();       }
  inline AngleDeg BallRelativeHeading() { return GetBall()->get_rel_heading(); }  
  inline AngleDeg BallAbsoluteHeading() { return GetBall()->get_abs_heading(); }  
  inline Vector   BallRelativeVelocity() { return GetBall()->get_rel_vel();    }  
  inline Vector   BallAbsoluteVelocity() { return GetBall()->get_abs_vel();    }  
  inline float    BallVelocityValid  () { return GetBall()->vel_valid();       }  
  inline Bool     BallMoving         () { return GetBall()->moving();          }
  inline Bool     BallKickable       (float buffer=0.0)
    { return GetBall()->kickable(buffer); }  
  inline Bool     BallCatchable      () 
    { return (GetBall()->catchable() && InOwnPenaltyArea() && 
	      CP_goalie && PlayMode == PM_Play_On)             ? TRUE : FALSE; }  
  inline float    BallKickRate       () { return GetBall()->get_kick_rate(CurrentTime);   }  

  /* Access shortcuts -- Players */
  inline float    PlayerDistance        (char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? 0                         : GetPlayer(s,n)->get_dist();    }
  inline AngleDeg PlayerAngle           (char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? 0                         : GetPlayer(s,n)->get_ang();     }
  inline Vector   PlayerAbsolutePosition(char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? MyPos()                   : GetPlayer(s,n)->get_abs_pos(); }
  inline Vector   PlayerRelativePosition(char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? Vector(0,0)               : GetPlayer(s,n)->get_rel_pos(); }
  inline float    PlayerX               (char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? MyX()                     : GetPlayer(s,n)->get_x();       }
  inline float    PlayerY               (char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? MyY()                     : GetPlayer(s,n)->get_y();       }
  float    PlayerPositionValid          (char s, Unum n);

  inline float    PlayerSpeed           (char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? MySpeed()                 : GetPlayer(s,n)->get_speed();       }
  inline AngleDeg PlayerRelativeHeading (char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? 0                         : GetPlayer(s,n)->get_rel_heading(); }  
  inline AngleDeg PlayerAbsoluteHeading (char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? MyDir()                   : GetPlayer(s,n)->get_abs_heading(); }  
  inline Vector   PlayerAbsoluteVelocity(char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? MyVel()                   : GetPlayer(s,n)->get_abs_vel();     }  
  inline Vector   PlayerRelativeVelocity(char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? Polar2Vector(MySpeed(),0) : GetPlayer(s,n)->get_rel_vel();     }  
  float    PlayerVelocityValid          (char s, Unum n); 
  inline AngleDeg PlayerRelativeFace    (char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? 0                         : GetPlayer(s,n)->get_rel_face();    }  
  inline AngleDeg PlayerAbsoluteFace    (char s, Unum n) 
    { return (s==MySide && n==MyNumber) ? MyAng()                   : GetPlayer(s,n)->get_abs_face();    }  
  float    PlayerFaceValid              (char s, Unum n);

  /* Access shortcuts -- Teammates */
  inline float    TeammateDistance        (Unum n) { return PlayerDistance        (MySide,n); }
  inline AngleDeg TeammateAngle           (Unum n) { return PlayerAngle           (MySide,n); }
  inline Vector   TeammateAbsolutePosition(Unum n) { return PlayerAbsolutePosition(MySide,n); }
  inline Vector   TeammateRelativePosition(Unum n) { return PlayerRelativePosition(MySide,n); }
  inline float    TeammateX               (Unum n) { return PlayerX               (MySide,n); }
  inline float    TeammateY               (Unum n) { return PlayerY               (MySide,n); }
  inline float    TeammatePositionValid   (Unum n) { return PlayerPositionValid   (MySide,n); }

  inline float    TeammateSpeed           (Unum n) { return PlayerSpeed           (MySide,n); }
  inline AngleDeg TeammateRelativeHeading (Unum n) { return PlayerRelativeHeading (MySide,n); }
  inline AngleDeg TeammateAbsoluteHeading (Unum n) { return PlayerAbsoluteHeading (MySide,n); }
  inline Vector   TeammateAbsoluteVelocity(Unum n) { return PlayerAbsoluteVelocity(MySide,n); }
  inline Vector   TeammateRelativeVelocity(Unum n) { return PlayerRelativeVelocity(MySide,n); }
  inline float    TeammateVelocityValid   (Unum n) { return PlayerVelocityValid   (MySide,n); }
  inline AngleDeg TeammateRelativeFace    (Unum n) { return PlayerRelativeFace    (MySide,n); }
  inline AngleDeg TeammateAbsoluteFace    (Unum n) { return PlayerAbsoluteFace    (MySide,n); }
  inline float    TeammateFaceValid       (Unum n) { return PlayerFaceValid       (MySide,n); }

  /* Access shortcuts -- Opponents */
  inline float    OpponentDistance        (Unum n) { return PlayerDistance        (TheirSide,n); }
  inline AngleDeg OpponentAngle           (Unum n) { return PlayerAngle           (TheirSide,n); }
  inline Vector   OpponentAbsolutePosition(Unum n) { return PlayerAbsolutePosition(TheirSide,n); }
  inline Vector   OpponentRelativePosition(Unum n) { return PlayerRelativePosition(TheirSide,n); }
  inline float    OpponentX               (Unum n) { return PlayerX               (TheirSide,n); }
  inline float    OpponentY               (Unum n) { return PlayerY               (TheirSide,n); }
  inline float    OpponentPositionValid   (Unum n) { return PlayerPositionValid   (TheirSide,n); }

  inline float    OpponentSpeed           (Unum n) { return PlayerSpeed           (TheirSide,n); }
  inline AngleDeg OpponentRelativeHeading (Unum n) { return PlayerRelativeHeading (TheirSide,n); }
  inline AngleDeg OpponentAbsoluteHeading (Unum n) { return PlayerAbsoluteHeading (TheirSide,n); }
  inline Vector   OpponentAbsoluteVelocity(Unum n) { return PlayerAbsoluteVelocity(TheirSide,n); }
  inline Vector   OpponentRelativeVelocity(Unum n) { return PlayerRelativeVelocity(TheirSide,n); }
  inline float    OpponentVelocityValid   (Unum n) { return PlayerVelocityValid   (TheirSide,n); }
  inline AngleDeg OpponentRelativeFace    (Unum n) { return PlayerRelativeFace    (TheirSide,n); }
  inline AngleDeg OpponentAbsoluteFace    (Unum n) { return PlayerAbsoluteFace    (TheirSide,n); }
  inline float    OpponentFaceValid       (Unum n) { return PlayerFaceValid       (TheirSide,n); }

  /* more complex shortcuts */
  /* kickable for other players */
  Bool BallKickableForPlayer(char s, Unum n, float buffer=0);
  inline Bool BallKickableForTeammate(Unum n, float buffer=0) 
    { return BallKickableForPlayer(MySide, n, buffer); }
  inline Bool BallKickableForOpponent(Unum n, float buffer=0) 
    { return BallKickableForPlayer(TheirSide, n, buffer); }

  /* is player behind us */
  /* three forms for third arg:
     angle: absolute angle indicating the front
     Vector: position whose direction indicates the front
     none: use MyAng */
  /* is point behind us */
  /* three forms for third arg:
     angle: absolute angle indicating the front
     Vector: position whose direction indicates the front
     none: use MyAng */
  inline Bool IsPointBehind(Vector pt, AngleDeg ang)
    {
      return (fabs(GetNormalizeAngleDeg(AngleTo(pt) + MyAng() - ang)) > CP_behind_angle)
	? TRUE : FALSE;      
    }
  inline Bool IsPointBehind(Vector pt, Vector targPos)
    {
      return IsPointBehind(pt, (targPos - MyPos()).dir());
    }
  inline Bool IsPointBehind(Vector pt)
    {
      return (fabs(AngleTo(pt)) > CP_behind_angle)
	? TRUE : FALSE;
    }

  inline Bool IsPlayerBehind(char s, Unum n, AngleDeg ang)
    {      
      if (!PlayerPositionValid(s, n))
	my_error("Can't tell if player is behind if I don't know where he is");
      return (fabs(PlayerAngle(s, n) + MyAng() - ang) > CP_behind_angle)
	? TRUE : FALSE;
    }
  inline Bool IsPlayerBehind(char s, Unum n, Vector targPos)
    {
      return IsPlayerBehind(s, n, (targPos - MyPos()).dir());
    }
  inline Bool IsPlayerBehind(char s, Unum n)
    {
      if (!PlayerPositionValid(s, n))
	my_error("Can't tell if player is behind if I don't know where he is");
      return (fabs(PlayerAngle(s, n)) > CP_behind_angle)
	? TRUE : FALSE;
    }

  inline Bool IsTeammateBehind(Unum n, AngleDeg ang)
    { return IsPlayerBehind(MySide, n, ang); }
  inline Bool IsTeammateBehind(Unum n, Vector targPos)
    { return IsPlayerBehind(MySide, n, targPos); }
  inline Bool IsTeammateBehind(Unum n)
    { return IsPlayerBehind(MySide, n); }
  
  inline Bool IsOpponentBehind(Unum n, AngleDeg ang)
    { return IsPlayerBehind(TheirSide, n, ang); }
  inline Bool IsOpponentBehind(Unum n, Vector targPos)
    { return IsPlayerBehind(TheirSide, n, targPos); }
  inline Bool IsOpponentBehind(Unum n)
    { return IsPlayerBehind(TheirSide, n); }
  
  
  /* Relative markers */
  MarkerType RM_My_Goal;
  MarkerType RM_Their_Goal;
  MarkerType RM_LB_Flag;
  MarkerType RM_LC_Flag;
  MarkerType RM_LF_Flag;
  MarkerType RM_RB_Flag;
  MarkerType RM_RC_Flag;
  MarkerType RM_RF_Flag;
  MarkerType RM_My_PC_Flag;    /* Center of my penalty area */
  MarkerType RM_Their_PC_Flag; /* Center of theirs          */

  /* predicted positions */
  Vector BallPredictedPosition(int steps, float kick_power, AngleDeg kick_dir);
  inline Vector BallPredictedPosition(int steps = 1) 
    { return GetBall()->estimate_future_pos(steps); }


  inline Vector PlayerPredictedPosition(char side, Unum num, int steps = 1, Vector dash_per=Vector(0,0))
    { return GetPlayer(side,num)->estimate_future_pos(steps, 0, dash_per); }
  inline Vector TeammatePredictedPosition(Unum num, int steps = 1, Vector dash_per=Vector(0,0)) 
    { return PlayerPredictedPosition(MySide,num,steps, dash_per); }
  inline Vector OpponentPredictedPosition(Unum num, int steps = 1, Vector dash_per=Vector(0,0)) 
    { return PlayerPredictedPosition(TheirSide,num,steps, dash_per); }

  Bool BallWillBeKickable(int steps = 1, float dash_power = 0, float buffer = 0);

  Bool WillKickBeCollision(float kick_power, AngleDeg kick_dir, float buffer = 0) 
    { return ((BallPredictedPosition(1, kick_power, kick_dir) - MyPredictedPosition()).mod() <= 
	      SP_player_size + SP_ball_size + buffer)  ? TRUE : FALSE;  } 
  Bool WillDashBeCollision(float dash_power, float buffer = 0) 
    { return ((BallPredictedPosition() - MyPredictedPosition(1, dash_power)).mod() <= 
          SP_player_size + SP_ball_size + buffer) ? TRUE : FALSE;  } 

  Unum PlayerWithBall(float buffer=0);    /* with ball means actually kickable      */
  Unum TeammateWithBall(float buffer=0);  
  Unum OpponentWithBall(float buffer=0);  
  char TeamWithBall(float buffer=0);   

  /* Direct access functions */
  inline StationaryObject *GetMarker(MarkerType m) { return &Marker[m]; }

  inline BallObject *GetBall() { return &Ball; }

  inline PlayerObject **Teammates() { return MyPlayer; }
  inline PlayerObject **Opponents() { return TheirPlayer; }
  inline PlayerObject **TeamlessPlayers() { return TeamlessPlayer; }

  inline int NumTeammates() { return num_my_players; }
  inline int NumOpponents() { return num_their_players; }
  inline int NumTeamlessPlayers()  { return num_teamless_players; }
  inline int NumPlayers()   { return num_my_players + num_their_players + num_teamless_players; }

  PlayerObject *GetTeammate(Unum num);
  PlayerObject *GetOpponent(Unum num);
  PlayerObject *GetPlayer(char side, Unum num);

  /* Player data-structure management */
  PlayerObject *GetNewPlayer(char side, Unum num);
  Bool         ForgetAPlayer(char side);
  void         CleanMyPlayers();
  void         CleanTheirPlayers();
  void         CleanTeamlessPlayers();
  void         CleanAllPlayers();
  Bool         ResetMyDuplicatePlayers();
  Bool         ResetTheirDuplicatePlayers();

  MarkerType ClosestMarker;
  MarkerType ClosestMotionMarker;

  MarkerType ClosestGoal();
  MarkerType ClosestFlagTo();
  
  int SortPlayersBy(char side, char keyFunc, float keyNum, int* players);
                                    /* side = m,t,b (both) keyFunc = d,a
				       (dist/ang), keyNum = central val.
				       players = array.  Return num found */
  int SortPlayersByDistanceToPoint(char side, Vector point, Unum* players);
  int SortPlayersByDistanceToLine(char side, Line line, Unum* players, 
				  Bool TestEndPoints=FALSE, Vector ep1=0, Vector ep2=0);

  int NumTeammatesWithin(float Dist, Vector of_pos);
  inline int NumTeammatesWithin(float Dist) 
    { if (!MyConf()) my_error("nope"); return NumTeammatesWithin(Dist, MyPos()); }
  int NumOpponentsWithin(float Dist, Vector of_pos);
  inline int NumOpponentsWithin(float Dist) 
    { if (!MyConf()) my_error("nope"); return NumOpponentsWithin(Dist, MyPos()); }
  inline int NumPlayersWithin(float Dist, Vector of_pos)
    { return NumTeammatesWithin(Dist,of_pos) +  NumOpponentsWithin(Dist,of_pos); }
  inline int NumPlayersWithin(float Dist)   
    { if (!MyConf()) my_error("nope"); return NumPlayersWithin(Dist,MyPos()); }

  /* These are needed for the Decision Tree, Dodge player -- use dist and ang */
  int NumTeammatesWithin(float Dist, AngleDeg Ang, float ofDist, AngleDeg ofAng);
  inline int NumTeammatesWithin(float Dist, AngleDeg Ang) {return NumTeammatesWithin(Dist, Ang, 0, 0);}
  int NumOpponentsWithin(float Dist, AngleDeg Ang, float ofDist, AngleDeg ofAng);
  inline int NumOpponentsWithin(float Dist, AngleDeg Ang) {return NumOpponentsWithin(Dist, Ang, 0, 0);}
  inline int NumPlayersWithin  (float Dist, AngleDeg Ang, float ofDist, AngleDeg ofAng)
    { return NumTeammatesWithin(Dist,Ang,ofDist,ofAng) +  NumOpponentsWithin(Dist,Ang,ofDist,ofAng); }
  inline int NumPlayersWithin(float Dist, AngleDeg Ang)   {return NumPlayersWithin(Dist, Ang, 0, 0);}

  PlayerObject *GetPlayerWithin(float Dist, Vector ofPos);
  PlayerObject *GetPlayerWithin(float Dist, AngleDeg Ang, float ofDist, AngleDeg ofAng);

  int NumOpponentsInCone(float wid_dist_ratio, Vector end, Vector vert);
  int NumTeammatesInCone(float wid_dist_ratio, Vector end, Vector vert, Bool IncludeMe = FALSE);
  inline int NumPlayersInCone(float wid_dist_ratio, Vector end, Vector vert)
    { return NumOpponentsInCone(wid_dist_ratio, end, vert) +
	NumTeammatesInCone(wid_dist_ratio, end, vert); 
    }

  int NumOpponentsInCone(float wid_dist_ratio, Vector end)
    { return NumOpponentsInCone(wid_dist_ratio, end, MyPos()); }
  int NumTeammatesInCone(float wid_dist_ratio, Vector end)
    { return NumTeammatesInCone(wid_dist_ratio, end, MyPos()); }
  int NumPlayersInCone(float wid_dist_ratio, Vector end)
    { return NumPlayersInCone(wid_dist_ratio, end, MyPos()); }

  /* which is 'm' (my team), 't' (thier team), 'b' (both) */
  /* this should probably be private */
  int NumPlayersInConeToPlayer(char which,
			       float wid_dist_ratio, char side, Unum num,
			       float extra_len, Vector vert);

  int NumPlayersInConeToPlayer(float wid_dist_ratio, char side, Unum num, float extra_len, Vector vert)
    { return NumPlayersInConeToPlayer('b', wid_dist_ratio, side, num, extra_len, vert); }
  int NumTeammatesInConeToPlayer(float wid_dist_ratio, char side, Unum num, float extra_len, Vector vert)
    { return NumPlayersInConeToPlayer('m', wid_dist_ratio, side, num, extra_len, vert); }
  int NumOpponentsInConeToPlayer(float wid_dist_ratio, char side, Unum num, float extra_len, Vector vert)
    { return NumPlayersInConeToPlayer('t', wid_dist_ratio, side, num, extra_len, vert); }
  int NumPlayersInConeToTeammate(float wid_dist_ratio, Unum num, float extra_len, Vector vert)
    { return NumPlayersInConeToPlayer('b', wid_dist_ratio, MySide, num, extra_len, vert); }
  int NumTeammatesInConeToTeammate(float wid_dist_ratio, Unum num, float extra_len, Vector vert)
    { return NumPlayersInConeToPlayer('m', wid_dist_ratio, MySide, num, extra_len, vert); }
  int NumOpponentsInConeToTeammate(float wid_dist_ratio, Unum num, float extra_len, Vector vert)
    { return NumPlayersInConeToPlayer('t', wid_dist_ratio, MySide, num, extra_len, vert); }
  int NumPlayersInConeToOpponent(float wid_dist_ratio, Unum num, float extra_len, Vector vert)
    { return NumPlayersInConeToPlayer('b', wid_dist_ratio, TheirSide, num, extra_len, vert); }
  int NumTeammatesInConeToOpponent(float wid_dist_ratio, Unum num, float extra_len, Vector vert)
    { return NumPlayersInConeToPlayer('m', wid_dist_ratio, TheirSide, num, extra_len, vert); }
  int NumOpponentsInConeToOpponent(float wid_dist_ratio, Unum num, float extra_len, Vector vert)
    { return NumPlayersInConeToPlayer('t', wid_dist_ratio, TheirSide, num, extra_len, vert); }

  int NumPlayersInConeToPlayer(float wid_dist_ratio, char side, Unum num, float extra_len = 0.0)
    { return NumPlayersInConeToPlayer('b', wid_dist_ratio, side, num, extra_len, MyPos()); }
  int NumTeammatesInConeToPlayer(float wid_dist_ratio, char side, Unum num, float extra_len = 0.0)
    { return NumPlayersInConeToPlayer('m', wid_dist_ratio, side, num, extra_len, MyPos()); }
  int NumOpponentsInConeToPlayer(float wid_dist_ratio, char side, Unum num, float extra_len = 0.0)
    { return NumPlayersInConeToPlayer('t', wid_dist_ratio, side, num, extra_len, MyPos()); }
  int NumPlayersInConeToTeammate(float wid_dist_ratio, Unum num, float extra_len = 0.0)
    { return NumPlayersInConeToPlayer('b', wid_dist_ratio, MySide, num, extra_len, MyPos()); }
  int NumTeammatesInConeToTeammate(float wid_dist_ratio, Unum num, float extra_len = 0.0)
    { return NumPlayersInConeToPlayer('m', wid_dist_ratio, MySide, num, extra_len, MyPos()); }
  int NumOpponentsInConeToTeammate(float wid_dist_ratio, Unum num, float extra_len = 0.0)
    { return NumPlayersInConeToPlayer('t', wid_dist_ratio, MySide, num, extra_len, MyPos()); }
  int NumPlayersInConeToOpponent(float wid_dist_ratio, Unum num, float extra_len = 0.0)
    { return NumPlayersInConeToPlayer('b', wid_dist_ratio, TheirSide, num, extra_len, MyPos()); }
  int NumTeammatesInConeToOpponent(float wid_dist_ratio, Unum num, float extra_len = 0.0)
    { return NumPlayersInConeToPlayer('m', wid_dist_ratio, TheirSide, num, extra_len, MyPos()); }
  int NumOpponentsInConeToOpponent(float wid_dist_ratio, Unum num, float extra_len = 0.0)
    { return NumPlayersInConeToPlayer('t', wid_dist_ratio, TheirSide, num, extra_len, MyPos()); }

  Unum ClosestTeammateTo(Vector p, Bool include_me = TRUE);
  Unum ClosestOpponentTo(Vector p);
  Vector ClosestTeamlessPlayerPosition();
  Unum ClosestTeammateToBall();
  Unum ClosestOpponentToBall();
  float ClosestTeammateToBallDistance(); 

  inline int NumTeammatesCloserTo(Vector p) { return NumTeammatesWithin(DistanceTo(p),p) - 1; }
  inline int NumOpponentsCloserTo(Vector p) { return NumOpponentsWithin(DistanceTo(p),p); }
  inline int NumPlayersCloserTo(Vector p)   { return NumTeammatesCloserTo(p) + NumOpponentsCloserTo(p); }
  inline int NumTeammatesCloserToBall()     { return NumTeammatesCloserTo(BallAbsolutePosition()); }
  inline int NumOpponentsCloserToBall()     { return NumOpponentsCloserTo(BallAbsolutePosition()); }
  inline int NumPlayersCloserToBall()       { return NumPlayersCloserTo(BallAbsolutePosition()); }

  float PlayerDistanceTo(char s, Unum n, Vector p);
  inline float TeammateDistanceTo(Unum n, Vector p) { return PlayerDistanceTo(MySide,n,p); }
  inline float OpponentDistanceTo(Unum n, Vector p) { return PlayerDistanceTo(TheirSide,n,p); }

  float PlayerDistance2To(char s, Unum n, Vector p);
  inline float TeammateDistance2To(Unum n, Vector p) { return PlayerDistance2To(MySide,n,p); }
  inline float OpponentDistance2To(Unum n, Vector p) { return PlayerDistance2To(TheirSide,n,p); }

  float PlayerDistanceToLine(char s, Unum n, Line l);
  inline float TeammateDistanceToLine(Unum n, Line l) { return PlayerDistanceToLine(MySide,n,l); }
  inline float OpponentDistanceToLine(Unum n, Line l) { return PlayerDistanceToLine(TheirSide,n,l); }

  float PlayerDistance2ToLine(char s, Unum n, Line l);
  inline float TeammateDistance2ToLine(Unum n, Line l) { return PlayerDistance2ToLine(MySide,n,l); }
  inline float OpponentDistance2ToLine(Unum n, Line l) { return PlayerDistance2ToLine(TheirSide,n,l); }

  inline float PlayerDistanceToBall(char side, Unum num)  
    { if (!BallPositionValid()) my_error("Ball unknown"); 
      return PlayerDistanceTo(side,num,BallAbsolutePosition()); } 
  inline float TeammateDistanceToBall(Unum num) { return PlayerDistanceToBall(MySide,num); } 
  inline float OpponentDistanceToBall(Unum num) { return PlayerDistanceToBall(TheirSide,num); } 

  /* Which side of the field is the object on? */
  inline Fside BallLocationSide() { return LocationSide(BallAbsolutePosition()); }
  inline Fside TeammateLocationSide(Unum t) { return LocationSide(TeammateAbsolutePosition(t)); }
  inline Fside OpponentLocationSide(Unum o) { return LocationSide(OpponentAbsolutePosition(o)); }
  inline Fside PlayerLocationSide(char s, Unum p) { return LocationSide(PlayerAbsolutePosition(s,p)); }

  inline Unum ClosestTeammate() 
    { if (!MyConf()) my_error("I'm lost"); return ClosestTeammateTo(MyPos(),FALSE); }
  inline Unum ClosestOpponent() 
    { if (!MyConf()) my_error("I'm lost"); return ClosestOpponentTo(MyPos()); }


  Unum FurthestBackTeammate(Bool IncludeUs = TRUE);
  Unum FurthestBackOpponent();
  /* if we can't find any players, this will return (0,0) */
  Vector PositionOfFurthestBackPlayer(Bool IncludeUs = TRUE);

  Unum FurthestForwardTeammate(Bool IncludeUs = TRUE);
  Unum FurthestForwardOpponent(Bool IncludeGoalie = TRUE);
  /* if we can't find any players, this will return (0,0) */
  Vector PositionOfFurthestForwardPlayer(Bool IncludeUs = TRUE);

  float AngleBetweenClosestTwoOpponents(Vector p);

  Bool InOwnPenaltyArea(Vector p);
  Bool InOwnPenaltyArea();
  Bool BallInOwnPenaltyArea();
  Bool FacingBackInOwnPA();
  Bool FacingBackNearOwnGoal();

  Bool IsPointInBounds(float x, float y, float buffer=0);
  Bool IsPointInBounds(Vector pt, float buffer=0) { return IsPointInBounds(pt.x, pt.y, buffer); }
  Rectangle ShiftRectangleInBounds(Rectangle *rect);

  void update();
  void update_self_seen(Time time);
  void update_self(Time time);
  void update_ball(Time time);
  void update_players(Time time);
  void reconcile_unknown_players();

  void VerifyDash(float *dash_power);

  Vector PositionToKickoffPosition(const Vector pos);

  /* offside stuff */

  float my_offside_line;    /* In front of which I'm offside      */
  float their_offside_line; /* Behind which other team is offside */
  void  update_offside_lines();

  Bool OffsidePosition(float x, char side);
  inline Bool OffsidePosition(Vector pos, char side) { return OffsidePosition(pos.x,side); }
  Bool TeammateInOffsidePosition(Unum num);
  Bool OpponentInOffsidePosition(Unum num);
  Bool PlayerInOffsidePosition(char side, Unum num);
  inline Bool InOffsidePosition() { return TeammateInOffsidePosition(MyNumber); }

  float XToAdjustForOffsideX(float x, float buffer);
  inline float XToAdjustForOffsideX(float x) { return XToAdjustForOffsideX(x,CP_at_point_buffer); }
  inline Vector PositionToAdjustForOffsidePosition(Vector pos, float buffer)
    { pos.x = XToAdjustForOffsideX(pos.x,buffer); return pos; }
  inline Vector PositionToAdjustForOffsidePosition(Vector pos)
    { pos.x = XToAdjustForOffsideX(pos.x,CP_at_point_buffer); return pos; }
  Rectangle RectangleToAdjustForOffsideRectangle(Rectangle *rect, float buffer);
  inline Rectangle RectangleToAdjustForOffsideRectangle(Rectangle *rect)
    { return RectangleToAdjustForOffsideRectangle(rect,CP_at_point_buffer); }
  
  inline float XToOnsideX(float x, float buffer) { return Min(x, my_offside_line-buffer); }
  inline float XToOnsideX(float x) { return XToOnsideX(x,CP_at_point_buffer); }


  inline Vector PositionToOnsidePosition(Vector pos, float buffer) 
    { pos.x = XToOnsideX( pos.x, buffer ); return pos; }
  inline Vector PositionToOnsidePosition(Vector pos) 
    { return PositionToOnsidePosition(pos,CP_at_point_buffer); }

  Vector PositionToPullOffsidePosition(Vector pos, float buffer);
  inline Vector PositionToPullOffsidePosition(Vector pos) 
    { return PositionToPullOffsidePosition(pos,CP_at_point_buffer); }

  Bool  PullOffside;
  float PullOffsidePosition;

  /* congestion stuff */
#ifndef RELEASE_VERSION  
  float Congestion(Vector pos, Bool consider_me = FALSE);
  float TeammateCongestion(Unum teammate, Bool consider_me = TRUE);
  inline float MyCongestion() { return TeammateCongestion(MyNumber,FALSE); }
  Unum  LeastCongestedTeammate();
  Vector LeastCongestedValidPointInRectangle(Rectangle *rect, Bool attract = FALSE, Vector attract_point = 0);
  Vector LeastCongestedValidPointForPassFromInRectangle(Rectangle *rect, Vector from, 
					     Bool attract = FALSE, Vector attract_point = 0);
#endif

  inline void SetTeammateTired(Unum num, Time time) { TiredTimes[num] = time; }
  inline Bool TeammateTired(Unum num) 
    { 
      if ( num == MyNumber ) 
	return Tired();
      else 
	return (CurrentTime.t - CP_say_tired_interval > SecondLastStartClockTime.t &&
	      CurrentTime - CP_say_tired_interval < TiredTimes[num]) ? TRUE : FALSE; 
    }

  inline void SetAnnouncedTired(Time time) { SetTeammateTired(MyNumber,time); }
  inline Bool NeedToAnnounceTired()
    { if (!Tired()) my_error("I'm not tired"); 
      return (CurrentTime.t - CP_say_tired_interval > SecondLastStartClockTime.t &&
	      CurrentTime - CP_say_tired_interval < TiredTimes[MyNumber]) ? FALSE : TRUE; }
    
  Rectangle OwnPenaltyArea;
  Rectangle OwnGoalieArea;
  Rectangle TheirPenaltyArea;
  Rectangle TheirGoalieArea;
  Rectangle FieldRectangle;

  /* for the new position based elcoity correction */
  Time   sight_position_correction_time;
  Vector sight_position_correction;

private:

  Time TiredTimes[MAX_PLAYERS];

  PlayerObject *ClosestPlayerObjectTo(Vector gpos); 
  PlayerObject *ClosestTeammateObjectTo(Vector gpos);
  PlayerObject *ClosestOpponentObjectTo(Vector gpos);
  PlayerObject *ClosestPlayerObjectTo(char side, Vector gpos); 

  StationaryObject *Fieldline;
  StationaryObject *Marker;            
  BallObject       Ball;
  PlayerObject     *Player;        

  SideLine      SeenLine;
  MarkerType    SeenMarker[MAX_MARKERS];      /* for clearing the seen flag            */
  PlayerObject  *MyPlayer[MAX_PLAYERS];       /* pointers to the players on my team    */
  PlayerObject  *TheirPlayer[MAX_PLAYERS];    /* pointers to the players on their team */
  PlayerObject  *TeamlessPlayer[MAX_PLAYERS]; /* pointers to the players with no team  */
  PlayerObject  *FreePlayer[MAX_PLAYERS];     /* elements in Player not assigned       */

  TempPlayerObject  *UnknownPlayer;           /* Players seen with missing team or num */

  int num_seen_markers;
  int num_my_players;
  int num_their_players;
  int num_teamless_players;
  int num_free_players;
  int num_unknown_players;
  int num_players;         /* The maximum -- doesn't include me */
};

#endif
