/* -*- Mode: C -*- */

/* MemPlayer.h
 * CMUnited98 (soccer client for Robocup98)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1998 Peter Stone
 *
 * CMUnited-98 was created by Peter Stone, Manuela Veloso, and Patrick Riley
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* MemPlayer.C stores all the information relating to the client itself:
 * position on the field, speed, stamina, etc.
 */

#ifndef __player_h
#define __player_h

#include "netif.h"
#include "types.h"
#include "MemOption.h"
#include "utils.h"
#include "geometry.h"

class Command {
public:

  CMDType type;
  
  float power;
  float angle;
  float x;
  float y;
  Vqual qual;
  Vwidth width;

  char  command[MAXMESG];
  Time  time;

  Command() { type = CMD_none; }

  inline Bool valid() { return (Bool) (type != CMD_none); }
  inline Bool valid(Time t) { return (Bool) (type != CMD_none && t == time); }
};



/*****************************************/
/*****************************************/
/*****************************************/

class PlayerInfo : public OptionInfo{
public:

  PlayerInfo();
  virtual ~PlayerInfo();
  void Initialize();

  void SetPlayMode(Pmode mode);
#ifndef RELEASE_VERSION
  virtual void EndSetPlay() = 0; /* defined in MemSetplay.C */
#endif
  void sanitize_time(Time &tm);

  void EstimateMyPos();
  void EstimateMyVel(Time time);
  Vector NewVelFromDash(Vector old_vel, float dash_power);
  void UpdateFromMyAction(Time time);
  void update_self_estimate(Time time);
  void update_stamina(Time time);
  void reset_stamina();
  Time update_time(int time);

  virtual void VerifyDash(float *dash_power);

  Socket           *sock;

  char             SaveLogFileName[MAX_FILE_LEN];
  FILE             *SaveLogFile;
  int              SaveLogCounter;
  char             SaveSoundLogFileName[MAX_FILE_LEN];
  FILE             *SaveSoundLogFile;
  int              SaveSoundLogCounter;

  char             MySide;            // 'l' or 'r'                          
  char             TheirSide;         // 'l' or 'r'                          
  char             TheirTeamName[MAX_TEAMNAME_LEN];    // The name of their team                 
  int              MyTeamNameLen;     // strlen of MyTeamName
  Unum		   MyNumber;          // uniform number                      
  int              Initialized;					   
  Bool             ServerAlive;       // is the server going?
  Bool             TestVersion;       // is this the test version of the program?			   

  int              TimerInterval;
  Bool             ClockStopped;
  int              StoppedClockMSec;  // time elapsed in before_kick_off_mode (msec)

  Time             CurrentTime;       // Most recently observed time         
  Time             LastTime;          // previous time				   
  Time             LastSightTime;     // last time I saw			      
  Time             LastSenseTime;     // last time I sensed			   
  Time             LastSoundTime;     // last time I heard		      	   
					   
  int              LastSightInterval; // cycles between last 2 sights 
  inline Time      PreviousSightTime() { return LastSightTime - LastSightInterval; }  // previous time I saw 

  Command       *Action;
  Command       *LastAction;
  Command       ChangeView;
  inline Bool   ResendNeeded() 
    { return (RequestResend && CurrentTime == LastActionTime() 
	      && ResendTime == CurrentTime && LastActionType() == ResendType) ? TRUE : FALSE; }
  Bool          RequestResend;
  CMDType       ResendType;
  Time          ResendTime;

  inline AngleDeg LastActionAngle() { return LastAction->angle; }
  inline float    LastActionPower() { return LastAction->power; }
  inline float    LastActionX()     { return LastAction->y; }
  inline float    LastActionY()     { return LastAction->x; }
  inline CMDType  LastActionType()  { return LastAction->type; }
  inline Time     LastActionTime()  { return LastAction->time; }
  inline Bool     LastActionValid() { return LastAction->valid(); }
  inline Bool     LastActionValid(Time t) { return LastAction->valid(t); }

  Time             LastBehaveTime;           // last time into behave		   	  
  Time             LastActionOpTime;         // last time I could have acted		   	  
  Time             LastStartClockTime;       // time when server clock started again
  Time             SecondLastStartClockTime; // time when server clock started again

  Bool             NewSight;						      
  Bool             NewAction;
  Bool             FirstActionOpSinceLastSight;

  SenseType        LastSenseType;					   

  Vqual            ViewQuality;
  Vwidth           ViewWidth;
  Vwidth           LastViewWidth;
  Time             ViewWidthTime;
  

  Pmode            PlayMode;
  Time             PlayModeTime;
  Kmode            KickOffMode;    
  int              MyScore;
  int              TheirScore;

  AngleDeg         MyViewAngle(Time time);
  inline AngleDeg  MyViewAngle() { return MyViewAngle(CurrentTime); }
  Bool             InViewAngle(Time time, AngleDeg ang, float buffer=5.0);
  inline Bool      InViewAngle(AngleDeg ang, float buffer=5.0) { return InViewAngle(CurrentTime,ang,buffer); }
  int              MySightInterval();
  int              PredictedNextSightInterval();

  inline void      SetMyPos(Vector p, Time t)   { pos = p; conf = 1; my_pos_time = t;}
  inline void      SetMyAng(AngleDeg a)        { ang = a; }
  inline void      SetMyVel(Vector v, Time t)   { vel = v; vel_conf = 1; my_vel_time = t; }
  
  void  SetMySensedInfo(float st, float e, float sp, int k, int d, int tu, int sa, Time ti);
  float GetMySensedSpeed(Time time);
  float GetMySensedStamina(Time time);
  float GetMySensedEffort(Time time);
  int   GetMySensedKicks(Time time);
  int   GetMySensedDashes(Time time);
  int   GetMySensedTurns(Time time);
  int   GetMySensedSays(Time time);

  inline float MyStamina()  { return stamina; }
  inline float MyEffort()   { return effort; }
  inline float MyRecovery() { return recovery; }
  inline Bool  Tired()      
    { return ( MyStamina() < EffortDecThreshold + SP_max_power + CP_tired_buffer ) ? TRUE : FALSE; }

  float RecoveryDecThreshold;
  float EffortDecThreshold;
  float EffortIncThreshold;
  float CorrectDashPowerForStamina(float dash_power, float stamina, float effort, float recovery);
  inline float CorrectDashPowerForStamina(float dash_power, float stamina)
    { return CorrectDashPowerForStamina(dash_power, stamina, MyEffort(), MyRecovery()); }
  inline float CorrectDashPowerForStamina(float dash_power)
    { return CorrectDashPowerForStamina(dash_power, MyStamina(), MyEffort(), MyRecovery()); }

  inline Bool SensedInfoKnown(Time time) { return (Bool) (sense_time == time || prev_sense_time == time); }

  inline Vector    MyPos()   { if (!MyConf())    my_error("Don't know my pos"); return pos; }
  inline float     MyX()     { return MyPos().x; }
  inline float     MyY()     { return MyPos().y; }
  inline AngleDeg  MyAng()   { return ang; }
  inline Vector    MyVel()   { if (!MyVelConf()) my_error("Don't know my vel"); return vel; }
  inline float     MySpeed() { return MyVel().mod(); }
  inline float     MyDir()   { return MyVel().dir(); }
  inline float     MyConf()  { return (conf > CP_min_valid_conf) ? conf : 0; }
  inline float     MyVelConf() { return (vel_conf > CP_min_valid_conf) ? vel_conf : 0; }
  inline Time      MyPosTime() { return my_pos_time; }
  inline Time      MyVelTime() { return my_vel_time; }
  inline Time      MyUpdateTime() { return Min(my_pos_time,my_vel_time); }

  inline float     DistanceTo(Vector vec)  
    { if (!MyConf()) my_error("DistanceTo: no conf"); return (vec - pos).mod(); }
  inline AngleDeg  AngleTo   (Vector vec)
    { if (!MyConf()) my_error("AngleTo: no conf"); 
      AngleDeg angto = (vec - pos).dir() - ang; NormalizeAngleDeg(&angto); return angto; }
  inline Vector    Polar2Gpos(float dist, AngleDeg ang)
    { if (!MyConf()) my_error("Polar2Gpos: no conf"); 
      Vector rpos = Polar2Vector(dist,ang); return MyPos() + rpos.rotate(MyAng()); }

  inline Bool AtPos(Vector p, float buffer) 
    { return (DistanceTo(p) <= buffer) ? TRUE : FALSE; }
  inline Bool AtPos(Vector p)
    { return AtPos(p, CP_at_point_buffer); }
  
  inline Vector Global2RelativeToMe(Vector v)  { return v.Global2Relative(MyPos(), MyAng()); } 
  inline Vector RelativeToMe2Global(Vector v)  { return v.Relative2Global(MyPos(), MyAng()); } 

  /* Predicted Position */
  Vector MyPredictedPositionWithTurn(float angle,
				     int steps = 1, float dash_power = 0,
				     bool with_turn = TRUE);
  Vector MyPredictedPosition(int steps = 1, float dash_power = 0)
    { return MyPredictedPositionWithTurn(0.0, steps, dash_power, FALSE); }

  Vector MyPredictedPositionAtMaxSpeed(int steps = 1);
  
  /* Which side of the field am I on? */
  inline Fside LocationSide(Vector p) { return p.y < 0 ? FS_Left : FS_Right; } 
  inline Fside MyLocationSide() { return LocationSide(MyPos()); }

  /* Pure virtual function: the actual definition is in MemCommunicate.C */
#ifndef RELEASE_VERSION
   virtual void Say          (MessageType mtype)                 = 0;  
   virtual void Say          (MessageType mtype, int d1)         = 0;  
   virtual void Say          (MessageType mtype, int d1, int d2) = 0;  

   virtual void SayToPlayer  (Unum player, MessageType mtype)    = 0;  
   virtual void SayToPosition(Pnum position, MessageType mtype)  = 0;  
   virtual void SayToUnit    (int unit, MessageType mtype)       = 0;  
#endif

  int              kicks;
  int              dashes;
  int              turns;
  int              says;

protected:
  Time             my_pos_time;
  Time             my_vel_time;

  AngleDeg         my_last_ang;

private:
  Vector           pos;
  AngleDeg         ang;
  Vector           vel;
  float            conf;
  float            vel_conf;

  float            stamina;
  float            effort;
  float            recovery;

  Time             sense_time;
  Time             prev_sense_time;

  float            last_speed; /* from sense_body */
  float            prev_speed;

  float            last_stamina;
  float            prev_stamina;

  float            last_effort;
  float            prev_effort;

  int              last_kicks;
  int              prev_kicks;
  int              last_dashes;
  int              prev_dashes;
  int              last_turns;
  int              prev_turns;
  int              last_says;
  int              prev_says;
};

#endif



