/* -*- Mode: C -*- */

/* MemPlayer.C
 * CMUnited98 (soccer client for Robocup98)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1998 Peter Stone
 *
 * CMUnited-98 was created by Peter Stone, Manuela Veloso, and Patrick Riley
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* MemPlayer.C stores all the information relating to the client itself:
 * position on the field, speed, stamina, etc.
 */

#include "types.h"
#include "MemPlayer.h"
#include "client.h"

/*********************************************************************************/
/*********************************************************************************/
/*********************************************************************************/

PlayerInfo::PlayerInfo()
{
  Initialized = FALSE;
  ServerAlive = FALSE;
  ViewQuality = VQ_High;
  ViewWidth   = VW_Normal;
  NewSight = FALSE;
  FirstActionOpSinceLastSight = FALSE;
  ClockStopped = FALSE;
  StoppedClockMSec = 0;
  LastStartClockTime = Time(-1,0); /* If a problem, change back to 0,0 */
  SecondLastStartClockTime = LastStartClockTime;

  Action = new Command;
  LastAction = new Command;
  RequestResend = FALSE;

  last_dashes = prev_dashes = dashes = 0;
  last_turns  = prev_turns  = turns  = 0;
  last_kicks  = prev_kicks  = kicks  = 0;
  last_says   = prev_says   = says   = 0;

  TheirTeamName[0] = '\n';
}

/*********************************************************************************/

PlayerInfo::~PlayerInfo()
{
  if (CP_save_log)
    fclose(SaveLogFile);

  if (CP_save_sound_log)
    fclose(SaveSoundLogFile);

  delete Action;
  delete LastAction;
}

/*********************************************************************************/

void PlayerInfo::Initialize()
{
  /* printf("Calling Player Initialize\n"); */

  TheirSide = ( MySide == 'l' ? 'r' : 'l' );
  MyTeamNameLen = strlen(MyTeamName);

  TestVersion = (VP_test || ( MySide == 'l' && VP_test_l ) || ( MySide == 'r' && VP_test_r )) ? TRUE : FALSE;
  if ( TestVersion == TRUE ) printf("%d : test version\n",MyNumber);
  if ( VP_train_DT == TRUE ) printf("%d : training DT\n",MyNumber);

  MyScore    = IP_my_score;
  TheirScore = IP_their_score;

  if (CP_save_log){
    sprintf(SaveLogFileName,"Logfiles/%s%d-%c.log",MyTeamName,(int)MyNumber,MySide);
    SaveLogFile = fopen(SaveLogFileName,"w");
    SaveLogCounter = 0;
  }

  if (CP_save_sound_log){
    sprintf(SaveSoundLogFileName,"Logfiles/%s%d-%c-sounds.log",MyTeamName,(int)MyNumber,MySide);
    SaveSoundLogFile = fopen(SaveSoundLogFileName,"w");
    SaveSoundLogCounter = 0;
  }

  TimerInterval = SP_simulator_step/CP_senses_per_cycle;

  stamina = SP_stamina_max;
  effort = 1;
  recovery = 1;

  RecoveryDecThreshold = SP_stamina_max * SP_recover_dec_thr;
  EffortDecThreshold   = SP_stamina_max * SP_effort_dec_thr;
  EffortIncThreshold   = SP_stamina_max * SP_effort_inc_thr;

  my_vel_time = my_pos_time = 0;

  Initialized = TRUE;
}




/*********************************************************************************/
/*********************************************************************************/
/*********************************************************************************/

void PlayerInfo::SetPlayMode(Pmode mode)
{
  /* If clock is starting, save the old time */
  if ( ClockStopped ){
    ClockStopped = FALSE;
    SecondLastStartClockTime = LastStartClockTime;
    LastStartClockTime = LastActionOpTime;
    StoppedClockMSec = 0; 

    sanitize_time(CurrentTime);
    sanitize_time(LastSightTime);
    sanitize_time(LastSoundTime);
    sanitize_time(sense_time);
  }

  PlayMode = mode;
  PlayModeTime = CurrentTime;

  if ( mode == PM_Before_Kick_Off || mode == PM_My_Offside_Kick || mode == PM_Their_Offside_Kick ){
    if ( StoppedClockMSec != 0 ) my_error("StoppedClockMSec should have been reset already");
    ClockStopped = TRUE;
  }
  
  if (mode == PM_Half_Time || mode == PM_Extended_Time)
    reset_stamina();

#ifndef RELEASE_VERSION
  if ( PlayMode != PM_Play_On ) 
    EndSetPlay(); 
#endif

/* in others: */
//    if ( Mem->QActionTaken )
//      Mem->CloseRewards();

}

/*********************************************************************************/

void PlayerInfo::sanitize_time(Time &tm)
{
  if ( !LastStartClockTime ) return;

  /* This is to take care of times that were prematurely updated before we knew that
     the clock was about to start again */
  if ( tm.t == LastStartClockTime.t && tm.s > LastStartClockTime.s ){
    tm = Time( LastStartClockTime.t + (tm.s - LastStartClockTime.s), 0 );
  }
}

/*********************************************************************************/

void PlayerInfo::EstimateMyPos()
{
  /* Takes me from previous time to time */
  if ( MyConf() && MyVelConf() )
    pos += vel;
}

/*********************************************************************************/

void PlayerInfo::EstimateMyVel(Time time)
{
  if ( my_vel_time == time && vel_conf == CP_max_conf )
    return;
  
  /* Takes me from previous time to time */
  if ( SensedInfoKnown(time) ){
    float old_speed = vel.mod();
    if ( old_speed )
      vel = vel * GetMySensedSpeed(time)/old_speed; /* don't change the direction */
    else 
      vel = Polar2Vector( GetMySensedSpeed(time), MyAng() );  /* use my direction */
    vel_conf = CP_max_conf;
  }
  else if ( vel_conf < CP_max_conf && SensedInfoKnown(time-1) ) {
    vel = Polar2Vector( GetMySensedSpeed(time-1)*SP_player_decay, MyAng() );
    vel_conf = CP_conf_decay;
  }
  else if ( !MyVelConf() )
    return;
  else if ( my_vel_time == time-1 ){
    vel *= SP_player_decay;
    vel_conf *= CP_conf_decay;
  }
  else if ( my_vel_time > time-10 ){  /* missed up to 10 cycles */
    while ( my_vel_time < time && MyVelConf() ){
      vel *= SP_player_decay;
      vel_conf *= CP_conf_decay;
      ++my_vel_time;
    }
  }
  else
    my_error("Having trouble in vel update -- must have missed at least 10 cycles %.1f %.1f    %f",
	     (float)my_vel_time.t,(float)my_vel_time.s,MyVelConf());

  my_vel_time = time;
}

/*********************************************************************************/

Vector PlayerInfo::NewVelFromDash(Vector old_vel, float dash_power)
{
   float effective_power = MyEffort() * dash_power;
   effective_power *= SP_dash_power_rate;
   Vector new_vel = old_vel +  Polar2Vector( effective_power, MyAng() );

   if ( new_vel.mod() > SP_player_speed_max )
     new_vel *= ( SP_player_speed_max/new_vel.mod() );

   return new_vel;
}

/*********************************************************************************/

void PlayerInfo::VerifyDash(float *dash_power)
{
  /* Check if recovery going down, or max_speed exceeded */

  float new_stamina = MyStamina() -  MyEffort() * (fabs(*dash_power));
  if ( new_stamina <= SP_recover_dec_thr * SP_stamina_max && recovery > SP_recover_min ){
    /* printf("%d:%d.%d ",MyNumber,CurrentTime.t,CurrentTime.s); */
    /* printf("WARNING: recovery about to go to %.3f\n",recovery - SP_recover_dec); */
    ;
  }
  if ( new_stamina <= SP_effort_dec_thr * SP_stamina_max && effort > SP_effort_min ){
    /* printf("WARNING: effort about to go to %.2f\n",MyEffort() - SP_effort_dec); */
  }
  if ( new_stamina < 0 ){
    /* printf("%d:%d.%d ",MyNumber,CurrentTime.t,CurrentTime.s); */
    /* printf("WARNING: not enough stamina for dash\n"); */
    *dash_power = signf(*dash_power)*MyStamina()/MyEffort();
  }

  if ( NewVelFromDash( MyVel(), *dash_power ).mod() > SP_player_speed_max ){
    /* printf("%d:%d.%d ",MyNumber,CurrentTime.t,CurrentTime.s); */
    /* printf("WARNING: can't move that fast (assuming vel and dash in same dir)\n"); */
    /* printf("my speed %f   dash_power %f   ",MySpeed(),*dash_power); */
    *dash_power = signf(*dash_power)*(SP_player_speed_max - MySpeed())/(MyEffort()*SP_dash_power_rate);
    /* printf("new dash_power %f\n",*dash_power); */
  }
}

/*********************************************************************************/

void PlayerInfo::UpdateFromMyAction(Time time)
{
  /* Assume vel and pos are correct for time -- going to time+1 */
  if ( !MyConf() ) my_error("Can't update from action if not localized"); 
  /* But I'm pretty good at estimating... up conf_decay?? */
  if ( !NewAction || !(LastActionValid(time)) ) my_error("No action at that time");

  AngleDeg delta_ang, expected_delta;
  switch(LastActionType()){
  case CMD_turn:
    if ( my_pos_time > time ) break;
    /* be careful not to estimate in a turn that's already been seen -- 
       server updates turns instantaneously */
    delta_ang = GetNormalizeAngleDeg(ang - my_last_ang);
    expected_delta = LastActionAngle()/(1.0 + SP_inertia_moment * MySpeed());

    /* only if the change is closer to 0 than to the expected change */
    if ( fabs(delta_ang) < fabs(delta_ang-expected_delta) ){
      ang += expected_delta;
      NormalizeAngleDeg(&ang);
    }
    break;
  case CMD_dash:
    if ( my_vel_time > time ) break;
    vel = NewVelFromDash( vel, LastActionPower() );
    break;
  default: ;
  }
}

/*********************************************************************************/

void PlayerInfo::update_self_estimate(Time time)
{
  if ( !MyConf() ){
    vel_conf = 0;   /* If don't know my position, can't know my velocity */
    return;
  }

  if (CP_use_new_position_based_vel) {
    if ( my_pos_time == time ){ /* just vel */
      if ( my_vel_time == time )
	return;
      if ( NewAction && LastActionValid(my_vel_time) )
	UpdateFromMyAction(my_vel_time);
      
      EstimateMyVel(time);
    }
  } else {
    if ( my_pos_time == time ){ /* just vel */
      if ( my_vel_time == time ) my_error("my pos and vel already updated\n");
      if ( NewAction && LastActionValid(my_vel_time) )
	UpdateFromMyAction(my_vel_time);
      
      EstimateMyVel(time);
    }
  }
    
  while ( my_pos_time < time ){
    if ( NewAction && LastActionValid(my_pos_time) )
      UpdateFromMyAction(my_pos_time);

    ++my_pos_time;

    EstimateMyPos();
    EstimateMyVel(time);

    conf *= CP_conf_decay;
  }
}

/*********************************************************************************/

void PlayerInfo::update_stamina(Time time)
{
  if ( NewAction && LastActionType() == CMD_dash )
    stamina -= effort * fabs(LastActionPower());

  if ( stamina < 0 ) stamina = 0;

  if ( stamina <= SP_recover_dec_thr * SP_stamina_max && recovery > SP_recover_min ) {
    recovery -= SP_recover_dec;
  }

  if ( SensedInfoKnown(time) ){
    stamina = GetMySensedStamina(time);
    effort  = GetMySensedEffort(time);
  }
  else {
    if ( stamina <= SP_effort_dec_thr * SP_stamina_max && effort > SP_effort_min )
      effort -= SP_effort_dec;
    if (stamina >= SP_effort_inc_thr * SP_stamina_max && effort < 1.0){
      effort += SP_effort_inc;
      if ( effort > 1.0 )
	effort = 1.0;
    }
    stamina += recovery * SP_stamina_inc;
    if ( stamina > SP_stamina_max )
      stamina = SP_stamina_max;
  }
}

/*********************************************************************************/

void PlayerInfo::reset_stamina()
{
  /* right now, effort and recovery aren't reset */
  stamina = SP_stamina_inc;
  /* effort = 1.0; */
}

/*********************************************************************************/

Time PlayerInfo::update_time(int time)
{
  LastTime = CurrentTime;

  if ( ClockStopped ){
    if ( CurrentTime.t != time ){
      if ( CurrentTime.t == time - 1 ) /* Sometimes happens in offsides mode */
	CurrentTime = Time(time,0);
      else
	my_error("server time should be the same %d %d %d",CurrentTime.t, CurrentTime.s, time);
    }
    else 
      CurrentTime.s = StoppedClockMSec/SP_simulator_step;
  }
  else if ( LastStartClockTime.t == time )
    CurrentTime = LastStartClockTime;
  else
    CurrentTime = Time(time,0);
  
  return CurrentTime;
}




/*********************************************************************************/
/*********************************************************************************/
/*********************************************************************************/

AngleDeg PlayerInfo::MyViewAngle(Time time)
{
  AngleDeg view_angle = SP_visible_angle;
  Vwidth width;

  if ( time < ViewWidthTime ) 
    width = LastViewWidth;
  else 
    width = ViewWidth;

  if ( width == VW_Narrow ) view_angle /= 2;
  else if ( width == VW_Wide ) view_angle *= 2;

  return view_angle/2;
}

/*********************************************************************************/

Bool PlayerInfo::InViewAngle(Time time, AngleDeg ang, float buffer)
{
  if ( fabs(ang) < MyViewAngle(time) - buffer ) return TRUE;
  return FALSE;
}

/*********************************************************************************/

int PlayerInfo::MySightInterval()
{
  int interval = SP_send_step;
  
  if ( ViewWidth == VW_Narrow ) interval /= 2;
  else if ( ViewWidth == VW_Wide ) interval *= 2;

  if ( ViewQuality == VQ_Low ) interval /=2;

  return interval;
}

/*********************************************************************************/

int PlayerInfo::PredictedNextSightInterval()
{
  switch(MySightInterval()){
  case 37: 
  case 75: return 1;
  case 300: return 3;
  case 150: return (LastSightInterval <= 1 ? 2 : 1);
  default: my_error("Sight interval should be 37, 75, 150, or 300: %d",MySightInterval());
  }
  return 0;
}

/*********************************************************************************/

void PlayerInfo::SetMySensedInfo(float st, float e, float sp, int k, int d, int tu, int sa, Time ti)
{
  if ( sense_time == ti )
    return;

  prev_sense_time = sense_time;
  sense_time      = ti;

  prev_stamina    = last_stamina;
  last_stamina    = st;
  prev_effort     = last_effort;
  last_effort     = e;
  prev_speed      = last_speed;
  last_speed      = sp;

  prev_kicks     = last_kicks;
  last_kicks     = k;
  if ( last_kicks != kicks ){
     my_error("Server missed a kick at time %d (%d %d)",prev_sense_time.t,last_kicks,kicks);
     LastAction->type = CMD_none;
     kicks = last_kicks;
     Mem->GetBall()->forget_past_kick(LastAction->time);
     /* RequestResend = TRUE;
     ResendType    = CMD_kick;
     ResendTime    = LastActionTime(); */
  }

  prev_dashes    = last_dashes;
  last_dashes    = d;
  if ( last_dashes != dashes ){
     my_error("Server missed a dash at time %d (%d %d)",prev_sense_time.t,last_dashes,dashes);
     LastAction->type = CMD_none;
     dashes = last_dashes;
     /* RequestResend = TRUE;
     ResendType   = CMD_dash;
     ResendTime    = LastActionTime(); */
  }

  prev_turns     = last_turns;
  last_turns     = tu;
  if ( last_turns != turns ){
     my_error("Server missed a turn at time %d (%d %d)",prev_sense_time.t,last_turns,turns);
     LastAction->type = CMD_none;
     turns = last_turns;
     /* RequestResend = TRUE;
     ResendType   = CMD_turn;
     ResendTime    = LastActionTime(); */
  }

  prev_says      = last_says;
  last_says      = sa;
  if ( last_says != says ){
    says = last_says;
  }
}

/*********************************************************************************/

float PlayerInfo::GetMySensedSpeed(Time time){

  if (time == sense_time)
    return last_speed;
  if (time == prev_sense_time)
    return prev_speed;

  my_error("Don't know my speed at time %d",time.t);
  return 0;
}

/*********************************************************************************/

float PlayerInfo::GetMySensedStamina(Time time){

  if (time == sense_time)
    return last_stamina;
  if (time == prev_sense_time)
    return prev_stamina;

  my_error("Don't know my stamina at time %d",time.t);
  return 0;
}

/*********************************************************************************/

float PlayerInfo::GetMySensedEffort(Time time){

  if (time == sense_time)
    return last_effort;
  if (time == prev_sense_time)
    return prev_effort;

  my_error("Don't know my effort at time %d",time.t);
  return 0;
}

/*********************************************************************************/

int PlayerInfo::GetMySensedKicks(Time time){

  if (time == sense_time)
    return last_kicks;
  if (time == prev_sense_time)
    return prev_kicks;

  my_error("Don't know my kicks at time %d",time.t);
  return 0;
}

/*********************************************************************************/

int PlayerInfo::GetMySensedDashes(Time time){

  if (time == sense_time)
    return last_dashes;
  if (time == prev_sense_time)
    return prev_dashes;

  my_error("Don't know my dashes at time %d",time.t);
  return 0;
}

/*********************************************************************************/

int PlayerInfo::GetMySensedTurns(Time time){

  if (time == sense_time)
    return last_turns;
  if (time == prev_sense_time)
    return prev_turns;

  my_error("Don't know my turns at time %d",time.t);
  return 0;
}

/*********************************************************************************/

int PlayerInfo::GetMySensedSays(Time time){

  if (time == sense_time)
    return last_says;
  if (time == prev_sense_time)
    return prev_says;

  my_error("Don't know my says at time %d",time.t);
  return 0;
}

/*********************************************************************************/

float PlayerInfo::CorrectDashPowerForStamina(float dash_power, float stamina, float, float)
{
  int sign = Sign(dash_power);
  float new_power = fabs(dash_power);
  new_power = Min( new_power, stamina-(EffortDecThreshold+CP_tired_buffer) );

  if ( new_power < 0 ) new_power = 0;
  new_power *= sign;

  return new_power;
}

/*********************************************************************************/

Vector PlayerInfo::MyPredictedPositionAtMaxSpeed(int steps)
{
  if ( !MyConf() ) my_error("Can't estimate future if don't know present (max speed)");

  Vector new_position = MyPos();
  Vector max_velocity = Polar2Vector(SP_player_speed_max,MyAng());
  for (int i=0; i<steps; i++){
    new_position += max_velocity;
  }
  return new_position;
}

/*********************************************************************************/

Vector PlayerInfo::MyPredictedPositionWithTurn(float turn_ang,
					       int steps, float dash_power,
					       bool with_turn)
{
  if ( !MyConf() ) my_error("Can't estimate future if don't know present");
  
  float corrected_dash_power = dash_power;
  float effective_power;
  float predicted_stamina = MyStamina();
  float predicted_effort = MyEffort();
  float predicted_recovery = MyRecovery();
  float myang = MyAng();
  Vector position = MyPos();
  Vector velocity;
  if ( !MyVelConf() ) velocity = 0;
  else                velocity = MyVel();
  /* debug code 
  cout << "steps: " << steps << "\tpow: " << dash_power << "\tmyang: " << myang
       << "\tposition: " << position << "\tvel: " << velocity
       << "\tturn?: " << turn_first << "\tturn_ang: " << turn_angle
       << "\tstam: " << predicted_stamina << "\teff: " << predicted_effort
       << "\trec: " << predicted_recovery << endl; */
    
  for (int i=0; i<steps; i++){
    corrected_dash_power = CorrectDashPowerForStamina(dash_power,predicted_stamina);
    /* cout << " in func: i=" << i << "\tpos" << position << endl; */
    if (i==0 && with_turn) {
      myang += turn_ang;
      effective_power = 0;
    } else if (fabs(corrected_dash_power) > predicted_stamina)
      effective_power = Sign(corrected_dash_power) * predicted_stamina ;
    else
      effective_power = corrected_dash_power;
    
    effective_power *= predicted_effort;
    effective_power *= SP_dash_power_rate;
    velocity += Polar2Vector( effective_power, myang );
    /* cout << " in func: i=" << i << "\tvel" << velocity << endl; */
    
    if ( velocity.mod() > SP_player_speed_max )
      velocity *= ( SP_player_speed_max/velocity.mod() );
    
    position += velocity;
    velocity *= SP_player_decay;

    predicted_stamina -= predicted_effort * fabs(corrected_dash_power);
    if (predicted_stamina < 0) predicted_stamina = 0;

    if ( predicted_stamina <= SP_recover_dec_thr * SP_stamina_max && predicted_recovery > SP_recover_min ) {
      predicted_recovery -= SP_recover_dec;
    }

    if ( predicted_stamina <= SP_effort_dec_thr * SP_stamina_max && predicted_effort > SP_effort_min )
      predicted_effort -= SP_effort_dec;
    if (predicted_stamina >= SP_effort_inc_thr * SP_stamina_max && predicted_effort < 1.0){
      predicted_effort += SP_effort_inc;
      if ( predicted_effort > 1.0 )
	predicted_effort = 1.0;
    }
    predicted_stamina += predicted_recovery * SP_stamina_inc;
    if ( predicted_stamina > SP_stamina_max )
      predicted_stamina = SP_stamina_max;
  }

  /* cout << "returning " << position << endl; */
  return position;
  
}

