/* -*- Mode: C++ -*- */

/* MemOption.C
 * CMUnited98 (soccer client for Robocup98)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1998 Peter Stone
 *
 * CMUnited-98 was created by Peter Stone, Manuela Veloso, and Patrick Riley
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* MemOption.C contains all the client parameters
 * They can be specified as .conf files or command line options in the same style
 * as for soccerserver.  The code is based on Noda's soccerserver code.
 *
 * Parameters starting with SP_ are server parameters.  Most can be read from
 * the standard server.conf file
 *
 * Not all paramters listed here have meaning in the release version of CMUnited98
 */

#include "netif.h"
#include "MemOption.h"

/* setting defaults to match version 4.06 server.conf */
OptionInfo::OptionInfo(){

  VP_test_l   = FALSE;
  VP_test_r   = FALSE;
  VP_test     = FALSE;
  VP_train_DT = FALSE;
  VP_use_DT   = FALSE;

  IP_my_score             = 0;
  IP_their_score          = 0;
  IP_reconnect            = 0;

  sprintf(MyTeamName,     "CMUnited");

  /* no option flags for these */
  SP_pitch_length 	  = 105.0;
  SP_pitch_width 	  = 68.0;
  SP_pitch_margin	  = 5.0;
  SP_penalty_area_length  = 16.5;
  SP_penalty_area_width   = 40.32;
  SP_goal_area_length	  = 5.5;
  SP_goal_area_width	  = 18.32;
  SP_penalty_spot_dist    = 11.0;
  SP_corner_arc_r	  = 1.0;
  SP_free_kick_buffer     = 9.15;
  SP_after_goal_wait  	  = 50;
  SP_feel_distance        = 3.0;
  SP_num_lines            = 4;
  SP_num_markers          = 55;
  SP_unum_far_length      = 20.0;
  SP_unum_too_far_length  = 40.0;
  SP_team_far_length      = 40.0;
  SP_team_too_far_length  = 60.0;

  SP_version              = 4.06;
  SP_team_size            = 11;
  SP_half                 = 1;
  sprintf(SP_host,        "localhost");
  SP_goal_width           = 14.02;       
  SP_player_size          = 0.8;
  SP_player_decay         = 0.4;
  SP_player_rand 	  = 0.1;
  SP_player_weight	  = 60.0;
  SP_player_speed_max     = 32.0;
  SP_stamina_max	  = 2000.0;
  SP_stamina_inc          = 20.0;
  SP_recover_dec_thr      = 0.3;	
  SP_recover_dec          = 0.05; 	
  SP_recover_min          = 0.1; 	
  SP_effort_dec_thr       = 0.4;	
  SP_effort_min           = 0.1;	
  SP_effort_dec           = 0.05;	
  SP_effort_inc_thr       = 0.9;	
  SP_effort_inc           = 0.05;	
  SP_ball_size            = 0.085;		
  SP_ball_decay           = 0.96;	
  SP_ball_rand            = 0.05;		
  SP_ball_weight          = 0.2;;	
  SP_ball_speed_max       = 32.0;	
  SP_dash_power_rate      = 0.01;	
  SP_kick_power_rate      = 0.016;
  SP_kickable_margin      = 1.0;	
  SP_kickable_area        = SP_kickable_margin + SP_ball_size + SP_player_size;
  SP_catch_prob           = 1.0;	
  SP_catch_area_l         = 2.0;	
  SP_catch_area_w         = 1.0;	
  SP_catch_ban_cycle      = 5;
  SP_max_power            = 100;		
  SP_min_power            = -30;		
  SP_max_moment           = 180;		
  SP_min_moment           = -180;		
  SP_visible_angle        = 90.0;	
  SP_audio_cut_dist       = 50.0;	
  SP_dist_qstep           = 0.1;	
  SP_land_qstep           = 0.01;	
  SP_ckmargin             = 1.0;	
  SP_wind_dir             = 0.0;	
  SP_wind_force           = 0.0;
  SP_wind_rand            = 0.0;	
  SP_wind_none            = FALSE;
  SP_wind_random          = FALSE;
  SP_half_time            = 300;
  SP_port                 = 6000;
  SP_coach_port           = 6001;
  SP_simulator_step       = 100;
  SP_send_step            = 150;
  SP_recv_step            = 10;
  SP_say_msg_size         = 512;
  SP_hear_max             = 2;
  SP_hear_inc             = 1;
  SP_hear_decay           = 2;
  SP_coach_mode           = FALSE;
  SP_coach_w_referee_mode = FALSE;
  SP_use_offside          = FALSE;
  SP_forbid_kickoff_offside = TRUE;
  SP_verbose              = TRUE;
  SP_offside_area         = 9.15;
  SP_inertia_moment       = 5.0;

  CP_goalie               = FALSE;
  CP_save_log             = FALSE;
  CP_save_freq            = 10;
  CP_save_sound_log       = FALSE;
  CP_save_sound_freq      = 10;
  CP_senses_per_cycle     = 2;
  CP_max_conf             = 1.0;
  CP_min_valid_conf       = 0.5;
  CP_conf_decay           = 0.98;
  CP_player_conf_decay    = 0.99;
  CP_ball_conf_decay      = 0.9;
  CP_max_player_move_factor = 4;
  CP_max_say_interval     = 100;
  CP_ball_moving_threshold = .2; /* experimentally checked -- ball still, player fast => .15 ball speed */
  CP_dodge_angle_buffer    = 25;
  CP_dodge_distance_buffer = 3.5;
  CP_dodge_power           = 30;
  sprintf(CP_tree_stem,      "pass");
  CP_DT_evaluate_interval  = 10;
  CP_say_tired_interval    = 20;
  CP_tired_buffer          = 10;
  CP_set_plays             = FALSE;
  CP_Setplay_Delay         = 5;
  CP_Setplay_Max_Delay     = 100;
  CP_Setplay_Time_Limit    = 150;
  CP_kickable_buffer       = .1;
  CP_mark_persist_time     = 100;
  CP_track_min_distance    = 0;
  CP_track_max_distance    = 15;
  CP_pull_offsides         = FALSE;
  CP_pull_offsides_when_winning = TRUE;
  CP_spar                  = TRUE;
  CP_mark                  = TRUE;
  CP_take_stats            = FALSE;

  /* pat added these */
  CP_use_new_position_based_vel = TRUE;;

  CP_opt_ctrl_dist        = 1.6;//(SP_player_size + .8 *SP_kickable_margin);
  CP_KickTo_err           = 3;
  CP_closest_margin       = .97;//(SP_player_size + 2.0*SP_ball_size);
  CP_dokick_factor        = .22;
  CP_max_turn_kick_pow    = 20;
  CP_kick_time_space      = 1;
  CP_max_ignore_vel       = .005;
  CP_max_est_err          = .3;
  CP_hard_kick_margin     = .97;//(SP_player_size + 2.0*SP_ball_size);
  CP_hard_kick_err        = .1; 
  CP_hard_kick_factor     = .25;
  CP_max_hard_kick_angle_err  = 5;
  CP_max_go_to_point_angle_err = 5;
  CP_max_int_lookahead    = 20;
  CP_hardest_kick_angle_disp = 15;
  CP_hard_kick_end_turn_dist = 1.1;//(SP_player_size + .3 *SP_kickable_margin);
  CP_intercept_step       = 2;
  CP_turnball_opp_worry_dist = 5; 
  CP_collision_buffer = .1;
  CP_behind_angle  = 80;
  CP_time_for_full_rotation = 24; /* average guestimate */
  CP_ball_invalidation_dist_err = .3;

  CP_dribble_ball_dist    = 1.2;
  CP_dribble_ignore_opp_dist = 15;
  CP_dribble_worry_opp_dist = 4;
  CP_dribble_angle_norm = 60;
  CP_dribble_dodge_max_dist = 6;
  CP_dribble_dodge_angle_err = 15;
  CP_dribble_exp_angle_buffer = 10;
  CP_dribble_angle_ignore_buffer = 5;
  CP_dribble_dodge_close_dist = 2;
  CP_can_dribble_cone_ratio = .75;
  CP_dribble_towards_length = 10;
  CP_dribble_sideline_buffer = 1.5;
  CP_dribble_sphere_rad = 3;

  CP_move_imp_1v1_initial =  0.0;
  CP_move_imp_1v1_inc =       .2;
  CP_move_imp_1v1_threshold = 1.0 ;
  CP_at_point_buffer = 1;
  CP_overrun_dist = 3;
  CP_def_block_dist = 2;
  CP_def_block_dist_ratio = .5;
  CP_overrun_buffer = 2.5;
  CP_cycles_to_kick = 4;
  CP_breakaway_buffer = 3;
  CP_breakaway_kickable_buffer = 1.5;

  CP_static_kick_dist = .985;
  //CP_static_kick_ang = 47;  /* caculated value! */
  CP_static_kick_ang = 42;  /* caculated value!- extar buffer */
  CP_static_kick_ang_err = 15;//old: 5
  CP_static_kick_dist_err = .3;//old: .14
  CP_static_kick_overrun_dist = 4;

  CP_goalie_baseline_buffer = 1;
  CP_goalie_scan_angle_err = 10;
  CP_goalie_at_point_buffer = .1;
  CP_goalie_vis_angle_err = 5;
  CP_goalie_max_shot_distance = 40;
  CP_goalie_min_pos_dist = 15;
  CP_goalie_max_pos_dist = SP_pitch_length * .75;
  CP_goalie_max_forward_percent = .75;
  CP_goalie_ball_ang_for_corner = 90;
  CP_goalie_max_come_out_dist = 10;
  CP_goalie_ball_dist_for_corner = SP_penalty_area_length;
  CP_goalie_ball_dist_for_center = SP_pitch_length / 2;
  CP_goalie_free_kick_dist = 3;
  CP_goalie_go_to_ball_cone_ratio = .25;
  CP_goalie_warn_space = 10;
  CP_goalie_comes_out = TRUE;

  CP_clear_ball_ang_step = 5.0;
  CP_clear_ball_cone_ratio = .5;
  CP_clear_ball_max_dist = 30;
  CP_clear_offensive_min_horiz_dist = 20;
  CP_clear_offensive_min_angle = 60;

  CP_hardest_kick_shot_distance = 13;
  CP_moderate_kick_shot_distance = 9;
  CP_good_shot_distance = 20;
  CP_shot_distance = 30;
  CP_cycles_to_kick_buffer = 1;
  CP_breakaway_shot_distance = 16;

  sprintf(FP_initial_formation, "433");
  sprintf(FP_formation_when_tied, "initial");
  sprintf(FP_formation_when_losing, "initial");
  sprintf(FP_formation_when_losing_lots, "initial");
  sprintf(FP_formation_when_winning, "initial");
  sprintf(FP_initial_hc_method, "Obey");
  sprintf(FP_initial_mc_method, "Obey");
  FP_initial_player_1_pos     = 1;
  FP_initial_player_2_pos     = 2;
  FP_initial_player_3_pos     = 3;
  FP_initial_player_4_pos     = 4;
  FP_initial_player_5_pos     = 5;
  FP_initial_player_6_pos     = 6;
  FP_initial_player_7_pos     = 7;
  FP_initial_player_8_pos     = 8;
  FP_initial_player_9_pos     = 9;
  FP_initial_player_10_pos    = 10;
  FP_initial_player_11_pos    = 11;
  FP_goalie_number            = 0;
}

void OptionInfo::GetOptions(int argc, char **argv)
{
  option_t opt[] = {
    {"test_l",        		(void *)&VP_test_l,             V_BOOL},
    {"test_r",        		(void *)&VP_test_r,             V_BOOL},
    {"test",        		(void *)&VP_test,               V_BOOL},
    {"train_DT",                    (void *)&VP_train_DT,           V_BOOL},
    {"use_DT",                      (void *)&VP_use_DT,             V_BOOL},

    {"my_score",        		(void *)&IP_my_score,           V_INT},
    {"their_score",        		(void *)&IP_their_score,        V_INT},
    {"reconnect",                   (void *)&IP_reconnect,          V_INT},

    {"team_name",        		(void *)&MyTeamName,            V_STRING},
    {"goalie",        		(void *)&CP_goalie,             V_BOOL},
    {"save_log",        		(void *)&CP_save_log,           V_BOOL},
    {"save_freq",                   (void *)&CP_save_freq,          V_INT},
    {"save_sound_log",  		(void *)&CP_save_sound_log,     V_BOOL},
    {"save_sound_freq",             (void *)&CP_save_sound_freq,    V_INT},
    {"senses_per_cycle",            (void *)&CP_senses_per_cycle,   V_INT},
    {"max_conf",                    (void *)&CP_max_conf,           V_FLOAT},
    {"min_conf",                    (void *)&CP_min_valid_conf,     V_FLOAT},
    {"conf_decay",                  (void *)&CP_conf_decay,         V_FLOAT},
    {"player_conf_decay",           (void *)&CP_player_conf_decay,  V_FLOAT},
    {"ball_conf_decay",             (void *)&CP_ball_conf_decay,    V_FLOAT},
    {"max_player_move_factor",      (void *)&CP_max_player_move_factor, V_FLOAT},
    {"max_say_interval",            (void *)&CP_max_say_interval,   V_INT},
    {"ball_moving_threshold",       (void *)&CP_ball_moving_threshold, V_FLOAT},
    {"dodge_distance_buffer",       (void *)&CP_dodge_distance_buffer, V_FLOAT},
    {"dodge_angle_buffer",          (void *)&CP_dodge_angle_buffer, V_FLOAT},
    {"dodge_power",                 (void *)&CP_dodge_power,        V_FLOAT},
    {"tree_stem",                   (void *)&CP_tree_stem,          V_STRING},
    {"DT_evaluate_interval",        (void *)&CP_DT_evaluate_interval, V_INT},
    {"say_tired_interval",          (void *)&CP_say_tired_interval, V_INT},
    {"tired_buffer",                (void *)&CP_tired_buffer,       V_FLOAT},
    {"set_plays",                   (void *)&CP_set_plays,          V_BOOL},
    {"set_play_delay",              (void *)&CP_Setplay_Delay,     V_INT},
    {"set_play_time_limit",         (void *)&CP_Setplay_Time_Limit, V_INT},
    {"kickable_buffer",             (void *)&CP_kickable_buffer,    V_FLOAT},
    {"mark_persist_time",           (void *)&CP_mark_persist_time,  V_INT},
    {"track_max_distance",          (void *)&CP_track_max_distance, V_FLOAT},
    {"track_min_distance",          (void *)&CP_track_min_distance, V_FLOAT},
    {"pull_offsides",               (void *)&CP_pull_offsides,      V_BOOL},
    {"pull_offsides_when_winning",  (void *)&CP_pull_offsides_when_winning, V_ONOFF},
    {"spar",                        (void *)&CP_spar,            V_ONOFF},
    {"mark",                        (void *)&CP_mark,            V_ONOFF},
    {"take_stats",                  (void *)&CP_take_stats,      V_BOOL},

  
    {"use_new_position_based_vel",  (void *)&CP_use_new_position_based_vel, V_BOOL},
    {"opt_ctrl_dist",               (void *)&CP_opt_ctrl_dist,      V_FLOAT},
    {"KickTo_err",                  (void *)&CP_KickTo_err,         V_FLOAT},
    {"closest_margin",              (void *)&CP_closest_margin,     V_FLOAT},
    {"dokick_factor",               (void *)&CP_dokick_factor,      V_FLOAT},
    {"max_turn_kick_pow",           (void *)&CP_max_turn_kick_pow,  V_FLOAT},
    {"kick_time_space",             (void *)&CP_kick_time_space,    V_INT},
    {"max_ignore_vel",              (void *)&CP_max_ignore_vel,     V_FLOAT},
    {"max_est_err",                 (void *)&CP_max_est_err,        V_FLOAT},
    {"hard_kick_margin",            (void *)&CP_hard_kick_margin,   V_FLOAT},
    {"hard_kick_err",               (void *)&CP_hard_kick_err,      V_FLOAT},
    {"hard_kick_factor",            (void *)&CP_hard_kick_factor,   V_FLOAT},
    {"max_hard_kick_angle_err",     (void *)&CP_max_hard_kick_angle_err, V_INT},		
    {"max_go_to_point_angle_err",   (void *)&CP_max_go_to_point_angle_err, V_INT},		
    {"max_int_lookahead",           (void *)&CP_max_int_lookahead,  V_INT},		
    {"intercept_close_dist",        (void *)&CP_intercept_close_dist,  V_FLOAT},		
    {"intercept_step",              (void *)&CP_intercept_step,     V_INT},		
    {"hardest_kick_angle_disp",     (void *)&CP_hardest_kick_angle_disp,  V_INT},		
    {"hard_kick_end_turn_dist",     (void *)&CP_hard_kick_end_turn_dist,  V_FLOAT},		
    {"turnball_opp_worry_dist",     (void *)&CP_turnball_opp_worry_dist,V_FLOAT},		
    {"collision_buffer",            (void *)&CP_collision_buffer,  V_FLOAT},
    {"behind_angle",                (void *)&CP_behind_angle, V_FLOAT},
    {"ball_invalidation_dist_err",  (void *)&CP_ball_invalidation_dist_err , V_FLOAT},
    {"time_for_full_rotation",      (void *)&CP_time_for_full_rotation, V_INT},
    {"dribble_ball_dist",           (void *)&CP_dribble_ball_dist,  V_FLOAT},		
    {"dribble_ignore_opp_dist",     (void *)&CP_dribble_ignore_opp_dist, V_FLOAT},		
    {"dribble_worry_opp_dist",      (void *)&CP_dribble_worry_opp_dist, V_FLOAT},		
    {"dribble_angle_norm",          (void *)&CP_dribble_angle_norm, V_FLOAT},		
    {"dribble_dodge_max_dist",      (void *)&CP_dribble_dodge_max_dist, V_FLOAT},		
    {"dribble_dodge_angle_err",     (void *)&CP_dribble_dodge_angle_err, V_FLOAT},		
    {"dribble_exp_angle_buffer",    (void *)&CP_dribble_exp_angle_buffer, V_FLOAT},		
    {"dribble_angle_ignore_buffer", (void *)&CP_dribble_angle_ignore_buffer, V_FLOAT},		
    {"dribble_dodge_close_dist",    (void *)&CP_dribble_dodge_close_dist, V_FLOAT},
    {"can_dribble_cone_ratio",      (void *)&CP_can_dribble_cone_ratio, V_FLOAT},
    {"dribble_towards_length",      (void *)&CP_dribble_towards_length, V_FLOAT},
    {"dribble_sideline_buffer",     (void *)&CP_dribble_sideline_buffer, V_FLOAT},
    {"dribble_sphere_rad",          (void *)&CP_dribble_sphere_rad, V_FLOAT},

 
    {"move_imp_1v1_initial",        (void *)&CP_move_imp_1v1_initial, V_FLOAT},
    {"move_imp_1v1_inc",            (void *)&CP_move_imp_1v1_inc, V_FLOAT},
    {"move_imp_1v1_threshold",      (void *)&CP_move_imp_1v1_threshold, V_FLOAT},
    {"at_point_buffer",             (void *)&CP_at_point_buffer, V_FLOAT},
    {"overrun_dist",                (void *)&CP_overrun_dist, V_FLOAT},
    {"def_block_dist",              (void *)&CP_def_block_dist, V_FLOAT},
    {"def_block_dist_ratio",        (void *)&CP_def_block_dist_ratio, V_FLOAT},
    {"overrun_buffer",              (void *)&CP_overrun_buffer, V_FLOAT},
    {"cycles_to_kick",              (void *)&CP_cycles_to_kick, V_FLOAT},
    {"breakaway_buffer",            (void *)&CP_breakaway_buffer, V_FLOAT},
    {"breakaway_kickable_buffer",   (void *)&CP_breakaway_kickable_buffer, V_FLOAT},
    {"static_kick_dist",            (void *)&CP_static_kick_dist, V_FLOAT},
    {"static_kick_ang",             (void *)&CP_static_kick_ang, V_FLOAT},
    {"static_kick_ang_err",         (void *)&CP_static_kick_ang_err, V_FLOAT},
    {"static_kick_dist_err",        (void *)&CP_static_kick_dist_err, V_FLOAT},
    {"static_kick_overrun_dist",    (void *)&CP_static_kick_overrun_dist, V_FLOAT},

    {"goalie_baseline_buffer",      (void *)&CP_goalie_baseline_buffer, V_FLOAT},
    {"goalie_scan_angle_err",       (void *)&CP_goalie_scan_angle_err, V_FLOAT},
    {"goalie_at_point_buffer",      (void *)&CP_goalie_at_point_buffer, V_FLOAT},
    {"goalie_vis_angle_err",        (void *)&CP_goalie_vis_angle_err, V_FLOAT},
    {"goalie_max_shot_distance",    (void *)&CP_goalie_max_shot_distance, V_FLOAT},
    {"goalie_min_pos_dist",         (void *)&CP_goalie_min_pos_dist, V_FLOAT},
    {"goalie_max_pos_dist",         (void *)&CP_goalie_max_pos_dist, V_FLOAT},
    {"goalie_max_forward_percent",  (void *)&CP_goalie_max_forward_percent, V_FLOAT},
    {"goalie_ball_ang_for_corner",  (void *)&CP_goalie_ball_ang_for_corner, V_FLOAT},
    {"goalie_max_come_out_dist",    (void *)&CP_goalie_max_come_out_dist, V_FLOAT},
    {"goalie_ball_dist_for_corner", (void *)&CP_goalie_ball_dist_for_corner, V_FLOAT},
    {"goalie_ball_dist_for_center", (void *)&CP_goalie_ball_dist_for_center, V_FLOAT},
    {"goalie_free_kick_dist",       (void *)&CP_goalie_free_kick_dist, V_FLOAT},
    {"goalie_go_to_ball_cone_ratio",(void *)&CP_goalie_go_to_ball_cone_ratio, V_FLOAT},
    {"goalie_warn_space",           (void *)&CP_goalie_warn_space, V_INT},
    {"goalie_comes_out",            (void *)&CP_goalie_comes_out, V_ONOFF},

    {"clear_ball_ang_step",         (void *)&CP_clear_ball_ang_step, V_FLOAT},
    {"clear_ball_cone_ratio",       (void *)&CP_clear_ball_cone_ratio, V_FLOAT},
    {"clear_ball_max_dist",         (void *)&CP_clear_ball_max_dist, V_FLOAT},
    {"clear_offensive_min_horiz_dist", (void *)&CP_clear_offensive_min_horiz_dist, V_FLOAT},
    {"clear_offensive_min_angle",   (void *)&CP_clear_offensive_min_angle, V_FLOAT},

    {"hardest_kick_shot_distance",  (void *)&CP_hardest_kick_shot_distance, V_FLOAT},
    {"moderate_kick_shot_distance", (void *)&CP_moderate_kick_shot_distance, V_FLOAT},
    {"good_shot_distance",          (void *)&CP_good_shot_distance, V_FLOAT},
    {"shot_distance",               (void *)&CP_shot_distance, V_FLOAT},
    {"cycles_to_kick_buffer",       (void *)&CP_cycles_to_kick_buffer, V_INT},
    {"breakaway_shot_distance",     (void *)&CP_breakaway_shot_distance, V_FLOAT},
		
    {"formation",                   (void *)&FP_initial_formation,  V_STRING},
    {"formation_when_losing",       (void *)&FP_formation_when_losing,  V_STRING},
    {"formation_when_losing_lots",  (void *)&FP_formation_when_losing_lots,  V_STRING},
    {"formation_when_winning",      (void *)&FP_formation_when_winning,  V_STRING},
    {"formation_when_tied",         (void *)&FP_formation_when_tied,  V_STRING},

    {"home_change",                 (void *)&FP_initial_hc_method,  V_STRING},
    {"mark_change",                 (void *)&FP_initial_mc_method,  V_STRING},
    {"player_1_pos",                (void *)&FP_initial_player_1_pos,  V_INT},
    {"player_2_pos",                (void *)&FP_initial_player_2_pos,  V_INT},
    {"player_3_pos",                (void *)&FP_initial_player_3_pos,  V_INT},
    {"player_4_pos",                (void *)&FP_initial_player_4_pos,  V_INT},
    {"player_5_pos",                (void *)&FP_initial_player_5_pos,  V_INT},
    {"player_6_pos",                (void *)&FP_initial_player_6_pos,  V_INT},
    {"player_7_pos",                (void *)&FP_initial_player_7_pos,  V_INT},
    {"player_8_pos",                (void *)&FP_initial_player_8_pos,  V_INT},
    {"player_9_pos",                (void *)&FP_initial_player_9_pos,  V_INT},
    {"player_10_pos",               (void *)&FP_initial_player_10_pos, V_INT},
    {"player_11_pos",               (void *)&FP_initial_player_11_pos, V_INT},
    {"goalie_number",               (void *)&FP_goalie_number        , V_INT},

    {"version",        		(void *)&SP_version,          V_FLOAT},
    {"team_size",        		(void *)&SP_team_size,          V_INT},
    {"half",        		(void *)&SP_half,          V_INT},
    {"host",             		(void *)&SP_host,               V_STRING},
    {"goal_width",   		(void *)&SP_goal_width,         V_FLOAT},
    {"player_size",   		(void *)&SP_player_size,	V_FLOAT},
    {"player_decay",   	        (void *)&SP_player_decay,    	V_FLOAT},
    {"player_rand",   		(void *)&SP_player_rand,	V_FLOAT},
    {"player_weight",   	        (void *)&SP_player_weight,	V_FLOAT},
    {"player_speed_max",            (void *)&SP_player_speed_max,	V_FLOAT},
    {"stamina_max",   		(void *)&SP_stamina_max,	V_FLOAT},
    {"stamina_inc_max",   	        (void *)&SP_stamina_inc,	V_FLOAT},
    {"recover_dec_thr",   	        (void *)&SP_recover_dec_thr,	V_FLOAT},
    {"recover_min",   		(void *)&SP_recover_min, 	V_FLOAT},
    {"recover_dec",   		(void *)&SP_recover_dec, 	V_FLOAT},
    {"effort_dec_thr",   	        (void *)&SP_effort_dec_thr,	V_FLOAT},
    {"effort_min",   		(void *)&SP_effort_min, 	V_FLOAT},
    {"effort_dec",   		(void *)&SP_effort_dec, 	V_FLOAT},
    {"effort_inc_thr",   	        (void *)&SP_effort_inc_thr,	V_FLOAT},
    {"effort_inc",   		(void *)&SP_effort_inc,	        V_FLOAT},
    {"ball_size",   		(void *)&SP_ball_size,		V_FLOAT},
    {"ball_decay",   		(void *)&SP_ball_decay, 	V_FLOAT},
    {"ball_rand",   		(void *)&SP_ball_rand,		V_FLOAT},
    {"ball_weight",   		(void *)&SP_ball_weight,	V_FLOAT},
    {"ball_speed_max",   	        (void *)&SP_ball_speed_max,	V_FLOAT},
    {"dash_power_rate",   	        (void *)&SP_dash_power_rate,	V_FLOAT},
    {"kick_power_rate",   	        (void *)&SP_kick_power_rate,	V_FLOAT},
    {"kickable_margin",   	        (void *)&SP_kickable_margin,	V_FLOAT},
    {"catch_probability",   	(void *)&SP_catch_prob, 	V_FLOAT},
    {"catchable_area_l",            (void *)&SP_catch_area_l,	V_FLOAT},
    {"catchable_area_w",            (void *)&SP_catch_area_w,	V_FLOAT},
    {"maxpower",   		        (void *)&SP_max_power,		V_FLOAT},
    {"minpower",   		        (void *)&SP_min_power,		V_FLOAT},
    {"maxmoment",   		(void *)&SP_max_moment,		V_FLOAT},
    {"minmoment",   		(void *)&SP_min_moment,		V_FLOAT},
    {"visible_angle",   	        (void *)&SP_visible_angle,	V_FLOAT},
    {"visible_distance",            (void *)&SP_visible_dist,	V_FLOAT},
    {"audio_cut_dist",   	        (void *)&SP_audio_cut_dist,	V_FLOAT},
    {"quantize_step",   	        (void *)&SP_dist_qstep, 	V_FLOAT},
    {"quantize_step_l",   	        (void *)&SP_land_qstep, 	V_FLOAT},
    {"ckick_margin",   	        (void *)&SP_ckmargin,   	V_FLOAT},
    {"wind_dir",   		        (void *)&SP_wind_dir,   	V_FLOAT},
    {"wind_force",   		(void *)&SP_wind_force,  	V_FLOAT},
    {"wind_rand",   		(void *)&SP_wind_rand,  	V_FLOAT},
    {"wind_none",   		(void *)&SP_wind_none,  	V_BOOL},
    {"wind_random",   		(void *)&SP_wind_random,	V_BOOL},
    {"half_time",   		(void *)&SP_half_time,  	V_INT},
    {"port",   			(void *)&SP_port,       	V_INT},
    {"coach_port",   		(void *)&SP_coach_port,  	V_INT},
    {"simulator_step",   	        (void *)&SP_simulator_step,	V_INT},
    {"send_step",   		(void *)&SP_send_step,  	V_INT},
    {"recv_step",   		(void *)&SP_recv_step,  	V_INT},
    {"say_msg_size",   	        (void *)&SP_say_msg_size,	V_INT},
    {"hear_max",   		        (void *)&SP_hear_max,   	V_INT},
    {"hear_inc",   		        (void *)&SP_hear_inc,   	V_INT},
    {"hear_decay",   		(void *)&SP_hear_decay,  	V_INT},
    {"catch_ban_cycle",             (void *)&SP_catch_ban_cycle,	V_INT},
    {"coach",   			(void *)&SP_coach_mode,  	V_BOOL},
    {"coach_w_referee",   	        (void *)&SP_coach_w_referee_mode,V_BOOL},
    {"use_offside",   		(void *)&SP_use_offside,	V_ONOFF},
    {"forbid_kick_off_offside",     (void *)&SP_forbid_kickoff_offside, V_ONOFF},
    {"log_file",		        (void *)&SP_logfile,		V_STRING},
    {"record",			(void *)&SP_recfile,		V_STRING},
    {"record_log",		        (void *)&SP_rec_log,		V_BOOL},
    {"record_version",	        (void *)&SP_rec_ver,		V_INT},
    {"send_log",	        (void *)&SP_send_log,		V_BOOL},
    {"replay",			(void *)&SP_replay,		V_STRING},
    {"verbose",                     (void *)&SP_verbose,		V_ONOFF},
    {"offside_active_area_size",    (void *)&SP_offside_area,	V_FLOAT},
    {"inertia_moment",      	(void *)&SP_inertia_moment,     V_FLOAT},
    {"\0",				NULL, 			       	0}
  } ;

  /* skip command name */
  argv++ ; argc-- ;

  /* first, search option '-file' */
  int i ;
  FILE *fp ;
  for(i = 0 ; i < argc ; i++) {
    if (!strcmp(*(argv + i),"-file")) {
      if ((fp = fopen(*(argv+i+1),"r")) == NULL) {
	cerr << "cat't open config file " << *(argv+i+1) << endl ;
	break ;
      }
			
      char buf[100] ;
      while(fgets(buf,100,fp) != NULL) {
				/* ignore remark line */
	if (buf[0] == '#' || buf[0] == '\n')
	  continue ;

				/* replace from ':' to ' ' */
	char *t = buf ;
	while(*t != NULLCHAR) {
	  if (*t == ':') *t = ' ' ;
	  t++ ;
	}

	int n, p ;
	char com[256] ;
	char onoff[16] ;
	n = sscanf(buf,"%s", com) ;
	if (n < 1) {
	  cerr << "Illegal line : " << buf ;
	  continue ;
	}

	for (p = 0 ; opt[p].vptr != NULL ; p++) {
	  if (strcmp(com, opt[p].optname))
	    continue ;

	  /* match */
	  switch(opt[p].vsize) {
	  case V_INT:
	    n = sscanf(buf, "%s %d", com, (int *)opt[p].vptr) ;
	    break ;

	  case V_STRING:
	    n = sscanf(buf, "%s %s", com, (char *)opt[p].vptr) ;
	    break ;

	  case V_FLOAT:
	    n = sscanf(buf, "%s %f", com, (float *)opt[p].vptr) ;
	    break ;

	  case V_BOOL:
	    n = 2 ;
	    *((Bool *)opt[p].vptr) = TRUE ;
	    break ;

	  case V_ONOFF:
	    n = sscanf(buf, "%s %s", com, onoff) ;
	    *((Bool *)opt[p].vptr) = (!strcmp(onoff, "on")) ? TRUE :FALSE;
	    break ;
	  }

	  if (n < 2)
	    cerr << "Illegal line (" << com << ") " << endl ;

	  break ;
	}
				
	if (opt[p].vptr == NULL)
	  cerr << "Illegal line (" << com << ") " << endl ;
      }

      fclose(fp) ;
      /* break ; */   // Without this, more than one file can be read
    }
  }

  /* next, analyze command line option */
  int p ;

  while (argc) {
    if (!strcmp(*argv, "-file")) {
      argv += 2 ;
      argc -= 2 ;
      continue ;
    }
			
    for (p = 0 ; opt[p].vptr != NULL ; p++) {
      if (strcmp(*argv + 1, opt[p].optname))
	continue ;

      /* match */
      argv++ ;
      argc-- ;

      switch(opt[p].vsize) {
      case V_INT:
	*((int *)opt[p].vptr) = atoi(*argv) ;
	break ;

      case V_STRING:
	strcpy((char *)opt[p].vptr, *argv) ;
	break ;

      case V_FLOAT:
	*((float *)opt[p].vptr) = atof(*argv) ;
	break ;

      case V_BOOL:
	*((Bool *)opt[p].vptr) = TRUE ;
	argv-- ;
	argc++ ;
	break ;

      case V_ONOFF:
	*((Bool *)opt[p].vptr) = (!strcmp(*argv, "on")) ? TRUE : FALSE ;
	break ;
      }

      break ;
    }

    if (opt[p].vptr == NULL)
      cerr << "Unrecognized Option : " << *argv << endl ;

    argv++ ;
    argc-- ;
  }

  SP_half_time = SP_half_time * 1000 / SP_simulator_step ;
  SP_kickable_area = SP_kickable_margin + SP_ball_size + SP_player_size ;
}



