/* usetrees.c
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

#include "global.h"
#include "trees/consult.h"
#include "trees/getnames.h"
#include "trees/trees.h"
#include "trees/userint.h"
#include "trees/defns.i"

#define CLOSEST_COMPARE 0
#define TREE_DEBUG      0

/*****************************************************************/

int ChooseReceiverRandom(){
  
  int teammates[TEAM_SIZE];
  int NumOpen = Mem->FindTeammatesWithStatus(OPEN_STAT, teammates);

  if ( NumOpen == 0 )
    return FALSE;

  if ( NumOpen < 3 ) /* HACK for testing pass3.tree: wait for 3 open receivers */
    return FALSE;

  for (int i=0; i<NumOpen; i++){  /* Some aren't visible */
    if ( !Mem->PlayerDistanceValid(Mem->MySide,teammates[i]) ||
	 Mem->GetPlayerDistance(Mem->MySide,teammates[i]) < 10 ){
      FACETEAMMATE(teammates[i]);
      return FALSE;
    }
  }

  int receiver = teammates[int_random(NumOpen)];  /* Choose one randomly */

  if ( receiver > 0 &&   /* This is redundant now */
       Mem->PlayerDistanceValid(Mem->MySide,receiver) &&
       Mem->GetPlayerDistance(Mem->MySide,receiver) < 10){  
    /* Ignore players who are too close */
    Mem->SetStatus(Mem->MySide,receiver,BLANK_STAT);
    receiver = ChooseReceiverRandom();
  }

  return receiver;
}

/*****************************************************************/

int ChooseReceiverClosest(){
  
  int teammates[TEAM_SIZE];
  int NumOpen = Mem->FindTeammatesWithStatus(OPEN_STAT, teammates);

  if ( NumOpen == 0 )
    return FALSE;

  if ( NumOpen < 3 ) /* HACK for testing pass3.tree: wait for 3 open receivers */
    return FALSE;

  for (int i=0; i<NumOpen; i++){  /* Some aren't visible */
    if ( !Mem->PlayerDistanceValid(Mem->MySide,teammates[i]) ||
	 Mem->GetPlayerDistance(Mem->MySide,teammates[i]) < 10 ){
      FACETEAMMATE(teammates[i]);
      return FALSE;
    }
  }

  int receiver = teammates[0];  
  for (int i=1; i<NumOpen; i++){
    if ( Mem->GetPlayerDistance(Mem->MySide,teammates[i]) <
	 Mem->GetPlayerDistance(Mem->MySide,receiver) ){
      receiver = teammates[i];
    }
  }

  if ( receiver > 0 &&
       Mem->PlayerDistanceValid(Mem->MySide,receiver) &&
       Mem->GetPlayerDistance(Mem->MySide,receiver) < 10){  
    /* Ignore players who are too close */
    Mem->SetStatus(Mem->MySide,receiver,BLANK_STAT);
    receiver = ChooseReceiverRandom();
  }

  for (int i=0; i<3; i++)
    printf("Receiver %d: %f\n",teammates[i],Mem->GetPlayerDistance(Mem->MySide,teammates[i]));
  printf(">> Passing to %d\n",receiver);

  return receiver;
}

/*****************************************************************/

void FillAttributes(int receiver){
  if ( !Mem->TeammateAngleValid(receiver) )
    printf("%d: Need to know player (%d) angle to get tree stats!!\n",
	   Mem->MyNumber,receiver);

  float angle = Mem->GetTeammateAngle(receiver);
  float distance;

  if ( !Mem->TeammateDistanceValid(receiver) )
    distance = 0;
  else 
    distance = Mem->GetTeammateDistance(receiver);
  
  FillAttributes(distance,angle,Mem->GetData(receiver));
}

/* Can be used for points, not just markers */
void FillAttributes(float receiverDist, float receiverAng, char *receiverData){
  /* go through all of RangeDesc[a]'s, setting values as in userint.c */
  /* Gonna have to convert from strings */
  char dataStream[MAXMESG];

  GetPassData(dataStream,receiverDist,receiverAng,'a');

  if ( !strcmp(receiverData,BLANK_DAT) ){
    /* This can be used to see what happens without communication */
    int len = strlen(dataStream);
    for (int j=0; j<87; j++){
      dataStream[len + 2*j] = '?';
      dataStream[len + 2*j+1] = ',';
    }
  }
  else 
    /* Now copy in receiver data */
    strcpy(&dataStream[strlen(dataStream)],receiverData);

  int i = 0;
  for (int att=0; att<=MaxAtt; ++att){
    while (dataStream[i] == ' ') i++;
    if (dataStream[i] == '?'){
      RangeDesc[att].Known = FALSE;
    }
    else if ( MaxAttVal[att] )
      my_error(" Can't handle discrete attributes here yet. ");
    else{ /* It's a continuous attribute */
      /* sscanf(&dataStream[i], "%lf", &(RangeDesc[att].LowerBound) ); */
      RangeDesc[att].LowerBound = get_double(&dataStream[i]); 
      /* Not handling ranges here--just single values. See userint.c */
      RangeDesc[att].UpperBound = RangeDesc[att].LowerBound;
      RangeDesc[att].Known = TRUE;
    }
    RangeDesc[att].Asked = TRUE; /* Set it to known */

    /* IF CORE DUMPING HERE, CHECK MESG LENGTH (NEED > 512 for DTs) */
    while (dataStream[i] != ',') i++; /* Advance to next attribute */
    i++; /* Advance past comma */
  }
/*  my_error("FillAttributes");*/
}

/*****************************************************************/

#if DETAILED_PASS_STATS
char  _PredictedClass_;
float _PredictedProb_;
#endif

int FindTreeConfidences(int NumTeammates, int *teammates, float *Confidences){
  int types[NumTeammates];
  for (int i=0; i<NumTeammates; i++)
    types[i] = RL_PASS_ACTION;
  return FindTreeConfidences(NumTeammates,teammates,types,Confidences);
}

int FindTreeConfidences(int NumActions, int *actions, int *action_types, float *Confidences){

  static int FIRST_TIME = TRUE;
  int a;

  if (FIRST_TIME){
    
    /* Initialize the tree */
    FileName = TREE_STEM;   
    printf("%s\n",TREE_STEM);
    GetNames();
    DecisionTree = GetTree(".tree");
  
    RangeDesc = new ValRange[MaxAtt+1];
    ForEach(a, 0, MaxAtt){
	if ( MaxAttVal[a] ){
	    RangeDesc[a].Probability = new double[MaxAttVal[a]+1];
	}
    } 
    FIRST_TIME = FALSE;
  }

  float HighConf = -1;
  int   HighTeammateIndex = -1;
  for (int i=0; i<NumActions; i++){ /* Cycle through open teammates       */
    Clear(); 
    if (action_types[i] == RL_PASS_ACTION)
      FillAttributes(actions[i]);   /* Set the teammate for whom checking */
    else /* A Knock action */
      FillAttributes(MIN(40,Mem->GetMarkerDistance(actions[i])),
		     Mem->GetMarkerAngle(actions[i]),
		     BLANK_DAT);      /* Set the marker for whom checking */
    int TreeClass = InterpretTree(&Confidences[i]);     /* Store confidence */
#if TREE_DEBUG
    printf("%d: %s with conf %.2f\n",actions[i],ClassName[TreeClass],Confidences[i]);
#endif

    if ( !strcmp(ClassName[TreeClass],"F") ){
      Confidences[i] = -Confidences[i];   /* Failures with low conf, better */
    }
    if ( !strcmp(ClassName[TreeClass],"M") ){
      Confidences[i] = 0;                /* Misses better than failures    */
    }

    if ( Confidences[i] > HighConf ||    /* Possibly reset if they're =    */
	 (Confidences[i] == HighConf && int_random(2)) ){  
      HighTeammateIndex = i;         /* Reset best values                  */
      HighConf = Confidences[i];
#if DETAILED_PASS_STATS
      _PredictedClass_ = *ClassName[TreeClass];
      _PredictedProb_  = HighConf;
      if (_PredictedClass_ == 'F')  /* Turn it back to positive           */
	_PredictedProb_ *= -1;
#endif
    }
  }
  return HighTeammateIndex;
}

/*****************************************************************/

int ChooseReceiverTree(){

  int teammates[TEAM_SIZE];

  int NumOpen = Mem->FindTeammatesWithStatus(OPEN_STAT, teammates);

  float confidence[NumOpen];  /* Array to keep confidence that pass will succeed */

  if ( NumOpen == 0 )
    return FALSE;

#if SAVE_PASS_DATA
  if ( NumOpen < 3 ) /* HACK for testing pass3.tree: wait for 3 open receivers */
    return FALSE;
#endif

#if SAVE_PASS_DATA
  /* Only if you want to be sure the data's current */
  for (int i=0; i<NumOpen; i++){  /* Some aren't visible */
    if ( !Mem->PlayerDistanceValid(Mem->MySide,teammates[i]) ||
	 Mem->GetPlayerDistance(Mem->MySide,teammates[i]) < 10 ){
      FACETEAMMATE(teammates[i]);
      return FALSE;
    }
  }
#endif

  int WaitForData = FALSE;
  for (int j=0; j<NumOpen; j++){
    if ( !strcmp(Mem->GetData(teammates[j]),BLANK_DAT) ){
      SAYTO(teammates[j],GIVE_DATA_MSG);    /* Ping the individual for data */
      WaitForData = TRUE;
    }
  }
  if ( WaitForData )      /* waiting for data:  don't bother evaluating yet */
    return FALSE;

#if 0
  if ( !Mem->BallValid() || Mem->GetBallDistance() > KICK_DISTANCE ) /* can't kick yet */
    return Mem->MyNumber; /* But when, there I can pass, so send my number as a holder */
#endif

  int HighTeammateIndex = FindTreeConfidences(NumOpen, teammates, confidence);

  int receiver;
  if ( HighTeammateIndex < 0 )      /* Success predicted for nobody       */
    receiver = ChooseReceiverRandom(); /* Choose randomly    */
  else
    receiver = teammates[HighTeammateIndex];        /* Take best choice   */

#if CLOSEST_COMPARE
  int close_receiver = ChooseReceiverClosest();
/*  if (close_receiver == receiver){
    _PredictedClass_ = 'F';
    _PredictedProb_  = 1.0;
  }
  else */
  receiver = close_receiver;
#endif  

  printf(">>>Passing to %d\n",receiver);
  return receiver;
}




