/* setplay.c
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

#include "global.h"

/******************** SETPLAYPOSITION CLASS FUNCTIONS ******************/

void SetPlayPosition::SetHome(float x, float y, float buf, int type){
  HomeX = x;
  HomeY = y;
  HomeBuffer = buf;
  PositionType = type;
}

void SetPlayPosition::SetAim(float x, float y, int wait){
  AimX = x;
  AimY = y;
  AimPosition = UNKNOWN_POSITION;
  WaitTime = wait;
}

void SetPlayPosition::SetAim(int position, int wait){
  AimX = AimY = NODIR;
  AimPosition = position;
  WaitTime = wait;
}

void SetPlayPosition::SetAim(int position, float x, float y, int wait){
  AimX = x; AimY = y;
  AimPosition = position;
  WaitTime = wait;
}

/******************** SETPLAYFORMATION CLASS FUNCTIONS ******************/

SetPlayFormation::SetPlayFormation(){
  positions = NULL;
  positionPlayers = NULL;
}

SetPlayFormation::~SetPlayFormation(){
  delete positions;
  delete positionPlayers;
}
   
void SetPlayFormation::SpecifySetPlayPosition(int pos, float x, float y, 
					      float buf, int type){
  if (pos >= num_positions) my_error("Too many set play positions");
  positions[pos].SetHome(x,y,buf,type);
}

/* SETPLAY_STARTER can be treated either as a passer or as a knocker */
/* SETPLAY_STARTER should always be set as pos 0 (used in memory.c)  */

void SetPlayFormation::SpecifySetPlayAction  (int pos, float x, float y, int wait){
  if (pos >= num_positions) my_error("Too many set play positions");
  if ( positions[pos].GetType() == SETPLAY_PASSER )
    my_error("Passers should be given a target player");
  positions[pos].SetAim(x,y,wait);
}

void SetPlayFormation::SpecifySetPlayAction  (int pos, int target, int wait){
  if (pos >= num_positions) my_error("Too many set play positions");
  if ( positions[pos].GetType() == SETPLAY_SHOOTER || 
       positions[pos].GetType() == SETPLAY_KNOCKER || 
       positions[pos].GetType() == SETPLAY_BLASTER   )
    my_error("Shooters should be given a target point");
  float x,y;
  GetSetPlayPositionXY(target,&x,&y);
  positions[pos].SetAim(target,x,y,wait);
}

void SetPlayFormation::AssignSetPlayPosition (int pos, int formationPos){
  if (pos >= num_positions) my_error("Too many set play positions");
  positionPlayers[pos] = formationPos;
}

int  SetPlayFormation::SetPlayPositionPlayer(int pos){
  return positionPlayers[pos];
}

int  SetPlayFormation::SetPlayPlayerPosition(int formationPos){
  for (int i=0; i<num_positions; i++){
    if (SetPlayPositionPlayer(i) == formationPos)
      return i;
  }
  return UNKNOWN_POSITION;
}
	
float SetPlayFormation::GetSetPlayPositionX(int pos){
  return positions[pos].GetHomeX();
}

float SetPlayFormation::GetSetPlayPositionY(int pos){
  return positions[pos].GetHomeY();
}

void  SetPlayFormation::GetSetPlayPositionXY(int pos, float *x, float *y){
  *x = GetSetPlayPositionX(pos); *y = GetSetPlayPositionY(pos);
}

float SetPlayFormation::GetSetPlayPlayerX(int formationPos){
  int pos = SetPlayPlayerPosition(formationPos);
  if (pos == UNKNOWN_POSITION)
    my_error("Can't get the x of an unknown (sp) position");
  return GetSetPlayPositionX(pos);
}

float SetPlayFormation::GetSetPlayPlayerY(int formationPos){
  int pos = SetPlayPlayerPosition(formationPos);
  if (pos == UNKNOWN_POSITION)
    my_error("Can't get the y of an unknown (sp) position");
  return GetSetPlayPositionY(pos);
}

void  SetPlayFormation::GetSetPlayPlayerXY(int formationPos, float *x, float *y){
  *x = GetSetPlayPlayerX(formationPos); *y = GetSetPlayPlayerY(formationPos);
}

float SetPlayFormation::GetSetPlayPlayerAimX(int formationPos){
  int pos = SetPlayPlayerPosition(formationPos);
  if (pos == UNKNOWN_POSITION)
    my_error("Can't get the aimx of an unknown (sp) position");
  return positions[pos].GetAimX();
}

float SetPlayFormation::GetSetPlayPlayerAimY(int formationPos){
  int pos = SetPlayPlayerPosition(formationPos);
  if (pos == UNKNOWN_POSITION)
    my_error("Can't get the aimy of an unknown (sp) position");
  return positions[pos].GetAimY();
}

void  SetPlayFormation::GetSetPlayPlayerAimXY(int formationPos, float *x, float *y){
  *x = GetSetPlayPlayerAimX(formationPos); *y = GetSetPlayPlayerAimY(formationPos);
}

float SetPlayFormation::GetPositionBuffer(int formationPos){
  int pos = SetPlayPlayerPosition(formationPos);
  if (pos == UNKNOWN_POSITION)
    my_error("Can't get the buffer of an unknown (sp) position");
  return positions[pos].GetHomeBuffer();
}

int   SetPlayFormation::GetWaitTime      (int formationPos){
  int pos = SetPlayPlayerPosition(formationPos);
  if (pos == UNKNOWN_POSITION)
    my_error("Can't get the wait time of an unknown (sp) position");
  return positions[pos].GetWaitTime();
}

int   SetPlayFormation::GetPositionType  (int formationPos){
  int pos = SetPlayPlayerPosition(formationPos);
  if (pos == UNKNOWN_POSITION){
    char msg[100];
    sprintf(msg,"Can't get the type of an unknown (sp) position (formationPos = %d)",
	    formationPos);
    my_error(msg);
  }
  return positions[pos].GetType();
}

int   SetPlayFormation::GetStarter   (){
  int starter = UNKNOWN_POSITION;
  for (int pos=0; pos< num_positions; pos++){
    if ( positions[pos].GetType() == SETPLAY_STARTER )
      starter = SetPlayPositionPlayer(pos);
  }
  return starter;  /* Could be UNKNOWN_POSITION */
}

int   SetPlayFormation::GetAimPosition   (int formationPos){
  int setplayPos = positions[SetPlayPlayerPosition(formationPos)].GetAimPosition();
  if ( setplayPos == UNKNOWN_POSITION )
    return UNKNOWN_POSITION;
  else 
    return SetPlayPositionPlayer(setplayPos);
}

int  SetPlayFormation::Initialize(int PlayMode, float ballx, float bally){

  if ( !Mem->BallInSetPlayPosition() ){
    SAY(PING_BALL_MSG);
    return FALSE;
  }

  switch(PlayMode){
  case MY_KICK_IN:
    if ( ballx < X0-45 )
      num_positions = 3;
    else
      num_positions = 5; /* going to corner kick mode */
    break;
  case MY_CORNER_KICK:
    num_positions = 5;
    break;
  case MY_KICK_OFF:
    num_positions = 5;
    break;
  case MY_GOAL_KICK:
    num_positions = 2;
    break;
  case THEIR_KICK_IN:
    num_positions = 1;
    break;
  case THEIR_CORNER_KICK:
    num_positions = 5;
    break;
  case THEIR_GOAL_KICK:
    num_positions = 3;
    break;
  case THEIR_KICK_OFF:
    num_positions = 3;
    break;
  default: return FALSE;  /* No set play for that mode */
  }

  /* Reinitialize the structure */
  if ( positions != NULL ) delete positions;
  if ( positionPlayers != NULL ) delete positionPlayers;
  positions = new SetPlayPosition[num_positions];
  positionPlayers = new int[num_positions];
  for (int i=0; i<num_positions; i++)
    positionPlayers[i] = UNKNOWN_PLAYER;

  int ballside = Mem->BallLocationSide();
  int buffer = 1;
  float tempx,tempy,dist,angle;

  switch(PlayMode){
  case MY_KICK_OFF:
    /* changing to relative, 
       Could be used for freekicks (if not within 15 of sideline) */
    /* Need to use absolute here -- might not see ball            */
    SpecifySetPlayPosition(0,  0,    0, buffer, SETPLAY_STARTER);
#if 1
    SpecifySetPlayPosition(1, 12,   15, buffer, SETPLAY_BLASTER);
    SpecifySetPlayPosition(2, 12,  -15, buffer, SETPLAY_BLASTER);
    SpecifySetPlayPosition(3, 16,   14.5, buffer, SETPLAY_BLASTER);
    SpecifySetPlayPosition(4, 16,  -14.5, buffer, SETPLAY_BLASTER);

    SpecifySetPlayAction  (0,12, int_random(2) ? 15 : -15, 10);
#else
    SpecifySetPlayPosition(1, 5,   15, buffer, SETPLAY_BLASTER);
    SpecifySetPlayPosition(2, 5,  -15, buffer, SETPLAY_BLASTER);
    SpecifySetPlayPosition(3, 7,   14.5, buffer, SETPLAY_BLASTER);
    SpecifySetPlayPosition(4, 7,  -14.5, buffer, SETPLAY_BLASTER);

    SpecifySetPlayAction  (0,5, int_random(2) ? 15 : -15, 10);
#endif

    SpecifySetPlayAction  (1,X0,-10,20);   /* aim to far side of goal */
    SpecifySetPlayAction  (2,X0, 10,20);
    SpecifySetPlayAction  (3,X0,-10,20);
    SpecifySetPlayAction  (4,X0, 10,20);
    break;
  case MY_KICK_IN:
#if 0
    if ( ballx < X0-40 ){
      SpecifySetPlayPosition(0, ballx,bally, buffer, SETPLAY_STARTER);
      SpecifySetPlayPosition(1, ballx,  (Y0-6)*ballside, buffer, SETPLAY_BLASTER);
      SpecifySetPlayPosition(2, ballx+2,(Y0-7)*ballside, buffer, SETPLAY_BLASTER);

      SpecifySetPlayAction  (0, 1, 10); 
      SpecifySetPlayAction  (1,X0, 10*ballside, 15);  /* Shoot near side of the goal */
      SpecifySetPlayAction  (2,X0, 10*ballside, 15);
#endif
    if ( ballx < X0-45 ){
      SpecifySetPlayPosition(0, ballx ,   bally, buffer, SETPLAY_STARTER);
      SpecifySetPlayPosition(1, ballx+20, (Y0-3)*ballside , buffer, SETPLAY_PASSER);
      SpecifySetPlayPosition(2, ballx+15, (Y0-15)*ballside, buffer, SETPLAY_KNOCKER);
      
      SpecifySetPlayAction  (0, 1, 10); 
      SpecifySetPlayAction  (1, 2, 20);  
      SpecifySetPlayAction  (2,X0, 0, 40);  /* Shoot on goal */
      break;
    }
    else{
      ; /* same as corner kick */
    }
  case MY_CORNER_KICK:
#if 1
    SpecifySetPlayPosition(0, ballx,   bally, buffer, SETPLAY_STARTER);
    SpecifySetPlayPosition(1, X0-20,    (Y0-3)   *ballside, buffer, SETPLAY_PASSER);
    SpecifySetPlayPosition(2, PA_X+2,   (PA_Y-12)*ballside, buffer, SETPLAY_SHOOTER);
    SpecifySetPlayPosition(3, PA_X,     (PA_Y-10)*ballside, buffer, SETPLAY_SHOOTER);
    SpecifySetPlayPosition(4, PA_X+10, -(PA_Y-10)*ballside, buffer, SETPLAY_SHOOTER);

    SpecifySetPlayAction  (0,1,20);
    SpecifySetPlayAction  (1,2,30);
    SpecifySetPlayAction  (2,X0,0,40);
    SpecifySetPlayAction  (3,X0,0,40);
    SpecifySetPlayAction  (4,X0,0,50);
#else
    SpecifySetPlayPosition(0, ballx,     bally,            buffer, SETPLAY_STARTER);
    SpecifySetPlayPosition(1, X0-20,    (Y0-3)   *ballside, buffer, SETPLAY_PASSER);
    SpecifySetPlayPosition(2, PA_X+2,   (PA_Y-12)*ballside, buffer, SETPLAY_BLASTER);
    SpecifySetPlayPosition(3, PA_X,     (PA_Y-10)*ballside, buffer, SETPLAY_BLASTER);
    SpecifySetPlayPosition(4, PA_X+10, -(PA_Y-10)*ballside, buffer, SETPLAY_SHOOTER);

    SpecifySetPlayAction  (0,PA_X+1,(PA_Y-11)*ballside,20);
    SpecifySetPlayAction  (1,2,30);
    SpecifySetPlayAction  (2,X0,0,40);
    SpecifySetPlayAction  (3,X0,0,40);
    SpecifySetPlayAction  (4,X0,0,50);
#endif
    break;
  case MY_GOAL_KICK:
    SpecifySetPlayPosition(0,  ballx, bally, buffer, SETPLAY_STARTER);
    SpecifySetPlayPosition(1, -PA_X, (Y0-5)*ballside, buffer, SETPLAY_KNOCKER);
    
    SpecifySetPlayAction  (0, 1 ,10);
    SpecifySetPlayAction  (1, X0, 0, 10); /* make own decision */
    break;
  case THEIR_KICK_IN:
    dist  = FREE_KICK_BUFFER+1;
    angle = my_atan2(Y0,ballx+X0);
    tempx = ballx - dist * cos(angle);
    tempy = Y0    - dist * sin(angle);
    /* tempx = ballx-xdist;
       tempy = Y0 - Y0*(xdist/(ballx+X0)); */
    SpecifySetPlayPosition(0, tempx, tempy*ballside, buffer, SETPLAY_KNOCKER);

    SpecifySetPlayAction  (0, X0, 0 , 0);  /* Just get to position */
    break;
  case THEIR_KICK_OFF:
    /* Need to use absolute here -- might not see ball            */
    SpecifySetPlayPosition(0, -10, 0, buffer, SETPLAY_KNOCKER);
    SpecifySetPlayPosition(1, -8,  5, buffer, SETPLAY_KNOCKER);
    SpecifySetPlayPosition(2, -8,  -5, buffer, SETPLAY_KNOCKER);

    SpecifySetPlayAction  (0, X0, 0 , 0);  /* Just get to position */
    SpecifySetPlayAction  (1, X0, 0 , 0);  
    SpecifySetPlayAction  (2, X0, 0 , 0);  
    break;
  case THEIR_GOAL_KICK:
    SpecifySetPlayPosition(0, GA_X-10, (GA_Y  )*ballside, buffer, SETPLAY_KNOCKER);
    SpecifySetPlayPosition(1, GA_X-8,  (GA_Y+5)*ballside, buffer, SETPLAY_KNOCKER);
    SpecifySetPlayPosition(2, GA_X-8,  (GA_Y-5)*ballside, buffer, SETPLAY_KNOCKER);

    SpecifySetPlayAction  (0, X0, 0 , 0);  /* Just get to position */
    SpecifySetPlayAction  (1, X0, 0 , 0);  
    SpecifySetPlayAction  (2, X0, 0 , 0);  
    break;
  case THEIR_CORNER_KICK:
    SpecifySetPlayPosition(0, -(X0-2), (Y0-10)*ballside, buffer, SETPLAY_KNOCKER);
    SpecifySetPlayPosition(1, -(X0-5), (Y0-10)*ballside, buffer, SETPLAY_KNOCKER);
    SpecifySetPlayPosition(2, -PA_X,  PA_Y*ballside, buffer, SETPLAY_KNOCKER);
    SpecifySetPlayPosition(3, -GA_X, -GA_Y*ballside, buffer, SETPLAY_KNOCKER);
    SpecifySetPlayPosition(4, -PA_X,  0,             buffer, SETPLAY_KNOCKER);
    
    SpecifySetPlayAction  (0, X0, 0 , 0);  /* Just get to position */
    SpecifySetPlayAction  (1, X0, 0 , 0);  
    SpecifySetPlayAction  (2, 0, Y0*ballside, 20);  
    SpecifySetPlayAction  (3, 0,-Y0*ballside, 20);  
    SpecifySetPlayAction  (4, 0, Y0*ballside, 20);  
    break;
  default: return FALSE;  /* No set play for that mode */
  }

  return TRUE;
}


void SetPlayFormation::Print(){
  printf("spf(%d): ",num_positions);
  for (int i=0; i<num_positions; i++)
    printf("%d->%d ",i,positionPlayers[i]);
  printf("\n");
}
