/* position.h
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* -*- Mode: C -*- */

#ifndef _POSITION_H
#define _POSITION_H

#include <math.h>
#include "peterclient.h"
#include "utils.h"

#define TEAM_SIZE 11

#define NUM_EDGE_MARKERS 8
#define NUM_INTERNAL_MARKERS 6
#define NUM_MARKERS (NUM_EDGE_MARKERS+NUM_INTERNAL_MARKERS)

#define GOAL_R   0
#define GOAL_L  (GOAL_R +1)
#define FLAG_RB (GOAL_L +1)
#define FLAG_B  (FLAG_RB+1)
#define FLAG_LB (FLAG_B +1)
#define FLAG_LT (FLAG_LB+1)
#define FLAG_T  (FLAG_LT+1)
#define FLAG_RT (FLAG_T +1)

#define FLAG_PRT (FLAG_RT +1)
#define FLAG_PRC (FLAG_PRT+1)
#define FLAG_PRB (FLAG_PRC+1)
#define FLAG_PLT (FLAG_PRB+1)
#define FLAG_PLC (FLAG_PLT+1)
#define FLAG_PLB (FLAG_PLC+1)

#define MY_GOAL     NUM_MARKERS
#define THEIR_GOAL (MY_GOAL   +1)
#define LB_FLAG    (THEIR_GOAL+1)
#define LC_FLAG    (LB_FLAG   +1)
#define LF_FLAG    (LC_FLAG   +1)
#define RB_FLAG    (LF_FLAG   +1)
#define RC_FLAG    (RB_FLAG   +1)
#define RF_FLAG    (RC_FLAG   +1)

#define MY_PL_FLAG    (RF_FLAG      +1)
#define MY_PC_FLAG    (MY_PL_FLAG   +1)
#define MY_PR_FLAG    (MY_PC_FLAG   +1)
#define THEIR_PL_FLAG (MY_PR_FLAG   +1)  // Left as I face THEIR goal
#define THEIR_PC_FLAG (THEIR_PL_FLAG+1)
#define THEIR_PR_FLAG (THEIR_PC_FLAG+1)

#define NO_LINE -1
#define LINE_R  GOAL_R
#define LINE_L  GOAL_L
#define LINE_B  FLAG_B
#define LINE_T  FLAG_T

#define MY_LINE    MY_GOAL
#define THEIR_LINE THEIR_GOAL
#define L_LINE     LC_FLAG
#define R_LINE     RC_FLAG

#define NUM_OBJECTS (NUM_MARKERS + (TEAM_SIZE+1)*2 + 1)

/* Vision widths and qualities */
#define SAME_WIDTH     0
#define NARROW_WIDTH  (SAME_WIDTH+1)
#define NORMAL_WIDTH  (NARROW_WIDTH+1)
#define WIDE_WIDTH    (NORMAL_WIDTH+1)

#define SAME_QUALITY  0
#define LOW_QUALITY  (SAME_QUALITY+1)
#define HIGH_QUALITY (LOW_QUALITY+1)

#define X0    52.5       // Dimensions of the field:  105 x 68 
#define Y0    34
#define PA_X (X0-16.5)   // Penalty area is 40.32 x 16.5
#define PA_Y (20.16)
#define GA_X (X0-5.5)    // goal area is 5.5 x 18.32
#define GA_Y (9.16)

#define FREE_KICK_BUFFER 9.15

#define PLAYER_SIZE 0.8
#define BALL_SIZE 0.085
#define KICK_DISTANCE (1 + PLAYER_SIZE + BALL_SIZE)
#define FEEL_DISTANCE 3
#define GOAL_WIDTH 14
#define DEFAULT_VIEW_ANGLE 90
#define STAMINA_MAX 2000
#define STAMINA_INC 20
#define GAME_LENGTH 6000

#define TRUE 1
#define FALSE 0

// Parameters
#define MINIMUM_OVERALL_CONFIDENCE 0.5
#define VALID_THRESHOLD            0.4
#define HEARSAY_CONFIDENCE         0.99

#define DECAY_S_ESTIMATE          0.9
#if SAVE_PASS_DATA
#define DECAY_M_ESTIMATE          0.99
#define DECAY_M_TICK               0.99
#else
#define DECAY_M_ESTIMATE          0.92    /* was .9, .96 */
#define DECAY_M_TICK               0.97  /* per time step */
#endif

// Geometry Functions
inline float rad_to_deg(float x)
{ return x * 180 / M_PI; }

inline float deg_to_rad(float x)
{ return x * M_PI / 180; }

// The coordinate system here gets a little tricky.  We work in two
//  coordinate systems, a polar and a cartesian.  Unfortunately
//  the angle in the polar coordinate system is with respect to
//  the y-axis in the cartesian coordinate system.  So, there
//  are two sets of conversion functions.  Ones between polar
//  and xy, the normal conversion, and the ones between polar
//  and relative xy, which reverses x and y.
inline void xy_to_polar(float x, float y, float *r, float *t)
{ *r = sqrt((x*x) + (y*y)); *t = (x==0 && y==0) ? 0 : atan2(y, x); }
inline void relxy_to_polar(float x, float y, float *r, float *t)
{ *r = sqrt((x*x) + (y*y)); *t = (x==0 && y==0) ? 0 : atan2(x, y); }

inline float polar_to_y(float r, float t)
{ return r * sin(t); }
inline float polar_to_x(float r, float t)
{ return r * cos(t); }
inline void polar_to_xy(float r, float t, float *x, float *y)
{ *x = polar_to_x(r,t); *y = polar_to_y(r,t); }

inline float polar_to_rely(float r, float t)
{ return r * cos(t); }
inline float polar_to_relx(float r, float t)
{ return r * sin(t); }
inline void polar_to_relxy(float r, float t, float *x, float *y)
{ *x = polar_to_relx(r,t); *y = polar_to_rely(r,t); }

inline float normalize(float a)
{ return a - ((2*M_PI) * (int)(a / M_PI)); }

inline int InPitch(float x, float y) { return ( fabs(x) < X0+10 && fabs(y) < Y0+10 );}


class Position {
public:
  Position();

  // Methods to get the position
  inline float get_t() { return cur_t; }
  inline float get_tDEG() { return rad_to_deg(cur_t); }
  inline float get_r() { return cur_r; }
  inline float get_x() { return polar_to_relx(cur_r, cur_t); }
  inline float get_y() { return polar_to_rely(cur_r, cur_t); }
  inline float get_rconf() { return ( cur_rconf > VALID_THRESHOLD ) ? 
			       cur_rconf : 0; }
  inline float get_tconf() { return ( cur_tconf > VALID_THRESHOLD ) ?
			       cur_tconf : 0; }
  inline float get_conf() { return (get_rconf() < get_tconf()) ?
			      get_rconf() : get_tconf(); }

  inline float get_new_r() { return new_r; }
  inline float get_new_t() { return new_t; }
  inline float get_new_tDEG() { return rad_to_deg(new_t); }
  inline float get_new_x() { return polar_to_relx(new_r, new_t); }
  inline float get_new_y() { return polar_to_rely(new_r, new_t); }
  inline float get_new_rconf() { return new_rconf; }
  inline float get_new_tconf() { return new_tconf; }
  inline float get_new_conf()  { return (new_rconf < new_tconf) ?
				   new_rconf : new_tconf; }

  // Methods to change the position
  // Changes do not take place until tick() is called.
  virtual float set_polar(float r, float t, float c);
  virtual float set_xy(float x, float y, float c);
  virtual float set_r(float r, float c);
  virtual float set_t(float t, float c);
  virtual float set_t_gospel(float t, float c);

  virtual float estimate_degrade(float c);
  virtual float translate(float dr, float dt, float c);

  // Method which processes all changes
  virtual void tick();
  virtual void reset();
  
  // Was the new_* position set via a set_* call.  If so, we accept
  //  it as gospel and rotate/translate calls cannot override it.
  int gospel;
  
protected:
  float cur_r, cur_t;  // relative polar coordinates
  float cur_rconf, cur_tconf;

  float new_r, new_t;
  float new_rconf, new_tconf;
};

class PositionStationary : public Position {
public:
  virtual float estimate_degrade();
  virtual float translate(float dr, float dt, float c);

  virtual void tick();

  void get_translate(float *dr, float *dt, float *c);
};

class PositionMobile : public Position {
public:
  PositionMobile();
  ~PositionMobile();

  virtual float estimate_degrade();
  virtual float translate(float dr, float dt, float c);
  virtual void tick(int time_steps);
  virtual void reset();

  float get_rel_vel_r(int dt); /* computed from relative ball positions */
  float get_rel_vel_t();
  float get_rel_vel_tDEG();

  float get_traj_mag();        /* computed from global ball positions   */
  float get_traj_dir();
  float get_traj_conf(int time, int dt);

  inline float get_old_r() { return old_r; }
  inline float get_old_t() { return old_t; }
  inline float get_old_x() { return polar_to_relx(old_r, old_t); }
  inline float get_old_y() { return polar_to_rely(old_r, old_t); }
  inline float get_old_conf() { return (old_rconf < old_tconf) ?
			      old_rconf : old_tconf; }

  inline float get_vel_dir() { return cur_vel_dir; }
  virtual void set_vel_dir(float );

  inline float get_vel_mag() { return cur_vel_mag; }
  virtual void set_vel_mag(float );

  virtual void set_vel(float, float);

  void clear_globalxy();
  void update_globalxy(float x, float y, int t); /* Not from external source */
  void set_globalxy(float x, float y, int t);
  inline void get_globalxy(float *x, float *y)
    {*x = global_x; *y = global_y;}

  inline int get_global_set_time() {return global_set_time;}
  int velocity_valid();

  int got_global_info;
  char side;
  int num;
  
private:
  float global_x, global_y;
  float old_global_x, old_global_y;
  PointQueue *points;

  int   global_set_time, old_global_set_time;  /* Times at which global values
						  were set */
  float old_r, old_t;            // Used to calculate relative velocity
  float old_rconf, old_tconf;

  float cur_vel_dir;
  float new_vel_dir;
  float cur_vel_mag;
  float new_vel_mag;
};

/*--------------------------------------------------------------*/

class PositionInfo {

public:
  PositionInfo();  
  Initialize(const char team, const int number, char *team_name);  
  ~PositionInfo();

  float GetMarkerDistance(int marker);
  float GetMarkerAngle(int marker);
  float GetMarkerGlobalX(int marker);
  float GetMarkerGlobalY(int marker);
  inline void  GetMarkerGlobalXY(int marker, float *x, float *y)
    {*x = GetMarkerGlobalX(marker), *y = GetMarkerGlobalY(marker);}
  float MarkerValid(int marker);

  int   ClosestGoal();
  int   ClosestFlag();
  int   ClosestUnseenPlayer(char *side, float *global_x, float *global_y);

  int   MyVelValid();
  float GetMyVelMagnitude();
  float GetMyVelDirection();

  float GetPlayerDistance(char side, int number);
  float GetPlayerAngle(char side, int number);
  float GetPlayerAngleRad(char side, int number);
  float PlayerDistanceValid(char side, int number);
  float PlayerAngleValid(char side, int number);
  float PlayerValid(char side, int number);
  float GetPlayerVelMagnitude(char side, int num); /* from the server */
  float GetPlayerVelDirection(char side, int num); /* In relative terms */
  int   PlayerVelValid(char side, int num);

  inline float GetTeammateDistance(int number)  
    {return GetPlayerDistance(MySide,number);}
  inline float GetTeammateAngle(int number)
    {return GetPlayerAngle(MySide,number);}
  inline float GetTeammateAngleRad(int number)
    {return GetPlayerAngleRad(MySide,number);}
  inline float TeammateDistanceValid(int number)
    {return PlayerDistanceValid(MySide,number);}
  inline float TeammateAngleValid(int number)
    {return PlayerAngleValid(MySide,number);}
  inline float TeammateValid(int number)
    {return PlayerValid(MySide,number);}
  inline float GetTeammateVelMagnitude(int num)
    {return GetPlayerVelMagnitude(MySide,num);}
  inline float GetTeammateVelDirection(int num)
    {return GetPlayerVelDirection(MySide,num);}
  inline int   TeammateVelValid(int num)
    {return PlayerVelValid(MySide,num);}

  inline float GetOpponentDistance(int number)  
    {return GetPlayerDistance(TheirSide,number);}
  inline float GetOpponentAngle(int number)
    {return GetPlayerAngle(TheirSide,number);}
  inline float GetOpponentAngleRad(int number)
    {return GetPlayerAngleRad(TheirSide,number);}
  inline float OpponentDistanceValid(int number)
    {return PlayerDistanceValid(TheirSide,number);}
  inline float OpponentAngleValid(int number)
    {return PlayerAngleValid(TheirSide,number);}
  inline float OpponentValid(int number)
    {return PlayerValid(TheirSide,number);}
  inline float GetOpponentVelMagnitude(int num)
    {return GetPlayerVelMagnitude(TheirSide,num);}
  inline float GetOpponentVelDirection(int num)
    {return GetPlayerVelDirection(TheirSide,num);}
  inline int   OpponentVelValid(int num)
    {return PlayerVelValid(TheirSide,num);}
  
  float GetBallDistance();
  float GetBallAngle();
  float GetBallAngleRad();
  float GetBallRelVelDirection();  /* My calculation using relatives */
  float GetBallRelVelMagnitude();
  float GetBallTrajectoryAbsoluteDirection();  /* My calculation using globals */
  float GetBallTrajectoryRelativeDirection();
  float GetBallTrajectoryMagnitude();
  float GetBallVelMagnitude();     /* from the server   */
  float GetBallVelDirection();     /* In relative terms */

  float BallDistanceValid();
  float BallAngleValid();
  float BallValid();
  float BallRelVelocityValid();
  float BallTrajectoryValid();
  int   BallVelValid();
  
  void SeeLine(int line, float dist, float ang, int time);
  void SeeLine(int line, float ang, int time);

  void SeeMarker(int marker, float dist, float ang, int time);
  void SeeMarker(int marker,  float ang, int time);
  void SeeMarker(int marker, 
		 float dist, float ang, float distChng, float dirChng, int time);

  void SeeBall(float ang, int time);
  void SeeBall(float dist, float ang, int time);
  void SeeBall(float dist, float ang, float distChng, float dirChng, int time);

  void SeePlayer(char side, int num, float ang, int time);
  void SeePlayer(char side, int num, float dist, float ang, int time);
  void SeePlayer(char side, int num, 
		 float dist, float ang, float distChng, float dirChng, int time);
  void SeePlayer(float dist, float ang, float distChng, float dirChng, int time);
  void SeePlayer(char side, float dist, float ang, int time);
  void SeePlayer(char side, float ang, int time);
  void SeePlayer(float dist, float ang, int time);
  void SeePlayer(float ang, int time);

  void SeeExtraPlayer(char side, float dist, float ang, int time);
  void ClearExtraPlayer();

  virtual void HearPlayer(char side, int num, float ang, char *msg, int time);

  inline void HearPlayerGlobalPosition(char side, int num, float x, float y, int time)
    { ((side == MySide) ? MyTeam[num] : TheirTeam[num])->set_globalxy(x,y,time); }
  inline void HearTeammateGlobalPosition(int num, float x, float y, int time)
    { HearPlayerGlobalPosition(MySide,num,x,y,time); }
  inline void HearOpponentGlobalPosition(int num, float x, float y, int time)
    { HearPlayerGlobalPosition(TheirSide,num,x,y,time); }
  inline void HearBallGlobalPosition(float x, float y, int time)
    { Ball->set_globalxy(x,y,time); }

  void UpdatePlayerGlobalPosition(char side, int num);
  void UpdateBallGlobalPosition();
  void ClearBallGlobalPosition();

  void Tick(int marker,int time);
  void UpdateMobileObjectPositions(int erase_data);
  void DisambiguateExtraPlayers();
  void Reset();
  void ClearAction();

  void  Dashing(float power);
  void  Turning(float angle);
  float LastTurnAngle;
  void  Kicking(float power,float angle);
  int   Dashed;  /* Did I dash in the last sensor interval? */

  void Acting(int code);

  void Print();

  int view_width;
  int view_quality;
  int GetViewAngle();
  int GetViewAngle(int width);
  int TimePerSight();

  inline  float	GetGlobalAngle(){return global_angle;}
  inline  float	GetGlobalX()    {return global_x;}
  inline  float	GetGlobalY()    {return global_y;}
  inline  void GetGlobalXY(float *x, float *y) { *x = global_x; *y = global_y; }

  float AngleToPoint(float x, float y); // In degrees
  float DistanceToPoint(float x, float y);

  inline float polar_to_globalx(float r, float t)
    { return global_x + r * cos(global_angle - t); }
  inline float polar_to_globaly(float r, float t)
    { return global_y + r * sin(global_angle - t); }

  /* Uses the global position of the client to calculate global
     position of viewed objects.  Could be stored, but risky */
  /* Both global_angle and t should be in RADIANS            */
  inline void polar_to_globalxy(float r, float t, float *x, float *y)
    {*x = global_x + r*cos(global_angle - t); *y = global_y + r*sin(global_angle - t);}

  float globalxy_to_polar_r(float x, float y);
  float globalxy_to_polar_t(float x, float y);
  inline void globalxy_to_polar(float x, float y, float *r, float *t)
    {*r = globalxy_to_polar_r(x,y); *t = globalxy_to_polar_t(x,y);}

  inline void GetMobileGlobalXY(PositionMobile *MobileObject, float *x, float *y)
    { polar_to_globalxy( MobileObject->get_r(), MobileObject->get_t(), x, y ); }
  inline float GetMobileGlobalX(PositionMobile *MobileObject)
    { return polar_to_globalx( MobileObject->get_r(), MobileObject->get_t() ); }
  inline float GetMobileGlobalY(PositionMobile *MobileObject)
    { return polar_to_globaly( MobileObject->get_r(), MobileObject->get_t() ); }
  inline void GetMobileLastGlobalXY(PositionMobile *MobileObject, float *x, float *y)
    { MobileObject->get_globalxy(x,y); }
  inline int GetMobileGlobalSetTime(PositionMobile *MobileObject)
    { return MobileObject->get_global_set_time(); }

  inline void GetPlayerGlobalXY(char side, int num, float *x, float *y)
    { GetMobileGlobalXY( (side == MySide ? MyTeam[num] : TheirTeam[num]), x, y ); }
  inline float GetPlayerGlobalX(char side, int num)
    { return GetMobileGlobalX( (side == MySide ? MyTeam[num] : TheirTeam[num]) ); }
  inline float GetPlayerGlobalY(char side, int num)
    { return GetMobileGlobalY( (side == MySide ? MyTeam[num] : TheirTeam[num]) ); }
  inline void GetPlayerLastGlobalXY(char side, int num, float *x, float *y)
    { GetMobileLastGlobalXY( (side == MySide ? MyTeam[num] : TheirTeam[num]), x, y ); }
  inline int GetPlayerGlobalSetTime(char side, int num)
    { return GetMobileGlobalSetTime( (side == MySide ? MyTeam[num] : TheirTeam[num]) ); }

  inline void GetTeammateGlobalXY(int num, float *x, float *y)
    { GetPlayerGlobalXY( MySide, num, x, y ); }
  inline float GetTeammateGlobalX(int num)
    { return GetPlayerGlobalX( MySide, num); }
  inline float GetTeammateGlobalY(int num)
    { return GetPlayerGlobalY( MySide, num ); }
  inline void GetTeammateLastGlobalXY(int num, float *x, float *y)
    { GetPlayerLastGlobalXY( MySide, num, x, y ); }
  inline int GetTeammateGlobalSetTime(int num)
    { return GetPlayerGlobalSetTime( MySide, num ); }

  inline void GetOpponentGlobalXY(int num, float *x, float *y)
    { GetPlayerGlobalXY( TheirSide, num, x, y ); }
  inline float GetOpponentGlobalX(int num)
    { return GetPlayerGlobalX( TheirSide, num); }
  inline float GetOpponentGlobalY(int num)
    { return GetPlayerGlobalY( TheirSide, num ); }
  inline void GetOpponentLastGlobalXY(int num, float *x, float *y)
    { GetPlayerLastGlobalXY( TheirSide, num, x, y ); }
  inline int GetOpponentGlobalSetTime(int num)
    { return GetPlayerGlobalSetTime( TheirSide, num ); }

  inline void GetBallGlobalXY(float *x, float *y)
    { GetMobileGlobalXY( Ball, x, y ); }
  inline float GetBallGlobalX()
    { return GetMobileGlobalX( Ball ); }
  inline float GetBallGlobalY()
    { return GetMobileGlobalY( Ball ); }

  inline void GetTeammateHeardGlobalXY(int num, float *x, float *y)
    { MyTeam[num]->get_globalxy(x,y); }

  char             MySide;            // 'l' or 'r'                          
  char             TheirSide;            // 'l' or 'r'                          
  char             *MyTeamName;      // The name of my team                 
  char             *TheirTeamName;      // The name of my team                 
  int              MyTeamNameLen;    // strlen of MyTeamName
  int              MyNumber;          // uniform number                      
  int              CurrentTime;       // Most recently observed time         
  int              LastTime;         // previous time				   

  Socket           *sock;            // default socket					
	  
private:
  int ConvertMarker(int marker, float *x, float *y); /* get the x,y coords of marker */
  inline int ConvertMarker(int marker)  /* ignore the actual position of the marker  */
    { float x; return ConvertMarker(marker, &x, &x); } 
  float overall_conf();
  float triangulate(int m);

  // This is an array of all the objects, access to these objects
  //  should be done through the more specific pointers below.
  //  They point to the same objects.
  Position *objects[NUM_OBJECTS];

  PositionStationary *Markers[NUM_MARKERS];
  PositionMobile *Ball;
  PositionMobile *MyTeam[TEAM_SIZE + 1];
  PositionMobile *TheirTeam[TEAM_SIZE + 1];

  PositionMobile *ExtraPlayers[2*TEAM_SIZE]; /* The unidentified ones */
  int NumExtraPlayers;

  int line; // This should be in the side-independent format
  float line_r, line_t;

  float global_angle, old_global_angle; // Zero is side-independent right
  float global_x, global_y;             // (0,0) is center, (-X0,0) is my goal
					// (-X0,-Y0) is my right defensive flag
  float my_vel_x;  /* relative to my heading*/
  float my_vel_y;
  float my_vel_dir;
  float my_vel_mag;

  int turning;
  int dashing;
  /* float turning_angle;*/
};

#endif
