/* position.c
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

#include "global.h"

#define DEBUG                   0
#define PLAYER_ID_DEBUG         0

struct {
  float x, y;
} marker_global[] = { 
  {X0, 0},      /* GOAL_R   */
  {-X0, 0},     /* GOAL_L   */
  {X0, -Y0},    /* FLAG_RB  */
  {0, -Y0},     /* FLAG_B   */
  {-X0, -Y0},   /* FLAG_LB  */
  {-X0, Y0},    /* FLAG_LT  */
  {0, Y0},      /* FLAG_T   */
  {X0, Y0},     /* FLAG_RT  */
  {PA_X,PA_Y},  /* FLAG_PRT */
  {PA_X,0},     /* FLAG_PRC */
  {PA_X,-PA_Y}, /* FLAG_PRB */
  {-PA_X,PA_Y}, /* FLAG_PLT */
  {-PA_X,0},    /* FLAG_PLC */
  {-PA_X,-PA_Y} /* FLAG_PLB */
};

struct {
  int a, b;
} line_endpoints[] = {
  {2, 7},
  {5, 4},
  {-1, -1}, // Not really a line.
  {4, 2}, 
  {-1, -1}, 
  {-1, -1},
  {7, 5}
};

//
// Position
//

Position::Position()
{
  new_rconf = new_tconf 
    = cur_rconf = cur_tconf 
    = 0.0;
  gospel = 0;
}

float Position::estimate_degrade(float c){
  if (!gospel){
    new_rconf *= c;
    new_tconf *= c;
  }
}

float Position::translate(float dr, float dt, float c)
{
  float x, y;

  if (gospel) return get_new_conf();

  // Do rotation first.
  new_t += dt;

  // Now, movement along y-axis.

  /* Distance not valid, but need some magnitude for direction */
  if (!new_r) new_r = 100; 

  x = get_new_x(); y = get_new_y() + dr;
  relxy_to_polar(x, y, &new_r, &new_t);

  new_rconf *= c;
  new_tconf *= c;
}

float Position::set_polar(float r, float t, float c)
{
  new_r = r;
  new_t = normalize(t);
  new_rconf = c;
  new_tconf = c;
  gospel = 1;
  return get_new_conf();
}

float Position::set_r(float r, float c)
{
  new_r = r;
  new_rconf = c;
  return get_new_conf();
}

float Position::set_t(float t, float c)
{
  new_t = normalize(t);
  new_tconf = c;
  return get_new_conf();
}

float Position::set_t_gospel(float t, float c)
{
  gospel = 1;  /* Assume that you do set_t when you know the angle */
  return set_t(t, c);
}

float Position::set_xy(float x, float y, float c)
{
  relxy_to_polar(x, y, &new_r, &new_t);
  new_rconf = c;
  new_tconf = c;
  gospel = 1;
  return get_new_conf();
}

void Position::tick()
{
  cur_r = new_r;
  cur_t = new_t;
  cur_rconf = new_rconf;
  cur_tconf = new_tconf;

  gospel = 0;
}

void Position::reset()
{
  set_polar(0,0,0);
}

//
// PositionStationary
//

float PositionStationary::estimate_degrade()
{
  Position::estimate_degrade(DECAY_S_ESTIMATE);
}

void PositionStationary::get_translate(float *dr, float *dt, float *c)
{
  *dr = new_r - cur_r;
  *dt = new_t - cur_t;
  *c = get_new_conf();
}

float PositionStationary::translate(float dr, float dt, float c)
{
  Position::translate(dr, dt, c);
}

void PositionStationary::tick()
{
  Position::tick();
}

//
// PositionMobile
//

PositionMobile::PositionMobile()
{
  got_global_info = FALSE;
  old_global_set_time = global_set_time = 0;
  points = new PointQueue(5);
  global_x = global_y = NODIR;
  side = 'b'; /* if not otherwise set, ball */
  num = 0;
  Position::Position();
}

PositionMobile::~PositionMobile()
{
  delete points;
}

float PositionMobile::estimate_degrade()
{
  Position::estimate_degrade(DECAY_M_ESTIMATE);
}

float PositionMobile::translate(float dr, float dt, float c)
{
  Position::translate(dr, dt, c);
}

void PositionMobile::tick(int time_steps)
{
  old_r = cur_r;
  old_t = cur_t;
  old_rconf = cur_rconf;
  old_tconf = cur_tconf;

  if (!gospel) {
    for (int i=0; i<time_steps; i++){
      new_rconf *= DECAY_M_TICK;
      new_tconf *= DECAY_M_TICK;
    }
  }

  cur_vel_dir = new_vel_dir;
  new_vel_dir = NODIR;
  cur_vel_mag = new_vel_mag;
  new_vel_mag = 0;

  Position::tick();
}

void PositionMobile::reset()
{
  Position::reset();
  old_r = old_t = old_rconf = old_tconf = 0;
  old_global_set_time = global_set_time = 0;
}
  
float PositionMobile::get_rel_vel_t()
{
  float dx = get_x() - get_old_x();
  float dy = get_y() - get_old_y();

  return my_atan2(dx, dy);
}

float PositionMobile::get_rel_vel_tDEG()
{
  return rad_to_deg(get_rel_vel_t());
}

float PositionMobile::get_rel_vel_r(int dt)
{
  float dx = get_x() - get_old_x(); 
  float dy = get_y() - get_old_y();
  
  return sqrt(dx*dx + dy*dy)/dt;
}
 
#define TRAJ_VALID_TIME 15
float PositionMobile::get_traj_mag(){
  int dt = global_set_time - old_global_set_time;
  if ( dt > TRAJ_VALID_TIME )
    my_error("Some global info is out of date (traj mag)");

  return GetDistance(&global_x,&global_y,&old_global_x,&old_global_y)/dt;
}

float PositionMobile::get_traj_dir(){

  int dt;
  float dx,dy,dir;

#if 0
  dt = global_set_time - old_global_set_time;
  if ( dt > 15 )
    my_error("Some global info is out of date (traj dir) ");
#endif
  dx = global_x - old_global_x;
  dy = global_y - old_global_y;

  float short_dir = my_atan2(dy, dx);

  return short_dir;
/*
  printf(" **** now(%d) (%.1f,%.1f)   then(%d) (%.1f,%.1f)   dx/dy %.1f %.1f\n",
	 global_set_time,global_x,global_y,
	 old_global_set_time,old_global_x,old_global_y,dx,dy);
*/

  int index = points->OldestIndexWithin(15);
  if ( index==END_OF_QUEUE )
    my_error("Some global info is out of date (traj dir) ");

  if ( global_x != points->GetTailX() )
    my_error("Did I update queue correctly?? ");
  
  dx = global_x - points->GetIndexX(index);
  dy = global_y - points->GetIndexY(index);
  dir = my_atan2(dy, dx);

#if 1
  float diff = rad_to_deg(short_dir - dir);
  CleanAngle(&diff);
  if ( fabs(diff) > 10 )
    printf("%d:%d  going back to %d -- %.1f.  One step -- %.1f\n",
	   Mem->MyNumber,Mem->CurrentTime,index,rad_to_deg(dir),rad_to_deg(short_dir));
#endif

  return dir;
  /*return short_dir;*/
}

float PositionMobile::get_traj_conf(int time, int dt){
  /* Both current and old globals are recent */
  if ( time != global_set_time || global_set_time - old_global_set_time > dt )
    return FALSE;
  else
    return get_conf();
/*
  if ( time - global_set_time > 10 || time - old_global_set_time > 15 )
    return FALSE;

  if ( time == global_set_time )
    return get_conf();
  else 
    return pow(DECAY_M_TICK,time - old_global_set_time);
*/      
}
 
void PositionMobile::set_vel_dir(float d)
{
  new_vel_dir = d;
}

void PositionMobile::set_vel_mag(float m)
{
  new_vel_mag = m;
}

void PositionMobile::set_vel(float d, float m)
{
  set_vel_dir(d);
  set_vel_mag(m);
}

void PositionMobile::clear_globalxy(){
  global_set_time = old_global_set_time = 0;
}

void PositionMobile::set_globalxy(float x, float y, int time){
  if ( InPitch(x,y) ){  
    got_global_info = TRUE; 
    update_globalxy(x,y,time);
  }
}

void PositionMobile::update_globalxy(float x, float y, int time)
{
  old_global_x        = global_x;
  old_global_y        = global_y;
  old_global_set_time = global_set_time;

  global_x        = x; 
  global_y        = y;
  global_set_time = time;

  points->Add(x,y,time);
  /*points->Print();*/
}

//
// PositionInfo
//

PositionInfo::PositionInfo()
{
  int obj=0;
  int i;

  for(i=0; i<NUM_MARKERS; i++) {
    Markers[i] = new PositionStationary();
    objects[obj++] = Markers[i];
  }

  for(i=0; i<TEAM_SIZE+1; i++) {
    MyTeam[i] = new PositionMobile();
    objects[obj++] = MyTeam[i];
    TheirTeam[i] = new PositionMobile();
    objects[obj++] = TheirTeam[i];
  }

  for(i=0; i<2*TEAM_SIZE; i++) {
    ExtraPlayers[i] = new PositionMobile();
  }
  NumExtraPlayers = 0;

  Ball = new PositionMobile();
  objects[obj++] = Ball;

  Dashed = FALSE;

  view_width   = NORMAL_WIDTH;   
  view_quality = HIGH_QUALITY;

  CurrentTime = LastTime = 0;

  my_vel_x = my_vel_y = NODIR;
}

PositionInfo::Initialize(const char s, const int n, char *tn){
  MySide = s;
  TheirSide = (s == 'l') ? 'r' : 'l';
  MyTeamName = tn;
  MyTeamNameLen = strlen(MyTeamName);
  MyNumber = n;

  for(int i=0; i<TEAM_SIZE+1; i++) {
    MyTeam[i]->side = MySide;
    MyTeam[i]->num = i;
    TheirTeam[i]->side = TheirSide;
    TheirTeam[i]->num = i;
  }
}

PositionInfo::~PositionInfo()
{
  int i;

  for(i=0; i<NUM_MARKERS; i++)
    delete Markers[i];

  for (i=0; i<TEAM_SIZE+1; i++) {
    delete MyTeam[i];
    delete TheirTeam[i];
  }

  delete Ball;
}

int PositionInfo::ConvertMarker(int marker, float *x, float *y)
{
  if (marker < NUM_MARKERS)
    return marker;

  int ActualMarker;

  if (MySide == 'l') {
    switch(marker) {
    case MY_GOAL:       ActualMarker = GOAL_L;   break;
    case THEIR_GOAL:    ActualMarker = GOAL_R;   break;
    case LB_FLAG:       ActualMarker = FLAG_LT;  break;
    case LC_FLAG:       ActualMarker = FLAG_T;   break;
    case LF_FLAG:       ActualMarker = FLAG_RT;  break;
    case RB_FLAG:       ActualMarker = FLAG_LB;  break;
    case RC_FLAG:       ActualMarker = FLAG_B;   break;
    case RF_FLAG:       ActualMarker = FLAG_RB;  break;
    case MY_PL_FLAG:    ActualMarker = FLAG_PLT; break;
    case MY_PC_FLAG:    ActualMarker = FLAG_PLC; break;
    case MY_PR_FLAG:    ActualMarker = FLAG_PLB; break;
    case THEIR_PL_FLAG: ActualMarker = FLAG_PRT; break;
    case THEIR_PC_FLAG: ActualMarker = FLAG_PRC; break;
    case THEIR_PR_FLAG: ActualMarker = FLAG_PRB; break;
    }
  } else {
    switch(marker) {
    case MY_GOAL:       ActualMarker = GOAL_R;   break;
    case THEIR_GOAL:    ActualMarker = GOAL_L;   break;
    case LB_FLAG:       ActualMarker = FLAG_RB;  break;
    case LC_FLAG:       ActualMarker = FLAG_B;   break;
    case LF_FLAG:       ActualMarker = FLAG_LB;  break;
    case RB_FLAG:       ActualMarker = FLAG_RT;  break;
    case RC_FLAG:       ActualMarker = FLAG_T;   break;
    case RF_FLAG:       ActualMarker = FLAG_LT;  break;
    case MY_PL_FLAG:    ActualMarker = FLAG_PRB; break;
    case MY_PC_FLAG:    ActualMarker = FLAG_PRC; break;
    case MY_PR_FLAG:    ActualMarker = FLAG_PRT; break;
    case THEIR_PL_FLAG: ActualMarker = FLAG_PLB; break;
    case THEIR_PC_FLAG: ActualMarker = FLAG_PLC; break;
    case THEIR_PR_FLAG: ActualMarker = FLAG_PLT; break;
    }
  }
  *x = marker_global[ActualMarker].x;
  *y = marker_global[ActualMarker].y;

  return ActualMarker;
}

float PositionInfo::GetMarkerDistance(int marker)
{
  int m = ConvertMarker(marker);
  return Markers[m]->get_r();
}

float PositionInfo::GetMarkerAngle(int marker)
{
  int m = ConvertMarker(marker);
  return Markers[m]->get_tDEG();
}

float PositionInfo::GetMarkerGlobalX(int marker)
{
  int m = ConvertMarker(marker);
  return marker_global[m].x;
}

float PositionInfo::GetMarkerGlobalY(int marker)
{
  int m = ConvertMarker(marker);
  return marker_global[m].y;
}

float PositionInfo::MarkerValid (int marker)
{
  int m = ConvertMarker(marker);
  return Markers[m]->get_conf();
}

int PositionInfo::ClosestGoal()
{
  return (global_x < 0) ? MY_GOAL : THEIR_GOAL;
}

int PositionInfo::ClosestFlag()
{
  /* Chop the field into a 5 x 5 grid.  
     If there's no flag in the current section, it's an error */

  if ( global_x < -PA_X - (X0-PA_X)/2 ){
    if ( global_y < -PA_Y - (Y0-PA_Y)/2 )     return RB_FLAG;
    else if ( global_y > PA_Y + (Y0-PA_Y)/2 ) return LB_FLAG;
    else if ( Mem->PlayMode != BEFORE_KICK_OFF) printf("%d:%d Flag Confusion 1\n",MyNumber,CurrentTime);
  }
  else if ( global_x < -PA_X/2 ){
    if ( global_y < -PA_Y - (Y0-PA_Y)/2 ){ if ( Mem->PlayMode != BEFORE_KICK_OFF) printf("%d:%d Flag Confusion 2\n",MyNumber,CurrentTime);}
    else if ( global_y < -PA_Y/2 )            return MY_PR_FLAG; 
    else if ( global_y < PA_Y/2 )             return MY_PC_FLAG;
    else if ( global_y < PA_Y + (Y0-PA_Y)/2 ) return MY_PL_FLAG;
    else if ( Mem->PlayMode != BEFORE_KICK_OFF) printf("%d:%d Flag Confusion 3\n",MyNumber,CurrentTime);
  }
  else if ( global_x < PA_X/2 ){
    if ( global_y < -PA_Y - (Y0-PA_Y)/2 )     return RC_FLAG;
    else if ( global_y > PA_Y + (Y0-PA_Y)/2 ) return LC_FLAG;
    else if ( Mem->PlayMode != BEFORE_KICK_OFF) printf("%d:%d Flag Confusion 4\n",MyNumber,CurrentTime);
  }
  else if ( global_x < PA_X + (X0-PA_X)/2 ){
    if ( global_y < -PA_Y - (Y0-PA_Y)/2 ){  if ( Mem->PlayMode != BEFORE_KICK_OFF) printf("%d:%d Flag Confusion 5\n",MyNumber,CurrentTime);}
    else if ( global_y < -PA_Y/2 )            return THEIR_PR_FLAG; 
    else if ( global_y < PA_Y/2 )             return THEIR_PC_FLAG;
    else if ( global_y < PA_Y + (Y0-PA_Y)/2 ) return THEIR_PL_FLAG;
    else if ( Mem->PlayMode != BEFORE_KICK_OFF) printf("%d:%d Flag Confusion 6\n",MyNumber,CurrentTime);
  }
  else{
    if ( global_y < -PA_Y - (Y0-PA_Y)/2 )     return RF_FLAG;
    else if ( global_y > PA_Y + (Y0-PA_Y)/2 ) return LF_FLAG;
    else if ( Mem->PlayMode != BEFORE_KICK_OFF) printf("%d:%d Flag Confusion 7\n",MyNumber,CurrentTime);
  }
  return FALSE;
}

int PositionInfo::ClosestUnseenPlayer(char *side, float *global_x, float *global_y){
  float minDistance = 1000;
  float Distance;
  PositionMobile *closestPlayer;
  float player_x,player_y;
  
  if ( *side != TheirSide ){ /* MySide or unknown */
    for (int i=1; i<TEAM_SIZE+1; i++){
      if ( !MyTeam[i]->gospel && MyNumber != i && 
	   CurrentTime - MyTeam[i]->get_global_set_time() < 100 ){    /* set recently */
	MyTeam[i]->get_globalxy(&player_x,&player_y); /* Stored, not computed values */
	Distance = GetDistance(&player_x,&player_y,global_x,global_y);
	if ( Distance < minDistance ){
	  minDistance = Distance;
	  closestPlayer = MyTeam[i];
	}
      }
    }
  }
  if ( *side != MySide ){ /* MySide or unknown */
    for (int i=1; i<TEAM_SIZE+1; i++){
      if ( !TheirTeam[i]->gospel ){
	GetPlayerGlobalXY(TheirSide,i,&player_x,&player_y);
	Distance = GetDistance(&player_x,&player_y,global_x,global_y);
	if ( Distance < minDistance ){
	  minDistance = Distance;
	  closestPlayer = TheirTeam[i];
	}
      }
    }
  }
  
  if ( minDistance > 10 )
    return 0;         /* No player is close enough to guess */

#if 0
  printf("Min distance from (%.1f, %.1f) = %.1f \n",*global_x,*global_y,minDistance);
  printf("Chose player %d\n",closestPlayer->num);
#endif

  *side = closestPlayer->side;
  return closestPlayer->num;
}

int   PositionInfo::MyVelValid(){
  if ( my_vel_dir == NODIR )
    return FALSE;
  else 
    return TRUE;
}

float PositionInfo::GetMyVelMagnitude(){ 
  return my_vel_mag;
}

float PositionInfo::GetMyVelDirection(){
  return my_vel_dir;
}

float PositionInfo::GetPlayerDistance(char side, int number)
{
  if (side == MySide)
    return MyTeam[number]->get_r();
  else
    return TheirTeam[number]->get_r();
}

float PositionInfo::GetPlayerAngle(char side, int number)
{
  if (side == MySide)
    return MyTeam[number]->get_tDEG();
  else
    return TheirTeam[number]->get_tDEG();
}

float PositionInfo::GetPlayerAngleRad(char side, int number)
{
  if (side == MySide)
    return MyTeam[number]->get_t();
  else
    return TheirTeam[number]->get_t();
}

float PositionInfo::GetPlayerVelDirection(char side, int number)
{
  float dir;  /* was in global coordinates, want relative to me */

  if (side == MySide)
    dir = MyTeam[number]->get_vel_dir();
  else if (side == TheirSide)
    dir = TheirTeam[number]->get_vel_dir();
  else 
    my_error("Which side?");

  if ( dir == NODIR )
    return NODIR;    
  dir = dir - rad_to_deg(global_angle);

  CleanAngle(&dir);
  return dir;
}

float PositionInfo::GetPlayerVelMagnitude(char side, int number)
{
  if (side == MySide)
    return MyTeam[number]->get_vel_mag();
  if (side == TheirSide)
    return TheirTeam[number]->get_vel_mag();
}

float PositionInfo::PlayerDistanceValid(char side, int number)
{
  if (side == MySide)
    return MyTeam[number]->get_rconf();
  else
    return TheirTeam[number]->get_rconf();
}
  
float PositionInfo::PlayerAngleValid(char side, int number)
{
  if (side == MySide)
    return MyTeam[number]->get_tconf();
  else
    return TheirTeam[number]->get_tconf();
}

float PositionInfo::PlayerValid(char side, int number)
{
  float conf = MIN(PlayerDistanceValid(side, number),PlayerAngleValid(side, number)); 
  
  if ( conf > 0 ){
    float x,y;
    GetPlayerGlobalXY(side,number,&x,&y);
    if ( !InPitch(x,y) ) conf = 0;
  }

  return conf;
}

int   PositionInfo::PlayerVelValid(char side, int number)
{
  if ( side == MySide )
    return ( (MyTeam[number]->get_vel_dir() == NODIR) ? FALSE : TRUE); 
  else 
    return ( (TheirTeam[number]->get_vel_dir() == NODIR) ? FALSE : TRUE); 
}

float PositionInfo::GetBallDistance() { return Ball->get_r(); }

float PositionInfo::GetBallAngle() { return Ball->get_tDEG(); }

float PositionInfo::GetBallAngleRad() { return Ball->get_t(); }

float PositionInfo::GetBallVelDirection() { 
  float dir =  Ball->get_vel_dir();

  if ( dir == NODIR )
    return NODIR;    
  dir = dir - rad_to_deg(global_angle);

  CleanAngle(&dir);   /* was in global coordinates, want relative to me */
  return dir;
}

float PositionInfo::GetBallVelMagnitude() { return Ball->get_vel_mag(); }

float PositionInfo::GetBallRelVelDirection() { return Ball->get_rel_vel_tDEG(); }

float PositionInfo::GetBallRelVelMagnitude() { return Ball->get_rel_vel_r(CurrentTime-LastTime);}

float PositionInfo::GetBallTrajectoryAbsoluteDirection() { return Ball->get_traj_dir(); }

float PositionInfo::GetBallTrajectoryRelativeDirection() { 
  float dir =  GetBallTrajectoryAbsoluteDirection();
  dir -= global_angle;  /* was in global coordinates, want relative to me */
  CleanAngleRad(&dir);   

  dir = rad_to_deg(dir);
  return dir;
}

float PositionInfo::GetBallTrajectoryMagnitude() { return Ball->get_traj_mag(); }

float PositionInfo::BallTrajectoryValid() { 
  return Ball->get_traj_conf(CurrentTime,MIN(TRAJ_VALID_TIME,TimePerSight()*3)); 
}

float PositionInfo::BallDistanceValid() { return (float) Ball->get_rconf(); }

float PositionInfo::BallAngleValid() { return (float) Ball->get_tconf(); }

float PositionInfo::BallValid()
{
  return MIN(BallDistanceValid(),BallAngleValid()); 
}

float PositionInfo::BallRelVelocityValid() { return Ball->get_old_conf(); }

int   PositionInfo::BallVelValid() { 
  return ( (Ball->get_vel_dir() == NODIR) ? FALSE : TRUE); 
}

/********************************************************************************/
/********************************************************************************/
/********************************************************************************/

void PositionInfo::SeeLine(int l, float dist, float ang, int tm)
{
  line = l;
  line_r = dist; line_t = fabs(deg_to_rad(ang));

  switch (line) {
  case LINE_T : global_angle = (ang < 0) ? 
    normalize(M_PI - line_t) : normalize(line_t); break;
  case LINE_B : global_angle = (ang < 0) ? 
    normalize(-line_t) : normalize(M_PI + line_t); break;
  case LINE_R : global_angle = (ang < 0) ? 
    normalize(M_PI_2 - line_t) : normalize(-M_PI_2 + line_t); break;
  case LINE_L : global_angle = (ang < 0) ? 
    normalize(-M_PI_2 - line_t) : normalize(M_PI_2 + line_t); break;
  }
}

void PositionInfo::SeeLine(int l, float ang, int tm)
{
  /*******  Can't use -1 if later on dists to lines are < 0 when off field ******/
  float dist = NO_LINE; /* Indicates that we no longer know how far the line is */
                        /* Be careful how it's used now (triangulation)         */
  /*******  Can't use -1 if later on dists to lines are < 0 when off field ******/

  SeeLine(l,dist,ang,tm);
}

/********************************************************************************/

void PositionInfo::SeeMarker(int marker, float dist, float ang, int tm)
{
  int m = ConvertMarker(marker);
  Markers[m]->set_polar(dist, deg_to_rad(ang), 1.0);
}

void PositionInfo::SeeMarker(int marker, float ang, int tm)
{
  my_error("Shouldn't process markers when using low quality -- no info");
  int m = ConvertMarker(marker);
  /* For purpose of triangulation, it's not really visble */
  Markers[m]->set_t_gospel(deg_to_rad(ang), 1.0);
}

void PositionInfo::SeeMarker(int marker, float dist, float ang, 
			     float distChng, float dirChng, int tm){

  if (dirChng == NODIR)
    my_error("Shouldn't be here without a valid dirChng");

  float rad_ang = deg_to_rad(ang);
  my_vel_x = -(distChng*cos(rad_ang) - deg_to_rad(dirChng)*dist*sin(rad_ang));
  my_vel_y = -(distChng*sin(rad_ang) + deg_to_rad(dirChng)*dist*cos(rad_ang));

  SeeMarker(marker,dist,ang,tm);
}

/********************************************************************************/

void PositionInfo::SeeBall(float ang, int tm)
{
  Ball->set_t_gospel(deg_to_rad(ang), 1.0);
}

void PositionInfo::SeeBall(float dist, float ang, int tm)
{
  Ball->set_polar(dist, deg_to_rad(ang), 1.0);
}

void PositionInfo::SeeBall(float dist, float ang, float distChng, float dirChng, int tm)
{
  if ( (my_vel_x!=NODIR && my_vel_y!=NODIR) ||
       !dashing ){ /* If I didn't just move, assume no motion */

    if (!dashing)
      my_vel_x = my_vel_y = 0;

    float vel_mag, vel_dir;

    float rad_ang = deg_to_rad(ang);
    float Wx = distChng*cos(rad_ang) - deg_to_rad(dirChng)*dist*sin(rad_ang);
    float Wy = distChng*sin(rad_ang) + deg_to_rad(dirChng)*dist*cos(rad_ang);

    float Ux = Wx + my_vel_x;
    float Uy = Wy + my_vel_y;

    vel_mag = sqrt(Ux*Ux + Uy*Uy);
    vel_dir = rad_to_deg(my_atan2(Uy,Ux));

    /* Don't know if this should be in here or not */
/*
    if (Wx < 0) { 
      vel_dir += 180;
      CleanAngle(&vel_dir);
    }
*/  
    Ball->set_vel(vel_dir,vel_mag);  /* This is now in global coordinates               */
                                     /* changes to relative angle in GetBallVelDirection*/
#if 0
    printf("*****Global vel (dir,mag):  (%.1f, %.1f)\n",vel_dir,vel_mag);
#endif
  }

  /* Use for NN training? */

  SeeBall(dist,ang,tm);
}

/********************************************************************************/

void PositionInfo::SeePlayer(char side, int num, float ang, int time)
{
  if (side == MySide)
    MyTeam[num]->set_t_gospel(deg_to_rad(ang), 1.0);
  else
    TheirTeam[num]->set_t_gospel(deg_to_rad(ang), 1.0);
}

void PositionInfo::SeePlayer(char side, int num, float dist, float ang, int time)
{
  if (side == MySide)
    MyTeam[num]->set_polar(dist, deg_to_rad(ang), 1.0);
  else
    TheirTeam[num]->set_polar(dist, deg_to_rad(ang), 1.0);
}

void PositionInfo::SeePlayer(char side, int num, 
			     float dist, float ang, float distChng, float dirChng, int time)
{
  PositionMobile *Player;

  if (side == MySide)
    Player = MyTeam[num];
  if (side == TheirSide)
    Player = TheirTeam[num];

  if ( (my_vel_x!=NODIR && my_vel_y!=NODIR) ||
       !dashing ){ /* If I didn't just move, assume no motion */

    if (!dashing)
      my_vel_x = my_vel_y = 0;

    float vel_mag, vel_dir;

    float rad_ang = deg_to_rad(ang);
    float Wx = distChng*cos(rad_ang) - deg_to_rad(dirChng)*dist*sin(rad_ang);
    float Wy = distChng*sin(rad_ang) + deg_to_rad(dirChng)*dist*cos(rad_ang);

    float Ux = Wx + my_vel_x;
    float Uy = Wy + my_vel_y;

    vel_mag = sqrt(Ux*Ux + Uy*Uy);
    vel_dir = rad_to_deg(my_atan2(Uy,Ux));
    if (Wx < 0) { 
      vel_dir += 180;
      CleanAngle(&vel_dir);
    }
  
    Player->set_vel(vel_dir,vel_mag);/* This is now in global coordinates               */
                                     /* changes to relative angle in GetBallVelDirection*/
  }

  SeePlayer(side,num,dist,ang,time);
}

void SeePlayer(float dist, float ang, float distChng, float dirChng, int time){
  my_error("shouldn't be getting here");
}

void PositionInfo::SeePlayer(char side, float dist, float ang, int time)
{
  int num=0;

  for (int i=1; i<TEAM_SIZE; i++){

    if ( PlayerAngleValid(side,i) /*> PlayerDistanceValid(side,i)*/
	 && fabs(GetPlayerAngle(side,i) - ang) < 1 ){
      if ( num ){ /* Already assigned */
	if ( fabs(GetPlayerDistance(side,i) - dist) < 2 ||
	     ( GetPlayerDistance(side,num) != dist &&
	       PlayerAngleValid(side,i) > PlayerAngleValid(side,num)) )
	  num = i;
	/*my_error("Two players at same angle!!!");*/
      }
      else       /* not assigned yet  */
	num = i;
    }
  }

  if ( num ){
#if PLAYER_ID_DEBUG
    printf("SeePlayer1 %d:  Player %d thinks Player %d is at %.1f, %.1f\n",
	   CurrentTime,MyNumber,num,dist,ang);
#endif
    SeePlayer(side, num, dist, ang, time);
  }
  else
    SeeExtraPlayer(side, dist, ang, time);
}

void PositionInfo::SeePlayer(char side, float ang, int time)
{
  int num=0;
  
  for (int i=1; i<TEAM_SIZE; i++){

    if ( PlayerAngleValid(side,i) /*> PlayerDistanceValid(side,i)*/
	 && fabs(GetPlayerAngle(side,i) - ang) < 1 ){
      if ( num ){ /* Already assigned */
	if ( PlayerAngleValid(side,i) > PlayerAngleValid(side,num) )
	  num = i;
	/*my_error("Two players at same angle!!!"); */
      }
      else       /* not assigned yet  */
	num = i;
    }
  }

  if ( num ){
#if PLAYER_ID_DEBUG
    printf("SeePlayer2 %d:  Player %d thinks Player %d is at ang %.1f\n",
	   CurrentTime,MyNumber,num,ang);
#endif
    SeePlayer(side, num, ang, time);
  }
}

void PositionInfo::SeePlayer(float dist, float ang, int time)
{
  int num=0;
  char side = '0';

  /* Close players that are "felt" come here.
    count number of close players
    if 1, you know who it is (right?)
    else disambiguate by angles?
    Or turn quickly to look?
  */

  for (int i=1; i<TEAM_SIZE; i++){
    if ( PlayerAngleValid(MySide,i) /*> PlayerDistanceValid(MySide,i) */
	 && fabs(GetPlayerAngle(MySide,i) - ang) < 1 ){
      if ( num ){ /* Already assigned */
	if ( fabs(GetPlayerDistance(MySide,i) - dist) < 2 ||
	     ( GetPlayerDistance(MySide,num) != dist &&
	       PlayerAngleValid(MySide,i) > PlayerAngleValid(side,num)) ){
	  side = MySide; 
	  num = i;
	}
	/* my_error("Two players at same angle!!!"); */
      }
      else{       /* not assigned yet  */
	side = MySide; 
	num = i;
      }
    }
    if ( PlayerAngleValid(TheirSide,i) /*> PlayerDistanceValid(TheirSide,i) */
	 && fabs(GetPlayerAngle(TheirSide,i) - ang) < 1 ){
      if ( num ){ /* Already assigned */
	if ( fabs(GetPlayerDistance(TheirSide,i) - dist) < 2 ||
	     ( GetPlayerDistance(TheirSide,num) != dist &&
	       PlayerAngleValid(TheirSide,i) > PlayerAngleValid(side,num)) ){
	  side = TheirSide; 
	  num = i;
	}
	/*my_error("Two players at same angle!!!");*/
      }
      else{       /* not assigned yet  */
	side = TheirSide; 
	num = i;
      }
    }
  }

  if (num){
#if PLAYER_ID_DEBUG 
    printf("SeePlayer3 %d:  Player %d thinks Player %d is at %.1f, %.1f\n",
	   CurrentTime,MyNumber,num,dist,ang);
#endif
    SeePlayer(side, num, dist, ang , time);
  }
  else
    SeeExtraPlayer('?', dist, ang, time);
}

void PositionInfo::SeePlayer(float ang, int time)
{

  if ( fabs(ang) > GetViewAngle()/2 ){
    SeePlayer((float)FEEL_DISTANCE,ang,time);   /* dist must be less than 3 */
    return;
  }

  int num=0;
  char side = '0';

  for (int i=1; i<TEAM_SIZE; i++){
    if ( PlayerAngleValid(MySide,i) /*> PlayerDistanceValid(MySide,i) */
	 && fabs(GetPlayerAngle(MySide,i) - ang) < 1 ){
      if ( num ){ /* Already assigned */
	if ( PlayerAngleValid(MySide,i) > PlayerAngleValid(side,num) ){
	  side = MySide; 
	  num = i;
	}
	/*my_error("Two players at same angle!!!"); */
      }
      else{       /* not assigned yet  */
	side = MySide; 
	num = i;
      }
    }
    if ( PlayerAngleValid(TheirSide,i) /*> PlayerDistanceValid(TheirSide,i) */
	 && fabs(GetPlayerAngle(TheirSide,i) - ang) < 1 ){
      if ( num ){ /* Already assigned */
	if ( PlayerAngleValid(TheirSide,i) > PlayerAngleValid(side,num) ){
	  side = TheirSide; 
	  num = i;
	}
	/*my_error("Two players at same angle!!!"); */
      }
      else{       /* not assigned yet  */
	side = TheirSide; 
	num = i;
      }
    }
  }

  if (num){
#if PLAYER_ID_DEBUG 
    printf("SeePlayer4 %d:  Player %d thinks Player %d is at ang %.1f\n",
	   CurrentTime,MyNumber,num,ang);
#endif
    SeePlayer(side, num, ang ,time);
  }
}

void PositionInfo::SeeExtraPlayer(char side, float dist, float ang, int time){

  ExtraPlayers[NumExtraPlayers]->side = side;
  ExtraPlayers[NumExtraPlayers]->set_polar(dist, deg_to_rad(ang), 1);

  NumExtraPlayers++;
}

/********************************************************************************/

void PositionInfo::HearPlayer(char side, int num, 
			      float ang, char *msg, int time)
{
  /* Assume you're not hearing your own message (filtered in parse.c) */
  /* Don't do anything for opponent messages.                         */
  if ( side == MySide ){
    MyTeam[num]->set_t_gospel(deg_to_rad(ang), 1.0);
  }
}

/********************************************************************************/
/********************************************************************************/
/********************************************************************************/

void PositionInfo::ClearBallGlobalPosition()
{
  Ball->clear_globalxy();
}

void PositionInfo::UpdateBallGlobalPosition()
{
  if ( BallValid()==1 && (CurrentTime > Ball->get_global_set_time()) ){  
    /* don't do it if the ball's not valid or if the info's not new */
    float x,y;
    GetBallGlobalXY(&x, &y);
    Ball->update_globalxy(x,y,CurrentTime); 
  }
}

void PositionInfo::UpdatePlayerGlobalPosition(char side, int num)
{
  PositionMobile *Player = ((side == MySide) ? MyTeam[num] : TheirTeam[num]);
  if ( PlayerValid(side,num) && (CurrentTime > Player->get_global_set_time()) ){  
    /* don't do it if the player's not valid  or if the info's not new */
    float x,y;
    GetPlayerGlobalXY(side, num, &x, &y);
    Player->update_globalxy(x,y,CurrentTime); 
  }
}

void PositionInfo::Tick(int marker, int time)
{
  /* rely on marker == -1 if none were visible */
  int i;

  LastTime    = CurrentTime;
  CurrentTime = time;

  if (dashing)  /* I moved since the last time step */
    Dashed = TRUE;
  else          /* I didn't                         */
    Dashed = FALSE;

  /* Calculate change in direction. */
  if ( line==NO_LINE ){ /* If I didn't see a line, assume turn worked */
    global_angle -= deg_to_rad(LastTurnAngle);
    CleanAngleRad(&global_angle);
  }
  float t_change = global_angle - old_global_angle;
  float r_change;

  /* If we see a marker, triangulate to find others.  Otherwise
     estimate movement from visible line. */
  if (marker>=0) {
    r_change = triangulate(marker);
#if DEBUG
    printf("Triangulating...\n");
#endif
  } else /* if (dashing || turning) */ {
    /* We are only here if there are no visible markers. */
#if DEBUG
    printf("Estimating movement...\n");
#endif

    /* Calculate change in distance (r_change) using the endpoints
       of the visible line. */
    {
      float alpha,aep;

      if ( line != NO_LINE ){
	float ag = global_angle;
	/* aep is angle to end point of the line */
	aep = Markers[line_endpoints[line].a]->get_t() + t_change;
	CleanAngleRad(&aep); /* So it's not over PI */
	
	/* Do rotations, so calculations can be done as if the line
        was right.  Pretend line we're facing is right line. */
	switch(line) {
	case LINE_T: ag = normalize(ag - M_PI_2); break;
	case LINE_L: ag = normalize(ag - M_PI); break;
	case LINE_B: ag = normalize(ag + M_PI_2); break;
	case LINE_R: break;
	}
	
	/* Do calculation of old distance, by the law of sines. */
	if (aep > 0) alpha = M_PI_2 + ag - aep;
	else alpha = 3*M_PI_2 - ag + aep;
#if DEBUG
	printf("ag = %.1f, aep = %.1f, alpha = %.1f, oldr = %.1f\n",
	       rad_to_deg(ag),rad_to_deg(aep),rad_to_deg(alpha),
	       Markers[line_endpoints[line].a]->get_r());
#endif
      }

      if (line_r == NO_LINE || line == NO_LINE)    
	                 /* Low quality vision, so don't know line distance */
	r_change = 0;    /* Therefore, can't tell translational change      */
      else               /* High quality vision                             */
	r_change = (Markers[line_endpoints[line].a]->get_r() *
		    sin(alpha) / sin(alpha + aep)) - line_r;
#if DEBUG
      printf("Estimated movement, dr = %f, dt = %f\n", r_change, t_change);
#endif
    }

    /* Update markers based on estimation of movement. */
    for(i=0; i<NUM_EDGE_MARKERS; i++)
      Markers[i]->translate(-r_change, t_change, 1);
    /* Peter's addition */  /* This is if you didn't triangulate */
    if ( MySide == 'r' ){  /* x & y were flipped before, and will be flipped again */
      global_x *= -1;      /* here they need to be in team-independent format      */
      global_y *= -1;
    }
    global_x += r_change * cos(global_angle);
    global_y += r_change * sin(global_angle);
    /* Peter's addition */

    for (i=0; i<NUM_OBJECTS; i++){
      if ( i<NUM_MARKERS )
	((PositionStationary *)objects[i])->estimate_degrade();
      else
	((PositionMobile *)objects[i])->estimate_degrade();
    }
  }

  /* New: 12/16/96 */
  /* Update the positions of the mobile objects based on my move too */
  for(i=1; i<=TEAM_SIZE; i++){
    if (!MyTeam[i]->gospel && i != MyNumber){ /* If I saw it, don't update it */
      if ( TeammateValid(i) )
	MyTeam[i]->translate(-r_change, t_change, 1);
    }
    if (!TheirTeam[i]->gospel){ /* If I saw it, don't update it */
      if ( OpponentValid(i) )
	TheirTeam[i]->translate(-r_change, t_change, 1);
    }
  }
  if (!Ball->gospel){ /* If I saw it, don't update it */
    Ball->translate(-r_change, t_change, 1);
  }
  /* New: 12/16/96 */
	
#define ANG_DIV_FACT 3 

  if ( view_quality == HIGH_QUALITY ) 
    for (i=0; i<NUM_EDGE_MARKERS; i++)
      if (objects[i]->get_conf() && /* if it wasn't in view (know angle from talk)
				       then let it be                             */
	  fabs(objects[i]->get_new_tDEG()) < GetViewAngle()/ANG_DIV_FACT && 
	  !objects[i]->gospel){     /* actual width must be different from view_width */
	/*printf("%d:%d Adjusting view width from %d due to object %d\n",
	  MyNumber,CurrentTime,view_width,i); */
      if ( view_width == NARROW_WIDTH )  
	; /* Couldn't see because I'm confused about my angle */
      else if ( fabs(objects[i]->get_new_tDEG()) < GetViewAngle(NORMAL_WIDTH)/ANG_DIV_FACT )
	view_width = NARROW_WIDTH;
      else
	view_width = NORMAL_WIDTH; /* was wide_width */
      }

#if 1
  if ( view_quality == HIGH_QUALITY ) /* high */
    /* Make sure that objects which are supposed to be in view are in view. */
      for(i=NUM_MARKERS; i<NUM_OBJECTS; i++)
	i = NUM_OBJECTS-1;  /* only do this for the ball for now */
	if (objects[i]->get_conf() && /* if it wasn't in view (know angle from talk)
					 then let it be                             */
	    ( fabs(objects[i]->get_new_tDEG()) < GetViewAngle()/ANG_DIV_FACT || 
	      objects[i]->get_new_r() < FEEL_DISTANCE ) &&
	    !objects[i]->gospel){
	  objects[i]->set_r(0,0);  /* don't make it gospel */
	  objects[i]->set_t(0,0); 
	  //printf("%d:%d forgetting ball\n",Mem->MyNumber,Mem->CurrentTime);
	}
#endif

  old_global_angle = global_angle;
  if (MySide == 'r'){  /* Used the same orientation for both sides above*/
    global_x *= -1;    /* Now convert positions back to team-relative   */
    global_y *= -1;    /* Before using the communicated global info     */
    global_angle += M_PI;
  }
  CleanAngleRad(&global_angle);

  /* Process updates--first stationary objects (to get own motion) */
  for(i=0; i<NUM_MARKERS; i++){
    objects[i]->tick();
  }

  /* Now update mobile objects */
  UpdateMobileObjectPositions(TRUE);

  if ( BallValid() == 1 )  /* For trajectory calculations */
    UpdateBallGlobalPosition();

  /* Compute the magnitude and direction of my velocity */
  if ( my_vel_x != NODIR && my_vel_y != NODIR ){
    my_vel_mag = sqrt(my_vel_x*my_vel_x + my_vel_y*my_vel_y);
    my_vel_dir = rad_to_deg(my_atan2(my_vel_y,my_vel_x));
    if (my_vel_x < 0) { 
      my_vel_dir += 180;
      CleanAngle(&my_vel_dir);
    }
  }
  else if ( !Dashed ){
    my_vel_mag = 0;
    my_vel_dir = 0; /* Assume no motion if I didn't dash ??? */
  }
  else {
    my_vel_mag = 0;
    my_vel_dir = NODIR;
  }

  /* Clean up for next tick */
  my_vel_x = my_vel_y = NODIR; /* No longer needed--conversions already done */
  NumExtraPlayers = 0;
  line = NO_LINE;

  if ( !InPitch(global_x,global_y) && MarkerValid(THEIR_GOAL) ){
    /* printf("%d:%d  Thought I was at (%.1f, %.1f) -- conf %.1f.  Must be confused.  Resetting\n",Mem->MyNumber,Mem->CurrentTime,global_x,global_y,MarkerValid(THEIR_GOAL));*/
    PositionInfo::Reset();
  }

#if DEBUG
  Print();
#endif
}

void PositionInfo::UpdateMobileObjectPositions(int erase_data){
  PositionMobile *MobileObject;
  for(int i=NUM_MARKERS; i<NUM_OBJECTS; i++){

    /* Process the heard globals                     */
    /* Must be after my global position has been set */
    MobileObject = (PositionMobile *) objects[i]; /* Just for type purposes */
    if ( MobileObject->got_global_info ){   
      float x,y,r,t;

      /* If I didn't just see the object */
      if ( !MobileObject->gospel || MobileObject->get_new_conf() < 1 ){ 
	MobileObject->get_globalxy(&x,&y);     /* retrieve communicated values     */
	globalxy_to_polar(x,y,&r,&t);          /* convert them to polar            */
	MobileObject->set_polar(r,t,HEARSAY_CONFIDENCE); /* It's hearsay, so not sure */
      }
      if (erase_data || (!turning && !dashing)) /* want to save it to reuse
						  after next sight */
	MobileObject->got_global_info = FALSE;
    }
    
    if ( erase_data ){                            /* coming from Tick() */
      int store_globals = FALSE;
      if ( MobileObject->gospel && MobileObject->side != 'b' ) /* not for the ball */
	store_globals = TRUE;

      MobileObject->tick(CurrentTime - LastTime); /* equivalent to objects[i]->tick();*/

      if (store_globals)
	UpdatePlayerGlobalPosition(MobileObject->side,MobileObject->num); 
    }
    else
      MobileObject->tick(0);                      /* don't degrade confidence */
  }
  
  DisambiguateExtraPlayers(); /* includes doing another tick */
}

void PositionInfo::DisambiguateExtraPlayers(){
  float x,y;
  int num=0;

  for (int i=0; i<NumExtraPlayers; i++){
    ExtraPlayers[i]->tick(0); /* Move the new values to cur values */
    GetMobileGlobalXY(ExtraPlayers[i],&x,&y);
#if 0
    printf("extra player %d.  R,d (%.1f,%.1f),  x,y (%.1f,%.1f)\n",
	   i,ExtraPlayers[i]->get_r(),ExtraPlayers[i]->get_tDEG(),x,y);
#endif
    num = ClosestUnseenPlayer(&(ExtraPlayers[i]->side),&x,&y);

    if (num){
      SeePlayer(ExtraPlayers[i]->side,num,
		ExtraPlayers[i]->get_r(),ExtraPlayers[i]->get_tDEG(),CurrentTime);

      if (ExtraPlayers[i]->side == MySide){
	UpdatePlayerGlobalPosition(MySide,num); 
	MyTeam[num]->tick(0);
      }
      else if (ExtraPlayers[i]->side == TheirSide){
	UpdatePlayerGlobalPosition(TheirSide,num); 
	TheirTeam[num]->tick(0); 
      }
      else 
	my_error("Should know the team at this point");
    }
  }
  NumExtraPlayers = 0;
}

void PositionInfo::ClearAction()
{
  turning = 0; dashing = 0;
  LastTurnAngle = 0;
}

void PositionInfo::Reset()
{
  int i;

  for(i=NUM_MARKERS; i<NUM_OBJECTS; i++)
    objects[i]->reset();  /* only reset mobile objects */
}

void PositionInfo::Turning(float angle)
{
  turning = 1;
  LastTurnAngle = angle;

#if 0
  /*  This is no longer needed--it happens in Tick based on the ACTUAL
      change of position instead of an estimate:  12/16/96           */
  /*  turning_angle = deg_to_rad(angle);*/
  for (int i=0; i<TEAM_SIZE; i++){
    MyTeam[i]->translate(0,-deg_to_rad(angle),1);    /* The degradation of confidence */
    TheirTeam[i]->translate(0,-deg_to_rad(angle),1); /* happens in Tick               */
  }
  Ball->translate(0,-deg_to_rad(angle),1);
#endif
}

void PositionInfo::Dashing(float power)
{
  dashing=1;
}

void PositionInfo::Kicking(float power, float angle)
{
  /* ball's angle is probably where you kicked it */
  Ball->set_t(deg_to_rad(angle), 0.8*Ball->get_tconf());
  Ball->set_r(power/6, 0.6*Ball->get_rconf());
}

void PositionInfo::Acting(int code) {
#if 0
  switch(code){
  case FORG: ForgetAll(); break;
  case FORGBALL: Ball->Forget(); break;
  case FORGMYGOAL: MyGoal->Forget(); break;
  case FORGTHEIRGOAL: TheirGoal->Forget(); break;
  default: my_error("don't know that acting code");
  }
#endif
}

void PositionInfo::Print(){
  int i;
  printf("  +--------------------------------------------------+\n");
  printf("  + Time = %d\n", CurrentTime); 
  printf("  + Global Angle = %f\n", rad_to_deg(global_angle));
  printf("  + Global Position= (%.1f,%.1f)\n",global_x,global_y);
  printf("  + Velocity = (%.1f,%.1f)\n",my_vel_dir,my_vel_mag);
  printf("  + Stamina Lower bound = %d\n",Mem->GetMyStamina());
  float x,y;

  for(i=0; i<NUM_MARKERS; i++) { 
    printf("  + Marker[%d] : Dist = %f, Theta = %f, Conf = %f\n",
	   i, Markers[i]->get_r(), Markers[i]->get_tDEG(), 
	   Markers[i]->get_conf());
    polar_to_globalxy(Markers[i]->get_r(), Markers[i]->get_t(), &x, &y);
    printf("  +              x = %.1f, y = %.1f\n",x,y);
    if ( i < 8 && 
	 (fabs(fabs(x)-fabs(marker_global[i].x)) > .1 || 
	  fabs(fabs(y)-fabs(marker_global[i].y)) > .1)){
      printf("  +              *****ALERT*****\n");
    }
  }

  printf("  + Ball : Dist = %f, Theta = %f, Conf = %f\n",
	 Ball->get_r(), Ball->get_tDEG(), Ball->get_conf());
  polar_to_globalxy(Ball->get_r(), Ball->get_t(), &x, &y);
  printf("  +              x = %.1f, y = %.1f, VelDir = %.1f, VelMag = %.1f\n",
	 x,y,GetBallVelDirection(),GetBallVelMagnitude());
/*
  for(i=1; i<3; i++) {
    printf("  + Teammate[%d] : Dist = %.2f, Theta = %.2f, Conf = %.2f, AngleConf = %.2f\n",
	   i, MyTeam[i]->get_r(), MyTeam[i]->get_tDEG(), 
	   MyTeam[i]->get_conf(), MyTeam[i]->get_tconf());
    polar_to_globalxy(MyTeam[i]->get_r(), MyTeam[i]->get_t(), &x, &y);
    printf("  +              x = %.1f, y = %.1f, VelDir = %.1f, VelMag = %.1f\n",
	   x,y,GetPlayerVelDirection(MySide,i),GetPlayerVelMagnitude(MySide,i));
  }
*/
/*
  for(i=1; i<12; i++) {
    printf("  + Opponent[%d] : Dist = %.2f, Theta = %.2f, Conf = %.2f, AngleConf = %.2f\n",
	   i, TheirTeam[i]->get_r(), TheirTeam[i]->get_tDEG(), 
	   TheirTeam[i]->get_conf(), TheirTeam[i]->get_tconf());
    polar_to_globalxy(TheirTeam[i]->get_r(), TheirTeam[i]->get_t(), &x, &y);
    printf("  +              x = %.1f, y = %.1f, VelDir = %.1f, VelMag = %.1f\n",
	   x,y,GetPlayerVelDirection(TheirSide,i),GetPlayerVelMagnitude(TheirSide,i));
  }
*/
  printf("  +--------------------------------------------------+\n\n");
}

float PositionInfo::triangulate(int m)
{
  float mx = Markers[m]->get_new_x();
  float my = Markers[m]->get_new_y();
  float gx, gy;
  int i;

  // Calculate global position
  polar_to_xy(Markers[m]->get_new_r(), 
	      global_angle - Markers[m]->get_new_t(), 
	      &gx, &gy);

  gx = marker_global[m].x - gx; gy = marker_global[m].y - gy;

  /* Peter's addition */
  if ( MySide == 'r' ){  /* x & y were flipped before, and will be overwritten soon */
    global_x *= -1;      /* here they need to be in team-independent format         */
    global_y *= -1;
  }
  float r_change = GetDistance(&gx,&gy,&global_x,&global_y);
  global_x = gx;
  global_y = gy;
  /* Peter's addition */

  // Set the positions of the markers
  for(i=0; i<NUM_EDGE_MARKERS; i++) { /* Do all markers if I want to move to P_FLAGs */
    float r, t;
    xy_to_polar(marker_global[i].x - gx, marker_global[i].y - gy,
		&r, &t);
    Markers[i]->set_polar(r, global_angle - t, 1.0);
  }

  return r_change;
}

float PositionInfo::overall_conf()
{
  int i;
  float min=1;

  for (i=0; i<NUM_MARKERS; i++) {
    float c = Markers[i]->get_conf();
    if (c < min) min = c;
  }

  return min;
}

float PositionInfo::globalxy_to_polar_r(float x, float y){
  return GetDistance(&x,&y,&global_x,&global_y);
}

float PositionInfo::globalxy_to_polar_t(float x, float y){
  float dx = x - global_x;
  float dy = y - global_y;
  float angle = my_atan2(dy,dx);
  return global_angle - angle;
}

int PositionInfo::GetViewAngle(int width){

  int ang = DEFAULT_VIEW_ANGLE;

  if ( width == NARROW_WIDTH )       /* "narrow" */
    ang /= 2;
  else if ( width == WIDE_WIDTH )  /* "wide"   */
    ang *= 2;

  return ang;
}

int PositionInfo::GetViewAngle(){
  return GetViewAngle(view_width);
}

int PositionInfo::TimePerSight(){

  int time = SEND_STEP;

  if ( view_width == NARROW_WIDTH )     
    time /= 2;
  else if ( view_width == WIDE_WIDTH )  
    time*= 2;
  if ( view_quality == LOW_QUALITY )   
    time /= 2;

  return time/SIMULATOR_STEP;  /* gives a lower bound on the time that's passed */
}

float PositionInfo::AngleToPoint(float x, float y){
  return rad_to_deg(globalxy_to_polar_t(x,y));
}

float PositionInfo::DistanceToPoint(float x, float y){
  float myX,myY;
  GetGlobalXY(&myX,&myY);
  return GetDistance(&myX,&myY,&x,&y);
}
