/* peterclient.c
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/*
 *  udp client program.
 */
#include "global.h"

#if PRACTICE
#define SAVE_LOG                0
#define SAVE_SOUND_LOG          0
#else
#define SAVE_LOG                0
#define SAVE_SOUND_LOG          0
#endif
#define RECV_DEBUG              0
#define TAKE_KBD_INPUT          0
#define PRINT_MISSED_MSGS       0

int SIMULATOR_STEP = DEFAULT_SIMULATOR_STEP;

Socket init_connection(char* host, int port);
int    send_message(char* buf, Socket *sock);
int    receive_message(char *buf, int size, Socket *sock, int *port, int flags);
int    ProcessInitializeMessage(Socket *sock, char *team_name, int unum);
void   ProcessIncomingMessage(Socket *sock);
int    ProcessKeyboardInput(Socket *sock, int *behavior, FILE *fpin);
void   message_loop(FILE* fpin, char* team_name, int unum, Socket *sock, int behavior, int my_score, int their_score);
void   close_connection(Socket sock);

/* Make these Global instead of passing them around */
Socket GLOBAL_SOCK;
Socket *GLOBAL_sock = &GLOBAL_SOCK;
Memory  GLOBAL_MEM;
Memory *Mem = &GLOBAL_MEM;

#if SAVE_LOG
FILE *fpout; 
char  fpoutName[MAXMESG];
#endif

#if SAVE_SOUND_LOG
FILE *fpSoundout; 
char  fpSoundoutName[MAXMESG];
#endif

/*
 * UDP connection γ
 *
 *  :
 * host : host name ⤷ host  IP address
 * port : port ֹ
 * 
 *  :
 * error ȯ硢Socket.socketfd  -1 ֤
 * 
 */
 
Socket init_connection(char *host, int port)
{
    struct hostent 	*host_ent ;
    struct in_addr      *addr_ptr ;
    struct sockaddr_in	cli_addr ;
    int			sockfd, val ;
    Socket		sock ;

    sock.socketfd = -1 ;

    if((host_ent = (struct hostent *)gethostbyname(host)) == NULL) {
	/* Check if a numeric address */
	if(inet_addr(host) == -1){
	    return sock ;
	}
    }
    else{
	addr_ptr = (struct in_addr *) *host_ent->h_addr_list ;
	host = inet_ntoa(*addr_ptr) ;
    }


    /*
     *  Open UDP socket.
     */
    if( (sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0){
	return sock ;  	/* Can't open socket. */
    }

   val = fcntl(sockfd, F_GETFL, 0) ;
   /* 5/17/96 Dave found that O_NONBLOCK didn't work for setting up a*/
   /* non-blocking socket.  O_NDELAY seems to work, though!          */
   /*    val |= O_NONBLOCK ;*/
    val |= O_NDELAY;
    fcntl(sockfd, F_SETFL, val) ;

    /*
     *  Bind any local address.
     */
    bzero((char *) &cli_addr, sizeof(cli_addr)) ;
    cli_addr.sin_family		= AF_INET ;
    cli_addr.sin_addr.s_addr	= htonl(INADDR_ANY) ;
    cli_addr.sin_port		= htons(0) ;


    if(bind(sockfd, (struct sockaddr *) &cli_addr,
	    sizeof(cli_addr)) < 0){
	return sock ;  /* Can't bind local address */
    }
    

    /*
     *  Fill in the structure with the address of the server.
     */
    sock.socketfd = sockfd ;

    bzero((char *) &sock.serv_addr, sizeof(sock.serv_addr)) ;
    sock.serv_addr.sin_family		= AF_INET ;
    sock.serv_addr.sin_addr.s_addr	= inet_addr(host) ;
    sock.serv_addr.sin_port		= htons(port) ;
    
    return sock ;
}


/*
 * socket  message 
 *
 *  :
 * buf  : message 
 * sock : socket
 * 
 *  :
 * Ф˼Ԥ硢-1 ֤
 * 
 */

#define TIMER_INIT_SECS  999999  /* Enough so it won't expire */
#define TIMER_INIT_USECS 1000*SIMULATOR_STEP
#define CHECK_SOCKET_INTERVAL_USECS 10000

/* Returns FALSE if there's no time to wait--ready to act right away */
/* Returns TRUE if the time ran out: any new messages were parsed    */
int message_poll_while_waiting(Socket *sock){
  struct itimerval itv2;
  int              dummy_port;
  char             buf[MAXMESG]; /* For peaking, this must be big */
  struct timeval   timeout;

  timeout.tv_sec = 0;
  timeout.tv_usec = CHECK_SOCKET_INTERVAL_USECS;

  getitimer(ITIMER_REAL, &itv2);
  if ( itv2.it_value.tv_sec != TIMER_INIT_SECS )
    return FALSE;

  while (itv2.it_value.tv_sec == TIMER_INIT_SECS){  /* changes after TIMER_INIT_USECS */
    /* Check for new messages while waiting to act                  */
    /* Right now this is all busy wait.  Put in some short usleeps? */

    if( receive_message(buf, MAXMESG, sock, &dummy_port, MSG_PEEK) > 0 ){
#if RECV_DEBUG
      /*
      getitimer(ITIMER_REAL, &itv2);
      fprintf(fpout,"Player %d, time %d: About to process, timer secs-usecs = %d-%d\n",
	     Mem->MyNumber,Mem->CurrentTime,itv2.it_value.tv_sec,itv2.it_value.tv_usec); 
      */
#endif
      ProcessIncomingMessage(sock);
#if RECV_DEBUG
      /*
      getitimer(ITIMER_REAL, &itv2);
      fprintf(fpout,"Player %d, time %d: Processed       , timer secs-usecs = %d-%d\n",
	     Mem->MyNumber,Mem->CurrentTime,itv2.it_value.tv_sec,itv2.it_value.tv_usec); 
      */
#endif
    }
    select(1,0,0,0,&timeout);  /* Pause for a bit so as not to busy wait */
    getitimer(ITIMER_REAL, &itv2);
  }
  return TRUE;
}

int message_poll_until_sight(Socket *sock)
{
  fd_set	readfds;
  int		in, max_fd;

  Mem->NewSight = FALSE;

  while ( !Mem->NewSight ){
    FD_ZERO(&readfds) ;
#if 0
    FD_SET(in, &readfds) ;
#else 
    in = 0;
#endif
    FD_SET(sock->socketfd, &readfds) ;
    
    max_fd = ((in > sock->socketfd) ? in : sock->socketfd) + 1 ;
    /* Select waits for something on the socket before acting         */
    if(select(max_fd, &readfds, 0, 0, 0) == -1){
      perror("select") ;
      break ;
    }

    if( FD_ISSET(sock->socketfd, &readfds) ){
      ProcessIncomingMessage(sock);
    }

#if 0
    if( FD_ISSET(in, &readfds) ){
      ProcessKeyboardInput(sock,&behavior,fpin);
    }
#endif
  }
  Mem->NewSight = FALSE;
}

int send_message(char *buf, Socket *sock)
{
  int n;
  static int initialized = FALSE;

  static struct itimerval itv;

#if 0
  int StartTime = Mem->CurrentTime;
#endif  
  
  if ( initialized ){ /* Don't act unless it's been enough time */
    message_poll_while_waiting(sock);
  }

  n = strlen(buf) ;
  if( sendto(sock->socketfd, buf, n, 0,
	        (struct sockaddr *)&sock->serv_addr, sizeof(sock->serv_addr))
      != n ){
    my_error("Why getting here? client.c: send_message");
    return (-1) ;
  }

  if ( initialized ){
    setitimer(ITIMER_REAL, &itv, NULL); 
  }
  else {                                   /* This is the initialize message */
    itv.it_interval.tv_sec = 0 ;           /* Don't have an fpout yet        */
    itv.it_interval.tv_usec = 0 ;
    itv.it_value.tv_sec = TIMER_INIT_SECS; /* So the clock doesn't run down  */
    itv.it_value.tv_usec = TIMER_INIT_USECS;
    initialized = TRUE; 
#if PRACTICE
    itv.it_value.tv_usec *= DELAY_FACTOR;
#endif
    return 0;
  }
  
#if SAVE_LOG
  /*if (Mem->CurrentTime != StartTime)
    fprintf(fpout, "   &&& Following command issued-sent at %d-%d &&&\n",
	    StartTime,Mem->CurrentTime);*/
  fprintf(fpout,">>");
  fputs(buf, fpout) ;
#endif

  /*usleep(1000 * SIMULATOR_STEP);    /* Wait simulator_step to let command execute */
  
  return 0;
}
	   
/*
 * socket  message μ
 *
 *  :
 * buf  : message  buffer
 * size : sizeof(buf)
 * sock : socket
 * 
 *  :
 * ˼Ԥ硢-1 ֤
 * Ƥ̵硢 0 ֤
 * Ƥ硢 1 ֤
 * 
 */

int receive_message(char *buf, int size, Socket *sock, int *port, int flags)
{
  /* flags is usually 0.  If it's MSG_PEEK, just check if the message is there */
    int			n,servlen ;
    struct sockaddr_in	serv_addr ;

    servlen = sizeof(serv_addr) ;

#if RECV_DEBUG
    struct itimerval itv2;
    getitimer(ITIMER_REAL, &itv2);
    int start_secs  = itv2.it_value.tv_sec; 
    int start_usecs = itv2.it_value.tv_usec;
#endif

    n = recvfrom(sock->socketfd, buf, size, flags,
			 (struct sockaddr *)&serv_addr, &servlen);

#if RECV_DEBUG
    getitimer(ITIMER_REAL, &itv2);
    int end_secs  = itv2.it_value.tv_sec; 
    int end_usecs = itv2.it_value.tv_usec;
    if (Mem->CurrentTime > 0)
      fprintf(fpout,"  #### Receiving took from %d-%d to %d-%d (%d)####\n",
	      start_secs,start_usecs,end_secs,end_usecs,flags);
#endif


    /* Remember the port for this client */
    if (*port == DEFAULT_PORT_NUMBER)
      /* If this doesn't work, use ntohs(serv_addr.sin_port) */
      *port = serv_addr.sin_port;

    if(n < 0)
	if( n == -1 && errno == EWOULDBLOCK){
	    buf[0] = '\0' ;
	    return 0 ;
	}
	else
	    return (-1) ;
    else{
	buf[n] = '\0' ;

	if(n == 0)
	    return 0 ;
	else
	    return 1 ;
    }
}


/*
 * messege 
 *
 *  :
 * fpin : (Ƥ socket ޤ)
 * fpout: (socket μƤޤ)
 * sock : socket
 * 
 */

int ProcessInitializeMessage(Socket *sock, char *team_name, int unum){

  int        port = DEFAULT_PORT_NUMBER, n;
  int        number = unum;
  char	     buf[MAXMESG];
  char       team, playmode[MAXMESG];

  n = receive_message(buf, MAXMESG, sock, &port, 0); 

  if(n == -1)
    my_error("n == -1 in initialize.  Why?");

  /************************************/
  /* First message from server used to*/
  /* set up client's memory and       */
  /* determine name of the log file   */
  /************************************/
  if ( !(strncmp(buf,"(init",4)) ) {
    /* It's an init msg */
    sscanf(buf,"(init %c %d %[^)]",&team, &number, playmode);
    Mem->Initialize(team,number,team_name,playmode,port);
  }
  else if ( !(strncmp(buf,"(reconnect",4)) ) {
    /* It's a reconnect msg */
    sscanf(buf,"(reconnect %c %[^)]",&team, playmode);
    printf("reconnecting to %d on side %c!\n",number,team);
    Mem->Initialize(team,number,team_name,playmode,port);
  }
  else /* Not an init message */
    return FALSE;

  /* Reset the port number to be used by this client */
  sock->serv_addr.sin_port = htons(port);

#if SAVE_LOG
  /**********************/
  /* Clear the log file */
  /**********************/
  sprintf(fpoutName,"%s%d-%c.log",team_name,number,team);
  fpout = fopen(fpoutName,"w");
#endif

#if SAVE_SOUND_LOG
  /**********************/
  /* Clear the log file */
  /**********************/
  sprintf(fpSoundoutName,"%s%d-%c-sounds.log",team_name,number,team);
  fpSoundout = fopen(fpSoundoutName,"w");
#endif

  return TRUE; /* successfully initialized */
}

/*********************************************************************************/

void ProcessIncomingMessage(Socket *sock){

  char	     buf[MAXMESG];
  int        MessageCounter = 0, n, dummy_port; /* port doesn't matter at this point */
  static int counter = 0;
   
  /* keep receiving messages until the socket is empty (n==0) */
  while((n = receive_message(buf, MAXMESG, sock, &dummy_port, 0)) > 0){

    if (MessageCounter > 0){    /* not first time through while */

#if PRINT_MISSED_MSGS	      
      printf("Player %d-%c skipped a message at time %d\n",
	     Mem->MyNumber,Mem->MySide,Mem->CurrentTime);
#endif
#if SAVE_LOG
      fputs("   *** SKIPPED LAST MESSAGE ***\n",fpout);
#endif

    }
    MessageCounter++;

#if SAVE_LOG
    /***************************************/
    /* Store messages received to the file */
    /***************************************/
    fputs(buf, fpout);
    
    /** Every once in a while, store data **/
    if (counter%SAVE_FREQ == 0){
      fclose(fpout);
      fpout = fopen(fpoutName,"a");
    }
#endif

#if SAVE_SOUND_LOG
    /***************************************/
    /* Store messages received to the file */
    /***************************************/
    if (buf[1] == 'h') /* only hear messages */
      fputs(buf, fpSoundout);
    
    /** Every once in a while, store data **/
    if (counter%SAVE_SOUND_FREQ == 0){
      fclose(fpSoundout);
      fpSoundout = fopen(fpSoundoutName,"a");
    }
#endif

    counter++;

    ParseSensoryData(buf);      /* parse the sound OR sight           */

    /* Respond immediately if the delay is 0 */
    if ( Mem->HeardNewSound() && !Mem->CommunicateDelay ){
                                /* respond to sound                   */ 
      Communicate(0);           /* Could make this behavior-dependent */
      Mem->ClearSound();        /* Undo the sound bit                 */
    }

  }
  
  if(n == -1)
    my_error("n == -1.  Why?");
}

/*********************************************************************************/

int ProcessKeyboardInput(Socket *sock, int *behavior, FILE *fpin){

  char	     buf[MAXMESG];

  fgets(buf, MAXMESG, fpin) ;
  sscanf(buf,"%d",behavior);

  if ( !(strncmp(buf,"q",1)) ){
    return FALSE;
    sscanf(buf,"(say (%s %d) %s)\n",Mem->MyTeamName,Mem->MyNumber,
	   BYE_MSG);
    if(send_message(buf, sock) == -1) my_error("send didn't work.  Why not");
  }
  if ( buf[0] == '(' ) {         /* Still take kbd input */
    if(send_message(buf, sock) == -1) ("send didn't work.  Why not? (2)");
  }
  else{
    switch(*behavior){
    case -8: Change_view(NORMAL_WIDTH,LOW_QUALITY); break;
    case -7: Change_view(NORMAL_WIDTH,HIGH_QUALITY); break;
    case FORGET: printf("Forget\n"); break;
    case RANDOM_INTERCEPT:  printf("Random Intercept\n"); break;
    case INTERCEPT:  printf("Intercept\n"); break;
    case SHOOT:  printf("Shoot\n"); break;
    case RANDOM_PASS:  printf("Random Pass\n"); break;
    case LEARNED_PASS:  printf("Learned Pass\n"); break;
    case COLLECT_PASS:  printf("Collect Pass\n"); break;
    case INTERCEPT_PASS:  printf("Intercept Pass\n"); break;
    case CHASE_AND_SHOOT: printf("ChaseAndShoot\n"); break;
    case DRIBBLE_AND_SHOOT: printf("DribbleAndShoot\n"); break;
    case DEFEND: printf("Defend\n"); break;
    case KAMRAM: printf("Kamram\n"); break;
    case GOALIE: printf("Goalie\n"); break;
    case CENTER_SWEEPER: printf("Sweeper\n"); break;
    case RIGHT_DEFENSE: printf("Right Defense\n"); break;
    case CENTER_DEFENSE: printf("Center Defense\n"); break;
    case LEFT_DEFENSE: printf("Left Defense\n"); break;
    case RIGHT_MIDFIELD: printf("Right Midfield\n"); break;
    case CENTER_MIDFIELD: printf("Center Midfield\n"); break;
    case LEFT_MIDFIELD: printf("Left Midfield\n"); break;
    case RIGHT_WING: printf("Right Wing\n"); break;
    case CENTER_FORWARD: printf("Center Forward\n"); break;
    case LEFT_WING: printf("Left Wing\n"); break;
    case START_SET_PLAY: printf("Start Set Play\n"); break;
    case RIGHT_MID_SET_PLAY: printf("Right Mid Set Play\n"); break;
    case LEFT_MID_SET_PLAY: printf("Left Mid Set Play\n"); break;
    case RIGHT_WINGER_SET_PLAY: printf("Right Winger Set Play\n"); break;
    case LEFT_WINGER_SET_PLAY: printf("Left Winger Set Play\n"); break;
    case FINISH_SET_PLAY: printf("Finish Set Play\n"); break;
    default: printf("Unknown behavior");
    }
  }
}

/*********************************************************************************/

void message_loop(FILE* fpin, char* team_name, int unum, Socket *sock, int behavior, int my_score, int their_score)
{
    fd_set	readfds;
    int		in, max_fd, FIRST_TIME = TRUE;
    int         retrieve_message = FALSE;

    in = fileno(fpin) ;
    while(1){
	FD_ZERO(&readfds) ;
	FD_SET(sock->socketfd, &readfds) ;

#if TAKE_KBD_INPUT
	FD_SET(in, &readfds) ;
	max_fd = ((in > sock->socketfd) ? in : sock->socketfd) + 1 ;
#else
	max_fd = sock->socketfd + 1 ;
#endif
	/* Select waits for something on the socket before acting         */
	/* If I want to speed through the loop again, take this out       */
	/* For some reason I seem to at least need it with TAKE_KBD_INPUT */
	if ( !Mem->NewSight ) { /* If free to act, zip right to action */
	  if(select(max_fd, &readfds, 0, 0, 0) == -1){
	    perror("select") ;
	    break ;
	  }
	  retrieve_message = TRUE;
	}

	if (retrieve_message){

/***********************************************************/
/* Send incoming messages to file and parse them to memory */
/***********************************************************/
	
	  if( FD_ISSET(sock->socketfd, &readfds) ){
	    if ( FIRST_TIME ){
	      if ( ProcessInitializeMessage(sock,team_name,unum) ){
		Mem->OriginalBehavior = behavior;
		Mem->MyScore = my_score;
		Mem->TheirScore = their_score;
		//printf("my_score/theirs(2):   %d/%d\n",my_score,their_score);
		FIRST_TIME = FALSE;  /* Otherwise, keep waiting */
	      }
	    }
	    else
	      ProcessIncomingMessage(sock);
	  }
    
/***************************************************/
/* Take the input from keyboard to choose behavior */
/***************************************************/

#if TAKE_KBD_INPUT	
	  if( FD_ISSET(in, &readfds) ){
	    if ( !ProcessKeyboardInput(sock,&behavior,fpin) )
	      break;            /* Received a quit message */
	  }
#endif
	  
	  retrieve_message = FALSE;
	}
/***************************/
/* Do programmed behaviors */
/***************************/

	if ( Mem->NewSight ){  /* For now, only act after a see--not a hear */
	  Mem->NewSight = FALSE;    /* Undo the sight bit */
	  Mem->ClearAction();
	  DoBehavior(behavior);
	}

	if ( Mem->HeardNewSound() ){ 
	  int TimeSinceSound;
	  if ( Mem->PlayMode == BEFORE_KICK_OFF ){
	    /* time measured in terms of sights */
	    if ( Mem->StoppedClockTime >= Mem->GetNewSoundTime() ) 
	      TimeSinceSound = Mem->StoppedClockTime - Mem->GetNewSoundTime();
	    else 
	      TimeSinceSound = Mem->StoppedClockTime; /* just changed to this mode */
	  }
	  else
	    TimeSinceSound = Mem->CurrentTime - Mem->GetNewSoundTime();
	  /* priority to behaviors */
	  if ( (TimeSinceSound >= Mem->CommunicateDelay) )
	    if ( !Mem->NewSight || TimeSinceSound >= MAX_COMMUNICATE_DELAY ){
	      Communicate(0);         /* Could make this behavior-dependent */
	      Mem->ClearSound();      /* Undo the sound bit                 */
	    }
	}
    }
#if SAVE_LOG  
    fclose(fpout);  /* Only stores the data if client ended with "q" */
#endif

#if SAVE_SOUND_LOG  
    fclose(fpSoundout);  /* Only stores the data if client ended with "q" */
#endif
}


/*
 * socket  close
 *
 *  :
 * sock : socket
 * 
 */

void close_connection(Socket sock)
{
    close(sock.socketfd) ;
}

/* for test */
#ifdef TEST
void main(int argc, char **argv)
{
    char *TeamName;
    int  unum = 0, my_score = 0, their_score = 0;
    int  arg_index = 1;
    char initMsg[MAXMESG];

    if(argc < 2) exit(1) ;

    GLOBAL_SOCK = init_connection(argv[arg_index++], DEFAULT_PORT_NUMBER) ;
    if(GLOBAL_sock->socketfd == -1)
	exit(1) ;

    if ( argv[arg_index][0] >= '0' && argv[arg_index][0] <= '9' ){
      /* It's a timing argument */
      SIMULATOR_STEP = atoi(argv[arg_index++]);
    }

    if (argc > 2) { 
      TeamName = strdup(argv[arg_index++]);
      sprintf(initMsg,"(init %s)\n", TeamName);   /*send the init msg*/
    }
    else 
      strcpy(TeamName, "CMUnited\n");

    if ( !strncmp(argv[0],"reconnect",8) ){
      if ( argc < arg_index+1 )
	my_error("Need a uniform number");
      unum = atoi(argv[arg_index++]);

      int score_arg_index = arg_index +1;

      if ( argc < score_arg_index+1 )
	printf("Assuming my score == 0\n");
      else
	my_score = atoi(argv[score_arg_index++]);

      if ( argc < score_arg_index+1 )
	printf("Assuming their score == 0\n");
      else
	their_score = atoi(argv[score_arg_index++]);

      sprintf(initMsg,"(reconnect %s %d)\n", TeamName, unum);   /*send the reconnect msg*/
    }

    if (send_message(initMsg, GLOBAL_sock) == -1)
      my_error("initMSG");

    int behavior = -2;
    if (argc > arg_index) {
      sscanf(argv[arg_index],"%d",&behavior);
    }

#if TAKE_KBD_INPUT 
    message_loop(stdin, TeamName, unum, GLOBAL_sock, behavior, my_score, their_score) ; 
#else
    FILE *fpin;
    fpin = fopen("blank-file","r");
    message_loop(fpin, TeamName, unum, GLOBAL_sock, behavior, my_score, their_score) ; 
    fclose(fpin);
#endif

    close_connection(GLOBAL_SOCK) ;
}
#endif /* TEST */

