/* parse.c
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

#include "global.h"

#define SELF      0
#define REFEREE  (SELF+1)
#define COACH    (REFEREE+1)
#define TEAMMATE (COACH+1)
#define OPPONENT (TEAMMATE+1)

int ParseSights(int, char*);
int ParseSound(int, char*);
void ParseRefereeMessage(char*, int);

void ParseSensoryData(char *SensoryInfo){

  //printf("%s\n",SensoryInfo);

  char *message;
  int time;

  if ( SensoryInfo[1]=='s' ){            /* see */
    SensoryInfo += 5;                    /* "(see " */
    time = get_int(SensoryInfo);         /* %d */
    while ( isdigit(*SensoryInfo) ) SensoryInfo++; /* advance past int */
    int closestMarker = ParseSights(time, SensoryInfo);
    Mem->NewSight = TRUE;
    if (Mem->PlayMode == BEFORE_KICK_OFF)
      Mem->StoppedClockTime += Mem->TimePerSight();
    Mem->Tick(closestMarker,time);  /* Tick relies on closestMarker < 0 if none visible */
  }
  else if ( SensoryInfo[1]=='h' ){       /* hear */
    SensoryInfo += 6;                    /* "(hear " */
    time = get_int(SensoryInfo);         /* %d */
    while ( isdigit(*SensoryInfo) ) SensoryInfo++; /* advance past int */
    int type = ParseSound(time, SensoryInfo);
    if ( type == TEAMMATE)
      Mem->UpdateMobileObjectPositions(FALSE); /* might have ball or player info */
  }
}

/*********************************************************************************/
/*********************************************************************************/
/*********************************************************************************/

int ParseSights(int time, char *buffer){

  float dist, ang;
  float dirChng = NODIR; 
  float distChng;
  char object;     /* just the first letter */
  char player_side, player_number;
  int marker, line;
  int view_qual = LOW_QUALITY;
  /* Tick relies on closestMarker < 0 if none visible */
  int closestMarker = -1, processThisMarker;
  float closestMarkerDist, motionInfoDist = 1000;

  while (*buffer != ')'){

    line = marker = NUM_MARKERS;
    dirChng = NODIR;
    player_number = player_side = 0;
    processThisMarker = FALSE;

    buffer+=3;            /* " ((" */
    object = *buffer;     /* object's first letter */

    if ( object=='g' ){
      buffer+=5;          /* "goal " */
      if ( *buffer=='r' )         marker = GOAL_R; else
      if ( *buffer=='l' )         marker = GOAL_L; else
      my_error("goal ?");
    } else
    if ( object=='G' )            marker = Mem->ClosestGoal(); else
    if ( object=='f' ){
      buffer+=5;          /* "flag " */
      if ( *buffer=='r' ){
	buffer+=2;
	if ( *buffer=='b' )       marker = FLAG_RB; else
	if ( *buffer=='t' )       marker = FLAG_RT; else
	my_error("flag r ?");
      } else
      if ( *buffer=='l' ){
	buffer+=2;
	if ( *buffer=='b' )       marker = FLAG_LB; else
	if ( *buffer=='t' )       marker = FLAG_LT; else
	my_error("flag l ?");
      } else
      if ( *buffer=='c' ){
	buffer+=2;
	if ( *buffer=='b' )       marker = FLAG_B; else
	if ( *buffer=='t' )       marker = FLAG_T; else
	my_error("flag c ?");
      } else
      if ( *buffer=='p' ){
	buffer+=2;
	if ( *buffer=='r' ){
	  buffer+=2;
	  if ( *buffer=='t')      marker = FLAG_PRT; else
	  if ( *buffer=='c')      marker = FLAG_PRC; else
	  if ( *buffer=='b')      marker = FLAG_PRB; else	    
	  my_error("flag p r ?");
	} else
	if ( *buffer=='l' ){
	  buffer+=2;
	  if ( *buffer=='t')      marker = FLAG_PLT; else
	  if ( *buffer=='c')      marker = FLAG_PLC; else
	  if ( *buffer=='b')      marker = FLAG_PLB; else	    
	  my_error("flag p l ?");
	} else
	my_error("flag p ?");
      } else
      my_error("flag ?");
    } else
    if ( object=='F' ){
      if ( Mem->ClosestFlag() )   marker = Mem->ClosestFlag();
      else ; /* it's a flag I don't handle yet */
    } else
    if ( object=='l' ){
      buffer+=5;          /* "line " */
      if ( *buffer=='r' )         line   = LINE_R; else
      if ( *buffer=='l' )         line   = LINE_L; else
      if ( *buffer=='t' )         line   = LINE_T; else
      if ( *buffer=='b' )         line   = LINE_B; else
      my_error("line ?");
    } else
    if ( object=='p' ){
      buffer+=6;          /* "player" */
      if ( *buffer == ' ' ){              /* there's a team */ 
	buffer++;
	if ( !strncmp(buffer,Mem->MyTeamName,Mem->MyTeamNameLen) )
	  player_side = Mem->MySide;
	else
	  player_side = Mem->TheirSide;
	while ( isalnum(*buffer) ) buffer++;   /* advance past team name */
	if ( *buffer== ' ' ){             /* there's a number */
	  buffer++;
	  player_number = get_int(buffer);
	}
      }
    }

    while (*(buffer++) != ')'); /* advance to end of object */
    buffer++;                   /* " " */        

    ang = get_float(buffer);
    while ( *buffer != ' ' && *buffer != ')' ) buffer++;

    if ( *buffer != ')' ) {                  /* 'high' quality     */
      buffer++;                              /* " " */
      view_qual = HIGH_QUALITY;
      Mem->view_quality = HIGH_QUALITY;      /* should be 'high'   */
      dist = ang;
      ang = get_float(buffer);
      while ( *buffer != ' ' && *buffer != ')' ) buffer++;
    }
    else 
      Mem->view_quality = LOW_QUALITY;       /* should be 'low'    */

    if ( *buffer != ')' ){
      buffer++;                              /* " " */
      distChng = get_float(buffer);
      while ( *(buffer++) != ' ' );          /* advance past num and space */
      dirChng  = get_float(buffer);
      while ( *buffer != ')' ) buffer++;
    }

    if ( *buffer != ')' ) my_error("Should be done with object info here");
    buffer++;                                /* ")" */

    if ( Mem->view_quality == HIGH_QUALITY &&/* angle is larger than I though */
	 object != 'l' &&                    /* Not a line  */
	 dist > FEEL_DISTANCE && fabs(ang) > Mem->GetViewAngle()/2 ){
      /* printf("%d:%d dist = %.1f, ang = %.1f so angle's wider than I think\n",
	     Mem->MyNumber,Mem->CurrentTime,dist,ang);*/
      if ( fabs(ang) > Mem->GetViewAngle(NORMAL_WIDTH)/2 )
	Mem->view_width = WIDE_WIDTH;
      else
	Mem->view_width = NORMAL_WIDTH; 
    }
    

/* marker */
    if ( marker < NUM_MARKERS ){                  /* It's a marker */

      if ( view_qual == LOW_QUALITY ){               /* Low quality   */
	; 
        /* DON'T BOTHER PROCESSING ANY??? I don't think it helps ... */
	/*if ( closestMarkerDist > 0 ){
	/*  closestMarkerDist = 0;                   /* Only process 1*/
	/*  processThisMarker = TRUE;
	/*}*/
      }
      else{                                          /* high quality  */
	if ( closestMarker < 0 || dist < closestMarkerDist){
	  closestMarker = marker;
	  closestMarkerDist = dist;
	  processThisMarker = TRUE;
	}
	if ( dirChng != NODIR && dist < motionInfoDist ){
	  motionInfoDist = dist;
	  processThisMarker = TRUE;
	}
      }
      
      if ( processThisMarker ){
	if ( view_qual == LOW_QUALITY )              /* low quality   */
	  Mem->SeeMarker(marker, ang, time);
	else if (dirChng == NODIR)                   /* high quality  */
	  Mem->SeeMarker(marker, dist, ang, time);   /* No motion info*/
	else
	  Mem->SeeMarker(marker, dist, ang, distChng, dirChng, time);
      }
    }

/* line */
    else if ( line < NUM_MARKERS ){               /* It's a line   */

      if ( *buffer != ')' )
	/* There's another line coming.  Assuming lines happen
	   last in the visual string and the closer line comes first */
	; 
      else if ( view_qual == LOW_QUALITY )           /* low quality   */
	Mem->SeeLine(line, ang, time);
      else                                           /* high quality  */
	Mem->SeeLine(line, dist, ang, time);
    }

/* ball */
    else if ( object=='b' ){                      /* It's the ball */

      if ( view_qual == LOW_QUALITY )                /* low quality   */
	Mem->SeeBall(ang, time);
      else if ( dirChng == NODIR )                   /* high quality  */
	Mem->SeeBall(dist, ang, time);
      else                                           /* know direction*/
	Mem->SeeBall(dist, ang, distChng, dirChng, time);         
    }

    else if ( object=='B' ){                      /* It's the ball */
                                                     /* out of view   */
      if ( view_qual == LOW_QUALITY )                /* low quality   */
	Mem->SeeBall(ang, time);  
      else                                           /* high quality  */
	Mem->SeeBall(dist, ang, time);         
    }

/* player */
    else if ( object=='p' ){                      /* It's another player  */

      if ( !player_side ){                      /* Too far for team or num */
	if ( view_qual == LOW_QUALITY )                /* low quality   */
	  Mem->SeePlayer(ang, time);
	else if ( dirChng == NODIR )                   /* high quality  */
	  Mem->SeePlayer(dist, ang, time);
	else                                           /* know direction*/
	  ;//my_error("Shouldn't know dirChng when the player's far");
      }  

      else{
	if ( !player_number ){                  /* Too far for number     */
	  if ( view_qual == LOW_QUALITY )                /* low quality   */
	    Mem->SeePlayer(player_side, ang, time);
	  else if ( dirChng == NODIR )                   /* high quality  */
	    Mem->SeePlayer(player_side, dist, ang, time);
	  else                                           /* know direction*/
	   ;//my_error("Shouldn't know dirChng when the team member's far");
	}

	else{                                   /* Know side AND number   */
	  if ( view_qual == LOW_QUALITY )                /* low quality   */
	    Mem->SeePlayer(player_side, player_number, ang, time);
	  else if ( dirChng == NODIR ){                  /* high quality  */
	    my_error("Should know dirChng when know number");
	    Mem->SeePlayer(player_side, player_number, dist, ang, time);
	  }
	  else                                           /* know direction*/
	    Mem->SeePlayer(player_side, player_number, dist, ang, distChng, dirChng, time);
	}
      }
    }

    else if ( object == 'P' ){                    /* It's another player    */
      if ( view_qual == LOW_QUALITY )                    /* low quality   */
	Mem->SeePlayer(ang, time);
      else if ( dirChng == NODIR )                       /* high quality  */
	Mem->SeePlayer(dist, ang, time);
	/*my_error("Should know dirChng when a player's within 3");*/
      else                                               /* know direction*/
	/* Mem->SeePlayer(dist, ang, distChng, dirChng, time);*/
        my_error("Shouldn't know dirChng when out of view"); 
    }
  }
  return closestMarker;
}

/*********************************************************************************/

int ParseSound(int time, char *msg){

  /** printf("sndr: %s  msg: %s\n",sndr,msg);          **/

  msg++; /* " " */

  if ( *msg=='r' ){         /* referee (or coach) */ 
    msg+=8;                 /* "referee " */
    if ( isupper(*msg) ){   /* coach messages are capitalized */
      HearCoach(msg,time);  
      return COACH;
    }
    else{
      ParseRefereeMessage(msg,time);
      return REFEREE;
    }
  }
  else if ( *msg != 's' ){ /* not "self": it's not your own message    */

    int num,form,formtime,pos,key;
    char team[50];
    float x,y;

    float ang = get_float(msg);
    while ( *(msg++) != ' ' );  /* advance past ang and space */

    char side;
    if ( !strncmp(msg,Mem->MyTeamName,Mem->MyTeamNameLen) )
      side = Mem->MySide;
    else
      side = Mem->TheirSide;

    if ( side == Mem->MySide ){
      while ( isalnum(*msg++) );     /* advance past team name and space */
      sscanf(msg,"%d %d %d %d %d %f %f %[^)]",&num,&key,&form,&formtime,&pos,&x,&y,msg);
      /* Call the communications.c HearPlayer which calls Mem->HearPlayer   */
      HearTeammate(num, key, ang, form, formtime, pos, x, y, msg, time);  
      return TEAMMATE;
    }
    else {
      HearOpponent(ang, msg, time, TRUE); /* It's the opponent--don't know number */
      return OPPONENT;
    }
  }
  return SELF;
}

/*********************************************************************************/

void ParseRefereeMessage(char *msg, int time){

  switch( msg[0] ){
  case 'p': Mem->PlayMode = PLAY_ON; break;                /* play_on */
  case 'k': 
    Mem->SetPlay = FALSE;
    Mem->SetMyselfInactive();
    if ( msg[5] == 'i' ){                                  /* kick_in */
      if ( msg[8] == Mem->MySide )
	Mem->PlayMode = MY_KICK_IN;
      else if ( msg[8] == Mem->TheirSide )
	Mem->PlayMode = THEIR_KICK_IN;
      else 
	my_error("kick_in_?");
      if ( Mem->QActionTaken )
	Mem->CloseRewards();
    }
    else if ( msg[5] == 'o' ){                            /* kick_off */
      if ( msg[9] == Mem->MySide )
	Mem->PlayMode = MY_KICK_OFF;
      else if ( msg[9] == Mem->TheirSide )
	Mem->PlayMode = THEIR_KICK_OFF;
      else 
	my_error("kick_off_?");
    }
    else
      my_error("referee k..?");
    break;
  case 'g': 
    Mem->SetPlay = FALSE;
    Mem->SetMyselfInactive();
    if ( msg[5] == 'k' ){                                 /* goal_kick */
      if ( msg[10] == Mem->MySide )
	Mem->PlayMode = MY_GOAL_KICK;
      else if ( msg[10] == Mem->TheirSide )
	Mem->PlayMode = THEIR_GOAL_KICK;
      else 
	my_error("goal_kick_?");
      if ( Mem->QActionTaken )
	Mem->CloseRewards();
    }
    else if ( msg[5] == Mem->MySide ){                    /* goal */
      Mem->MyScore++;
      /*
      printf("Server says my score = %c, I say %d\n",msg[7],
	     Mem->MyScore);
	     */

      Mem->StoppedClockTime=0; /* Measures sights since clock stopped */
      Mem->KickOffMode = THEIR_KICK_OFF;
      Mem->PlayMode = BEFORE_KICK_OFF;
    }
    else if ( msg[5] == Mem->TheirSide ){
      Mem->TheirScore++;
      /*
      printf("Server says their score = %c, I say %d\n",msg[7],
	     Mem->TheirScore);
	     */
      Mem->StoppedClockTime=0; /* Measures sights since clock stopped */
      Mem->KickOffMode = MY_KICK_OFF;
      Mem->PlayMode = BEFORE_KICK_OFF;
    }
    else 
      my_error("referee g..?");
    if ( Mem->QActionTaken )
      Mem->CloseRewards();
    break;
  case 'c':                                               /* corner_kick */
    Mem->SetPlay = FALSE;
    Mem->SetMyselfInactive();
    if ( msg[12] == Mem->MySide )
      Mem->PlayMode = MY_CORNER_KICK;
    else if ( msg[12] == Mem->TheirSide )
      Mem->PlayMode = THEIR_CORNER_KICK;
    else 
      my_error("corner_kick_?");
    if ( Mem->QActionTaken )
      Mem->CloseRewards();
    break;
  case 'd': Mem->PlayMode = DROP_BALL; break;             /* drop_ball */    
  case 'f':
    Mem->SetPlay = FALSE;
    Mem->SetMyselfInactive();
    if ( msg[5] == 'k' ){                                 /* free_kick */
      if ( msg[10] == Mem->MySide )
	Mem->PlayMode = MY_FREE_KICK;
      else if ( msg[10] == Mem->TheirSide )
	Mem->PlayMode = THEIR_FREE_KICK;
      else 
	my_error("free_kick_?");
    }
    else if ( msg[5] == Mem->MySide )                     /* foul */
      ;
    else if ( msg[5] == Mem->TheirSide )
      ;
    else 
      my_error("referee f..?");
    break;
  case 'h':                                               /* half_time */
    Mem->StoppedClockTime=0;    /* Measures sights since clock stopped */
    Mem->StaminaReset();        /* this is only chance before changing */
    Mem->PlayMode = HALF_TIME;  /* play_mode to before_kick_off        */
    Mem->SetPlay = FALSE;
    Mem->SetMyselfInactive();
    if ( Mem->MySide == 'l' )
      Mem->KickOffMode = THEIR_KICK_OFF;
    else 
      Mem->KickOffMode = MY_KICK_OFF;
    break;             
  case 'b': Mem->PlayMode = BEFORE_KICK_OFF; break;       /* before_kick_off */
  case 't': 
    if ( msg[5] == 'u' )                             /* time_up */
      Mem->PlayMode = TIME_UP; 
    else if ( msg[5] == 'o' )                             /* time_over */
      break;
    else if ( msg[5] == 'e' ){                            /* time_extended */
      Mem->StoppedClockTime = 0;
      Mem->StaminaReset();
      Mem->PlayMode = EXTENDED_TIME;
      Mem->SetPlay = FALSE;
      Mem->SetMyselfInactive();
      if ( Mem->MySide == 'l' )
	Mem->KickOffMode = MY_KICK_OFF;
      else 
	Mem->KickOffMode = THEIR_KICK_OFF;
    }
    else 
      my_error("referee t..?");
    break;
  /* case 'e': break;                                        /* extend */
  default: my_error("Referee msg ????");
  }
}

