/* memory.c
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

#include "global.h"

#define ACTIVE_DEBUG 0

/*--------------------------------------------------------------*/

Memory::Memory() : PositionInfo(), TeamPositionInfo(), RewardInfo(){

  NewSight = FALSE;
  StoppedClockTime = 0;
  strcpy(new_sound,BLANK_MSG);

  ClearActives();
  ClearStatuses();
  ClearData();
  ClearTheirSounds();

  ChangePositions = FALSE;
  UseUnitCoaching = FALSE;
  EchoSounds = FALSE;

  UseSetPlays = FALSE;
  SetPlay = FALSE;
  CurrentSetPlayFormation = (SetPlayFormation *) new SetPlayFormation;

  MoveRecorded = FALSE;
  LastDashPower = LastDashTimes = 0;
  PreDashGlobalX = PreDashGlobalY = 0;

  stamina_estimate = STAMINA_MAX;
}

Memory::~Memory(){
  delete CurrentSetPlayFormation;
}

Memory::Initialize(const char side, const int number, char *team_name, 
		   char *mode, int sock){

  PositionInfo::Initialize(side, number, team_name);

  socket = sock;

  my_team_status[0] = my_team_status[number]; /* Again, keep own status in 0 */
  my_team_data[0] = my_team_data[number];

  if ( mode[0] == 'b' ){ /* Before_kick_off */ 
    PlayMode = BEFORE_KICK_OFF;
    if ( side == 'l' )
      KickOffMode = MY_KICK_OFF;
    else 
      KickOffMode = THEIR_KICK_OFF;
  }
  else                  /* Act as if the game's in progress */
    PlayMode = PLAY_ON;
}

void Memory::Reset(){

  PositionInfo::Reset();
  if ( GetCurrentFormation() != NULL )
    TeamPositionInfo::Reset();

  SetMyselfInactive();
  ClearStatuses();
  ClearData();
}

void Memory::Dashing(float power){
 
  PositionInfo::Dashing(power);
  LastDashPower=power;

  stamina_estimate -= (int)power;
  if ( stamina_estimate < 0 )
    stamina_estimate = 0;
}

void Memory::Tick(int marker,int time){

  stamina_estimate += STAMINA_INC*(time - CurrentTime); /* Since it's before PI::Tick */

  if (stamina_estimate > STAMINA_MAX)
    stamina_estimate = STAMINA_MAX;

  PositionInfo::Tick(marker,time);
}

#if 0
float Memory::GetPlayerDistance(char side, int number){
  if ( number > TEAM_SIZE ) /* It's a position, not a uniform number */
    number = GetPositionHolder(number);

  return PositionInfo::GetPlayerDistance(side, number);
}

float Memory::GetPlayerAngle(char side, int number){
  if ( number > TEAM_SIZE ) /* It's a position, not a uniform number */
    number = GetPositionHolder(number);

  return PositionInfo::GetPlayerAngle(side, number);
}

float Memory::PlayerDistanceValid(char side, int number){
  if ( number > TEAM_SIZE ) /* It's a position, not a uniform number */
    number = GetPositionHolder(number);

  return PositionInfo::PlayerDistanceValid(side, number);
}

float Memory::PlayerAngleValid(char side, int number){
  if ( number > TEAM_SIZE ) /* It's a position, not a uniform number */
    number = GetPositionHolder(number);

  return PositionInfo::PlayerAngleValid(side, number);
}

float Memory::PlayerValid(char side, int number){
  if ( number > TEAM_SIZE ) /* It's a position, not a uniform number */
    number = GetPositionHolder(number);

  return PositionInfo::PlayerValid(side, number);
}
#endif

int Memory::FindTeammateWithStatus(char *status){

/*  if ( !strcmp(status,RECEIVER_STAT) )
    printf("2: %s  3: %s\n",my_team_status[2],my_team_status[3]);*/

  for (int i=1; i<TEAM_SIZE+1; i++){
    if ( !strcmp(my_team_status[i], status) )
      return i;
  }
  return FALSE;
}

int Memory::FindTeammatesWithStatus(char *status, int *teammates){

  int NumWithStatus = 0;

  for (int i=1; i<TEAM_SIZE+1; i++){
    if ( !strcmp(my_team_status[i], status) ){
      teammates[NumWithStatus] = i;
      NumWithStatus++;
    }
  }
  return NumWithStatus;
}

int Memory::SortPlayersBy(char side, char KeyFunc, float KeyNum, int* players){
  int result = 0; /*Number of players sorted */

  float (PositionInfo::*KeyFunction)(char,int);
  KeyFunction =
    (( KeyFunc == 'd' ) ? 
     &PositionInfo::GetPlayerDistance : &PositionInfo::GetPlayerAngle);
  float (PositionInfo::*TestFunction)(char,int);
  TestFunction =
    (( KeyFunc == 'd' ) ? 
     &PositionInfo::GetPlayerDistance : &PositionInfo::GetPlayerAngle);

  int num = (( side == 'b') ? TEAM_SIZE*2 : TEAM_SIZE); /* Make aux array big 
							   enough */
  float vals[num];
  char team = (( side == 't' ) ? TheirSide : MySide);
  for (int i=1; i<=TEAM_SIZE; i++){
    if ( (TestFunction)(team,i) ){
      players[result]=i;
      vals[result]   =fabs(KeyNum - (KeyFunction)(team,i));  /* use diff from key*/
      result++;
    }
  }
  
  if ( side == 'b' ){   /* Need to put in Their  team too */
    team = TheirSide;
    for (int i=1; i<=TEAM_SIZE; i++){
      if ( (TestFunction)(team,i) ){
	players[result]=i+TEAM_SIZE;    /* to distinguish from my team */
	vals[result]   =fabs(KeyNum - (KeyFunction)(team,i));  /* use diff from key*/
	result++;
      }
    } 
  }
		
/* Now should have all values in question in vals, with uniform number in 
   corresponding position of players ( +TEAM_SIZE for their team if 
   side == 'b'):  Just sort em */

  BubbleSort(result,players,vals);
  return result;
}

int Memory::NumTeammatesWithin(float Dist, float Ang, float ofDist, float ofAng){
  int result = 0;
  for (int i=1; i<=TEAM_SIZE; i++){
    if ( PlayerValid(MySide,i) &&
	 fabs(GetPlayerDistance(MySide,i) - ofDist) <= Dist &&
	 fabs(GetPlayerAngle   (MySide,i) - ofAng ) <= Ang )
      result++;
  }
  return result;
}

int Memory::NumOpponentsWithin(float Dist, float Ang, float ofDist, float ofAng){
  int result = 0;
  for (int i=1; i<=TEAM_SIZE; i++){
    if ( PlayerValid(TheirSide,i) &&
	 fabs(GetPlayerDistance(TheirSide,i) - ofDist) <= Dist &&
	 fabs(GetPlayerAngle   (TheirSide,i) - ofAng ) <= Ang )
      result++;
  }
  return result;
}

int Memory::NumPlayersWithin  (float Dist, float Ang, float ofDist, float ofAng){
  return 
    NumTeammatesWithin(Dist,Ang,ofDist,ofAng) + 
    NumOpponentsWithin(Dist,Ang,ofDist,ofAng);
}

int Memory::GetClosestTeammate(){
  int ClosestPlayer = 0;
  float dist, ClosestDist = 1000;
  for (int i=1; i<TEAM_SIZE+1; i++){
    if ( PlayerValid(MySide,i) &&
	 (dist=GetPlayerDistance(MySide,i)) < ClosestDist ){
      ClosestDist = dist;
      ClosestPlayer = i;
    }
  }
  return ClosestPlayer;
}

int Memory::GetClosestOpponent(){
  int ClosestPlayer = 0;
  float dist, ClosestDist = 1000;
  for (int i=1; i<TEAM_SIZE+1; i++){
    if ( PlayerValid(TheirSide,i) &&
	 (dist=GetPlayerDistance(TheirSide,i)) < ClosestDist ){
      ClosestDist = dist;
      ClosestPlayer = i;
    }
  }
  return ClosestPlayer;
}


  /* This is expensive.  If I do it too much, I should store global positions
     in the position structure---but remember to erase them every tick           */
float Memory::ClosestTeammateTo(float toX, float toY, int *teammate_number){
  float x,y,dist,BestDist = 10000;
  for (int i=1; i<=TEAM_SIZE; i++){
    if ( i == MyNumber || PlayerValid(MySide,i) ){ /* Exclude myself here? no. */
      GetPlayerGlobalXY(MySide,i,&x,&y);
      dist = GetDistance( &x, &y, &toX, &toY );
      if (dist < BestDist){
	*teammate_number = i;
	BestDist = dist;
      }
    }
  }
  return BestDist;
}

int Memory::InOwnPenaltyArea(){
  if ( MarkerValid(THEIR_GOAL) && GetGlobalX() < -PA_X && fabs(GetGlobalY()) < PA_Y )
    return TRUE;
  else 
    return FALSE;
}

int Memory::FacingBackInOwnPA(){
  if ( InOwnPenaltyArea() && fabs(rad_to_deg(GetGlobalAngle())) > 90 ){
    /* printf("%d facing back near own goal\n",MyNumber); */
    return TRUE;
  }
  else 
    return FALSE;
}

int Memory::FacingBackNearOwnGoal(){
  if ( MarkerValid(MY_GOAL) && GetMarkerDistance(MY_GOAL) < 25 && 
       fabs(rad_to_deg(GetGlobalAngle())) > 90 ){
    /* printf("%d facing back near own goal\n",MyNumber); */
    return TRUE;
  }
  else 
    return FALSE;
}

void Memory::ClearStatus(char side, int num){
    SetStatus(side,num, BLANK_STAT);  // Get rid of the open status
}

void Memory::ClearStatuses(){
  for (int i=1; i<=TEAM_SIZE; i++)
    SetStatus(MySide,i, BLANK_STAT);  // Get rid of all the open stats
}

void Memory::ClearStatuses(char *status){
  for (int i=1; i<=TEAM_SIZE; i++){
    if ( !strcmp(GetStatus(MySide,i),status) )
      SetStatus(MySide,i, BLANK_STAT);  /* Get rid of all the stats = status*/
  }
}

void Memory::ClearOtherStatuses(){
  for (int i=1; i<=TEAM_SIZE; i++) {
    if (i != MyNumber)
      SetStatus(MySide,i,BLANK_STAT);  // Get rid of all the open stats
  }
}

void Memory::ClearData(){
  for (int i=1; i<=TEAM_SIZE; i++)
    SetData(i, BLANK_DAT);  // Get rid of all the open stats
}

int Memory::HeardNewSound(){
  return ( strcmp(new_sound,BLANK_MSG) ? TRUE : FALSE );
}

Memory::NewSound(int time, int speaker, char *msg){  
  if (Mem->PlayMode == BEFORE_KICK_OFF)
    new_sound_time = Mem->StoppedClockTime;  /* time's not actually advancing */
  else
    new_sound_time = time;
  new_sound_speaker = speaker;
  strcpy(new_sound,msg);
}

Memory::ClearSound(){  strcpy(new_sound,BLANK_MSG);}

void Memory::ClearActives(){
  for (int i=0; i<=TEAM_SIZE; i++)
    SetInactive(i);
}

void Memory::SetMyselfActive(){
  SetActive(0);
  SetActive(MyNumber);
}

void Memory::SetMyselfInactive(){
  SetInactive(0);
  SetInactive(MyNumber);
}

void Memory::SetAtAttention(int player){
  if (player != 0 && player != MyNumber)
    my_error("other players should be marked as inactive when at attention");
  my_team_active[player] = AT_ATTENTION; 
}

void Memory::SetMyselfAtAttention(){
  SetAtAttention(0);
  SetAtAttention(MyNumber);
}

void Memory::SetAuxiliary(int player){
  if (player != 0 && player != MyNumber)
    my_error("other players should be marked as active when auxiliary");
  my_team_active[player] = AUXILIARY; 
}

void Memory::SetMyselfAuxiliary(){
  SetAuxiliary(0);
  SetAuxiliary(MyNumber);
}

void Memory::SetInSetPlay(int player){
  if (player != 0 && player != MyNumber)
    my_error("other players should be marked as active when InSetPlay");
  my_team_active[player] = INSETPLAY; 
}

void Memory::SetMyselfInSetPlay(){
  SetInSetPlay(0);
  SetInSetPlay(MyNumber);
}


void Memory::UpdateActive(){

/*  if ( PlayMode != PLAY_ON ) */
  if ( CurrentTime == 0 )
    return;

  /* Do this in terms of distance so it works even if the players not 
     back in position yet */
  float cdist, max_cdist = GetMaxActiveDistance();
  float ball_dist = GetBallDistance();

  switch( GetCurrentFormationType() ){
  case fRT_FORMATION: 
    if ( GetBallGlobalY() > 5 && PlayMode == PLAY_ON){
      SetMyselfInactive();
      return;
    }
    break;
  case fLT_FORMATION: 
    if ( GetBallGlobalY() < -5 && PlayMode == PLAY_ON){
      SetMyselfInactive();
      return;
    }
    break;
  }

  if ( strcmp(GetMyStatus(),RECEIVER_STAT) ){ /* Stay active if receiving */
    if ( PlayMode != PLAY_ON && GetMyPositionType() != GOALTENDER )
      max_cdist = 100;
  }

  if ( OnlyActiveIfClosest() ) /* not set in the previous switch function */
    cdist = MIN(max_cdist,.2+ClosestTeammateToBall());
  else
    cdist = max_cdist;

  cdist = MAX(GetMinInactiveDistance(), cdist);

#if 0
  float ball_vel_dir            = GetBallVelDirection();
  int   ball_vel_known          = ball_vel_dir != NODIR;
  int   ball_moving_towards     = ball_vel_known && fabs(ball_vel_dir) > 135;
  int   ball_not_moving_towards = ball_vel_known && !ball_moving_towards;
#else
  int   ball_vel_known          = BallTrajectoryValid() > 0 ? TRUE : FALSE;
  float ball_vel_dir;
  if ( ball_vel_known) 
        ball_vel_dir            = GetBallTrajectoryRelativeDirection();
  int   ball_moving_towards     = ball_vel_known && fabs(ball_vel_dir) > 150;
  int   ball_not_moving_towards = ball_vel_known && fabs(ball_vel_dir) < 120;
#endif

  if ( !strcmp(GetMyStatus(),RECEIVER_STAT) ){ /* I'm the receiver */
    if ( GetMyActiveStatus() != ACTIVE ){
#if ACTIVE_DEBUG
      if (Mem->MyNumber == 4)
      printf("%d                                     ----->    ACTIVE  (%d)\n",
	     Mem->CurrentTime,Mem->MyNumber);
#endif
      SetMyselfActive();
    }
    if ( ball_dist < cdist ||             /* Active in my own right */
	 GetMyStamina() < 500 )           /* Or getting tired       */
      Mem->ClearMyStatus();
    return;
  }

/* 
After you were active, if the ball's still within 40 and if you
 don't know what direction it's going, stay at attention until you do.
 If you KNOW it's not coming towards you, you can go inactive, if you
 KNOW it is, then you go back active...  Don't say going to position
 on transition from active to at attention.  But do from at attention
 to inactive 
*/

  /* when inactive or at attention, check if should go active */
  if ( GetMyActiveStatus() != ACTIVE && ball_dist < max_cdist ){
    if ( ball_dist < cdist || ball_moving_towards ){
      SAY(MY_BALL_MSG);
#if ACTIVE_DEBUG
      if (MyNumber == 4)
      printf("%d                                     ----->    ACTIVE  (%d)\n",
	     Mem->CurrentTime,Mem->MyNumber);
      if (Mem->MyNumber == 4 && ball_moving_towards)
	printf("   *** %.1f ***   \n",ball_vel_dir);
#endif
      SetMyselfActive();
      return;
    }
  }

#if PRACTICE
#else
  float targetX,targetY;
  if ( ball_vel_known && 
       GetCollectTargetXY(&targetX,&targetY) && /* I can get there */
       DistanceToPoint(targetX,targetY) < 10 &&
       (Mem->GetMyPosition() == UNKNOWN_POSITION || 
	TeamPositionInfo::IsWithinMyMaxRange(targetX,targetY)) ){
    int closest;
    ClosestTeammateTo(targetX,targetY,&closest);
    if ( closest == MyNumber ){
      /* printf("%d:%d Target (%.1f %.1f)\n",Mem->MyNumber,Mem->CurrentTime,
	     targetX,targetY); */
      SetMyselfActive();
      return;
    }
  }
#endif

  /* when active, check if should go inactive                */
  if ( GetMyActiveStatus() == ACTIVE && ball_dist > cdist ){

    if ( !ball_vel_known ){
#if ACTIVE_DEBUG
      if (Mem->MyNumber == 4)
      printf("%d                    AT ATTENTION     <-----    ACTIVE  (%d)\n",
	     Mem->CurrentTime,Mem->MyNumber);
#endif
      SetMyselfAtAttention();
    }
    else if ( ball_not_moving_towards ){
#if ACTIVE_DEBUG
      if (Mem->MyNumber == 4)
      printf("%d INACTIVE             <-----                   ACTIVE  (%d)\n",
	     Mem->CurrentTime,Mem->MyNumber);
      if (Mem->MyNumber == 4 && ball_not_moving_towards)
	printf("   *** %.1f ***   \n",ball_vel_dir);
#endif
      SetMyselfInactive();
      /* printf("%d:%d going inactive            bdist = %.1f, cdist =%.1f\n", 
	     Mem->MyNumber,Mem->CurrentTime,ball_dist,cdist);*/
      if ( Mem->GetMyPosition() == UNKNOWN_POSITION )
	Mem->SetMyPosition(GetNewPosition());
      ANNOUNCEMYPOSITION;
      return;
    }
  }


  /* when at attention, check if should go inactive          */
  if ( GetMyActiveStatus() == AT_ATTENTION ){
    if ( ball_dist > max_cdist ||                      /* or ballvel !known? */
	 (ball_dist > cdist && ball_not_moving_towards) ){ 
      SetMyselfInactive();
#if ACTIVE_DEBUG
      if (Mem->MyNumber == 4)
      printf("%d INACTIVE  <------- AT ATTENTION                       (%d)\n",
	     Mem->CurrentTime,Mem->MyNumber);
      if (Mem->MyNumber == 4 && ball_not_moving_towards)
	printf("   *** %.1f ***   \n",ball_vel_dir);
#endif
      if ( Mem->GetMyPosition() == UNKNOWN_POSITION )
	Mem->SetMyPosition(GetNewPosition());
      ANNOUNCEMYPOSITION;
      return;
    }
  }
}

void Memory::SetStatus(char side, int num, char *status){

  if ( side == MySide ){
    strcpy(my_team_status[num],status);
  } 
  else { /* side == TheirSide */
    my_error(" Not dealing with their_team_status");
    /* strcpy(their_team_status[num],status); */
  }
}

char *Memory::GetStatus(char side, int num){
  
  if ( side == MySide ){
    return my_team_status[num];
  }
  else { /* side == TheirSide */
    my_error(" Not dealing with their_team_status (get)");
    /* return their_team_status[num]; */
  }
}

void Memory::SetData(int num, char *data){
  strcpy(my_team_data[num],data);
}

char *Memory::GetData(int num){
  return my_team_data[num];
}

int Memory::InPosition(){
  float x = GetGlobalX(), y = GetGlobalY();
  float homeX = GetCurrentHomeX(), homeY = GetCurrentHomeY();
  float dist = GetDistance(&homeX,&homeY,&x,&y);

  if ( dist > GetMyPositionBuffer() ) 
    return FALSE;
  else
    return TRUE;
}

int Memory::InKickoffPosition(){
  if ( GetMyPosition() == UNKNOWN_POSITION )
    return FALSE;

  float x = GetGlobalX(), y = GetGlobalY();

  float goalX,goalY;
  if ( SetPlay && MyPositionInSetPlay() )
    GetMySetPlayXY(&goalX,&goalY);
  else{
    goalX = Mem->GetCurrentHomeX();
    goalY = Mem->GetCurrentHomeY();
  }

  PositionToKickoffPosition( &goalX, &goalY );
  float dist = GetDistance(&goalX,&goalY,&x,&y);

  if ( dist > GetMyPositionBuffer() ) 
    return FALSE;
  else
    return TRUE;
}

int Memory::GetPositionOfMyLocation(){
  return LocationToPosition(GetGlobalX(), GetGlobalY());
}

int Memory::GetMyLocationsPositionType(){
  return PositionType(GetPositionOfMyLocation());
}

int Memory::GetMyLocationsPositionSide(){
  return PositionSide(GetPositionOfMyLocation());
}
 
void Memory::CheckForPlayersInMyPosition(){

  for (int teammate=1; teammate<=TEAM_SIZE; teammate++){

    if ( teammate != MyNumber && TeammateValid(teammate) && 
	 GetTeammateDistance(teammate) < 10 ){
      int teammate_position = GetPlayerPosition(teammate);
      
      if ( 1 /* || GetActiveStatus(teammate) != ACTIVE */ ){
	if ( teammate_position == UNKNOWN_POSITION ){
	  /*printf("%d:%d  %d at unknown_position\n",
		 Mem->MyNumber,Mem->CurrentTime,teammate);*/
	  SAYTO(teammate,PING_MSG);
	}
	else if ( teammate_position == GetMyPosition() ){
	  /*printf("%d:%d  %d at my position (%d)\n",
		 Mem->MyNumber,Mem->CurrentTime,teammate,teammate_position);*/
	  SAYTO(teammate, ALREADY_THERE_MSG);
	}
      }
    }
  }
}

int   Memory::OnlyActiveIfClosest(){
  int position = GetMyPosition();
  if (position == UNKNOWN_POSITION)
    position = GetPositionOfMyLocation();
  return GetCurrentFormation()->GetPosition(position)->OnlyActiveIfClosest();
}

float Memory::GetMaxActiveDistance(){
  int position = GetMyPosition();
  if (position == UNKNOWN_POSITION)
    position = GetPositionOfMyLocation();
  return GetCurrentFormation()->GetPosition(position)->GetMaxActiveDistance();
}

float Memory::GetMinInactiveDistance(){
  int position = GetMyPosition();
  if (position == UNKNOWN_POSITION)
    position = GetPositionOfMyLocation();
  return GetCurrentFormation()->GetPosition(position)->GetMinInactiveDistance();
}

int Memory::IsWithinMyHomeRange(char side, int number){
  if ( !PlayerValid(side,number) )
    return FALSE;

  float x,y;
  GetPlayerGlobalXY(side, number, &x, &y);
  return TeamPositionInfo::IsWithinMyHomeRange(x,y);
}

int Memory::IsWithinMyMaxRange(char side, int number){
  if ( !PlayerValid(side,number) )
    return FALSE;

  float x,y;
  GetPlayerGlobalXY(side, number, &x, &y);
  return TeamPositionInfo::IsWithinMyMaxRange(x,y);
}

int Memory::AmWithinMyHomeRange(){
  return TeamPositionInfo::IsWithinMyHomeRange(GetGlobalX(),GetGlobalY());
}

int Memory::AmWithinMyMaxRange(){
  return TeamPositionInfo::IsWithinMyMaxRange(GetGlobalX(),GetGlobalY());
}

int Memory::BallIsWithinMyHomeRange(){
  float x,y;
  GetBallGlobalXY(&x,&y);
  return TeamPositionInfo::IsWithinMyHomeRange(x,y);
}

int Memory::BallIsWithinMyMaxRange(){
  float x,y;
  GetBallGlobalXY(&x,&y);
/*  return TeamPositionInfo::IsWithinMyMaxRange(x,y);*/
  int result =  TeamPositionInfo::IsWithinMyMaxRange(x,y);
/*
  if (!result){
    printf("%d: Ball is outside my max range (%.1f, %.1f)\n",
	   MyNumber,GetBallGlobalX(),GetBallGlobalY());
    PositionMaxRange(Mem->GetMyPosition())->Print();
  }
*/
  return result;
}

int   Memory::OpponentIsWithinPlayerHomeRange(int opponent, int player){
  float opponentx,opponenty;
  GetPlayerGlobalXY(TheirSide,opponent,&opponentx,&opponenty);

  return PositionHomeRange(GetPlayerPosition(player))->IsWithin(opponentx,opponenty);
}

int   Memory::OpponentIsWithinPlayerMaxRange(int opponent, int player){
  float opponentx,opponenty;
  GetPlayerGlobalXY(TheirSide,opponent,&opponentx,&opponenty);

  return PositionMaxRange(GetPlayerPosition(player))->IsWithin(opponentx,opponenty);
}

float Memory::PlayerDistanceToPositionHome(int player, int position){
  if ( player && player != MyNumber && !TeammateValid(player) )
    my_error("Don't know where the player is (PlayerDistanceToPositionHome)");
  
  float x,y;
  if ( !player || player == MyNumber ){
    x = GetGlobalX(); y = GetGlobalY();
  }
  else 
    GetTeammateGlobalXY(player,&x,&y);

  return DistanceToPositionHome(position,x,y);
}

void  Memory::UpdateHome(){

  /* need a HOME_GETOPEN */

  switch(HomeChangeMethod){
  case HOME_OBEY: /*printf("%d:%d home obey??\n",Mem->MyNumber,Mem->CurrentTime);*/ break;
  case HOME_MARK: /*printf("%d:%d home mark??\n",Mem->MyNumber,Mem->CurrentTime);*/ break;
  case HOME_MARK_ELSE_SHIFT:
  case HOME_SHIFT:
    /* Was Shade to ball side */
    if ( !Mem->BallValid() )
      my_error("Shouldn't be updating home if ball's not valid");

    float defaultY = GetMyPositionY(); 
    float defaultX = GetMyPositionX(); 
    float ballX = GetBallGlobalX();
    float ballY = GetBallGlobalY();
    float ShadeFactor = GetMyHomeRangeHeight()/(Y0*2);  /* Scale by field width     */
    float ShadeMag = (ballY - defaultY)*ShadeFactor;

    float ballDist = fabs(ballX - defaultX);
    ShadeMag *= MAX(0,(1 - ballDist/(X0*2)));           /* Scale by ball's distance */
    SetCurrentHomeY(defaultY + ShadeMag);

    /* drop back if the ball's on the other side of the field */
    ShadeFactor = (GetMyHomeRangeWidth()/(X0*2));      /* Scale by field length    */
    ShadeMag = fabs((ballY - defaultY)*ShadeFactor);
    ShadeMag *= fabs(ballY)/Y0;                        /* Scale by ball's position */

    if ( ballY*defaultY < 0 ) /* other side */
      SetCurrentHomeX(defaultX - ShadeMag);
    else
      SetCurrentHomeX(defaultX);
  }
}

void  Memory::UpdateMark(){

  int ClosestOpponent;

  switch(MarkChangeMethod){
  case MARK_OBEY: break;
  case MARK_CLOSEST: 
    /* Might want to prevent switching too hastily--first look for mark? */
    /* Don't yet Make sure the player's in home range */
    if (GetMark() != UNKNOWN_PLAYER)
      return;
    ClosestOpponent = GetClosestOpponent();
    if ( ClosestOpponent && IsWithinMyMaxRange(TheirSide,ClosestOpponent) )
      SetMark(ClosestOpponent); 
    else
      SetMark(UNKNOWN_PLAYER);
    break;
  case MARK_OPEN: my_error("Haven't defined MARK_OPEN yet"); break;
  }
}

int   Memory::GetMyReceiverList(int *ReceiverList){
  
  /* Go by my actual location, not my position */
  int MyActualLocation = GetPositionOfMyLocation();
  int *Rlist = GetCurrentFormation()->GetCandidateReceivers(MyActualLocation);
  for (int i=0; i<TEAM_SIZE; i++)
    ReceiverList[i] = Rlist[i]; /* Copy the list into here */

  int myPosition = GetMyPosition();
  int NumOptions = 0;
  while (ReceiverList[NumOptions] != MyActualLocation && NumOptions < TEAM_SIZE)
    NumOptions++;
  
  return NumOptions;
}

#define SETPLAY_TIME_LIMIT 100
void Memory::InitializeSetPlay(){

#if PRACTICE
  return;
#endif

  float ballx = GetBallGlobalX();
  float bally = GetBallGlobalY();

  int NextPlayMode;
  if ( PlayMode == BEFORE_KICK_OFF )
    NextPlayMode = KickOffMode;
  else
    NextPlayMode = PlayMode;

  /* To assign the x and y positions -- depends on where ball is*/
  if ( !CurrentSetPlayFormation->Initialize(NextPlayMode,ballx,bally) )
    return;

  /* Go through the positions in order, finding the closest player
     to that position, and mark the player as going there       */
  /* When it's me set my mode to Insetplay                      */

  float minDist, dist, x, y;
  int closestPosition, i, closestPlayer;
  int PositionAssigned[TEAM_SIZE];
  for (i=0; i<TEAM_SIZE; i++) PositionAssigned[i] = FALSE;

  for (i=0; i<CurrentSetPlayFormation->GetNumPositions(); i++){
    minDist = 1000;
    CurrentSetPlayFormation->GetSetPlayPositionXY(i,&x,&y);
    for (int position=0; position<TEAM_SIZE; position++){
      dist = DistanceToPositionHome(position,x,y);
      if (dist < minDist && !PositionAssigned[position] && 
	  PositionType(position) != GOALTENDER){ /* goalie not in set play */
	minDist = dist;
	closestPosition = position;
      }
    }

    closestPlayer = GetPositionPlayer(closestPosition);
    if ( closestPlayer == UNKNOWN_PLAYER && GetMyPosition() == UNKNOWN_POSITION ){
      SetMyPosition(GetNewPosition());
      ANNOUNCEMYPOSITION;
      closestPlayer = GetPositionPlayer(closestPosition);
    }
   
    CurrentSetPlayFormation->AssignSetPlayPosition(i,closestPosition);
    PositionAssigned[closestPosition] = TRUE;

    if ( closestPlayer == MyNumber ){
      SetMyselfInSetPlay();
    }
    /* printf("%d  closest postion: %d   player: %d  ball (%.1f, %.1f)  pos (%.1f %.1f)\n",
	     MyNumber,closestPosition,closestPlayer,ballx,bally,x,y); */
  }

  ClearStatuses();
  SetPlay = TRUE;
  SetPlayStartTime = 0;                /* Not started yet */
  SetPlayInvokeTime = CurrentTime;
  SetPlayTimeLimit = SETPLAY_TIME_LIMIT;
  SetPlayStartActionChosen = 0;
  SetPlayStartReceiver = 0;
}

int Memory::SetPlayPositionFilled(int pos){
  /* Pos is the SetPlay position */
  int formationPos = CurrentSetPlayFormation->SetPlayPositionPlayer(pos);
  if ( formationPos == UNKNOWN_POSITION ){
    my_error("Why aren't all set play positions filled?");
  }
  int player = GetPositionPlayer(formationPos);
  if ( player == UNKNOWN_PLAYER ) {
    /* printf("don't know who's playing position  %d   at %d\n",
	   formationPos, CurrentTime);*/
    return FALSE;
  }

  /* Assuming I know if I'm in place */
  if ( player == MyNumber || !strcmp(GetStatus(MySide,player),OPEN_STAT) )
    return TRUE;
  else 
    return FALSE;
}

int Memory::AllInSetPlayPosition(){
  float positionx,positiony,gx,gy,dist;
  for (int pos=0; pos<CurrentSetPlayFormation->GetNumPositions(); pos++){

    CurrentSetPlayFormation->GetSetPlayPositionXY(pos,&positionx,&positiony);
    GetMySetPlayXY(&gx,&gy);
    dist = GetDistance(&positionx,&positiony,&gx,&gy);

    /* Don't worry about far players being in position */
    if ( !SetPlayPositionFilled(pos) && dist < MAX_COMMUNICATE_DIST - 5){
      /* printf(" %d not filled\n",pos); */
      return FALSE;
    }
  }
  return TRUE;
}

int Memory::GetSetPlayStarterPosition(){
  return CurrentSetPlayFormation->GetStarter();  /* could be UNKNOWN_POSITION */
}

int Memory::PositionInSetPlay(int pos){
  for (int i=0; i<CurrentSetPlayFormation->GetNumPositions(); i++){
    if ( CurrentSetPlayFormation->SetPlayPositionPlayer(i) == pos )
      return TRUE;
  }
  return FALSE;
}

int Memory::MyPositionInSetPlay(){
  return PositionInSetPlay(GetMyPosition());
}

int Memory::BallInSetPlayPosition(){

  float ballx,bally;
  Mem->GetBallGlobalXY(&ballx,&bally);

  float ball_leeway = 1;
  float balldist  = Mem->GetBallDistance();
  if ( balldist > 50 ) ball_leeway = 3; /* Might not see it well enough */
  else ball_leeway += 2*(balldist/50);

  switch(PlayMode){
  case THEIR_KICK_IN:
  case MY_KICK_IN:
    if ( fabs(fabs(bally) - Y0) > ball_leeway ){ return FALSE; }
    break;
  case MY_CORNER_KICK:
    if ( fabs(ballx - X0) > ball_leeway )      { return FALSE; }
    if ( fabs(fabs(bally) - Y0) > ball_leeway ){ return FALSE; }
    break;
  case THEIR_CORNER_KICK:
    if ( fabs(ballx + X0) > ball_leeway )      { return FALSE; }
    if ( fabs(fabs(bally) - Y0) > ball_leeway ){ return FALSE; }
    break;
  case BEFORE_KICK_OFF:
  case THEIR_KICK_OFF:
  case MY_KICK_OFF:
    /* Ball is definitely at 0,0 */
    /* if ( fabs(ballx) > ball_leeway ){ return FALSE; }
       if ( fabs(bally) > ball_leeway ){ return FALSE; }    */
    break;
  case MY_GOAL_KICK:
    if ( fabs(ballx + GA_X) > ball_leeway ){ return FALSE;}
    if ( fabs(fabs(bally) - GA_Y) > ball_leeway ){ return FALSE;}
    break;
  case THEIR_GOAL_KICK:
    if ( fabs(ballx - GA_X) > ball_leeway ){ return FALSE;}
    if ( fabs(fabs(bally) - GA_Y) > ball_leeway ){ return FALSE;}
    break;
  case THEIR_FREE_KICK:
  case MY_FREE_KICK:
    break;
  default: return FALSE;  /* No set play for that mode */
  }
  return TRUE;
}

void Memory::ModifySetPlayStartAction(){
  if ( GetMySetPlayPositionType() != SETPLAY_STARTER )
    my_error("Only the starter should have to modify setplay action");

  /* Send the ball to the earliest filled position */
  /* Otherwise just shoot                          */
  /* Start with pos 1 so not to consider self      */

  int NumSetPlayPositions = CurrentSetPlayFormation->GetNumPositions();

  int choicePosition = 0;

  if ( SetPlayStartActionChosen > 1 ){ /* don't switch more than twice */
    choicePosition = SetPlayStartReceiver;
  }
  else if ( NumSetPlayPositions > 2 ){  /* There's a choice */
    int filledPositions[NumSetPlayPositions];

    for (int pos=1; pos<NumSetPlayPositions; pos++){
      if ( SetPlayPositionFilled(pos) )
	filledPositions[pos] = TRUE;
      else 
	filledPositions[pos] = FALSE;
    }
    
    int formationPos;
    int teammates[2];
    
    formationPos = CurrentSetPlayFormation->SetPlayPositionPlayer(1);
    teammates[0] = GetPositionPlayer(formationPos);
    formationPos = CurrentSetPlayFormation->SetPlayPositionPlayer(2);
    teammates[1] = GetPositionPlayer(formationPos);
    float confidence[2];
    
    if ( filledPositions[1] && filledPositions[2] ){
      choicePosition = FindTreeConfidences(2, teammates, confidence) + 1;    
      SetPlayStartActionChosen++;
      SetPlayStartReceiver = choicePosition;
      /* choose better of two */
    }
    else {
      for (int pos=1; pos<NumSetPlayPositions; pos++)
	if ( filledPositions[pos] ){  /* Take first filled position */
	  choicePosition = pos;
	  break;
	}
    }
  }
  else { /* There's no choice.  Just check if the player's there */
    if ( SetPlayPositionFilled(1) )
      choicePosition = 1;
  }

  if ( choicePosition ){
    CurrentSetPlayFormation->SpecifySetPlayAction(0,choicePosition,10);
    return;
  }

  /* Nobody in position */

  if ( PlayMode == MY_GOAL_KICK )
    CurrentSetPlayFormation->SpecifySetPlayAction(0,0,Y0*BallLocationSide(),0);
  else
    CurrentSetPlayFormation->SpecifySetPlayAction(0,X0,0,0);
}

float Memory::GetPositionSetPlayX(int pos){
  return CurrentSetPlayFormation->GetSetPlayPlayerX(pos);
}

float Memory::GetPositionSetPlayY(int pos){
  return CurrentSetPlayFormation->GetSetPlayPlayerY(pos);
}

void  Memory::GetPositionSetPlayXY(int pos, float *x, float *y){
  *x = GetPositionSetPlayX(pos); *y = GetPositionSetPlayY(pos);
}

float Memory::GetMySetPlayX(){
  return GetPositionSetPlayX(GetMyPosition());
}

float Memory::GetMySetPlayY(){
  return GetPositionSetPlayY(GetMyPosition());
}

void  Memory::GetMySetPlayXY(float *x, float *y){
  *x = GetMySetPlayX(); *y = GetMySetPlayY();
}

float  Memory::GetMySetPlayPositionBuffer(){
  return CurrentSetPlayFormation->GetPositionBuffer(GetMyPosition());
}

int Memory::InMySetPlayPosition(){
  float x,y,gx,gy;
  GetMySetPlayXY(&x,&y);
  GetGlobalXY(&gx,&gy);
  float dist = GetDistance(&x,&y,&gx,&gy);
  if (dist < GetMySetPlayPositionBuffer())
    return TRUE;
  else
    return FALSE;
}

float Memory::GetMySetPlayAimX(){
  return CurrentSetPlayFormation->GetSetPlayPlayerAimX(GetMyPosition());
}

float Memory::GetMySetPlayAimY(){
  return CurrentSetPlayFormation->GetSetPlayPlayerAimY(GetMyPosition());
}

void  Memory::GetMySetPlayAimXY(float *x, float *y){
  *x = GetMySetPlayAimX(); *y = GetMySetPlayAimY();
}

int   Memory::GetMySetPlayAimPosition(){
  return CurrentSetPlayFormation->GetAimPosition(GetMyPosition());
}

int   Memory::GetMySetPlayWaitTime(){
  return CurrentSetPlayFormation->GetWaitTime(GetMyPosition());
}

int   Memory::GetMySetPlayPositionType(){
  int position = GetMyPosition();
  if ( position == UNKNOWN_POSITION ) 
    my_error("Setplay position type with no position?");
  return CurrentSetPlayFormation->GetPositionType(position);
}

void  Memory::SetTheirSound(int num, char *msg, int time){
  strcpy(their_sounds[num],msg);
  their_sounds_times[num] = time;
}
  
char *Memory::GetTheirSound(int num){
  return their_sounds[num];
}

int   Memory::GetTheirSoundTime(int num){
  return their_sounds_times[num];
}

void  Memory::ClearTheirSound(int num){
  SetTheirSound(num,"blank",-1);
}

void  Memory::ClearTheirSounds(){
  for (int num=0; num< NUM_THEIR_SOUNDS_SAVED; num++)
    ClearTheirSound(num);
}

int   Memory::GetTheirOldestSoundNumber(){
  int oldest=-1, oldesttime = 1000000;
  for (int num=0; num< NUM_THEIR_SOUNDS_SAVED; num++)
    if ( GetTheirSoundTime(num) < oldesttime && GetTheirSoundTime(num) >=0 ){
      oldest = num;
      oldesttime = GetTheirSoundTime(num);
    }

  return oldest;
}

int   Memory::GetTheirBlankSoundNumber(){
  for (int num=0; num< NUM_THEIR_SOUNDS_SAVED; num++)
    if ( GetTheirSoundTime(num) < 0 )
      return num;

  return -1;
}

char *Memory::GetTheirOldestSound(){
  int oldest = GetTheirOldestSoundNumber();

  if (oldest < 0)
    return NULL;
  else
    return GetTheirSound(oldest);  /* could be NULL */
}

void  Memory::ReplaceTheirOldestSound(char *msg){
  int oldest = GetTheirBlankSoundNumber();
  if ( oldest < 0)
    oldest = GetTheirOldestSoundNumber();

  SetTheirSound(oldest,msg,CurrentTime);
}

