/* formation.h
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

#ifndef _FORMATION_H
#define _FORMATION_H

#include "position.h"

/*************************************************************************/
/************************ FORMATION DEFINITIONS **************************/
/*************************************************************************/

/************************ 4-3-3 Formation ********************************/

#define f433_FORMATION       0

#define f433_GOALIE           0
#define f433_SWEEPER         (f433_GOALIE         +1)
#define f433_RIGHT_DEFENSE   (f433_SWEEPER        +1)
#define f433_CENTER_DEFENSE  (f433_RIGHT_DEFENSE  +1)
#define f433_LEFT_DEFENSE    (f433_CENTER_DEFENSE +1)
#define f433_RIGHT_MIDFIELD  (f433_LEFT_DEFENSE   +1)
#define f433_CENTER_MIDFIELD (f433_RIGHT_MIDFIELD +1) 
#define f433_LEFT_MIDFIELD   (f433_CENTER_MIDFIELD+1) 
#define f433_RIGHT_WING      (f433_LEFT_MIDFIELD  +1) 
#define f433_CENTER_FORWARD  (f433_RIGHT_WING     +1) 
#define f433_LEFT_WING       (f433_CENTER_FORWARD +1) 


/************************ 4-4-2 Formation ********************************/

#define f442_FORMATION                 (f433_FORMATION      +1)

#define f442_GOALIE                     0
#define f442_SWEEPER                   (f442_GOALIE         +1)
#define f442_RIGHT_DEFENSE             (f442_SWEEPER        +1)
#define f442_CENTER_DEFENSE            (f442_RIGHT_DEFENSE  +1)
#define f442_LEFT_DEFENSE              (f442_CENTER_DEFENSE +1)
#define f442_RIGHT_MIDFIELD            (f442_LEFT_DEFENSE   +1)
#define f442_DEFENSIVE_CENTER_MIDFIELD (f442_RIGHT_MIDFIELD +1) 
#define f442_OFFENSIVE_CENTER_MIDFIELD (f442_DEFENSIVE_CENTER_MIDFIELD +1) 
#define f442_LEFT_MIDFIELD             (f442_OFFENSIVE_CENTER_MIDFIELD+1) 
#define f442_RIGHT_WING                (f442_LEFT_MIDFIELD  +1) 
#define f442_LEFT_WING                 (f442_RIGHT_WING +1) 


/************************ 3-5-2 Formation ********************************/

#define f352_FORMATION               (f442_FORMATION + 1)

#define f352_GOALIE                   0
#define f352_SWEEPER                 (f352_GOALIE         +1)
#define f352_RIGHT_DEFENSE           (f352_SWEEPER        +1)
#define f352_LEFT_DEFENSE            (f352_RIGHT_DEFENSE  +1)
#define f352_RIGHT_OUTSIDE_MIDFIELD  (f352_LEFT_DEFENSE   +1)
#define f352_RIGHT_INSIDE_MIDFIELD   (f352_RIGHT_OUTSIDE_MIDFIELD   +1)
#define f352_CENTER_MIDFIELD         (f352_RIGHT_INSIDE_MIDFIELD +1) 
#define f352_LEFT_INSIDE_MIDFIELD    (f352_CENTER_MIDFIELD+1) 
#define f352_LEFT_OUTSIDE_MIDFIELD   (f352_LEFT_INSIDE_MIDFIELD+1) 
#define f352_RIGHT_WING              (f352_LEFT_OUTSIDE_MIDFIELD  +1) 
#define f352_LEFT_WING               (f352_RIGHT_WING     +1) 


/************************ 7-2 Formation ********************************/

#define f72_FORMATION               (f352_FORMATION + 1)

#define f72_RIGHT_GOALIE             0
#define f72_LEFT_GOALIE             (f72_RIGHT_GOALIE   +1)
#define f72_RIGHT_DEFENSE           (f72_LEFT_GOALIE    +1)
#define f72_CENTER_DEFENSE          (f72_RIGHT_DEFENSE  +1)
#define f72_LEFT_DEFENSE            (f72_CENTER_DEFENSE +1)
#define f72_RIGHT_OUTSIDE_MIDFIELD  (f72_LEFT_DEFENSE   +1)
#define f72_RIGHT_INSIDE_MIDFIELD   (f72_RIGHT_OUTSIDE_MIDFIELD   +1)
#define f72_LEFT_INSIDE_MIDFIELD    (f72_RIGHT_INSIDE_MIDFIELD +1) 
#define f72_LEFT_OUTSIDE_MIDFIELD   (f72_LEFT_INSIDE_MIDFIELD+1) 
#define f72_RIGHT_WING              (f72_LEFT_OUTSIDE_MIDFIELD  +1) 
#define f72_LEFT_WING               (f72_RIGHT_WING     +1) 

/************************ 3-3-4 Formation ********************************/

#define f334_FORMATION               (f72_FORMATION + 1)

#define f334_GOALIE                   0
#define f334_SWEEPER                 (f334_GOALIE         +1)
#define f334_RIGHT_DEFENSE           (f334_SWEEPER        +1)
#define f334_LEFT_DEFENSE            (f334_RIGHT_DEFENSE  +1)
#define f334_RIGHT_MIDFIELD          (f334_LEFT_DEFENSE   +1)
#define f334_CENTER_MIDFIELD         (f334_RIGHT_MIDFIELD +1) 
#define f334_LEFT_MIDFIELD           (f334_CENTER_MIDFIELD+1) 
#define f334_RIGHT_WING              (f334_LEFT_MIDFIELD  +1) 
#define f334_RIGHT_INSIDE_FORWARD    (f334_RIGHT_WING     +1) 
#define f334_LEFT_INSIDE_FORWARD     (f334_RIGHT_INSIDE_FORWARD +1) 
#define f334_LEFT_WING               (f334_LEFT_INSIDE_FORWARD  +1) 

/************************ 2-4-4 Formation ********************************/

#define f244_FORMATION               (f334_FORMATION + 1)

#define f244_GOALIE                   0
#define f244_RIGHT_DEFENSE           (f244_GOALIE         +1)
#define f244_LEFT_DEFENSE            (f244_RIGHT_DEFENSE  +1)
#define f244_RIGHT_MIDFIELD          (f244_LEFT_DEFENSE   +1)
#define f244_DEFENSIVE_CENTER_MIDFIELD (f244_RIGHT_MIDFIELD +1) 
#define f244_OFFENSIVE_CENTER_MIDFIELD (f244_DEFENSIVE_CENTER_MIDFIELD+1) 
#define f244_LEFT_MIDFIELD             (f244_OFFENSIVE_CENTER_MIDFIELD+1) 
#define f244_RIGHT_WING              (f244_LEFT_MIDFIELD  +1) 
#define f244_RIGHT_INSIDE_FORWARD    (f244_RIGHT_WING     +1) 
#define f244_LEFT_INSIDE_FORWARD     (f244_RIGHT_INSIDE_FORWARD +1) 
#define f244_LEFT_WING               (f244_LEFT_INSIDE_FORWARD  +1) 

/************************ 4-3-2 Formation ********************************/

#define f432_FORMATION                 (f244_FORMATION      +1)

#define f432_RIGHT_GOALIE              0
#define f432_LEFT_GOALIE               (f432_RIGHT_GOALIE   +1)
#define f432_SWEEPER                   (f432_LEFT_GOALIE         +1)
#define f432_RIGHT_DEFENSE             (f432_SWEEPER        +1)
#define f432_CENTER_DEFENSE            (f432_RIGHT_DEFENSE  +1)
#define f432_LEFT_DEFENSE              (f432_CENTER_DEFENSE +1)
#define f432_RIGHT_MIDFIELD            (f432_LEFT_DEFENSE   +1)
#define f432_CENTER_MIDFIELD           (f432_RIGHT_MIDFIELD +1) 
#define f432_LEFT_MIDFIELD             (f432_CENTER_MIDFIELD+1) 
#define f432_RIGHT_WING                (f432_LEFT_MIDFIELD  +1) 
#define f432_LEFT_WING                 (f432_RIGHT_WING +1) 

/************************ RIGHT Formation ********************************/

#define fRT_FORMATION                 (f432_FORMATION      +1)

#define fRT_GOALIE                    0
#define fRT_INSIDE_FULLBACK           (fRT_GOALIE          +1)
#define fRT_OUTSIDE_FULLBACK          (fRT_INSIDE_FULLBACK +1)
#define fRT_INSIDE_QTRBACK            (fRT_OUTSIDE_FULLBACK+1)
#define fRT_OUTSIDE_QTRBACK           (fRT_INSIDE_QTRBACK  +1)
#define fRT_INSIDE_HALFBACK           (fRT_OUTSIDE_QTRBACK +1)
#define fRT_OUTSIDE_HALFBACK          (fRT_INSIDE_HALFBACK +1)
#define fRT_INSIDE_3QTRBACK           (fRT_OUTSIDE_HALFBACK+1)
#define fRT_OUTSIDE_3QTRBACK          (fRT_INSIDE_3QTRBACK +1) 
#define fRT_INSIDE_FORWARD            (fRT_OUTSIDE_3QTRBACK+1) 
#define fRT_OUTSIDE_FORWARD           (fRT_INSIDE_FORWARD  +1) 

/************************ LEFT  Formation ********************************/

#define fLT_FORMATION                 (fRT_FORMATION      +1)

#define fLT_GOALIE                    0
#define fLT_INSIDE_FULLBACK           (fLT_GOALIE          +1)
#define fLT_OUTSIDE_FULLBACK          (fLT_INSIDE_FULLBACK +1)
#define fLT_INSIDE_QTRBACK            (fLT_OUTSIDE_FULLBACK+1)
#define fLT_OUTSIDE_QTRBACK           (fLT_INSIDE_QTRBACK  +1)
#define fLT_INSIDE_HALFBACK           (fLT_OUTSIDE_QTRBACK +1)
#define fLT_OUTSIDE_HALFBACK          (fLT_INSIDE_HALFBACK +1)
#define fLT_INSIDE_3QTRBACK           (fLT_OUTSIDE_HALFBACK+1)
#define fLT_OUTSIDE_3QTRBACK          (fLT_INSIDE_3QTRBACK +1) 
#define fLT_INSIDE_FORWARD            (fLT_OUTSIDE_3QTRBACK+1) 
#define fLT_OUTSIDE_FORWARD           (fLT_INSIDE_FORWARD  +1) 

/*************************************************************************/

#define NUM_FORMATIONS fLT_FORMATION

/*************************************************************************/
/************************ CLASS DEFINITIONS ******************************/
/*************************************************************************/

/************************ RECTANGLE CLASS DEFINITION *********************/

class Rectangle {
public:
  void DefineRectangle(float x,float y,float w,float h);
  void DefineRectangleFromTLCorner(float x,float y,float w,float h);

  inline float TopY()     { return centerY+height/2; }
  inline float BottomY()  { return centerY-height/2; }
  inline float RightX()   { return centerX+width/2; }
  inline float LeftX()    { return centerX-width/2; }

  inline float Width()    { return width; }   /* X range */
  inline float Height()   { return height; }  /* Y range */

  float DistanceToEdge(float x, float y);
  int   IsWithin      (float x, float y);
  void  AdjustToWithin(float *x, float *y);

  void Print();
  
private:
  float centerX;  /* Coordinates of the center */
  float centerY;
  float width;
  float height;
};

/********************** TEAMPOSITION CLASS DEFINITION *********************/

/* Types and sides */
#define GOALTENDER 1
#define DEFENDER   (GOALTENDER + 1)
#define MIDFIELDER (DEFENDER   + 1)
#define FORWARD    (MIDFIELDER + 1)
#define LEFT       1
#define CENTER     0
#define RIGHT      (-LEFT)

/* Use for determining types and sides of positions */
#define ROUGH_GOALTENDERX -52
#define ROUGH_DEFENDERX   -35
#define ROUGH_MIDFIELDERX   0
#define ROUGH_FORWARDX     30
#define ROUGH_LEFTY        25
#define ROUGH_CENTERY       0
#define ROUGH_RIGHTY      -25

class TeamPosition {
public:
  TeamPosition();
  ~TeamPosition();

  void SetHome(float x, float y, float buf);
  void SetHomeRange(float width, float height);
  void SetMaxRange (float width, float height);

  void PrintHomeRange();
  void PrintMaxRange();

  inline float GetHomeX()      { return HomeX;      }
  inline float GetHomeY()      { return HomeY;      }
  inline float GetHomeBuffer() { return HomeBuffer; }

  inline Rectangle *GetHomeRange() { return HomeRange; }
  inline Rectangle *GetMaxRange()  { return MaxRange;  }

  inline void SetMaxActiveDistance(float x)  { MaxActiveDistance      = x; }
  inline void SetMinInactiveDistance(float x){ MinInactiveDistance    = x; }
  inline void SetOnlyActiveIfClosest(int x)  { only_active_if_closest = x; }

  inline float GetMaxActiveDistance()   { return MaxActiveDistance; }
  inline float GetMinInactiveDistance() { return MinInactiveDistance; }
  inline int   OnlyActiveIfClosest()    { return only_active_if_closest; }
  
  inline int GetType() { return type; }
  inline int GetSide() { return side; }

private:
  float HomeX;
  float HomeY;
  float HomeBuffer;

  Rectangle *HomeRange;
  Rectangle *MaxRange;

  float MinInactiveDistance;    /* Stay active if the ball is closer */
  float MaxActiveDistance;      /* Never active if ball is farther   */
  int   only_active_if_closest; /* If false, always go active if ball is closer 
				   than MaxActiveDistance */
  int type;
  int side; 
};

/*************************** UNIT CLASS DEFINITION ************************/

#define DEFENSIVE_UNIT  0
#define MIDFIELD_UNIT  (DEFENSIVE_UNIT+1)
#define FORWARD_UNIT   (MIDFIELD_UNIT +1)
#define LEFT_UNIT      (FORWARD_UNIT  +1)
#define CENTER_UNIT    (LEFT_UNIT     +1)
#define RIGHT_UNIT     (CENTER_UNIT   +1)
class Unit {
public:
  Unit();
  ~Unit();

  InitializeUnit(int tp, int sz, int *mems, int capt);

  inline int GetType()     { return type;    }
  inline int GetSize()     { return size;    }
  inline int GetCaptain()  { return captain; }
  inline int *GetMembers() { return members; }
  
  int IsMember(int position);
  int IsCaptain(int position);

  int ImportantPosition(int p1, int p2);

private:
  int type;      /* defensive, mid, left, center, etc. */
  int size;      /* number of members                  */
  int *members;  /* The POSITIONS in the unit          */
  int captain;   /* The Captain has special privileges */
};

/************************ FORMATION CLASS DEFINITION **********************/

class Formation {
public:

  Formation(int type);
  ~Formation();

  AllocateUnits(int num);

  inline int GetType() { return type; }

  void SetPosition(int num, float x, float y, float buffer,
		   float home_width, float home_height, 
		   float max_width, float max_height, float max_act_dist,
		   float min_inact_dist, int act_if_closest);

  TeamPosition *GetPosition(int num);
  
  void   SetCandidateReceivers(int position, int num, int *PrefList);
  int   *GetCandidateReceivers(int position);

  Unit *GetUnit(int unit);
  inline int GetNumUnits() {return NumUnits;}
  int IsUnitMember(int unit, int position);
  int IsUnitCaptain(int unit, int position);

  int IsMoreImportantPosition(int pos1, int pos2);

private:
  int type;

  TeamPosition PositionList[TEAM_SIZE];  /* A static list of positions:     */
                                         /* Shouldn't change once defined   */
  int ReceiverList[TEAM_SIZE][TEAM_SIZE];/* map position to candidate       */
                                         /* receiver positions              */
  int TempReceiverList[TEAM_SIZE];       /* Used to pass back possible receivers */

  int  NumUnits;
  Unit *FormationUnits;
};

/******************** TEAMPOSITIONINFO CLASS DEFINITION ******************/

#define UNKNOWN_POSITION  -1
#define UNKNOWN_PLAYER    -1
#define UNKNOWN_FORMATION NULL
#define UNKNOWN_FORMATION_TYPE -1

#define MARK_OBEY    0
#define MARK_CLOSEST (MARK_OBEY+1   )
#define MARK_OPEN    (MARK_CLOSEST+1)

#define HOME_OBEY             0
#define HOME_SHIFT           (HOME_OBEY+1   )
#define HOME_MARK            (HOME_SHIFT+1  )
#define HOME_MARK_ELSE_SHIFT (HOME_MARK +1  )  /* If I don't have a mark, shift */
 
class TeamPositionInfo {
public:

  TeamPositionInfo();
  ~TeamPositionInfo();
  
  void Reset();

  void SetPlayerPosition(int player, int position);
  void ClearPlayerPosition(int player);
  void ClearPositionPlayer(int position); 

  void SetMyPosition  (int position);
  void ClearMyPosition();

  int PositionOccupied (int position);

  int LocationToPosition(float x, float y);
  float DistanceToPositionHome(int position, float x, float y);
  
  float PositionHomeX          (int position);
  float PositionHomeY          (int position);
  float PositionHomeBuffer     (int position);
  int   PositionType           (int position);
  int   PositionSide           (int position);
  Rectangle *PositionHomeRange (int position);
  Rectangle *PositionMaxRange  (int position);
  float PositionHomeRangeWidth (int position);
  float PositionHomeRangeHeight(int position);
  float PositionMaxRangeWidth (int position);
  float PositionMaxRangeHeight(int position);
  float PlayerHomeX            (int player  );
  float PlayerHomeY            (int player  );
  float PlayerHomeBuffer       (int player  );
  int   PlayerType             (int player  );
  int   PlayerSide             (int player  );

  int   GetPlayerPosition(int player);
  int   GetPositionPlayer(int position);

  int   GetMyPosition();
  float GetMyPositionX();
  float GetMyPositionY();
  inline void GetMyPositionXY(float *x, float *y) 
    { *x = GetMyPositionX(); *y = GetMyPositionY(); }

  int   GetMyPositionType();
  int   GetMyPositionSide();
  float GetMyPositionBuffer();
  Rectangle *GetMyHomeRange();
  Rectangle *GetMyMaxRange();
  float GetMyHomeRangeWidth();
  float GetMyHomeRangeHeight();
  float GetMyMaxRangeWidth();
  float GetMyMaxRangeHeight();

  void PrintMyHomeRange();
  void PrintMyMaxRange();

  int   IsWithinMyHomeRange(float x, float y);
  int   IsWithinMyMaxRange (float x, float y);

  void  ResetCurrentHome();

  void  SetFormation (int formation_number, int time);
  inline Formation *GetCurrentFormation() { return CurrentFormation; }
  inline int GetCurrentFormationType() { return CurrentFormation->GetType(); }

  inline float GetCurrentHomeX() { return CurrentHomeX; }
  inline float GetCurrentHomeY() { return CurrentHomeY; }
  inline int   GetMark()         { return Mark;         }
  inline float GetMarkBuffer()   { return MarkBuffer;   }

  int SetCurrentHome (float x, float y);
  int SetCurrentHomeX(float x );
  int SetCurrentHomeY(float y );
  int SetMark        (int mark);
  inline void SetMarkBuffer(float buffer) { MarkBuffer = buffer; }

  void HearTeammateFormationPosition(int player, int form, int formtime, int pos);

  int GetUnitSize(int unit);
  int *GetUnitMembers(int unit);
  int IsUnitMember(int unit, int player);
  int IsUnitCaptain(int unit, int player);
  int IsMyUnit(int unit);
  int AmUnitCaptain(int unit);
  int IsMyUnitCaptain(int player);

  int AmCandidateReceiverOfPlayer(int player);

  int IsMoreImportantPosition(int pos1, int pos2);

  /**** Public Variables  ***********************************************/

  int   MarkChangeMethod;       /* There might be different ways of     */
  int   PositionChangeMethod;   /* deciding who to mark, when to change */ 
  int   HomeChangeMethod;       /* positions, and where to make home    */

  int   MarkPursuitRange;       /* range, max_range, everywhere, etc.   */
  int   PositionStrategy;       /* evade, mark, hold home, etc.         */

  int   FormationTime;          /* Time the current formation was set   */

private:
  float CurrentHomeX;  /* These could be different from HomeX and HomeY */
  float CurrentHomeY;  /* In my position                                */

  int   Mark;          /* The number of the opponent whom I'm marking   */
  float MarkBuffer;    /* The distance at which I should mark           */

  int PositionPlayers[TEAM_SIZE+1];  /* Maps positions to their players */
  int PlayerPositions[TEAM_SIZE+1];  /* Maps players to their positions */
  
  Formation *CurrentFormation;
  Formation *f433_Formation;
  Formation *f442_Formation;
  Formation *f352_Formation;
  Formation *f72_Formation;
  Formation *f334_Formation;
  Formation *f244_Formation;
  Formation *f432_Formation;
  Formation *fRT_Formation;
  Formation *fLT_Formation;
};


#endif

