/* formation.c
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

#include "global.h"

#define MARK_DEBUG          0
#define TEAM_POSITION_DEBUG 0

/*************************************************************************/
/************************ FORMATION DEFINITIONS **************************/
/*************************************************************************/

#define GOALIEMAXACTIVEDIST 14
#define MAXACTIVEDIST       40
#define MININACTIVEDIST     10

/************************ 4-3-3 Formation ********************************/

#define f433_GOALIE_BUF      2
#define f433_POSITION_BUF    3

#define f433_GOALTENDERX  (-52)
#define f433_SWEEPERX     (-43)
#define f433_DEFENDERX    (-35)
#define f433_MIDFIELDERX     0
#define f433_FORWARDX       30
#define f433_LEFTY          25
#define f433_CENTERY         0
#define f433_RIGHTY       (-25)

#define f433_HOMEXRANGE f433_FORWARDX
#define f433_HOMEYRANGE f433_LEFTY
#define f433_MAXXRANGE  (f433_FORWARDX*2)
#define f433_MAXYRANGE  (((Y0+5) - f433_LEFTY)*2)

void Initialize_433_Formation(Formation *formation){
  
  /* set position within goal stay, in penalty box      */
  formation->SetPosition(f433_GOALIE, f433_GOALTENDERX, f433_CENTERY, f433_GOALIE_BUF, 
			 (X0- (-f433_GOALTENDERX))*2, GOAL_WIDTH,
			 /*(-f433_GOALTENDERX - PA_X)*2, PA_Y*2,*/
			 (X0- (-f433_GOALTENDERX))*2, GOAL_WIDTH,
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, FALSE);  

  /* set position inside penalty box, don't go more than half-way to mid  */
  formation->SetPosition(f433_SWEEPER, f433_SWEEPERX, f433_CENTERY, f433_POSITION_BUF, 
			 (X0- (-f433_SWEEPERX))*2, PA_Y*2,        
			 -f433_SWEEPERX - X0/2, (Y0+5)*2,
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, TRUE);           

  formation->SetPosition(f433_RIGHT_DEFENSE, f433_DEFENDERX, f433_RIGHTY, f433_POSITION_BUF, 
			 f433_HOMEXRANGE,f433_HOMEYRANGE,
			 f433_MAXXRANGE ,f433_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f433_CENTER_DEFENSE, f433_DEFENDERX+5, f433_CENTERY, f433_POSITION_BUF, 
			 f433_HOMEXRANGE,f433_HOMEYRANGE,
			 f433_MAXXRANGE ,f433_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f433_LEFT_DEFENSE, f433_DEFENDERX, f433_LEFTY, f433_POSITION_BUF, 
			 f433_HOMEXRANGE,f433_HOMEYRANGE,
			 f433_MAXXRANGE ,f433_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f433_RIGHT_MIDFIELD, f433_MIDFIELDERX, f433_RIGHTY, f433_POSITION_BUF, 
			 f433_HOMEXRANGE,f433_HOMEYRANGE,
			 f433_MAXXRANGE ,f433_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f433_CENTER_MIDFIELD, f433_MIDFIELDERX, f433_CENTERY, f433_POSITION_BUF, 
			 f433_HOMEXRANGE,f433_HOMEYRANGE,
			 f433_MAXXRANGE ,f433_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f433_LEFT_MIDFIELD, f433_MIDFIELDERX, f433_LEFTY, f433_POSITION_BUF, 
			 f433_HOMEXRANGE,f433_HOMEYRANGE,
			 f433_MAXXRANGE ,f433_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);
  
  formation->SetPosition(f433_RIGHT_WING, f433_FORWARDX, f433_RIGHTY, f433_POSITION_BUF, 
			 f433_HOMEXRANGE,f433_HOMEYRANGE,
			 f433_MAXXRANGE ,f433_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f433_CENTER_FORWARD, f433_FORWARDX+5, f433_CENTERY, f433_POSITION_BUF, 
			 f433_HOMEXRANGE,f433_HOMEYRANGE,
			 f433_MAXXRANGE ,f433_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f433_LEFT_WING, f433_FORWARDX, f433_LEFTY, f433_POSITION_BUF, 
			 f433_HOMEXRANGE,f433_HOMEYRANGE,
			 f433_MAXXRANGE ,f433_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  /* Candidate receivers will be chosen on the fly here */
  int dummylist[TEAM_SIZE];
  for (int i=0; i<TEAM_SIZE; i++)
    formation->SetCandidateReceivers(i,0,dummylist);

  formation->AllocateUnits(6);

  /* Positions listed in order of importance, captain indicated as index into list */

  int defenders[5] = {f433_GOALIE, f433_SWEEPER,
		      f433_CENTER_DEFENSE, f433_RIGHT_DEFENSE, f433_LEFT_DEFENSE};
  formation->GetUnit(DEFENSIVE_UNIT)->InitializeUnit(DEFENSIVE_UNIT, 5, defenders, 0);

  int midfielders[3] = {f433_CENTER_MIDFIELD, f433_RIGHT_MIDFIELD, f433_LEFT_MIDFIELD};
  formation->GetUnit(MIDFIELD_UNIT)->InitializeUnit(MIDFIELD_UNIT, 3, midfielders, 0);

  int forwards[3] = {f433_CENTER_FORWARD, f433_RIGHT_WING, f433_LEFT_WING};
  formation->GetUnit(FORWARD_UNIT)->InitializeUnit(FORWARD_UNIT, 3, forwards, 0);

  int rights[3] = {f433_RIGHT_DEFENSE, f433_RIGHT_MIDFIELD, f433_RIGHT_WING};
  formation->GetUnit(RIGHT_UNIT)->InitializeUnit(RIGHT_UNIT, 3, rights, 1);

  int centers[5] = {f433_GOALIE, f433_SWEEPER, f433_CENTER_DEFENSE, 
		    f433_CENTER_MIDFIELD, f433_CENTER_FORWARD};
  formation->GetUnit(CENTER_UNIT)->InitializeUnit(CENTER_UNIT, 5, centers, 3);

  int lefts[3] = {f433_LEFT_DEFENSE, f433_LEFT_MIDFIELD, f433_LEFT_WING};
  formation->GetUnit(LEFT_UNIT)->InitializeUnit(LEFT_UNIT, 3, lefts, 1);
}


int *Build_433_ReceiverList(int *PreferenceList, int position){

  int NumOptions=0;
  
  int LocationSide = Mem->GetMyLocationSide();

  switch ( position ){
  case f433_CENTER_FORWARD: break;
  case f433_LEFT_WING:
  case f433_RIGHT_WING:
    NumOptions = (int_random(3) ? 1 : 0); /* shoot 1 in 3 times */
    PreferenceList[0] = f433_CENTER_FORWARD; 
    break;
  case f433_CENTER_MIDFIELD:
    NumOptions = 5;
    PreferenceList[0] = (LocationSide == RIGHT) ? f433_RIGHT_WING : f433_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f433_LEFT_WING : f433_RIGHT_WING;
    PreferenceList[2] = f433_CENTER_FORWARD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f433_RIGHT_MIDFIELD :f433_LEFT_MIDFIELD;
    PreferenceList[4] = (LocationSide == RIGHT) ? f433_LEFT_MIDFIELD :f433_RIGHT_MIDFIELD;
    break;
  case f433_RIGHT_MIDFIELD:
  case f433_LEFT_MIDFIELD:
    NumOptions = 4;
    PreferenceList[0] = f433_CENTER_FORWARD;
    PreferenceList[1] = f433_CENTER_MIDFIELD;
    PreferenceList[2] = (LocationSide == RIGHT) ? f433_RIGHT_WING : f433_LEFT_WING;
    PreferenceList[3] = (LocationSide == RIGHT) ? f433_LEFT_WING : f433_RIGHT_WING;
    break;
  case f433_CENTER_DEFENSE:    
    NumOptions = 3;
    PreferenceList[0] = (LocationSide == RIGHT) ? f433_RIGHT_MIDFIELD : f433_LEFT_MIDFIELD;
    PreferenceList[1] = (LocationSide == RIGHT) ? f433_LEFT_MIDFIELD : f433_RIGHT_MIDFIELD;
    PreferenceList[2] = f433_CENTER_MIDFIELD;
    break;
  case f433_RIGHT_DEFENSE:
  case f433_LEFT_DEFENSE:
    NumOptions = 3;
    PreferenceList[0] = (LocationSide == RIGHT) ? f433_RIGHT_MIDFIELD : f433_LEFT_MIDFIELD;
    PreferenceList[1] = (LocationSide == RIGHT) ? f433_RIGHT_WING : f433_LEFT_WING;
    PreferenceList[2] = f433_CENTER_MIDFIELD;
    break;
  case f433_SWEEPER:
  case f433_GOALIE:
    NumOptions = 7;
    PreferenceList[0] = (LocationSide == RIGHT) ? f433_RIGHT_DEFENSE : f433_LEFT_DEFENSE;
    PreferenceList[1] = (LocationSide == RIGHT) ? f433_LEFT_DEFENSE : f433_RIGHT_DEFENSE;
    PreferenceList[2] = (LocationSide == RIGHT) ? f433_RIGHT_MIDFIELD : f433_LEFT_MIDFIELD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f433_LEFT_MIDFIELD : f433_RIGHT_MIDFIELD;
    PreferenceList[4] = f433_CENTER_MIDFIELD;
    PreferenceList[5] = (LocationSide == RIGHT) ? f433_RIGHT_WING : f433_LEFT_WING;
    PreferenceList[6] = (LocationSide == RIGHT) ? f433_LEFT_WING : f433_RIGHT_WING;
    break;
  }	

  for (int i = NumOptions; i<TEAM_SIZE; i++)
    PreferenceList[i] = position;  /* Pad list with own number, meaning no receiver */
}


/************************ 4-4-2 Formation ********************************/

#define f442_GOALIE_BUF      2
#define f442_POSITION_BUF    3

#define f442_GOALTENDERX  (-52)
#define f442_SWEEPERX     (-43)
#define f442_DEFENDERX    (-35)
#define f442_MIDFIELDERX     0
#define f442_FORWARDX       30
#define f442_LEFTY          25
#define f442_INSIDELEFTY    15
#define f442_CENTERY         0
#define f442_INSIDERIGHTY (-f442_INSIDELEFTY)
#define f442_RIGHTY       (-f442_LEFTY)

#define f442_HOMEXRANGE f442_FORWARDX
#define f442_HOMEYRANGE f442_LEFTY
#define f442_MAXXRANGE  (f442_FORWARDX*2)
#define f442_MAXYRANGE  (((Y0+5) - f442_LEFTY)*2)
#define f442_INSIDE_MAXYRANGE (((Y0+5) - f442_INSIDELEFTY)*2)


void Initialize_442_Formation(Formation *formation){
  
  /* set position within goal stay, in penalty box      */
  formation->SetPosition(f442_GOALIE, f442_GOALTENDERX, f442_CENTERY, f442_GOALIE_BUF, 
			 (X0- (-f442_GOALTENDERX))*2, GOAL_WIDTH,
			 /*(-f442_GOALTENDERX - PA_X)*2, PA_Y*2,*/
			 (X0- (-f442_GOALTENDERX))*2, GOAL_WIDTH,
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, FALSE);  

  /* set position inside penalty box, don't go more than half-way to mid  */
  formation->SetPosition(f442_SWEEPER, f442_SWEEPERX, f442_CENTERY, f442_POSITION_BUF, 
			 (X0- (-f442_SWEEPERX))*2, PA_Y*2,        
			 -f442_SWEEPERX - X0/2, (Y0+5)*2,
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, TRUE);           

  formation->SetPosition(f442_RIGHT_DEFENSE, f442_DEFENDERX, f442_RIGHTY, f442_POSITION_BUF, 
			 f442_HOMEXRANGE,f442_HOMEYRANGE,
			 f442_MAXXRANGE ,f442_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f442_CENTER_DEFENSE, f442_DEFENDERX+5, f442_CENTERY, f442_POSITION_BUF, 
			 f442_HOMEXRANGE,f442_HOMEYRANGE,
			 f442_MAXXRANGE ,f442_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f442_LEFT_DEFENSE, f442_DEFENDERX, f442_LEFTY, f442_POSITION_BUF, 
			 f442_HOMEXRANGE,f442_HOMEYRANGE,
			 f442_MAXXRANGE ,f442_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f442_RIGHT_MIDFIELD, f442_MIDFIELDERX, f442_RIGHTY, f442_POSITION_BUF, 
			 f442_HOMEXRANGE,f442_HOMEYRANGE,
			 f442_MAXXRANGE ,f442_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f442_DEFENSIVE_CENTER_MIDFIELD, f442_MIDFIELDERX-10, f442_CENTERY, f442_POSITION_BUF, 
			 f442_HOMEXRANGE,f442_HOMEYRANGE,
			 f442_MAXXRANGE ,f442_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f442_OFFENSIVE_CENTER_MIDFIELD, f442_MIDFIELDERX+10, f442_CENTERY, f442_POSITION_BUF, 
			 f442_HOMEXRANGE,f442_HOMEYRANGE,
			 f442_MAXXRANGE ,f442_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f442_LEFT_MIDFIELD, f442_MIDFIELDERX, f442_LEFTY, f442_POSITION_BUF, 
			 f442_HOMEXRANGE,f442_HOMEYRANGE,
			 f442_MAXXRANGE ,f442_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);
  
  formation->SetPosition(f442_RIGHT_WING, f442_FORWARDX, f442_INSIDERIGHTY, f442_POSITION_BUF, 
			 f442_HOMEXRANGE,f442_HOMEYRANGE,
			 f442_MAXXRANGE ,f442_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f442_LEFT_WING, f442_FORWARDX, f442_INSIDELEFTY, f442_POSITION_BUF, 
			 f442_HOMEXRANGE,f442_HOMEYRANGE,
			 f442_MAXXRANGE ,f442_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  /* Candidate receivers will be chosen on the fly here */
  int dummylist[TEAM_SIZE];
  for (int i=0; i<TEAM_SIZE; i++)
    formation->SetCandidateReceivers(i,0,dummylist);

  formation->AllocateUnits(6);

  /* Positions listed in order of importance, captain indicated as index into list */

  int defenders[5] = {f442_GOALIE, f442_SWEEPER,
		      f442_CENTER_DEFENSE, f442_RIGHT_DEFENSE, f442_LEFT_DEFENSE};
  formation->GetUnit(DEFENSIVE_UNIT)->InitializeUnit(DEFENSIVE_UNIT, 5, defenders, 0);

  int midfielders[4] = {f442_DEFENSIVE_CENTER_MIDFIELD, f442_OFFENSIVE_CENTER_MIDFIELD,
			f442_RIGHT_MIDFIELD, f442_LEFT_MIDFIELD};
  formation->GetUnit(MIDFIELD_UNIT)->InitializeUnit(MIDFIELD_UNIT, 4, midfielders, 0);

  int forwards[2] = {f442_RIGHT_WING, f442_LEFT_WING};
  formation->GetUnit(FORWARD_UNIT)->InitializeUnit(FORWARD_UNIT, 2, forwards, 0);

  int rights[3] = {f442_RIGHT_DEFENSE, f442_RIGHT_MIDFIELD, f442_RIGHT_WING};
  formation->GetUnit(RIGHT_UNIT)->InitializeUnit(RIGHT_UNIT, 3, rights, 1);

  int centers[5] = {f442_GOALIE, f442_SWEEPER, f442_CENTER_DEFENSE, 
		    f442_DEFENSIVE_CENTER_MIDFIELD, f442_OFFENSIVE_CENTER_MIDFIELD};
  formation->GetUnit(CENTER_UNIT)->InitializeUnit(CENTER_UNIT, 5, centers, 4);

  int lefts[3] = {f442_LEFT_DEFENSE, f442_LEFT_MIDFIELD, f442_LEFT_WING};
  formation->GetUnit(LEFT_UNIT)->InitializeUnit(LEFT_UNIT, 3, lefts, 1);
}


int *Build_442_ReceiverList(int *PreferenceList, int position){

  int NumOptions=0;
  
  int Side = Mem->GetMyPositionSide();
  int LocationSide = Mem->GetMyLocationSide();

  switch ( position ){
  case f442_LEFT_WING:
  case f442_RIGHT_WING:
    NumOptions = (int_random(3) ? 1 : 0); /* shoot 1 in 3 times */
    PreferenceList[0] =  (Side == RIGHT) ? f442_LEFT_WING : f442_RIGHT_WING;
    break;
  case f442_OFFENSIVE_CENTER_MIDFIELD:
    NumOptions = 2;
    PreferenceList[0] = (LocationSide == RIGHT) ? f442_RIGHT_WING : f442_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f442_LEFT_WING : f442_RIGHT_WING;
    break;
  case f442_DEFENSIVE_CENTER_MIDFIELD:
    NumOptions = 5;
    PreferenceList[0] = (LocationSide == RIGHT) ? f442_RIGHT_WING : f442_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f442_LEFT_WING : f442_RIGHT_WING;
    PreferenceList[2] = f442_OFFENSIVE_CENTER_MIDFIELD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f442_RIGHT_MIDFIELD :f442_LEFT_MIDFIELD;
    PreferenceList[4] = (LocationSide == RIGHT) ? f442_LEFT_MIDFIELD :f442_RIGHT_MIDFIELD;
    break;
  case f442_RIGHT_MIDFIELD:
  case f442_LEFT_MIDFIELD:
    NumOptions = 4;
    PreferenceList[0] = (LocationSide == RIGHT) ? f442_RIGHT_WING : f442_LEFT_WING;
    PreferenceList[1] = f442_OFFENSIVE_CENTER_MIDFIELD;
    PreferenceList[2] = (LocationSide == RIGHT) ? f442_LEFT_WING : f442_RIGHT_WING;
    PreferenceList[3] = f442_DEFENSIVE_CENTER_MIDFIELD;
    break;
  case f442_CENTER_DEFENSE:    
    NumOptions = 4;
    PreferenceList[0] = (LocationSide == RIGHT) ? f442_RIGHT_MIDFIELD : f442_LEFT_MIDFIELD;
    PreferenceList[1] = (LocationSide == RIGHT) ? f442_LEFT_MIDFIELD : f442_RIGHT_MIDFIELD;
    PreferenceList[2] = f442_OFFENSIVE_CENTER_MIDFIELD;
    PreferenceList[3] = f442_DEFENSIVE_CENTER_MIDFIELD;
    break;
  case f442_RIGHT_DEFENSE:
  case f442_LEFT_DEFENSE:
    NumOptions = 4;
    PreferenceList[0] = (LocationSide == RIGHT) ? f442_RIGHT_MIDFIELD : f442_LEFT_MIDFIELD;
    PreferenceList[1] = (LocationSide == RIGHT) ? f442_RIGHT_WING : f442_LEFT_WING;
    PreferenceList[2] = f442_OFFENSIVE_CENTER_MIDFIELD;
    PreferenceList[3] = f442_DEFENSIVE_CENTER_MIDFIELD;
    break;
  case f442_SWEEPER:
  case f442_GOALIE:
    NumOptions = 8;
    PreferenceList[0] = (LocationSide == RIGHT) ? f442_RIGHT_DEFENSE : f442_LEFT_DEFENSE;
    PreferenceList[1] = (LocationSide == RIGHT) ? f442_LEFT_DEFENSE : f442_RIGHT_DEFENSE;
    PreferenceList[2] = (LocationSide == RIGHT) ? f442_RIGHT_MIDFIELD : f442_LEFT_MIDFIELD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f442_LEFT_MIDFIELD : f442_RIGHT_MIDFIELD;
    PreferenceList[4] = f442_OFFENSIVE_CENTER_MIDFIELD;
    PreferenceList[5] = f442_DEFENSIVE_CENTER_MIDFIELD;
    PreferenceList[6] = (LocationSide == RIGHT) ? f442_RIGHT_WING : f442_LEFT_WING;
    PreferenceList[7] = (LocationSide == RIGHT) ? f442_LEFT_WING : f442_RIGHT_WING;
    break;
  }	

  for (int i = NumOptions; i<TEAM_SIZE; i++)
    PreferenceList[i] = position;  /* Pad list with own number, meaning no receiver */
}


/************************ 3-5-2 Formation ********************************/

#define f352_GOALIE_BUF      2
#define f352_POSITION_BUF    3

#define f352_GOALTENDERX  (-52)
#define f352_SWEEPERX     (-43)
#define f352_DEFENDERX    (-35)
#define f352_MIDFIELDERX     0
#define f352_FORWARDX       30
#define f352_LEFTY          30 /*25*/
#define f352_INSIDELEFTY    15
#define f352_CENTERY         0
#define f352_INSIDERIGHTY (-f352_INSIDELEFTY)
#define f352_RIGHTY       (-f352_LEFTY)

#define f352_HOMEXRANGE f352_FORWARDX
#define f352_HOMEYRANGE f352_LEFTY
#define f352_MAXXRANGE  (f352_FORWARDX*2)
#define f352_MAXYRANGE  (((Y0+5) - f352_LEFTY)*2)
#define f352_INSIDE_MAXYRANGE (((Y0+5) - f352_INSIDELEFTY)*2)

void Initialize_352_Formation(Formation *formation){
  
  /* set position within goal stay, in penalty box      */
  formation->SetPosition(f352_GOALIE, f352_GOALTENDERX, f352_CENTERY, f352_GOALIE_BUF, 
			 (X0- (-f352_GOALTENDERX))*2, GOAL_WIDTH, 
			 /*(-f352_GOALTENDERX - PA_X)*2, PA_Y*2, */
			 (X0- (-f352_GOALTENDERX))*2, GOAL_WIDTH, 
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, FALSE);  

  /* set position inside penalty box, don't go more than half-way to mid  */
  formation->SetPosition(f352_SWEEPER, f352_SWEEPERX, f352_CENTERY, f352_POSITION_BUF, 
			 (X0- (-f352_SWEEPERX))*2, PA_Y*2,         
			 -f352_SWEEPERX - X0/2, (Y0+5)*2,
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, TRUE);  

  formation->SetPosition(f352_RIGHT_DEFENSE, f352_DEFENDERX, f352_INSIDERIGHTY, f352_POSITION_BUF, 
			 f352_HOMEXRANGE,f352_HOMEYRANGE,
			 f352_MAXXRANGE ,f352_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f352_LEFT_DEFENSE, f352_DEFENDERX, f352_INSIDELEFTY, f352_POSITION_BUF, 
			 f352_HOMEXRANGE,f352_HOMEYRANGE,
			 f352_MAXXRANGE ,f352_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f352_RIGHT_OUTSIDE_MIDFIELD, f352_MIDFIELDERX, f352_RIGHTY, f352_POSITION_BUF, 
			 f352_HOMEXRANGE,f352_HOMEYRANGE,
			 (X0+5)*2,f352_MAXYRANGE, /* can go whole length of field */
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE); 

  formation->SetPosition(f352_RIGHT_INSIDE_MIDFIELD, f352_MIDFIELDERX-10, f352_INSIDERIGHTY, f352_POSITION_BUF, 
			 f352_HOMEXRANGE,f352_HOMEYRANGE,
			 f352_MAXXRANGE ,f352_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f352_CENTER_MIDFIELD, f352_MIDFIELDERX+9, f352_CENTERY, f352_POSITION_BUF, 
			 f352_HOMEXRANGE,f352_HOMEYRANGE,
			 f352_MAXXRANGE ,f352_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f352_LEFT_INSIDE_MIDFIELD, f352_MIDFIELDERX-10, f352_INSIDELEFTY, f352_POSITION_BUF, 
			 f352_HOMEXRANGE,f352_HOMEYRANGE,
			 f352_MAXXRANGE ,f352_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f352_LEFT_OUTSIDE_MIDFIELD, f352_MIDFIELDERX, f352_LEFTY, f352_POSITION_BUF, 
			 f352_HOMEXRANGE,f352_HOMEYRANGE,
			 (X0+5)*2, f352_MAXYRANGE, /* can go whole length of field */
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE); 
  
  formation->SetPosition(f352_RIGHT_WING, f352_FORWARDX, f352_INSIDERIGHTY, f352_POSITION_BUF, 
			 f352_HOMEXRANGE,f352_HOMEYRANGE,
			 f352_MAXXRANGE ,f352_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f352_LEFT_WING, f352_FORWARDX, f352_INSIDELEFTY, f352_POSITION_BUF, 
			 f352_HOMEXRANGE,f352_HOMEYRANGE,
			 f352_MAXXRANGE ,f352_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  /* Candidate receivers will be chosen on the fly here */
  int dummylist[TEAM_SIZE];
  for (int i=0; i<TEAM_SIZE; i++)
    formation->SetCandidateReceivers(i,0,dummylist);

  formation->AllocateUnits(6);

  int defenders[4] = {f352_GOALIE, f352_SWEEPER, f352_RIGHT_DEFENSE, f352_LEFT_DEFENSE};
  formation->GetUnit(DEFENSIVE_UNIT)->InitializeUnit(DEFENSIVE_UNIT, 4, defenders, 0);

  int midfielders[5] = {f352_CENTER_MIDFIELD, 
			f352_RIGHT_OUTSIDE_MIDFIELD, f352_LEFT_OUTSIDE_MIDFIELD,
			f352_RIGHT_INSIDE_MIDFIELD, f352_LEFT_INSIDE_MIDFIELD};
  formation->GetUnit(MIDFIELD_UNIT)->InitializeUnit(MIDFIELD_UNIT, 5, midfielders, 0);

  int forwards[2] = {f352_RIGHT_WING, f352_LEFT_WING};
  formation->GetUnit(FORWARD_UNIT)->InitializeUnit(FORWARD_UNIT, 2, forwards, 0);

  int rights[3] = {f352_RIGHT_DEFENSE, f352_RIGHT_OUTSIDE_MIDFIELD, f352_RIGHT_WING};
  formation->GetUnit(RIGHT_UNIT)->InitializeUnit(RIGHT_UNIT, 3, rights, 1);

  int centers[5] = {f352_GOALIE, f352_SWEEPER, 
		    f352_CENTER_MIDFIELD, f352_RIGHT_INSIDE_MIDFIELD, 
		    f352_LEFT_INSIDE_MIDFIELD};
  formation->GetUnit(CENTER_UNIT)->InitializeUnit(CENTER_UNIT, 5, centers, 2);

  int lefts[3] = {f352_LEFT_DEFENSE, f352_LEFT_OUTSIDE_MIDFIELD, f352_LEFT_WING};
  formation->GetUnit(LEFT_UNIT)->InitializeUnit(LEFT_UNIT, 3, lefts, 1);
}


int *Build_352_ReceiverList(int *PreferenceList, int position){

  int NumOptions=0;
  
  int Side = Mem->GetMyPositionSide();
  int LocationSide = Mem->GetMyLocationSide();

  switch ( position ){
  case f352_LEFT_WING:
    NumOptions = (int_random(3) ? 1 : 0); /* shoot 1 in 3 times */
    PreferenceList[0] = f352_RIGHT_WING; 
    break;
  case f352_RIGHT_WING:
    NumOptions = (int_random(3) ? 1 : 0); /* shoot 1 in 3 times */
    PreferenceList[0] = f352_LEFT_WING; 
    break;
  case f352_CENTER_MIDFIELD:
    NumOptions = 4;
    PreferenceList[0] = (LocationSide == RIGHT) ? f352_RIGHT_WING : f352_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f352_LEFT_WING : f352_RIGHT_WING;
    PreferenceList[2] = (LocationSide == RIGHT) ? f352_RIGHT_OUTSIDE_MIDFIELD : 
      f352_LEFT_OUTSIDE_MIDFIELD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f352_LEFT_OUTSIDE_MIDFIELD : 
      f352_RIGHT_OUTSIDE_MIDFIELD;
    break;
  case f352_RIGHT_OUTSIDE_MIDFIELD:
  case f352_LEFT_OUTSIDE_MIDFIELD:
    NumOptions = 3;
    PreferenceList[0] = (LocationSide == RIGHT) ? f352_RIGHT_WING : f352_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f352_LEFT_WING : f352_RIGHT_WING; 
    PreferenceList[2] = f352_CENTER_MIDFIELD;
    break;
  case f352_RIGHT_INSIDE_MIDFIELD:
  case f352_LEFT_INSIDE_MIDFIELD:
    NumOptions = 6;
    PreferenceList[0] = (LocationSide == RIGHT) ? f352_RIGHT_WING : f352_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f352_RIGHT_OUTSIDE_MIDFIELD : 
      f352_LEFT_OUTSIDE_MIDFIELD;
    PreferenceList[2] = f352_CENTER_MIDFIELD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f352_LEFT_WING : f352_RIGHT_WING;
    PreferenceList[4] = (LocationSide == RIGHT) ? f352_LEFT_OUTSIDE_MIDFIELD : 
      f352_RIGHT_OUTSIDE_MIDFIELD;
    PreferenceList[5] = (Side == RIGHT) ? f352_LEFT_INSIDE_MIDFIELD : 
      f352_RIGHT_INSIDE_MIDFIELD;
    break;
  case f352_RIGHT_DEFENSE:
  case f352_LEFT_DEFENSE:
  case f352_SWEEPER:
  case f352_GOALIE:
    NumOptions = 7;
    PreferenceList[0] = (LocationSide == RIGHT) ? f352_RIGHT_OUTSIDE_MIDFIELD : 
      f352_LEFT_OUTSIDE_MIDFIELD;
    PreferenceList[1] = (LocationSide == RIGHT) ? f352_RIGHT_INSIDE_MIDFIELD : 
      f352_LEFT_INSIDE_MIDFIELD;
    PreferenceList[2] = (LocationSide == RIGHT) ? f352_LEFT_OUTSIDE_MIDFIELD : 
      f352_RIGHT_OUTSIDE_MIDFIELD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f352_LEFT_INSIDE_MIDFIELD : 
      f352_RIGHT_INSIDE_MIDFIELD;
    PreferenceList[4] = f352_CENTER_MIDFIELD;
    PreferenceList[5] = (LocationSide == RIGHT) ? f352_RIGHT_WING : f352_LEFT_WING;
    PreferenceList[6] = (LocationSide == RIGHT) ? f352_LEFT_WING : f352_RIGHT_WING;
    break;
  }	

  for (int i = NumOptions; i<TEAM_SIZE; i++)
    PreferenceList[i] = position;  /* Pad list with own number, meaning no receiver */
}

/************************ 7-2 Formation ********************************/

#define f72_GOALIE_BUF      2
#define f72_POSITION_BUF    3

#define f72_GOALTENDERX  (-52)
#define f72_DEFENDERX    (-43)
#define f72_MIDFIELDERX  (-20)
#define f72_FORWARDX       17   /* > ROUGH_FORWARDX/2 so treated as a forward */
#define f72_LEFTY          25
#define f72_INSIDELEFTY    15
#define f72_CENTERY         0
#define f72_INSIDERIGHTY (-f72_INSIDELEFTY)
#define f72_RIGHTY       (-f72_LEFTY)

#define f72_HOMEXRANGE 15
#define f72_HOMEYRANGE f72_INSIDELEFTY
#define f72_MAXXRANGE  30
#define f72_MAXYRANGE  (((Y0+5) - f72_LEFTY)*2)
#define f72_INSIDE_MAXYRANGE (((Y0+5) - f72_INSIDELEFTY)*2)

void Initialize_72_Formation(Formation *formation){

  /* set position within goal stay, in penalty box      */
  formation->SetPosition(f72_RIGHT_GOALIE, f72_GOALTENDERX, f72_CENTERY-(GOAL_WIDTH/4), f72_GOALIE_BUF, 
			 (X0- (-f72_GOALTENDERX))*2, GOAL_WIDTH/2, 
			 /*(-f72_GOALTENDERX - PA_X)*2, PA_Y*2, */
			 (X0- (-f72_GOALTENDERX))*2, GOAL_WIDTH, 
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, FALSE);  

  formation->SetPosition(f72_LEFT_GOALIE, f72_GOALTENDERX, f72_CENTERY+(GOAL_WIDTH/4), f72_GOALIE_BUF, 
			 (X0- (-f72_GOALTENDERX))*2, GOAL_WIDTH/2, 
			 /*(-f72_GOALTENDERX - PA_X)*2, PA_Y*2, */
			 (X0- (-f72_GOALTENDERX))*2, GOAL_WIDTH, 
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, FALSE);  

  formation->SetPosition(f72_RIGHT_DEFENSE, f72_DEFENDERX, f72_INSIDERIGHTY, f72_POSITION_BUF, 
			 f72_HOMEXRANGE,f72_HOMEYRANGE,
			 f72_MAXXRANGE ,f72_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f72_CENTER_DEFENSE, f72_DEFENDERX+5, f72_CENTERY, f72_POSITION_BUF, 
			 f72_HOMEXRANGE,f72_HOMEYRANGE,
			 f72_MAXXRANGE ,f72_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f72_LEFT_DEFENSE, f72_DEFENDERX, f72_INSIDELEFTY, f72_POSITION_BUF, 
			 f72_HOMEXRANGE,f72_HOMEYRANGE,
			 f72_MAXXRANGE ,f72_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f72_RIGHT_OUTSIDE_MIDFIELD, f72_MIDFIELDERX, f72_RIGHTY, f72_POSITION_BUF, 
			 f72_HOMEXRANGE,f72_HOMEYRANGE,
			 f72_MAXXRANGE ,f72_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE); 

  formation->SetPosition(f72_RIGHT_INSIDE_MIDFIELD, f72_MIDFIELDERX-5, f72_INSIDERIGHTY+5, f72_POSITION_BUF, 
			 f72_HOMEXRANGE,f72_HOMEYRANGE,
			 f72_MAXXRANGE ,f72_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f72_LEFT_INSIDE_MIDFIELD, f72_MIDFIELDERX-5, f72_INSIDELEFTY-5, f72_POSITION_BUF, 
			 f72_HOMEXRANGE,f72_HOMEYRANGE,
			 f72_MAXXRANGE ,f72_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f72_LEFT_OUTSIDE_MIDFIELD, f72_MIDFIELDERX, f72_LEFTY, f72_POSITION_BUF, 
			 f72_HOMEXRANGE,f72_HOMEYRANGE,
			 f72_MAXXRANGE ,f72_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE); 
  
  formation->SetPosition(f72_RIGHT_WING, f72_FORWARDX, f72_INSIDERIGHTY, f72_POSITION_BUF, 
			 f72_HOMEXRANGE,f72_HOMEYRANGE,
			 f72_MAXXRANGE ,f72_INSIDE_MAXYRANGE, 
			 60, MININACTIVEDIST, TRUE);   /* Need to go to the corners */

  formation->SetPosition(f72_LEFT_WING, f72_FORWARDX, f72_INSIDELEFTY, f72_POSITION_BUF, 
			 f72_HOMEXRANGE,f72_HOMEYRANGE,
			 f72_MAXXRANGE ,f72_INSIDE_MAXYRANGE, 
			 60, MININACTIVEDIST, TRUE);   /* Need to go to the corners */

  /* Candidate receivers will be chosen on the fly here */
  int dummylist[TEAM_SIZE];
  for (int i=0; i<TEAM_SIZE; i++)
    formation->SetCandidateReceivers(i,0,dummylist);

  formation->AllocateUnits(6);

  int defenders[9] = {f72_RIGHT_GOALIE, f72_LEFT_GOALIE, f72_RIGHT_DEFENSE, 
		      f72_CENTER_DEFENSE, f72_LEFT_DEFENSE, 
		      f72_RIGHT_INSIDE_MIDFIELD, f72_LEFT_INSIDE_MIDFIELD,
		      f72_RIGHT_OUTSIDE_MIDFIELD, f72_LEFT_OUTSIDE_MIDFIELD};
  formation->GetUnit(DEFENSIVE_UNIT)->InitializeUnit(DEFENSIVE_UNIT, 9, defenders, 0);

  int midfielders[4] = {f72_RIGHT_INSIDE_MIDFIELD, f72_LEFT_INSIDE_MIDFIELD,
			f72_RIGHT_OUTSIDE_MIDFIELD, f72_LEFT_OUTSIDE_MIDFIELD};
  formation->GetUnit(MIDFIELD_UNIT)->InitializeUnit(MIDFIELD_UNIT, 4, midfielders, 0);

  int forwards[2] = {f72_RIGHT_WING, f72_LEFT_WING};
  formation->GetUnit(FORWARD_UNIT)->InitializeUnit(FORWARD_UNIT, 2, forwards, 0);

  int rights[3] = {f72_RIGHT_DEFENSE, f72_RIGHT_OUTSIDE_MIDFIELD, f72_RIGHT_WING};
  formation->GetUnit(RIGHT_UNIT)->InitializeUnit(RIGHT_UNIT, 3, rights, 1);

  int centers[5] = {f72_RIGHT_GOALIE, f72_LEFT_GOALIE, 
		    f72_CENTER_DEFENSE,
		    f72_RIGHT_INSIDE_MIDFIELD, f72_LEFT_INSIDE_MIDFIELD};
  formation->GetUnit(CENTER_UNIT)->InitializeUnit(CENTER_UNIT, 5, centers, 3);

  int lefts[3] = {f72_LEFT_DEFENSE, f72_LEFT_OUTSIDE_MIDFIELD, f72_LEFT_WING};
  formation->GetUnit(LEFT_UNIT)->InitializeUnit(LEFT_UNIT, 3, lefts, 1);
}

int *Build_72_ReceiverList(int *PreferenceList, int position){

  int NumOptions=0;
  
  int LocationSide = Mem->GetMyLocationSide();

  switch ( position ){
  case f72_LEFT_WING:
  case f72_RIGHT_WING:
    break;  /* shoot */
  case f72_RIGHT_OUTSIDE_MIDFIELD:
  case f72_LEFT_OUTSIDE_MIDFIELD:
  case f72_RIGHT_INSIDE_MIDFIELD:
  case f72_LEFT_INSIDE_MIDFIELD:
  case f72_RIGHT_DEFENSE:
  case f72_CENTER_DEFENSE:
  case f72_LEFT_DEFENSE:
    NumOptions = 2;
    PreferenceList[0] = (LocationSide == RIGHT) ? f72_RIGHT_WING : f72_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f72_LEFT_WING : f72_RIGHT_WING;
    break;
  case f72_RIGHT_GOALIE:
  case f72_LEFT_GOALIE:
    break; /* clear */
  }	

  for (int i = NumOptions; i<TEAM_SIZE; i++)
    PreferenceList[i] = position;  /* Pad list with own number, meaning no receiver */
}

/************************ 3-3-4 Formation ********************************/

#define f334_GOALIE_BUF      2
#define f334_POSITION_BUF    3

#define f334_GOALTENDERX  (-52)
#define f334_SWEEPERX     (-43)
#define f334_DEFENDERX    (-35)
#define f334_MIDFIELDERX     0
#define f334_FORWARDX       30
#define f334_LEFTY          25
#define f334_INSIDELEFTY    15
#define f334_CENTERY         0
#define f334_INSIDERIGHTY (-f334_INSIDELEFTY)
#define f334_RIGHTY       (-f334_LEFTY)

#define f334_HOMEXRANGE f334_FORWARDX
#define f334_HOMEYRANGE f334_LEFTY
#define f334_MAXXRANGE  (f334_FORWARDX*2)
#define f334_MAXYRANGE  (((Y0+5) - f334_LEFTY)*2)
#define f334_INSIDE_MAXYRANGE (((Y0+5) - f334_INSIDELEFTY)*2)

void Initialize_334_Formation(Formation *formation){
  
  /* set position within goal stay, in penalty box      */
  formation->SetPosition(f334_GOALIE, f334_GOALTENDERX, f334_CENTERY, f334_GOALIE_BUF, 
			 (X0- (-f334_GOALTENDERX))*2, GOAL_WIDTH, 
			 /*(-f334_GOALTENDERX - PA_X)*2, PA_Y*2, */
			 (X0- (-f334_GOALTENDERX))*2, GOAL_WIDTH, 
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, FALSE);  

  /* set position inside penalty box, don't go more than half-way to mid  */
  formation->SetPosition(f334_SWEEPER, f334_SWEEPERX, f334_CENTERY, f334_POSITION_BUF, 
			 (X0- (-f334_SWEEPERX))*2, PA_Y*2,         
			 -f334_SWEEPERX - X0/2, (Y0+5)*2,
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, TRUE);  

  formation->SetPosition(f334_RIGHT_DEFENSE, f334_DEFENDERX, f334_INSIDERIGHTY, f334_POSITION_BUF, 
			 f334_HOMEXRANGE,f334_HOMEYRANGE,
			 f334_MAXXRANGE ,f334_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f334_LEFT_DEFENSE, f334_DEFENDERX, f334_INSIDELEFTY, f334_POSITION_BUF, 
			 f334_HOMEXRANGE,f334_HOMEYRANGE,
			 f334_MAXXRANGE ,f334_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f334_RIGHT_MIDFIELD, f334_MIDFIELDERX, f334_RIGHTY, f334_POSITION_BUF, 
			 f334_HOMEXRANGE,f334_HOMEYRANGE,
			 f334_MAXXRANGE ,f334_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f334_CENTER_MIDFIELD, f334_MIDFIELDERX, f334_CENTERY, f334_POSITION_BUF, 
			 f334_HOMEXRANGE,f334_HOMEYRANGE,
			 f334_MAXXRANGE ,f334_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f334_LEFT_MIDFIELD, f334_MIDFIELDERX, f334_LEFTY, f334_POSITION_BUF, 
			 f334_HOMEXRANGE,f334_HOMEYRANGE,
			 f334_MAXXRANGE ,f334_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);
  
  formation->SetPosition(f334_RIGHT_WING, f334_FORWARDX+5, f334_RIGHTY, f334_POSITION_BUF, 
			 f334_HOMEXRANGE,f334_HOMEYRANGE,
			 f334_MAXXRANGE ,f334_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f334_RIGHT_INSIDE_FORWARD, f334_FORWARDX, f334_CENTERY-10, f334_POSITION_BUF, 
			 f334_HOMEXRANGE,f334_HOMEYRANGE,
			 f334_MAXXRANGE ,f334_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f334_LEFT_INSIDE_FORWARD, f334_FORWARDX, f334_CENTERY+10, f334_POSITION_BUF, 
			 f334_HOMEXRANGE,f334_HOMEYRANGE,
			 f334_MAXXRANGE ,f334_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f334_LEFT_WING, f334_FORWARDX+5, f334_LEFTY, f334_POSITION_BUF, 
			 f334_HOMEXRANGE,f334_HOMEYRANGE,
			 f334_MAXXRANGE ,f334_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);


  /* Candidate receivers will be chosen on the fly here */
  int dummylist[TEAM_SIZE];
  for (int i=0; i<TEAM_SIZE; i++)
    formation->SetCandidateReceivers(i,0,dummylist);

  formation->AllocateUnits(6);

  int defenders[4] = {f334_GOALIE, f334_SWEEPER, f334_RIGHT_DEFENSE, f334_LEFT_DEFENSE};
  formation->GetUnit(DEFENSIVE_UNIT)->InitializeUnit(DEFENSIVE_UNIT, 4, defenders, 0);

  int midfielders[3] = {f334_CENTER_MIDFIELD, 
			f334_RIGHT_MIDFIELD, f334_LEFT_MIDFIELD};
  formation->GetUnit(MIDFIELD_UNIT)->InitializeUnit(MIDFIELD_UNIT, 3, midfielders, 0);

  int forwards[4] = {f334_RIGHT_INSIDE_FORWARD, f334_LEFT_INSIDE_FORWARD,
		     f334_RIGHT_WING, f334_LEFT_WING};
  formation->GetUnit(FORWARD_UNIT)->InitializeUnit(FORWARD_UNIT, 4, forwards, 0);

  int rights[3] = {f334_RIGHT_DEFENSE, f334_RIGHT_MIDFIELD, f334_RIGHT_WING};
  formation->GetUnit(RIGHT_UNIT)->InitializeUnit(RIGHT_UNIT, 3, rights, 1);

  int centers[5] = {f334_GOALIE, f334_SWEEPER, 
		    f334_CENTER_MIDFIELD, f334_RIGHT_INSIDE_FORWARD, 
		    f334_LEFT_INSIDE_FORWARD};
  formation->GetUnit(CENTER_UNIT)->InitializeUnit(CENTER_UNIT, 5, centers, 2);

  int lefts[3] = {f334_LEFT_DEFENSE, f334_LEFT_MIDFIELD, f334_LEFT_WING};
  formation->GetUnit(LEFT_UNIT)->InitializeUnit(LEFT_UNIT, 3, lefts, 1);
}


int *Build_334_ReceiverList(int *PreferenceList, int position){

  int NumOptions=0;
  
  int Side = Mem->GetMyPositionSide();
  int LocationSide = Mem->GetMyLocationSide();

  switch ( position ){
  case f334_LEFT_INSIDE_FORWARD:
  case f334_RIGHT_INSIDE_FORWARD:
      break; /* shoot */
  case f334_LEFT_WING:
    NumOptions = (int_random(3) ? 3 : 0); /* shoot 1 in 3 times */
    PreferenceList[0] = f334_LEFT_INSIDE_FORWARD; 
    PreferenceList[1] = f334_RIGHT_INSIDE_FORWARD; 
    PreferenceList[2] = f334_RIGHT_WING; 
    break;
  case f334_RIGHT_WING:
    NumOptions = (int_random(3) ? 3 : 0); /* shoot 1 in 3 times */
    PreferenceList[0] = f334_RIGHT_INSIDE_FORWARD; 
    PreferenceList[1] = f334_LEFT_INSIDE_FORWARD; 
    PreferenceList[2] = f334_LEFT_WING; 
    break;
  case f334_CENTER_MIDFIELD:
    NumOptions = 6;
    PreferenceList[0] = (LocationSide == RIGHT) ? f334_RIGHT_WING : f334_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f334_LEFT_WING : f334_RIGHT_WING;
    PreferenceList[2] = (LocationSide == RIGHT) ? f334_RIGHT_INSIDE_FORWARD : 
      f334_LEFT_INSIDE_FORWARD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f334_LEFT_INSIDE_FORWARD : 
      f334_RIGHT_INSIDE_FORWARD;
    PreferenceList[4] = (LocationSide == RIGHT) ? f334_RIGHT_MIDFIELD : 
      f334_LEFT_MIDFIELD;
    PreferenceList[5] = (LocationSide == RIGHT) ? f334_LEFT_MIDFIELD : 
      f334_RIGHT_MIDFIELD;
    break;
  case f334_RIGHT_MIDFIELD:
  case f334_LEFT_MIDFIELD:
    NumOptions = 5;
    PreferenceList[0] = (Side == RIGHT) ? f334_RIGHT_WING : f334_LEFT_WING;
    PreferenceList[1] = (Side == RIGHT) ? f334_RIGHT_INSIDE_FORWARD : 
      f334_LEFT_INSIDE_FORWARD;
    PreferenceList[2] = (Side == RIGHT) ? f334_LEFT_INSIDE_FORWARD :
      f334_RIGHT_INSIDE_FORWARD;
    PreferenceList[3] = f334_CENTER_MIDFIELD;
    PreferenceList[4] = (Side == RIGHT) ? f334_LEFT_WING : RIGHT_WING;
    break;
  case f334_RIGHT_DEFENSE:
  case f334_LEFT_DEFENSE:
  case f334_SWEEPER:
  case f334_GOALIE:
    NumOptions = 7;
    PreferenceList[0] = (LocationSide == RIGHT) ? f334_RIGHT_MIDFIELD : 
      f334_LEFT_MIDFIELD;
    PreferenceList[1] = (LocationSide == RIGHT) ? f334_RIGHT_WING : f334_LEFT_WING;
    PreferenceList[2] = (Side == RIGHT) ? f334_RIGHT_INSIDE_FORWARD : 
      f334_LEFT_INSIDE_FORWARD;
    PreferenceList[3] = f334_CENTER_MIDFIELD;
    PreferenceList[4] = (LocationSide == RIGHT) ? f334_LEFT_MIDFIELD : 
      f334_RIGHT_MIDFIELD;
    PreferenceList[5] = (Side == RIGHT) ? f334_LEFT_INSIDE_FORWARD :
      f334_RIGHT_INSIDE_FORWARD;
    PreferenceList[6] = (LocationSide == RIGHT) ? f334_LEFT_WING : f334_RIGHT_WING;
    break;
  }	

  for (int i = NumOptions; i<TEAM_SIZE; i++)
    PreferenceList[i] = position;  /* Pad list with own number, meaning no receiver */
}


/************************ 2-4-4 Formation ********************************/

#define f244_GOALIE_BUF      2
#define f244_POSITION_BUF    3

#define f244_GOALTENDERX  (-52)
#define f244_SWEEPERX     (-43)
#define f244_DEFENDERX    (-35)
#define f244_MIDFIELDERX     0
#define f244_FORWARDX       30
#define f244_LEFTY          25
#define f244_INSIDELEFTY    15
#define f244_CENTERY         0
#define f244_INSIDERIGHTY (-f244_INSIDELEFTY)
#define f244_RIGHTY       (-f244_LEFTY)

#define f244_HOMEXRANGE f244_FORWARDX
#define f244_HOMEYRANGE f244_LEFTY
#define f244_MAXXRANGE  (f244_FORWARDX*2)
#define f244_MAXYRANGE  (((Y0+5) - f244_LEFTY)*2)
#define f244_INSIDE_MAXYRANGE (((Y0+5) - f244_INSIDELEFTY)*2)

void Initialize_244_Formation(Formation *formation){
  
  /* set position within goal stay, in penalty box      */
  formation->SetPosition(f244_GOALIE, f244_GOALTENDERX, f244_CENTERY, f244_GOALIE_BUF, 
			 (X0- (-f244_GOALTENDERX))*2, GOAL_WIDTH, 
			 /*(-f244_GOALTENDERX - PA_X)*2, PA_Y*2, */
			 (X0- (-f244_GOALTENDERX))*2, GOAL_WIDTH, 
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, FALSE);  

  formation->SetPosition(f244_RIGHT_DEFENSE, f244_DEFENDERX, f244_INSIDERIGHTY, f244_POSITION_BUF, 
			 f244_HOMEXRANGE,f244_HOMEYRANGE,
			 f244_MAXXRANGE ,f244_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f244_LEFT_DEFENSE, f244_DEFENDERX, f244_INSIDELEFTY, f244_POSITION_BUF, 
			 f244_HOMEXRANGE,f244_HOMEYRANGE,
			 f244_MAXXRANGE ,f244_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f244_RIGHT_MIDFIELD, f244_MIDFIELDERX, f244_RIGHTY, f244_POSITION_BUF, 
			 f244_HOMEXRANGE,f244_HOMEYRANGE,
			 f244_MAXXRANGE ,f244_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f244_DEFENSIVE_CENTER_MIDFIELD, f244_MIDFIELDERX-10, f244_CENTERY, f244_POSITION_BUF, 
			 f244_HOMEXRANGE,f244_HOMEYRANGE,
			 f244_MAXXRANGE ,f244_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f244_OFFENSIVE_CENTER_MIDFIELD, f244_MIDFIELDERX+10, f244_CENTERY, f244_POSITION_BUF, 
			 f244_HOMEXRANGE,f244_HOMEYRANGE,
			 f244_MAXXRANGE ,f244_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f244_LEFT_MIDFIELD, f244_MIDFIELDERX, f244_LEFTY, f244_POSITION_BUF, 
			 f244_HOMEXRANGE,f244_HOMEYRANGE,
			 f244_MAXXRANGE ,f244_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);
  
  formation->SetPosition(f244_RIGHT_WING, f244_FORWARDX+5, f244_RIGHTY, f244_POSITION_BUF, 
			 f244_HOMEXRANGE,f244_HOMEYRANGE,
			 f244_MAXXRANGE ,f244_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f244_RIGHT_INSIDE_FORWARD, f244_FORWARDX, f244_CENTERY-10, f244_POSITION_BUF, 
			 f244_HOMEXRANGE,f244_HOMEYRANGE,
			 f244_MAXXRANGE ,f244_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f244_LEFT_INSIDE_FORWARD, f244_FORWARDX, f244_CENTERY+10, f244_POSITION_BUF, 
			 f244_HOMEXRANGE,f244_HOMEYRANGE,
			 f244_MAXXRANGE ,f244_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f244_LEFT_WING, f244_FORWARDX+5, f244_LEFTY, f244_POSITION_BUF, 
			 f244_HOMEXRANGE,f244_HOMEYRANGE,
			 f244_MAXXRANGE ,f244_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);


  /* Candidate receivers will be chosen on the fly here */
  int dummylist[TEAM_SIZE];
  for (int i=0; i<TEAM_SIZE; i++)
    formation->SetCandidateReceivers(i,0,dummylist);

  formation->AllocateUnits(6);

  int defenders[3] = {f244_GOALIE, f244_RIGHT_DEFENSE, f244_LEFT_DEFENSE};
  formation->GetUnit(DEFENSIVE_UNIT)->InitializeUnit(DEFENSIVE_UNIT, 3, defenders, 0);

  int midfielders[4] = {f244_DEFENSIVE_CENTER_MIDFIELD, f244_OFFENSIVE_CENTER_MIDFIELD,
			f244_RIGHT_MIDFIELD, f244_LEFT_MIDFIELD};
  formation->GetUnit(MIDFIELD_UNIT)->InitializeUnit(MIDFIELD_UNIT, 4, midfielders, 0);

  int forwards[4] = {f244_RIGHT_INSIDE_FORWARD, f244_LEFT_INSIDE_FORWARD,
		     f244_RIGHT_WING, f244_LEFT_WING};
  formation->GetUnit(FORWARD_UNIT)->InitializeUnit(FORWARD_UNIT, 4, forwards, 0);

  int rights[3] = {f244_RIGHT_DEFENSE, f244_RIGHT_MIDFIELD, f244_RIGHT_WING};
  formation->GetUnit(RIGHT_UNIT)->InitializeUnit(RIGHT_UNIT, 3, rights, 1);

  int centers[5] = {f244_GOALIE, 
		    f244_DEFENSIVE_CENTER_MIDFIELD, f244_OFFENSIVE_CENTER_MIDFIELD,
		    f244_RIGHT_INSIDE_FORWARD, f244_LEFT_INSIDE_FORWARD};
  formation->GetUnit(CENTER_UNIT)->InitializeUnit(CENTER_UNIT, 5, centers, 2);

  int lefts[3] = {f244_LEFT_DEFENSE, f244_LEFT_MIDFIELD, f244_LEFT_WING};
  formation->GetUnit(LEFT_UNIT)->InitializeUnit(LEFT_UNIT, 3, lefts, 1);
}


int *Build_244_ReceiverList(int *PreferenceList, int position){

  int NumOptions=0;
  
  int Side = Mem->GetMyPositionSide();
  int LocationSide = Mem->GetMyLocationSide();

  switch ( position ){
  case f244_LEFT_INSIDE_FORWARD:
  case f244_RIGHT_INSIDE_FORWARD:
      break; /* shoot */
  case f244_LEFT_WING:
    NumOptions = (int_random(3) ? 3 : 0); /* shoot 1 in 3 times */
    PreferenceList[0] = f244_LEFT_INSIDE_FORWARD; 
    PreferenceList[1] = f244_RIGHT_INSIDE_FORWARD; 
    PreferenceList[2] = f244_RIGHT_WING; 
    break;
  case f244_RIGHT_WING:
    NumOptions = (int_random(3) ? 3 : 0); /* shoot 1 in 3 times */
    PreferenceList[0] = f244_RIGHT_INSIDE_FORWARD; 
    PreferenceList[1] = f244_LEFT_INSIDE_FORWARD; 
    PreferenceList[2] = f244_LEFT_WING; 
    break;
  case f244_OFFENSIVE_CENTER_MIDFIELD:
    NumOptions = 4;
    PreferenceList[0] = (LocationSide == RIGHT) ? f244_RIGHT_WING : f244_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f244_LEFT_WING : f244_RIGHT_WING;
    PreferenceList[2] = (LocationSide == RIGHT) ? f244_RIGHT_INSIDE_FORWARD : 
      f244_LEFT_INSIDE_FORWARD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f244_LEFT_INSIDE_FORWARD : 
      f244_RIGHT_INSIDE_FORWARD;
    break;
  case f244_DEFENSIVE_CENTER_MIDFIELD:
    NumOptions = 7;
    PreferenceList[0] = (LocationSide == RIGHT) ? f244_RIGHT_WING : f244_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f244_LEFT_WING : f244_RIGHT_WING;
    PreferenceList[2] = (LocationSide == RIGHT) ? f244_RIGHT_INSIDE_FORWARD : 
      f244_LEFT_INSIDE_FORWARD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f244_LEFT_INSIDE_FORWARD : 
      f244_RIGHT_INSIDE_FORWARD;
    PreferenceList[4] = f244_OFFENSIVE_CENTER_MIDFIELD;
    PreferenceList[5] = (LocationSide == RIGHT) ? f244_RIGHT_MIDFIELD :f244_LEFT_MIDFIELD;
    PreferenceList[6] = (LocationSide == RIGHT) ? f244_LEFT_MIDFIELD :f244_RIGHT_MIDFIELD;
    break;
  case f244_RIGHT_MIDFIELD:
  case f244_LEFT_MIDFIELD:
    NumOptions = 5;
    PreferenceList[0] = (Side == RIGHT) ? f244_RIGHT_WING : f244_LEFT_WING;
    PreferenceList[1] = (Side == RIGHT) ? f244_RIGHT_INSIDE_FORWARD : 
      f244_LEFT_INSIDE_FORWARD;
    PreferenceList[2] = (Side == RIGHT) ? f244_LEFT_INSIDE_FORWARD :
      f244_RIGHT_INSIDE_FORWARD;
    PreferenceList[3] = f244_OFFENSIVE_CENTER_MIDFIELD;
    PreferenceList[4] = (Side == RIGHT) ? f244_LEFT_WING : RIGHT_WING;
    break;
  case f244_RIGHT_DEFENSE:
  case f244_LEFT_DEFENSE:
  case f244_GOALIE:
    NumOptions = 7;
    PreferenceList[0] = (LocationSide == RIGHT) ? f244_RIGHT_MIDFIELD : 
      f244_LEFT_MIDFIELD;
    PreferenceList[1] = (LocationSide == RIGHT) ? f244_RIGHT_WING : f244_LEFT_WING;
    PreferenceList[2] = (Side == RIGHT) ? f244_RIGHT_INSIDE_FORWARD : 
      f244_LEFT_INSIDE_FORWARD;
    PreferenceList[3] = f244_OFFENSIVE_CENTER_MIDFIELD;
    PreferenceList[4] = (LocationSide == RIGHT) ? f244_LEFT_MIDFIELD : 
      f244_RIGHT_MIDFIELD;
    PreferenceList[5] = (Side == RIGHT) ? f244_LEFT_INSIDE_FORWARD :
      f244_RIGHT_INSIDE_FORWARD;
    PreferenceList[6] = (LocationSide == RIGHT) ? f244_LEFT_WING : f244_RIGHT_WING;
    break;
  }	

  for (int i = NumOptions; i<TEAM_SIZE; i++)
    PreferenceList[i] = position;  /* Pad list with own number, meaning no receiver */
}

/************************ 4-3-2 Formation ********************************/

#define f432_GOALIE_BUF      2
#define f432_POSITION_BUF    3

#define f432_GOALTENDERX  (-52)
#define f432_SWEEPERX     (-43)
#define f432_DEFENDERX    (-35)
#define f432_MIDFIELDERX     0
#define f432_FORWARDX       30
#define f432_LEFTY          25
#define f432_INSIDELEFTY    15
#define f432_CENTERY         0
#define f432_INSIDERIGHTY (-f432_INSIDELEFTY)
#define f432_RIGHTY       (-f432_LEFTY)

#define f432_HOMEXRANGE f432_FORWARDX
#define f432_HOMEYRANGE f432_LEFTY
#define f432_MAXXRANGE  (f432_FORWARDX*2)
#define f432_MAXYRANGE  (((Y0+5) - f432_LEFTY)*2)
#define f432_INSIDE_MAXYRANGE (((Y0+5) - f432_INSIDELEFTY)*2)


void Initialize_432_Formation(Formation *formation){
  
  /* set position within goal stay, in penalty box      */
  formation->SetPosition(f432_RIGHT_GOALIE, f432_GOALTENDERX, f432_CENTERY-(GOAL_WIDTH/4), f432_GOALIE_BUF, 
			 (X0- (-f432_GOALTENDERX))*2, GOAL_WIDTH/2, 
			 /*(-f432_GOALTENDERX - PA_X)*2, PA_Y*2, */
			 (X0- (-f432_GOALTENDERX))*2, GOAL_WIDTH, 
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, FALSE);  

  formation->SetPosition(f432_LEFT_GOALIE, f432_GOALTENDERX, f432_CENTERY+(GOAL_WIDTH/4), f432_GOALIE_BUF, 
			 (X0- (-f432_GOALTENDERX))*2, GOAL_WIDTH/2, 
			 /*(-f432_GOALTENDERX - PA_X)*2, PA_Y*2, */
			 (X0- (-f432_GOALTENDERX))*2, GOAL_WIDTH, 
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, FALSE);  

  /* set position inside penalty box, don't go more than half-way to mid  */
  formation->SetPosition(f432_SWEEPER, f432_SWEEPERX, f432_CENTERY, f432_POSITION_BUF, 
			 (X0- (-f432_SWEEPERX))*2, PA_Y*2,        
			 -f432_SWEEPERX - X0/2, (Y0+5)*2,
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, TRUE);           

  formation->SetPosition(f432_RIGHT_DEFENSE, f432_DEFENDERX, f432_RIGHTY+13, f432_POSITION_BUF, 
			 f432_HOMEXRANGE,f432_HOMEYRANGE,
			 f432_MAXXRANGE ,f432_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f432_CENTER_DEFENSE, f432_DEFENDERX+5, f432_CENTERY, f432_POSITION_BUF, 
			 f432_HOMEXRANGE,f432_HOMEYRANGE,
			 f432_MAXXRANGE ,f432_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f432_LEFT_DEFENSE, f432_DEFENDERX, f432_LEFTY-13, f432_POSITION_BUF, 
			 f432_HOMEXRANGE,f432_HOMEYRANGE,
			 f432_MAXXRANGE ,f432_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f432_RIGHT_MIDFIELD, f432_MIDFIELDERX, f432_RIGHTY, f432_POSITION_BUF, 
			 f432_HOMEXRANGE,f432_HOMEYRANGE,
			 f432_MAXXRANGE ,f432_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f432_CENTER_MIDFIELD, f432_MIDFIELDERX, f432_CENTERY, f432_POSITION_BUF, 
			 f432_HOMEXRANGE,f432_HOMEYRANGE,
			 f432_MAXXRANGE ,f432_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f432_LEFT_MIDFIELD, f432_MIDFIELDERX, f432_LEFTY, f432_POSITION_BUF, 
			 f432_HOMEXRANGE,f432_HOMEYRANGE,
			 f432_MAXXRANGE ,f432_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);
  
  formation->SetPosition(f432_RIGHT_WING, f432_FORWARDX, f432_INSIDERIGHTY, f432_POSITION_BUF, 
			 f432_HOMEXRANGE,f432_HOMEYRANGE,
			 f432_MAXXRANGE ,f432_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(f432_LEFT_WING, f432_FORWARDX, f432_INSIDELEFTY, f432_POSITION_BUF, 
			 f432_HOMEXRANGE,f432_HOMEYRANGE,
			 f432_MAXXRANGE ,f432_INSIDE_MAXYRANGE, 
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  /* Candidate receivers will be chosen on the fly here */
  int dummylist[TEAM_SIZE];
  for (int i=0; i<TEAM_SIZE; i++)
    formation->SetCandidateReceivers(i,0,dummylist);

  formation->AllocateUnits(6);

  /* Positions listed in order of importance, captain indicated as index into list */

  int defenders[6] = {f432_RIGHT_GOALIE, f432_LEFT_GOALIE, f432_SWEEPER,
		      f432_CENTER_DEFENSE, f432_RIGHT_DEFENSE, f432_LEFT_DEFENSE};
  formation->GetUnit(DEFENSIVE_UNIT)->InitializeUnit(DEFENSIVE_UNIT, 6, defenders, 0);

  int midfielders[3] = {f432_CENTER_MIDFIELD, 
			f432_RIGHT_MIDFIELD, f432_LEFT_MIDFIELD};
  formation->GetUnit(MIDFIELD_UNIT)->InitializeUnit(MIDFIELD_UNIT, 3, midfielders, 0);

  int forwards[2] = {f432_RIGHT_WING, f432_LEFT_WING};
  formation->GetUnit(FORWARD_UNIT)->InitializeUnit(FORWARD_UNIT, 3, forwards, 0);

  int rights[3] = {f432_RIGHT_DEFENSE, f432_RIGHT_MIDFIELD, f432_RIGHT_WING};
  formation->GetUnit(RIGHT_UNIT)->InitializeUnit(RIGHT_UNIT, 3, rights, 1);

  int centers[5] = {f432_RIGHT_GOALIE, f432_LEFT_GOALIE, f432_SWEEPER, f432_CENTER_DEFENSE, 
		    f432_CENTER_MIDFIELD};
  formation->GetUnit(CENTER_UNIT)->InitializeUnit(CENTER_UNIT, 5, centers, 4);

  int lefts[3] = {f432_LEFT_DEFENSE, f432_LEFT_MIDFIELD, f432_LEFT_WING};
  formation->GetUnit(LEFT_UNIT)->InitializeUnit(LEFT_UNIT, 3, lefts, 1);
}


int *Build_432_ReceiverList(int *PreferenceList, int position){

  int NumOptions=0;
  
  int Side = Mem->GetMyPositionSide();
  int LocationSide = Mem->GetMyLocationSide();

  switch ( position ){
  case f442_LEFT_WING:
  case f442_RIGHT_WING:
    NumOptions = (int_random(3) ? 1 : 0); /* shoot 1 in 3 times */
    PreferenceList[0] =  (Side == RIGHT) ? f442_LEFT_WING : f442_RIGHT_WING;
    break;
  case f432_CENTER_MIDFIELD:
    NumOptions = 4;
    PreferenceList[0] = (LocationSide == RIGHT) ? f432_RIGHT_WING : f432_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f432_LEFT_WING : f432_RIGHT_WING;
    PreferenceList[2] = (LocationSide == RIGHT) ? f432_RIGHT_MIDFIELD :f432_LEFT_MIDFIELD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f432_LEFT_MIDFIELD :f432_RIGHT_MIDFIELD;
    break;
  case f432_RIGHT_MIDFIELD:
  case f432_LEFT_MIDFIELD:
    NumOptions = 3;
    PreferenceList[0] = (LocationSide == RIGHT) ? f432_RIGHT_WING : f432_LEFT_WING;
    PreferenceList[1] = (LocationSide == RIGHT) ? f432_LEFT_WING : f432_RIGHT_WING;
    PreferenceList[2] = f432_CENTER_MIDFIELD;
    break;
  case f432_CENTER_DEFENSE:    
    NumOptions = 3;
    PreferenceList[0] = (LocationSide == RIGHT) ? f432_RIGHT_MIDFIELD : f432_LEFT_MIDFIELD;
    PreferenceList[1] = (LocationSide == RIGHT) ? f432_LEFT_MIDFIELD : f432_RIGHT_MIDFIELD;
    PreferenceList[2] = f432_CENTER_MIDFIELD;
    break;
  case f432_RIGHT_DEFENSE:
  case f432_LEFT_DEFENSE:
    NumOptions = 3;
    PreferenceList[0] = (LocationSide == RIGHT) ? f432_RIGHT_MIDFIELD : f432_LEFT_MIDFIELD;
    PreferenceList[1] = (LocationSide == RIGHT) ? f432_RIGHT_WING : f432_LEFT_WING;
    PreferenceList[2] = f432_CENTER_MIDFIELD;
    break;
  case f432_SWEEPER:
    NumOptions = 7;
    PreferenceList[0] = (LocationSide == RIGHT) ? f432_RIGHT_DEFENSE : f432_LEFT_DEFENSE;
    PreferenceList[1] = (LocationSide == RIGHT) ? f432_LEFT_DEFENSE : f432_RIGHT_DEFENSE;
    PreferenceList[2] = (LocationSide == RIGHT) ? f432_RIGHT_MIDFIELD : f432_LEFT_MIDFIELD;
    PreferenceList[3] = (LocationSide == RIGHT) ? f432_LEFT_MIDFIELD : f432_RIGHT_MIDFIELD;
    PreferenceList[4] = f432_CENTER_MIDFIELD;
    PreferenceList[5] = (LocationSide == RIGHT) ? f432_RIGHT_WING : f432_LEFT_WING;
    PreferenceList[6] = (LocationSide == RIGHT) ? f432_LEFT_WING : f432_RIGHT_WING;
    break;
  case f432_RIGHT_GOALIE:
  case f432_LEFT_GOALIE:
    break; /* clear */
  }	

  for (int i = NumOptions; i<TEAM_SIZE; i++)
    PreferenceList[i] = position;  /* Pad list with own number, meaning no receiver */
}

/************************ RIGHT Formation ********************************/

#define fRT_GOALIE_BUF      2
#define fRT_POSITION_BUF    3

#define fRT_GOALTENDERX   (-52)
#define fRT_FULLBACKX     (-35)
#define fRT_QTRBACKX      (-15)
#define fRT_HALFBACKX      0
#define fRT_3QTRBACKX      12
#define fRT_FORWARDX       30
#define fRT_OUTSIDEY      (-22)
#define fRT_INSIDEY       (-7)
#define fRT_CENTERY        0

#define fRT_HOMEXRANGE    20
#define fRT_HOMEYRANGE    10
#define fRT_MAXXRANGE     30
#define fRT_MAXYRANGE     30

void Initialize_RT_Formation(Formation *formation){
  
  /* set position within goal stay, in penalty box      */
  formation->SetPosition(fRT_GOALIE, fRT_GOALTENDERX, fRT_CENTERY, fRT_GOALIE_BUF, 
			 (X0- (-fRT_GOALTENDERX))*2, GOAL_WIDTH, 
			 /*(-fRT_GOALTENDERX - PA_X)*2, PA_Y*2, */
			 (X0- (-fRT_GOALTENDERX))*2, GOAL_WIDTH, 
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, FALSE);  

  formation->SetPosition(fRT_INSIDE_FULLBACK, fRT_FULLBACKX, fRT_INSIDEY, fRT_POSITION_BUF, 
			 fRT_HOMEXRANGE,fRT_HOMEYRANGE,
			 fRT_MAXXRANGE ,fRT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fRT_OUTSIDE_FULLBACK, fRT_FULLBACKX, fRT_OUTSIDEY, fRT_POSITION_BUF, 
			 fRT_HOMEXRANGE,fRT_HOMEYRANGE,
			 fRT_MAXXRANGE ,fRT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fRT_INSIDE_QTRBACK, fRT_QTRBACKX, fRT_INSIDEY, fRT_POSITION_BUF, 
			 fRT_HOMEXRANGE,fRT_HOMEYRANGE,
			 fRT_MAXXRANGE ,fRT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fRT_OUTSIDE_QTRBACK, fRT_QTRBACKX, fRT_OUTSIDEY, fRT_POSITION_BUF, 
			 fRT_HOMEXRANGE,fRT_HOMEYRANGE,
			 fRT_MAXXRANGE ,fRT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fRT_INSIDE_HALFBACK, fRT_HALFBACKX, fRT_INSIDEY, fRT_POSITION_BUF, 
			 fRT_HOMEXRANGE,fRT_HOMEYRANGE,
			 fRT_MAXXRANGE ,fRT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fRT_OUTSIDE_HALFBACK, fRT_HALFBACKX, fRT_OUTSIDEY, fRT_POSITION_BUF, 
			 fRT_HOMEXRANGE,fRT_HOMEYRANGE,
			 fRT_MAXXRANGE ,fRT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fRT_INSIDE_3QTRBACK, fRT_3QTRBACKX, fRT_INSIDEY, fRT_POSITION_BUF, 
			 fRT_HOMEXRANGE,fRT_HOMEYRANGE,
			 fRT_MAXXRANGE ,fRT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fRT_OUTSIDE_3QTRBACK, fRT_3QTRBACKX, fRT_OUTSIDEY, fRT_POSITION_BUF, 
			 fRT_HOMEXRANGE,fRT_HOMEYRANGE,
			 fRT_MAXXRANGE ,fRT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fRT_INSIDE_FORWARD, fRT_FORWARDX, fRT_INSIDEY, fRT_POSITION_BUF, 
			 fRT_HOMEXRANGE,fRT_HOMEYRANGE,
			 fRT_MAXXRANGE ,fRT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fRT_OUTSIDE_FORWARD, fRT_FORWARDX, fRT_OUTSIDEY, fRT_POSITION_BUF, 
			 fRT_HOMEXRANGE,fRT_HOMEYRANGE,
			 fRT_MAXXRANGE ,fRT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);


  /* Candidate receivers will be chosen on the fly here */
  int receiverlist[TEAM_SIZE] = {fRT_INSIDE_FORWARD,fRT_OUTSIDE_FORWARD,
				 fRT_INSIDE_3QTRBACK,fRT_OUTSIDE_3QTRBACK,
				 fRT_INSIDE_HALFBACK,fRT_OUTSIDE_HALFBACK,
				 fRT_INSIDE_QTRBACK,fRT_OUTSIDE_QTRBACK,
				 fRT_INSIDE_FULLBACK,fRT_OUTSIDE_FULLBACK,0};
  for (int i=0; i<TEAM_SIZE; i++)
    formation->SetCandidateReceivers(i,TEAM_SIZE-i-1,receiverlist);

  formation->AllocateUnits(3);

  /* Positions listed in order of importance, captain indicated as index into list */

  int defenders[3] = {fRT_GOALIE, fRT_INSIDE_FULLBACK, fRT_OUTSIDE_FULLBACK};
  formation->GetUnit(DEFENSIVE_UNIT)->InitializeUnit(DEFENSIVE_UNIT, 3, defenders, 0);

  int midfielders[6] = {fRT_INSIDE_QTRBACK,  fRT_OUTSIDE_QTRBACK,
			fRT_INSIDE_HALFBACK, fRT_OUTSIDE_HALFBACK,
			fRT_INSIDE_3QTRBACK,  fRT_OUTSIDE_3QTRBACK};
  formation->GetUnit(MIDFIELD_UNIT)->InitializeUnit(MIDFIELD_UNIT, 6, midfielders, 2);

  int forwards[2] = {fRT_INSIDE_FORWARD, fRT_OUTSIDE_FORWARD};
  formation->GetUnit(FORWARD_UNIT)->InitializeUnit(FORWARD_UNIT, 2, forwards, 0);
}

/************************ LEFT  Formation ********************************/

#define fLT_GOALIE_BUF      2
#define fLT_POSITION_BUF    3

#define fLT_GOALTENDERX   (-52)
#define fLT_FULLBACKX     (-35)
#define fLT_QTRBACKX      (-15)
#define fLT_HALFBACKX      0
#define fLT_3QTRBACKX      12
#define fLT_FORWARDX       30
#define fLT_OUTSIDEY       22
#define fLT_INSIDEY        7
#define fLT_CENTERY        0

#define fLT_HOMEXRANGE    20
#define fLT_HOMEYRANGE    10
#define fLT_MAXXRANGE     30
#define fLT_MAXYRANGE     30

void Initialize_LT_Formation(Formation *formation){
  
  /* set position within goal stay, in penalty box      */
  formation->SetPosition(fLT_GOALIE, fLT_GOALTENDERX, fLT_CENTERY, fLT_GOALIE_BUF, 
			 (X0- (-fLT_GOALTENDERX))*2, GOAL_WIDTH, 
			 /*(-fLT_GOALTENDERX - PA_X)*2, PA_Y*2, */
			 (X0- (-fLT_GOALTENDERX))*2, GOAL_WIDTH, 
			 GOALIEMAXACTIVEDIST, MININACTIVEDIST, FALSE);  

  formation->SetPosition(fLT_INSIDE_FULLBACK, fLT_FULLBACKX, fLT_INSIDEY, fLT_POSITION_BUF, 
			 fLT_HOMEXRANGE,fLT_HOMEYRANGE,
			 fLT_MAXXRANGE ,fLT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fLT_OUTSIDE_FULLBACK, fLT_FULLBACKX, fLT_OUTSIDEY, fLT_POSITION_BUF, 
			 fLT_HOMEXRANGE,fLT_HOMEYRANGE,
			 fLT_MAXXRANGE ,fLT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fLT_INSIDE_QTRBACK, fLT_QTRBACKX, fLT_INSIDEY, fLT_POSITION_BUF, 
			 fLT_HOMEXRANGE,fLT_HOMEYRANGE,
			 fLT_MAXXRANGE ,fLT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fLT_OUTSIDE_QTRBACK, fLT_QTRBACKX, fLT_OUTSIDEY, fLT_POSITION_BUF, 
			 fLT_HOMEXRANGE,fLT_HOMEYRANGE,
			 fLT_MAXXRANGE ,fLT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fLT_INSIDE_HALFBACK, fLT_HALFBACKX, fLT_INSIDEY, fLT_POSITION_BUF, 
			 fLT_HOMEXRANGE,fLT_HOMEYRANGE,
			 fLT_MAXXRANGE ,fLT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fLT_OUTSIDE_HALFBACK, fLT_HALFBACKX, fLT_OUTSIDEY, fLT_POSITION_BUF, 
			 fLT_HOMEXRANGE,fLT_HOMEYRANGE,
			 fLT_MAXXRANGE ,fLT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fLT_INSIDE_3QTRBACK, fLT_3QTRBACKX, fLT_INSIDEY, fLT_POSITION_BUF, 
			 fLT_HOMEXRANGE,fLT_HOMEYRANGE,
			 fLT_MAXXRANGE ,fLT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fLT_OUTSIDE_3QTRBACK, fLT_3QTRBACKX, fLT_OUTSIDEY, fLT_POSITION_BUF, 
			 fLT_HOMEXRANGE,fLT_HOMEYRANGE,
			 fLT_MAXXRANGE ,fLT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fLT_INSIDE_FORWARD, fLT_FORWARDX, fLT_INSIDEY, fLT_POSITION_BUF, 
			 fLT_HOMEXRANGE,fLT_HOMEYRANGE,
			 fLT_MAXXRANGE ,fLT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);

  formation->SetPosition(fLT_OUTSIDE_FORWARD, fLT_FORWARDX, fLT_OUTSIDEY, fLT_POSITION_BUF, 
			 fLT_HOMEXRANGE,fLT_HOMEYRANGE,
			 fLT_MAXXRANGE ,fLT_MAXYRANGE,
			 MAXACTIVEDIST, MININACTIVEDIST, TRUE);


  /* Candidate receivers will be chosen on the fly here */
  int receiverlist[TEAM_SIZE] = {fLT_INSIDE_FORWARD,fLT_OUTSIDE_FORWARD,
				 fLT_INSIDE_3QTRBACK,fLT_OUTSIDE_3QTRBACK,
				 fLT_INSIDE_HALFBACK,fLT_OUTSIDE_HALFBACK,
				 fLT_INSIDE_QTRBACK,fLT_OUTSIDE_QTRBACK,
				 fLT_INSIDE_FULLBACK,fLT_OUTSIDE_FULLBACK,0};
  for (int i=0; i<TEAM_SIZE; i++)
    formation->SetCandidateReceivers(i,TEAM_SIZE-i-1,receiverlist);

  formation->AllocateUnits(3);

  /* Positions listed in order of importance, captain indicated as index into list */

  int defenders[3] = {fLT_GOALIE, fLT_INSIDE_FULLBACK, fLT_OUTSIDE_FULLBACK};
  formation->GetUnit(DEFENSIVE_UNIT)->InitializeUnit(DEFENSIVE_UNIT, 3, defenders, 0);

  int midfielders[6] = {fLT_INSIDE_QTRBACK,  fLT_OUTSIDE_QTRBACK,
			fLT_INSIDE_HALFBACK, fLT_OUTSIDE_HALFBACK,
			fLT_INSIDE_3QTRBACK,  fLT_OUTSIDE_3QTRBACK};
  formation->GetUnit(MIDFIELD_UNIT)->InitializeUnit(MIDFIELD_UNIT, 6, midfielders, 2);

  int forwards[2] = {fLT_INSIDE_FORWARD, fLT_OUTSIDE_FORWARD};
  formation->GetUnit(FORWARD_UNIT)->InitializeUnit(FORWARD_UNIT, 2, forwards, 0);
}


/*************************************************************************/
/************************ FUNCTION DEFINITIONS ***************************/
/*************************************************************************/

/************************ RECTANGLE CLASS FUNCTIONS **********************/

void Rectangle::DefineRectangle(float x,float y,float w,float h){
  centerX = x;
  centerY = y;
  width   = w;
  height  = h;
}

void  Rectangle::DefineRectangleFromTLCorner(float x,float y,float w,float h){
  DefineRectangle(x+w/2,y+h/2,w,h);
}

float Rectangle::DistanceToEdge(float x, float y){
  if ( IsWithin(x,y) )  /* return a positive number */
    return MIN(TopY()-y, MIN(y-BottomY(), MIN(LeftX()-x, x-RightX())));

  float xdist = -1000; /* want maximum negative distance (min abs) */
  float ydist = -1000; /* want maximum negative distance (min abs) */
  if (TopY()-y    <= 0 && TopY()-y    > ydist) ydist = TopY()-y;
  if (y-BottomY() <= 0 && y-BottomY() > ydist) ydist = y-BottomY();
  if (LeftX()-x   <= 0 && LeftX()-x   > xdist) xdist = LeftX()-x;
  if (x-RightX()  <= 0 && x-RightX()  > xdist) xdist = x-RightX();

  /* If it's outside in 2 dimensions, want min negative distance (max abs) */
  float result;
  if ( xdist == -1000 )
    result = ydist;
  else if ( ydist == -1000 )
    result = xdist;
  else
    result = MIN(xdist,ydist);

  if (result == -1000){
    printf("(%.1f, %.1f) distance to edge of \n",x,y);
    this->Print();
    my_error("Well then IsWithin should have been true");
  }

  return result;
}

int   Rectangle::IsWithin(float x, float y){
  if ( x<LeftX() || x>RightX() || y>TopY() || y<BottomY() )
    return FALSE;
  
  return TRUE;
}

void  Rectangle::AdjustToWithin(float *x, float *y){
  if (TopY()-*y    <0) *y = TopY();
  if (*y-BottomY() <0) *y = BottomY();
  if (RightX()-*x  <0) *x = RightX();
  if (*x-LeftX()   <0) *x = LeftX();
}

void  Rectangle::Print(){
  printf("RECTANGLE:  x = %.1f to %.1f   y = %.1f to %.1f\n",
	 LeftX(),RightX(),BottomY(),TopY());
}

/************************ TEAMPOSITION CLASS FUNCTIONS *****************/

TeamPosition::TeamPosition(){
  HomeRange = new Rectangle;
  MaxRange  = new Rectangle;
  HomeX = HomeY = HomeBuffer = 0;
  MaxActiveDistance = 0;
  only_active_if_closest = TRUE;
  type = side = 0;
}

TeamPosition::~TeamPosition(){
  delete HomeRange;
  delete MaxRange;
}

void TeamPosition::SetHome(float x, float y, float buf){
  HomeX = x;
  HomeY = y;
  HomeBuffer = buf;

  /* Determine the type and the side of a position */
  if      ( x < (ROUGH_GOALTENDERX+ROUGH_DEFENDERX)/2 ) type = GOALTENDER;
  else if ( x < (ROUGH_DEFENDERX+ROUGH_MIDFIELDERX)/2 ) type = DEFENDER;
  else if ( x < (ROUGH_MIDFIELDERX+ROUGH_FORWARDX) /2 ) type = MIDFIELDER;
  else                                                  type = FORWARD;

  if      ( y < (ROUGH_RIGHTY+ROUGH_CENTERY)/2 ) side = RIGHT;
  else if ( y < (ROUGH_CENTERY+ROUGH_LEFTY) /2 ) side = CENTER;
  else                                           side = LEFT;
}

void TeamPosition::SetHomeRange(float width, float height){
  HomeRange->DefineRectangle(HomeX,HomeY,width,height);
}

void TeamPosition::SetMaxRange (float width, float height){
  MaxRange->DefineRectangle(HomeX,HomeY,width,height);
}

void TeamPosition::PrintHomeRange (){
  HomeRange->Print();
}

void TeamPosition::PrintMaxRange (){
  MaxRange->Print();
}

/************************** UNIT CLASS FUNCTIONS ***********************/

Unit::Unit(){
  type = size = captain = 0;
  members = NULL;
}

Unit::~Unit(){
  if ( members != NULL )
    delete members;
}

Unit::InitializeUnit(int tp, int sz, int *mems, int capt){
  type = tp;
  size = sz;
  captain = mems[capt];
  members = new int[size];

  for (int i=0; i<size; i++)
    members[i] = mems[i];
}

int Unit::IsMember(int position){
  for (int i=0; i<size; i++){
    if ( members[i] == position )
      return TRUE;
  }
  return FALSE;
}

int Unit::IsCaptain(int position){
  if ( members[0] == position )
      return TRUE;
  return FALSE;
}

int Unit::ImportantPosition(int p1, int p2){
  int index1, index2;

  for (int i=0; i<size; i++){
    if ( members[i] == p1 )
      index1 = i;
    if ( members[i] == p2 )
      index2 = i;
  }

  if ( index1 < index2 )
    return p1;
  else if ( index2 < index1 )
    return p2;
  else
    my_error("indeces shouldn't be the same");
}

/************************ FORMATION CLASS FUNCTIONS ********************/

Formation::Formation(int formation_number){
  type = formation_number;
  FormationUnits = NULL;
}

Formation::~Formation(){
  if ( FormationUnits != NULL )
    delete FormationUnits;
}

Formation::AllocateUnits(int num){
  NumUnits = num;
  FormationUnits = new Unit[NumUnits];
}

void Formation::SetPosition(int num, float x, float y, float buffer,
			    float home_width, float home_height, 
			    float max_width, float max_height, float max_act_dist,
			    float min_inact_dist, int act_if_closest){
  /* Use this to put in the parameters for the positions in a given formation */
  PositionList[num].SetHome(x,y,buffer);
  PositionList[num].SetHomeRange(home_width,home_height);
  PositionList[num].SetMaxRange(max_width,max_height);

  PositionList[num].SetMaxActiveDistance  (max_act_dist  );
  PositionList[num].SetMinInactiveDistance(min_inact_dist);
  PositionList[num].SetOnlyActiveIfClosest(act_if_closest);
}

TeamPosition *Formation::GetPosition(int num){
  if (num >= TEAM_SIZE) 
    my_error("position list index only goes from 0 to TEAM_SIZE -1");
  return &(PositionList[num]); 
}

void Formation::SetCandidateReceivers(int position, int num, int *PrefList){
  int i;

  for (i=0; i<num; i++)
    ReceiverList[position][i] = PrefList[i];

  for (; i<TEAM_SIZE; i++)
    ReceiverList[position][i] = position;    /* End the list with sequence       */
}					     /*	of own number--means no reciever */

int *Formation::GetCandidateReceivers(int position){
  
  /* Either call a function to build the list on the fly, 
     or return the precanned ReceiverList */
  switch(type){
  case f433_FORMATION: 
    Build_433_ReceiverList(TempReceiverList,position); 
    return TempReceiverList;
  case f442_FORMATION: 
    Build_442_ReceiverList(TempReceiverList,position); 
    return TempReceiverList;
  case f352_FORMATION: 
    Build_352_ReceiverList(TempReceiverList,position); 
    return TempReceiverList;
  case f72_FORMATION: 
    Build_72_ReceiverList(TempReceiverList,position); 
    return TempReceiverList;
  case f334_FORMATION: 
    Build_334_ReceiverList(TempReceiverList,position); 
    return TempReceiverList;
  case f244_FORMATION: 
    Build_244_ReceiverList(TempReceiverList,position); 
    return TempReceiverList;
  case f432_FORMATION: 
    Build_432_ReceiverList(TempReceiverList,position); 
    return TempReceiverList;
  default: 
    return  ReceiverList[position]; 
  }
}

Unit *Formation::GetUnit(int num){
  return &FormationUnits[num];
}

int Formation::IsUnitMember(int unit, int position){
  return GetUnit(unit)->IsMember(position);
}

int Formation::IsUnitCaptain(int unit, int position){
  return GetUnit(unit)->IsCaptain(position);
}

int Formation::IsMoreImportantPosition(int pos1, int pos2){
  if ( pos1 == pos2 )
    my_error("same positions more important???");

  for (int unit=0; unit<NumUnits; unit++){
    if ( IsUnitMember(unit,pos1) && IsUnitMember(unit,pos2) ){
      if ( GetUnit(unit)->ImportantPosition(pos1,pos2) == pos1 )
	return TRUE;
    }
  }
  return FALSE;
}

/******************** TEAMPOSITIONINFO CLASS FUNCTIONS *****************/

TeamPositionInfo::TeamPositionInfo(){
  CurrentHomeX = CurrentHomeY = 0;
  Mark = UNKNOWN_PLAYER;
  MarkChangeMethod = MARK_OBEY;
  HomeChangeMethod = HOME_OBEY;
  PositionChangeMethod = 0;
  MarkPursuitRange = PositionStrategy = 0; 
  CurrentFormation = UNKNOWN_FORMATION;
  FormationTime = 0;

  for(int i=0; i<TEAM_SIZE+1; i++){
    PositionPlayers[i] = UNKNOWN_PLAYER;
    PlayerPositions[i] = UNKNOWN_POSITION;
  }

  f433_Formation = new Formation(f433_FORMATION);
  Initialize_433_Formation(f433_Formation);

  f442_Formation = new Formation(f442_FORMATION);
  Initialize_442_Formation(f442_Formation);

  f352_Formation = new Formation(f352_FORMATION);
  Initialize_352_Formation(f352_Formation);

  f72_Formation = new Formation(f72_FORMATION);
  Initialize_72_Formation(f72_Formation);

  f334_Formation = new Formation(f334_FORMATION);
  Initialize_334_Formation(f334_Formation);

  f244_Formation = new Formation(f244_FORMATION);
  Initialize_244_Formation(f244_Formation);

  f432_Formation = new Formation(f432_FORMATION);
  Initialize_432_Formation(f432_Formation);

  fRT_Formation = new Formation(fRT_FORMATION);
  Initialize_RT_Formation(fRT_Formation);

  fLT_Formation = new Formation(fLT_FORMATION);
  Initialize_LT_Formation(fLT_Formation);
}

TeamPositionInfo::~TeamPositionInfo(){
  delete f433_Formation;
  delete f442_Formation;
  delete f352_Formation;
  delete f72_Formation;
  delete f334_Formation;
  delete f244_Formation;
  delete f432_Formation;
  delete fRT_Formation;
  delete fLT_Formation;
}

void TeamPositionInfo::Reset(){
  if (GetMyPosition() == UNKNOWN_POSITION )
    SetMyPosition(GetNewPosition());
  ResetCurrentHome();
}

void TeamPositionInfo::SetPlayerPosition(int player, int position){

  /* printf("%d: changing %d's position from %d to %d\n",
	 Mem->MyNumber,player,Mem->GetPlayerPosition(player),position); */

  if ( player == UNKNOWN_PLAYER )
    my_error("Should we be setting unknown player's position?");
  else{
    /* First check to see if some other player is in the same position */
    int old_position = PlayerPositions[player];
    int old_position_player = UNKNOWN_PLAYER;
    if ( old_position != UNKNOWN_POSITION && PositionPlayers[old_position] == player ){
      for (int i=1; i<TEAM_SIZE; i++) {
	if ( i != player && PlayerPositions[i] == old_position )
	  old_position_player = i;
      }
      PositionPlayers[old_position] = old_position_player;
    }
  }

  if (position != UNKNOWN_POSITION)
    PositionPlayers[position] = player;
  if (player   != UNKNOWN_PLAYER  )
    PlayerPositions[player]   = position;
}

void TeamPositionInfo::ClearPlayerPosition(int player){
  int position = PlayerPositions[player];
  PlayerPositions[player] = UNKNOWN_POSITION;

  if ( player == Mem->MyNumber )
    PlayerPositions[0] = UNKNOWN_POSITION;

  if ( PositionPlayers[position] == player )
    PositionPlayers[position] = UNKNOWN_PLAYER;
}

void TeamPositionInfo::ClearPositionPlayer(int position){
  for (int i=0; i<TEAM_SIZE; i++){
    if ( PlayerPositions[i] == position )
      PlayerPositions[i] = UNKNOWN_POSITION;
  }
  PositionPlayers[position] = UNKNOWN_PLAYER;
}

void TeamPositionInfo::SetMyPosition(int position){
  SetPlayerPosition(0            , position);  /* Position 0 hold me too */
  SetPlayerPosition(Mem->MyNumber, position);
  ResetCurrentHome();
#if TEAM_POSITION_DEBUG
  printf("player %d, time %d:  going to position %d\n",
	 Mem->MyNumber,Mem->CurrentTime, position);
#endif
}

void TeamPositionInfo::ClearMyPosition(){
#if 0
  printf("player %d, time %d:  clearing position %d\n",
	 Mem->MyNumber,Mem->CurrentTime,Mem->GetMyPosition());
#endif
  ClearPlayerPosition(Mem->MyNumber);
}

int  TeamPositionInfo::PositionOccupied (int position){
  if (GetPositionPlayer(position) == UNKNOWN_PLAYER )
    return FALSE;
  else 
    return TRUE;
}

int TeamPositionInfo::LocationToPosition(float x, float y){
  /* Finds the position whose home range (x,y) falls.  If more than one,
     takes the one closest to the home position.  If none, takes the one
     whose home range rectangle is closest */

  float tiebreakDist = -1000;
  int position;
  Rectangle *HomeRange;
  
  for (int i=0; i<TEAM_SIZE; i++){
    HomeRange = CurrentFormation->GetPosition(i)->GetHomeRange();
    if ( HomeRange->IsWithin(x,y) ){  /* Check if (x,y) is within the home range */
      float HomeDist = DistanceToPositionHome(i,x,y);
      if ( tiebreakDist < 0 || HomeDist < tiebreakDist ){
	position = i;
	tiebreakDist = HomeDist;
      }
    }    /* We're not in the range of this position. Now check if closest so far */
    else if ( tiebreakDist < 0 && HomeRange->DistanceToEdge(x,y) > tiebreakDist ){
      position = i;
      tiebreakDist = HomeRange->DistanceToEdge(x,y);    /* a negative number     */
    }
  }
  return position;
}

float TeamPositionInfo::DistanceToPositionHome(int position, float x, float y){
  
  float HomeX = PositionHomeX(position);
  float HomeY = PositionHomeY(position);
  return GetDistance(&x,&y,&HomeX,&HomeY);
}

float TeamPositionInfo::PositionHomeX     (int position){
  return CurrentFormation->GetPosition(position)->GetHomeX();
}

float TeamPositionInfo::PositionHomeY     (int position){
  return CurrentFormation->GetPosition(position)->GetHomeY();
}

float TeamPositionInfo::PositionHomeBuffer(int position){
  return CurrentFormation->GetPosition(position)->GetHomeBuffer();
}

int   TeamPositionInfo::PositionType      (int position){
  return CurrentFormation->GetPosition(position)->GetType();
}

int   TeamPositionInfo::PositionSide      (int position){
  return CurrentFormation->GetPosition(position)->GetSide();
}

Rectangle *TeamPositionInfo::PositionHomeRange (int position){
  return CurrentFormation->GetPosition(position)->GetHomeRange();
}

Rectangle *TeamPositionInfo::PositionMaxRange  (int position){
  return CurrentFormation->GetPosition(position)->GetMaxRange();
}

float TeamPositionInfo::PositionHomeRangeWidth (int position){
  return PositionHomeRange(position)->Width();
}

float TeamPositionInfo::PositionHomeRangeHeight(int position){
  return PositionHomeRange(position)->Height();
}

float TeamPositionInfo::PositionMaxRangeWidth (int position){
  return PositionMaxRange(position)->Width();
}

float TeamPositionInfo::PositionMaxRangeHeight(int position){
  return PositionMaxRange(position)->Height();
}

float TeamPositionInfo::PlayerHomeX       (int player  ){
  return CurrentFormation->GetPosition(GetPlayerPosition(player))->GetHomeX();
}

float TeamPositionInfo::PlayerHomeY       (int player  ){
  return CurrentFormation->GetPosition(GetPlayerPosition(player))->GetHomeY();
}

float TeamPositionInfo::PlayerHomeBuffer  (int player  ){
  return CurrentFormation->GetPosition(GetPlayerPosition(player))->GetHomeBuffer();
}

int   TeamPositionInfo::PlayerType        (int player  ){
  return CurrentFormation->GetPosition(GetPlayerPosition(player))->GetType();
}

int   TeamPositionInfo::PlayerSide        (int player  ){
  return CurrentFormation->GetPosition(GetPlayerPosition(player))->GetSide();
}

int TeamPositionInfo::GetPlayerPosition(int player){
  return PlayerPositions[player];
}

int TeamPositionInfo::GetPositionPlayer(int position){
  return PositionPlayers[position];
}

int TeamPositionInfo::GetMyPosition(){
  return GetPlayerPosition(0);
}

float TeamPositionInfo::GetMyPositionX(){
  return PlayerHomeX(0);
}

float TeamPositionInfo::GetMyPositionY(){
  return PlayerHomeY(0);
}

int   TeamPositionInfo::GetMyPositionType(){
  return PlayerType(0);
}

int   TeamPositionInfo::GetMyPositionSide(){
  return PlayerSide(0);
}

float TeamPositionInfo::GetMyPositionBuffer(){
  return PositionHomeBuffer(GetMyPosition());
}

Rectangle *TeamPositionInfo::GetMyHomeRange(){
  return PositionHomeRange(GetMyPosition());
}

Rectangle *TeamPositionInfo::GetMyMaxRange(){
  return PositionMaxRange(GetMyPosition());
}

float TeamPositionInfo::GetMyHomeRangeWidth(){
  return PositionHomeRangeWidth(GetMyPosition());
}

float TeamPositionInfo::GetMyHomeRangeHeight(){
  return PositionHomeRangeHeight(GetMyPosition());
}

float TeamPositionInfo::GetMyMaxRangeWidth(){
  return PositionMaxRangeWidth(GetMyPosition());
}

float TeamPositionInfo::GetMyMaxRangeHeight(){
  return PositionMaxRangeHeight(GetMyPosition());
}

void TeamPositionInfo::PrintMyHomeRange(){
  GetMyHomeRange()->Print();
}

void TeamPositionInfo::PrintMyMaxRange(){
  GetMyMaxRange()->Print();
}

int   TeamPositionInfo::IsWithinMyHomeRange(float x, float y){
  return GetMyHomeRange()->IsWithin(x,y);
}

int   TeamPositionInfo::IsWithinMyMaxRange(float x, float y){
  return GetMyMaxRange()->IsWithin(x,y);
}
  
void  TeamPositionInfo::ResetCurrentHome(){
  int position = GetMyPosition();

  if (position != UNKNOWN_POSITION)
    SetCurrentHome(PositionHomeX(position),PositionHomeY(position));
}

void  TeamPositionInfo::SetFormation(int formation_number, int time){
  FormationTime = time;

  switch(formation_number){
  case f433_FORMATION: CurrentFormation = f433_Formation; break;
  case f442_FORMATION: CurrentFormation = f442_Formation; break;
  case f352_FORMATION: CurrentFormation = f352_Formation; break;
  case f72_FORMATION:  CurrentFormation = f72_Formation;  break;
  case f334_FORMATION: CurrentFormation = f334_Formation; break;
  case f244_FORMATION: CurrentFormation = f244_Formation; break;
  case f432_FORMATION: CurrentFormation = f432_Formation; break;
  case fRT_FORMATION: CurrentFormation = fRT_Formation; break;
  case fLT_FORMATION: CurrentFormation = fLT_Formation; break;
  default: my_error("Not a valid formation number");
  }
  ResetCurrentHome();

  if ( GetCurrentFormation() != UNKNOWN_FORMATION &&
       formation_number != GetCurrentFormationType() )
    SAY(HELLO_MSG); /* Announce the new formation */
}

int TeamPositionInfo::SetCurrentHome(float x, float y){

/*  printf("%d trying to set to (%.1f,%.1f) -- %d\n",
	 Mem->MyNumber,x,y,IsWithinMyHomeRange(x,y)); */

  if ( IsWithinMyHomeRange(x,y) ){
    CurrentHomeX = x;
    CurrentHomeY = y;
    return TRUE;
  }

  /* Need to adjust x and y to be in homerange */
  GetMyHomeRange()->AdjustToWithin(&x,&y);
  CurrentHomeX = x;
  CurrentHomeY = y;
  return FALSE;
}

int TeamPositionInfo::SetCurrentHomeX(float x ){
  return SetCurrentHome(x,GetCurrentHomeY());
}

int TeamPositionInfo::SetCurrentHomeY(float y ){
  return SetCurrentHome(GetCurrentHomeX(),y);
}

int TeamPositionInfo::SetMark        (int mark){
  /* Check if mark is in the maxrange?  No--you can mark a player far away
     Just don't move to it until it's in range */
  /* Announce the player you're leaving?? */
  Mark = mark;
#if MARK_DEBUG
  printf("%d: Marking %d\n",Mem->MyNumber,mark); 
#endif
}

void TeamPositionInfo::HearTeammateFormationPosition(int player, int formation, 
						     int formtime, int position){

  if ( formation == UNKNOWN_FORMATION_TYPE )
    return;

  if ( GetCurrentFormation() != UNKNOWN_FORMATION && /* Need to be initialized */
       formation != GetCurrentFormationType() && formtime > FormationTime ){
    /* It's a new formation! */
    FILE *outFile = fopen("out.dat","a");
    fprintf(outFile,"%d:%d Changed formations from %d to %d at time %d (according to %d)\n",
	   Mem->MyNumber,Mem->CurrentTime,
	   GetCurrentFormationType(),formation,formtime,player);
    fclose(outFile);
    SetFormation(formation,formtime);
  }

  if ( GetPlayerPosition(player) != position ){
    SetPlayerPosition(player,position);
  }
}

int TeamPositionInfo::GetUnitSize(int unit){
  return CurrentFormation->GetUnit(unit)->GetSize();
}

int *TeamPositionInfo::GetUnitMembers(int unit){
  return CurrentFormation->GetUnit(unit)->GetMembers();
}

int TeamPositionInfo::IsUnitMember(int unit, int player){
  return CurrentFormation->IsUnitMember(unit, GetPlayerPosition(player));
}

int TeamPositionInfo::IsUnitCaptain(int unit, int player){
  return CurrentFormation->IsUnitCaptain(unit, GetPlayerPosition(player));
}

int TeamPositionInfo::IsMyUnit(int unit){
  return IsUnitMember(unit,0);
}

int TeamPositionInfo::AmUnitCaptain(int unit){
  return IsUnitCaptain(unit,0);
}

int TeamPositionInfo::IsMyUnitCaptain(int player){
  for (int i=0; i<CurrentFormation->GetNumUnits(); i++){
    if ( IsMyUnit(i) && IsUnitCaptain(i,player) )
      return TRUE;
  }
  return FALSE;
}

int TeamPositionInfo::AmCandidateReceiverOfPlayer(int player){
  int position = GetPlayerPosition(player);
  int *candidates = CurrentFormation->GetCandidateReceivers(position);
  for (int i=0; i<=TEAM_SIZE; i++){
    if ( candidates[i] = GetMyPosition() )
      return TRUE;
    if ( candidates[i] = position )
      return FALSE;
  }
  return FALSE;
}
							    
int TeamPositionInfo::IsMoreImportantPosition(int pos1, int pos2){
  return CurrentFormation->IsMoreImportantPosition(pos1, pos2);
}
