/* data.c
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

#include "global.h"

void GetStampedName( char *name, char *outputName ){
  char date[100],weekday[10],month[10],temp[10];
  int  day,hour,min,sec,year;
  FILE *dateFile;
  
  /* Coach client does this                                      */
  /* system("date > ../date.log");        /* Put the date in a file */

  dateFile = fopen("../date.log","r");
  fscanf(dateFile,"%[^\n]",date);         /* Scan it in             */
  fclose(dateFile);
  
  sscanf(date,"%s %s %d %d:%d:%d %s %d",
	 weekday,month,&day,&hour,&min,&sec,temp,&year);
  sprintf(name,"../%s-%s%d-%d:%d:%d",outputName,month,day,hour,min,sec);
}

/*****************************************************************************/

#if SAVE_COLLECT_DATA
static FILE *CdataFile = NULL;
#endif

void RecordCollectData( float TurnAngle, float GoPower){

#if SAVE_COLLECT_DATA   
  static char CdataFileName[100];

  if ( Mem->NewCoach ){     /** Open the output file with a time stamp **/
    GetStampedName(CdataFileName,"collect.dat");
    CdataFile = fopen(CdataFileName,"a");
    
    fprintf(CdataFile,"%% -----------------------------------------\n");
    fprintf(CdataFile,"%% Net used (if any): %s\n",INIT_ToUse_FILE);
    fprintf(CdataFile,"%% -----------------------------------------\n");
    fprintf(CdataFile,"%% Dist  Ang  VMag  TAng  GPow R\n");
    fprintf(CdataFile,"%% -----------------------------------------\n");

    /** Unsetting Mem->NewCoach in RecordCollectResult **/
  }

  if (CdataFile == NULL) {
    CdataFile = fopen(CdataFileName,"a");
  }

  /* vdir was 4th input before 6/12/96                                        */
  /* float vdir = 180 - Mem->GetBallRelVelDirection();  /* Reverse the direction */
  /* if (vdir == 180) vdir = 0;                      /* Unreverse if it was 0 */
  /* CleanAngle(&vdir);                                                       */
  fprintf(CdataFile,"  %4.1f %5.1f %4.1f ",
	  Mem->GetBallDistance(),Mem->GetBallAngle(),
	  Mem->GetBallRelVelMagnitude());
	  /* Mem->GetBallRelVelMagnitude()*(Mem->CurrentTime - Mem->LastTime) ); */
          /* Assuming always on narrow view when using nns */
  fprintf(CdataFile,"%6.2f %4.2f ", TurnAngle, GoPower);
#endif

}

/*****************************************************************************/

void RecordCollectResult(char Result){

#if SAVE_COLLECT_DATA   

  if (Mem->RecordCollectData)   /** Never collected the data, so don't **/
    return;                     /** record the result                  **/

  if ( CdataFile != NULL ){     /** Only the client that recorded data **/
    static int counter = 0;
    static int misses  = 0;
    static int saves   = 0;
    static int goals   = 0;

    if ( Mem->NewCoach ){ /** First time through, reset everything **/
      counter = misses = saves = 0;
      Mem->NewCoach = FALSE;
    }

    fprintf(CdataFile,"%c\n",Result);

    switch(Result){
    case COLLECT_MISS: misses++; break; 
    case COLLECT_SAVE: saves++;  break;
    case COLLECT_GOAL: goals++;  break;
    }
    
    if ( ++counter%(SAVE_FREQ)==0 ){
      fprintf(CdataFile,"%%misses: %d/%d = %4.1f%%   saves: %d/%d = %4.1f%%   goals: %d/%d = %4.1f%%\n",
	      misses,counter,(float) 100*misses/counter,
	      saves,counter,(float) 100*saves/counter,
	      goals,counter,(float) 100*goals/counter);
      fprintf(CdataFile,"%%saves/goals: %d/%d = %4.1f%%\n",
	      saves,goals+saves,(float) 100*saves/(goals+saves));
      fclose(CdataFile);
      CdataFile = NULL; /** Set it back to Null so RecordCollectData knows to reopen **/
    }
  }
#endif

}

/*****************************************************************************/

void GetPassData(char *outStream, int receiver, char sortBy){
  float angle,distance;

  if ( !Mem->TeammateAngleValid(receiver) ){
    if (sortBy == 'a')
      my_error("Need to know player angle to get tree stats!!");
    else 
      angle = 0;
  }
  else 
    angle = Mem->GetTeammateAngle(receiver);

  if ( !Mem->TeammateDistanceValid(receiver) )
    distance = 0;
  else 
    distance = Mem->GetTeammateDistance(receiver);
  
  GetPassData(outStream,distance,angle,sortBy);
}

/* Can be used to get data to a point--not just a receiver */

void GetPassData(char *outStream, float receiverDist, float receiverAng, 
		 char sortBy){

  char MySide = Mem->MySide, TheirSide = Mem->TheirSide;

  /* If sorting by angle, put receiver's angle at 0           */
  /* If sorting by distance, use 0 as the key (closest first) */
  float key = 0;
  /* Assume that if I'm passing, I know at least the angle    */
  if ( sortBy == 'a' ){
    key = receiverAng;
  }

  int teammates[TEAM_SIZE];
  int NumTeammates = Mem->SortPlayersBy('m',sortBy,key,teammates);
  int opponents[TEAM_SIZE];
  int NumOpponents = Mem->SortPlayersBy('t',sortBy,key,opponents);

  int counter = 0;
  for (int i=0; i<NumTeammates; i++){  /* Sorted by angle insures receiver's first */
    if ( Mem->PlayerValid(MySide, teammates[i]) ){

      if ( Mem->GetTeammateDistance(teammates[i]) >= 150 ){
	char msg[200];
	sprintf(msg,"player %d valid (%.1f)  and  > 150 (%.1f)?  I'm at (%.1f,%.1f) (conf %.1f),  other's at (%.1f, %.1f) (conf %.1f)\n",
		teammates[i],Mem->TeammateValid(teammates[i]),
		Mem->GetTeammateDistance(teammates[i]),
		Mem->GetGlobalX(),Mem->GetGlobalY(),Mem->MarkerValid(THEIR_GOAL),
		Mem->GetPlayerGlobalX(Mem->MySide,teammates[i]),
		Mem->GetPlayerGlobalY(Mem->MySide,teammates[i]),
		Mem->PlayerValid(Mem->MySide,teammates[i]));
	my_error(msg);
      }

      sprintf(&outStream[counter],"%5.1f,%6.1f,",
	      Mem->GetPlayerDistance(MySide, teammates[i]), 
	      fabs(Mem->GetPlayerAngle(MySide, teammates[i]) - key));
      counter+=13;
    }
    else{ /* Only angle is valid */
      sprintf(&outStream[counter],"?,%6.1f,",
	      fabs(Mem->GetPlayerAngle(MySide, teammates[i]) - key));
      counter+=9;
    }
  }
  for (int i=NumTeammates; i<TEAM_SIZE-1; i++){ /* -1 to leave out self */
    sprintf(&outStream[counter],"?,?,");
    counter+=4;
  }
  int NumOpsEntered = 0;
  for (int i=0; i<NumOpponents; i++){  /* Sorted by angle insures receiver's first */
    if ( opponents[i] == 1 &&
	 Mem->MarkerValid(THEIR_GOAL) &&
	 receiverDist == Mem->GetMarkerDistance(THEIR_GOAL) &&
	 receiverAng  == Mem->GetMarkerAngle(THEIR_GOAL) ){
      /* Special Case:  Disregard the goalie if shooting */
      i++;
    }
    else if ( Mem->PlayerValid(TheirSide, opponents[i]) ){
      sprintf(&outStream[counter],"%5.1f,%6.1f,",
	      Mem->GetPlayerDistance(TheirSide, opponents[i]), 
	      fabs(Mem->GetPlayerAngle(TheirSide, opponents[i]) - key));
      counter+=13;
      NumOpsEntered++;
    }
    else{ /* Only angle is valid */
      sprintf(&outStream[counter],"?,%6.1f,",
	      fabs(Mem->GetPlayerAngle(TheirSide, opponents[i]) - key));
      counter+=9;
      NumOpsEntered++;
    }
  }
  for (int i=NumOpsEntered; i<TEAM_SIZE; i++){
    sprintf(&outStream[counter],"?,?,");
    counter+=4;
  }

  /* Now some stats. If want more, make dist,ang into arrays, use loop*/
  /* What if distance isn't valid? */ 

  if ( receiverDist ){
    float RD = receiverDist;
    float RA = receiverAng;

#define NUM_STATS 15
    float dist[NUM_STATS]     = {4 ,8 ,12,4 ,8 ,12,4 ,8 ,12,RD,RD,RD,RD,RD,RD};
    float ang[NUM_STATS]      = {4 ,4 ,4 ,8 ,8 ,8 ,12,12,12,1 ,2 ,3 ,4 ,5 ,6 };
    float fromDist[NUM_STATS] = {RD,RD,RD,RD,RD,RD,RD,RD,RD,0 ,0 ,0 ,0 ,0 ,0 };
    float fromAng[NUM_STATS]  = {RA,RA,RA,RA,RA,RA,RA,RA,RA,RA,RA,RA,RA,RA,RA};
    
    for (int i=0; i<NUM_STATS; i++){
      sprintf(&outStream[counter],"%d, %d, %d, ",
	      Mem->NumTeammatesWithin(dist[i],ang[i],fromDist[i],fromAng[i]),
	      Mem->NumOpponentsWithin(dist[i],ang[i],fromDist[i],fromAng[i]),
	      Mem->NumPlayersWithin  (dist[i],ang[i],fromDist[i],fromAng[i]) );
      counter += 9;
    }
  }
  else{ /* distance isn't valid */
    for (int i=0; i<NUM_STATS; i++){
      sprintf(&outStream[counter],"?, ?, ?, ");
      counter += 9;
    }
  }
}

/*****************************************************************************/

#if SAVE_PASS_DATA
static FILE *PdataFile = NULL;
#endif

void RecordPassData( int receiver ){

#if SAVE_PASS_DATA   
  static char PdataFileName[100];

  if ( Mem->NewCoach ){     /** Open the output file with a time stamp **/
    GetStampedName(PdataFileName,"pass.dat");
    PdataFile = fopen(PdataFileName,"a");
    
    fprintf(PdataFile,"| -----------------------------------------\n");
    fprintf(PdataFile,"| Tree used (if any): %s\n",TREE_STEM);
    fprintf(PdataFile,"| -----------------------------------------\n");
    fprintf(PdataFile,"| teammates(me) opponents(me) stats(me) teammates(rec) opponents(rec) stats(rec) res\n");
    fprintf(PdataFile,"| -----------------------------------------\n");

    /** Unsetting Mem->NewCoach in RecordCollectResult **/
  }

  if (PdataFile == NULL) {
    PdataFile = fopen(PdataFileName,"a");
  }
  
/* Player distance, other 9 teammates angs and dists sorted by ang? 
11 opponents sorted by ang , teammates 9 players sorted by dist, 
11 opponents sorted by dist */

  char outStream[MAXMESG];
  GetPassData(outStream,receiver,'a',Mem);
  fprintf(PdataFile,"%s",outStream);

  /* Put in the receiver's data.  Fast forward past the data msg */
  
  fprintf(PdataFile,"%s",Mem->GetData(receiver));

#endif

}

/*****************************************************************************/

#if DETAILED_PASS_STATS
extern char  _PredictedClass_;
extern float _PredictedProb_;
#endif

void RecordPassResult(char Result){

#if SAVE_PASS_DATA   

  if ( PdataFile != NULL ){     /** Only the client that recorded data **/
    static int counter   = 0;
    static int successes = 0;
    static int failures  = 0;
    static int misses    = 0;
#if DETAILED_PASS_STATS
    static int successesSpoint9 = 0;
    static int failuresSpoint9  = 0;
    static int missesSpoint9    = 0;
    static int successesSpoint8 = 0;
    static int failuresSpoint8  = 0;
    static int missesSpoint8    = 0;
    static int successesSpoint7 = 0;
    static int failuresSpoint7  = 0;
    static int missesSpoint7    = 0;
    static int successesSpoint6 = 0;
    static int failuresSpoint6  = 0;
    static int missesSpoint6    = 0;
    static int successesSpoint5 = 0;
    static int failuresSpoint5  = 0;
    static int missesSpoint5    = 0;
    static int successesSpoint0 = 0;
    static int failuresSpoint0  = 0;
    static int missesSpoint0    = 0;
    static int successesFpoint9 = 0;
    static int failuresFpoint9  = 0;
    static int missesFpoint9    = 0;
    static int successesFpoint8 = 0;
    static int failuresFpoint8  = 0;
    static int missesFpoint8    = 0;
    static int successesFpoint7 = 0;
    static int failuresFpoint7  = 0;
    static int missesFpoint7    = 0;
    static int successesFpoint6 = 0;
    static int failuresFpoint6  = 0;
    static int missesFpoint6    = 0;
    static int successesFpoint5 = 0;
    static int failuresFpoint5  = 0;
    static int missesFpoint5    = 0;
    static int successesFpoint0 = 0;
    static int failuresFpoint0  = 0;
    static int missesFpoint0    = 0;
#endif

    if ( Mem->NewCoach ){ /** First time through, reset everything **/
      counter = successes = failures = 0;
      Mem->NewCoach = FALSE;
    }

    fprintf(PdataFile,"%c\n",Result);

    switch(Result){
    case PASS_SUCCESS: 
      successes++;  
#if DETAILED_PASS_STATS
      if (_PredictedClass_ == 'S'){
	if (_PredictedProb_ > .9)
	  successesSpoint9++;
        else if (_PredictedProb_ > .8)
	  successesSpoint8++;
        else if (_PredictedProb_ > .7)
	  successesSpoint7++;
        else if (_PredictedProb_ > .6)
	  successesSpoint6++;
        else if (_PredictedProb_ > .5)
	  successesSpoint5++;
	else 
	  successesSpoint0++;
      }
      else if (_PredictedClass_ == 'F'){
	if (_PredictedProb_ > .9)
	  successesFpoint9++;
        else if (_PredictedProb_ > .8)
	  successesFpoint8++;
        else if (_PredictedProb_ > .7)
	  successesFpoint7++;
        else if (_PredictedProb_ > .6)
	  successesFpoint6++;
        else if (_PredictedProb_ > .5)
	  successesFpoint5++;
	else 
	  successesFpoint0++;
      }
#endif
      break;
    case PASS_FAILURE: 
      failures++;   
#if DETAILED_PASS_STATS
      if (_PredictedClass_ == 'S'){
	if (_PredictedProb_ > .9)
	  failuresSpoint9++;
        else if (_PredictedProb_ > .8)
	  failuresSpoint8++;
        else if (_PredictedProb_ > .7)
	  failuresSpoint7++;
        else if (_PredictedProb_ > .6)
	  failuresSpoint6++;
        else if (_PredictedProb_ > .5)
	  failuresSpoint5++;
	else 
	  failuresSpoint0++;
      }
      else if (_PredictedClass_ == 'F'){
	if (_PredictedProb_ > .9)
	  failuresFpoint9++;
        else if (_PredictedProb_ > .8)
	  failuresFpoint8++;
        else if (_PredictedProb_ > .7)
	  failuresFpoint7++;
        else if (_PredictedProb_ > .6)
	  failuresFpoint6++;
        else if (_PredictedProb_ > .5)
	  failuresFpoint5++;
	else 
	  failuresFpoint0++;
      }
#endif
      break; 
    case PASS_MISS:    
      misses++;     
#if DETAILED_PASS_STATS
      if (_PredictedClass_ == 'S'){
	if (_PredictedProb_ > .9)
	  missesSpoint9++;
        else if (_PredictedProb_ > .8)
	  missesSpoint8++;
        else if (_PredictedProb_ > .7)
	  missesSpoint7++;
        else if (_PredictedProb_ > .6)
	  missesSpoint6++;
        else if (_PredictedProb_ > .5)
	  missesSpoint5++;
	else 
	  missesSpoint0++;
      }
      else if (_PredictedClass_ == 'F'){
	if (_PredictedProb_ > .9)
	  missesFpoint9++;
        else if (_PredictedProb_ > .8)
	  missesFpoint8++;
        else if (_PredictedProb_ > .7)
	  missesFpoint7++;
        else if (_PredictedProb_ > .6)
	  missesFpoint6++;
        else if (_PredictedProb_ > .5)
	  missesFpoint5++;
	else 
	  missesFpoint0++;
      }
#endif
      break;
    }
    
    if ( !strcmp(BehaviorName,"TreePass") )
      printf("%c %.2f\n",_PredictedClass_,_PredictedProb_);

    if ( ++counter%(SAVE_FREQ)==0 ){
      fprintf(PdataFile,"| misses: %d/%d = %4.1f%%   successes: %d/%d = %4.1f%%   failures: %d/%d = %4.1f%%\n",
	      misses,counter,(float)100*misses/counter,
	      successes,counter,(float)100*successes/counter,
	      failures,counter,(float) 100*failures/counter);
#if DETAILED_PASS_STATS
if ( !strcmp(BehaviorName,"TreePass") ){
      int counterSpoint9 = missesSpoint9+failuresSpoint9+successesSpoint9;
      fprintf(PdataFile,"| missesS.9: %d/%d = %4.1f%%   successesS.9: %d/%d = %4.1f%%   failuresS.9: %d/%d = %4.1f%%\n",
	      missesSpoint9,counterSpoint9,(float)100*missesSpoint9/counterSpoint9,
	      successesSpoint9,counterSpoint9,(float)100*successesSpoint9/counterSpoint9,
	      failuresSpoint9,counterSpoint9,(float) 100*failuresSpoint9/counterSpoint9);
      int counterSpoint8 = missesSpoint8+failuresSpoint8+successesSpoint8;
      fprintf(PdataFile,"| missesS.8: %d/%d = %4.1f%%   successesS.8: %d/%d = %4.1f%%   failuresS.8: %d/%d = %4.1f%%\n",
	      missesSpoint8,counterSpoint8,(float)100*missesSpoint8/counterSpoint8,
	      successesSpoint8,counterSpoint8,(float)100*successesSpoint8/counterSpoint8,
	      failuresSpoint8,counterSpoint8,(float) 100*failuresSpoint8/counterSpoint8);
      int counterSpoint7 = missesSpoint7+failuresSpoint7+successesSpoint7;
      fprintf(PdataFile,"| missesS.7: %d/%d = %4.1f%%   successesS.7: %d/%d = %4.1f%%   failuresS.7: %d/%d = %4.1f%%\n",
	      missesSpoint7,counterSpoint7,(float)100*missesSpoint7/counterSpoint7,
	      successesSpoint7,counterSpoint7,(float)100*successesSpoint7/counterSpoint7,
	      failuresSpoint7,counterSpoint7,(float) 100*failuresSpoint7/counterSpoint7);
      int counterSpoint6 = missesSpoint6+failuresSpoint6+successesSpoint6;
      fprintf(PdataFile,"| missesS.6: %d/%d = %4.1f%%   successesS.6: %d/%d = %4.1f%%   failuresS.6: %d/%d = %4.1f%%\n",
	      missesSpoint6,counterSpoint6,(float)100*missesSpoint6/counterSpoint6,
	      successesSpoint6,counterSpoint6,(float)100*successesSpoint6/counterSpoint6,
	      failuresSpoint6,counterSpoint6,(float) 100*failuresSpoint6/counterSpoint6);
      int counterSpoint5 = missesSpoint5+failuresSpoint5+successesSpoint5;
      fprintf(PdataFile,"| missesS.5: %d/%d = %4.1f%%   successesS.5: %d/%d = %4.1f%%   failuresS.5: %d/%d = %4.1f%%\n",
	      missesSpoint5,counterSpoint5,(float)100*missesSpoint5/counterSpoint5,
	      successesSpoint5,counterSpoint5,(float)100*successesSpoint5/counterSpoint5,
	      failuresSpoint5,counterSpoint5,(float) 100*failuresSpoint5/counterSpoint5);
      int counterSpoint0 = missesSpoint0+failuresSpoint0+successesSpoint0;
      fprintf(PdataFile,"| missesS.0: %d/%d = %4.1f%%   successesS.0: %d/%d = %4.1f%%   failuresS.0: %d/%d = %4.1f%%\n",
	      missesSpoint0,counterSpoint0,(float)100*missesSpoint0/counterSpoint0,
	      successesSpoint0,counterSpoint0,(float)100*successesSpoint0/counterSpoint0,
	      failuresSpoint0,counterSpoint0,(float) 100*failuresSpoint0/counterSpoint0);
      int counterFpoint9 = missesFpoint9+failuresFpoint9+successesFpoint9;
      fprintf(PdataFile,"| missesF.9: %d/%d = %4.1f%%   successesF.9: %d/%d = %4.1f%%   failuresF.9: %d/%d = %4.1f%%\n",
	      missesFpoint9,counterFpoint9,(float)100*missesFpoint9/counterFpoint9,
	      successesFpoint9,counterFpoint9,(float)100*successesFpoint9/counterFpoint9,
	      failuresFpoint9,counterFpoint9,(float) 100*failuresFpoint9/counterFpoint9);
      int counterFpoint8 = missesFpoint8+failuresFpoint8+successesFpoint8;
      fprintf(PdataFile,"| missesF.8: %d/%d = %4.1f%%   successesF.8: %d/%d = %4.1f%%   failuresF.8: %d/%d = %4.1f%%\n",
	      missesFpoint8,counterFpoint8,(float)100*missesFpoint8/counterFpoint8,
	      successesFpoint8,counterFpoint8,(float)100*successesFpoint8/counterFpoint8,
	      failuresFpoint8,counterFpoint8,(float) 100*failuresFpoint8/counterFpoint8);
      int counterFpoint7 = missesFpoint7+failuresFpoint7+successesFpoint7;
      fprintf(PdataFile,"| missesF.7: %d/%d = %4.1f%%   successesF.7: %d/%d = %4.1f%%   failuresF.7: %d/%d = %4.1f%%\n",
	      missesFpoint7,counterFpoint7,(float)100*missesFpoint7/counterFpoint7,
	      successesFpoint7,counterFpoint7,(float)100*successesFpoint7/counterFpoint7,
	      failuresFpoint7,counterFpoint7,(float) 100*failuresFpoint7/counterFpoint7);
      int counterFpoint6 = missesFpoint6+failuresFpoint6+successesFpoint6;
      fprintf(PdataFile,"| missesF.6: %d/%d = %4.1f%%   successesF.6: %d/%d = %4.1f%%   failuresF.6: %d/%d = %4.1f%%\n",
	      missesFpoint6,counterFpoint6,(float)100*missesFpoint6/counterFpoint6,
	      successesFpoint6,counterFpoint6,(float)100*successesFpoint6/counterFpoint6,
	      failuresFpoint6,counterFpoint6,(float) 100*failuresFpoint6/counterFpoint6);
      int counterFpoint5 = missesFpoint5+failuresFpoint5+successesFpoint5;
      fprintf(PdataFile,"| missesF.5: %d/%d = %4.1f%%   successesF.5: %d/%d = %4.1f%%   failuresF.5: %d/%d = %4.1f%%\n",
	      missesFpoint5,counterFpoint5,(float)100*missesFpoint5/counterFpoint5,
	      successesFpoint5,counterFpoint5,(float)100*successesFpoint5/counterFpoint5,
	      failuresFpoint5,counterFpoint5,(float) 100*failuresFpoint5/counterFpoint5);
      int counterFpoint0 = missesFpoint0+failuresFpoint0+successesFpoint0;
      fprintf(PdataFile,"| missesF.0: %d/%d = %4.1f%%   successesF.0: %d/%d = %4.1f%%   failuresF.0: %d/%d = %4.1f%%\n",
	      missesFpoint0,counterFpoint0,(float)100*missesFpoint0/counterFpoint0,
	      successesFpoint0,counterFpoint0,(float)100*successesFpoint0/counterFpoint0,
	      failuresFpoint0,counterFpoint0,(float) 100*failuresFpoint0/counterFpoint0);
} 
#endif
      fclose(PdataFile);
      PdataFile = NULL; /** Set it back to Null so RecordPassData knows to reopen **/
    }
  }
#endif

}

