/* behave.c
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

#include "global.h"

static char *BehaviorName = "Dunno";

/*****************************************************************************/

int Defend(float dist, float gpow, float spow){

  static int _INPOSITION_ = FALSE;
  static int _DEFENDING_ = FALSE;
  
/*printf("%d/%d - ",_INPOSITION_,_DEFENDING_);*/

  if ( !_INPOSITION_ && !_DEFENDING_) {         /* If out of position, go home */
    if ( GOALIEHOME(gpow) >= TRIED )        /* Turned to face ball         */
      _INPOSITION_ = TRUE;
  }
  else if ( !_INPOSITION_ && _DEFENDING_ ){
    if ( (Mem->BallValid() && Mem->GetBallDistance() > dist) ||
	 (Mem->MarkerValid(MY_GOAL) && Mem->GetMarkerDistance(MY_GOAL) > dist*2) )
      _DEFENDING_ = FALSE;                    /* Stop and go back home       */
    else 
      CHASEANDSHOOT( gpow,spow );
  }
  else { /* _INPOSITION_ */
   _DEFENDING_ = TRUE;                       /* _DEFENDING_ Mode until ball
				              comes close and then leaves */
   if ( COLLECTANDSHOOT( &ChooseNN,dist,gpow,spow ) >= TRYING )
     _INPOSITION_ = FALSE;                   /* Moved to ball               */
 }
  return _INPOSITION_;
}

/*****************************************************************************/

int Kamram(float dist, float gpow, float spow){

static int _INPOSITION_ = FALSE;
static int _ATTACKING_ = FALSE;

/*printf("%d/%d - ",_INPOSITION_,_ATTACKING_);*/

  if ( !_INPOSITION_ && !_ATTACKING_) {         /* If out of position, go home */
    if ( KAMRAMHOME(gpow) >= TRIED )        /* Turned to face ball         */
      _INPOSITION_ = TRUE;
  }
  else if ( !_INPOSITION_ && _ATTACKING_ ){
    if ( (Mem->BallValid()  && Mem->GetBallDistance() > dist) ||
	 (Mem->MarkerValid(THEIR_GOAL) && Mem->GetMarkerDistance(THEIR_GOAL) > dist*2) )
      _ATTACKING_ = FALSE;                    /* Stop and go back home       */
    else 
      CHASEANDSHOOT( gpow,spow );
  }
  else { /* _INPOSITION_ */
   _ATTACKING_ = TRUE;                       /* _ATTACKING_ Mode until ball
				              comes close and then leaves */
   if ( COLLECTANDSHOOT( &ChooseNN,dist,gpow,spow ) >= TRYING )
     _INPOSITION_ = FALSE;                   /* Moved to ball               */
 }
  return _INPOSITION_;
}

/*****************************************************************************/

int TreePass(int (*ReceiverChoiceFunction)(), 
	     float cdist, float pdist, float gpow, float dpow, float spow){

  #define _TREEPASS_LOOKING_   0
  #define _TREEPASS_INFORMING_ _TREEPASS_LOOKING_   +4
  #define _TREEPASS_CHOOSING_  _TREEPASS_INFORMING_ +1
  #define _TREEPASS_PINGING_   _TREEPASS_CHOOSING_  +1
  #define _TREEPASS_PASSING_   _TREEPASS_PINGING_   +1
  #define _TREEPASS_RESET_     _TREEPASS_PASSING_   +1

  static int Phase = _TREEPASS_LOOKING_;
  static int receiver = 0;
  static int HangCounter = 0;
  static int HangCounter2 = 0;
  int Result, ball_coming;

  switch(Phase){
  case _TREEPASS_LOOKING_: /* Waiting to collect the ball */
#if 0
    ball_coming = Mem->FindTeammateWithStatus(RECEIVER_STAT); /* The player to whom
								     the ball is coming */
    if ( ball_coming == 0 || ball_coming == Mem->MyNumber ){
      Result = CHASEBALL(gpow);
    }
    else{
      Result = 0;
    }  /* Look for the ball */
#else
    Result = COLLECTBALL(&ChooseNN,cdist,gpow);
#endif

    if ( Result == SUCCESS ){    /* Or if Result == TRIED? */
      Phase++;
    }
    else if ( HangCounter++ > 100 ){/* Coach needs to reset ball */
      SAYTOCOACH(WAKEUP_MSG);
      HangCounter = 0;
    }
    break;
  case _TREEPASS_INFORMING_:
    if ( Mem->BallValid() && Mem->GetBallDistance() > KICK_DISTANCE ){
      Phase = _TREEPASS_LOOKING_;  /* Return to beginning phase if ball's gone */
      break;
    }
    HangCounter = 0;
    Mem->ClearOtherStatuses(); /* Get rid of all the open stats */
    Mem->ClearData();
    SAY(READY_TO_PASS_MSG);
    receiver = 0;
    Phase++;
    break;
  case _TREEPASS_CHOOSING_: /* Waiting for answer */
    if ( Mem->BallValid() && Mem->GetBallDistance() > KICK_DISTANCE ){
      Phase = _TREEPASS_LOOKING_;  /* Return to beginning phase if ball's gone */
      break;
    }
    receiver = (*ReceiverChoiceFunction)();
    if ( receiver ){
      char Announcement[MAXMESG];
      sprintf(Announcement,PASSING_DECISION,receiver);
      SAY(Announcement);
      Phase++;
    }
    else if (HangCounter++ > 10){
      SAY(READY_TO_PASS_MSG);  /* One of the players may have lost a message */
      HangCounter = 0;
#if SAVE_PASS_DATA
      HangCounter2++;          /* Or one player might be stuck at global angle 0 */
#else
      Phase = _TREEPASS_LOOKING_;
#endif
    }
    else if (HangCounter2 > 10){
      SAY(JOLT_MSG);
      HangCounter2 = 0;
    }
    break;
  case _TREEPASS_PINGING_:
    if ( Mem->BallValid() && Mem->GetBallDistance() > KICK_DISTANCE ){
      Phase = _TREEPASS_LOOKING_;  /* Return to beginning phase if ball's gone */
      break;
    }
    if ( !receiver ){ 
      Phase = _TREEPASS_CHOOSING_;
      break;
    }
    if ( FACETEAMMATE(receiver) == SUCCESS ) /* Sure I'm facing the player */
      Phase++;
    if ( !strcmp(Mem->GetData(receiver),BLANK_DAT) )
      SAYTO(receiver,GIVE_DATA_MSG);  /* Ping the individual */
    break;
  case _TREEPASS_PASSING_: /* Pass the ball */
    if ( Mem->BallValid() && Mem->GetBallDistance() > 15 ){
      Phase = _TREEPASS_LOOKING_;  /* Return to beginning phase if ball's gone */
      break;
    }
    if ( receiver ){
	Result = APPROACHANDPASSTO( receiver, 150, pdist, gpow, dpow, spow );
	if ( Result == TRIED )
	  Phase++;
      }
    else /* !receiver */
      my_error("should have a receiver");
    break;
  case _TREEPASS_RESET_:  
      Phase = _TREEPASS_LOOKING_;
      Mem->SetStatus(Mem->MySide,receiver,BLANK_STAT);  /* Clear that the player's open
				             	      eventually should clear ALL opens */
      /*Mem->Acting(FORG);*/
      break;
  default: Phase++;
  }
  return Phase;
}

/*****************************************************************************/

int TreeTestReceiveAndAttack(int (*ChoiceFunction)(float*,float*), 
		     float CollectDist, float DashPower, float ShotPower){

  static int _RECEIVED_ = 0;
  static int counter = 0;
  int ball_coming;

  switch(_RECEIVED_){
  case 0: 
    if ( fabs(rad_to_deg(Mem->GetGlobalAngle())) < .0001 ){  
      /* Position just got set, wait for RESET msg */
      if ( counter++ < 2 )
	break;
      else
	counter = 0;
    }

    /* Hack to see the other players with high quality while looking for ball */
    ball_coming = Mem->FindTeammateWithStatus(RECEIVER_STAT); /* The player to whom
							       the ball is coming */
    /* Just so the player can see with high quality     */
    /* Requires a line commented out from Lookforplayer */
    if( !ball_coming && !Mem->BallValid() )
      LOOKFORTEAMMATE(1);
    /* Hack to see the other players with high quality while looking for ball */

    else if ( RECEIVEPASS(ChoiceFunction, CollectDist, DashPower) == SUCCESS )
      _RECEIVED_ = 1;
    break;
  case 1: 
    int result = COLLECTANDKNOCK(LC_FLAG,&ChooseNN,CollectDist,DashPower,ShotPower);
    if ( result == FAILURE )
      _RECEIVED_ = 0;  /* Ball's out of range again */
    else if ( result == TRIED )
      SAYTOCOACH(KICKED_MSG); /* 0 is for coach */
    break;
  }
}

/*****************************************************************************/

int TreeReceiveAndKnock(int (*ChoiceFunction)(float*,float*), 
			float CollectDist, float DashPower, int Marker, 
			float ShotPower){

  #define _RECEIVEANDKNOCK_POSITIONING_ 0
  #define _RECEIVEANDKNOCK_RECEIVING_   _RECEIVEANDKNOCK_POSITIONING_+1
  #define _RECEIVEANDKNOCK_KNOCKING_    _RECEIVEANDKNOCK_RECEIVING_+1

  static int Phase = _RECEIVEANDKNOCK_POSITIONING_;

  int ball_coming;

  switch(Phase){
  case _RECEIVEANDKNOCK_POSITIONING_:
    if ( HOLDPOSITION(DashPower) == SUCCESS )
      Phase++;
    break;
  case _RECEIVEANDKNOCK_RECEIVING_:
    ball_coming = Mem->FindTeammateWithStatus(RECEIVER_STAT); /* The player to whom
							       the ball is coming */
    if( !ball_coming && !Mem->BallValid() )
      LOOKFORBALL;
    else if ( RECEIVEPASS(ChoiceFunction, CollectDist, DashPower) == SUCCESS )
      Phase++;  /* Received it */
    break;
  case _RECEIVEANDKNOCK_KNOCKING_: 
    int result = COLLECTANDKNOCK(Marker,&ChooseNN,CollectDist,DashPower,ShotPower);
    if ( result == FAILURE )
      Phase = _RECEIVEANDKNOCK_POSITIONING_; /* Ball's out of range again */
    break;
  }
}

/*****************************************************************************/

int StartSetPlay(int (*ReceiverChoiceFunction)(), 
		 float cdist, float pdist, 
		 float gpow, float dpow, float spow){

  #define _STARTSETPLAY_DRIBBLING_ 0
  #define _STARTSETPLAY_PASSING_   _STARTSETPLAY_DRIBBLING_ +1
  #define _STARTSETPLAY_RESET_     _STARTSETPLAY_PASSING_   +1

  static int Phase = _STARTSETPLAY_DRIBBLING_;
  static int HangCounter = 0;

  switch(Phase){
  case _STARTSETPLAY_DRIBBLING_:
    if ( !Mem->BallValid() || Mem->GetBallDistance() > cdist )
      HOLDPOSITION(gpow);                  /* Ball's gone   */
    else if ( !Mem->NumOpponentsWithin(pdist) ){ 
      /* Go to the ball and dribble at their goal */
      COLLECTANDDRIBBLE(THEIR_GOAL,&ChooseNN,cdist,gpow,dpow);
    }
    else{                                  /* Near defender */
      Phase++;
      HangCounter = 0;
      TreePass(ReceiverChoiceFunction,cdist,pdist,gpow,dpow,spow);
    }
    break;
  case _STARTSETPLAY_PASSING_:
    if ( TreePass(ReceiverChoiceFunction,cdist,pdist,gpow,dpow,spow) == 
	 _TREEPASS_RESET_ ){ /* Once more for Treepass to reset, then go on */
      TreePass(ReceiverChoiceFunction,cdist,pdist,gpow,dpow,spow);
      Phase++;
    }
    else if ( HangCounter++ > 20 ) {
	Phase = _STARTSETPLAY_DRIBBLING_;
    }	
    break;
  case _STARTSETPLAY_RESET_:
    if ( HOLDPOSITION(gpow) == SUCCESS )
      Phase = _STARTSETPLAY_DRIBBLING_;
    break;
  default: Phase++;
  }

  return Phase;
}

/*****************************************************************************/

#define FIND_BALL_TEST 0
int ChangePositionsAndAct (int (*ActFunction)(char*,int*),
			   float DashPower, float ShotPower){  
  
  /*** Should pass CollectDist into HoldPositionAndAct--do the 3 collectdist
    options here INSTEAD of there        ***/
  
  #define _CHANGEPANDA_LOOKING_   0
  #define _CHANGEPANDA_GOING_     _CHANGEPANDA_LOOKING_+1
  #define _CHANGEPANDA_RETURNING_ _CHANGEPANDA_GOING_  +1

  static int LastPhase = _CHANGEPANDA_LOOKING_;

  if ( Mem->PlayMode==BEFORE_KICK_OFF ) {
    for ( int i=1; i<=TEAM_SIZE; i++){
      if ( Mem->TeammateValid(i) && Mem->GetPlayerPosition(i) == UNKNOWN_POSITION 
	   && !int_random(10)){
	SAYTO(i,PING_MSG);
	return LastPhase;
      }
    }
  }

  if ( !Mem->SetPlay && Mem->GetMyActiveStatus() != INSETPLAY )
    Mem->UpdateActive(); 

#if FIND_BALL_TEST
  static int looking = 0;
#endif

  if ( !Mem->BallValid() ){ /* Just looking for the ball */
    if ( LastPhase != _CHANGEPANDA_LOOKING_ ) 
      LastPhase = _CHANGEPANDA_LOOKING_;
#if FIND_BALL_TEST
    looking = Mem->CurrentTime;
#endif
    FINDBALL;  
  }

#if 1
  else if (1){

#if FIND_BALL_TEST
    if (looking){
      printf("%d Found the ball in %d (%d)\n",Mem->MyNumber,
	     Mem->CurrentTime - looking, Mem->view_width);
      looking = 0;
    }
#endif

    HOLDPOSITIONANDACT(ActFunction,DashPower,ShotPower);
    return 0;
  }
#endif

  else if ( Mem->GetMyActiveStatus() == ACTIVE ){
    if ( HOLDPOSITIONANDACT(ActFunction,DashPower,ShotPower) != TRIED )
      my_error( "Should return TRIED");

    if ( LastPhase != _CHANGEPANDA_GOING_ ) {  /* Say after doing */
      /*************************************************************/
      /* Announce that I'm going to the ball if I wasn't last step */
      /* Others should free my position to fill                    */
      /*************************************************************/ 
      SAY(MY_BALL_MSG); 
      LastPhase = _CHANGEPANDA_GOING_;
    }
    else{
      ;/*if ( !int_random(4) )
	SAY(MY_BALL_MSG);      /* Gives far away players my position */
    }
  }
  else { /* ball is farther than CollectDist:  going to position */
    if ( LastPhase != _CHANGEPANDA_RETURNING_ ){
      /*************************************************************/
      /* Consider changing positions--announce if I do             */
      /* When I was chasing ball, see if someone else filled my    */
      /* position.  If not, go back, if so look for a new one      */
      /*************************************************************/
      if ( Mem->GetPositionPlayer(Mem->GetMyPosition()) != Mem->MyNumber ){
	/* Someone took my position   */
	Mem->SetMyPosition(GetNewPosition());  
      }
      ANNOUNCEMYPOSITION;  
      LastPhase = _CHANGEPANDA_RETURNING_;
    }
    else { /* I'm happy holding my position, but I need to consider if another
	      is better */
      int position;
      if ( (position = ConsiderNewPosition()) != UNKNOWN_POSITION ){
	Mem->SetMyPosition(position);
	ANNOUNCEMYPOSITION; 
      }
    }
    int result = HOLDPOSITIONANDACT(ActFunction,DashPower,ShotPower);

    if ( result != TRYING && result != SUCCESS )
      my_error( "Should return TRYING or SUCCESS");

  }
  return LastPhase;
}

/*****************************************************************************/
/*****************     Switch Function     ***********************************/
/*****************************************************************************/

void DoBehavior(int behavior){
  
  static int FIRST_TIME = TRUE;
  static int BEHAVIOR = 0;
  float cdist=0, sdist, pdist, gpow, dpow, spow, pbuf, x, y; 
  int Marker;
  int formation, position;

#if MOVE_LEARN
  if (Mem->MoveRecorded){
    RecordDashResult(Mem);
    Mem->MoveRecorded = FALSE;
  }
#endif

  switch (behavior){
  case FORGET: 
    Mem->Reset();
    break;
  case RANDOM_INTERCEPT:
    /*if ( Mem->view_width != NARROW_WIDTH )
      NARROW_HIGH_VIEW;*/
    /*    gpow = 6, cdist = 20;*/
    gpow = 50, cdist = 14;  /* Wait longer to go, but then go faster */
    if (BEHAVIOR != RANDOM_INTERCEPT) BEHAVIOR = RANDOM_INTERCEPT;
    BehaviorName = strdup("Random Intercept");
    COLLECTBALL( &ChooseRandom, cdist, gpow );
    break;
  case INTERCEPT:
    /*if ( Mem->view_width != NARROW_WIDTH )
      NARROW_HIGH_VIEW;*/
    NARROW_HIGH_VIEW;
    gpow = 50, cdist = 30;  /* Wait longer to go, but then go faster */
    if (BEHAVIOR != INTERCEPT) BEHAVIOR = INTERCEPT;
    BehaviorName = strdup("Intercept");
    /* TRAINCOLLECT( &ChooseNN, cdist, gpow ); */
    ANALYTICCOLLECTBALL( cdist, gpow ); 
    break;
  case SHOOT:
    if ( Mem->view_width != WIDE_WIDTH )  
      WIDE_HIGH_VIEW;
    /* usleep(1000 * SEND_STEP);  /* Wait a bit so defender sees ball */
    spow = 3; 
    if (BEHAVIOR != SHOOT) BEHAVIOR = SHOOT;
    BehaviorName = strdup("Shoot");
    /* STAYANDSHOOT( spow ); */
    SHOOTONGOAL( 100 );
    break;
  case CHASE_AND_SHOOT: 
    gpow = 6, spow = 4; 
    if (BEHAVIOR != CHASE_AND_SHOOT) BEHAVIOR = CHASE_AND_SHOOT;
    BehaviorName = strdup("ChaseAndShoot");
    CHASEANDSHOOT( gpow, spow );
    break;
  case DRIBBLE_AND_SHOOT: 
    gpow = 6, dpow = 6, spow = 4, sdist = 40; 
    if (BEHAVIOR != DRIBBLE_AND_SHOOT) BEHAVIOR = DRIBBLE_AND_SHOOT;
    BehaviorName = strdup("DribbleAndShoot");
    DRIBBLEANDSHOOT( sdist, gpow, dpow, spow );
    break;
  case DEFEND: 
    cdist = 20, gpow = 10, spow = 4;  
    if (BEHAVIOR != DEFEND) BEHAVIOR = DEFEND;
    BehaviorName = strdup("Defend");
    Defend( cdist, gpow, spow);
    break;
  case KAMRAM: 
    cdist = 20, gpow = 10, spow = 4;  
    if (BEHAVIOR != KAMRAM) BEHAVIOR = KAMRAM;
    BehaviorName = strdup("Kamram");
    Kamram( cdist, gpow, spow);
    break;
  case RANDOM_PASS: 
    if ( Mem->view_width != NORMAL_WIDTH )  
      NORMAL_HIGH_VIEW;
    cdist = 0, pdist = 35, gpow = 0, dpow = 10, spow = 3;  
    if (BEHAVIOR != RANDOM_PASS) BEHAVIOR = RANDOM_PASS;
    BehaviorName = strdup("RandomPass");
    TreePass( &ChooseReceiverRandom, cdist, pdist, gpow, dpow, spow);
    break;
  case LEARNED_PASS: 
    if ( Mem->view_width != NORMAL_WIDTH )  
      NORMAL_HIGH_VIEW;
    cdist = 0, pdist = 35, gpow = 0, dpow = 10, spow = 3;  
    if (BEHAVIOR != LEARNED_PASS) BEHAVIOR = LEARNED_PASS;
    BehaviorName = strdup("TreePass");
    TreePass( &ChooseReceiverTree, cdist, pdist, gpow, dpow, spow);
    break;
  case COLLECT_PASS:
    if ( Mem->view_width != NARROW_WIDTH )
      NARROW_HIGH_VIEW;
    /*    gpow = 6, cdist = 20;*/
    gpow = 10, cdist = 14;  /* Wait longer to go, but then go faster */
    spow = 100;
    if ( Mem->GetGlobalAngle() < 0 ) gpow = 0;   /* Don't run back after ball */
    if (BEHAVIOR != COLLECT_PASS) BEHAVIOR = COLLECT_PASS;
    BehaviorName = strdup("CollectPass");
    TreeTestReceiveAndAttack( &ChooseNN, cdist, gpow, spow);
    break;
  case INTERCEPT_PASS:
    if ( Mem->view_width != NARROW_WIDTH )
      NARROW_HIGH_VIEW;
    cdist = 14, gpow = 10, spow = 100;
    if ( Mem->GetGlobalAngle() > 0 ) gpow = 0;   /* Don't run back after ball */
    if (BEHAVIOR != INTERCEPT_PASS) BEHAVIOR = INTERCEPT_PASS;
    BehaviorName = strdup("InterceptPass");
    Marker = int_random(2) ? THEIR_GOAL : MY_GOAL;
    COLLECTANDKNOCK( Marker ,&ChooseNN, cdist, gpow, spow );
    break;
  case GOALIE:
    gpow = 50, pbuf = 2, spow = 5;
    formation = f433_FORMATION;
    position = f433_GOALIE;
    BehaviorName = strdup("Goalie");
    cdist = 14;
    break;
  case CENTER_SWEEPER:
    gpow = 50, pbuf = 3, spow = 5;
    formation = f433_FORMATION;
    position = f433_SWEEPER;
    BehaviorName = strdup("Sweeper");
    break;
  case RIGHT_DEFENSE:
    gpow = 50, pbuf = 3, spow = 5;
    formation = f433_FORMATION;
    position = f433_RIGHT_DEFENSE;
    BehaviorName = strdup("RightDefense");
    break;
  case CENTER_DEFENSE:
    gpow = 50, pbuf = 3, spow = 5;
    formation = f433_FORMATION;
    position = f433_CENTER_DEFENSE;
    BehaviorName = strdup("CenterDefense");
    break;
  case LEFT_DEFENSE:
    gpow = 50, pbuf = 3, spow = 5;
    formation = f433_FORMATION;
    position = f433_LEFT_DEFENSE;
    BehaviorName = strdup("LeftDefense");
    break;
  case RIGHT_MIDFIELD:
    gpow = 50, pbuf = 3, spow = 5;
    formation = f433_FORMATION;
    position = f433_RIGHT_MIDFIELD;
    BehaviorName = strdup("RightMidfield");
    break;
  case CENTER_MIDFIELD:
    gpow = 50, pbuf = 3, spow = 5;
    formation = f433_FORMATION;
    position = f433_CENTER_MIDFIELD;
    BehaviorName = strdup("CenterMidfield");
    break;
  case LEFT_MIDFIELD:
    gpow = 50, pbuf = 3, spow = 5;
    formation = f433_FORMATION;
    position = f433_LEFT_MIDFIELD;
    BehaviorName = strdup("LeftMidfield");
    break;
  case RIGHT_WING:
    gpow = 50, pbuf = 3, spow = 5;
    formation = f433_FORMATION;
    position = f433_RIGHT_WING;
    BehaviorName = strdup("RightWing");
    break;
  case CENTER_FORWARD:
    gpow = 50, pbuf = 3, spow = 5;
    formation = f433_FORMATION;
    position = f433_CENTER_FORWARD;
    BehaviorName = strdup("CenterForward");
    break;
  case LEFT_WING:
    gpow = 50, pbuf = 3, spow = 5;
    formation = f433_FORMATION;
    position = f433_LEFT_WING;
    BehaviorName = strdup("LeftWing");
    break;
  case START_SET_PLAY:
    x = -45, y = 0;
    pbuf = 10, cdist = 30, pdist = 25;
    gpow = 6, dpow = 6, spow = 3;
    if ( FIRST_TIME ){
      NORMAL_HIGH_VIEW;
      PREGAMEMOVE(x,y);
      FIRST_TIME = FALSE;
    }
    if (BEHAVIOR != START_SET_PLAY){
      Mem->SetCurrentHome(x,y);
      BEHAVIOR = START_SET_PLAY;
    }
    BehaviorName = strdup("StartSetPlay");
    StartSetPlay( &ChooseReceiverTree, cdist, pdist, gpow, 
		  dpow, spow);
    break;
  case RIGHT_MID_SET_PLAY:
    x = 0, y = -25, pbuf = 10;
    cdist = 14, gpow = 10, spow = 6;
    Marker = RF_FLAG;
    if ( FIRST_TIME ){
      NORMAL_HIGH_VIEW;
      PREGAMEMOVE(x,y);
      FIRST_TIME = FALSE;
    }
    if (BEHAVIOR != RIGHT_MID_SET_PLAY){
      Mem->SetCurrentHome(x,y);
      BEHAVIOR = RIGHT_MID_SET_PLAY;
    }
    BehaviorName = strdup("RightMidSetPlay");
    TreeReceiveAndKnock(&ChooseNN,cdist,gpow,Marker,spow);
    break;
  case LEFT_MID_SET_PLAY:
    x = 0, y = 25, pbuf = 10;
    cdist = 14, gpow = 10, spow = 6;
    Marker = LF_FLAG;
    if ( FIRST_TIME ){
      NORMAL_HIGH_VIEW;
      PREGAMEMOVE(x,y);
      FIRST_TIME = FALSE;
    }
    if (BEHAVIOR != LEFT_MID_SET_PLAY){
      Mem->SetCurrentHome(x,y);
      BEHAVIOR = LEFT_MID_SET_PLAY;
    }
    BehaviorName = strdup("LeftMidSetPlay");
    TreeReceiveAndKnock(&ChooseNN,cdist,gpow,Marker,spow);
    break;
  case RIGHT_WINGER_SET_PLAY:
    x = 40, y = -30;
    pbuf = 10, cdist = 15, pdist = 35;
    gpow = 10, dpow = 6, spow = 3;
    if ( FIRST_TIME ){
      NORMAL_HIGH_VIEW;
      PREGAMEMOVE(x,y);
      FIRST_TIME = FALSE;
    }
    if (BEHAVIOR != RIGHT_WINGER_SET_PLAY){
      Mem->SetCurrentHome(x,y);
      BEHAVIOR = RIGHT_WINGER_SET_PLAY;
    }
    BehaviorName = strdup("RightWingerSetPlay");
    StartSetPlay( &ChooseReceiverTree, cdist, pdist, gpow, 
		  dpow, spow);
    break;
  case LEFT_WINGER_SET_PLAY:
    x = 40, y = 30;
    pbuf = 10, cdist = 15, pdist = 35;
    gpow = 10, dpow = 6, spow = 3;
    if ( FIRST_TIME ){
      NORMAL_HIGH_VIEW;
      PREGAMEMOVE(x,y);
      FIRST_TIME = FALSE;
    }
    if (BEHAVIOR != LEFT_WINGER_SET_PLAY){
      Mem->SetCurrentHome(x,y);
      BEHAVIOR = LEFT_WINGER_SET_PLAY;
    }
    BehaviorName = strdup("LeftWingerSetPlay");
    StartSetPlay( &ChooseReceiverTree, cdist, pdist, gpow, 
		  dpow, spow);
    break;
  case FINISH_SET_PLAY:
    x = 35, y = 0, pbuf = 10;
    cdist = 14, gpow = 10, spow = 6;
    Marker = THEIR_GOAL;
    if ( FIRST_TIME ){
      NORMAL_HIGH_VIEW;
      PREGAMEMOVE(x,y);
      FIRST_TIME = FALSE;
    }
    if (BEHAVIOR != FINISH_SET_PLAY){
      Mem->SetCurrentHome(x,y);
      BEHAVIOR = FINISH_SET_PLAY;
    }
    BehaviorName = strdup("FinishSetPlay");
    TreeReceiveAndKnock(&ChooseNN,cdist,gpow,Marker,spow);
    break;
  case JUSTIN_KICK:
    break; /* Action triggered by message from coach */
  case JUSTIN_COLLECT:
    break; /* Action triggered by message from coach */
  }

#if USE_RL
  #define ACTFUNC &UseRL
#else
/*#define ACTFUNC &RandomOccupied*/
/*#define ACTFUNC &WorkTheWings*/
/*#define ACTFUNC &HighestDTOutput */
#define ACTFUNC &ConsiderDTOutput
#endif

  switch(behavior){
  case GOALIE:
  case CENTER_SWEEPER:
  case RIGHT_DEFENSE:
  case CENTER_DEFENSE:
  case LEFT_DEFENSE:
  case RIGHT_MIDFIELD:
  case CENTER_MIDFIELD:
  case LEFT_MIDFIELD:
  case RIGHT_WING:
  case CENTER_FORWARD:
  case LEFT_WING:
    if ( FIRST_TIME ){
      NORMAL_HIGH_VIEW;
      /* formation = f352_FORMATION;         */
      formation = f442_FORMATION;         
      /* formation = f72_FORMATION;          */
      /* formation = f334_FORMATION;         */
      /* formation = f244_FORMATION;         */
#if PRACTICE
      formation = f433_FORMATION;
#endif
      Mem->SetFormation(formation,Mem->CurrentTime);
      Mem->ChangePositions = FALSE; 
      Mem->HomeChangeMethod = HOME_SHIFT; 
      Mem->UseSetPlays = FALSE; 
      Mem->UseUnitCoaching = FALSE;  /* makes marking happen */
      Mem->EchoSounds = FALSE;
      FIRST_TIME = FALSE;
    }
    if (BEHAVIOR != behavior){
      Mem->SetMyPosition(position);
      BEHAVIOR = behavior;
    }

    /* ReactToScoreAndTime(); */
    /* SwitchRightLeft();     */
    /* SinglePlayerSwitch();  */

    if ( Mem->PlayMode == BEFORE_KICK_OFF ){
      if( Mem->InKickoffPosition() ){
	Mem->CheckForPlayersInMyPosition();
      }
      else{  /* Not in kickoff position */

	Mem->Reset();  /* Erases memory, active status, currenthome, etc. */
	
	position = Mem->GetMyPosition();

	float goalX,goalY;
	if ( Mem->SetPlay && Mem->MyPositionInSetPlay() )
	  Mem->GetMySetPlayXY(&goalX,&goalY);
	else if ( Mem->BallValid() ){
	  Mem->UpdateHome();
	  goalX = Mem->GetCurrentHomeX();
	  goalY = Mem->GetCurrentHomeY();
	}
	else{
	  Mem->GetMyPositionXY(&goalX,&goalY);
	}
	
	PREGAMEMOVE(goalX,goalY);
	
	WAITFORSIGHT; /* Wait until in the actual position */
	if ( Mem->InKickoffPosition() )
	  ANNOUNCEMYPOSITION;
      }
    }

    if ( Mem->UseSetPlays && Mem->PlayMode != PLAY_ON ){
      if ( !Mem->SetPlay )
	Mem->InitializeSetPlay();
      else if ( Mem->MyPositionInSetPlay() )  /* There's a set play on */
	Mem->SetMyselfInSetPlay();
      else                                    /* I'm not in it         */
	Mem->SetMyselfInactive();
    }
    else 
      Mem->SetPlay = FALSE;  /* Should already be in SetPlay ActiveStatus (if in play) */

#if 0
    int ChangeTime = 18000*6;
    int NumFormations = 6;
    if ( Mem->CurrentTime>10 && (Mem->CurrentTime)%ChangeTime < 7){
      int newFormationType = ((Mem->CurrentTime/ChangeTime)%NumFormations)+1;
      Mem->SetFormation(newFormationType,Mem->CurrentTime);
    }
    /* Mem->HomeChangeMethod = HOME_OBEY; */
#endif

#if 1
    /* Asssuming we always want high quality view if we're not in a tight loop */
    /* Corrects for possible missed high_quality command                       */
    if (Mem->BallValid()){
      float bdist = Mem->GetBallDistance();
      if ( bdist > 50 )
	WIDE_HIGH_VIEW;
      else if (bdist > 25)
	NORMAL_HIGH_VIEW;
      else
	NARROW_HIGH_VIEW;
    }
#endif

    /* if ( strcmp(Mem->GetMyStatus(),BLANK_STAT) )
      printf("%d:%d status== %s\n",Mem->MyNumber,Mem->CurrentTime,Mem->GetMyStatus());*/

#if PRACTICE 
    Mem->ChangePositions = FALSE;
    Mem->HomeChangeMethod = HOME_OBEY;
    Mem->SetPlay = FALSE;
#endif

#if 0
      Mem->HomeChangeMethod = HOME_OBEY; 
      Mem->SetPlay = FALSE;
#endif

    ChangePositionsAndAct( ACTFUNC, gpow, spow );

    if ( Mem->UseUnitCoaching && !Mem->NewSight ){
      /* Do captainly stuff if there's time left over */
      for (int i=0; i<Mem->GetCurrentFormation()->GetNumUnits(); i++){
	if ( Mem->AmUnitCaptain(i) )
	  DirectUnit(i);
      }
    }

    if ( Mem->EchoSounds && !Mem->NewSight ){
      if ( !int_random(1000) )
	ECHOTHEIRSOUND;
    }

    if ( Mem->QActionTaken )
      Mem->LookForRewards();

    break;
  }
}
