/* act.c
 * CMUnited-97 (soccer client for Robocup-97)
 * Peter Stone <pstone@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1997 Peter Stone
 *
 * CMUnited-97 was created by Peter Stone and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

#include "global.h"

/* Everything in here should return SUCCESS, TRIED, TRYING, or FAILURE = 0 */
/* SUCCESS = positive of success, failure = positive of failure, 
   TRYING = I need more cycles, TRIED = pretty sure of success */

#define COLLECT_DEBUG         0
#define CLEAR_DEBUG           0
#define DRIBBLE_DEBUG         0
#define CHOICE_FUNCTION_DEBUG 0

#define TUNED_PARAMS 0
#if TUNED_PARAMS
/* playerSpeed, timeMultiplier1, timeMultiplier2, dashTwiceBuffer,
   facePointBuffer, DashPower, GoToPointBuffer */
/* Backwards movement discouraged */
float TunedParams[7] = { 0.590, 1.165, 1.0,   100.0, 9.191, 53.5, 0.87 };
/* When back was allowed */
//float TunedParams[7] = { 0.591, 1.448, 1.058,   100.0, 0.76, 50.0, 0.87 };
#endif

/*****************************************************************************/
/************        LOW LEVELS       ****************************************/
/*****************************************************************************/

/*****************************************************************************/

int FaceBall(){
  
  if ( !Mem->BallAngleValid() ) {
    return FAILURE;
  }
  else if ( Mem->GetBallAngle() ) {           /* If not facing ball*/
    TURN(Mem->GetBallAngle());                /* Face it           */
    return TRIED;
  }
  else if ( Mem->BallAngleValid() == 1 ) {    /* Sure I'm facing   */
    return SUCCESS;
  }
  else              /* Think I'm facing the ball, but not positive */
     return TRIED;
}
/*#define FACEBALL FaceBall()*/

/*****************************************************************************/

int FaceMarker(int marker ){

  if ( !Mem->MarkerValid(marker) ) {
    return FAILURE;
  }
  else if ( Mem->GetMarkerAngle(marker) ) {           /* If not facing marker*/
    TURN(Mem->GetMarkerAngle(marker));                /* Face it           */
    return TRIED;
  }
  else if ( Mem->MarkerValid(marker) == 1 ) {         /* I see it in front */
    return SUCCESS;
  }
  else                    /* Think I'm facing the marker, but not positive */
     return TRIED;
}
/*#define FACEMARKER(a) FaceMarker(a)*/

/*****************************************************************************/

int FacePlayer(char side, int player ){

  if ( !Mem->PlayerAngleValid(side, player) ) {
    return FAILURE;
  }
  else if ( Mem->GetPlayerAngle(side, player) ) {  /* If not facing player*/
    TURN(Mem->GetPlayerAngle(side, player));       /* Face it           */
    return TRIED;
  }
  else if ( Mem->PlayerValid(side, player) == 1 ) {/* I see it in front */
    return SUCCESS;
  }
  else                   /* Think I'm facing the player, but not positive */
     return TRIED;
}
/*#define FACEPLAYER(a,b) FacePlayer(a,b)
#define FACETEAMMATE(a) FacePlayer(Mem->MySide,a)
#define FACEOPPONENT(a) FacePlayer(Mem->TheirSide,a)*/

/*****************************************************************************/

int FacePoint(float x, float y, float buffer){

  float dx = x - Mem->GetGlobalX();
  float dy = y - Mem->GetGlobalY();
  float angle = my_atan2(dy,dx);
  float da = rad_to_deg(Mem->GetGlobalAngle() - angle);
  CleanAngle(&da);

#if 0
  printf("dx: %f,dy: %f,angle: %f, da: %f\n",dx,dy,180*angle/M_PI,da);
#endif

  if ( fabs(da) < buffer )                               /* <1 is close enough */
    return SUCCESS;

  int viewAngle = Mem->GetViewAngle()/2;

  if ( fabs(da) > viewAngle && Mem->Dashed ){       /* If it's a big turn and*/
    int counter = 0;                                /* I was moving, verify  */
    
    if ( Mem->PlayMode != PLAY_ON )
      counter = 10; /* No rush, don't want to look silly */
    else
      if ( Mem->view_width != NARROW_WIDTH ) 
	LOW_QUAL_VIEW;                    /* not so much advantage if narrow */

    while ( fabs(da) > viewAngle ) {

      if (counter++ > 10){
	  if (counter > 20)
	    return FAILURE;

	HIGH_QUAL_VIEW;  /* I must be confused about my position */
	dx  = x - Mem->GetGlobalX();   /* reevaluate it          */
	dy  = y - Mem->GetGlobalY();
	angle = my_atan2(dy,dx);
      }

      da = rad_to_deg(Mem->GetGlobalAngle() - angle);
      CleanAngle(&da);

#if 0
      if ( (counter > 10 && Mem->PlayMode == PLAY_ON) 
	   || counter > 15 )
	printf("%2d:%d (%d) counter = %d Facing %.1f (%.1f %.1f). Now at %.1f (%.1f %.1f). Turning  %.1f\n",
		 Mem->MyNumber,Mem->CurrentTime,Mem->PlayMode,counter,
		 rad_to_deg(angle),x,y,
		 rad_to_deg(Mem->GetGlobalAngle()),Mem->GetGlobalX(),Mem->GetGlobalY(),da);
#endif

      TURN(da);
      WAITFORSIGHT;
    }
    HIGH_QUAL_VIEW;
  }
  if ( fabs(da) > buffer )                             /* <1 is close enough */
    TURN(da);

  return TRIED;
}
/*#define FACEPOINT(a,b) FacePoint(a,b,1)*/

/*****************************************************************************/

int LookForBall(char dir){
  /* Moving first, pinging only if time is left */
  if ( !(Mem->BallAngleValid()) ) {       /* If Can't see ball */
#if 0    
    float ang;
    if ((ang=Mem->GetBallAngle()) > Mem->GetViewAngle()/2){
      TURN(ang);                      /* Look for where ball was last */
      printf("remembering ball\n");
    }
    else
#endif
      LOOK(dir);                                 /* Look for it         */ 

    WAITUNTILFREETOACT;
    if ( !Mem->NewSight ){         /* Don't talk when action's critical */
      SAY(PING_BALL_MSG); 
    }  
    return TRYING;
  }
  else 
    return FACEBALL;               /* SUCCESS if sure, else TRIED       */
}
/*#define LOOKFORBALL LookForBall('r')*/

/*****************************************************************************/

/* Don't return until ball is found */
int FindBall(){
  char dir = (int_random(2) ? 'r' : 'l');
  int oldWidth = Mem->view_width;

  int newWidth;
  if (oldWidth == NARROW_WIDTH)
    newWidth = NORMAL_WIDTH;
  else 
    newWidth = oldWidth;

#if 0
  newWidth = oldWidth;  /* Allows you to look at narrow, low */
#endif

  if ( !Mem->BallAngleValid() ) {
    CHANGE_VIEW(newWidth,LOW_QUALITY);
    SAY(PING_BALL_MSG);
    WAITUNTILFREETOACT;
  }

  while ( !Mem->BallAngleValid() ) {
    LOOK(dir);
    WAITFORSIGHT;
  }

  CHANGE_VIEW(oldWidth,HIGH_QUALITY);
  WAITUNTILFREETOACT;
  if ( !Mem->BallAngleValid() ){ /* The ball just moved again          */
    return FindBall(); 
  }
  return FACEBALL;              /* SUCCESS if sure, else TRIED        */
}
/* #define FINDBALL FindBall() */

/*****************************************************************************/

int LookForMarker(int marker, char dir ){
  
  if ( !(Mem->MarkerValid(marker)) ) {     /* If Can't see marker */
    LOOK(dir);                             /* Look for it         */ 
    return TRYING;
  }
  else if (Mem->GetMarkerAngle(marker)) {  /* If not facing marker*/
    TURN(Mem->GetMarkerAngle(marker));     /* Face it             */
    return TRIED;
  }
  else if ( Mem->MarkerValid(marker) == 1 ){/* Sure you're facing */
    return SUCCESS;
  }
  else{        /* Think you're facing the marker but not positive */
    return TRIED;
  }
}
/*#define LOOKFORMARKER(a) LookForMarker(a,'r')*/

/*****************************************************************************/

int LookForPlayer(char side, int player, char dir ){
  
  if ( !(Mem->PlayerAngleValid(side, player)) ) {/* If Can't see player */
    if ( side == Mem->MySide ){
      SAYTO(player,PING_MSG);                      /* Ping the player     */
      WAITUNTILFREETOACT;
    }
    if ( !(Mem->PlayerAngleValid(side, player)) ) {/* check again       */
#if 0
      float ang;
      if ((ang=Mem->GetPlayerAngle(side,player)) > 30){
	TURN(ang);                      /* Look for where player was last */
	printf("remembering player\n");
      }
      else{
#endif
	LOOK(dir);                                 /* Look for it         */ 
    /*}*/
      return TRYING;
    }
  }

  /* Player angle is valid */
  if (Mem->GetPlayerAngle(side, player)) {  /* If not facing player*/
    TURN(Mem->GetPlayerAngle(side, player));     /* Face it             */
    return TRIED;
  }
  else if ( Mem->PlayerAngleValid(side, player) == 1 ){/* Sure you're facing */
    return SUCCESS;
  }
  else{              /* Think you're facing the player but not positive */
    return TRIED;
  }
}
/*#define LOOKFORPLAYER(a,b) LookForPlayer(a,b,'r')
#define LOOKFORTEAMMATE(a) LookForPlayer(Mem->MySide,a,'r')
#define LOOKFOROPPONENT(a) LookForPlayer(Mem->TheirSide,a,'r')*/

/*****************************************************************************/

/* Don't return until ball is found */
int FindPlayer(char side, int player){
  char dir = (int_random(2) ? 'r' : 'l');

  if ( !Mem->PlayerAngleValid(side,player) ) {
    PINGPLAYER(side,player);
    WAITUNTILFREETOACT;
  }

  int counter=0;
  int limit= 360/Mem->GetViewAngle() +1;  /* Turn more than 360 degrees */
  while ( !Mem->PlayerAngleValid(side,player) ) {
    if (counter++ > limit)
      return FAILURE;
    LOOK(dir);
    WAITFORSIGHT;
  }
  return TRIED;  /* Know where it is, but not facing */
}
/* #define FINDPLAYER(a,b) FindPlayer(a,b) 
#define FINDTEAMMATE(a) FindPlayer(Mem->MySide,a) 
#define FINDOPPONENT(a) FindPlayer(Mem->TheirSide,a) 
*/

/*****************************************************************************/

int Move(float dist, float ang, float dpow, int MinDashTimes, int MaxDashTimes ){

  float dodgeBuffer = 35;

#if 0
  /* Avoid getting stuck going to ball for a setplay */
  if ( Mem->GetMyActiveStatus() == INSETPLAY &&
       Mem->MyPositionInSetPlay() &&
       Mem->GetMySetPlayPositionType() == SETPLAY_STARTER )
    dodgeBuffer = 80;
#endif

  /* check to which side the player is (players within 22 of -22 or 22)      */
  /* Don't need to dodge if the point I'm going to is closer than the player */
  if ( Mem->NumPlayersWithin(MIN(3,dist),dodgeBuffer,0,ang-dodgeBuffer) && 
       Mem->PlayMode != BEFORE_KICK_OFF){ 
    DODGEPLAYER(ang+90,100,-135);     
    return TRYING;
  }
  else if ( Mem->NumPlayersWithin(MIN(3,dist),dodgeBuffer,0,ang+dodgeBuffer) && 
	    Mem->PlayMode != BEFORE_KICK_OFF){
    DODGEPLAYER(ang-90,100,135);     
    return TRYING;
  }
  else {
    if ( dist<15 || fabs(ang)>3 ){ /* don't turn if far */
      TURN(ang);
      WAITUNTILFREETOACT;
    }
    GO(dist,dpow,MinDashTimes,MaxDashTimes);
    return TRIED;
  }
}
/*
#define DASHMOVE(a,b)   Move(0,a,b,1,2)
#define GOMOVE(a,b,c,d,e) Move(a,b,c,d,e)
*/

/*****************************************************************************/

/* was 160, 90 */
#define CIRCLE_ANGLE 130
#define CIRCLE_POWER 70

float NeedToCircle(float GoalAngle ){

  float BallAngle = Mem->GetBallAngle();
  float BGdiff = GoalAngle-BallAngle;
  CleanAngle(&BGdiff);
  if ( fabs(BGdiff) > CIRCLE_ANGLE )
    return BGdiff;
  else
    return 0;
}
/*#define NEEDTOCIRCLE(a) NeedToCircle(a)*/

/*****************************************************************************/

int Shoot(float kpow, float ang, int shootTimes, int trap, int hard_shoot){

  if ( !Mem->BallValid() || Mem->GetBallDistance() > KICK_DISTANCE )
    return TRYING;  /* should be failure */  /* Ball's not in range*/

#if PRACTICE
  shootTimes = 1;
  trap = 0;
  hard_shoot = 0;
#endif

  switch( Mem->GetCurrentFormationType() ){
  case fRT_FORMATION: 
  case fLT_FORMATION: 
    hard_shoot=0;
    shootTimes=1;
    trap=0;
    break;
  }

  float BGdiff;
  if ( (BGdiff = NEEDTOCIRCLE(ang)) ){         /* Trying to kick backwards */
    CIRCLEBALL(CIRCLE_POWER,BGdiff);           /* Move around the ball     */
    return TRYING;
  }
  else {
    if (trap && Mem->PlayMode == PLAY_ON){     /* Not when ball's out      */
      TRAPBALL; 
    }
    if (hard_shoot){
      KICK(30,ang);
    }
    for (int i=0; i<shootTimes; i++){
      KICK(kpow,ang);
      WAITUNTILFREETOACT;
      if ( Mem->NewSight ) 
	break;     /* Go back for the new sight */
    }
    return TRIED;                               /* Kicked             */
  }
}
/*
#define SHOOTONGOAL(a) Shoot(a,Mem->GetMarkerAngle(THEIR_GOAL),MAX_ACTS,1,1)
#define SHOOTATMARKER(a,b) Shoot(b*Mem->GetMarkerDistance(a),Mem->GetMarkerAngle(a),MAX_ACTS,1,0)
#define SHOOTATANGLE(a,b) Shoot(a,b,MAX_ACTS,0,0)
#define DRIBBLEATMARKER(a,b) Shoot(b,Mem->GetMarkerAngle(a),1,0,0)
#define DRIBBLEATTEAMMATE(a,b) Shoot(b,Mem->GetPlayerAngle(Mem->MySide,a),1,0,0)
#define DRIBBLEATOPPONENT(a,b) Shoot(b,Mem->GetPlayerAngle(Mem->TheirSide,a),1,0,0)
#define PASSTOTEAMMATE(a,b) Shoot(b*Mem->GetPlayerDistance(Mem->MySide,a),Mem->GetPlayerAngle(Mem->MySide,a),MAX_ACTS,1,0)
#define EASYPASSTOTEAMMATE(a,b) Shoot(b*Mem->GetPlayerDistance(Mem->MySide,a),Mem->GetPlayerAngle(Mem->MySide,a),MAX_ACTS,0,0)
/* If don't know player's distance 
#define PASSTOWARDTEAMMATE(a,b) Shoot(b,Mem->GetPlayerAngle(Mem->MySide,a),MAX_ACTS,1,0)
#define EASYPASSTOWARDTEAMMATE(a,b) Shoot(b,Mem->GetPlayerAngle(Mem->MySide,a),MAX_ACTS,0,0)
*/

/*****************************************************************************/

#define HARD_SHOT_DIST  15
int ShootAtPoint(float kpow, float x, float y, int trap_ball, int hard_shot){

  float dx = x - Mem->GetGlobalX();
  float dy = y - Mem->GetGlobalY();
  float dist = sqrt(dx*dx + dy*dy);
  float angle = my_atan2(dy,dx);
  float da = rad_to_deg(Mem->GetGlobalAngle() - angle);

#if 0
  printf("(shoot) dx: %f,dy: %f,angle: %f, da: %f\n",dx,dy,180*angle/M_PI,da);
#endif

  if ( dist < HARD_SHOT_DIST ){
    hard_shot = FALSE;
    trap_ball = FALSE;
  }
    
  return Shoot(kpow,da,MAX_ACTS,trap_ball,hard_shot);
}
/* 
#define SHOOTATPOINT(a,b,c) ShootAtPoint(a,b,c,0,1)
#define BLASTATPOINT(a,b) ShootAtPoint(100,a,b,0,0)
#define SHOOTONGOALATCORNER(a) ShootAtPoint(a,X0,(GOAL_WIDTH/2 - 3) * (int_random(2) ? 1 : -1),0,1) 
#define SHOOTATNEARPOST(a) ShootAtPoint(a,X0,(GOAL_WIDTH/2 -3) * (Mem->GetGlobalY()>0 ? 1 : -1),0,1)
#define SHOOTATFARPOST(a) ShootAtPoint(a,X0,(GOAL_WIDTH/2 -3) * (Mem->GetGlobalY()>0 ? -1 : 1),0,1)
*/

/*****************************************************************************/

#define CHECK_CLEAR_DIST  12  /* There should be no players closer than this dist */
#define CHECK_CLEAR_ANGLE 10  /* whithin this ang of the ball's trajectory        */
#define MAX_CLEAR_OFFSET  90  /* Don't consider angles this much different        */
#define SUPER_DANGER_DIST 10  /* When this close to own goal, clear sideways      */

int ClearTowards(float kpow, float ang){

  if ( !Mem->BallValid() || Mem->GetBallDistance() > KICK_DISTANCE )
    return FAILURE;                             /* Ball's not in range*/

  if ( Mem->FacingBackNearOwnGoal() /*FacingBackInOwnPA()*/ 
       || Mem->InOwnPenaltyArea() ){  
    /* If in goal mouth, just get it out */
    float clearSide = Mem->GetMyLocationSide();

    switch( Mem->GetCurrentFormationType() ){
    case fRT_FORMATION: 
      //clearSide = RIGHT; break;
      clearSide = 0; break;
    case fLT_FORMATION: 
      //clearSide = LEFT; break;
      clearSide = 0; break;
    }

    float Xtarget = 0;
    if ( Mem->GetMarkerDistance(MY_GOAL) < SUPER_DANGER_DIST )
      Xtarget = -X0/2;
    ang = Mem->AngleToPoint(Xtarget, Y0*clearSide); /* Clear to center flag           */
    float startAng = ang;
#if CLEAR_DEBUG  
  printf("%d:%d  want to clear to point (%.1f %.1f)\n",
	 Mem->MyNumber, Mem->CurrentTime,Xtarget,Y0*clearSide);
#endif
    while ( NEEDTOCIRCLE(ang) ){                 /* Don't want to circle              */
      if ( startAng > 0 )
	ang -= 5;
      else
	ang += 5;
    }
  }

  float clearAng = ang;
  float clearOffset = 0;

#if CLEAR_DEBUG  
  printf("%d:%d  want to clear %.1f\n",Mem->MyNumber, Mem->CurrentTime,ang);
  printf("       %d in way\n",Mem->NumPlayersWithin(CHECK_CLEAR_DIST/2, CHECK_CLEAR_ANGLE,
 						    CHECK_CLEAR_DIST/2, clearAng));
#endif

  while ( Mem->NumPlayersWithin(CHECK_CLEAR_DIST/2, CHECK_CLEAR_ANGLE, 
				CHECK_CLEAR_DIST/2, clearAng)         ){
    clearOffset *= -1;                     /* Flip offset angle              */
    if (clearOffset >= 0)                  /* increment it if even iteration */
      clearOffset += 2*CHECK_CLEAR_ANGLE;
    
    if ( clearOffset >= MAX_CLEAR_OFFSET ){
      clearAng = ang;
      break;
    }
    else
      clearAng = ang + clearOffset;
    
    CleanAngle(&clearAng);

#if CLEAR_DEBUG
    printf("  ---  checking %.1f\n",clearAng);
    printf("     %d in way\n",Mem->NumPlayersWithin(CHECK_CLEAR_DIST/2, CHECK_CLEAR_ANGLE,
						     CHECK_CLEAR_DIST/2, clearAng));
#endif

  }

  return SHOOTATANGLE(kpow, clearAng);
}
/*
#define CLEARTOWARDSMARKER(a,b) ClearTowards(b*Mem->GetMarkerDistance(a),Mem->GetMarkerAngle(a))
#define CLEARTOWARDSPOINT(a,b,c) ClearTowards(c*Mem->DistanceToPoint(a,b),Mem->AngleToPoint(a,b))
*/

/*****************************************************************************/

int CollectStoppedBall(float CollectDist, float DashPower){

 if ( !Mem->BallValid() ) {        /* If Can't see ball */
    FINDBALL;                       /* Look for it       */ 
    WAITUNTILFREETOACT;
  }
  if ( !Mem->BallValid() ) {
    return FAILURE;
  }
  else if ( (Mem->PlayMode==THEIR_KICK_OFF ||  
	     Mem->PlayMode==THEIR_KICK_IN ||  
	     Mem->PlayMode==THEIR_GOAL_KICK ||  
	     Mem->PlayMode==THEIR_FREE_KICK ||
	     Mem->PlayMode==THEIR_CORNER_KICK)  &&
	    Mem->GetBallDistance() < 10 ) {         /* If can't go closer  */
    return SUCCESS;                                 /* Don't waste stamina */
  }
  else if ( Mem->GetBallDistance() > CollectDist) { /* If Far from ball  */
    FACEBALL;                                       /* wait              */
    return TRYING;
  }
  else if ( Mem->GetBallDistance() > KICK_DISTANCE - .2 )  { /* If it's in range  */
    GOMOVE(Mem->GetBallDistance(),Mem->GetBallAngle(),DashPower*Mem->GetBallDistance(),
	   1,2);
    return TRYING;
  }
  else if ( Mem->BallValid() == 1 )  {
    return SUCCESS;                             /* Sure you're at it */
  }
  else {
    FACEBALL;
    return TRIED;                               /* Not sure          */
  }
}
/*#define COLLECTSTOPPEDBALL(a) CollectStoppedBall(150,a)*/

/*****************************************************************************/

int CollectBall (int (*ChoiceFunction)(float*,float*), 
		 float CollectDist, float DashPower){

#if PRACTICE
  if (0) /* Don't use analytic method */
#endif
  if ( Mem->PlayMode == PLAY_ON )
    return ANALYTICCOLLECTBALL(CollectDist,DashPower); 

  float TurnAng = 0;        /* By default, just go straight at ball */
  
  if ( !Mem->BallValid() ) {        /* If Can't see ball */
    FINDBALL;                       /* Look for it       */ 
    WAITUNTILFREETOACT;
  }

#if 0

    float bdist = Mem->GetBallDistance();
    float ang  = Mem->GetBallAngle();
    float relvelmag  = Mem->GetBallRelVelMagnitude();
    float relvelang  = Mem->GetBallRelVelDirection();  /* My calculation  */
    float velmag     = Mem->GetBallVelMagnitude();     /* from the server   */
    float velang     = Mem->GetBallVelDirection();     /* In relative terms */
    float ballvalid  = Mem->BallValid();
    float relvelvalid = Mem->BallRelVelocityValid();
    int   velvalid   = Mem->BallVelValid();
    float ballx       = Mem->GetBallGlobalX();
    float bally       = Mem->GetBallGlobalY();
    float trajvalid   = Mem->BallTrajectoryValid();
    float trajdir,trajmag;
    if (trajvalid){
      trajdir     = rad_to_deg(Mem->GetBallTrajectoryAbsoluteDirection());
      trajmag     = Mem->GetBallTrajectoryMagnitude();
    }
    
    printf ("%d:%d me(%.1f) (%.1f,%.1f %.1f)   ball(%.1f): (%.1f %.1f)   global: (%.1f,%.1f)\n",
	    Mem->MyNumber,Mem->CurrentTime,
	    Mem->MarkerValid(THEIR_GOAL), Mem->GetGlobalX(),Mem->GetGlobalY(),
	    rad_to_deg(Mem->GetGlobalAngle()),
	    ballvalid,bdist,ang,  ballx,bally);
    if (trajvalid && trajmag < 5) /* else a jump */
      printf ("  **** traj conf = (%.1f)    dir = %.1f   mag = %.1f **** \n",
	      trajvalid,trajdir,trajmag);
    
    /*
      printf ("%d(%.1f,%.1f)  ball(%.1f): (%.1f %.1f)   relvel(%.1f): (%.1f %.1f)   vel(%d): (%.1f %.1f)\n",
      Mem->CurrentTime,Mem->GetGlobalX(),Mem->GetGlobalY(),
      ballvalid,dist,ang,  relvelvalid,relvelmag,relvelang,  velvalid,velmag,velang);
      */
#endif

  if ( !Mem->BallValid() ) {
    return FAILURE;
  }
  else if ( (Mem->PlayMode==THEIR_KICK_OFF ||  
	     Mem->PlayMode==THEIR_KICK_IN ||  
	     Mem->PlayMode==THEIR_GOAL_KICK ||  
	     Mem->PlayMode==THEIR_FREE_KICK ||
	     Mem->PlayMode==THEIR_CORNER_KICK)  &&
	    Mem->GetBallDistance() < 10 ) {         /* If can't go closer  */
    return SUCCESS;                                 /* Don't waste stamina */
  }
  else if ( Mem->GetBallDistance() > CollectDist) { /* If Far from ball  */
    FACEBALL;                                       /* wait              */
    return TRYING;
  }
  else if ( Mem->GetBallDistance() > KICK_DISTANCE )  { /* If it's in range  */

#if CHOICE_FUNCTION_DEBUG
    if ( ChoiceFunction == &ChooseNN )
      printf("Player %d is trying to use the NN --- ",Mem->MyNumber);
#endif

#if 0
    float tmp1,tmp2;
    if ( /* (*ChoiceFunction)( &tmp1, &tmp2) == WAIT || */
	 Mem->BallValid() < 1 || !Mem->BallRelVelocityValid() || 
	 /* Really want to do this if I moved on last step (Dashed) */
	 Mem->GetBallDistance() < 6 || Mem->GetBallDistance() > NN_CDIST+10 || 
	 Mem->Dashed || Mem->GetBallVelDirection() == NODIR ||
	 !Mem->GetBallVelMagnitude() || 
	 fabs(Mem->GetBallVelDirection()) < 135 || 
	 Mem->GetBallRelVelMagnitude() < NN_MIN_VEL)/* Moving too slowly for NN */
      ChoiceFunction = &ChooseStraight;             /* Only use nets when ball
						       is approaching and near*/
#else
    if ( Mem->BallValid() < 1 || !Mem->BallTrajectoryValid() || 
	 Mem->GetBallTrajectoryMagnitude() > 5 || 
	 Mem->GetBallTrajectoryMagnitude() < .3)
      ChoiceFunction = &ChooseStraight;  
    else 
      ChoiceFunction = &ChooseAnalytic;        
#endif

#if CHOICE_FUNCTION_DEBUG
    printf("relvelvalid:  %.1f -- RelVelMag:  %.1f -- ",
	   Mem->BallRelVelocityValid(),Mem->GetBallRelVelMagnitude());
    if ( ChoiceFunction == &ChooseNN )
      printf("YES using it(%.1f)\n",Mem->GetBallVelDirection());
    else
      printf("NOT using it\n");
#endif
    if ( ((*ChoiceFunction)( &TurnAng, &DashPower) == TRY) &&
	 Mem->PlayMode != BEFORE_KICK_OFF ){ /* Not at a pause    */
#if SAVE_COLLECT_DATA
      if ( Mem->RecordCollectData ){
	RecordCollectData(TurnAng,DashPower);   /* Record now and not */
	Mem->RecordCollectData = FALSE;             /* again 'til result  */
      }
      else TurnAng =0;                              /* Stop after dash    */
#endif
      float BallAng = Mem->GetBallAngle();

      int moveRes;
      /* Go more quickly after the NN attempt */
      if (ChoiceFunction != &ChooseNN){
	int dashTimes = MAX_ACTS;
	float distFactor = 1;
	if (Mem->GetBallDistance() < 5)
	  dashTimes =2;
	moveRes = GOMOVE(distFactor*Mem->GetBallDistance(),TurnAng+BallAng, 
			 DashPower*Mem->GetBallDistance(),1,dashTimes);
      }
      else{
	moveRes = DASHMOVE(TurnAng+BallAng, DashPower*Mem->GetBallDistance());
                  /* move: returns TRIED if dash, TRYING if dodgeplayer */
      }
      /* if ( TurnAng ) TURN ( -TurnAng );
                     /* If you didn't move straight, look back           */
      return moveRes;
    }
    else { /* ChoiceFunction == WAIT or In between trials (before_kick_off) */
      FACEBALL;
      return TRYING;
    }
  }
  else if ( Mem->BallValid() == 1 )  {
    Mem->ClearBallGlobalPosition();
    return SUCCESS;                             /* Sure you're at it */
  }
  else {
    Mem->ClearBallGlobalPosition();
    FACEBALL;
    return TRIED;                               /* Not sure          */
  }
}
/*#define CHASEBALL(a) CollectBall(&ChooseStraight,150,a)
#define COLLECTBALL(a,b,c) CollectBall(a,b,c)
#define ACTIVECOLLECTBALL(a,b) Collectball(a,150,b)
*/

/*****************************************************************************/

/* Returns TRUE if I can intercept it, else FALSE */    
int GetCollectTargetXY(float *targetX, float *targetY){

  if ( !Mem->BallTrajectoryValid() )
    printf("Why here if ball trajectory isn't valid??");

#if TUNED_PARAMS
  float playerSpeed = TunedParams[0];
#else
  float playerSpeed = .5;
#endif

  float ballX,ballY, myX,myY;
  Mem->GetGlobalXY(&myX,&myY);
  Mem->GetBallGlobalXY(&ballX,&ballY);

  /* Convert my position to origin (0,0)*/
  ballX -= myX; ballY -= myY;
    
  float ballDistance = Mem->GetBallDistance();
  float ballDirection = Mem->GetBallTrajectoryAbsoluteDirection();
  float ballSpeed = Mem->GetBallTrajectoryMagnitude();

  float ballDx = ballSpeed * cos(ballDirection);
  float ballDy = ballSpeed * sin(ballDirection);

  float a = ballX*ballX + ballY*ballY;
  float b = 2*(ballX*ballDx + ballY*ballDy); 
  float c = ballDx*ballDx + ballDy*ballDy - playerSpeed*playerSpeed; 
  float disc = b*b - 4*a*c;

#if COLLECT_DEBUG    
  printf("%d:%d me (%.1f, %.1f), ball (%.1f, %.1f) -- s/d: %.1f %.1f\n",
	 Mem->MyNumber,Mem->CurrentTime,
	 myX,myY,ballX,ballY,ballSpeed,rad_to_deg(ballDirection));
  printf("           ballD (%.1f, %.1f)......",ballDx,ballDy);
#endif

  int BallTooFast = FALSE;
  if (disc < 0){
#if 0
    printf("%d:%d The ball is too fast!\n",Mem->MyNumber,Mem->CurrentTime);
#endif
    BallTooFast = TRUE;
  }

  float t;  /* Time to intersection point */
  if ( !BallTooFast ){
    float u1 = (-b + sqrt(disc)) / (2*a); 
    float u2 = (-b - sqrt(disc)) / (2*a); 
    
    float u = (u1 > u2) ? u1 : u2; 
    t = 1.0 / u; 
#if TUNED_PARAMS
    t *= TunedParams[1];
#endif

#if 0      
    printf("%d:%d Chose between %.1f and %.1f\n",
	   Mem->MyNumber,Mem->CurrentTime,
	   1.0/u1, 1.0/u2);
#endif

    if (u <= 0){
#if 0
      printf("%d:%d Why negative time? Chose between %.1f and %.1f\n",
	     Mem->MyNumber,Mem->CurrentTime,
	     1.0/u1, 1.0/u2);
#endif
      BallTooFast = TRUE;
    }
  }

  if ( BallTooFast )
#if TUNED_PARAMS
    t = (ballDistance/ballSpeed)*TunedParams[2];
#else
    t = (int)(ballDistance/ballSpeed); /* Aim for where ball is going anyway */
#endif
  /*  else
      printf("%d:%d Using analytic method\n",Mem->MyNumber,Mem->CurrentTime); */

#if TUNED_PARAMS
#else
  t *= 1.1;
#endif

  *targetX = ballX + ballDx * t; 
  *targetY = ballY + ballDy * t; 
    
  /* First convert target back to relative to my position */
  *targetX += myX; *targetY += myY;

#if 0
  printf("target: (%.1f, %.1f)\n",*targetX,*targetY);
#endif

  return !BallTooFast;
}

/****    ****    ****    ****    ****    ****    ****    ****    ****    ****/

int AnalyticCollectBall (float CollectDist, float DashPower){

#if SAVE_COLLECT_DATA
      if ( Mem->RecordCollectData ){
	RecordCollectData(0,0);   /* Record now and not */
	Mem->RecordCollectData = FALSE;             /* again 'til result  */
      }
#endif

  if ( !Mem->BallValid() ) {        /* If Can't see ball */
    FINDBALL;                       /* Look for it       */ 
    WAITUNTILFREETOACT;
  }

  if ( !Mem->BallValid() ) {
    return FAILURE;
  }
  else if ( Mem->GetBallDistance() > CollectDist) { /* If Far from ball  */
    FACEBALL;                                       /* wait              */
    return TRYING;
  }
  else if ( Mem->GetBallDistance() > KICK_DISTANCE &&
	    Mem->BallTrajectoryValid() )  { /* If it's in range  */

    int minDashes = 1;
    float targetX,targetY;
    GetCollectTargetXY(&targetX,&targetY);

#if TUNED_PARAMS
    /* Blowing up GOTOPOINT to put in params */
    float dist = Mem->DistanceToPoint(targetX,targetY);
    if ( dist > TunedParams[6] ) {   /* GoToPointBuffer */
      FacePoint(targetX,targetY,TunedParams[4]); /* facePointBuffer */
      if (dist >= TunedParams[3])  /* dashTwiceBuffer */
	minDashes = 2;
      else if ( Mem->BallTrajectoryValid() &&
	        fabs(Mem->GetBallTrajectoryRelativeDirection()) < 60 && /* ball going away */
		fabs(Mem->GetBallAngle()) < 35 &&                       /* facing the ball */
		Mem->GetMyStamina() > 200 )
	minDashes =  MIN(3,((int)dist)/5 +3);
      else if ( 0 && dist > 8 && Mem->GetMyStamina() > 200 ){
	minDashes = MAX(3,MIN(2,((int)dist)/5));
	//printf("minDashes = %d\n",minDashes);
      }
      else                         /* DashPower */
	minDashes = 1;            /* One dash at least      */

      GOMOVE(dist,0,TunedParams[5]*dist,minDashes,3);            
      return TRYING;
    }
    else                                              /* Find ball          */
      return TRIED;     /* TRYING, TRIED (turned), or SUCCESS (seeing)*/

#else

    /*if ( Mem->DistanceToPoint(targetX,targetY) > 15 &&
	 Mem->GetMyStamina() > 200 )
      minDashes = 2;*/

    if ( Mem->BallTrajectoryValid() &&
	 fabs(Mem->GetBallTrajectoryRelativeDirection()) < 35 && /* ball going away */
	 fabs(Mem->GetBallAngle()) < 35 &&                        /* facing the ball */
	 Mem->GetMyStamina() > 200 ){
      /* printf("%d:%d minDashes at 3\n",Mem->MyNumber,Mem->CurrentTime); */
      /* printf("no 3 dashes\n"); */
      minDashes = 3; /* Hurry if ball going away */
    }

    return GOTOPOINTMINDASHES(targetX,targetY,DashPower,1,minDashes);

#endif

  }
  else if ( Mem->BallValid() == 1 )  {
    /* Mem->ClearBallGlobalPosition(); */
    return SUCCESS;                             /* Sure you're at it */
  }
  else {
    /* Mem->ClearBallGlobalPosition(); */
    FACEBALL;
    return TRIED;                               /* Not sure          */
  }
}
/*#define ANALYTICCOLLECTBALL(a,b) AnalyticCollectBall(a,b)*/

/*****************************************************************************/

int GoalieHome (float DashPower ){
  
  if ( !Mem->MarkerValid(MY_GOAL) ){
    LOOKFORMARKER(MY_GOAL);
    return FAILURE;
  }
  else if ( (Mem->GetMarkerDistance(MY_GOAL) > 5) ) {         
    FACEMARKER(MY_GOAL);
    DASHMOVE(0,DashPower*Mem->GetMarkerDistance(MY_GOAL) ); /* If far, Move to it */
    return TRYING;
  }
  else                                              /* Find ball          */
    return LOOKFORBALL;     /* TRYING, TRIED (turned), or SUCCESS (seeing)*/
}
/*#define GOALIEHOME(a) GoalieHome(a)*/

/*****************************************************************************/

int KamramHome (float DashPower ){
  
  if ( !Mem->MarkerValid(THEIR_GOAL) ){
    LOOKFORMARKER(THEIR_GOAL);
    return FAILURE;
  }
  else if ( (Mem->GetMarkerDistance(THEIR_GOAL) > 15) ) {         
    FACEMARKER(THEIR_GOAL);
    DASHMOVE(0,DashPower*Mem->GetMarkerDistance(THEIR_GOAL) );/* If far, Move to it */
    return TRYING;
  }
  else                                              /* Find ball          */
    return LOOKFORBALL;     /* TRYING, TRIED (turned), or SUCCESS (seeing)*/
}
/*#define KAMRAMHOME(a) KamramHome(a)*/

/*****************************************************************************/

int GotoPoint (float x, float y, float DashPower, float Buffer, int minDashes){ 
  
  float gx = Mem->GetGlobalX();
  float gy = Mem->GetGlobalY();
  float dist = GetDistance(&x,&y,&gx,&gy);

  if (Mem->SetPlay){  /* Don't let the ball get in your way on set plays */
    float balldist = Mem->GetBallDistance();
    if ( balldist < KICK_DISTANCE ){
      if (Mem->MyPositionInSetPlay() && 
	  Mem->GetMySetPlayPositionType() == SETPLAY_STARTER ){
	float aimx, aimy;
	Mem->GetMySetPlayAimXY(&aimx, &aimy);
	float goalAng = Mem->AngleToPoint(aimx,aimy);
	if ( !NEEDTOCIRCLE(goalAng) )
	  return SUCCESS;
      }

      /* WIDE_VIEW; */
      float ballAng = Mem->GetBallAngle();
      if (ballAng > 0)
	DODGEBALL(ballAng-90,100);     
      else
	DODGEBALL(ballAng+90,100);     
      FACEPOINT(x,y);
      return FAILURE;
    }
  }

  if ( dist > Buffer ) {         
    FACEPOINT(x,y);
    GOMOVE(dist,0,DashPower*dist,minDashes,3);      /* If far, Move to it */
    return TRYING;
  }
  else                                              /* Find ball          */
    return LOOKFORBALL;     /* TRYING, TRIED (turned), or SUCCESS (seeing)*/
}
/*
#define GOTOPOINT(x,y,z,a) GotoPoint(x,y,z,a,1)
#define GOTOPOINTMINDASHES(x,y,z,a,b) GotoPoint(x,y,z,a,b)
*/

/*****************************************************************************/

int HoldHome(float DashPower){

  Mem->UpdateHome();

  if ( Mem->PlayMode != BEFORE_KICK_OFF && 
       GOTOPOINT(Mem->GetCurrentHomeX(),Mem->GetCurrentHomeY(), /* Going to position */
		 DashPower,Mem->GetMyPositionBuffer()) <= TRIED )
    return TRYING;
    /* Perhaps check if there's already a person there before returning */
    /* If there is, say an announcemyposition message                   */
    /* But just once--not repeatedly                                    */

  /* Free time to talk at this point            */
  /* This often says things to the wrong person */
  Mem->CheckForPlayersInMyPosition();
  
  return SUCCESS;
}
/*
#define HOLDHOME(a) HoldHome(a)
*/

/*****************************************************************************/

int MarkPlayer( float DashPower ){

  Mem->UpdateMark();

  char side = Mem->TheirSide;
  int  mark = Mem->GetMark();

  /* No player to mark */
  if ( mark == UNKNOWN_PLAYER){
   /* printf("%d: No player to mark\n",Mem->MyNumber);*/
    if ( !Mem->BallValid() ) {        /* If Can't see ball */
      FINDBALL;                       /* Look for it       */ 
      WAITUNTILFREETOACT;
    }
    return HOLDHOME( DashPower );
  }

  if ( !Mem->PlayerValid(side,mark) ) {      /* If Can't see mark */
    /* printf("%d:%d Can't find my mark (%d)\n",Mem->MyNumber,Mem->CurrentTime,mark);*/
    if ( FINDPLAYER(side,mark) == FAILURE ){      /* Look for it         */ 
      Mem->SetMark(UNKNOWN_PLAYER);               /* Can't find it       */
      return FAILURE;
    }
  }

  if (!Mem->IsWithinMyMaxRange(side,mark)){
    /* Should not call markplayer from hold position in this situation */
    /* Or update mark should give a different player                   */

#if 0
    printf("%d: My mark (%d) is out of range:  ",Mem->MyNumber,mark);
    printf("(%.1f, %.1f)\n",Mem->GetPlayerGlobalX(side,mark),
	   Mem->GetPlayerGlobalX(side,mark));
#endif

    if ( !Mem->BallValid() ) {        /* If Can't see ball */
      FINDBALL;                       /* Look for it       */ 
      WAITUNTILFREETOACT;
    }
    return HOLDHOME( DashPower );

  }

  int MarkDist = 1;  /* How close you need to be to the correct position */

  float playerX,playerY,markX,markY;
  Mem->GetPlayerGlobalXY(side,mark,&playerX,&playerY);

  markX = playerX - Mem->GetMarkBuffer()/2;
  markY = playerY - playerY/10;

  if ( GOTOPOINT(markX,markY,DashPower,MarkDist) <= TRIED )
    /* gotopoint: returns TRIED if in position, looking for Ball, 
       SUCCESS if in position, seeing ball */
    return TRYING;
  else 
    return SUCCESS;
}
/*
#define MARKPLAYER(a) MarkPlayer(a)
*/

/*****************************************************************************/

int HoldPosition( float DashPower ){

  if ( Mem->GetMyPosition() == UNKNOWN_POSITION ){
    /* printf("%d:%d Can't hold an unknown position",Mem->MyNumber, Mem->CurrentTime); */
    Mem->SetMyPosition(GetNewPosition());
  }

  if ( Mem->HomeChangeMethod == HOME_MARK || 
       Mem->HomeChangeMethod == HOME_MARK_ELSE_SHIFT )
    return MARKPLAYER( DashPower );
  else
    return HOLDHOME( DashPower );
}
/* 
#define HOLDPOSITION(a) HoldPosition(a)
*/

/*****************************************************************************/
/************        HIGHER LEVELS    ****************************************/
/*****************************************************************************/

int ExecuteSetPlay(float DashPower, float ShotPower){
  if ( Mem->GetMyActiveStatus() != INSETPLAY ) my_error("Shouldn't be doing a set play");

  if ( !Mem->MyPositionInSetPlay() ){
    /* printf("%d:%d  My position (%d) no longer in set play\n",
       Mem->MyNumber,Mem->CurrentTime,Mem->GetMyPosition()); */
    Mem->SetMyselfInactive();
    return FAILURE;
  }

  int spPosType = Mem->GetMySetPlayPositionType();
  int spStarterPos = Mem->GetSetPlayStarterPosition();

  /* make sure there's a starter if there should be one */
  if ( spPosType == SETPLAY_STARTER ){
    if ( strcmp(Mem->GetMyStatus(),SETPLAY_STARTER_STAT) ){
      Mem->SetMyStatus(SETPLAY_STARTER_STAT);
      SAY(SETPLAY_STARTER_MSG);
    }
  }
  else if ( spStarterPos != UNKNOWN_POSITION ){  /* there is a starter */
    int player = Mem->GetPositionPlayer(spStarterPos);
    if ( player == UNKNOWN_PLAYER ){
      SAYTOPOSITION(spStarterPos,PING_MSG);
    }
    else if ( strcmp(Mem->GetStatus(Mem->MySide,player),SETPLAY_STARTER_STAT)){
      if ( Mem->CurrentTime > Mem->SetPlayInvokeTime+Mem->SetPlayTimeLimit/4 &&
	   (Mem->PlayMode == MY_KICK_IN || Mem->PlayMode == MY_FREE_KICK) ){
	/* Only have to worry if ball's position could be in doubt */
	/*printf("%d:%d Going to have to reinitialize (time)\n",
	  Mem->MyNumber,Mem->CurrentTime);*/
	Mem->SetPlay = FALSE; /* force reinitialization */
	return FAILURE;
      }
      else if ( Mem->CurrentTime > Mem->SetPlayInvokeTime+10 ){
	SAYTO(player,ARE_YOU_STARTER_MSG);  /* Wait before forcing starter issue */
      }
    }
  }

  float goalx, goaly, buffer;
  if ( spPosType == SETPLAY_STARTER ){
    Mem->GetBallGlobalXY(&goalx,&goaly); 
    buffer = BALL_SIZE + PLAYER_SIZE;
  }
  else{
    Mem->GetMySetPlayXY(&goalx,&goaly);
    buffer = Mem->GetMySetPlayPositionBuffer();
  }
  
  if ( Mem->PlayMode != PLAY_ON ){
    if (spPosType == SETPLAY_STARTER){
      if ( COLLECTSTOPPEDBALL(10) != SUCCESS )
	return FAILURE;
    }
    else{
      if ( GOTOPOINT(goalx,goaly,10,buffer) != SUCCESS ) /* Not worried about getting */
	return FAILURE;                                  /* There quickly             */
    }

    if ( strcmp(Mem->GetMyStatus(),OPEN_STAT) &&       /* no ack yet         */
	 spStarterPos != UNKNOWN_POSITION &&           /* no point in saying */
	 spPosType != SETPLAY_STARTER ){           
      SAY(READY_TO_RECEIVE_MSG);
    }

    /* If you're the one near the ball, wait until everyone's in position */
    if ( spPosType == SETPLAY_STARTER ){
      if ( Mem->AllInSetPlayPosition() ){
	Mem->ModifySetPlayStartAction();          /* Choose best of 1st 2 */
	/* Mem->SetPlayStartTime = Mem->CurrentTime; /* Starting */
      }
      else if (Mem->CurrentTime >Mem->SetPlayInvokeTime+Mem->SetPlayTimeLimit){
	Mem->ModifySetPlayStartAction();          /* Just go with what you've got */
	/* Mem->SetPlayStartTime = Mem->CurrentTime; /* Starting */
      }	
      else{
	Mem->CheckForPlayersInMyPosition();
	return TRYING;
      }
    }
    else{  /* Not the starter */
      Mem->CheckForPlayersInMyPosition();
      return TRYING;
    }
  }
  else if ( !Mem->SetPlayStartTime ){         /* We're in PLAY_ON mode                */
    Mem->SetPlayStartTime = Mem->CurrentTime; /* Time that mode is changed to PLAY_ON */
  }

  /* Play has started */

  int position = Mem->GetMySetPlayAimPosition();
  int player   = UNKNOWN_PLAYER;
  float aimx,aimy;
  if ( position == UNKNOWN_POSITION ){ /* We're going for a point */
    Mem->GetMySetPlayAimXY(&aimx,&aimy);
  }
  else {                              /* We're going for a player */
    player = Mem->GetPositionPlayer(position);
    if ( player == UNKNOWN_PLAYER )
      my_error("Should know who's playing the position");
  } 
 
  int SetPlayTime = Mem->GetMySetPlayWaitTime();
  int CollectDist = 14;

  if (spPosType == SETPLAY_STARTER)
    DashPower = 0;

  switch(spPosType){
  case SETPLAY_SHOOTER:
    COLLECTANDSHOOT(&ChooseNN,CollectDist,DashPower,100);
    break;
  case SETPLAY_STARTER:
    /* Assuming the starter either passes or knocks */
    if ( player != UNKNOWN_PLAYER )
      EASYCOLLECTANDPASSTO(player,&ChooseNN,CollectDist,DashPower,3);
    else 
      COLLECTANDCLEARTOPOINT(aimx,aimy,&ChooseNN,CollectDist,DashPower,100);
    break;
  case SETPLAY_PASSER:
    EASYCOLLECTANDPASSTO(player,&ChooseNN,CollectDist,DashPower,ShotPower);
    break;
  case SETPLAY_KNOCKER:
    COLLECTANDKNOCKTOPOINT(aimx,aimy,&ChooseNN,CollectDist,DashPower,100);
    break;
  case SETPLAY_BLASTER:
    BLASTATPOINT(aimx,aimy);
    break;
  default: my_error("what's my set play position type???");
  }
  
  //printf("%d   SetPlayTime: %d   currentTime: %d   started at:%d\n",
  //  Mem->MyNumber,SetPlayTime,Mem->CurrentTime,Mem->SetPlayStartTime); 

  if ( Mem->SetPlayStartTime &&  /* into play_on mode */
       Mem->CurrentTime - Mem->SetPlayStartTime >= SetPlayTime ){
    Mem->SetMyselfAtAttention();
    return SUCCESS;
  }

  return TRIED;
}
/* #define EXECUTESETPLAY(a,b) ExecuteSetPlay(a,b) */

/*****************************************************************************/

int DribbleToMarker(int Marker, float DashPower, float ShotPower){
  
  if ( CHASEBALL(DashPower) != SUCCESS ) {      /* If not at ball    */
    return TRYING;                              /* Not done          */
  }
  else {
    if ( Marker > 0 && Mem->MarkerValid(Marker) ){
      if ( DRIBBLEATMARKER(Marker,ShotPower) == TRYING )  /* Dribble */
	return TRYING;                         /* Circling Ball      */
      else 
#if DRIBBLE_DEBUG
	printf("%d:%d Facing Marker %d\n",Mem->MyNumber, Mem->CurrentTime, Marker);
#endif
      if ( fabs(Mem->GetMarkerAngle(Marker)) > 3 ){  /* Arbitrary */
	WAITUNTILFREETOACT;
	if ( !Mem->NewSight ){
	  FACEMARKER(Marker);
	  return TRIED;    /* Means you won't dash after facing */
	}
      }
    }
    else { /* No marker or it's not valid */
      KICKANDFACE(ShotPower,Mem->GetBallAngle()); /* Dribble straight*/
#if DRIBBLE_DEBUG
	printf("%d:%d Kicking and facing\n",Mem->MyNumber, Mem->CurrentTime );
#endif
    }

    if ( fabs(Mem->GetBallAngle()) < 45 ){     /* Don't move if ball was behind */
      /* WAITUNTILFREETOACT;
      if ( !Mem->NewSight ){ 
	DASHMOVE(0,DashPower);
      }*/
      DASHMOVE(0,DashPower);  /* Always dash at least once after a kick */
    }
    return TRIED;
  }
}
/*#define DRIBBLETOMARKER(a,b,c) DribbleToMarker(a,b,c)
#define DRIBBLESTRAIGHT(a,b) DribbleToMarker(-1,a,b)*/

/*****************************************************************************/

int CollectAndKnock (int Marker, float x, float y,
		     int (*ChoiceFunction)(float*,float*), 
		     float CollectDist, float ShootDist, float DashPower, 
		     float DribblePower, float ShotPower, int clear ){
  
#if 0
  printf(">>>BValid=%f\n",Mem->BallValid());
  printf(">>>GValid=%f\n",Mem->MarkerValid(THEIR_GOAL));
#endif

  /* Can put in a different choice function here for CollectBall */
  
  if ( !Mem->BallValid() || Mem->GetBallDistance() > KICK_DISTANCE ){
    if ( COLLECTBALL(ChoiceFunction, CollectDist, DashPower) <= TRYING )
      return FAILURE;                             /* Ball's not in range*/
    else                                          
      return TRYING;                              /* Dashing for it     */
  }
  
  if ( !Mem->MarkerValid(Marker) ){
    LOOKFORMARKER(Marker);
    return TRYING;                                /* Facing goal        */
  }
  else if ( Mem->GetMarkerDistance(Marker) > ShootDist ){ /* goal is far */
    DRIBBLETOMARKER(Marker,50,DribblePower);
#if DRIBBLE_DEBUG
    printf("DRIBBLING to %d (%d)\n",Marker,Mem->MyNumber);
#endif
    return TRYING;
  }
  else{                                        /* Goal is near           */
#if DRIBBLE_DEBUG
    printf("KNOCKING to %d (%d)\n",Marker,Mem->MyNumber);
#endif

#if PRACTICE
  return SHOOTATMARKER(Marker, ShotPower);
#endif

  /* printf("clear: %d,   Marker: %d\n",clear,Marker); */

  if ( clear ){ /* Don't worry about exact aim.  Instead avoid players */
    if ( !Marker )
      CLEARTOWARDSPOINT(x,y,ShotPower);
    else
      return CLEARTOWARDSMARKER(Marker, ShotPower);
  }

  if ( !Marker )
    return SHOOTATPOINT(ShotPower,x,y);

  if (Marker == THEIR_GOAL)
    /* If far enough away, shoot at center??? */
    return SHOOTATNEARPOST(100);
  else 
    return SHOOTATMARKER(Marker,ShotPower);  /* circling ball (trying) */
                                             /* or Kicked  (tried)     */
  }
}

int CollectAndKnock (int Marker, int (*ChoiceFunction)(float*,float*), 
		     float CollectDist, float ShootDist, float DashPower, 
		     float DribblePower, float ShotPower, int clear ){
  /* ignore the x and y positions */
  return CollectAndKnock(Marker,0,0,ChoiceFunction,CollectDist,ShootDist,
			 DashPower,DribblePower,ShotPower,clear);
}
/*#define COLLECTANDSHOOT(a,b,c,d) CollectAndKnock(THEIR_GOAL,a,b,150,c,0,d,0)
#define CHASEANDSHOOT(a,b)     CollectAndKnock(THEIR_GOAL,&ChooseStraight,150,150,a,0,b,0)
#define DRIBBLEANDSHOOT(a,b,c,d)   CollectAndKnock(THEIR_GOAL,&ChooseStraight,150,a,b,c,d,0)
#define STAYANDSHOOT(a)   CollectAndKnock(THEIR_GOAL,&ChooseStraight,0,150,0,0,a,0)
#define COLLECTANDKNOCK(a,b,c,d,e) CollectAndKnock(a,b,c,150,d,0,e,0)
#define ACTIVECOLLECTANDKNOCK(a,b,c,d) CollectAndKnock(a,b,150,150,c,0,d,0)
#define CHASEANDKNOCK(a,b,c)     CollectAndKnock(a,&ChooseStraight,150,150,b,0,c,0)
#define DRIBBLEANDKNOCK(a,b,c,d,e)   CollectAndKnock(a,&ChooseStraight,150,b,c,d,e,0)
#define COLLECTANDDRIBBLE(a,b,c,d,e)   CollectAndKnock(a,b,c,0,d,e,0,0)
#define ACTIVECOLLECTANDDRIBBLE(a,b,c,d)   CollectAndKnock(a,b,150,0,c,d,0,0)
#define COLLECTANDCLEAR(a,b,c,d,e) CollectAndKnock(a,b,c,150,d,0,e,1)
#define COLLECTANDCLEARTOPOINT(a,b,c,d,e,f) CollectAndKnock(0,a,b,c,d,150,e,0,f,1)
#define COLLECTANDKNOCKTOPOINT(a,b,c,d,e,f) CollectAndKnock(0,a,b,c,d,150,e,0,f,0)
*/

/*****************************************************************************/

int CollectAndPassTo (int Receiver, int (*ChoiceFunction)(float*,float*), 
		       float CollectDist, float PassDist, float DashPower, 
		       float DribblePower, float PassPower, int trap ){
  
  char MySide = Mem->MySide;
  /* Can put in a different choice function here for CollectBall */

  if ( !Mem->BallValid() || Mem->GetBallDistance() > KICK_DISTANCE ){
    if ( COLLECTBALL(ChoiceFunction, CollectDist, DashPower) <= TRYING )
      return FAILURE;                             /* Ball's not in range*/
    else                                          
      return TRYING;                              /* Dashing for it     */
  }
  
  if ( Receiver > TEAM_SIZE ) {                   /* Going by position   */
    my_error("Receiver should never be > teamsize anymore");
  } 

  if ( Receiver == Mem->MyNumber ){
    my_error("Shouldn't be trying to pass to myself");
    return TRYING;                      /* Can't pass to myself             */
  }

  /*if ( !Mem->PlayerAngleValid(MySide,Receiver) ){*/
  /* For passes, only trusting visual or audial info from the last tick */
  if ( Mem->PlayerAngleValid(MySide,Receiver) < HEARSAY_CONFIDENCE - .01 ){
    LOOKFORTEAMMATE(Receiver);
    return TRYING;                                /* looking for player */
  }
  else if ( !Mem->PlayerDistanceValid(MySide,Receiver) ){
    /* The angle's valid, but the player's too far to see */
#if SAVE_PASS_DATA    
    RecordPassData(Receiver, Mem);
#endif
    if ( strcmp(Mem->GetStatus(Mem->MySide,Receiver),RECEIVER_STAT) ){
      /* Announce who's getting the ball */
      char Announcement[MAXMESG];
      sprintf(Announcement,PASSING_DECISION,Receiver);
      SAY(Announcement);
    }
#if DRIBBLE_DEBUG
    printf("PASSING toward %d (%d)\n",Receiver,Mem->MyNumber);
#endif
    Mem->ClearData();
    if (trap)
      return PASSTOWARDTEAMMATE(Receiver,80);   /* circling ball (trying) */
    else                                        /* or Kicked  (tried)     */
      return EASYPASSTOWARDTEAMMATE(Receiver,80);   
  }
  else if ( Mem->GetPlayerDistance(MySide,Receiver) > PassDist ){ /* player is far */
    DRIBBLEATTEAMMATE(Receiver,DribblePower);
    return TRYING;
  }
  else {                                       /* Player is near         */
#if SAVE_PASS_DATA    
    RecordPassData(Receiver, Mem);
#endif
    if ( Mem->GetMyActiveStatus() != INSETPLAY && 
	 /* Don't want to change receiver's Active Status by talking */
	 strcmp(Mem->GetStatus(Mem->MySide,Receiver),RECEIVER_STAT) ){
      /* Announce who's getting the ball */
      /* When passing, the receiver is marked (communications.c) */
      /* The indicated receiver then watches the ball until the  */
      /* ball is within CollectDist (HoldPositionAndAct).  then  */
      /* it marks itself as no longer the reciever.  When the    */
      /* pass is made, the old receiver is erased and the new    */
      /* receiver is marked (communications.c)                   */
      char Announcement[MAXMESG];
      sprintf(Announcement,PASSING_DECISION,Receiver);
      SAY(Announcement);
    }
#if DRIBBLE_DEBUG
    printf("PASSING to %d (%d)\n",Receiver,Mem->MyNumber);
    PassPower = 100; /* Just to test */
#endif
    Mem->ClearData();
    if (trap)                                      /* ball too far (failure) */
      return PASSTOTEAMMATE(Receiver,PassPower);   /* circling ball (trying) */
    else                                           /* or Kicked  (tried)     */
      return EASYPASSTOTEAMMATE(Receiver,PassPower);   
  }
}
/*
#define COLLECTANDPASSTO(a,b,c,d,e) CollectAndPassTo(a,b,c,150,d,10,e,1)
#define EASYCOLLECTANDPASSTO(a,b,c,d,e) CollectAndPassTo(a,b,c,150,d,10,e,0)
#define ACTIVECOLLECTANDPASSTO(a,b,c,d) CollectAndPassTo(a,b,150,150,c,10,d,1)
#define DRIBBLEANDPASSTO(a,b,c,d,e) CollectAndPassTo(a,&ChooseStraight,150,b,c,d,e,1)
#define APPROACHANDPASSTO(a,b,c,d,e,f) CollectAndPassTo(a,&ChooseNN,b,c,d,e,f,1)
/* a is receiver, b is CollectDist (collect when ball is < b), c is PassDist 
   (pass when ball < c), d is DashPower, e is DribblePower, f is PassPower 
#define STAYANDPASSTO(a,b)   CollectAndPassTo(a,&ChooseStraight,0,150,0,0,b,0)
/* a is receiver, b is PassPower 
*/

/*****************************************************************************/

int CollectAndAct (int (*ActFunction)(char*,int*),
		   int (*ChoiceFunction)(float*,float*),
		   float CollectDist, float ShotDist, float DashPower, 
		   float DribblePower, float ShotPower){

  char KnockOrPass;
  int BallTo;

  ActFunction( &KnockOrPass,&BallTo );  /* act */
  /* printf("Action %c, To %d\n",KnockOrPass,BallTo);   */
  switch (KnockOrPass){
  case 'k': /* knock */
    return COLLECTANDKNOCK( BallTo, ChoiceFunction, CollectDist, DashPower, ShotPower);
    break;
  case 'p': /* pass  */
    return COLLECTANDPASSTO( BallTo, ChoiceFunction, CollectDist, DashPower, ShotPower);
    break;
  case 'd': /*dribble*/  /* COLLECTANDDRIBBLE--enforces dribble */
    return COLLECTANDDRIBBLE( BallTo, ChoiceFunction, CollectDist, DashPower, DribblePower);
    break;
  case 'c': /* clear */
    return COLLECTANDCLEAR( BallTo, ChoiceFunction, CollectDist, DashPower, ShotPower);
    break;
  default:
    my_error("Should be a 'p' or a 'k' or a 'd' or a 'c'");
  }
}  
/*
#define COLLECTANDACT(a,b,c,d) CollectAndAct(a,&ChooseNN,b,0,c,10,d)
#define ACTIVECOLLECTANDACT(a,b,c) CollectAndAct(a,&ChooseNN,150,0,b,10,c)
*/


/*****************************************************************************/

int HoldPositionAndAct (int (*ActFunction)(char*,int*),
			int (*ChoiceFunction)(float*,float*),
			float ShotDist, float DashPower, 
			float DribblePower, float ShotPower){

  if ( !Mem->BallValid() ){
    my_error("should never get to ballvalid in Holdpositionandact");
    int with_ball;
    if ( !strcmp(Mem->GetMyStatus(),RECEIVER_STAT) && /*I'm receive          */
	 (with_ball = Mem->FindTeammateWithStatus(WITH_BALL_STAT)) ){ 
      FACETEAMMATE(with_ball);                        /*that's where the ball*/
      Mem->ClearStatus(Mem->MySide,with_ball);        /*should be            */
      /* Clearing the status, because this cue should only be used once      */
      /* Otherwise, the ball might never be found                            */
    } 
  }
  else if ( Mem->GetMyActiveStatus() == ACTIVE ){
#if 0
    if ( !strcmp(Mem->GetMyStatus(),RECEIVER_STAT) ){ /* I'm receiver        */
      Mem->ClearMyStatus();                           /* no need to remember */
    }
#endif
    if ( Mem->ChangePositions && Mem->GetMyPosition() != UNKNOWN_POSITION 
	 &&!Mem->AmWithinMyMaxRange() ){
      int old_position = Mem->GetMyPosition();
      if ( old_position != UNKNOWN_POSITION ){
	/*if ( !old_position )
	  printf ("%d:%d Goalie leaving\n",Mem->MyNumber,Mem->CurrentTime);*/
	Mem->ClearMyPosition();
	ANNOUNCELEAVINGPOSITION(old_position);
      }
    }
    ACTIVECOLLECTANDACT(ActFunction, DashPower, ShotPower);
    return TRIED;
  }
  else if ( Mem->GetMyActiveStatus() == AT_ATTENTION ){
    FACEBALL; /* Should know where the ball is if get here */
    return TRYING;
  }
  else if ( Mem->GetMyActiveStatus() == INSETPLAY ){
    return EXECUTESETPLAY(DashPower,ShotPower); 
  }
  else if ( Mem->GetMyActiveStatus() == AUXILIARY ){
    ;
  }
  else { /* active status is INACTIVE */
    /* Should take this out-- should be at attention */
    if ( !strcmp(Mem->GetMyStatus(),RECEIVER_STAT) ){ /*ball should be coming*/
      printf("shouldn't be inactive when receiver.  Clearing status?\n");
      Mem->ClearMyStatus();                           /* no need to remember */
    }

    DashPower = MIN(20,DashPower);
    return HOLDPOSITION(DashPower);
    /* TRYING if going or wandering, SUCCESS if in pos.*/
  }
}
   /* a is action func. b is collectdist. c,d are dash/shot power 
#define HOLDPOSITIONANDACT(a,b,c) HoldPositionAndAct(a,&ChooseNN,0,b,10,c)
*/

/*****************************************************************************/

int ReceivePass(int (*ChoiceFunction)(float*,float*), 
		float CollectDist, float DashPower){
  
  int Result, with_ball, ball_coming;
  
  ball_coming = Mem->FindTeammateWithStatus(RECEIVER_STAT); /* The player to whom
							       the ball is coming */
  if ( ball_coming && ball_coming != Mem->MyNumber ){
    return FAILURE; /* Someone else is getting the ball */
  }

  Result = COLLECTBALL(ChoiceFunction, CollectDist, DashPower);
  with_ball = Mem->FindTeammateWithStatus(WITH_BALL_STAT);

  if ( Result == TRYING ){  /* Ball's not in range (or dodging a player) */
    if ( strcmp(Mem->GetStatus(Mem->MySide,Mem->MyNumber),ANNOUNCED_OPEN_STAT)
	 && with_ball && Mem->PlayerDistanceValid(Mem->MySide,with_ball) ){
      /* Only say ready to receive if the passer is close enough to see distance */
      SAY(READY_TO_RECEIVE_MSG);
      Mem->SetStatus(Mem->MySide,with_ball,ANNOUNCED_OPEN_STAT);
    }
    return TRYING;
  }
  if ( Result == FAILURE )  /* Still trying to face the ball             */
    return TRYING; 
  else                      /* TRYING until at the ball                  */
    return Result;           
}
/*#define RECEIVEPASS(a,b,c) ReceivePass(a,b,c)*/



#if 0

OLD CODE FOR NUM DASHES

    /* Blowing up GOTOPOINT to put in params */
    float dist = Mem->DistanceToPoint(targetX,targetY);
    if ( dist > TunedParams[6] ) {   /* GoToPointBuffer */
      FacePoint(targetX,targetY,TunedParams[4]); /* facePointBuffer */
      if (dist >= TunedParams[3])  /* dashTwiceBuffer */
	minDashes = 2;
      else if ( Mem->BallTrajectoryValid() &&
	        fabs(Mem->GetBallTrajectoryRelativeDirection()) < 35 && /* ball going away */
		fabs(Mem->GetBallAngle()) < 35 &&                       /* facing the ball */
		Mem->GetMyStamina() > 200 )
	minDashes = 6;            /* Three dashes at least  */
      else                         /* DashPower */
	minDashes = 1;            /* One dash at least      */

      GOMOVE(dist,0,TunedParams[5]*dist,minDashes,3);            
      return TRYING;
    }
    else                                              /* Find ball          */
      return TRIED;     /* TRYING, TRIED (turned), or SUCCESS (seeing)*/
#endif
