/**
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2000-2001 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK 3.1 for the IXP2X00 Network Processor
 *
 * = FILENAME
 *      ix_cc_eth_tx_drv_wrapper.c
 *
 * = DESCRIPTION
 *      This file contains wrapper APIs for interfacing with the 
 *      kernel media driver, ixd2410.o.  Usually a Linux kernel driver
 *      can be acessed only by a user mode application through the system 
 *      calls to open, close, and ioctl. The API functions provided by 
 *      this wrapper provides a way for an application running in kernel 
 *      mode to use the driver from the kernel space.  This is the case 
 *      in SDK 3.1 where all core components are running in kernel 
 *      space.  Thus the client of the ixd2410.o (i.e. Eth Tx CC)
 *      needs to use the wrapper to access the driver.  
 *
 *      The wrapper utilizes the following internal driver functions:
 *
 *      gbe_mac_config_handler()
 *      read_32_bits_from_isr()
 *      gbic_status
 *      gbe_mac_phy_read()
 *
 *      The driver itself remains unchanged.
 *
 *      Note that the kernel mode application can not co-exist with a
 *      user mode application.  This means that when the wrapper is 
 *      being used by a kernel application, there should not be 
 *      any user mode application use the driver simultaneously 
 *      or the driver behavior will no longer be predictable. 
 *
 *      The use of the wrapper is as follows:
 *
 *      1. From Linux shell prompt, use insmod command to load ixd2410.o 
 *         into kerenl.
 *      2. From Linux shell prompt, use mknod command to create the driver 
 *         device. 
 *      2. From your kernel application, call any of the wrapper API 
 *         to initailize and configure the media driver.
 *
 * = CHANGE HISTORY
 *      04/31/2003 - Created.
 *
 * ============================================================================
 * $Id: ix_cc_eth_tx_drv_wrapper.c,v 1.5 2003/08/06 22:40:50 ktseng Exp $
 */

/**
 * System defined include files required.
 */
#include "ix_ossl.h"
#include <asm/arch/irqs.h>

/**
 * User defined include files required.
 */
#include "cc/ix_cc_eth_tx.h"
#include "cc/internal/ix_cc_eth_tx_int.h"
#include "cc/internal/ix_cc_eth_tx_drv_wrapper.h"

/* device driver common includes */
#include "ixf1104ce_linux_driver.h"
#include "ixf1104ce_ioctl.h"
#include "ixf1104ce_internal.h"

/**
 * Static function prototypes
 */

/**
 * Variable declarations global to this file only.
 */
extern unsigned int gbic_status;
static void ix_cc_eth_tx_drv_gbe_mac_isr(int, void *, struct pt_regs *);
void *g_pGbeMacUserContext = NULL;
extern uint32 s_PhyIntReadValue[2][4];
uint32 copper_link_status[IX_CC_ETH_TX_MAX_NUM_PORTS] = {0x04}; /* only bit 2 is used */


#define MDI_LINK_STATUS          0x01
#define MDI_INT_STATUS_OFFSET    0x1A

/**
 * function definitions.
 */

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)

gbe_mac_error GbEMAC_DeviceStart(
    uint32 arg_PortMask,
    uint32 *arg_Handle,
    uint32 arg_Mode)				
{

    uint32 s_DeviceNumber = 0; /* intended for only one card on media side */
    uint32 i;
    uint32 s_Result;
    gbe_mac_ioctl_ptr g_GbeMacIoctlPtr;
    gbe_mac_ioctl_ptr_64 g_GbeMacIoctlPtr64;

    /**
     * First free the original isr registered by the media driver.
     * Then register the wrapper isr.
     */
    free_irq(IRQ_IXDP2400_MEDIA_PCI, NULL);
    s_Result = request_irq(IRQ_IXDP2400_MEDIA_PCI, 
                           &ix_cc_eth_tx_drv_gbe_mac_isr,
                           SA_INTERRUPT,
                           "PtLone0",
                           NULL);
    if (s_Result != 0)
    {
        ix_ossl_message_log("Error in Registering the ISR for Media IF Card!!!\n");
    }

    for (i = 0; i < IX_CC_ETH_TX_MAX_NUM_PORTS; i++)
    {
        /* Fill the Port number and the value in the IOCTL structure */
        g_GbeMacIoctlPtr64.portIndex = i;
        g_GbeMacIoctlPtr64.valueHigh = arg_Mode;
        g_GbeMacIoctlPtr64.valueLow = 0x12344321; /* not used */

        /* Configure each port */
        s_Result = gbe_mac_config_handler(s_DeviceNumber,
                                          SET_CONFIG_MODE,
                                          (void *)&g_GbeMacIoctlPtr64);

        if(s_Result != SUCCESS)
        {
#ifdef IX_DEBUG
            ix_ossl_message_log("Media port %d configuration failed.\n", i);
#endif
            return(s_Result);
        }

        /* Enable each port */
        if ((arg_PortMask & (1 << i)) == 0)
        {
            /* This port is not to be enabled */
            continue; 
        }

        /* Fill the Port number and the value in the IOCTL structure */
        g_GbeMacIoctlPtr.portIndex = i;
        g_GbeMacIoctlPtr.value = (1 << i);

        /* Call the driver handler routine for opening the port */
        s_Result = gbe_mac_config_handler(s_DeviceNumber,
                                          SET_PORT_INIT_ENABLE,
                                          (void *)&g_GbeMacIoctlPtr);

        if(s_Result != SUCCESS)
        {
#ifdef IX_DEBUG
            ix_ossl_message_log("Media port %d initialization failed.\n", i);
#endif
            return(s_Result);
        }
    }

    return(s_Result);

} /* GbEMAC_DeviceStart */



gbe_mac_error GbEMAC_DeviceStop(
    uint32 arg_PortMask,
    uint32 *arg_Handle)				
{

    uint32 s_DeviceNumber = 0; /* intended for only one card */
    uint32 i;
    uint32 s_Result;
    gbe_mac_ioctl_ptr g_GbeMacIoctlPtr;

    for (i = 0; i < IX_CC_ETH_TX_MAX_NUM_PORTS; i++)
    {
        if ((arg_PortMask & (1 << i)) == 0)
        {
            /* This port is not to be disabled */
            continue; 
        }

        /* Fill the Port number and the value in the IOCTL structure */
        g_GbeMacIoctlPtr.portIndex = i;
        g_GbeMacIoctlPtr.value = (1 << i);

        /* Call the driver handler routine for closing the port */
        s_Result = gbe_mac_config_handler(s_DeviceNumber,
                                          SET_PORT_STATE_CLOSE,
                                          (void *)&g_GbeMacIoctlPtr);

        if(s_Result != SUCCESS)
        {
#ifdef IX_DEBUG
            ix_ossl_message_log("Media port %d closing failed.\n", i);
#endif
            return(s_Result);
        }
    }

    /* Clear the stored user context */     
    g_pGbeMacUserContext = NULL;

    return(SUCCESS);

} /* GbEMAC_DeviceStop */



gbe_mac_error GbEMAC_Ioctl(
    uint32 *arg_Handle, 
    uint32 ioctl_command, 
    void *ioctl_struct)
{
    uint32 s_Result;
    uint32 s_deviceNumber=0; /* intended for only 1 card */
    uint32 i=0;


    /* Check for NULL pointer */
    if ((void *)ioctl_struct == NULL)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("IOCTL_BUFFER_POINTER_NULL\n");
#endif
        return IOCTL_BUFFER_POINTER_NULL;
    }

    s_Result = gbe_mac_config_handler(s_deviceNumber, ioctl_command, ioctl_struct);
    if (s_Result != SUCCESS)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("Media port %d closing failed.\n", i);
#endif
        return(s_Result);
    }

    return s_Result;

} /* GbEMAC_Ioctl */



unsigned int GbEMAC_Callback(
    uint32 *argHandle,
    void * arg_pCallback,
    void * arg_pUserContext)
{
    if (arg_pUserContext == NULL)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("User context is NULL.\n");
#endif
        return(ERROR_INVALID_HANDLE);
    }

    if (arg_pCallback == NULL)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("User GBE MAC callback is NULL.\n");
#endif
        ((ix_cc_eth_tx_ctl_blk *)arg_pUserContext)->pGbeMacCallback = NULL;
        g_pGbeMacUserContext = NULL;
        return(SUCCESS);
    }

    /* Store the callback and user context */
    ((ix_cc_eth_tx_ctl_blk *)arg_pUserContext)->pGbeMacCallback = arg_pCallback;
    g_pGbeMacUserContext = arg_pUserContext;

    return(SUCCESS);

} /* GbEMAC_Callback */



/**
 * This is the wrapper internal interrupt service routine called by the kernel. 
 * 
 */
void ix_cc_eth_tx_drv_gbe_mac_isr(
    int arg_Irg, 
    void *arg_DevStruct,
    struct pt_regs *arg_PtRegs)
{
    uint32 readVal = 0;
    uint32 portId;


    /**
     * Check if ports are configured to fiber or copper.
     * For copper mode, read the phy register.  For fiber ports, 
     * read the GBIC status register.
     * (Note that the assumption here is that all ports are 
     * configured to either fiber or copper.  If this is not the case,
     * then the isr code needs to be re-arranged to take of
     * the mixed ports.)
     */
    if ((IX_CC_ETH_TX_DRIVER_MODE_DEFAULT & 0x7) != 0)
    {
        for (portId = 0; portId < IX_CC_ETH_TX_MAX_NUM_PORTS; portId++)
        {
            gbe_mac_phy_read(0, portId, MDI_LINK_STATUS, &readVal);

            /* Bit 2 indicates the link status */
            readVal = readVal & 0x4;

            if (copper_link_status[portId] != readVal)
            {
                /* Call the registered user callback */
                if (g_pGbeMacUserContext != NULL)
                {
                    ((ix_cc_eth_tx_ctl_blk *)g_pGbeMacUserContext)->pGbeMacCallback(g_pGbeMacUserContext, portId, ((readVal >> 2) << portId));
                }

                /* Update the state variable */
                copper_link_status[portId] = readVal;
            }
        }

        /** 
         * Reading PHY interrupt status register to enable the Interrupt,
         * this value is stored in a global array for future use.
         */
        for (portId = 0; portId < IX_CC_ETH_TX_MAX_NUM_PORTS; portId++)
        {
            gbe_mac_phy_read(0, 
                             portId,
                             MDI_INT_STATUS_OFFSET,
                             &(s_PhyIntReadValue[0][portId]));
        }
    }
    else /* fiber ports */
    {

        uint32 s_TempStatus;
        uint32 old_status;
        uint32 cur_status;

        /**
         *  Reading the GBIC Interrupt Status 
         */
        read_32_bits_from_isr(0, GBIC_STATUS_REG, &readVal);

        if (gbic_status != readVal)
        {
            old_status = (gbic_status >> 20) & 0xF;
            cur_status = (readVal >> 20) & 0xF;
            s_TempStatus = old_status ^ cur_status;

            for (portId = 0; portId < IX_CC_ETH_TX_MAX_NUM_PORTS; portId++)
            {
                if (s_TempStatus & (0x1 << portId))
                {
                    /* Call the registered user callback */
                    if (g_pGbeMacUserContext != NULL)
                    {
                        ((ix_cc_eth_tx_ctl_blk *)g_pGbeMacUserContext)->pGbeMacCallback(g_pGbeMacUserContext, portId, ~cur_status);
                    }
                }
            }

            /* Update the state variable */
            gbic_status = readVal;
        }
    } /* fiber ports */

    return;
}    

#endif /* _IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_ */
