/**;
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2000-2001 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK 3.0 for the IXP2000 Network Processor, Release 5
 *
 * = FILENAME
 *      ix_cc_stkdrv_vidd.h
 *
 * = DESCRIPTION
 *      Header file for the local VIDD component of the Stack Driver.
 *
 *  = CHANGE HISTORY
 *       08/20/2002 - initial revision
 *
 * ============================================================================
 * $Id: ix_cc_stkdrv_vidd_linux.h,v 1.6 2003/11/04 20:10:09 ktseng Exp $
 */
#if !defined(__IX_CC_STKDRV_VIDD_H__)
#define __IX_CC_STKDRV_VIDD_H__
#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/compatmac.h>    /* copy_to_user() */

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/rtnetlink.h>

#include <linux/in.h>
#include <linux/if.h>
#include <linux/if_arp.h>
#include <linux/inetdevice.h>
#include <linux/proc_fs.h>

#include <ix_cc_error.h>
#include <ix_ossl.h>
#include <ix_rm.h>
#include <ix_cc.h>
#include <ix_netmacros.h>


#if defined(__cplusplus)
extern "C"
{
#endif /* end defined(__cplusplus) */


#define STKDRV_BANNER	"Stack Driver"
#define STKDRV_VERSION_MAJOR	1
#define STKDRV_VERSION_MINOR	0


/**
 * Pre-processor symbol and macro definitions.
 */

#define IP_HEADER_IPID 4	/* Offset to IP ID field in IP header */
#define IP_HEADER_CSUM 10	/* Offset to checksum field in IP header */

#ifdef IX_DEBUG

extern int stkdrvTraceLevel;

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)

#define STKDRV_ASSERT(__expr, __code, __action)                              \
        if (__expr) {}                                                       \
        else                                                                 \
        {                                                                    \
            printk(KERN_CRIT "StkDrv: assertion failed at %s[%d]: %s() code" \
                   " 0x%x\n", __FILE__, __LINE__, __FUNCTION__, (int)__code);\
            __action;                                                        \
        }
#define STKDRV_TRACE(level, format, args...)                                 \
        (level > stkdrvTraceLevel) ?: printk("StkDrv: " format "\n", ##args)
#define STKDRV_DUMP(addr, len, string)                                       \
        (stkdrvTraceLevel < 4) ?: _ix_cc_stkdrv_dump_packet(addr, len, string)

#else /* not _IX_OS_LINUX_KERNEL_ */

#define STKDRV_ASSERT(__expr, __code, __action)                              \
        if (__expr) {}                                                       \
        else                                                                 \
        {                                                                    \
            printf("StkLib: assertion failed at %s[%d]: %s() code"           \
                   " 0x%x\n", __FILE__, __LINE__, __FUNCTION__, (int)__code);\
            __action;                                                        \
        }
#define STKDRV_TRACE(level, format, args...)                                 \
       (level > stkdrvTraceLevel) ?:  printf("StkLib: " format "\n", ##args)
#define STKDRV_DUMP(addr, len, string)          do {} while (0)

#endif /* _IX_OS_LINUX_KERNEL_ */

#else /* not IX_STKDRV_DEBUG */

#define STKDRV_ASSERT(__expr, __code, __action)  do {} while (0)
#define STKDRV_TRACE(level, format, args...)     do {} while (0)
#define STKDRV_DUMP(addr, len, string)           do {} while (0)

#endif /* IX_STKDRV_DEBUG */

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)

#define STKDRV_LOG(level, format, args...)                                   \
        printk(level "StkDrv: " format "\n", ##args)
#define STKDRV_ADVERT  ""           /* printed unconditinally */
#define STKDRV_EMERG   KERN_EMERG   /* system is unusable */
#define STKDRV_ALERT   KERN_ALERT   /* action must be taken immediately */
#define STKDRV_CRIT    KERN_CRIT    /* critical conditions */
#define STKDRV_ERR     KERN_ERR     /* error conditions */
#define STKDRV_WARNING KERN_WARNING /* warning conditions */
#define STKDRV_NOTICE  KERN_NOTICE  /* normal but significant condition */
#define STKDRV_INFO    KERN_INFO    /* informational */

#else  /* not _IX_OS_LINUX_KERNEL_ */

#define STKDRV_LOG(level, format, args...)                                   \
        printf("StkDrv: " format "\n", ##args)
#endif


/**
 * MACRO NAME: IX_CC_STKDRV_VIDD_GET_PHYSICAL_IF
 *
 * DESCRIPTION: Get the port of given port ID 
 *
 * @Param:  - IN arg_portID - ID of the desired port.
 * @Param:  - INOUT arg_pPhysicalIf - pointer to the list of
 *          ports to traverse.  If we find the desired
 *          port structure, it will be returned in this pointer, otherwise
 *          it will be NULL.
 *
 * @Return:
 */
#define _IX_CC_STKDRV_VIDD_GET_PHYSICAL_IF(                                 \
                                          arg_portId,                       \
                                          arg_pPhysicalIf                   \
                                          )                                 \
    do                                                                      \
    {                                                                       \
        /* Traverse the list of ports to find the port of given port ID. */ \
        while(arg_pPhysicalIf != NULL)                                      \
        {                                                                   \
            if(arg_pPhysicalIf->pPhysicalIfInfo->portId == arg_portId)      \
                break;                                                      \
            arg_pPhysicalIf = arg_pPhysicalIf->pNextPhysicalIf;             \
        }                                                                   \
    }                                                                       \
    while(0)

/**
 * TYPENAME: ix_cc_stkdrv_vidd_physical_if_node
 * 
 * DESCRIPTION: This list of data structures are created upon
 * initialization by the VIDD and represent the hardware network
 * interfaces.
 *
 */
typedef struct ix_s_cc_stkdrv_vidd_ctrl ix_cc_stkdrv_vidd_ctrl;
typedef struct ix_s_cc_stkdrv_vidd_physical_if_node ix_cc_stkdrv_vidd_physical_if_node;
struct ix_s_cc_stkdrv_vidd_physical_if_node
{
    struct net_device       netdev;    /* port API to protocol stack */
    struct net_device_stats stats;     /* port statistics */
    
    /* circular reference to VIDD control structure */
    ix_cc_stkdrv_vidd_ctrl* pViddCtrl;
    
    /* pointer to a context on the CC side, used when passing packets from the VIDD. */
    void* pCCPktContext;
    
    /* Pointer to port info shared between the CC Module and the VIDD. */
    ix_cc_stkdrv_physical_if_info* pPhysicalIfInfo;
    
    /* next interface structure in the list of the structures. */
    ix_cc_stkdrv_vidd_physical_if_node* pNextPhysicalIf;
};

/**
 * TYPENAME: ix_cc_stkdrv_vidd_fp_node
 * 
 * DESCRIPTION: This is the structure representing a Forwarding Plane Object.
 * It is constructed as a linked list for future expansion of FPs.
 * Created by VIDD during initialization.
 *
 */
typedef struct ix_s_cc_stkdrv_vidd_fp_node ix_cc_stkdrv_vidd_fp_node;
struct ix_s_cc_stkdrv_vidd_fp_node
{
    /* ID of this forwarding plane */
    ix_uint32 fpId;
    
    /* Head of linked list of physical interfaces for the forwarding plane */
    ix_cc_stkdrv_vidd_physical_if_node *pPhysicalIfs;
    
    /* number of ports on this FP. */
    ix_uint32 numPorts;

    /* Pointer to next FP in the list - in this release this will be NULL */
    ix_cc_stkdrv_vidd_fp_node *pNextFp;
};


/**
 * TYPENAME: ix_cc_stkdrv_vidd_ctrl
 * 
 * DESCRIPTION: This structure is defined by the driver and is created during
 * the initialization. It holds the control information about interfaces and
 * forwarding planes, and can be used as a context for all communication
 * between the CC Module and the VIDD.
 *
 */
struct ix_s_cc_stkdrv_vidd_ctrl
{
    struct tq_struct          ipSetupTask;
    int                       ipSetupStop;
#if defined(IX_IPV6_SUPPORT)
    struct tq_struct          ipPropUpdateTask; /* For setting and */
    int                       ipPropUpdateStop; /* deleting IPv6 address */
                                                /* of an interface */
    struct tq_struct          ipPropDeleteTask;
    int                       ipPropDeleteStop;
#endif
    ix_cc_stkdrv_vidd_fp_node *pFps;    /* list of forwarding planes */
    ix_buffer_free_list_handle hFreeList; /* a handle to tx buffers freelist */
};

#if defined(IX_IPV6_SUPPORT)
/**
 * TYPENAME: ix_s_cc_stkdrv_prop_update
 *
 * DESCRIPTION: This structure holds the IPv6 address to be added or deleted
 *              the Vidd control block, the operation as to whether to add or
 *              delete the IPv6 address. This is used by the Update property
 *              function to pass the above mentioned parameters to the bottom
 *              half. Memory is allocated to this structure and the memory is
 *              is freed by the bottom half itself.
 *
 */
typedef struct ix_s_cc_stkdrv_prop_update
{
   ix_uint32              propId;
   ix_cc_properties       property;
   ix_cc_stkdrv_vidd_ctrl *pViddCtrl;
} ix_cc_stkdrv_prop_update;
#endif

/**
 * Prototypes for interface functions.
 */
/**
 * NAME: ix_cc_stkdrv_vidd_fini
 *
 * DESCRIPTION: This function is called by the CC Module to shutdown the local
 * VIDD and free any memory allocated to it.
 * 
 * @Param:  - IN void *arg_pContext - pointer to context, in this case the VIDD
 *          control structure.
 *
 * @Return: IX_SUCCESS
 */
ix_error ix_cc_stkdrv_vidd_fini(
                                void *arg_pContext
                                );




/**
 * NAME: ix_cc_stkdrv_vidd_receive_pkt
 *
 * DESCRIPTION: This is the VIDD entry point (packet processing
 * callback invoked by the CC Module) that receives packets from
 * the CC Module and passes them to the local network stack.
 * 
 * @Param:  - IN ix_buffer_handle arg_hBuffer - handle to the input packet.
 * @Param:  - IN void* arg_pCtx - pointer to the context, in this case
 *          the port structure.
 * @Param:  - IN ix_cc_stkdrv_packet_type packetType - packet type.
 *          Ignored in the VIDD case.
 *
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_OOM_SYSTEM
 *          IX_CC_ERROR_ENTRY_NOT_FOUND
 *          IX_CC_ERROR_NULL
 */
ix_error ix_cc_stkdrv_vidd_receive_pkt(
                                       ix_buffer_handle arg_pBuffer,
                                       void *arg_pCtx, 
                                       ix_cc_stkdrv_packet_type packetType
                                       );


/**
 * NAME: ix_cc_stkdrv_vidd_receive_msg_int
 *
 * DESCRIPTION: This is the VIDD entry point (integer message processing
 * callback invoked by the CC Module) that receives integer messages
 * from the CC Module and processes the interface property updates
 * contained within those messages.
 * 
 * @Param:  - IN ix_cc_stkdrv_vidd_msg_id arg_MsgId - identification of the message.
 * @Param:  - IN ix_uint32 arg_msg - integer message value.
 * @Param:  - IN void *arg_pContext - pointer to the context.  Here the context
 *          is the physical interface structure.
 *
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_NULL
 *          IX_CC_STKDRV_VIDD_ERROR_MUX
 *          IX_CC_ERROR_UNDEFINED_MSG
 */
ix_error ix_cc_stkdrv_vidd_receive_msg_int(
                                           ix_cc_stkdrv_vidd_msg_id arg_msgId,
                                           const ix_uint32 arg_msg,
                                           void *arg_pContext,
					    void** arg_ppReplyMsg,
					    ix_uint32* arg_pReplyMsgSize

                                           );




/**
 * NAME: ix_cc_stkdrv_vidd_ifd_tx
 *
 * DESCRIPTION: transmit function called by the kernel for this driver. The packet
 * is sent to the output. The output is typically bound to IPv4 Fwdr CC.
 * This fn stores the packet data in an ix_buffer_handle before calling the stack driver
 * core component API ix_cc_stkdrv_send_packet() function.
 *  Note:
 * When a socket function like sendto is called by an application, it
 * translates in to a call to this function. If the amount of data to be
 * transmitted is small enough to be in a single packet then the kernel
 * may call this function in the context of the user process. This is not a 
 * problem. However if the data is large then the kernel queues multiple
 * packets. This results in multiple calls to ifd_tx and these are made 
 * from the bottom halves. i.e the context is arbitrary and many limitations
 * that apply to interrupt handlers apply to this function. e.g you 
 * cannot sleep/wait in this function.
 *
 * 
 * @Param:  - IN net_device *arg_pDev - pointer to device.It identifies the 
 *                                  transmit interface.
 * @Param:  - IN struct sk_buff *arg_pSkb - sk_buffer containing the network
 *          buffer. The data represents the full link layer frame. 
 * @Return: zero for success
 *          Non-zero for error
 */
int
ix_cc_stkdrv_vidd_ifd_tx(struct sk_buff *arg_pSkb, struct net_device *arg_pDev);

/**
 * Exported variables.
 */


#if defined(__cplusplus)
}
#endif /* end defined(__cplusplus) */

#endif /* end !defined(__IX_CC_STKDRV_VIDD_H__) */
