/**
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2000-2001 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK 3.0 for the IXP2000 Network Processor, Release 6
 *
 * = FILENAME
 *      ix_cc_stkdrv_pktclass.h
 *
 * = DESCRIPTION
 *      Header file for the packet classification in the Stack Driver CC Module.
 *
 * = AUTHOR
 *      Aaron Luk
 *      aaron.luk@intel.com
 *
 * = CHANGE HISTORY
 *       6/19/2002 - initial revision
 *
 * ============================================================================
 * $Id: ix_cc_stkdrv_pktclass.h,v 1.21 2003/12/16 23:46:52 ktseng Exp $
 */
#if !defined(__IX_CC_STKDRV_PKTCLASS_H__)
#define __IX_CC_STKDRV_PKTCLASS_H__

#if defined(IX_PLATFORM_2401) || defined(IX_PLATFORM_2801)
/* Improvement
 * Cleanup in headers - don't include user space stdio.h, use <> when including
 * headers from global include paths.
 */
#include <ix_ossl.h>
#include <ix_rm.h>

#include <ix_cc_error.h>
#include <ix_cc.h>
#include <ix_cc_properties.h>

#else /* IX_PLATFORM_2x01 */

#include "ix_cc_error.h"
#include <stdio.h>
#include "ix_ossl.h"
#include "ix_rm.h"
#include "ix_cc.h"
#include "ix_cc_properties.h"

#endif /* IX_PLATFORM_2x01 */

#if defined(__cplusplus)
extern "C"
{
#endif /* end defined(__cplusplus) */

/**
 * Pre-processor symbol and macro definitions.
 */

/* Maximum number of filters. */
#define IX_CC_STKDRV_NUM_FILTERS 16

/* Definitions of filter flag field */
#define IX_CC_STKDRV_FILTER_DIP_CLASS       0x01 /* classify by destination IP */
#define IX_CC_STKDRV_FILTER_INPORT_CLASS    0x02 /* classify by input port ID */
#define IX_CC_STKDRV_FILTER_ALL_CLASS       IX_CC_STKDRV_FILTER_DIP_CLASS | IX_CC_STKDRV_FILTER_INPORT_CLASS /* use all classifications */

/* V6 offset starting from start of header */
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
#define IX_CC_STKDRV_IPV6_DST_IP_OFFSET 6
#endif

#if defined(IX_PLATFORM_2800)
#define IX_CC_STKDRV_NUM_PHYS_PORTS        10
#else
#define IX_CC_STKDRV_NUM_PHYS_PORTS        4
#endif

/**
 * Type definitions.
 */
/**
 * TYPENAME: ix_cc_stkdrv_filter_type
 * 
 * DESCRIPTION: This data type holds values for possible future filter types, but
 * we will only be implementing the first filter type, based on IPv4 destination
 * IP and input port ID, for now.
 *
 */
typedef enum ix_e_stkdrv_filter_type
{
    IX_CC_STKDRV_FILTER_TYPE_FIRST = 0,
    /* IPv4 destination IP and input portID */
    IX_CC_STKDRV_FILTER_TYPE_IPV4_DIP_INPORT = IX_CC_STKDRV_FILTER_TYPE_FIRST,
    IX_CC_STKDRV_FILTER_TYPE_IPV6_DIP_INPORT,
    IX_CC_STKDRV_FILTER_TYPE_IPV4_5TUPLE,
    IX_CC_STKDRV_FILTER_TYPE_IPV4_6TUPLE,
    IX_CC_STKDRV_FILTER_TYPE_IPV6_SIMPLE,
    IX_CC_STKDRV_FILTER_TYPE_ARP_FILTER,
    IX_CC_STKDRV_FILTER_TYPE_ATM_FILTER,
    IX_CC_STKDRV_FILTER_TYPE_FRAME_RELAY,
    IX_CC_STKDRV_FILTER_TYPE_POS_FILTER,
    IX_CC_STKDRV_FILTER_TYPE_UNDEFINED,
    IX_CC_STKDRV_FILTER_TYPE_LAST
} ix_cc_stkdrv_filter_type;


/**
 * TYPENAME: ix_cc_stkdrv_filter_priority
 * 
 * DESCRIPTION: This data type determines whether a filter is of
 * high, medium, or low priority. Higher priority filters will be
 * checked first during packet classification.
 *
 */
typedef enum ix_e_cc_stkdrv_filter_priority
{
    IX_CC_STKDRV_FILTER_PRIORITY_FIRST = 0,
    IX_CC_STKDRV_FILTER_PRIORITY_HIGH = IX_CC_STKDRV_FILTER_PRIORITY_FIRST,
    IX_CC_STKDRV_FILTER_PRIORITY_LOW,
    IX_CC_STKDRV_FILTER_PRIORITY_LAST
} ix_cc_stkdrv_filter_priority;


/**
 * TYPENAME: ix_cc_stkdrv_ipv4_address_range
 * 
 * DESCRIPTION: Structure for specifying an inclusive address range.
 *
 */
typedef struct ix_s_cc_stkdrv_ipv4_address_range
{
    ix_uint32 start; /* Start address (inclusive) in host order */
    ix_uint32 end;   /* End address (inclusive) in host order.*/
} ix_cc_stkdrv_ipv4_address_range;


/**
 * TYPENAME: ix_cc_stkdrv_port_range
 * 
 * DESCRIPTION: Structure for specifying an inclusive port ID range.
 *
 */
typedef struct ix_s_cc_stkdrv_port_range
{
    ix_uint32 start; /* Start port ID */
    ix_uint32 end;   /* End port ID */
} ix_cc_stkdrv_port_range;


/**
 * TYPENAME: ix_cc_stkdrv_ipv4_range_filter
 * 
 * DESCRIPTION: Structure for specifying a range filter.
 *
 */
typedef struct ix_s_cc_stkdrv_ipv4_range_filter
{
    ix_cc_stkdrv_ipv4_address_range dstAddrs;
    ix_cc_stkdrv_port_range inPorts; 
    ix_uint16 flag; /* selects one or both fields for classification */
} ix_cc_stkdrv_ipv4_range_filter;


#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
/**
 * TYPENAME: ix_cc_stkdrv_og_filter_type
 * 
 * DESCRIPTION: 
 * This is used to label the filter type being used for any outgoing packet.
 */
typedef enum ix_e_cc_stkdrv_og_filter_type
{
    IX_CC_STKDRV_OG_FILTER_TYPE_FIRST = 0,
    IX_CC_STKDRV_OG_FILTER_TYPE_PKT,
    IX_CC_STKDRV_OG_FILTER_TYPE_USR_DEFINED_0,
    IX_CC_STKDRV_OG_FILTER_TYPE_UNDEFINED,
    IX_CC_STKDRV_OG_FILTER_LAST
} ix_cc_stkdrv_og_filter_type;


/**
 * TYPENAME: ix_cc_stkdrv_og_cc_type
 * 
 * DESCRIPTION: 
 * This is used to identify the outgoing core-component type
 */
typedef enum ix_e_cc_stkdrv_og_cc_type 
{
     IX_CC_STKDRV_OG_CC_TYPE_FIRST = 0,
     IX_CC_STKDRV_OG_CC_TYPE_IPV4_FWDER,
     IX_CC_STKDRV_OG_CC_TYPE_IPV6_FWDER,
     IX_CC_STKDRV_OG_CC_TYPE_Q_MGR,
     IX_CC_STKDRV_OG_CC_TYPE_UNDEFINED,
     IX_CC_STKDRV_OG_CC_TYPE_LAST
} ix_cc_stkdrv_og_cc_type;

typedef struct ix_s_cc_og_stkdrv_filter ix_cc_stkdrv_og_filter;

/**
 * TYPENAME: ix_cc_stkdrv_cb_og_chk_filter
 * 
 * DESCRIPTION: 
 * This is the function pointer typedef for the actual outgoing pkt 
 * filtering function
 */
typedef ix_error (*ix_cc_stkdrv_cb_og_chk_filter)(
                        ix_buffer_handle arg_hBuffer,
                        ix_uint8 arg_bufferType,
                        ix_cc_stkdrv_physical_if_node *arg_pPhysicalIf,
                        ix_cc_stkdrv_og_filter *arg_pFilter,
                        ix_cc_stkdrv_og_cc_type *arg_pCcType,
                        void                    **arg_ppData);

/**
 * TYPENAME: ix_cc_stkdrv_og_filter
 * 
 * DESCRIPTION: 
 * This is the outgoing filter structure. There would be one such structure 
 * per filter type in use 
 */
struct ix_s_cc_og_stkdrv_filter
{
    ix_cc_stkdrv_og_filter_type   filterType; /* Filter type for this filter */
    ix_cc_stkdrv_port_range       portRange; /* Applicable port range */
    void                         *pCritereon; /* Associated critereon */
    ix_cc_stkdrv_cb_og_chk_filter aOgChkFunc; /* Filter Function Pointer*/
};


/**
 * TYPENAME: ix_cc_stkdrv_og_comm_cc_map
 * 
 * DESCRIPTION: 
 * This is the map which is maintained between core component & its
 * corresponding communicationId
 */
typedef struct ix_s_cc_stkdrv_og_comm_cc_map
{
    ix_cc_stkdrv_og_cc_type  ccType;   /* Core Component Type */
    ix_communication_id   commId;   /* Associated Communication Id */
} ix_cc_stkdrv_og_comm_cc_map;


/**
 * TYPENAME: ix_cc_stkdrv_og_filter_ctrl
 * 
 * DESCRIPTION: 
 * This is the top level outgoing filter control structure. It contains 
 * all the defined filters and the commId/CCtype map
 */
typedef struct ix_s_cc_stkdrv_og_filter_ctrl
{
      /* Array containing the actual filter data. */
      ix_cc_stkdrv_og_filter  aFilterArray[IX_CC_STKDRV_OG_FILTER_LAST];
      /* communicationIds in use in outgoing path */
      ix_cc_stkdrv_og_comm_cc_map aMap[IX_CC_STKDRV_OG_CC_TYPE_LAST];  
      /* Number of filters in use. */
      ix_uint32 filterCount;
} ix_cc_stkdrv_og_filter_ctrl;


typedef struct ix_s_cc_stkdrv_ipv6_address_range
{
    ix_uint128 start;
    ix_uint128 end;
} ix_cc_stkdrv_ipv6_address_range;

typedef struct ix_s_cc_stkdrv_ipv6_range_filter
{
    ix_cc_stkdrv_ipv6_address_range dstAddrs;
    ix_cc_stkdrv_port_range inPorts; 
    ix_uint16 flag; /* selects one or both fields for classification */
}ix_cc_stkdrv_ipv6_range_filter;

#endif /* _IX_OS_LINUX_KERNEL_ */

/**
 * TYPENAME: ix_cc_stkdrv_filter
 * 
 * DESCRIPTION: Structure for the filter.
 *
 */
typedef struct ix_s_cc_stkdrv_filter
{
    ix_cc_stkdrv_handler_id handlerID; /* the communication handler associated with this filter */
    ix_cc_stkdrv_packet_type packetType; /* the type of packet associated with this filter */
    ix_cc_stkdrv_filter_type filterType; /* the type of filter */
    ix_cc_stkdrv_ipv4_range_filter filterCriteria; /* filter criteria */

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
    ix_cc_stkdrv_ipv6_range_filter v6FilterCriteria; /* V6 filter criteria */
#endif
} ix_cc_stkdrv_filter;


/**
 * TYPENAME: ix_cc_stkdrv_filter_handle
 * 
 * DESCRIPTION: Opaque handler to a filter.  Will be filled in when a filter
 * is added and passed in when a filter is removed.
 *
 *
 * The following is the fliter handle mapping.
 *      LW  Bits    Size    Field           Description
 *      -----------------------------------------------------------------
 *      0   31:0    32      Filter index    index of this filter in the filterArray
 *
 */
typedef ix_handle ix_cc_stkdrv_filter_handle;



/**
 * TYPENAME: ix_cc_stkdrv_filter_index_node
 * 
 * DESCRIPTION: Node in a linked list of filter indices.
 *
 */
typedef struct ix_s_cc_stkdrv_filter_index_node ix_cc_stkdrv_filter_index_node;
struct ix_s_cc_stkdrv_filter_index_node
{
    /* Value of the filter index. */
    ix_uint32 filterIndex;

    /* Pointer to the previous index in the list. */
    ix_cc_stkdrv_filter_index_node* pPrev;

    /* Pointer to the next index in the list. */
    ix_cc_stkdrv_filter_index_node* pNext;
};


/**
 * TYPENAME: ix_cc_stkdrv_cb_filter_ops
 * 
 * DESCRIPTION: Callback function from all asynchronous filter operations.
 *
 * @Param:  - IN ix_error arg_Result - error code returned from the
 *          messaging operation.
 * @Param:  - IN void *arg_pContext - pointer to user context.
 * @Param:  - IN ix_cc_stkdrv_filter_handle* arg_pFilterHandle - pointer to
 *          where the filter handle is returned, or NULL if it is irrelevant.
 *
 * @Return: IX_SUCCESS, or an appropriate error code.
 *
 */
typedef ix_error (*ix_cc_stkdrv_cb_filter_ops)(
                                               ix_error arg_Result,
                                               void *arg_pContext,
                                               ix_cc_stkdrv_filter_handle* arg_pFilterHandle
                                               );

/**
 * TYPENAME: ix_cc_stkdrv_cb_check_filter
 * 
 * DESCRIPTION: Function pointer to check a filter of a specific type.
 *
 * @Param:  - IN ix_buffer_handle arg_hBuffer - handle to the buffer to be classified.
 * @Param:  - IN ix_cc_stkdrv_filter* arg_pFilter - pointer to the filter to be checked.
 * @Param:  - OUT ix_uint32* arg_pFoundMatch - pointer to integer indicating whether
 *          a filter matched.
 *
 * @Return: IX_SUCCESS, or an appropriate error code.
 *
 */
typedef ix_error (*ix_cc_stkdrv_cb_check_filter)(
                                                 ix_buffer_handle arg_hBuffer,
                                                 ix_cc_stkdrv_filter* arg_pFilter,
                                                 ix_uint32* arg_pFoundMatch
                                                 );

/**
 * TYPENAME: ix_cc_stkdrv_filter_ctrl
 * 
 * DESCRIPTION: This is the control structure for the filters.  It will
 * be passed in as a context to all of the filtering API calls.
 *
 */
typedef struct ix_s_cc_stkdrv_filter_ctrl
{
    /* Array containing the actual filter data. */
    ix_cc_stkdrv_filter filterArray[IX_CC_STKDRV_NUM_FILTERS];

    /**
     * Array of linked lists of indices for filters in use -
     * one linked list for each priority.
     */
    ix_cc_stkdrv_filter_index_node* pUsedFilterIndices[IX_CC_STKDRV_FILTER_PRIORITY_LAST];

    /* Linked list of free filter indices. */
    ix_cc_stkdrv_filter_index_node* pFreeFilterIndices;

    /**
     * Array of all filter index nodes.  Entries from here are
     * placed in either the free index list of an appropriate
     * priority index list.
     */
    ix_cc_stkdrv_filter_index_node aAllFilterIndices[IX_CC_STKDRV_NUM_FILTERS];

    /* Number of filters in use. */
    ix_uint32 filterCount;

    /* Array of function pointers used to check filter of specific types. */
    ix_cc_stkdrv_cb_check_filter aCheckFilter[IX_CC_STKDRV_FILTER_TYPE_LAST];

} ix_cc_stkdrv_filter_ctrl;


/**
 * Prototypes for interface functions.
 */

#if defined(IX_CC_STKDRV_PKT_CLASS)
/**
 * NAME: ix_cc_stkdrv_init_filters
 *
 * DESCRIPTION: This function creates an empty array of filter handlers
 * and initializes each priority list to NULL.
 * 
 * @Param:  - OUT ix_cc_stkdrv_filter_ctrl** arg_ppFilterCtrl - location of
 *          pointer to filter control structure.
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_OOM_SYSTEM
 *          IX_CC_ERROR_INTERNAL
 *
 */
ix_error ix_cc_stkdrv_init_filters(
                                   ix_cc_stkdrv_filter_ctrl** arg_ppFilterCtrl
                                   );

/**
 * NAME: ix_cc_stkdrv_fini_filters
 *
 * DESCRIPTION: This function finalizes the filter control structure.
 * 
 * @Param:  - INOUT ix_cc_stkdrv_filter_ctrl** arg_ppFilterCtrl - location of
 *          pointer to filter control structure.
 * @Return: IX_SUCCESS
 */
ix_error ix_cc_stkdrv_fini_filters(
                                   ix_cc_stkdrv_filter_ctrl** arg_ppFilterCtrl
                                   );


/**
 * NAME: ix_cc_stkdrv_classify_pkt
 *
 * DESCRIPTION: This function classifies a packet according to the installed
 * packet filters.  Packets that do not match any classifiation will be
 * treated as local legacy packets.
 * 
 * @Param:  - IN ix_buffer_handle arg_hBuffer - handle to the packet to
 *          be classified.
 * @Param:  - IN ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl - pointer to the
 *          filter control structure.
 * @Param:  - OUT ix_cc_stkdrv_handler_id* arg_pHandlerID - pointer to the
 *          handler ID which is associated with the input packet's classification.
 *          The classification function tells us which communication handler the
 *          packet should be sent to.
 * @Param:  - OUT ix_cc_stkdrv_packet_type* arg_pPacketType - pointer to the
 *          packet type which is associated with the input packet's classification.
 *          The classification function tells us what type of packet this is
 *          (local or remote legacy stack, local or remote NPF stack).
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_INTERNAL
 *
 */
ix_error ix_cc_stkdrv_classify_pkt(
                                   ix_buffer_handle arg_hBuffer,
                                   ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl,
                                   ix_cc_stkdrv_handler_id* arg_pHandlerID,
                                   ix_cc_stkdrv_packet_type* arg_pPacketType
                                   );


/**
 * NAME: ix_cc_stkdrv_async_add_filter
 *
 * DESCRIPTION: This function is called to add a packet filter.
 * 
 * @Param:  - IN ix_cc_stkdrv_cb_get_num_ports arg_pCallback - pointer to the
 *          client's callback function to be invoked once the message returns.
 * @Param:  - IN void* arg_pContext - pointer to client context.
 * @Param:  - IN ix_cc_stkdrv_filter* arg_pFilter - pointer to the filter to be added.
 * @Param:  - IN ix_cc_stkdrv_filter_priority arg_priority - determines
 *          whether this filter is of high, medium, or low priority. During
 *          classification higher priority filters will be checked first.
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_OOM_SYSTEM
 */
ix_error ix_cc_stkdrv_async_add_filter(
                                       ix_cc_stkdrv_cb_filter_ops arg_pCallback,
                                       void* arg_pContext,
                                       ix_cc_stkdrv_filter* arg_pFilter,
                                       ix_cc_stkdrv_filter_priority arg_priority
                                       );

/**
 * NAME: ix_cc_stkdrv_async_remove_filter
 *
 * DESCRIPTION: This function is called to remove a packet filter.
 * 
 * @Param:  - IN ix_cc_stkdrv_cb_get_num_ports arg_pCallback - pointer to the
 *          client's callback function to be invoked once the message returns.
 * @Param:  - IN void* arg_pContext - pointer to client context.
 * @Param:  - IN ix_cc_stkdrv_filter_handle arg_filterHandle - handle to the
 *          filter to be removed.
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_NULL
 *          IX_CC_ERROR_OOM_SYSTEM
 */
ix_error ix_cc_stkdrv_async_remove_filter(
                                          ix_cc_stkdrv_cb_filter_ops arg_pCallback,
                                          void* arg_pContext,
                                          ix_cc_stkdrv_filter_handle arg_filterHandle
                                          );


/**
 * NAME: ix_cc_stkdrv_async_remove_all_filters
 *
 * DESCRIPTION: This function is called to remove all packet filters.
 * 
 * @Param:  - IN ix_cc_stkdrv_cb_get_num_ports arg_pCallback - pointer to the
 *          client's callback function to be invoked once the message returns.
 * @Param:  - IN void* arg_pContext - pointer to client context.
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_NULL
 *          IX_CC_ERROR_OOM_SYSTEM
 */
ix_error ix_cc_stkdrv_async_remove_all_filters(
                                               ix_cc_stkdrv_cb_filter_ops arg_pCallback,
                                               void* arg_pContext
                                               );

/**
 * NAME: ix_cc_stkdrv_async_modify_filter
 *
 * DESCRIPTION: This function is called to modify a packet filter.
 * 
 * @Param:  - IN ix_cc_stkdrv_cb_get_num_ports arg_pCallback - pointer to the
 *          client's callback function to be invoked once the message returns.
 * @Param:  - IN void* arg_pContext - pointer to client context.
 * @Param:  - IN ix_cc_stkdrv_filter* arg_pFilter - pointer to the new filter info.
 * @Param:  - IN ix_cc_stkdrv_filter_priority arg_priority - determines
 *          whether this filter is of high, medium, or low priority. During
 *          classification higher priority filters will be checked first.
 * @Param:  - IN ix_cc_stkdrv_filter_handle arg_filterHandle - handle to the
 *          filter to be modified.
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_NULL
 *          IX_CC_ERROR_OOM_SYSTEM
 */
ix_error ix_cc_stkdrv_async_modify_filter(
                                          ix_cc_stkdrv_cb_filter_ops arg_pCallback,
                                          void* arg_pContext,
                                          ix_cc_stkdrv_filter* arg_pFilter,
                                          ix_cc_stkdrv_filter_priority arg_priority,
                                          ix_cc_stkdrv_filter_handle arg_filterHandle
                                          );


/**
 * NAME: ix_cc_stkdrv_async_register_filter_type
 *
 * DESCRIPTION: This function is called to register a callback function for a
 *              specific filter type asynchronously.
 * 
 * @Param:  - IN ix_cc_stkdrv_cb_filter_ops arg_pCallback - pointer to the
 *          client's callback function to be invoked once the message returns.
 * @Param:  - IN void* arg_pContext - pointer to client context.
 * @Param:  - IN ix_cc_stkdrv_filter_type arg_filterType - type of the filter
 *          being registered.
 * @Param:  - IN ix_cc_stkdrv_cb_check_filter arg_checkFilterCB - callback
 *          function to check a filter of a specific type.
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_NULL
 *          IX_CC_ERROR_RANGE
 *          IX_CC_ERROR_OOM_SYSTEM
 */
ix_error ix_cc_stkdrv_async_register_filter_type(
                                                 ix_cc_stkdrv_cb_filter_ops arg_pCallback,
                                                 void* arg_pContext,
                                                 ix_cc_stkdrv_filter_type arg_filterType,
                                                 ix_cc_stkdrv_cb_check_filter arg_checkFilterCB
                                                 );


/**
 * NAME: ix_cc_stkdrv_async_unregister_filter_type
 *
 * DESCRIPTION: This function is called to unregister a callback function for a
 *              specific filter type asynchronously.
 * 
 * @Param:  - IN ix_cc_stkdrv_cb_filter_ops arg_pCallback - pointer to the
 *          client's callback function to be invoked once the message returns.
 * @Param:  - IN void* arg_pContext - pointer to client context.
 * @Param:  - IN ix_cc_stkdrv_filter_type arg_filterType - type of the filter
 *          being registered.
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_NULL
 *          IX_CC_ERROR_RANGE
 *          IX_CC_ERROR_OOM_SYSTEM
 */
ix_error ix_cc_stkdrv_async_unregister_filter_type(
                                                   ix_cc_stkdrv_cb_filter_ops arg_pCallback,
                                                   void* arg_pContext,
                                                   ix_cc_stkdrv_filter_type arg_filterType
                                                   );


#endif /* defined(IX_CC_STKDRV_PKT_CLASS) */

/**
 * Exported variables.
 */

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)

extern ix_cc_stkdrv_og_filter_ctrl *g_pFilterCtrl;

extern ix_error ix_cc_stkdrv_init_og_filters(
                  ix_cc_stkdrv_og_filter_ctrl **arg_ppFilterCtrl);

extern ix_error ix_cc_stkdrv_fini_og_filters(
                  ix_cc_stkdrv_og_filter_ctrl **arg_ppFilterCtrl);

extern ix_error _ix_cc_stkdrv_classify_output_id(
                  ix_buffer_handle              arg_hBuffer,
                  ix_uint8                      arg_pktType, 
                  ix_communication_id           *arg_pCommId,
                  ix_cc_stkdrv_physical_if_node *arg_pPhysicalIf,
                  void                          **arg_ppData);

#endif /* _IX_OS_LINUX_KERNEL_ */



#if defined(__cplusplus)
}
#endif /* end defined(__cplusplus) */

#endif /* end !defined(__IX_CC_STKDRV_PKTCLASS_H__) */

