/**
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2000-2003 Intel Corporation.All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK 3.1 for the IXP2000 Network Processor
 *
 * = FILENAME
 *      ix_cc_stkdrv_vidd.c
 *
 * = DESCRIPTION
 *      Source file for the VIDD component of the Stack Driver for Linux.
 *
 *
 * = CHANGE HISTORY
 *      08/18/2003 - creation time
 *
 * ============================================================================
 * $Id: ix_cc_stkdrv_vidd_linux.c,v 1.21.2.1 2004/03/16 19:16:38 rranjeet Exp $
 * 
 */

#define IX_ERROR_FILE_IDENT "$Id: ix_cc_stkdrv_vidd_linux.c,v 1.21.2.1 2004/03/16 19:16:38 rranjeet Exp $"

/**
 * User defined include files required.
 */

#include "ix_cc_error.h"
#include <linux/ctype.h>
#include <ix_ossl.h>
#include "ix_rm.h"
#include "ix_cc.h"
#include "cc/ix_cc_stkdrv_common.h"
#include "cc/ix_cc_stkdrv_pktclass.h"
#include "cc/ix_cc_stkdrv_vidd_linux.h"
#if defined(IX_IPV6_SUPPORT)
#include <linux/ipv6.h>
#include "definitions.h"
#endif
/*****************************************************************************
 * Definitions
 *****************************************************************************/

#ifndef BUILDNAME
#define BUILDNAME "undefined"
#endif

ix_cc_stkdrv_vidd_ctrl             *gViddCtrl;

int atoi(const char *nptr);

MODULE_PARM(stkdrvTraceLevel, "i");

int  stkdrvTraceLevel = 1;
/*
 * Trace levels shall be used in the following way:
 * 0 - stay quite
 * 1 - trace major errors
 * 2 - trace major operations and minor errors
 * 3 - trace all function calls except for periodic ones
 * 4 - trace all functions, even periodic, and packets
 */

/**
 * Types definitions whose scope is limited to this file.
 */
enum ifd_pioctl {
	  IFD_SET_MAC = SIOCDEVPRIVATE,
};



/****************************************************************************/

/**
 * Extern function prototypes.
 */

/**
 * Static function prototypes.
 */

PRIVATE ix_error _ix_cc_stkdrv_vidd_cleanup(ix_uint32 arg_checkErrors, void** arg_ppViddCtrl);

static int  ix_cc_stkdrv_vidd_change_mtu(struct net_device*, int);
static struct net_device_stats*
            ix_cc_stkdrv_vidd_ifd_get_stats(struct net_device*);
static int  ix_cc_stkdrv_vidd_hard_header(struct sk_buff*, struct
                          net_device*, unsigned short, void*, void*, unsigned);
static int  ix_cc_stkdrv_vidd_hard_parse(struct sk_buff*, unsigned char*);
static int  ix_cc_stkdrv_vidd_ifd_do_ioctl(struct net_device*, struct ifreq*, int);
static ix_error
            ix_cc_stkdrv_vidd_ip_setup(ix_cc_stkdrv_vidd_physical_if_node*);
static void ix_cc_stkdrv_vidd_ip_setup_task(void *);
#if defined(IX_IPV6_SUPPORT)
static int
ix_cc_stkdrv_delete_ipv6_addr(const struct net_device *arg_pDev,
    const ix_uint128 *arg_addr, const ix_uint8 arg_prefixlen);
static int ix_cc_stkdrv_set_ipv6_addr(const struct net_device *arg_pDev,
    const ix_uint128 *arg_addr, const ix_uint8 arg_prefixlen);
static void ix_cc_stkdrv_vidd_ipv6_setup_prop_update_task(void *arg);
#endif
PRIVATE ix_error
            _ix_cc_stkdrv_vidd_morph_ip_id(char* arg_pHeader);
static int  ix_cc_stkdrv_vidd_ifd_open(struct net_device*);
ix_error _ix_cc_stkdrv_vidd_set_property(
                         ix_uint32 arg_propID,
                         ix_cc_properties* arg_pProperty,
                         ix_cc_stkdrv_vidd_ctrl* arg_pViddCtrl
                              );

/*****************************************************************************
 * Variables
 *****************************************************************************/

/*****************************************************************************
 * Function definitions.
 *****************************************************************************/

/**
 * NAME: ix_cc_stkdrv_vidd_if_devinet_ioctl
 *
 * DESCRIPTION: Wrapper for kernel mode ioctl. If stack driver CC is running in user mode we could
 * simply use ioctl calls to set various properties of the ineterface. This is 
 * the kernel mode equivalent of that. The ioctl calls are divided in two classes
 * (by linux). One is protocol independent ioctl and other is protocol dependent
 * ioctl. This function is protocol dependent ioctl. The protocol is IP.
 *
 * 
 * @Param:  - IN arg_cmd - ioctl command value
 * @Param:  - IN arg_pIfReq - interface request structure
 *
 *
 * @Return: Zero for success, non-zero for failure 
 */
static int ix_cc_stkdrv_vidd_if_devinet_ioctl(unsigned int arg_cmd,
                                struct ifreq *arg_pIfReq)
{
    int err;
    mm_segment_t oldfs = get_fs();

    set_fs(get_ds());
    err = devinet_ioctl(arg_cmd, arg_pIfReq);
    set_fs(oldfs);

    return err;
}


/**
 * NAME: ix_cc_stkdrv_vidd_if_dev_ioctl
 *
 * DESCRIPTION:  Wrapper for kernel mode ioctl. If stack driver is running in user mode we could
 * simply use ioctl calls to set various properties of the ineterface. This is 
 * the kernel mode equivalent of that. The ioctl calls are divided in two classes
 * (by linux). One is protocol independent ioctl and other is protocol dependent
 * ioctl. This function is protocol independent ioctl.
 
 * 
 * @Param:  - IN arg_cmd - ioctl command value
 * @Param:  - IN arg_pIfReq - interface request structure
 *
 *
 * @Return: Zero for success, non-zero for failure 
 */
static int ix_cc_stkdrv_vidd_if_dev_ioctl(unsigned int arg_cmd, struct ifreq *arg_pIfReq)
{
    int err;
    mm_segment_t oldfs = get_fs();

    set_fs(get_ds());
    err = dev_ioctl(arg_cmd, arg_pIfReq);
    set_fs(oldfs);

    return err;
}

/**
 * NAME: ix_cc_stkdrv_vidd_ifd_open
 *
 * DESCRIPTION: Open function called by the kernel for this driver. 
 *              This is for compatibility with older versions.This
 *              is useful if your development environment is using
 *              a slightly older version of Linux (like RH 6.2) than
 *              the target Linux
 * 
 * @Param:  - IN net_device *arg_pDev  - pointer to net_device
 *
 * @Return: Zero
 */
static int
ix_cc_stkdrv_vidd_ifd_open(struct net_device *arg_pDev)
{
    return 0;
}


/**
 * NAME: ix_cc_stkdrv_vidd_ifd_stop
 *
 * DESCRIPTION: Stop function called by the kernel for this driver. 
 *              This is for compatibility with older versions.This
 *              is useful if your development environment is using
 *              a slightly older version of Linux (like RH 6.2) than
 *              the target Linux
 * 
 * @Param:  - IN net_device *arg_pDev  - pointer to net_device
 *
 * @Return: Zero
 */

int
ix_cc_stkdrv_vidd_ifd_stop(struct net_device *arg_pDev)
{
    return 0;
}

/**
 * NAME: ix_cc_stkdrv_vidd_ifd_set_config
 *
 * DESCRIPTION: Entry point for configuring the driver
 *              
 * 
 * @Param:  - IN net_device *arg_pDev  - pointer to net_device
 *            IN struct ifmap *arg_pMap - pointer to ifmap structure
 * @Return: Zero indicates success; negative error code indicates failure
 */
static int
ix_cc_stkdrv_vidd_ifd_set_config(struct net_device *arg_pDev, 
		struct ifmap *arg_pMap)
{
    ix_ossl_message_log("ix_cc_stkdrv_ifd_set_config: .....\n");
    return 0;
}

/**
 * NAME: ix_cc_stkdrv_vidd_ifd_do_ioctl
 *
 * DESCRIPTION: This function is called whenever a private ioctl call made for
 * this driver
 * Note
 * Ideally this function shouldn't be there at all. The only purpose it serves
 * is to set mac address for each interface.One should
 * use the standard SIOCSIFHWADDR ioctl call.
 *          
 * 
 * @Param:  - IN net_device *arg_pDev  - pointer to net_device
 *            IN struct ifreq *arg_pReq - ioctl request structure
 *            IN int arg_cmd - private ioctl command value
 * @Return: Zero
 */
static int
ix_cc_stkdrv_vidd_ifd_do_ioctl(struct net_device *arg_pDev,
	       	struct ifreq *arg_pReq, 
                          int arg_cmd)
{
    if (NULL == arg_pDev)
        return -1;

    switch(arg_cmd)
    {
        case IFD_SET_MAC:
            if (netif_running(arg_pDev))
            return -EBUSY;
            ix_ossl_memcpy(arg_pDev->dev_addr,
             arg_pReq->ifr_hwaddr.sa_data, arg_pDev->addr_len);

        break;
        default:
            ix_ossl_message_log("ix_cc_stkdrv_ifd_do_ioctl: Invalid Private Ioctl %d made for device %s\n",
              arg_cmd, arg_pDev->name);
        break;
    }

    return 0;
}

/**
 * NAME: ix_cc_stkdrv_vidd_ifd_get_stats
 *
 * DESCRIPTION: This function is called whenever the kernel wants to get the statistics
 * for a particular interface
 * 
 * @Param:  - IN net_device *arg_pDev - pointer to net_device
 *
 * @Return: Pointer to statistics
 */
struct net_device_stats *
ix_cc_stkdrv_vidd_ifd_get_stats(struct net_device *arg_pDev)
{
    return &((ix_cc_stkdrv_vidd_physical_if_node*)arg_pDev->priv)->stats;
}

/**
 * NAME: ix_cc_stkdrv_vidd_ifd_set_multicast_list
 *
 * DESCRIPTION: Apart from getting called for changes in multicast addresses this 
 * function is also called whenever there are some changes in the flag 
 * settings for a particular interface. (Anything to do with multicast is
 * not supported now). For now this function is only a place holder.
 *
 * 
 * @Param:  - IN net_device *arg_pDev - pointer to net_device
 *
 * @Return: None
 */
void
ix_cc_stkdrv_vidd_ifd_set_multicast_list(struct net_device *arg_pDev)
{
    ix_ossl_message_log("ifd_multicast_list:... \n");
    return;
}

/**
 *
 * NAME: ix_cc_stkdrv_vidd_set_macaddr
 * DESCRIPTION- This function sets MAC address for network device
 * @param arg_pDev  - IN pointer to the network device
 * @param arg_pAddr - IN new address
 * @return
 *  Zero indicates succcess, negative error code indicates failure.
 *****************************************************************************/
int
ix_cc_stkdrv_vidd_set_macaddr(
    struct net_device *arg_pDev,
    void              *arg_pAddr)
{
    STKDRV_TRACE(4, "%s()\n", __FUNCTION__);
    return -EOPNOTSUPP;
}

/**
 *
 * NAME: ix_cc_stkdrv_vidd_change_mtu
 * DESCRIPTION
 *  This function is configured as the net_device::change_mtu function in the
 *  network device object registered in the OS kernel. Here we do not allow
 *  changing MTU.
 * @param arg_pDev  - I/O pointer to the network device
 * @param arg_mtu  - IN new MTU
 * @return
 *  Zero indicates succcess, negative error code indicates failure.
 * @see
 *  BpdBpDevStartXmit()
 *****************************************************************************/
int
ix_cc_stkdrv_vidd_change_mtu(
    struct net_device *arg_pDev,
    int               arg_mtu)
{
    STKDRV_TRACE(4, "%s()\n", __FUNCTION__);
    return -EOPNOTSUPP;
}



/**
 *
 * NAME: ix_cc_stkdrv_vidd_hard_parse
 * DESCRIPTION
 * @param skb  - IN pointer to socket buffer (struct sk_buff) with the packet
 * @param saddr- IN source address
 * @return - zero
 *  
**/
static int
ix_cc_stkdrv_vidd_hard_parse(
    struct sk_buff *skb,
    unsigned char  *haddr)
{
    STKDRV_TRACE(4, "%s()\n", __FUNCTION__);
    return 0;
}


/**
 * NAME: ix_cc_stkdrv_vidd_hard_header
 * DESCRIPTION
 *  This function is configured as the net_device::hard_header function in the
 *  network device object registered in the OS kernel. It builds packet
 *  transport header. Here it it does not do anything, since we do not build
 *  L2 header in VIDD.
 * @param skb  - I/O pointer to socket buffer (struct sk_buff) with the packet
 * @param dev  - IN pointer to the network device
 * @param type - IN type of packet
 * @param daddr- IN destination address to use in the header
 * @param saddr- IN source address to use in the header
 * @param len  - IN length of packet
 * @return
 *  Length of the header created in 'skb' is returned.
 * @see
 *  BpdBpDevStartXmit()
**/
static int
ix_cc_stkdrv_vidd_hard_header(
    struct sk_buff    *skb,
    struct net_device *dev,
    unsigned short    type,
    void              *daddr,
    void              *saddr,
    unsigned          len)
{
    STKDRV_TRACE(4, "%s(0x%04x, %sdst, %ssrc, %d)\n", __FUNCTION__,
                 type, daddr ? "" : "no ", saddr ? "" : "no ", len);
    return 0;
} /* ix_cc_stkdrv_vidd_hard_header() */


/**
 * NAME: ix_cc_stkdrv_vidd_rebuild_header
 *
 * DESCRIPTION: This function is configured as the net_device::rebuild_header function in
 *  the network device object registered in the OS kernel. It would rebuild
 *  link layer header after address resolution. Here it is not expected to be
 *  ever called, but if so, just return OK.
 *
 * 
 * @Param:  - IN struct skbuff *skb - pointer to sk_buff
 *
 * @Return: Zero for success (always returns zero)
 */

int
ix_cc_stkdrv_vidd_rebuild_header(
    struct sk_buff    *skb)
{
    STKDRV_TRACE(4, "%s()\n", __FUNCTION__);
    
    return 0;
} /* ix_cc_stkdrv_vidd_rebuild_header() */


/**
 * NAME: ix_cc_stkdrv_vidd_set_sockaddr
 *
 * DESCRIPTION: Utility function
 *
 * 
 * @Param:  - IN struct sockaddr_in *arg_pSin - pointer to socket address
 * @Param:  - IN ix_uint32 arg_addr - address
 * @Param:  - IN ix_uint16 arg_port - port number
 *
 * @Return: None
 */
static void
ix_cc_stkdrv_vidd_set_sockaddr(struct sockaddr_in *arg_pSin,
	               	ix_uint32 arg_addr, ix_uint16 arg_port)
{
    arg_pSin->sin_family = AF_INET;
    arg_pSin->sin_addr.s_addr = arg_addr;
    arg_pSin->sin_port = arg_port;
}



/**
 * NAME: ix_cc_stkdrv_vidd_if_up
 *
 * DESCRIPTION:  This function enables an interface
 * 
 * @Param:  - IN arg_pIf - pointer to interface
 *
 * @Return: Zero for success, non-zero for failure 
 */
int
ix_cc_stkdrv_vidd_if_up(ix_cc_stkdrv_vidd_physical_if_node *arg_pIf)
{
    struct net_device *pDev = &arg_pIf->netdev;
    struct ifreq ir;
    int err;

    ix_ossl_memset(&ir, 0, sizeof(ir));
    strcpy(ir.ifr_ifrn.ifrn_name, pDev->name);

    if ((err = ix_cc_stkdrv_vidd_if_dev_ioctl(SIOCGIFFLAGS, &ir)) < 0)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("if_init: Unable to get interface flags (%d).\n",
                                                               err);
#endif
        return err;
    }

    ir.ifr_ifru.ifru_flags |= IFF_UP | IFF_RUNNING; 

    if ((err = ix_cc_stkdrv_vidd_if_dev_ioctl(SIOCSIFFLAGS, &ir)) < 0)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("if_init: Unable to set interface flags (%d).\n",
                                                               err);
#endif
        return err;
    }

    return 0;
}


/**
 * NAME: ix_cc_stkdrv_vidd_if_down
 *
 * DESCRIPTION:  This function disables an interface
 * Note:
 * But for a minor difference this function is very similar 
 * to interface_down. However interface_down is primarily used
 * on secondary/alias interfaces and as yet doesn't work on primary
 * interface. If and when we change that function, this function can
 * reuse interface_down.
 *
 * @Param:  - IN arg_pIf - pointer to interface
 *
 * @Return: Zero for success, non-zero for failure 
 */
int
ix_cc_stkdrv_vidd_if_down(ix_cc_stkdrv_vidd_physical_if_node *arg_pIf)
{
    struct net_device *pDev = &arg_pIf->netdev;
    struct ifreq ir;
    int err;

    ix_ossl_memset(&ir, 0, sizeof(ir));
    strcpy(ir.ifr_ifrn.ifrn_name, pDev->name);

    if ((err = ix_cc_stkdrv_vidd_if_dev_ioctl(SIOCGIFFLAGS, &ir)) < 0)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("if_init: Unable to get interface flags (%d).\n", err);
#endif
        return err;
    }

    ir.ifr_ifru.ifru_flags &= ~(IFF_UP | IFF_RUNNING); 

    if ((err = ix_cc_stkdrv_vidd_if_dev_ioctl(SIOCSIFFLAGS, &ir)) < 0)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("if_init: Unable to set interface flags (%d).\n", err);
#endif
        return err;
    }

    return 0;
}

/**
 * NAME: ix_cc_stkdrv_vidd_init
 *
 * DESCRIPTION: This function is called by the CC Module to initialize the
 * local VIDD and register the VIDD's entry points with the CC Module.
 * 
 * @Param:  - IN ix_buffer_free_list_handle arg_hFreeList - handle to
 *          freelist used to allocate buffers for passing packets down
 *          from the stack.
 * @Param:  - IN void *arg_pInitData- Initialization data that needs to be 
 *            passed in to init function 
 *
 * @Param:  - INOUT ix_cc_stkdrv_fp_node* arg_pFp - pointer to the FP structure
 *          used to get data for all ports.
 * @Param:  - INOUT ix_cc_stkdrv_handler_module* arg_pHandlerModule -
 *          the function will fill in this structure with its entry points
 *          and contexts.
 *
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_OOM_SYSTEM
 *          IX_CC_STKDRV_VIDD_ERROR_KERNEL
 */
ix_error ix_cc_stkdrv_vidd_init(
                                ix_buffer_free_list_handle arg_hFreeList,
                                ix_cc_stkdrv_fp_node *arg_pFp,
				void *arg_pInitData,
                                ix_cc_stkdrv_handler_module *arg_pHandlerModule
                                )
{
    ix_error                           err = IX_SUCCESS;
    
    struct net_device                  *pDev;
    ix_cc_stkdrv_vidd_ctrl             *pViddCtrl;
    ix_cc_stkdrv_vidd_fp_node          *pFp;
    ix_cc_stkdrv_vidd_physical_if_node *pPhysicalIf = NULL;
    ix_cc_stkdrv_physical_if_node      *pCurrPhysicalIf;
    
    STKDRV_TRACE(2, "%s()\n", __FUNCTION__);
    
    /* Allocate memory for VIDD control structure. */
    pViddCtrl = ix_ossl_malloc(sizeof(ix_cc_stkdrv_vidd_ctrl));
    gViddCtrl = pViddCtrl;
    if (pViddCtrl == NULL)
    {
        STKDRV_LOG(STKDRV_CRIT,
                   "Cannot allocate memory for VIDD control structure");
        return IX_ERROR_NEW(IX_CC_ERROR_OOM_SYSTEM, IX_ERROR_LEVEL_WARNING);
    }
    ix_ossl_memset(pViddCtrl, 0, sizeof(ix_cc_stkdrv_vidd_ctrl));
    
    /* Allocate memory for FP structure. */
    pFp = pViddCtrl->pFps = ix_ossl_malloc(sizeof(ix_cc_stkdrv_vidd_fp_node));
    if (pFp == NULL)
    {
        STKDRV_LOG(STKDRV_CRIT, "Could not allocate memory for FP structure");
        err = IX_ERROR_NEW(IX_CC_ERROR_OOM_SYSTEM, IX_ERROR_LEVEL_WARNING);
        goto label_1;
    }
    pFp->fpId = arg_pFp->fpId;
    pFp->pNextFp = NULL;
    pFp->numPorts = 0;
    pFp->pPhysicalIfs = NULL;
    
    /* Create physical interfaces for the FP. */
    for (pCurrPhysicalIf = arg_pFp->pPhysicalIfs;
         pCurrPhysicalIf != NULL;
         pCurrPhysicalIf = pCurrPhysicalIf->pNextPhysicalIf)
    {
        /* Allocate memory for physical if structure. */
        pPhysicalIf = ix_ossl_malloc(sizeof(*pPhysicalIf));
        if (pPhysicalIf == NULL)
        {
            STKDRV_LOG(STKDRV_CRIT,
                       "Could not allocate memory for physical if structure");
            err = IX_ERROR_NEW(IX_CC_ERROR_OOM_SYSTEM, IX_ERROR_LEVEL_WARNING);
            goto label_1;
        }
        ix_ossl_memset(pPhysicalIf, 0, sizeof(*pPhysicalIf));
        /* Store a pointer to the VIDD control structure in the PhysicalIf structure */
        pPhysicalIf->pViddCtrl = pViddCtrl;
        /*Store a pointer to the physicalIfInfo structure in the PhysicalIf structure */
        pPhysicalIf->pPhysicalIfInfo = pCurrPhysicalIf->pPhysicalIfInfo;
        
        /*
         * Register the interface in the kernel.
         */
        pDev = &pPhysicalIf->netdev;
        snprintf(pDev->name, IFNAMSIZ, "%s%u", IX_CC_STKDRV_DEV_NAME,
                 (unsigned)pPhysicalIf->pPhysicalIfInfo->portId);
        pDev->type = ARPHRD_VOID;     /* ??? */
        pDev->hard_header_len = 0;    /* ??? see EIOCGHDRLEN */
        pDev->mtu = pPhysicalIf->pPhysicalIfInfo->MTU;
        pDev->addr_len = IX_CC_MAC_ADDR_LEN;
        ix_ossl_memset(pDev->broadcast, 0xff, sizeof(pDev->broadcast));
        
        pDev->tx_queue_len = 0;
        pDev->flags = IFF_BROADCAST | IFF_NOARP;
                
        pDev->change_mtu = ix_cc_stkdrv_vidd_change_mtu;
        pDev->hard_header = ix_cc_stkdrv_vidd_hard_header;
        pDev->rebuild_header = ix_cc_stkdrv_vidd_rebuild_header;
        pDev->set_mac_address = ix_cc_stkdrv_vidd_set_macaddr;
        pDev->hard_header_cache = NULL;
        pDev->header_cache_update = NULL;
        pDev->hard_header_parse = ix_cc_stkdrv_vidd_hard_parse;
        /* MAC address on all the virtual IFs would be same
         * hence copy from the first one.
         */
        ix_ossl_memcpy(pDev->dev_addr,
               &pPhysicalIf->pPhysicalIfInfo->pVirtualIfs->macAddr,
               sizeof(pDev->dev_addr));
        pDev->priv = (void*)pPhysicalIf;
        
        pDev->get_stats = ix_cc_stkdrv_vidd_ifd_get_stats;
        pDev->open = ix_cc_stkdrv_vidd_ifd_open;
        pDev->stop = ix_cc_stkdrv_vidd_ifd_stop;
        pDev->hard_start_xmit = ix_cc_stkdrv_vidd_ifd_tx;
        pDev->set_multicast_list = ix_cc_stkdrv_vidd_ifd_set_multicast_list;
        pDev->do_ioctl = ix_cc_stkdrv_vidd_ifd_do_ioctl;
        pDev->set_config = ix_cc_stkdrv_vidd_ifd_set_config;
        
        SET_MODULE_OWNER(pDev);
        
        err = register_netdev(pDev);
        if (err != 0)
        {
            STKDRV_LOG(STKDRV_ERR, "Cannot register interface '%s' to the "
                       "protocol stack", pDev->name);
            err = IX_ERROR_NEW(IX_CC_STKDRV_VIDD_ERROR_MUX,
                               IX_ERROR_LEVEL_WARNING);
            goto label_1;
        }
        STKDRV_LOG(STKDRV_NOTICE, "Device '%s' created\n", pDev->name);
        /*
         * Fill in port-specific contexts for physical if structure and link it
         * to the list of ifs on FP.         
         */
        pPhysicalIf->pCCPktContext = pCurrPhysicalIf;
        pPhysicalIf->pNextPhysicalIf = pFp->pPhysicalIfs;
        pFp->pPhysicalIfs = pPhysicalIf;
        /* Increment number of ports on this FP */
        pFp->numPorts++;
	err= ix_cc_stkdrv_vidd_if_up(pPhysicalIf);
        if (err != 0)
        {
            STKDRV_LOG(STKDRV_ERR, "Cannot enable interface '%s' - status %d",
                       pDev->name, (int)err);
        }
    }

    /*
     * Perform handler registration - fill in the arg_pHandlerModule structure
     * - fill in pHandler_func with the function pointers and contexts for
     * the local VIDD.
     */
    arg_pHandlerModule->pHandler_func->receive_pkt = ix_cc_stkdrv_vidd_receive_pkt;
    arg_pHandlerModule->pHandler_func->pPktContext = (void*)pViddCtrl;
    arg_pHandlerModule->pHandler_func->receive_msg_str = NULL;
    arg_pHandlerModule->pHandler_func->pMsgStrContext = NULL;
    arg_pHandlerModule->pHandler_func->receive_msg_int = (ix_cc_stkdrv_msg_int_cb)ix_cc_stkdrv_vidd_receive_msg_int;
    arg_pHandlerModule->pHandler_func->pMsgIntContext = (void*)pViddCtrl;
    arg_pHandlerModule->pHandler_func->fini = ix_cc_stkdrv_vidd_fini;
    arg_pHandlerModule->pHandler_func->pFiniContext = pViddCtrl;
    /* Take the freelist handle and copy it in VIDD control structure */
    pViddCtrl->hFreeList = arg_hFreeList;
    
    /*
     * VIDD on VxWorks assignes IP addresses to its interfaces. It is not the
     * usual case in Linux that L2 interface manages its L3 addressing, 
     * but lets try to do it. We have to know when network layer is 
     * initialized, so let us schedule a task that will probe for 
     * IP initialization to finish and then assign IP addresses.
     */
    INIT_TQUEUE(&pViddCtrl->ipSetupTask,
                ix_cc_stkdrv_vidd_ip_setup_task, pViddCtrl);
    schedule_task(&pViddCtrl->ipSetupTask);
    return IX_SUCCESS;
    
  label_1:
    if(pPhysicalIf != NULL)
    {
        ix_ossl_free(pPhysicalIf);
        pPhysicalIf = NULL;
    }
    _ix_cc_stkdrv_vidd_cleanup(0, (void **)&pViddCtrl);
    return err;
} /* ix_cc_stkdrv_vidd_init() */


/**
 * NAME: ix_cc_stkdrv_vidd_fini
 *
 * DESCRIPTION: This function is called by the CC Module to shutdown the local
 * VIDD and free any memory allocated to it.
 * 
 * @Param:  - IN void *arg_pContext - pointer to context, in this case the VIDD
 *          control structure.
 *
 * @Return: IX_SUCCESS, or an appropriate error token for failure.
 */
ix_error ix_cc_stkdrv_vidd_fini(
                                void *arg_pContext
                                )
{
    return _ix_cc_stkdrv_vidd_cleanup(1, &arg_pContext);
}


/**
 * NAME: _ix_cc_stkdrv_vidd_cleanup
 *
 * DESCRIPTION: Cleans up VIDD memory and data structures created during
 *              initialiaztion.
 * 
 * @Param:  - IN ix_uint32 arg_checkErrors - 0 indicates the function will
 *          not check for errors.  1 indicates the function will check for
 *          errors.
 * @Param:  - INOUT void **arg_ppViddCtrl - pointer to location of VIDD
 *          control structure.
 *
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_NULL
 *          IX_CC_STKDRV_VIDD_ERROR_KERNEL
 */
ix_error _ix_cc_stkdrv_vidd_cleanup(
                                    ix_uint32 arg_checkErrors,
                                    void **arg_ppViddCtrl
                                    )
{
    ix_cc_stkdrv_vidd_ctrl	*pViddCtrl = *arg_ppViddCtrl;
    ix_cc_stkdrv_vidd_physical_if_node	*pPhysicalIf = NULL;
    
    STKDRV_TRACE(2, "%s()\n", __FUNCTION__);
    
    /* Check for NULL control structure. */
    STKDRV_ASSERT(pViddCtrl != NULL, arg_checkErrors, return IX_CC_ERROR_NULL);
    
    pViddCtrl->ipSetupStop = 1;
#if defined(IX_IPV6_SUPPORT)
    pViddCtrl->ipPropUpdateStop = 1;
    pViddCtrl->ipPropDeleteStop = 1;
#endif
    flush_scheduled_tasks();    /* make sure IP setup will not wake up   *
                                 * with rubbish at VIDD control location */
    if (pViddCtrl->pFps != NULL)
    {
        while ((pPhysicalIf = pViddCtrl->pFps->pPhysicalIfs) != NULL)
        {
            /*
             * Unregister the device from the protocol stack.
             * If device is running it will be stopped first.
             */
            unregister_netdev(&pPhysicalIf->netdev);
            pViddCtrl->pFps->numPorts--;
            pViddCtrl->pFps->pPhysicalIfs = pPhysicalIf->pNextPhysicalIf;
            ix_ossl_free(pPhysicalIf);
        }
        ix_ossl_free(pViddCtrl->pFps);
        pViddCtrl->pFps = NULL;
    }
    ix_ossl_free(pViddCtrl);
    *arg_ppViddCtrl = NULL;
    return IX_SUCCESS;
} /* ix_cc_stkdrv_vidd_cleanup() */


/**
 * NAME: ix_cc_stkdrv_vidd_receive_pkt
 *
 * DESCRIPTION: This is the VIDD entry point (packet processing
 * callback invoked by the CC Module) that receives packets from
 * the CC Module and passes them to the local network stack.The packet
 * handler for stack driver Core Component calls this function.
 * The received packets are handed over to the TCP/IP stack here. 
 * The packet is packaged in a sk_buff (it involves a copy) before 
 * sending it up the TCP/IP stack.
 * 
 * @Param:  - IN ix_buffer_handle arg_hBuffer - handle to the input packet.
 * @Param:  - IN void *arg_pCtx - pointer to the context, in this case
 *          the VIDD control structure.
 * @Param:  - IN ix_cc_stkdrv_packet_type packetType - packet type.
 *          Ignored in the VIDD case.
 *
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_OOM_SYSTEM
 *          IX_CC_ERROR_ENTRY_NOT_FOUND
 *          IX_CC_ERROR_NULL
 *          IX_CC_ERROR_INTERNAL
 */
ix_error ix_cc_stkdrv_vidd_receive_pkt(
                                       ix_buffer_handle arg_hBuffer,
                                       void *arg_pCtx, 
                                       ix_cc_stkdrv_packet_type packetType
                                       )
{
    ix_error          err;
    ix_uint32         portId;
    ix_uint32         pktLen, bufLen;
    ix_uint32         isEOP=0;
    ix_buffer_handle  hCurrBuffer = arg_hBuffer;
    ix_buffer_handle  hNextBuffer;
    
    ix_hw_buffer_meta *pMeta;
    void              *pData = NULL;
    
    ix_cc_stkdrv_vidd_ctrl             *pViddCtrl;
    ix_cc_stkdrv_vidd_physical_if_node *pPhysicalIf = NULL;
    
    struct net_device *pDev = NULL;
    struct sk_buff    *pSkb = NULL;
    ix_uint16         protocol_id;
#if defined(IX_IPV6_SUPPORT)
    ix_uint32          hdr_type;
#endif


    STKDRV_TRACE(4, "%s(0x%x)\n", __FUNCTION__, (int)arg_hBuffer);
    
    /*
     * Get meta from the buffer.
     */
    err = ix_rm_buffer_get_meta(arg_hBuffer, (void*)&pMeta);
    STKDRV_ASSERT(err == IX_SUCCESS, err,
                  err = IX_CC_ERROR_INTERNAL; goto label_1);
    /**
     * Extract the VIDD control structure from arg_pCtx and return
     * IX_CC_ERROR_NULL if it is invalid.
     */
    STKDRV_ASSERT(arg_pCtx != NULL, 0,
                  err = IX_CC_ERROR_NULL; goto label_1);
    pViddCtrl = (ix_cc_stkdrv_vidd_ctrl*)arg_pCtx;

#if defined(IX_IPV6_SUPPORT)
    /* 
     * We need to check up the type of IP packet, do it here as pMeta
     * would get changed further.
     */
    hdr_type = IX_RM_HW_META_GET_HEADER_TYPE(pMeta->m_BufferInfo);

    /* Check if the packet is IPv6 type */
    /* if(hdr_type & ETHER_IPV6_TYPE) */
    if(hdr_type & 0x02)
    {
      protocol_id = __constant_ntohs(ETH_P_IPV6);
    }
    else
#endif
    {
      /* Packet is IPv4 type */
      protocol_id = ETH_P_IP;
    }

    /**
     * Extract the input port ID from the meta-data of arg_hBuffer and look-up
     * the interface structure from the list of interfaces contained in the
     * VIDD control structure.
     */
    pPhysicalIf = pViddCtrl->pFps->pPhysicalIfs;
    portId = IX_NTOH16(pMeta->m_InputPort);
    _IX_CC_STKDRV_VIDD_GET_PHYSICAL_IF(portId, pPhysicalIf);
    if (pPhysicalIf == NULL)
    {
        STKDRV_TRACE(1, "Port %d not found\n", (int)portId);
        err = IX_CC_ERROR_ENTRY_NOT_FOUND;
        goto label_1;
    }
    pDev = &pPhysicalIf->netdev;
    
    /**
     * Get a socket buffer from the system. If we cannot get it return
     * IX_CC_ERROR_OOM_SYSTEM.
     */
    pktLen = IX_NTOH16(pMeta->m_PacketSize);
    bufLen = IX_NTOH16(pMeta->m_BufferSize);
    if (pktLen < bufLen)
    {
      STKDRV_TRACE(2, "PacketSize %d while BufferSize %d, fixing PacketSize\n",
                   (int)pktLen, (int)bufLen);
      pktLen = bufLen;
    }
    pSkb = dev_alloc_skb(pktLen);
    if (pSkb == NULL)
    {
        pPhysicalIf->stats.rx_dropped++;
        err = IX_ERROR_NEW(IX_CC_ERROR_OOM_SYSTEM, IX_ERROR_LEVEL_WARNING);
        STKDRV_TRACE(1, "Cannot get sk_buff for packet from port %d\n",
                     (int)portId);
        goto label_1;
    }
    pSkb->dev = pDev;
    err = ix_rm_buffer_get_data(hCurrBuffer, &pData);
    STKDRV_ASSERT(err == IX_SUCCESS, err,
                  err = IX_CC_ERROR_INTERNAL; goto label_1);
    pData += IX_NTOH16(pMeta->m_Offset);
    ix_ossl_memcpy(skb_put(pSkb, bufLen), pData, bufLen);
    _IX_CC_STKDRV_DEBUG_ERROR_CRT(ix_rm_buffer_get_next(hCurrBuffer, 
			    &hNextBuffer),
            IX_CC_ERROR_INTERNAL, IX_ERROR_LEVEL_WARNING);
    err = ix_rm_buffer_is_eop(hCurrBuffer, &isEOP);
    STKDRV_ASSERT(err == IX_SUCCESS, err,
                  err = IX_CC_ERROR_INTERNAL; goto label_1);
    /*
     * Handle buffer chains.
     */
    while (!isEOP) 
    {
        /* Translate the ix_buffer into a sk_buff buffer. */
        hCurrBuffer = hNextBuffer;
        err = ix_rm_buffer_get_meta(hCurrBuffer, (void**)&pMeta);
        STKDRV_ASSERT(err == IX_SUCCESS, err,
                      err = IX_CC_ERROR_INTERNAL; goto label_1);
        err = ix_rm_buffer_get_data(hCurrBuffer, &pData);
        STKDRV_ASSERT(err == IX_SUCCESS, err,
                      err = IX_CC_ERROR_INTERNAL; goto label_1);
        pData += IX_NTOH16(pMeta->m_Offset);
        
        bufLen = IX_NTOH16(IX_RM_MEM_UINT16_READ(&pMeta->m_BufferSize));
        ix_ossl_memcpy(skb_put(pSkb, bufLen), pData, bufLen);
        
        err = ix_rm_buffer_get_next(hCurrBuffer, &hNextBuffer);
        STKDRV_ASSERT(err == IX_SUCCESS, err,
                      err = IX_CC_ERROR_INTERNAL; goto label_1);
        err = ix_rm_buffer_is_eop(hCurrBuffer, &isEOP);
        STKDRV_ASSERT(err == IX_SUCCESS, err,
                      err = IX_CC_ERROR_INTERNAL; goto label_1);
    }
    pPhysicalIf->stats.rx_packets++;
    pPhysicalIf->stats.rx_bytes += pktLen;
    
    STKDRV_DUMP(pSkb->data, pSkb->len, "VIDD Rx");
 
#if !defined(IX_IPV6_SUPPORT)
    pSkb->mac.raw = pSkb->data;
#endif
    pSkb->protocol = protocol_id; 
    /*
     * Call the receive routine for the stack with a pointer to the buffer.
     */
    netif_rx(pSkb);

    /* Free the ix_buffer chain. */
    err = ix_rm_buffer_free_chain(arg_hBuffer);
    STKDRV_ASSERT(err == IX_SUCCESS, err,
                  err = IX_CC_ERROR_INTERNAL; goto label_1); 

    return err;
    
  label_1:
    if (pSkb != NULL)
    {
        pPhysicalIf->stats.rx_errors++;
        kfree_skb(pSkb);
    }
    return err;
} /* ix_cc_stkdrv_vidd_receive_pkt() */



/**
 * NAME: ix_cc_stkdrv_vidd_ifd_tx
 *
 * DESCRIPTION: transmit function called by the kernel for this driver. The packet
 * is sent to the output. The output is typically bound to IPv4 Fwdr CC.
 * This fn stores the packet data in an ix_buffer_handle before calling the stack driver
 * core component API ix_cc_stkdrv_send_packet() function.
 *  Note:
 * When a socket function like sendto is called by an application, it
 * translates in to a call to this function. If the amount of data to be
 * transmitted is small enough to be in a single packet then the kernel
 * may call this function in the context of the user process. This is not a 
 * problem. However if the data is large then the kernel queues multiple
 * packets. This results in multiple calls to ifd_tx and these are made 
 * from the bottom halves. i.e the context is arbitrary and many limitations
 * that apply to interrupt handlers apply to this function. e.g you 
 * cannot sleep/wait in this function.
 *
 * 
 * @Param:  - IN net_device *arg_pDev - pointer to device.It identifies the 
 *                                  transmit interface.
 * @Param:  - IN struct sk_buff *arg_pSkb - sk_buffer containing the network
 *          buffer. The data represents the full link layer frame. 
 * @Return: zero for success
 *          Non-zero for error
 */
int
ix_cc_stkdrv_vidd_ifd_tx(struct sk_buff *arg_pSkb, struct net_device *arg_pDev)
{
    ix_cc_stkdrv_vidd_physical_if_node *pPhysicalIf;
    ix_buffer_free_list_info   freeListInfo;
    ix_buffer_free_list_handle hFreeList;
    ix_buffer_handle           hHeadBuffer = IX_NULL_BUFFER_HANDLE;
    ix_buffer_handle           hCurrBuffer = IX_NULL_BUFFER_HANDLE;
    ix_buffer_handle           hPrevBuffer = IX_NULL_BUFFER_HANDLE;
    ix_hw_buffer_meta          *pMeta;
    ix_uint8                   *pData;
    int                        err = 0;
    ix_uint32                  pktLen, bufLen, bufMaxLen;
    ix_uint32 iterator=0;
    ix_uint16                  temp;
    ix_cc_stkdrv_og_pkt_type   pktType;


    STKDRV_TRACE(4, "%s()\n", __FUNCTION__);
    STKDRV_DUMP(arg_pSkb->data, arg_pSkb->len, "VIDD Tx");
    /*
     * If it is an IP packet, turn off the most significant bit in the IP ID.
	 * This is not required for IPv6.
     */
#if defined(IX_IPV6_SUPPORT)
    if (arg_pSkb->protocol == __constant_htons(ETH_P_IPV6))
    {
        pktType = IX_CC_STKDRV_OG_PKT_TYPE_IPV6;
    }
    else
#endif
    {
        pktType = IX_CC_STKDRV_OG_PKT_TYPE_IPV4;
        _ix_cc_stkdrv_vidd_morph_ip_id(arg_pSkb->data);
    }

    
    pPhysicalIf = (ix_cc_stkdrv_vidd_physical_if_node*)arg_pDev->priv;
    hFreeList = pPhysicalIf->pViddCtrl->hFreeList;
    ix_rm_buffer_free_list_get_info(hFreeList, &freeListInfo);
    bufMaxLen = freeListInfo.m_DataElementSize;
    
    pktLen = arg_pSkb->len;
    err = ix_rm_buffer_alloc(hFreeList, &hHeadBuffer);
    if (err != IX_SUCCESS)
    {
        STKDRV_TRACE(2, "Allocate buffer error %d\n", err);
        goto label_1;
    }
    err = ix_rm_buffer_get_meta(hHeadBuffer, (void**)&pMeta);
    STKDRV_ASSERT(err == IX_SUCCESS, err, goto label_1);
    err = ix_rm_buffer_get_data(hHeadBuffer, (void **)&pData);
    STKDRV_ASSERT(err == IX_SUCCESS, err, goto label_1);
    bufLen = (pktLen > bufMaxLen - IX_CC_SOP_HW_BUFFER_PREPEND) ?
                              bufMaxLen - IX_CC_SOP_HW_BUFFER_PREPEND : pktLen;
    pData += IX_CC_SOP_HW_BUFFER_PREPEND;
    temp = IX_RM_MEM_UINT16_WRITE(&(pMeta->m_Offset),
		    IX_HTON16((ix_uint16)IX_CC_SOP_HW_BUFFER_PREPEND));
    temp = IX_RM_MEM_UINT16_WRITE(&(pMeta->m_PacketSize),
		    IX_HTON16((ix_uint16)pktLen));
    temp = IX_RM_MEM_UINT16_WRITE(&(pMeta->m_BufferSize), 
                      IX_HTON16((ix_uint16)bufLen));
    temp = IX_RM_MEM_UINT16_WRITE(&(pMeta->m_InputPort),
                      IX_HTON16((ix_uint16)pPhysicalIf->pPhysicalIfInfo->portId));
    err = ix_rm_buffer_cell_count_set(&hHeadBuffer,
                                ix_cc_calculate_cell_count(bufLen, 1));
    STKDRV_ASSERT(err == IX_SUCCESS, err, goto label_1);
    ix_ossl_memcpy(pData, arg_pSkb->data, bufLen);
    skb_pull(arg_pSkb, bufLen);
    hCurrBuffer = hHeadBuffer;

    while (arg_pSkb->len > 0)
    {
        bufLen = (arg_pSkb->len > bufMaxLen) ? bufMaxLen : arg_pSkb->len;
        hPrevBuffer = hCurrBuffer;

        err = ix_rm_buffer_alloc(hFreeList, &hCurrBuffer);
        if (err != IX_SUCCESS)
        {
            STKDRV_TRACE(2, "Allocate buffer error %d\n", err);
            goto label_1;
        }
        err = ix_rm_buffer_get_meta(hCurrBuffer, (void**)&pMeta);
        STKDRV_ASSERT(err == IX_SUCCESS, err, goto label_1);
        err = ix_rm_buffer_get_data(hCurrBuffer, (void **)&pData);
        STKDRV_ASSERT(err == IX_SUCCESS, err, goto label_1);
        temp = IX_RM_MEM_UINT16_WRITE(&(pMeta->m_Offset), (ix_uint16)0);
        temp = IX_RM_MEM_UINT16_WRITE(&(pMeta->m_PacketSize),
		IX_HTON16((ix_uint16)pktLen));
        temp = IX_RM_MEM_UINT16_WRITE(&(pMeta->m_BufferSize),
		 IX_HTON16((ix_uint16)bufLen));	
        IX_RM_HW_META_SET_HEADER_TYPE(pMeta->m_BufferInfo, 0);
        
        ix_rm_buffer_cell_count_set(&hCurrBuffer,
                                    ix_cc_calculate_cell_count(bufLen, 0));
        ix_ossl_memcpy(pData, arg_pSkb->data, bufLen);
        skb_pull(arg_pSkb, bufLen);
        
        err = ix_rm_buffer_link(&hPrevBuffer, &hCurrBuffer);
        STKDRV_ASSERT(err == IX_SUCCESS, err, goto label_2);
        if (iterator==0)
        {
            hHeadBuffer = hPrevBuffer;
            iterator++;
        }
    }
    
    /* Send the packet to the CC side. */
    err = ix_cc_stkdrv_send_packet(hHeadBuffer, pPhysicalIf->pCCPktContext, pktType);
    if (err != IX_SUCCESS)
    {   
        pPhysicalIf->stats.tx_dropped++;
	dev_kfree_skb(arg_pSkb);
	return 0;
    }
    pPhysicalIf->stats.tx_packets++;
    pPhysicalIf->stats.tx_bytes += pktLen;
    
    dev_kfree_skb(arg_pSkb);
    return 0;
  label_2:
    if (hCurrBuffer != IX_NULL_BUFFER_HANDLE)
    {
        ix_rm_buffer_free(hCurrBuffer);
    }    
  label_1:
    pPhysicalIf->stats.tx_dropped++;
    dev_kfree_skb(arg_pSkb);
    if (hHeadBuffer != IX_NULL_BUFFER_HANDLE)
    {
        ix_rm_buffer_free_chain(hHeadBuffer);
    }
    /*
     * Signalling error here would mean asking for retry. But we set queue
     * len to 0 for our devices, we don't want queues (they are in lower
     * layers). Lets drop the packet immediatelly.
     */
    return 0;
} /* ix_cc_stkdrv_vidd_send() */



/**
 * NAME: _ix_cc_stkdrv_vidd_morph_ip_id
 *
 * DESCRIPTION: Turn off the most significant bit in the IP ID field of
 * the IP header.  IPv4 CC is capable of generating ICMP messages. In such
 * situations we have to make sure that IP packets generated by TCP/IP
 * stack and those (ICMP packets) generated by the IPv4 CC have unique IP ID
 * in the IP Header.  We ensure this by making IPv4 generate IP packets
 * with IP IDs with the most significant bit turned on and we turn off the
 * msbit for packets generated by the TCP/IP stack.  The checksum is
 * calculated incrementally. See RFC 1624 for details. Reads and writes
 * from the IP header are done by byte in case the IP header is not aligned.
 * This also ensures that the operations are endian-agnostic.
 * 
 * @Param:  - IN char *arg_pHeader - pointer to the start of the IP header.
 *
 * @Return: IX_SUCCESS
 */
ix_error _ix_cc_stkdrv_vidd_morph_ip_id(char *arg_pHeader)
{
    char *pIpId;
    ix_uint16 ipid;
    ix_uint16 csum;
    char *pCsum;
    ix_uint8  new_ipid;
    ix_uint8  new_csum;
    char  temp;

    /* read ipid from the header. ipid will be in big endian form in the header */
    pIpId = arg_pHeader + IP_HEADER_IPID;

    ipid = ((ix_uint8)(*pIpId)) << 8;
    ipid |= (ix_uint8)(*(pIpId+1));
   /* 
    * We will guarantee that the msb will always be set to 0. 
    * If it's already set to 0, we don't have anything to do
    */
    if (ipid & 0x8000)
    {

        /* read current value of csum */
        pCsum = arg_pHeader + IP_HEADER_CSUM;

        csum = ((ix_uint8)(*pCsum)) << 8;
        csum |= (ix_uint8)(*(pCsum+1));
    
        new_ipid = ipid & 0x7FFF;   /* turn off msbit */

        /* see RFC 1624 for the equation */
        new_csum = ~(~csum + ~ipid + new_ipid);

        /* write new_ipid value back to the header */
        temp = (char)(new_ipid >> 8);
        *pIpId = temp;
        temp = (char)(new_ipid & 0xff );
        *(pIpId+1) = temp;

        /* write new_csum value back to the header */
        temp = (char)(new_csum >> 8);
        *pCsum = temp;
        temp = (char)(new_csum & 0xff );
        *(pCsum+1) = temp;
    }

    return IX_SUCCESS;
}

ix_error
ix_cc_stkdrv_interface_down(const char *name)
{
    struct ifreq ir;
    ix_error err;

    ix_ossl_memset(&ir, 0, sizeof(ir));
    strcpy(ir.ifr_name, name);

    if ((err = ix_cc_stkdrv_vidd_if_dev_ioctl(SIOCGIFFLAGS, &ir)) < 0)
    {
        ix_ossl_message_log("interface_down: Unable to get interface flags \n");
        return err;
    }

    ir.ifr_ifru.ifru_flags &= ~(IFF_UP | IFF_RUNNING); 

    /*
    * We have to use if_devinet_ioctl instead of if_dev_ioctl for
    * disabling a secondary interface. if_dev_ioctl doesn't seem to work 
    * for secondary interfaces. And if_devinet_ioctl doesn't seem to work
    * for primary interface. Probably that was how they were intended to
    * work. 
    */
    /*** I AM NOT SURE IF WE NEED TO SUPPORT SECONDARY INTERFACE *****/
    if ((err = ix_cc_stkdrv_vidd_if_devinet_ioctl(SIOCSIFFLAGS, &ir)) < 0)
    {
        ix_ossl_message_log("interface_down: Unable to set interface flags \n");
        return err;
    }

    return 0;
}

/**
 * NAME: ix_cc_stkdrv_set_ip_mask
 *
 * DESCRIPTION: This function sets IP and netmask for a given interface
 * 
 * @Param:  - IN const ix_uint8 *arg_pName - Name of the interface like
 *            eth0, eth1. This name should be set by calling function 
 * @Param:  - IN int arg_addr - IP address to be set
 * @Param:  - IN int arg_mask - Netmask to be set
 *
 * @Return: Zero for success, non-zero for failure
 */

int
ix_cc_stkdrv_set_ip_mask(const ix_uint8 *arg_pName, int arg_addr, 
                   int arg_mask)
{

    struct ifreq ir;
    struct sockaddr_in *sin = (void *) &ir.ifr_ifru.ifru_addr;
    int err;

    ix_ossl_memset(&ir, 0, sizeof(ir));
    strcpy(ir.ifr_name, arg_pName);

    ix_cc_stkdrv_vidd_set_sockaddr (sin, cpu_to_be32(arg_addr), 0); 
	
    if ((err = ix_cc_stkdrv_vidd_if_devinet_ioctl(SIOCSIFADDR, &ir)) < 0)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("ix_cc_stkdrv_set_ip_mask: Unable to set interface address (%d).\n", err);
#endif
        return err;
    }

    ix_cc_stkdrv_vidd_set_sockaddr(sin, cpu_to_be32(arg_mask), 0); 
    if ((err = ix_cc_stkdrv_vidd_if_devinet_ioctl(SIOCSIFNETMASK, &ir)) < 0)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("ix_cc_stkdrv_set_ip_mask: Unable to set interface netmask (%d).\n", err);
#endif
        return err;
    }

    ix_cc_stkdrv_vidd_set_sockaddr(sin, cpu_to_be32(arg_addr | ~arg_mask), 0);
    if ((err = ix_cc_stkdrv_vidd_if_devinet_ioctl(SIOCSIFBRDADDR, &ir)) < 0)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("ix_cc_stkdrv_set_ip_mask: Unable to set interface broadcast address (%d).\n",
                                                                             err);
#endif
        return err;
    }

    return 0;
}

#if defined(IX_IPV6_SUPPORT)
/**
 * NAME: ix_cc_stkdrv_set_ipv6_addr
 *
 * DESCRIPTION: This is a utility function that sets the IPv6
 *              address and its prefix length for a given interface.
 * 
 * @Param:  - IN const net_device *arg_pDev  - Pointer to the net_device
 *               to which IPv6 address has to be added.
 * @Param:  - IN ix_uint128 arg_addr - IPv6 address to be added.
 * @Param:  - IN ix_uint8 arg_addr - IPv6 address prefix length.
 *
 * @Return: Zero for success, non-zero for failure
 */
static int
ix_cc_stkdrv_set_ipv6_addr(const struct net_device *arg_pDev,
  const ix_uint128 *arg_addr,
  const ix_uint8 arg_prefixlen)
{
    struct in6_ifreq ir6;

    ix_ossl_memset(&ir6, 0, sizeof(struct in6_ifreq));

    ix_ossl_memcpy(&ir6.ifr6_addr, arg_addr, sizeof (ix_uint128));
    ir6.ifr6_prefixlen = arg_prefixlen;
    ir6.ifr6_ifindex = arg_pDev->ifindex;

    return 0;
}
#endif /* IX_IPV6_SUPPORT */

/**
 * NAME: ix_cc_stkdrv_set_bcast_addr
 *
 * DESCRIPTION: This function sets broadcast address for a network interface
 * 
 * @Param:  - IN const ix_uint8 *arg_pName - Name of the interface like
 *            eth0, eth1. This name should be set by calling function 
 * @Param:  - IN int arg_addr - Broadcast IP address to be set
 * 
 * @Return: Zero for success, non-zero for failure
 */
int
ix_cc_stkdrv_set_bcast_addr(const ix_uint8 *arg_pName, int arg_addr) 
{

    struct ifreq ir;
    struct sockaddr_in *sin = (void *) &ir.ifr_ifru.ifru_addr;
    int err;

    ix_ossl_memset(&ir, 0, sizeof(ir));
    strcpy(ir.ifr_name, arg_pName);


    ix_cc_stkdrv_vidd_set_sockaddr(sin, cpu_to_be32(arg_addr), 0); 
    if ((err = ix_cc_stkdrv_vidd_if_devinet_ioctl(SIOCSIFBRDADDR, &ir)) < 0)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("ix_cc_stkdrv_set_bcast_addr: Unable to set interface broadcast address (%d).\n",
                                                                             err);
#endif
        return err;
    }

    return 0;
}


/**
 * NAME: ix_cc_stkdrv_delete_ip_addr
 *
 * DESCRIPTION: This function deletes IP address for a given interface
 * 
 * @Param:  - IN const ix_uint8 *arg_pName - Name of the interface like
 *            eth0, eth1. This name should be set by calling function 
 * @Param:  - IN int arg_addr - IP address to be deleted
 *
 *
 * @Return: Zero for success, non-zero for failure
 */

int
ix_cc_stkdrv_delete_ip_addr(const ix_uint8 *arg_pName, int arg_addr) 
{

    struct ifreq ir;
    struct sockaddr_in *sin = (void *) &ir.ifr_ifru.ifru_addr;
    int err;

    ix_ossl_memset(&ir, 0, sizeof(ir));
    strcpy(ir.ifr_name, arg_pName);

    ix_cc_stkdrv_vidd_set_sockaddr (sin, cpu_to_be32(arg_addr), 0);
	
    if ((err = ix_cc_stkdrv_vidd_if_devinet_ioctl(SIOCDIFADDR, &ir)) < 0)
    {
#ifdef IX_DEBUG
        ix_ossl_message_log("ix_cc_stkdrv_delete_ip_addr: Unable to delete IP address for interface (%d).\n",
                                                                                err);
#endif
        return err;
    }

    return 0;
}

#if defined(IX_IPV6_SUPPORT)
/**
 * NAME: ix_cc_stkdrv_delete_ipv6_addr
 *
 * DESCRIPTION: This is a utility function that deletes the IPv6
 *              of an interface.
 * 
 * @Param:  - IN const net_device *arg_pDev  - Pointer to the net_device
 *               to which IPv6 address has to be added.
 * @Param:  - IN ix_uint128 arg_addr - IPv6 address to be added.
 * @Param:  - IN ix_uint8 arg_addr - IPv6 address prefix length.
 *
 * @Return: Zero for success, non-zero for failure
 */

static int
ix_cc_stkdrv_delete_ipv6_addr(const struct net_device *arg_pDev,
  const ix_uint128 *arg_addr,
  const ix_uint8 arg_prefixlen)
{

    struct in6_ifreq ir6;

    ix_ossl_memset(&ir6, 0, sizeof(ir6));

    ix_ossl_memcpy(&ir6.ifr6_addr, arg_addr, sizeof (ix_uint128));
    ir6.ifr6_prefixlen = arg_prefixlen;
    ir6.ifr6_ifindex = arg_pDev->ifindex;

    return 0;
}
#endif /* IX_IPV6_SUPPORT */

/*****************************************************************************
 * Description
 *  Setup IP addressing on all interfaces on FP.
 *****************************************************************************/
void
ix_cc_stkdrv_vidd_ip_setup_task(
    void *ref)
{
    struct proc_dir_entry    *entry = NULL;
    ix_cc_stkdrv_vidd_ctrl   *pViddCtrl = (ix_cc_stkdrv_vidd_ctrl*)ref;
    ix_cc_stkdrv_vidd_physical_if_node *pPhysicalIf;
    
    STKDRV_TRACE(2, "%s()\n", __FUNCTION__);
    
    if (pViddCtrl->ipSetupStop)
    {
        STKDRV_TRACE(2, "IP setup procedure stopped\n");
        return;
    }
#ifdef CONFIG_PROC_FS
    for (entry = proc_net->subdir; entry != NULL; entry = entry->next)
    {
        if (entry->low_ino && entry->namelen == 8 &&
            memcmp(entry->name, "rt_cache", 8) == 0)
        {
            break;
        }
    }
#endif
    if (entry == NULL)
    {
        static int counter = 20;	/* it will be 10 s */
        if (--counter)
        {
            wait_queue_head_t wait;
            init_waitqueue_head(&wait);
            interruptible_sleep_on_timeout(&wait, HZ / 2);
            if (pViddCtrl->ipSetupStop)
            {
                STKDRV_TRACE(2, " IP setup procedure stopped\n");
            }
            else
            {
                schedule_task(&pViddCtrl->ipSetupTask); /* try again */
            }
        }
        else
        {
            STKDRV_LOG(STKDRV_ERR,
                       "Timeout when waiting for IP initialization");
        }
        return;
    }
    for (pPhysicalIf = pViddCtrl->pFps->pPhysicalIfs;
         pPhysicalIf != NULL; pPhysicalIf = pPhysicalIf->pNextPhysicalIf)
    {
        ix_cc_stkdrv_vidd_ip_setup(pPhysicalIf);
    }
    return;
} /* ix_cc_stkdrv_vidd_ip_setup_task() */

/**
 * NAME: ix_cc_stkdrv_vidd_ip_setup
 *
 * DESCRIPTION: This function sets up IP addressing on the device 
 * 
 * @Param:  - IN ix_cc_stkdrv_vidd_physical_if_node *pPhysicalIf-pointer
 * to the physical port structure 
 *
 * @Return: IX_SUCCESS if successful or a valid ix_error token on failure.
*/
ix_error
ix_cc_stkdrv_vidd_ip_setup(
    ix_cc_stkdrv_vidd_physical_if_node *pPhysicalIf)
{
    int          status;
    struct ifreq ifr;
    mm_segment_t oldfs;
    ix_cc_stkdrv_virtual_if* pCurrVirtualIf;
    
    struct sockaddr_in *sin = (struct sockaddr_in*)&ifr.ifr_ifru.ifru_addr;
    
    STKDRV_TRACE(2, "%s()\n", __FUNCTION__);
    
    STKDRV_ASSERT(pPhysicalIf != NULL, pPhysicalIf, return -1);

    /* Iterate through each of the virtual interfaces */
    for (pCurrVirtualIf = pPhysicalIf->pPhysicalIfInfo->pVirtualIfs;
         pCurrVirtualIf != NULL;
         pCurrVirtualIf = pCurrVirtualIf->pNextVirtualIf)
    {
#if defined(IX_IPV6_SUPPORT) 
        if (pCurrVirtualIf->ipProp.ip_version == IX_CC_IP_VERSION_IPV6)
        {
            STKDRV_LOG (STKDRV_NOTICE,
                "Change IP address on '%s' to %x:%x:%x:%x:%x:%x:%x:%x\n",
                pPhysicalIf->netdev.name,
                pCurrVirtualIf->ipProp.protocol.ipv6_prop.ipv6_address.addr.aUint16[0],
                pCurrVirtualIf->ipProp.protocol.ipv6_prop.ipv6_address.addr.aUint16[1],
                pCurrVirtualIf->ipProp.protocol.ipv6_prop.ipv6_address.addr.aUint16[2],
                pCurrVirtualIf->ipProp.protocol.ipv6_prop.ipv6_address.addr.aUint16[3],
                pCurrVirtualIf->ipProp.protocol.ipv6_prop.ipv6_address.addr.aUint16[4],
                pCurrVirtualIf->ipProp.protocol.ipv6_prop.ipv6_address.addr.aUint16[5],
                pCurrVirtualIf->ipProp.protocol.ipv6_prop.ipv6_address.addr.aUint16[6],
                pCurrVirtualIf->ipProp.protocol.ipv6_prop.ipv6_address.addr.aUint16[7]);
            status = ix_cc_stkdrv_set_ipv6_addr (&pPhysicalIf->netdev,
                         &pCurrVirtualIf->ipProp.protocol.ipv6_prop.ipv6_address.addr,
                         pCurrVirtualIf->ipProp.protocol.ipv6_prop.ipv6_address.prefixlen);
            if (status != 0)
            {
                STKDRV_LOG(STKDRV_ERR,
                    "Unable to set IPv6 address on device '%s' (error %d)",
                    pPhysicalIf->netdev.name, status);
                return IX_CC_ERROR_RANGE;
            }
        }
       else
#endif
       {
          STKDRV_LOG(STKDRV_NOTICE,
                  "Change IP address on '%s' to %d.%d.%d.%d\n",
                  pPhysicalIf->netdev.name,
                  NIPQUAD(pCurrVirtualIf->ipProp.protocol.ipv4_prop.ipv4_address));
          oldfs = get_fs();
          set_fs(get_ds());

          strcpy(ifr.ifr_ifrn.ifrn_name, pPhysicalIf->netdev.name);
          sin->sin_family = AF_INET;
          sin->sin_port = 0;
          sin->sin_addr.s_addr = pCurrVirtualIf->ipProp.protocol.ipv4_prop.ipv4_address;
          status = ix_cc_stkdrv_vidd_if_devinet_ioctl(SIOCSIFADDR, &ifr);
          if (status != 0)
          {
              STKDRV_LOG(STKDRV_ERR,
                      "Unable to set IP address on device '%s' (error %d)",
                      ifr.ifr_ifrn.ifrn_name, status);
              set_fs(oldfs);
              return IX_CC_ERROR_RANGE;
          }
          sin->sin_addr.s_addr = pCurrVirtualIf->
                                    ipProp.protocol.ipv4_prop.ipv4_subnet_mask;
          status = ix_cc_stkdrv_vidd_if_devinet_ioctl(SIOCSIFNETMASK, &ifr);
          if (status != 0)
          {
              STKDRV_LOG(STKDRV_WARNING,
                   "Unable to set IP mask on device '%s' (error %d)",
                   ifr.ifr_ifrn.ifrn_name, status);
          }
          sin->sin_addr.s_addr = pCurrVirtualIf->
                                     ipProp.protocol.ipv4_prop.ipv4_address;
          sin->sin_addr.s_addr |= ~pCurrVirtualIf->
                                       ipProp.protocol.ipv4_prop.ipv4_subnet_mask;
          status = ix_cc_stkdrv_vidd_if_devinet_ioctl(SIOCSIFBRDADDR, &ifr);
          if (status != 0)
          {
              STKDRV_LOG(STKDRV_WARNING,
                   "Unable to set IP broadcast on device '%s' (error %d)",
                   ifr.ifr_ifrn.ifrn_name, status);
          }
          set_fs(oldfs);
      }   /* else of IPv6 */
   } /* For loop */
   return IX_SUCCESS;
} /* ix_cc_stkdrv_vidd_ip_setup() */

/**
 * NAME: ix_cc_stkdrv_vidd_receive_msg_int
 *
 * DESCRIPTION: This is the VIDD entry point (integer message processing
 * callback invoked by the CC Module) that receives integer messages
 * from the CC Module and processes the interface property updates
 * contained within those messages.
 * 
 * @Param:  - IN ix_cc_stkdrv_vidd_msg_id arg_MsgId - identification of the message.
 * @Param:  - IN ix_uint32 arg_msg - integer message value.
 * @Param:  - IN void *arg_pContext - pointer to the context.  Here the context
 *          is the physical interface structure.
 *
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_NULL
 *          IX_CC_STKDRV_VIDD_ERROR_KERNEL
 *          IX_CC_ERROR_UNDEFINED_MSG
 */
ix_error ix_cc_stkdrv_vidd_receive_msg_int(
                                           ix_cc_stkdrv_vidd_msg_id arg_msgId,
                                           const ix_uint32 arg_msg,
                                           void *arg_pContext,
					   void** arg_ppReplyMsg,
					   ix_uint32* arg_pReplyMsgSize
                                           )
{

    ix_error err = IX_SUCCESS;

    ix_cc_stkdrv_vidd_ctrl* pViddCtrl = (ix_cc_stkdrv_vidd_ctrl*)arg_pContext;

    if((arg_pContext == NULL) || (arg_ppReplyMsg == NULL) || (arg_pReplyMsgSize == NULL))
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
         ("VIDD int msg handler invoked with NULL context or reply message pointer"));
    }

    /* Initialize reply message to be empty. */
    *arg_ppReplyMsg = NULL;
    *arg_pReplyMsgSize = 0;

    switch(arg_msgId)
    {
        case IX_CC_STKDRV_VIDD_MSG_ID_PROPERTY_UPDATE:
        {
            /**
             * Handle message to update port properties, make
             * kernel-specific calls to notify the kernel of these
             * property updates.
             */
            ix_cc_stkdrv_vidd_property_update_msg* pMsg = 
                (ix_cc_stkdrv_vidd_property_update_msg*)arg_msg;

            IX_ERROR_CR(_ix_cc_stkdrv_vidd_set_property(pMsg->propID, pMsg->pProperty, pViddCtrl));

            break;
        }
        
        default:
            return IX_ERROR_WARNING(IX_CC_ERROR_UNDEFINED_MSG, ("Invalid message ID %d for int msg to VIDD", arg_msgId));
    }

    return err;
}

/**
 * NAME: ix_cc_stkdrv_vidd_set_property
 *
 * DESCRIPTION: This is the function that takes care of updation 
 * of the interface properties, specified by the property ID
 * and the property data structure 
 * 
 * @Param:  - IN ix_uint32 arg_propID - identification of the
 * property to be set 
 * @Param:  - IN ix_cc_properties *arg_pProperty - pointer to the 
 * structure tha will provide property value 
 * @Param:  - IN ix_cc_stkdrv_vidd_ctrl *arg_pViddCtrl - pointer to 
 * the VIDD control structure
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error token on failure.
*/
ix_error _ix_cc_stkdrv_vidd_set_property(
                                         ix_uint32 arg_propID,
                                         ix_cc_properties *arg_pProperty,
                                         ix_cc_stkdrv_vidd_ctrl *arg_pViddCtrl
                                         )
{
    ix_cc_stkdrv_vidd_physical_if_node *pPhysicalIf;
#if defined(IX_IPV6_SUPPORT)
    ix_cc_stkdrv_prop_update *pPropUpdate;
#endif

    /* Get the port for this port ID. */
    pPhysicalIf = arg_pViddCtrl->pFps->pPhysicalIfs;
    _IX_CC_STKDRV_VIDD_GET_PHYSICAL_IF(arg_pProperty->port_id, pPhysicalIf);
    if(pPhysicalIf == NULL)
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL, ("Port %d not found", arg_pProperty->port_id));


    /* Handle message to set interface status. */
    if(arg_propID & IX_CC_SET_PROPERTY_PHYSICAL_IF_STATUS)
    {
        if((arg_pProperty->physical_if_state == IX_CC_PHYSICAL_IF_STATUS_UP) &&
            (arg_pProperty->link_state == IX_CC_LINK_STATUS_UP))
            ix_cc_stkdrv_vidd_if_up(pPhysicalIf);
        else if(arg_pProperty->physical_if_state == IX_CC_PHYSICAL_IF_STATUS_DOWN)
            ix_cc_stkdrv_vidd_if_down(pPhysicalIf);
    }

    /* Handle message to set link status. */
    if(arg_propID & IX_CC_SET_PROPERTY_LINK_STATUS)
    {
        if((arg_pProperty->physical_if_state == IX_CC_PHYSICAL_IF_STATUS_UP) &&
            (arg_pProperty->link_state == IX_CC_LINK_STATUS_UP))
            ix_cc_stkdrv_vidd_if_up(pPhysicalIf);
        else if(arg_pProperty->link_state == IX_CC_LINK_STATUS_DOWN)
            ix_cc_stkdrv_vidd_if_down(pPhysicalIf);
    }

    /* Handle message to add IPv4 address. */
    if(arg_propID & IX_CC_SET_PROPERTY_ADD_INTERFACE_IPV4)
    {
        ix_uint32 ret_status = 0;
        ix_uint32 loop_var;
        char *devName = IX_CC_STKDRV_DEV_NAME;

        char devUnit[IX_CC_STKDRV_DEV_NAME_LEN];
        char ipAddrString[IX_CC_STKDRV_IP_ADDR_STR_LEN];
        ix_uint32 ipAddr;

        for (loop_var = 0;
             loop_var < arg_pProperty->no_virtual_if;
             loop_var++)
        {
            if (arg_pProperty->ip_prop[loop_var].ip_version
                == IX_CC_IP_VERSION_IPV4)
            {
                ipAddr =
                    IX_HTON32(arg_pProperty->ip_prop[loop_var].protocol.ipv4_prop.ipv4_address);
                sprintf(devUnit, "%s%lu", devName,
                    pPhysicalIf->pPhysicalIfInfo->portId);
                sprintf(ipAddrString, "%lu.%lu.%lu.%lu",
                    ipAddr >> 24, (ipAddr >> 16) & 0xFF,
                    (ipAddr >> 8) & 0xFF, ipAddr & 0xFF);

                /* Add the virtual interface. */
                /* Set the IP address and subnet mask for the device. */
                ret_status =
                    ix_cc_stkdrv_set_ip_mask(devUnit,
                       arg_pProperty->ip_prop[loop_var].protocol.ipv4_prop.ipv4_address,
                       arg_pProperty->ip_prop[loop_var].protocol.ipv4_prop.ipv4_subnet_mask);
                if (ret_status != 0)
                    return IX_ERROR_WARNING(IX_CC_STKDRV_VIDD_ERROR_MUX,
                               ("Could not add IP address for port %lu",
                               arg_pProperty->port_id));
            }
        }
    }

    /**
     * Handle message to delete IPv4 address - in this release
     * we maintain only 1 IP address per port.
     */
    if(arg_propID & IX_CC_SET_PROPERTY_DEL_INTERFACE_IPV4)
    {
        ix_uint32 loop_var;
        int ret_status = 0;
        char *devName = IX_CC_STKDRV_DEV_NAME;

        char devUnit[IX_CC_STKDRV_DEV_NAME_LEN];
        char ipAddrString[IX_CC_STKDRV_IP_ADDR_STR_LEN];
        
        ix_uint32 ipAddr;

        for (loop_var = 0;
             loop_var < arg_pProperty->no_virtual_if;
             loop_var++)
        {
            if (arg_pProperty->ip_prop[loop_var].ip_version
                == IX_CC_IP_VERSION_IPV4)
            {
                ipAddr =
                    IX_HTON32(arg_pProperty->ip_prop[loop_var].protocol.ipv4_prop.ipv4_address);

                sprintf(devUnit, "%s%lu",
                    devName, pPhysicalIf->pPhysicalIfInfo->portId);
                sprintf(ipAddrString, "%lu.%lu.%lu.%lu",
                    ipAddr >> 24, (ipAddr >> 16) & 0xFF,
                    (ipAddr >> 8) & 0xFF, ipAddr & 0xFF);
                ret_status =
                    ix_cc_stkdrv_delete_ip_addr(devUnit, 
                    arg_pProperty->ip_prop[loop_var].protocol.ipv4_prop.ipv4_address);
                if(ret_status != 0)
                    return IX_ERROR_WARNING(IX_CC_STKDRV_VIDD_ERROR_KERNEL,
                               ("Could not delete IP address for port %lu",
                                arg_pProperty->port_id));
            }
        } 
    }

    /* Handle message to set IPv4 broadcast address. */
    if(arg_propID & IX_CC_SET_PROPERTY_ADD_BROADCAST_IPV4)
    {
        ix_uint32 loop_var;
        char *devName = IX_CC_STKDRV_DEV_NAME;

        char devUnit[IX_CC_STKDRV_DEV_NAME_LEN];
        char ipAddrString[IX_CC_STKDRV_IP_ADDR_STR_LEN];
        ix_uint32 ipAddr;
        for (loop_var = 0;
             loop_var < arg_pProperty->no_virtual_if;
             loop_var++)
        {
            if (arg_pProperty->ip_prop[loop_var].ip_version
                == IX_CC_IP_VERSION_IPV4)
            {
                ipAddr =
                    IX_HTON32(arg_pProperty->ip_prop[loop_var].protocol.ipv4_prop.ipv4_broadcast);
                sprintf(devUnit, "%s%lu",
                    devName, pPhysicalIf->pPhysicalIfInfo->portId);
                sprintf(ipAddrString, "%lu.%lu.%lu.%lu",
                    ipAddr >> 24, (ipAddr >> 16) & 0xFF,
                    (ipAddr >> 8) & 0xFF, ipAddr & 0xFF);

                /* Set the IPv4 broadcast address for this device. */
                ix_cc_stkdrv_set_bcast_addr(devUnit, 
                arg_pProperty->ip_prop[loop_var].protocol.ipv4_prop.ipv4_address);
            }
        }
    }

#if defined(IX_IPV6_SUPPORT)
    /* Handle message to add IPv6 address. */
    if(arg_propID & IX_CC_SET_PROPERTY_ADD_INTERFACE_IPV6)
    {
        /* Note that this memory allocated here will be freed in
           bottom half */
        pPropUpdate =
            ix_ossl_malloc (sizeof(ix_cc_stkdrv_prop_update));
        if (pPropUpdate == NULL)
        {
            STKDRV_LOG(STKDRV_CRIT,
                "Cannot allocate memory for prop update structure");
            return IX_ERROR_NEW(IX_CC_ERROR_OOM_SYSTEM, IX_ERROR_LEVEL_WARNING);
        }
        else
        {
            pPropUpdate->propId = IX_CC_SET_PROPERTY_ADD_INTERFACE_IPV6;
            ix_ossl_memcpy (&pPropUpdate->property, arg_pProperty,
                sizeof(ix_cc_properties));
            pPropUpdate->pViddCtrl = arg_pViddCtrl;
            /* Assign the IPv6 address in the bottom half, straight away
               assigning the IPv6 address is not working */
            INIT_TQUEUE(&arg_pViddCtrl->ipPropUpdateTask,
                ix_cc_stkdrv_vidd_ipv6_setup_prop_update_task, (void *)pPropUpdate);
            schedule_task(&arg_pViddCtrl->ipPropUpdateTask);
        }
    }
#endif

#if defined(IX_IPV6_SUPPORT)
    /**
     * Handle message to delete IPv6 address.
     */
    if(arg_propID & IX_CC_SET_PROPERTY_DEL_INTERFACE_IPV6)
    {
        /* Note that this memory allocated here will be freed in
           bottom half */
        pPropUpdate =
          ix_ossl_malloc (sizeof(ix_cc_stkdrv_prop_update));
        if (pPropUpdate == NULL)
        {
            STKDRV_LOG(STKDRV_CRIT,
                "Cannot allocate memory for prop update structure");
            return IX_ERROR_NEW(IX_CC_ERROR_OOM_SYSTEM, IX_ERROR_LEVEL_WARNING);
        }
        else
        {
            pPropUpdate->propId = IX_CC_SET_PROPERTY_DEL_INTERFACE_IPV6;
            ix_ossl_memcpy (&pPropUpdate->property, arg_pProperty,
                sizeof(ix_cc_properties));
            pPropUpdate->pViddCtrl = arg_pViddCtrl;
            /* Delete the IPv6 address in the bottom half, straight away
               deleting the IPv6 address is not working */
            INIT_TQUEUE(&arg_pViddCtrl->ipPropDeleteTask,
                ix_cc_stkdrv_vidd_ipv6_setup_prop_update_task, (void *)pPropUpdate);
            schedule_task(&arg_pViddCtrl->ipPropDeleteTask);
        }
    }
#endif

    return IX_SUCCESS;
}

#if defined(IX_IPV6_SUPPORT)
/**
 * NAME: ix_cc_stkdrv_vidd_ipv6_setup_prop_update_task 
 *
 * DESCRIPTION: This is the function implemented as bottom half
 *              and it adds or deletes the IPV6 address of an
 *              interface. This function takes care of freeing
 *              the memory allocated to the argument.
 *
 * @Param:  - IN  void *arg - Contains the VIDD control structure
 *                            the property update structure and
 *                            the command indicating whether to add
 *                            delete the IPv6 address. 
 *
 * @Return: Nothing
*/

static
void ix_cc_stkdrv_vidd_ipv6_setup_prop_update_task(void *arg_pPropUpdate)
{
    struct proc_dir_entry    *entry = NULL;
    ix_cc_stkdrv_prop_update *pPropUpdate =
      (ix_cc_stkdrv_prop_update *)arg_pPropUpdate;
    ix_cc_stkdrv_vidd_physical_if_node *pPhysicalIf;
    int ret_status;
    ix_uint32 loop_var;

    /* Get the port for this port ID. */
    pPhysicalIf = pPropUpdate->pViddCtrl->pFps->pPhysicalIfs;
    _IX_CC_STKDRV_VIDD_GET_PHYSICAL_IF (
        pPropUpdate->property.port_id,
        pPhysicalIf);

    STKDRV_TRACE(2, "%s()\n", __FUNCTION__);
    
    if (pPropUpdate->pViddCtrl->ipPropUpdateStop)
    {
        STKDRV_TRACE(2, "IP update procedure stopped\n");
        return;
    }
#ifdef CONFIG_PROC_FS
    for (entry = proc_net->subdir; entry != NULL; entry = entry->next)
    {
        if (entry->low_ino && entry->namelen == 8 &&
            memcmp(entry->name, "rt_cache", 8) == 0)
        {
            break;
        }
    }
#endif
    if (entry == NULL)
    {
        static int counter = 20;	/* it will be 10 s */
        if (--counter)
        {
            wait_queue_head_t wait;
            init_waitqueue_head(&wait);
            interruptible_sleep_on_timeout(&wait, HZ / 2);
            if (pPropUpdate->pViddCtrl->ipPropUpdateStop)
            {
                STKDRV_TRACE(2, " IP setup procedure stopped\n");
            }
            else
            {
                schedule_task(&pPropUpdate->pViddCtrl->ipPropUpdateTask); /* try again */
            }
        }
        else
        {
            STKDRV_LOG(STKDRV_ERR,
                       "Timeout when waiting for IP initialization");
        }
        return;
    }

    for (loop_var = 0;
         loop_var < pPropUpdate->property.no_virtual_if;
         loop_var++)
    {
        if (pPropUpdate->property.ip_prop[loop_var].ip_version
            == IX_CC_IP_VERSION_IPV6)
        {
            if (pPropUpdate->propId == IX_CC_SET_PROPERTY_ADD_INTERFACE_IPV6)
            {
                ret_status =
                    ix_cc_stkdrv_set_ipv6_addr(&pPhysicalIf->netdev,
                        &pPropUpdate->property.ip_prop[loop_var].protocol.ipv6_prop.ipv6_address.addr,
                        pPropUpdate->property.ip_prop[loop_var].protocol.ipv6_prop.ipv6_address.prefixlen);
                if (ret_status < 0)
                {
                    STKDRV_LOG(STKDRV_ERR,
                        "Could not set the IPv6 address");
                }
            }
            else if (pPropUpdate->propId == IX_CC_SET_PROPERTY_DEL_INTERFACE_IPV6)
            {
                ret_status =
                    ix_cc_stkdrv_delete_ipv6_addr(&pPhysicalIf->netdev,
                        &pPropUpdate->property.ip_prop[loop_var].protocol.ipv6_prop.ipv6_address.addr,
                        pPropUpdate->property.ip_prop[loop_var].protocol.ipv6_prop.ipv6_address.prefixlen);
                if (ret_status < 0)
                {
                    STKDRV_LOG(STKDRV_ERR,
                        "Could not delete the IPv6 address");
                }
            }
            else
            {
                STKDRV_LOG(STKDRV_ERR,
                    "Unrecognised update command");
            }
        }
    }
    /* Free the memory of the argument passed */
    ix_ossl_free (arg_pPropUpdate);
    return;
} /* ix_cc_stkdrv_vidd_ipv6_setup_prop_update_task() */
#endif
